

#include <stdlib.h>
#include <string.h>

#include "user_io.h"
#include "pstring.h"
#include "dataio.h"
#include "global.h"
#include "matrix.h"


#define MOUSEDEAD           0.1
#define MOUSEMAX            1.0

#ifdef WIN32	 
#define EXTRACT_KEY(xxx) atoi(xxx)
#else
#define EXTRACT_KEY(xxx) (0x1ff & atoi(xxx))
#endif

string_type FILENAME_KEYBOARD_DECODER("");
int CONTROLLER = CONTROLKEYBOARD;


/* **********************************************************************
********************************************************************** */
user_io::user_io() {

   flags = FLAG_IO_NULL;
   
   memset(control, 0, MAXAXIS<<2);

   // keyboard
   memset(status_keypress, 0, MAXKEYS);
   memset(status_keyold, 0, MAXKEYS);
   commandcount = 0;
   keyxlator = NULL;
   keymove[0] = keymove[1] = keymove[2] = 0;

   // mouse & joystick
   status_buttonpress = 0;
   buttoncount = 0;
   buttonxlator = NULL;
   povdegree = POV_DEFAULT;
}


/* **********************************************************************
********************************************************************** */
user_io::~user_io() {

   if (keyxlator)
      delete [] keyxlator;

   if (buttonxlator)
      delete [] buttonxlator;
}


/* **********************************************************************
********************************************************************** */
int user_io::read_commands(char *filename) {

   FILE *infile;
   int i, j;
   char token[MAXSTRLEN];
   
   if (!find_file(FILENAME_KEYBOARD_DECODER.string, "r", NULL, PLATFORM_SLASH, NULL, &infile))
      return 0;

   for (i=0; i<256; i++)
      if (get_token(infile, token))
         keyboard_driver[i] = EXTRACT_KEY(token);
      else
         break;
   
   for (; i<256; i++)
      keyboard_driver[i] = 0;

   fclose(infile);
   
   if (!find_file(filename, "r", NULL, PLATFORM_SLASH, NULL, &infile))
      return 0;

   if (keyxlator)
      delete [] keyxlator;

   get_token(infile, token);
   commandcount = atoi(token);

   keyxlator = new int[(commandcount+3) & 0xfffffffc];

   for (i=0; i<commandcount; i++) {
      get_token(infile, token);
      keyxlator[i] = keyboard_driver[atoi(token)];
   }

   if (buttonxlator)
      delete [] buttonxlator;

   get_token(infile, token);
   buttoncount = atoi(token);

   buttonxlator = new int[buttoncount];

   for (i=0; i<buttoncount; i++) {
      get_token(infile, token);
      j = atoi(token);
      buttonxlator[i] = j == 255 ? 255 : keyxlator[atoi(token)];
   }

   fclose(infile);
   return 1;
}


/* **********************************************************************
********************************************************************** */
void user_io::get_events() {

   poll_mouse();
   poll_joystick();
   poll_keyboard();
}


/* **********************************************************************
********************************************************************** */
void user_io::process_events(timer_type *timer) {

   int i, mask;
   float x, y;

   memcpy(status_keyold, status_keypress, MAXKEYS);
   memset(status_keypress, 0, MAXKEYS);
   memset(control, 0, MAXAXIS<<2);
   memset(joyaxis, 0, MAXAXIS<<2);
   status_buttonpress = 0;
   povdegree = POV_DEFAULT;

   get_events();

   memcpy(pure_keypress, status_keypress, MAXKEYS);

   // convert button presses to key presses
   for (i=0, mask = 1; i<buttoncount; i++, mask += mask)
      if (buttonxlator[i] != -1 && (status_buttonpress & mask))
         if (buttonxlator[i] < MAXKEYS)
            status_keypress[buttonxlator[i]] = 1;

   // process io to game movement
   switch (CONTROLLER) {

      case CONTROLMOUSE:
         x = joyaxis[AXIS_MOUSE_X];
         y = joyaxis[AXIS_MOUSE_Y];

         control[AXIS_PITCH] = (y < 0) ? y*y : -y*y;
         control[AXIS_YAW]   = (x < 0) ? x*x : -x*x;

         if (translate(KEYSTROKE_ROLL_RIGHT))
            if (keymove[AXIS_ROLL] < KEYMOVE_THRESHOLD)
               control[AXIS_ROLL] = keymove[AXIS_ROLL] = KEYMOVE;
            else {
               control[AXIS_ROLL] = (keymove[AXIS_ROLL] += (float)(timer->speedscale * KEYMOVE));
               if (control[AXIS_ROLL] > 1)
                  control[AXIS_ROLL] = 1;
            }

         else if (translate(KEYSTROKE_ROLL_LEFT))
            if (keymove[AXIS_ROLL] > -KEYMOVE_THRESHOLD)
               control[AXIS_ROLL] = keymove[AXIS_ROLL] = -KEYMOVE;
            else {
               control[AXIS_ROLL] = (keymove[AXIS_ROLL] -= (float)(timer->speedscale * KEYMOVE));
               if (control[AXIS_ROLL] < -1)
                  control[AXIS_ROLL] = -1;
            }

//         else
//            control[AXIS_ROLL] = keymove[AXIS_ROLL] = 0;

         if (translate(KEYSTROKE_HAT_UP))
            povdegree = 0.0f;
         else if (translate(KEYSTROKE_HAT_DOWN))
            povdegree = 180.0f;

         if (translate(KEYSTROKE_HAT_LEFT))
            povdegree = 270.0f;
         else if (translate(KEYSTROKE_HAT_RIGHT))
            povdegree = 90.0f;

         break;

      case CONTROLKEYBOARD:

         // pitch
         if (translate(KEYSTROKE_PITCH_FORWARD))
            if (keymove[AXIS_PITCH] < KEYMOVE_THRESHOLD)
               control[AXIS_PITCH] = keymove[AXIS_PITCH] = KEYMOVE;
            else {
               control[AXIS_PITCH] = (keymove[AXIS_PITCH] += (float)(timer->speedscale * KEYMOVE));
               if (control[AXIS_PITCH] > 1)
                  control[AXIS_PITCH] = 1;
            }

         else if (translate(KEYSTROKE_PITCH_BACKWARD))
            if (keymove[AXIS_PITCH] > -KEYMOVE_THRESHOLD)
               control[AXIS_PITCH] = keymove[AXIS_PITCH] = -KEYMOVE;
            else {
               control[AXIS_PITCH] = (keymove[AXIS_PITCH] -= (float)(timer->speedscale * KEYMOVE));
               if (control[AXIS_PITCH] < -1)
                  control[AXIS_PITCH] = -1;
            }

//         else
//            control[AXIS_PITCH] = keymove[AXIS_PITCH] = 0;

         // yaw
         if (translate(KEYSTROKE_YAW_LEFT))
            if (keymove[AXIS_YAW] < KEYMOVE_THRESHOLD)
               control[AXIS_YAW] = keymove[AXIS_YAW] = KEYMOVE;
            else {
               control[AXIS_YAW] = (keymove[AXIS_YAW] += (float)(timer->speedscale * KEYMOVE));
               if (control[AXIS_YAW] > 1)
                  control[AXIS_YAW] = 1;
            }

         else if (translate(KEYSTROKE_YAW_RIGHT))
            if (keymove[AXIS_YAW] > -KEYMOVE_THRESHOLD)
               control[AXIS_YAW] = keymove[AXIS_YAW] = -KEYMOVE;
            else {
               control[AXIS_YAW] = (keymove[AXIS_YAW] -= (float)(timer->speedscale * KEYMOVE));
               if (control[AXIS_YAW] < -1)
                  control[AXIS_YAW] = -1;
            }

//         else
//            control[AXIS_YAW] = keymove[AXIS_YAW] = 0;

         // roll
         if (translate(KEYSTROKE_ROLL_RIGHT))
            if (keymove[AXIS_ROLL] < KEYMOVE_THRESHOLD)
               control[AXIS_ROLL] = keymove[AXIS_ROLL] = KEYMOVE;
            else {
               control[AXIS_ROLL] = (keymove[AXIS_ROLL] += (float)(timer->speedscale * KEYMOVE));
               if (control[AXIS_ROLL] > 1)
                  control[AXIS_ROLL] = 1;
            }

         else if (translate(KEYSTROKE_ROLL_LEFT))
            if (keymove[AXIS_ROLL] > -KEYMOVE_THRESHOLD)
               control[AXIS_ROLL] = keymove[AXIS_ROLL] = -KEYMOVE;
            else {
               control[AXIS_ROLL] = (keymove[AXIS_ROLL] -= (float)(timer->speedscale * KEYMOVE));
               if (control[AXIS_ROLL] < -1)
                  control[AXIS_ROLL] = -1;
            }

//         else
//            control[AXIS_ROLL] = keymove[AXIS_ROLL] = 0;

         if (translate(KEYSTROKE_HAT_UP))
            povdegree = 0.0f;
         else if (translate(KEYSTROKE_HAT_DOWN))
            povdegree = 180.0f;

         if (translate(KEYSTROKE_HAT_LEFT))
            povdegree = 270.0f;
         else if (translate(KEYSTROKE_HAT_RIGHT))
            povdegree = 90.0f;

         break;

      default:      // joystick

         control[AXIS_THROTTLE] = (joyaxis[AXIS_THROTTLE] < -CORRECT) ? -joyaxis[AXIS_THROTTLE]*joyaxis[AXIS_THROTTLE] : joyaxis[AXIS_THROTTLE]*joyaxis[AXIS_THROTTLE];
         control[AXIS_PITCH]    = (joyaxis[AXIS_Y] < -CORRECT) ? joyaxis[AXIS_Y]*joyaxis[AXIS_Y] : -joyaxis[AXIS_Y]*joyaxis[AXIS_Y];
         control[AXIS_YAW]      = (joyaxis[AXIS_X] < -CORRECT) ? -joyaxis[AXIS_X]*joyaxis[AXIS_X] : joyaxis[AXIS_X]*joyaxis[AXIS_X];

         // roll
         if (joyaxis[AXIS_Z] < -CORRECT) {
            control[AXIS_ROLL] = -joyaxis[AXIS_Z]*joyaxis[AXIS_Z];
            keymove[AXIS_ROLL] = 0;
         }

         else if (joyaxis[AXIS_Z] > CORRECT) {
            control[AXIS_ROLL] = joyaxis[AXIS_Z]*joyaxis[AXIS_Z];
            keymove[AXIS_ROLL] = 0;
         }

         else if (translate(KEYSTROKE_ROLL_RIGHT))
            if (keymove[AXIS_ROLL] < KEYMOVE_THRESHOLD)
               control[AXIS_ROLL] = keymove[AXIS_ROLL] = KEYMOVE;
            else {
               control[AXIS_ROLL] = (keymove[AXIS_ROLL] += (float)(timer->speedscale * KEYMOVE));
               if (control[AXIS_ROLL] > 1)
                  control[AXIS_ROLL] = 1;
            }

         else if (translate(KEYSTROKE_ROLL_LEFT))
            if (keymove[AXIS_ROLL] > -KEYMOVE_THRESHOLD)
               control[AXIS_ROLL] = keymove[AXIS_ROLL] = -KEYMOVE;
            else {
               control[AXIS_ROLL] = (keymove[AXIS_ROLL] -= (float)(timer->speedscale * KEYMOVE));
               if (control[AXIS_ROLL] < -1)
                  control[AXIS_ROLL] = -1;
            }

//         else
//            control[AXIS_ROLL] = keymove[AXIS_ROLL] = 0;

         // hats
         if (translate(KEYSTROKE_HAT_UP))
            povdegree = 0.0f;
         else if (translate(KEYSTROKE_HAT_DOWN))
            povdegree = 180.0f;

         if (translate(KEYSTROKE_HAT_LEFT))
            povdegree = 270.0f;
         else if (translate(KEYSTROKE_HAT_RIGHT))
            povdegree = 90.0f;

         break;
   }

}


/* **********************************************************************
********************************************************************** */
void user_io::flag_on(unsigned int x) {

   flags |= x;
}


/* **********************************************************************
********************************************************************** */
void user_io::flag_off(unsigned int x) {

   flags &= ~x;
}


/* **********************************************************************
********************************************************************** */
void user_io::tether_mouse(int x, int y, int winx, int winy) {

   float m;

   m = joyaxis[AXIS_MOUSE_X];

   if ((flags & FLAG_IO_MOUSE_DEADZONE) && m > -MOUSEDEAD && m < MOUSEDEAD)
      m = 0;
   else if (m < -MOUSEMAX) {
      m = (float)-MOUSEMAX;
#ifdef WIN32
      if (flags & FLAG_IO_MOUSE_CONSTRAIN)
         center[0] = x + (winx>>1);
#endif
   }

   else if (m > MOUSEMAX) {
      m = (float)MOUSEMAX;
#ifdef WIN32
      if (flags & FLAG_IO_MOUSE_CONSTRAIN)
         center[0] = x - (winx>>1);
#endif
   }

   joyaxis[AXIS_MOUSE_X] = m;

   m = joyaxis[AXIS_MOUSE_Y];

   if ((flags & FLAG_IO_MOUSE_DEADZONE) && m > -MOUSEDEAD && m < MOUSEDEAD)
      m = 0;
   else if (m < -MOUSEMAX) {
      m = (float)-MOUSEMAX;
#ifdef WIN32
      if (flags & FLAG_IO_MOUSE_CONSTRAIN)
         center[1] = y + (winy>>1);
#endif
   }

   else if (m > MOUSEMAX) {
      m = (float)MOUSEMAX;
#ifdef WIN32
      if (flags & FLAG_IO_MOUSE_CONSTRAIN)
         center[1] = y - (winy>>1);
#endif
   }

   joyaxis[AXIS_MOUSE_Y] = m;
}
