
#include "shader.h"
#include "shaders.h"


/***************************************************************************
****************************************************************************/
float grey_marble_color(float value) {

   if (value < 0.4)
      return 1 - 0.571429f*value;
   if (value < 0.8)
      return 1.125f - 0.75f * value;

   return 1.7866336f - 1.4851485f * value;
}


/***************************************************************************
****************************************************************************/
float sub_marble_color(float value) {

   if (0.8 <= value)
      return -value + 1.2f;

   return -0.5f*value + 0.8f;
}


/***************************************************************************
****************************************************************************/
float main_marble_color(float value) {

   if (value < 0.3)
      return 1.0;
   if (value < 0.7)
      return -1.0f*value + 1.3f;
   if (value < 0.9)
      return -0.75f*value + 0.315f;

   return -0.015f*value + 0.315f;
}


/***************************************************************************
* marble - calculate marble color for given point.
*
* Input : float point[3]
*
* Output: float color[3]
*
* Fc call: turbulence, marble_color;
****************************************************************************/
int marble (void *x) {

//   float  size = 2;             // high freq
   float  isize = 0.5;
   float  ans;
   int    freq = 13;
   vector4f point;
   float temp;

   shaderparamtype *data;

   data = (shaderparamtype *)x;

   point[0] = data->in[0] * isize;
   point[1] = data->in[1] * isize;
   point[2] = data->in[2] * isize;
   point[3] = 1.0;

//   out[0] = main_marble_color(SIN((point[0]+turbulence(point, size)) * freq * PI));
//   out[1] = sub_marble_color(SIN((point[0]+turbulence(point, size)) * freq * PI));

   temp = (point[0]+turbulence(point, 300, 0.0033333333f)) * freq * PI;
   ans = grey_marble_color(SIN(temp));

   smultarray3(data->out, ans);

   return 1;
}


/***************************************************************************
****************************************************************************/
int wood (void *x) {

   float  size = 2;             // high freq
   float  isize = 0.5;
   float  ans;
   int    freq = 13;
   float  point[4];
   float  temp;

   shaderparamtype *data;

   data = (shaderparamtype *)x;

   point[0] = data->in[0] * isize;
   point[1] = data->in[1] * isize;
   point[2] = data->in[2] * isize;
   point[3] = 1;

//   out[0] = main_marble_color(SIN((point[0]+turbulence(point, size)) * freq * PI));
//   out[1] = sub_marble_color(SIN((point[0]+turbulence(point, size)) * freq * PI));

   temp = (point[0]+turbulence(point, size, isize)) * freq * PI;
   ans = main_marble_color(SIN(temp));

   data->out[0] *= ans*1.0f;
   data->out[1] *= ans*0.6f;
   data->out[2] *= ans*0.30196f;

   return 1;
}


/***************************************************************************
****************************************************************************/
int wood_panel (void *x) {

   float  size = 2;             // high freq
   float  isize = 0.5;
   float  ans;
   int    freq = 13;
   float  point[4];
   float  temp;

   shaderparamtype *data;

   data = (shaderparamtype *)x;

   point[0] = data->in[0] + 0.005f - round0(data->in[0]/4.0f)*4.0f;         // 4x1x8
   point[1] = data->in[1] + 0.005f - round0(data->in[1]);
   point[2] = data->in[2] + 0.005f - round0(data->in[2]/8.0f)*8.0f;

   if (point[0] < 0)
      point[0] = 4.0f + point[0];

   if (point[1] < 0)
      point[1] = 1.0f + point[1];

   if (point[2] < 0)
      point[2] = 8.0f + point[2];

   if ((point[0] > 0.01f && point[0] < 0.2f) ||
       (point[1] > 0.01f && point[1] < 0.2f) ||
       (point[2] > 0.01f && point[2] < 0.2f)) {
      smultarray3(data->out, 0.1f);

      return 1;
   }

   smultarray3(point, isize);

   temp = (point[0]+turbulence(point, size, isize)) * freq * PI;
   ans = main_marble_color(SIN(temp));

   data->out[0] *= ans*1.0f;
   data->out[1] *= ans*0.6f;
   data->out[2] *= ans*0.30196f;

   return 1;
}


/***************************************************************************
****************************************************************************/
int wave(void *x) {

   float value;

   shaderparamtype *data;

   data = (shaderparamtype *)x;
   value = turbulence(data->in, 5.745f, 0.174073f);
   saddarray3(data->normal, value);
   normalize3(data->normal);

   return 1;
}


/***************************************************************************
****************************************************************************/
int checker (void *x) {

   int value;
   shaderparamtype *data;

   data = (shaderparamtype *)x;

//   value = fabs((int)data->in[0]) + fabs((int)data->in[1]) + fabs((int)data->in[2]);
   value = (int)data->in[0] + (int)data->in[1] + (int)data->in[2] +
           (data->in[0] < 0) + (data->in[1] < 0) + (data->in[2] < 0) ;

   if (value & 0x01)
      data->out[0] = data->out[1] = data->out[2] = 0;

   return 1;
}


#define HH              0.632
#define LACUNARITY      2
#define OCTAVES         7

fractal_params moon_params(OCTAVES, (float)HH, LACUNARITY);

/***************************************************************************
****************************************************************************/
void moon_init () {

   moon_params.init();
}


/***************************************************************************
****************************************************************************/
int moon (void *x) {

   float high[3] = {0,0.7f,0.8f};
   float basehi[3] = {0.1f,0.3f,0.2f};

   float low[3]  = {0.1f,0.4f,0.3f};
   float baselow[3] = {0.1f,0.4f, 0.6f};
   float value;

   float threshold = 0.0;

   float p[3];
   shaderparamtype *data;

   data = (shaderparamtype *)x;

   p[0] = data->in[0] + data->in[0];
   p[1] = data->in[1] + data->in[1];
   p[2] = data->in[2] + data->in[2];

   value = fBm(p, &moon_params);

   if ( value > threshold) {
      data->out[0] *= (basehi[0] + high[0]*value);
      data->out[1] *= (basehi[1] + high[1]*value);
      data->out[2] *= (basehi[2] + high[2]*value);
      return 1;
   }

   data->out[0] *= (baselow[0] + low[0]*value);
   data->out[1] *= (baselow[1] + low[1]*value);
   data->out[2] *= (baselow[2] + low[2]*value);

   return 1;
}


#undef HH
#undef LACUNARITY
#undef OCTAVES


#define HH              0.632
#define LACUNARITY      2
#define OCTAVES         7


fractal_params planet2_params(OCTAVES, (float)HH, LACUNARITY);

/***************************************************************************
****************************************************************************/
void planet2_init () {

   planet2_params.init();
}


/***************************************************************************
****************************************************************************/
int planet2 (void *x) {

   vector3f high = {0,1.254f,0.2f};
   vector3f basehi = {0.429f,0.373f,0.135f};

   vector3f high2 = {0.196f,0.765f,0};

   vector3f low  = {0.1f,0.4f,0.3f};
   vector3f baselow = {0.1f,0.4f, 0.6f};

   float value;
   float temp;

   float threshold = -0.134544f;
   float threshold2 = 0.134544f;
   float threshold3 = 0.764352f;

   vector3f p;
   shaderparamtype *data;

   data = (shaderparamtype *)x;

   p[0] = data->in[0]*2.9987983f;
   p[1] = data->in[1]*2.9987983f;
   p[2] = data->in[2]*2.9987983f;

   value = fBm(p, &planet2_params);
   temp  = data->in[1] + 0.3f*value;

   if (fabs(temp) > 0.7f) {
      value = 0.25f*(3+value);
      smultarray3(data->out, value);
   }

   else
      if ( value > threshold) {
         if (value < threshold2) {
            multarray3(data->out, basehi);
         }

         else if (value < threshold3) {
            data->out[0] *= (basehi[0] + high[0]*value);
            data->out[1] *= (basehi[1] + high[1]*value);
            data->out[2] *= (basehi[2] + high[2]*value);
         }

         else {
            data->out[0] *= (basehi[0] + high2[0]*value);
            data->out[1] *= (basehi[1] + high2[1]*value);
            data->out[2] *= (basehi[2] + high2[2]*value);
         }

      }

      else {
         data->out[0] *= (baselow[0] + low[0]*value);
         data->out[1] *= (baselow[1] + low[1]*value);
         data->out[2] *= (baselow[2] + low[2]*value);
      }

   return 1;
}


#undef HH
#undef LACUNARITY
#undef OCTAVES


#define HH              0.632
#define LACUNARITY      2
#define OCTAVES         7


fractal_params planet_params(OCTAVES, (float)HH, LACUNARITY);

/***************************************************************************
****************************************************************************/
void planet_init () {

   planet_params.init();
}


/***************************************************************************
****************************************************************************/
int planet (void *x) {

   vector3f high    = {0,1.254f,0.2f};
   vector3f basehi  = {0.429f,0.373f,0.135f};

   vector3f high2   = {0.196f,0.765f,0};

   vector3f low     = {0.1f,0.4f,0.3f};
   vector3f baselow = {0.1f,0.4f, 0.6f};

   float value;

   float threshold = 0;
   float threshold2 = 0.134544f;
   float threshold3 = 0.764352f;

   vector3f p;
   shaderparamtype *data;

   data = (shaderparamtype *)x;

   p[0] = data->in[0]*9.9987983f;
   p[1] = data->in[1]*9.9987983f;
   p[2] = data->in[2]*9.9987983f;

   value = fBm(p, &planet_params);

   if ( value > threshold) {
      if (value < threshold2) {
         multarray3(data->out, basehi);
      }

      else if (value < threshold3) {
         data->out[0] *= (basehi[0] + high[0]*value);
         data->out[1] *= (basehi[1] + high[1]*value);
         data->out[2] *= (basehi[2] + high[2]*value);
      }

      else {
         data->out[0] *= (basehi[0] + high2[0]*value);
         data->out[1] *= (basehi[1] + high2[1]*value);
         data->out[2] *= (basehi[2] + high2[2]*value);
      }

   }

   else {
      data->out[0] *= (baselow[0] + low[0]*value);
      data->out[1] *= (baselow[1] + low[1]*value);
      data->out[2] *= (baselow[2] + low[2]*value);
   }

   return 1;
}


#undef HH
#undef LACUNARITY
#undef OCTAVES


#define HH              0.632
#define LACUNARITY      2
#define OCTAVES         7


fractal_params cloud_params(OCTAVES, (float)HH, LACUNARITY);

/***************************************************************************
****************************************************************************/
void cloud_init () {

   cloud_params.init();
}


/***************************************************************************
****************************************************************************/
int cloud (void *x) {

   float value;

   float high     = 0.4f;
   float basehigh = 0.175f;
   float low      = 0.175f;
   float baselow  = 0.15f;

   float threshold = 0.0f;

   vector3f p;
   shaderparamtype *data;

   data = (shaderparamtype *)x;

   p[0] = data->in[0]*0.09472388f;
   p[1] = data->in[1]*0.09472388f;
   p[2] = data->in[2]*0.09472388f;

   value = fBm(p, &cloud_params);

   if (value > threshold)
      value = basehigh + high*value;
   else
      value = baselow + low*value;

   smultarray3(data->out, value);

   return 1;
}


#undef HH
#undef LACUNARITY
#undef OCTAVES


#define HH              0.632
#define LACUNARITY      2
#define OCTAVES         7


fractal_params cloud2_params(OCTAVES, (float)HH, LACUNARITY);

/***************************************************************************
****************************************************************************/
void cloud2_init () {

   cloud2_params.init();
}


/***************************************************************************
        Note: planetclouds() seems to be better visually than cloud2()
****************************************************************************/
int cloud2 (void *x) {

   float value;

   float high     = 0.4f;
   float basehigh = 0.475f;
   float low      = 0.275f;
   float baselow  = 0.15f;

   float threshold = 0.1f;

   float p[3];
   shaderparamtype *data;

   marble(x);

   data = (shaderparamtype *)x;

   copyarray3(p, data->in);

   value = fBm(p, &cloud2_params);

   if (value > threshold)
      value = basehigh + high*value;
   else
      value = baselow + low*value;

   smultarray3(data->out, value);

   return 1;
}


#undef HH
#undef LACUNARITY
#undef OCTAVES


#define HH              0.632
#define LACUNARITY      2
#define OCTAVES         7


fractal_params warp_params(OCTAVES, (float)HH, LACUNARITY);

/***************************************************************************
****************************************************************************/
void warp_init () {

   warp_params.init();
}


/***************************************************************************
****************************************************************************/
int warp (void *x) {

   float value;

   float threshold = 0.6f;

   float p[3];
   shaderparamtype *data;

   data = (shaderparamtype *)x;

   copyarray3(p, data->in);

   value = fBm(p, &warp_params);

   if (value < threshold) {
      data->out[0] = data->out[1] = data->out[2] = 0;
      return 1;
   }

   p[2]=(float)fabs(data->in[2]);
   if (p[2] > 60)
      value *= 60/p[2];

   smultarray3(data->out, value);

   return 1;
}


#undef HH
#undef LACUNARITY
#undef OCTAVES


#define HH              0.632
#define LACUNARITY      2
#define OCTAVES         7


fractal_params stars_params(OCTAVES, (float)HH, LACUNARITY);

/***************************************************************************
****************************************************************************/
void stars_init () {

   stars_params.init();
}


/***************************************************************************
****************************************************************************/
int stars (void *x) {

   float value;
   float threshold = 0.6f;
   float p[3];
   shaderparamtype *data;

   data = (shaderparamtype *)x;

   p[0] = (float)fabs(data->in[0]);
   p[1] = data->in[1];
   p[2] = data->in[2];

   value = fBm(p, &stars_params);

   if (value < threshold) {
      data->out[0] = data->out[1] = data->out[2] = 0;
      return 1;
   }

   p[0] += value;
   p[1] -= value;
   p[2] += value;

   value = fBm(p, &stars_params);

   if (value < threshold) {
      data->out[0] = data->out[1] = data->out[2] = 0;
      return 1;
   }

// since value >= threshold = 0.6 ...
//   value = fabs(value);

   if (value > 1)
      value -= (int)value;

   if (data->pt[2] < -100)
      value *=-100/data->pt[2];
   else
      value += 0.5f*(1-value);

   smultarray3(data->out, value);

   return 1;
}


#undef HH
#undef LACUNARITY
#undef OCTAVES


#define HH              0.632
#define LACUNARITY      2
#define OCTAVES         7


fractal_params terrain_params(OCTAVES, (float)HH, LACUNARITY);

/***************************************************************************
****************************************************************************/
void terrain_init () {

   terrain_params.init();
}


/***************************************************************************
****************************************************************************/
int terrain (void *x) {

   float value;

   vector3f p, p2;
   shaderparamtype *data;

   data = (shaderparamtype *)x;

   p[0] = data->in[0]*0.0002f;
   p[1] = data->in[1]*0.0002f;
   p[2] = (float)(fabs(data->in[2])*0.0005);                      // remove negative to get bluemoon.rgb

   p[2] *= (float)(2000 * (1 + 0.732743*fBm(p, &terrain_params)));

   if (p[2] < 100) {            // water
      p[2] *= 0.01f;

      value = fBm(p, &terrain_params);
      value = (float)fabs(value);

      data->out[0] *= 0.1f;
      data->out[1] *= 0.3f - 0.2f*value;
      data->out[2] *= 0.6f - 0.25f*value;
      return 0;
   }

   copyarray3(p2, data->in);

   smultarray3(data->in, 0.0005f);

   marble(data);

   copyarray3(data->in, p2);

   if (p[2] > 1600) {           // snow
      p[2] *= 0.0005f;

      value = fBm(p, &terrain_params);
      value = (float)fabs(value);

      value = 0.5f * (1+ value);
      smultarray3(data->out, value);

      return 1;
   }

   if (p[2] > 1000) {           // red rock
      p[2] *= 0.000625f;

      value = fBm(p, &terrain_params);
      value = (float)fabs(value);

      data->out[0] *= 0.5f*(0.6f+value);
      data->out[1] *= 0.1f*(0.6f+value);
      data->out[2] *= 0.1f;
      return 1;
   }

   if (p[2] > 200) {            // vegetation
      p[2] *= 0.001f;

      value = fBm(p, &terrain_params);
      value = (float)fabs(value);

      data->out[0] *= 0.5f*(1-value);
      data->out[1] *= 0.5f*(1+value);
      data->out[2] *= 0.1f;
      return 1;
   }

   p[2] *= 0.005f;               // sand

   value = fBm(p, &terrain_params);
   value = (float)fabs(value);

   value = 0.3f * (1+value);

   data->out[0] *= value;
   data->out[1] *= value;
   data->out[2] = 0;

   return 1;
}


#undef HH
#undef LACUNARITY
#undef OCTAVES


#define HH              0.632
#define LACUNARITY      2
#define OCTAVES         7


fractal_params terrain2_params(OCTAVES, (float)HH, LACUNARITY);

/***************************************************************************
****************************************************************************/
void terrain2_init () {

   terrain2_params.init();
}


/***************************************************************************
****************************************************************************/
int terrain2 (void *x) {

   float value;

   float p[3], p2[3];
   shaderparamtype *data;

   data = (shaderparamtype *)x;

   p[0] = data->in[0]*0.002f;
   p[1] = data->in[1]*0.002f;
   p[2] = data->in[2]*0.006666667f;                      // remove negative to get bluemoon.rgb

   p[2] *= (float)(150 * (1 + 0.732743*fBm(p, &terrain2_params)));

   if (p[2] < -0.09) {            // water deep
      p[2] *= 0.006666667f;

      value = fBm(p, &terrain2_params);
      value = (float)fabs(value);

      data->out[0] *= 0.02f;
      data->out[1] *= 0.14f - 0.09f*value;
      data->out[2] *= 0.28f - 0.12f*value;

      return 1;
   }

   if (p[2] < 0.01f) {            // water
      p[2] *= 0.006666667f;

      value = fBm(p, &terrain2_params);
      value = (float)fabs(value);

      data->out[0] *= 0.1f;
      data->out[1] *= 0.3f - 0.2f*value;
      data->out[2] *= 0.6f - 0.25f*value;

      copyarray3(p2, data->in);

      smultarray3(data->in, 3.5473f);
      wave(data);

      copyarray3(data->in, p2);

      return 1;
   }

   copyarray3(p2, data->in);

   data->in[0] *= 0.0005f;
   data->in[1] *= 0.0005f;
   data->in[2] *= 0.001f;

   marble(data);

   copyarray3(data->in, p2);

   if (p[2] > 110) {           // snow
      p[2] *= 0.006666667f;

      value = fBm(p, &terrain2_params);
      value = (float)(0.5 * (1+fabs(value)));

      smultarray3(data->out, value);

      return 1;
   }

   if (p[2] > 70) {           // red rock
      p[2] *= 0.009090909f;

      value = fBm(p, &terrain2_params);
      value = (float)fabs(value);

      data->out[2] *= 0.1f * (1+ value);

      value = 0.08f * (1+ value);

      data->out[0] *= value;
      data->out[1] *= value;

      return 1;
   }

   if (p[2] > 10) {            // vegetation
      p[2] *= 0.014285714f;

      value = fBm(p, &terrain2_params);
      value = (float)fabs(value);

      data->out[0] *= 0.5f*(1-value);
      data->out[1] *= 0.5f*(1+value);
      data->out[2] *= 0.1f;
      return 1;
   }

   p[2] *= 0.1f;               // sand

   value = fBm(p, &terrain2_params);
   value = (float)fabs(value);
   value = 0.3f * (1+ value);

   data->out[0] *= value;
   data->out[1] *= value;
   data->out[2] = 0;

   return 1;
}


#undef HH
#undef LACUNARITY
#undef OCTAVES


#define HH              0.632
#define LACUNARITY      2
#define OCTAVES         7


fractal_params terrain3_params(OCTAVES, (float)HH, LACUNARITY);

/***************************************************************************
****************************************************************************/
void terrain3_init () {

   terrain3_params.init();
}


/***************************************************************************
****************************************************************************/
int terrain3 (void *x) {

   float value;

   vector3f p, p2;
   shaderparamtype *data;

   data = (shaderparamtype *)x;

   p[0] = data->in[0]*0.002f;
   p[1] = data->in[1]*0.002f;
   p[2] = data->in[2]*0.0066667f;                      // remove negative to get bluemoon.rgb

   p[2] *= (float)(150 * (1 + 0.732743*fBm(p, &terrain3_params)));

   if (p[2] < -0.09f) {            // water deep
      p[2] *= 0.0066667f;

      value = fBm(p, &terrain3_params);
      value = (float)fabs(value);

      data->out[0] *= 0.02f;
      data->out[1] *= 0.14f - 0.09f*value;
      data->out[2] *= 0.28f - 0.12f*value;

      return 1;
   }

   if (p[2] < 0.01f) {            // water
      p[2] *= 0.0066667f;

      value = fBm(p, &terrain3_params);
      value = (float)fabs(value);

      data->out[0] *= 0.1f;
      data->out[1] *= 0.3f - 0.2f*value;
      data->out[2] *= 0.6f - 0.25f*value;

      return 1;
   }

   copyarray3(p2, data->in);

   data->in[0] *= 0.0005f;
   data->in[1] *= 0.0005f;
   data->in[2] *= 0.001f;

   marble(data);

   copyarray3(data->in, p2);

   if (p[2] > 100) {           // snow
      p[2] *= 0.0066667f;

      value = fBm(p, &terrain3_params);
      value = (float)(0.5 * (1+ fabs(value)));

      smultarray3(data->out, value);

      return 1;
   }

   if (p[2] > 80) {
      p[2] *= 0.01f;

      value = fBm(p, &terrain3_params);
      value = (float)fabs(value);

      data->out[0] *= 0.73f + 0.27f*value;
      data->out[1] *= 0.75f + 0.25f*value;
      data->out[2] *= 0.69f + 0.31f*value;
      return 1;
   }

   if (p[2] > 60) {
      p[2] *= 0.0125f;

      value = fBm(p, &terrain3_params);
      value = (float)fabs(value);

      data->out[0] *= 0.79f - 0.06f*value;
      data->out[1] *= 0.75f;
      data->out[2] *= 0.59f + 0.10f*value;
      return 1;
   }

   if (p[2] > 40) {
      p[2] *= 0.0166667f;

      value = fBm(p, &terrain3_params);
      value = (float)fabs(value);

      data->out[0] *= 0.70f + 0.09f*value;
      data->out[1] *= 0.61f + 0.14f*value;
      data->out[2] *= 0.36f + 0.23f*value;
      return 1;
   }

   if (p[2] > 20) {
      p[2] *= 0.025f;

      value = fBm(p, &terrain3_params);
      value = (float)fabs(value);

      data->out[0] *= 0.34f + 0.36f*value;
      data->out[1] *= 0.65f - 0.04f*value;
      data->out[2] *= 0.04f + 0.32f*value;
      return 1;
   }

   p[2] *= 0.05f;

   value = fBm(p, &terrain3_params);
   value = (float)fabs(value);

   data->out[0] *= 0.29f + 0.06f*value;
   data->out[1] *= 0.36f + 0.29f*value;
   data->out[2] = 0.04f*value;

   return 1;
}


#undef HH
#undef LACUNARITY
#undef OCTAVES


#define HH              0.632
#define LACUNARITY      2
#define OCTAVES         7


fractal_params cave_floor_params(OCTAVES, (float)HH, LACUNARITY);
static float terrain_shades[3][3] = {   1, 0, 0,
                                        0.628f, 0, 0.08f,
                                        1, 0, 0.236f};


/***************************************************************************
****************************************************************************/
void cave_floor_init () {

   cave_floor_params.init();
}


/***************************************************************************
****************************************************************************/
int cave_floor (void *x) {

   float value;

   float p[3];
   shaderparamtype *data;

   data = (shaderparamtype *)x;
                        // calc general color
   p[0] = data->in[0]*0.002f;
   p[1] = data->in[1]*0.002f;
   p[2] = data->in[2]*0.006666667f;                      // remove negative to get bluemoon.rgb

   p[2] *= 1 + 0.732743f*fBm(p, &cave_floor_params);

   value = fBm(p, &cave_floor_params);
   value = (float)fabs(value);
                                // red bands
   switch (((int)(value*6.0)) % 6)      {
      case 0:
         data->out[0] *= value*terrain_shades[0][0];
         data->out[1] *= value*terrain_shades[0][1];
         data->out[2] *= value*terrain_shades[0][2];
         return 1;

      case 2:
         data->out[0] *= value*terrain_shades[1][0];
         data->out[1] *= value*terrain_shades[1][1];
         data->out[2] *= value*terrain_shades[1][2];
         return 1;

      default:
         data->out[0] *= value*terrain_shades[2][0];
         data->out[1] *= value*terrain_shades[2][1];
         data->out[2] *= value*terrain_shades[2][2];
         return 1;
   }

}


#undef HH
#undef LACUNARITY
#undef OCTAVES


#define HH              0.632
#define LACUNARITY      2
#define OCTAVES         7


fractal_params cave_ceiling_params(OCTAVES, (float)HH, LACUNARITY);

/***************************************************************************
****************************************************************************/
void cave_ceiling_init () {

   cave_ceiling_params.init();
}


/***************************************************************************
****************************************************************************/
int cave_ceiling (void *x) {

   float value;

   float p[3];
   shaderparamtype *data;

   data = (shaderparamtype *)x;
                        // calc general color
   p[0] = data->in[0]*0.002f;
   p[1] = data->in[1]*0.002f;
   p[2] = data->in[2]*0.006666667f;                      // remove negative to get bluemoon.rgb

   p[2] *= 1 + 0.732743f*fBm(p, &cave_ceiling_params);

   value = fBm(p, &cave_ceiling_params);
   value = (float)fabs(value);

   switch (((int)(value*6.0)) % 6)      {
      case 0:
         value *= 0.5f;
         data->out[0] *= value;
         data->out[1] *= value;
         data->out[2] *= value*1.2f;
         break;

      case 2:
         value *= 0.5;
         data->out[0] *= value*1.1f;
         data->out[1] *= value;
         data->out[2] *= value*1.2f;
         break;

      default:
         value *= 0.5;
         smultarray3(data->out, value);
         break;
   }

   return 1;
}


#undef HH
#undef LACUNARITY
#undef OCTAVES


/***************************************************************************
****************************************************************************/
int planetclouds (void *x) {

   float omega = 0.7f;
   float lambda = 2;
   int octaves = 9;

   vector3f PP, temp;           /* Point after distortion */
   float o, a, b;               /* Loop control for fractal sum */

   shaderparamtype *data;

   data = (shaderparamtype *)x;

   copyarray3(PP, data->in);

   /* Add in "distortion" vector */
   a = noise3(PP);
   saddarray3(PP, a);

        /* Second cirrus: replace DNoise with vector fBm */

   /* Compute VLfBm */
   o = 1;  a = 0;
   for (; octaves > 0  &&  o >= CORRECT;  octaves--) {
      b = noise3(PP);
      temp[0] = PP[0] + b;
      temp[1] = PP[1] + b;
      temp[2] = PP[2] + b;

      a += o * noise3(temp);

      smultarray3(PP, lambda);
      o *= omega;
   }

   if (a < 0) {
      data->out[0] = data->out[1] = data->out[2] = 0;
      return 1;
   }

   a += 0.3f;
   if (a > 1)
      return 1;

   smultarray3(data->out, a);
   return 1;
}


/***************************************************************************
****************************************************************************/
int contour (void *x) {

   float omega = 0.7f;
   float lambda = 2;
   int octaves = 9;

  vector3f PP;                  /* Point after distortion */
  float o, a, b;                /* Loop control for fractal sum */

   shaderparamtype *data;

   data = (shaderparamtype *)x;

   copyarray3(PP, data->in);

   /* Add in "distortion" vector */
   a = noise3(PP);
   saddarray3(PP, a);
   b = noise3(PP);

        /* Second cirrus: replace DNoise with vector fBm */

   /* Compute VLfBm */
   o = 1;  a = 0;
   for (; octaves > 0  &&  o >= CORRECT;  octaves--) {
      a += o*noise1(b+noise1(b));
      b *= lambda;
      o *= omega;
   }

   if (a < 0) {
      data->out[0] = data->out[1] = data->out[2] = 0;
      return 1;
   }

   a += 0.3f;
   if (a > 1)
      return 1;

   smultarray3(data->out, a);
   return 1;
}


/*-------------------------------------------------------------------------
 *      carpet - creates a carpeted surface with variable nap and scuffing.
 *      Makes a reasonable attempt at anti-aliasing.
 *
 *      Kd, Ka - the usual meaning
 *      scuff - the "amount of scuffing": footprints, vacuum cleaner marks,
 *         etc. (the relative frequency of intensity variation)
 *      nap - the "shagginess" of the carpet (size of the speckles)
 *-------------------------------------------------------------------------*/
int carpet (void *x) {

   shaderparamtype *data;
   float    scuff = 1.0, nap = 1.0;
   vector3f PP;
   float    pixelsize, twice, scale, iscale, weight, turb = 0, napfreq,
            speckle, mottling;
   float    temp;
   vector3f tempv;
   float    horizon = 7.85f, ihorizon = 0.12738854f;

   data = (shaderparamtype *)x;

        /* large-scale mottling */
   if (scuff < CORRECT)
      mottling = 1.0f;
   else {

        /* get pixel size */
      temp = scuff * 0.5f;
      PP[0] = data->in[0]*temp; /* frequency adjustment (S-shaped curve) */
      PP[1] = data->in[1]*temp;
      PP[2] = data->in[2]*temp;

      pixelsize = temp/(float)fabs(data->pt[2]);
      twice = pixelsize + pixelsize;

        // create some turbulence  - similar to turbulence(), but need variables and goes backwards...
      copyarray3(tempv, PP);
      for (scale = iscale = 1; scale > twice; scale *= 0.5f, iscale += iscale) {
         temp = noise3(tempv);
         turb += (float)fabs(temp) * scale;
         addarray3(tempv, tempv);
      }

      // gradual fade out of highest frequency component near limit
      if (scale > pixelsize) {
         weight = (scale/pixelsize) - 1;
         weight = clamp(weight, 0, 1);

         temp = noise3(tempv);
         turb += (float)fabs(temp) * scale * weight;
      }

      // scuffing will be between 0.5 and 1

      turb = 0.5f*(turb + 1.0f);
      mottling = min(turb, 1);
   }

   pixelsize = (float)fabs(data->pt[2]);        // distance factor

   // small-scale speckling
   if (pixelsize > horizon + horizon)
      speckle = 0;
   else if (nap < CORRECT)
      speckle = 1;
   else {
      // get high-frequency speckles
      napfreq = 84.456f/nap;     // scale up speckle frequency

      tempv[0] = data->in[0]*napfreq;
      tempv[1] = data->in[1]*napfreq;
      tempv[2] = data->in[2]*napfreq;

      speckle = noise3(tempv);
      speckle = (float)fabs(speckle);

      // two-level speckles
      speckle = (speckle <= 0.25f) ? 0.2f : 1.0f;

      // smooth out speckles as they get smaller
      if (speckle<1-CORRECT) {
         temp = pixelsize/napfreq;
         temp += temp;
         speckle = (float)smoothstep(temp, speckle, 1);
      }

   }

   speckle = (float)sqrt(speckle);

   temp = pixelsize - horizon;

   if (temp <= 0)
      temp = speckle;
   else if (speckle <= mottling) {
      temp *= ihorizon;
      temp = (float)smoothstep(temp, speckle, mottling);
   }

   else {
      temp = 1-temp*ihorizon;
      temp = (float)smoothstep(temp, mottling, speckle);
   }

   smultarray3(data->out, temp);

   return 1;
}


/***************************************************************************
****************************************************************************/
int energy_beam(void *x) {

   float temp, temp2;
   int i, j;
   shaderparamtype *data;
   float value = 0;
   float y;

   data = (shaderparamtype *)x;

   if (data->in[0] < -56 || data->in[0] > 56 || data->in[1] < -256 || data->in[1] > 256) {
      data->out[0] = data->out[1] = data->out[2] = 0;
      return 0;
   }

   y = data->in[1] + 256;

   // create a base/trunk
   if (y < 1) {
      if (data->in[0] <= 3.3333f && data->in[0] >= -3.3333f)
         return 1;

      data->out[0] = data->out[1] = data->out[2] = 0;
      return 0;
   }

   temp = 31.41632f*data->frame;
// asdf possible opt ->this is mostlikely to be zero
  j = (int)noise1(temp);
 
   i = (int)(54.4563*(noise1(y*0.011904762f+temp)-j));

   if (data->in[0] >= i-CORRECT && data->in[0] <= i+CORRECT)
      return 1;

   temp2 = i - data->in[0];
   if (temp2 <= 3.3333f && temp2 >= -3.3333f)
      value += (float)(1 - 0.3*fabs(temp2));

   temp *= 0.125f;
   i = (int)(54.4563f*(noise1(y*0.010416667f+temp)-j));

   if (data->in[0] >= i-CORRECT && data->in[0] <= i+CORRECT)
      return 1;

   temp2 = i - data->in[0];
   if (temp2 <= 3.3333f && temp2 >= -3.3333f)
      value += (float)(1 - 0.3*fabs(temp2));

   temp *= 0.25f;
   i = (int)(54.4563f*(noise1(y*0.0078125f+temp)-j));

   if (data->in[0] >= i-0.5-CORRECT && data->in[0] <= i+0.5+CORRECT)
      return 1;

   temp2 = i - data->in[0];
   if (temp2 <= 4 && temp2 >= -4)
      value += (float)(1 - 0.25*fabs(temp2));

   temp *= 0.25f;
   i = (int)(54.4563f*(noise1(y*0.00390625f+temp)-j));

   if (data->in[0] >= i-1-CORRECT && data->in[0] <= i+1+CORRECT)
      return 1;

   temp2 =  i - data->in[0];
   if (temp2 <= 5 && temp2 >= -5)
      value += (float)(1 - 0.2*fabs(temp2));

   if (value > CORRECT) {
      if (value < 1) {
         value *= HALFPI;
	 value = SIN(value);
         smultarray3(data->out, value);
      }

      // potential bug (?) - if here, then return full color strength
      //                     (should be zero (?) )
      return 1;
   }

   data->out[0] = data->out[1] = data->out[2] = 0;
   return 0;
}

