

#include <string.h>

#ifdef WIN32
#include "WIN_W32.h"
#else
#include "WIN_X.h"
#endif

#include "polyops.h"


/* *****************************************************************************
***************************************************************************** */
optimizer::optimizer() {

   vertex_data = NULL;
   vertexcount = 0;
}


/* *****************************************************************************
***************************************************************************** */
optimizer::~optimizer() {

   if (vertex_data)
      delete [] vertex_data;
}


/* **************************************************************
************************************************************** */
void optimizer::build_optimized_vertexlist() {

   face_work_type *ftr;
   int *index;
   char *useflag;
   int i, counter;
   vertex_type *vtr;
   vbtree bhead;
   vbdata *vdata;
   vbnode *vnode;
   
   if (!vertexcount)
      return;
      
   index = new int[vertexcount];
   useflag = new char[(vertexcount+3) & 0xfffffffc];																				// eliminate similar verticies

   counter = vertexcount;
   vertexcount = 0;
   index[0] = 0;

   for (i=0; i<counter; i++) {
      vnode = new vbnode;
      vnode->key = vdata = new vbdata;
      copyarray3(vdata->v, vertex_data[i]);
      vdata->id = vertexcount;
      
      vnode = (vbnode *)bhead.insert(vnode);
      index[i] = ((vbdata *)vnode->key)->id;
      
      vertexcount += (index[i] == vertexcount);
   }
	
   bhead.build(vertex_data);
   bhead.dest();

   memset(useflag, 0, (vertexcount+3) & 0xfffffffc);
   
   for (ftr=(face_work_type *)final_face_data.head; ftr; ftr=(face_work_type *)ftr->next)
      if (ftr->vlist.head) {
         vtr = (vertex_type *)ftr->vlist.head;

         do {
            vtr->vertex_index = index[vtr->vertex_index];
            useflag[vtr->vertex_index] = 1;
            vtr = (vertex_type *)vtr->next;
         } while (vtr != (vertex_type *)ftr->vlist.head);

      }

   // eliminate non-used verticies
   for (i=counter=0; i<vertexcount; i++)
      if (useflag[i]) {
         index[i] = counter;

         if (counter != i)
            copyarray3(vertex_data[counter], vertex_data[i]);
	 
         counter++;
      }

   // build new index list
   for (ftr=(face_work_type *)final_face_data.head; ftr; ftr=(face_work_type *)ftr->next)
      if (ftr->vlist.head) {
         vtr = (vertex_type *)ftr->vlist.head;
         do {
            vtr->vertex_index = index[vtr->vertex_index];
            vtr=(vertex_type *)vtr->next;
         } while (vtr != (vertex_type *)ftr->vlist.head);

      }

   vertexcount = counter;

   delete [] index;
   delete [] useflag;
}


/* ******************************************************************************************
   this will calculate the "height" ray of a triangle (as in height/base...)
****************************************************************************************** */
void optimizer::calc_height_split(float *res, float *a, float *b, float *c) {

   res[0] = b[0] - 0.5f*(c[0] + a[0]);
   res[1] = b[1] - 0.5f*(c[1] + a[1]);
   res[2] = b[2] - 0.5f*(c[2] + a[2]);

   normalize3(res);
}


/* ******************************************************************************************
		this will bisect the angle caused by ba and bc
****************************************************************************************** */
void optimizer::calc_bisect_split(float *res, float *a, float *b, float *c) {

   vector3f v1, v2;

   subeqarray3(v1, a, b);
   subeqarray3(v2, c, b);
   normalize3(v1);
   normalize3(v2);

   res[0] = -0.5f * (v1[0] + v2[0]);
   res[1] = -0.5f * (v1[1] + v2[1]);
   res[2] = -0.5f * (v1[2] + v2[2]);
}


/* ******************************************************************************************
   Note: ray is unit vector
****************************************************************************************** */
int optimizer::ray_intersect_edge(float *start, float *ray, float *pt1, float *pt2, float *intersect) {

   float l, lray2, lvray;
   vector3f v, ray2;

   subeqarray3(ray2, pt2, pt1);

   lray2 = dotproduct3(ray2, ray2);
    
   if (lray2 < CORRECT)
      return query_point_line_intersect(pt1, start, ray);
   
   if (!line3d_line3d_intersect(start, ray, pt1, ray2, intersect))
      return 0;
      
   subeqarray3(v, intersect, pt1);
   l = dotproduct3(v, v);

   if (l < CORRECT) {
      subeqarray3(v, intersect, pt2);
      l = dotproduct3(v, v);
   }
   
   lvray = dotproduct3(v, ray);
   
   // want only 1 intersection
//   if (lvray*lvray > COS_FIVE_DEG*COS_FIVE_DEG*dotproduct3(ray, ray)*l)
   if (lvray*lvray > COS_FIVE_DEG*COS_FIVE_DEG*l)
      return 0;
      
   lvray = dotproduct3(v, ray2);

   return (l <= lray2 && lvray*lvray > COS_FIVE_DEG*COS_FIVE_DEG*lray2*l);
}


/* **************************************************************
	returns true if the four uv coords are linear and "adjacent" and in the opposite direction
	vector2f p1, p2, t1, t2, test - uv coords
        plength, tlength - length of edge in 3D geometry space
************************************************************** */
int optimizer::query_mesh_uv(float *p1, float *p2, float plength, float *t1, float *t2, float tlength, float *test) {

   vector2f puv, tuv;
   float puvlength, tuvlength;

   subeqarray2(puv, p2, p1);
   puvlength = (float)magnitude2(puv);

   subeqarray2(tuv, t2, t1);
   tuvlength = (float)magnitude2(tuv);

   // check if vectors are in the opposite direction
   if (dotproduct2(puv, tuv) > -COS_FIVE_DEG*tuvlength*puvlength)
      return 0;

   // check to see if edges are really points
   if (puvlength < CORRECT) {
      if (tuvlength < CORRECT) {
         subeqarray2(puv, p2, t2);
         return (dotproduct2(puv, puv) < CORRECT);
      }

      return 0;
   }

   if (tuvlength < CORRECT)
      return 0;

   // check to see if there is a non-linear scale between UV and 3D space
   if (fabs(puvlength*tlength - tuvlength*plength) > CORRECT)
      return 0;

   // check if the "test" coord is on the "p" edge
   subeqarray2(tuv, test, p1);

   tuvlength = (float)magnitude2(tuv);

   tuv[0] = test[0] - (puv[0] * tuvlength + p1[0]);
   tuv[1] = test[1] - (puv[1] * tuvlength + p1[1]);

   return (dotproduct2(tuv, tuv) < CORRECT);
}


/* **************************************************************

  Rules: in order for "co-edges", at least one vertice of one must be on
		the edge of the other


		fail
			return 0

		2 edges "share" vertices and are in OPPOSITE order (otherwise they overlap)
			return 1

		t2 and p1 are "shared", and p2 lies between t1 and t2
			return 2

		t2 and p1 are "shared", and t1 lies between p1 and p2
			return 3

		t1 and p2 are "shared", and p1 lies between t1 and t2
			return 4

		t1 and p2 are "shared", and t2 lies between p1 and p2
			return 5

		if t is a subset of p
			return 6

		if p is a subset of t
			return 7

************************************************************** */
int optimizer::query_subset_edge(vertex_type *p1, vertex_type *p2, vertex_type *t1, vertex_type *t2, int textureflag, float *p) {

   vector3f pdirection, tdirection, temp_vector;
   float plength, tlength, temp_length;

   subeqarray3(pdirection, vertex_data[p2->vertex_index], vertex_data[p1->vertex_index]);
   plength = normalize3(pdirection);

   subeqarray3(tdirection, vertex_data[t2->vertex_index], vertex_data[t1->vertex_index]);
   tlength = normalize3(tdirection);

   *p = -dotproduct3(pdirection, tdirection);

   // edges are not collinear
   if (*p < COS_FIVE_DEG)
      return EC_FAIL;
 
   // one shared vertex
   if (p1->vertex_index == t2->vertex_index)	{

      // 2 shared vertices
      if (p2->vertex_index == t1->vertex_index) {
         *p = 1.0f;

         if (textureflag)
            return ( (similar2(p2->uv, t1->uv) && similar2(p1->uv, t2->uv)) ? EC_TWO_VERTICES : EC_FAIL);
   
         return EC_TWO_VERTICES;
      }

      // Note : if p2 on (t1,t2) , |pt2-t1| MUST be < 1.0 so check to see if vector is in the same as tdirection
      subeqarray3(temp_vector, vertex_data[p2->vertex_index], vertex_data[t1->vertex_index]);

      // same direction
      if (dotproduct3(temp_vector, tdirection) > COS_FIVE_DEG*magnitude3(temp_vector) ) {

         if (!textureflag)
            return EC_T2P1_P2T;

         if (!similar2(p1->uv, t2->uv))
            return EC_FAIL;

         if (query_mesh_uv(p1->uv, p2->uv, plength, t1->uv, t2->uv, tlength, t1->uv))
            return EC_T2P1_P2T;
      }

      // means t1 on (p1, p2)
      else {
         if (!textureflag)
            return EC_T2P1_T1P;

         if (!similar2(p1->uv, t2->uv))
            return EC_FAIL;

         if (query_mesh_uv(p1->uv, p2->uv, plength, t1->uv, t2->uv, tlength, t1->uv))
            return EC_T2P1_T1P;
      }

      return EC_FAIL;
   }

   // one shared vertex
   if (p2->vertex_index == t1->vertex_index) {

      // Note : if p1 on (t1,t2) , |pt1-t1| MUST be > 0.0 so check to see if |t1-p1| < tlength
      subeqarray3(temp_vector, vertex_data[p1->vertex_index], vertex_data[t1->vertex_index]);

      if (dotproduct3(temp_vector, temp_vector) <= tlength*tlength) {
         if (!textureflag)
            return EC_T1P2_P1T;

         if (!similar2(p2->uv, t1->uv))
            return EC_FAIL;

         if (query_mesh_uv(t1->uv, t2->uv, tlength, p1->uv, p2->uv, plength, p1->uv))
            return EC_T1P2_P1T;
      }

      // t2 on p
      else {
         if (!textureflag)
            return EC_T1P2_T2P;

         if (!similar2(p2->uv, t1->uv))
            return EC_FAIL;

         if (query_mesh_uv(t1->uv, t2->uv, tlength, p1->uv, p2->uv, plength, p1->uv))
            return EC_T1P2_T2P;
      }

      return EC_FAIL;
   }

   // last chance - find out if one edge intersects another (they are parallel)

   // t1 in p ?
   subeqarray3(temp_vector, vertex_data[t1->vertex_index], vertex_data[p1->vertex_index]);
   temp_length = (float)magnitude3(temp_vector);

   if (temp_length < plength && dotproduct3(temp_vector, pdirection) > COS_FIVE_DEG*temp_length) {
      temp_vector[0] = vertex_data[t1->vertex_index][0] - (vertex_data[p1->vertex_index][0] + temp_length * pdirection[0]);
      temp_vector[1] = vertex_data[t1->vertex_index][1] - (vertex_data[p1->vertex_index][1] + temp_length * pdirection[1]);
      temp_vector[2] = vertex_data[t1->vertex_index][2] - (vertex_data[p1->vertex_index][2] + temp_length * pdirection[2]);

      if (dotproduct3(temp_vector, temp_vector) < CORRECT)
         return (!textureflag || query_mesh_uv (p1->uv, p2->uv, plength, t1->uv, t2->uv, tlength, t1->uv)) ? EC_T_IN_P :EC_FAIL;
   }
   
   // t2 in p ?
   subeqarray3(temp_vector, vertex_data[t2->vertex_index], vertex_data[p1->vertex_index]);
   temp_length = (float)magnitude3(temp_vector);

   if (temp_length < plength  && dotproduct3(temp_vector, pdirection) > COS_FIVE_DEG*temp_length) {
      temp_vector[0] = vertex_data[t2->vertex_index][0] - (vertex_data[p1->vertex_index][0] + temp_length * pdirection[0]);
      temp_vector[1] = vertex_data[t2->vertex_index][1] - (vertex_data[p1->vertex_index][1] + temp_length * pdirection[1]);
      temp_vector[2] = vertex_data[t2->vertex_index][2] - (vertex_data[p1->vertex_index][2] + temp_length * pdirection[2]);

      if (dotproduct3(temp_vector, temp_vector) < CORRECT)
         return  (!textureflag || query_mesh_uv (p1->uv, p2->uv, plength, t1->uv, t2->uv, tlength, t2->uv)) ? EC_T_IN_P : EC_FAIL;
   }

   // p1 in t ?
   subeqarray3(temp_vector, vertex_data[p1->vertex_index], vertex_data[t1->vertex_index]);
   temp_length = (float)magnitude3(temp_vector);

   if (temp_length < tlength  && dotproduct3(temp_vector, tdirection) > COS_FIVE_DEG*temp_length) {
		
      temp_vector[0] = vertex_data[p1->vertex_index][0] - (vertex_data[t1->vertex_index][0] + temp_length * tdirection[0]);
      temp_vector[1] = vertex_data[p1->vertex_index][1] - (vertex_data[t1->vertex_index][1] + temp_length * tdirection[1]);
      temp_vector[2] = vertex_data[p1->vertex_index][2] - (vertex_data[t1->vertex_index][2] + temp_length * tdirection[2]);

      if (dotproduct3(temp_vector, temp_vector) < CORRECT)
         return (!textureflag || query_mesh_uv (t1->uv, t2->uv, tlength, p1->uv, p2->uv, plength, p1->uv)) ? EC_P_IN_T : EC_FAIL;
   }

   // p1 in t ?
   subeqarray3(temp_vector, vertex_data[p2->vertex_index], vertex_data[t1->vertex_index]);
   temp_length = (float)magnitude3(temp_vector);

   if (temp_length < tlength  && dotproduct3(temp_vector, tdirection) > COS_FIVE_DEG*temp_length) {
      temp_vector[0] = vertex_data[p2->vertex_index][0] - (vertex_data[t1->vertex_index][0] + temp_length * tdirection[0]);
      temp_vector[1] = vertex_data[p2->vertex_index][1] - (vertex_data[t1->vertex_index][1] + temp_length * tdirection[1]);
      temp_vector[2] = vertex_data[p2->vertex_index][2] - (vertex_data[t1->vertex_index][2] + temp_length * tdirection[2]);
		
      if (dotproduct3(temp_vector, temp_vector) < CORRECT)
         return (!textureflag || query_mesh_uv (t1->uv, t2->uv, tlength, p1->uv, p2->uv, plength, p1->uv)) ? EC_P_IN_T :EC_FAIL;
   }
	
   return EC_FAIL;
}


/* ******************************************************************************************
   splits up face between 2 concave verticies
****************************************************************************************** */
int optimizer::recurse_split_face(face_work_type *new_face, dbl_llist_manager *solution) {

   vertex_type	*ttr;
   face_work_type *ftr;

   if (solution->count == 1)
      return 1;

   ttr = (vertex_type *)new_face->vlist.head;

   do {
      if (ttr->concaveflag)
         return split_face(new_face, solution);

      ttr = (vertex_type *)ttr->next;
   } while (ttr != (vertex_type *)new_face->vlist.head);

   solution->dest();

   solution->insert(ftr = new face_work_type, NULL);

   copyarray4(ftr->face_normal, new_face->face_normal);
   ftr->material_index = new_face->material_index;
   ftr->textureflag = new_face->textureflag;
   new_face->vlist.clone_list(&ftr->vlist);

   return 1;
}


/* ******************************************************************************************
	splits up face between 1 concave and 1 normal vertex
****************************************************************************************** */
int optimizer::recurse_crack_face(face_work_type *new_face, dbl_llist_manager *solution) {

   vertex_type	*ttr;
   face_work_type *ftr;

   if (solution->count == 1)
      return 1;

   ttr = (vertex_type *)new_face->vlist.head;

   do {
      if (ttr->concaveflag)
         return crack_face(new_face, solution);

      ttr = (vertex_type *)ttr->next;
   } while (ttr != (vertex_type *)new_face->vlist.head);

   solution->dest();

   solution->insert(ftr = new face_work_type, NULL);

   copyarray4(ftr->face_normal, new_face->face_normal);
   ftr->material_index = new_face->material_index;
   ftr->textureflag = new_face->textureflag;
   new_face->vlist.clone_list(&ftr->vlist);

   return 1;
}


/* ******************************************************************************************
  any concave verticies left will be "cracked" by adding a vertex on the adjacent side, and splitting there

  Warning: this function can increase the # of verticies in the "vertex_data" list -> minor possibility
  will run out of entries 
****************************************************************************************** */
int optimizer::break_face(face_work_type *face, dbl_llist_manager *solution) {

   face_work_type *crackface, *subface;
   vertex_type *vtr, *wtr;
   int flag = 0;
   dbl_llist_manager subman, workman;

   if (solution->count == 1)
      return 1;
   
   if (!face->vlist.head)
      return 0;
   
   crackface = new face_work_type;
   copyarray4(crackface->face_normal, face->face_normal);
   crackface->material_index = face->material_index;
   crackface->textureflag = face->textureflag;
   face->vlist.clone_list(&crackface->vlist);

   vtr = (vertex_type *)crackface->vlist.head;

   do {
      if (vtr->concaveflag) {
         subface = crackface->make_subface((vertex_type *)vtr->back->back, vtr, vertex_data, OVERRIDE);
         if (subface)
            workman.insert(subface, NULL);
         else
            mbprintf("NVTZ - Loosing some back geometry...\n");	    	       

         subface = crackface->make_subface(vtr, (vertex_type *)vtr->next->next, vertex_data, OVERRIDE);
         if (subface)
            workman.insert(subface, NULL);
         else
            mbprintf("NVTZ - Loosing some next geometry...\n");	    	       

         wtr = (vertex_type *)vtr->back;
         crackface->vlist.remove(wtr);
         delete wtr;
	     
         wtr = (vertex_type *)vtr->next;
         crackface->vlist.remove(wtr);
         delete wtr;

         crackface->post_smash(vertex_data, 1);

         if (!crackface->vlist.validate(vertex_data)) {
            delete crackface;
            return 1;
         }

         wtr = (vertex_type *)crackface->vlist.head;
         do {
            if (wtr == vtr)
               break;

            wtr = (vertex_type *)wtr->next;
         } while (wtr != (vertex_type *)crackface->vlist.head);

         if (wtr == vtr) {
            calculate_normal(vertex_data[vtr->vertex_index], vertex_data[((vertex_type *)vtr->back)->vertex_index], vertex_data[((vertex_type *)vtr->back->back)->vertex_index], ((vertex_type *)vtr->back)->normal);
            ((vertex_type *)vtr->back)->concaveflag = (dotproduct3(((vertex_type *)vtr->back)->normal, crackface->face_normal) < CORRECT);

            calculate_normal(vertex_data[((vertex_type *)vtr->next)->vertex_index], vertex_data[vtr->vertex_index], vertex_data[((vertex_type *)vtr->back)->vertex_index], vtr->normal);
            vtr->concaveflag = (dotproduct3(vtr->normal, crackface->face_normal) < CORRECT);

            calculate_normal(vertex_data[((vertex_type *)vtr->next->next)->vertex_index], vertex_data[((vertex_type *)vtr->next)->vertex_index], vertex_data[vtr->vertex_index], ((vertex_type *)vtr->next)->normal);
            ((vertex_type *)vtr->next)->concaveflag = (dotproduct3(((vertex_type *)vtr->next)->normal, crackface->face_normal) < CORRECT);
         }

         else {
            crackface->vlist.calc_vnormal(vertex_data);
            crackface->calc_concave();
         }

         break;
      }

      vtr = (vertex_type *)vtr->next;
   } while (vtr != (vertex_type *)crackface->vlist.head);

   vtr = (vertex_type *)crackface->vlist.head;

   do {
      flag += vtr->concaveflag;
      vtr = (vertex_type *)vtr->next;
   } while (vtr != (vertex_type *)crackface->vlist.head);

   if (flag) {

      if (flag & 0x01)
         crack_face(crackface, &subman);
      else
         split_face(crackface, &subman);

      if (subman.head) {
         while (subman.head) {
            subman.remove(face = (face_work_type *)subman.head);
            workman.insert(face, NULL);
         }

      }
      
      else
         mbprintf("Lost data on final break...");
	 
      delete crackface;
   }

   else
      workman.insert(crackface, NULL);
   
   if (!solution->count || workman.count < solution->count) {
      solution->dest();
      while (workman.head) {
         workman.remove(face = (face_work_type *)workman.head);
         solution->insert(face, NULL);
      }

   }

   return 1;
}


/* ******************************************************************************************
****************************************************************************************** */
int optimizer::crack_face(face_work_type *face, dbl_llist_manager *solution) {

   vertex_type *vtr, *wtr;
   face_work_type *vface, *wface, *ftr;
   dbl_llist_manager s1, s2, workman;
   int concaveflag;

   if (solution->count == 1)
      return 1;

   if (!face->vlist.head)
      return 0;

   concaveflag = 0;
   vtr = (vertex_type *)face->vlist.head;

   do {
      if (vtr->concaveflag) {
         wtr = (vertex_type *)vtr->next->next;
         concaveflag = 1;

         do {
            if (!wtr->concaveflag && face->query_internal_diagonal(vertex_data, vtr, wtr)) {

               s1.dest();
               s2.dest();
               vface = wface = NULL;

               vface = face->make_subface(vtr, wtr, vertex_data, CHECK_NORMAL);
               if (vface) {
                  wface = face->make_subface(wtr, vtr, vertex_data, CHECK_NORMAL);
                  if (wface &&
                      recurse_crack_face(vface, &s1) &&
                      (!workman.count || s1.count < workman.count) &&
                      recurse_crack_face(wface, &s2) &&
                      (!workman.count || s1.count + s2.count < workman.count) &&
                      (!solution->count || s1.count+s2.count < solution->count)) {

                      // new solution better than last, get rid of last
                      workman.dest();

                      while (s1.head) {
                         s1.remove(ftr = (face_work_type *)s1.head);
                         workman.insert(ftr, NULL);
                      }

                      while (s2.head) {
                         s2.remove(ftr = (face_work_type *)s2.head);
                         workman.insert(ftr, NULL);
                      }

//asdf
delete vface;
delete wface;
break;
                  }

               }

               if (vface)
                  delete vface;

               if (wface)
                  delete wface;
            }

            wtr = (vertex_type *)wtr->next;
         } while (wtr != (vertex_type *)vtr->back);

      }

//asdf
if (workman.count > 0)
break;

      vtr = (vertex_type *)vtr->next;
   } while (vtr != (vertex_type *)face->vlist.head);

   if (!concaveflag) {
      workman.insert(ftr = new face_work_type, NULL);

      copyarray4(ftr->face_normal, face->face_normal);
      ftr->material_index = face->material_index;
      ftr->textureflag = face->textureflag;

      face->vlist.clone_list(&ftr->vlist);
   }

   // found a split above
   if (workman.head) {

      solution->dest();

      while (workman.head) {
         workman.remove(ftr = (face_work_type *)workman.head);
         solution->insert(ftr, NULL);
      }

      return 1;
   }

   return break_face(face, solution);
}


/* ******************************************************************************************

	First try to split between 2 "concave" verticies using internal diagonals.

	Second try to split between 1 concave and 1 normal vertex

	Last just split down the middle....

****************************************************************************************** */
int optimizer::split_face(face_work_type *face, dbl_llist_manager *solution) {

   vertex_type *vtr, *wtr;
   face_work_type *vface, *wface, *ftr;
   dbl_llist_manager s1, s2, workman;

   int concaveflag;        // flag for existance of concave pairs

   if (solution->count == 1)
      return 1;

   if (!face->vlist.head)
      return 0;

   concaveflag = 0;
   vtr = (vertex_type *)face->vlist.head;																					// find optimal concave pairs

   do {

      if (vtr->concaveflag) {
         concaveflag = 1;

         for (wtr = (vertex_type *)vtr->next; wtr != face->vlist.head; wtr = (vertex_type *)wtr->next)
            // dont check adj verticies
            if (wtr->concaveflag && vtr->next != wtr && wtr->next != vtr) {

               // split face, process first half
               if (face->query_internal_diagonal (vertex_data, vtr, wtr)) {

                  s1.dest();
                  s2.dest();
                  vface = wface = NULL;

                  vface = face->make_subface(vtr, wtr, vertex_data, CHECK_NORMAL);
                  if (vface) {

                     wface = face->make_subface(wtr, vtr, vertex_data, CHECK_NORMAL);
                     if (wface &&
                         recurse_split_face(vface, &s1) &&
                         (!workman.count || s1.count < workman.count) &&
                         recurse_split_face(wface, &s2) &&
                         (!workman.count || s1.count + s2.count < workman.count) &&
                         (!solution->count || s1.count+s2.count < solution->count)) {

                        // new solution better than last, get rid of last
                        workman.dest();

                        while (s1.head) {
                           s1.remove(ftr = (face_work_type *)s1.head);
                           workman.insert(ftr, NULL);
                        }

                        while (s2.head) {
                           s2.remove(ftr = (face_work_type *)s2.head);
                           workman.insert(ftr, NULL);
                        }

//asdf
delete vface;
delete wface;
break;
                     }

                  }

                  if (vface)
                     delete vface;

                  if (wface)
                     delete wface;
               }

            }

      }

//asdf
if (workman.count > 0)
break;

      vtr = (vertex_type *)vtr->next;
   } while (vtr != (vertex_type *)face->vlist.head);

   if (!concaveflag) {
      workman.insert(ftr = new face_work_type, NULL);

      copyarray4(ftr->face_normal, face->face_normal);
      ftr->material_index = face->material_index;
      ftr->textureflag = face->textureflag;

      face->vlist.clone_list(&ftr->vlist);
   }

   // found a split above
   if (workman.head) {
      
      solution->dest();

      while (workman.head) {
         workman.remove(ftr = (face_work_type *)workman.head);
         solution->insert(ftr, NULL);
      }

      return 1;
   }

   // no concave pairs -> start splitting each concave vertex -> a normal vertex, and look for optimum split
   // if we are here, then just subdivide like crazy -> any split will do (same for subsplits...)
   return crack_face(face, solution);
}


/***************************************************************
************************************************************** */
void optimizer::query_combine(face_work_type *face1, face_work_type *face2, dbl_llist_manager *out) {

   vertex_type *vtr, *wtr;	
   int result;
   float p;

   if (!face1->vlist.head || !face2->vlist.head)			// bad face(s)
      return;

   vtr = (vertex_type *)face1->vlist.head;

   do {
      wtr = (vertex_type *)face2->vlist.head;
      do {
         result = query_subset_edge(vtr, (vertex_type *)vtr->next, wtr, (vertex_type *)wtr->next, face1->textureflag, &p);

         if (result != EC_FAIL)
            out->append(new edge_match_type(vtr, (vertex_type *)vtr->next, wtr, (vertex_type *)wtr->next, result, p), NULL);

         wtr = (vertex_type *)wtr->next;
      } while(wtr != (vertex_type *)face2->vlist.head);

      vtr = (vertex_type *)vtr->next;
   } while (vtr != (vertex_type *)face1->vlist.head);

}


/* ******************************************************************************************
****************************************************************************************** */
int optimizer::remove_edge(int matchcount, edge_match_list *emlpresident, dbl_llist_manager *flist, face_work_type *ftr, vector3f *vertex_data, int *overlap_count) {

   edge_match_type *match, *ntr, *nntr;
   vertex_type *vtr;
   int i;

   match = (edge_match_type *)emlpresident->stats.head;


   if (matchcount == 1) {	// only 1 match :)
      flist->remove(emlpresident->candidate);

      join_face(ftr, match->v11, match->v12, emlpresident->candidate, match->v21, match->v22);

      delete emlpresident->candidate;
      return 1;
   }

   // overlap...
   if (ftr->vlist.count == 3) {
      overlap_count++;

      flist->remove(emlpresident->candidate);
      delete emlpresident->candidate;

      return 1;
   }

   ntr = (edge_match_type *)match->next;
   nntr = (edge_match_type *)ntr->next;
   
   // remove shared vertex
   if (matchcount == 2) {
      if (match->v12 == ntr->v11)
         vtr = match->v12;
      else if (match->v11 == ntr->v12)
         vtr = match->v11;
      else											// cork effect
         return 0;

      ftr->vlist.remove(vtr);
      delete vtr;

      flist->remove(emlpresident->candidate);
      delete emlpresident->candidate;

      return 1;
   }

   // cork -> remove all 3 verticies
   i = (match->v11 != ntr->v12 && match->v11 != nntr->v12) +
       (ntr->v11 != match->v12 && ntr->v11 != nntr->v12) +
       (nntr->v11 != match->v12 && nntr->v11 != ntr->v12);

   if (i != 1)
      return 0;

   ftr->vlist.remove(match->v11);
   delete match->v11;

   ftr->vlist.remove(ntr->v11);
   delete ntr->v11;

   ftr->vlist.remove(nntr->v11);
   delete nntr->v11;

   flist->remove(emlpresident->candidate);
   delete emlpresident->candidate;

   return 1;
}


/* ******************************************************************************************
****************************************************************************************** */
int optimizer::smash_edge(edge_match_list *emlpresident, dbl_llist_manager *flist, face_work_type *ftr, vector3f *vertex_data, int *overlap_count) {

   edge_match_type *match, *mtr, *ntr;

   int matchcount;

   match = (edge_match_type *)emlpresident->stats.head;
   matchcount = emlpresident->stats.count;

   // only 1 match :)
   if (matchcount == 1) {
      flist->remove(emlpresident->candidate);

      join_face(ftr, match->v11, match->v12, emlpresident->candidate, match->v21, match->v22);
      ftr->post_smash(vertex_data, 1);

      delete emlpresident->candidate;

      return 1;
   }

   // overlap...
   if (ftr->vlist.count == 3) {

      overlap_count++;

      flist->remove(emlpresident->candidate);
      delete emlpresident->candidate;

      return 1;
   }

   mtr = (edge_match_type *)emlpresident->stats.tail;
   for (ntr = match; ntr && !(ntr->v11 == mtr->v12 && ntr->v22 == mtr->v21 && ntr->v11->vsimilar(mtr->v21, ftr->textureflag)); mtr = ntr, ntr = (edge_match_type *)ntr->next); 

   if (!ntr)
      return 0;
					
   flist->remove(emlpresident->candidate);

   join_face(ftr, mtr->v11, ntr->v12, emlpresident->candidate, ntr->v21, mtr->v22);
   ftr->post_smash(vertex_data, 1);

   delete emlpresident->candidate;
							
   return 1;		
}


/***************************************************************
  note: only 2 triangles in "group" so ftr->next == NULL!
************************************************************** */
int optimizer::combine_2_triangles(group_type *group, int *overlap_count) {

   face_work_type *etr, *ftr;
   dbl_llist_manager match;
   edge_match_type *mtr;
   
   group->flist.remove(etr = (face_work_type *)group->flist.head);
   group->flist.remove(ftr = (face_work_type *)group->flist.head);

   if (!etr->vlist.head) {
      delete etr;
      if (ftr->vlist.head)
         final_face_data.insert(ftr, NULL);
      else
         delete ftr;
      return 1;
   }

   else if (!ftr->vlist.head) {
      delete ftr;
      final_face_data.insert(etr, NULL);
      return 1;
   }

   if (!etr->vlist.head || !ftr->vlist.head)	
      return 0;

   query_combine(etr, ftr, &match);

   if (match.count) {
      // multiple edges "connects" for 2 triangles results in overlap; ftr is chosen for deletion...
      if (match.count > 1) {					
         *overlap_count += 1;
         final_face_data.insert(etr, NULL);
         delete ftr;	
         return 1;
      }

      mtr = (edge_match_type *)match.head;
   
      if (mtr->resultflag == EC_TWO_VERTICES) {
         join_face(etr, mtr->v11, mtr->v12, ftr, mtr->v21, mtr->v22);
         etr->post_smash(vertex_data, 1);

         if (etr->vlist.count)
            final_face_data.insert(etr, NULL);
         else
            delete etr;

         delete ftr;
         return 1;
      }

   }

   // no valid combinations...
   final_face_data.insert(etr, NULL);
   final_face_data.insert(ftr, NULL);

   return 1;
}



/* **************************************************************
	This function combines a polygon and a triangle faces that are coplanar and
	share an edge.

  Assumptions:  
				- polygon/triangle are coplanar and have the same direction normal
				- vertex list contains no "similar" vertices
				- if vertices between "polybegin" and "polyend", similar vertices must exist between
						"tribegin" and "triend" and vice-versa		
		  
				- Polygon and Triangle dont overlap area -> just edges/partial edges
				- Polygon and Triangle are not "lines" or "points"
				- Polygon has no "holes"
				- Will not be combined if new triangle "makes" a hole
					ie it can only share 2 edges at most, and if it does, removes the middle vertex

************************************************************** */
void optimizer::join_face(face_work_type *poly, vertex_type *polystart, vertex_type *polyend, face_work_type *triangle, vertex_type *tristart, vertex_type *triend) {

   vertex_type *ptr;
   int flag;
	
   // remove "inbetween" vertices
   while ((vertex_type *)polystart->next != polyend) {
      ptr = (vertex_type *)polystart->next;
      poly->vlist.remove(ptr);
      delete ptr;
   }

   // remove "inbetween" vertices
   while (tristart->next != triend) {
      ptr = (vertex_type *)tristart->next;
      triangle->vlist.remove(ptr);
      delete ptr;
   }

   // reorder triangle : start->end
   poly->vlist.make_head(polyend);

   // reorder triangle : end -> start
   triangle->vlist.make_head(triend);

   while (triangle->vlist.head) {
      ptr = (vertex_type *)triangle->vlist.head;
      triangle->vlist.remove(ptr);
      poly->vlist.insert(ptr, polyend);
   }

   do {
      flag = 0;
      ptr = (vertex_type *)poly->vlist.head;

      if (!ptr)
         break;
	 
      do {
         if (ptr->vertex_index == ((vertex_type *)ptr->next)->vertex_index) {
            poly->vlist.remove(ptr);
            delete ptr;
            flag = 1;
            break;
         }

         ptr = (vertex_type *)ptr->next;
      } while (ptr != (vertex_type *)poly->vlist.head);

   } while (flag);

}


/* ******************************************************************************************
****************************************************************************************** */
int optimizer::split_weak_link(face_work_type *face, dbl_llist_manager *fman) {

   vertex_type *ptr, *qtr;
   face_work_type *ftr, *gtr;

   if (face->vlist.count < 4)
      return 0;

   ptr = (vertex_type *)face->vlist.head;

   do {
      for (qtr = (vertex_type *)ptr->next->next; qtr != (vertex_type *)face->vlist.head->back; qtr = (vertex_type *)qtr->next)
         if (ptr->vsimilar(qtr, face->textureflag)) {
            ftr = face->make_subface(ptr, qtr, vertex_data, OVERRIDE);
            if (ftr) {
               gtr = face->make_subface(qtr, ptr, vertex_data, OVERRIDE);
               if (gtr) {
                  fman->insert(gtr, NULL);
                  fman->insert(ftr, NULL);
                  return 1;
               }

               delete ftr;
            }

            return 0;
         }

      ptr = (vertex_type *)ptr->next;
   } while (ptr->next->next != face->vlist.head->back);

   return 0;
}


/* ******************************************************************************************
****************************************************************************************** */
int optimizer::decimate(face_work_type *face) {

   dbl_llist_manager solution;
   face_work_type *ftr;

   if (!split_face(face, &solution))
      return 0;

   if (!solution.head)
      return 0;

   while (solution.head) {
      solution.remove(ftr = (face_work_type *)solution.head);
      final_face_data.insert(ftr, NULL);
   }

   return 1;
}


/* *****************************************************************************
***************************************************************************** */
void optimizer::complex(dbl_llist_manager *groupman, int *overlap_count) {

   int i, j, k;

   group_type *gtr;
   face_work_type *etr, *ftr;
   dbl_llist_manager new_face_man;

   int matchcount;
   dbl_llist_manager match;
   edge_match_type *mtr;
   edge_match_list *emlptr, *emlpresident;
   dbl_llist_manager emlhead;
   
   // for each group of planar faces
   while (groupman->head) {
   
      groupman->remove(gtr = (group_type *)groupman->head);

      // for each face
      while (gtr->flist.head) {

         gtr->flist.remove(ftr = (face_work_type *)gtr->flist.head);

         do {
            for (etr = (face_work_type *)gtr->flist.head; etr; etr=(face_work_type *)etr->next) {
               query_combine(ftr, etr, &match);

               if (match.count)
                  emlhead.insert(new edge_match_list(&match, etr), NULL);
            }

            i = 0;

            while (emlhead.head) {

               emlpresident = NULL;
	       
               // find matches w/ shared edges
               for (matchcount = 0, emlptr=(edge_match_list *)emlhead.head; emlptr; emlptr=(edge_match_list *)emlptr->next) {
                  for (j=0,mtr=(edge_match_type *)emlptr->stats.head, k =0; mtr; mtr=(edge_match_type *)mtr->next, j++)
                     k += (mtr->resultflag == EC_TWO_VERTICES);

                  if (j == k && k > matchcount) {
                     emlpresident = emlptr;
                     matchcount = k;
                  }

               }

               if (emlpresident) {
                  emlhead.remove(emlpresident);

                  i = remove_edge(matchcount, emlpresident, &gtr->flist, ftr, vertex_data, overlap_count);

                  delete emlpresident;
		     
                  if (i) {
                     emlhead.dest();
                     break;
                  }

               }

               else {
                  ftr->post_smash(vertex_data, 1);
                  emlhead.dest();
                  break;
               }

            }

            if (i)
               continue;

            for (etr = (face_work_type *)gtr->flist.head; etr; etr=(face_work_type *)etr->next) {
               query_combine(ftr, etr, &match);

               if (match.count)
                  emlhead.insert(new edge_match_list(&match, etr), NULL);
            }

            while (emlhead.head) {
               emlpresident = NULL;

               for (emlptr=(edge_match_list *)emlhead.head; emlptr; emlptr=(edge_match_list *)emlptr->next)
                  if (!emlpresident || ((edge_match_type *)emlpresident->stats.head)->probability < ((edge_match_type *)emlptr->stats.head)->probability)
                     emlpresident = emlptr;

               emlhead.remove(emlpresident);

               i = smash_edge(emlpresident, &gtr->flist, ftr, vertex_data, overlap_count);

               delete emlpresident;

               if (i) {
                  ftr->post_smash(vertex_data, 1);
                  emlhead.dest();
                  break;
               }

            }

         } while (i);

         ftr->post_smash(vertex_data, 1);

         if (!ftr->vlist.count)
            delete ftr;
         else
            new_face_man.insert(ftr, NULL);
      }

      delete gtr;
   }

   while (new_face_man.head) {
	
      new_face_man.remove(ftr = (face_work_type *)new_face_man.head);
      ftr->vlist.calc_vnormal(vertex_data);

      if (ftr->calc_concave()) {
         if (!decimate(ftr))
            // if here, then we have some weird polygon caused by accuracy errors in the merge faces code...
            //      sol: look for "figure 8's" and decimate
            if (!split_weak_link(ftr, &new_face_man))
               mbprintf("WARNING: was not able to decimate a polygon");

         delete ftr;
      }
  
      else
         final_face_data.insert(ftr, NULL);
   }

}


/* *****************************************************************************
***************************************************************************** */
void optimizer::simple(dbl_llist_manager *groupman) {

   int i, j, k;

   group_type *gtr;
   face_work_type *dtr, *etr, *ftr;
   vertex_type *vtr, *wtr;

   int matchcount;
   dbl_llist_manager match;
   edge_match_type *mtr;
   edge_match_list *emlptr, *emlpresident;
   dbl_llist_manager emlhead;
   
   while (groupman->head) {
      groupman->remove(gtr = (group_type *)groupman->head);

      while (gtr->flist.head) {

         gtr->flist.remove(ftr = (face_work_type *)gtr->flist.head);
 
         do {

            i = 0;

            // calculate all possible merges w/ ftr
            for (etr = (face_work_type *)gtr->flist.head; etr; etr=(face_work_type *)etr->next) {
               query_combine(ftr, etr, &match);

               if (match.count)
                  emlhead.insert(new edge_match_list(&match, etr), NULL);
            }

            while (emlhead.head) {
               emlpresident = NULL;

               // find a face w/ shared edge of ftr
               for (matchcount = 0, emlptr=(edge_match_list *)emlhead.head; emlptr; emlptr=(edge_match_list *)emlptr->next) {
                  for (j=k=0, mtr=(edge_match_type *)emlptr->stats.head; mtr; mtr=(edge_match_type *)mtr->next, j++)
                     k += (mtr->resultflag == EC_TWO_VERTICES);

                  if (j == 1 && j == 1 && k > matchcount) {
                     emlpresident = emlptr;
                     matchcount = k;
                     break;
                  }

               }

               if (emlpresident) {

                  // pull face from face list
                  emlhead.remove(emlpresident);

                  // clone main face and set ptrs to shared vertices
                  etr = new face_work_type;

                  copyarray4(etr->face_normal, ftr->face_normal);
                  etr->material_index = ftr->material_index;
                  etr->textureflag = ftr->textureflag;
                  ftr->vlist.clone_list(&etr->vlist);

                  mtr = (edge_match_type *)emlpresident->stats.head;
		  
                  vtr = (vertex_type *)ftr->vlist.head;
                  wtr = (vertex_type *)etr->vlist.head;

                  do {
                     if (vtr == mtr->v11)
                        mtr->v11 = wtr;
                     else if (vtr == mtr->v12)
                        mtr->v12 = wtr;

                     vtr = (vertex_type *)vtr->next;
                     wtr = (vertex_type *)wtr->next;
                  } while (vtr != (vertex_type *)ftr->vlist.head);

                  // clone new face, and set ptrs to shared verticies
                  dtr = new face_work_type;

                  copyarray4(dtr->face_normal, emlpresident->candidate->face_normal);
                  dtr->material_index = emlpresident->candidate->material_index;
                  dtr->textureflag = emlpresident->candidate->textureflag;
                  emlpresident->candidate->vlist.clone_list(&dtr->vlist);

                  vtr = (vertex_type *)emlpresident->candidate->vlist.head;
                  wtr = (vertex_type *)dtr->vlist.head;

                  do {
                     if (vtr == mtr->v21)
                        mtr->v21 = wtr;
                     else if (vtr == mtr->v22)
                        mtr->v22 = wtr;

                     vtr = (vertex_type *)vtr->next;
                     wtr = (vertex_type *)wtr->next;
                  } while (vtr != (vertex_type *)emlpresident->candidate->vlist.head);

                  join_face(etr, mtr->v11, mtr->v12, dtr, mtr->v21, mtr->v22);

                  delete dtr;

                  etr->post_smash(vertex_data, 1);
                  etr->vlist.calc_vnormal(vertex_data);

                  if (!etr->calc_concave()) {
                     delete ftr;

                     ftr = etr;

                     gtr->flist.remove(emlpresident->candidate);

                     delete emlpresident->candidate;
                     delete emlpresident;
                     emlhead.dest();				

                     i = 1;
                     break;
                  }

                  delete etr;
                  delete emlpresident;
               }

               else {
                  emlhead.dest();				
                  break;
               }

            }

         } while (i);

         ftr->post_smash(vertex_data, 1);

         if (!ftr->vlist.count)
            delete ftr;			
         else
            final_face_data.insert(ftr, NULL);
      }

      delete gtr;
   }

}


/* *****************************************************************************
***************************************************************************** */
void optimizer::optimize_geometry(mtllist *mtl, int aggressive_flag, int spike_flag, int optimize_flag) {

   face_work_type *ftr, *etr;
   vertex_type *vtr;
   int i;
   dbl_llist_manager group_data;
   int overlap_count, oldvcount, oldfcount;
   group_type *gtr, *htr;
   char buffer[1024];

   oldvcount = vertexcount;
 
   // get rid of similar verticies
   if (optimize_flag)
      build_optimized_vertexlist();

   oldfcount = final_face_data.count;

   // calc vertex and face normals
   // opt trick - read till end of list, but insert at start of list
   etr = (face_work_type *)final_face_data.head;

   while (etr) {
      ftr = etr;
      etr = (face_work_type *)etr->next;
      final_face_data.remove(ftr);
      
      ftr->post_smash(vertex_data, 1);
      
      if (!ftr->vlist.validate(vertex_data)) {
         delete ftr;
         continue;
      }
           
      i = 0;
      vtr = (vertex_type *)ftr->vlist.head;
      
      do {
         if (query_calc_normal(vertex_data[vtr->vertex_index],
                               vertex_data[((vertex_type *)vtr->next)->vertex_index],
                               vertex_data[((vertex_type *)vtr->next->next)->vertex_index])) {
            i = 1;
            break;
         }
	  
         vtr = (vertex_type *)vtr->next;
      } while (vtr != (vertex_type *)ftr->vlist.head);
	
      if (i) {
         ftr->vlist.make_head(vtr);

         calculate_normal(vertex_data[((vertex_type *)vtr->next->next)->vertex_index],
                          vertex_data[((vertex_type *)vtr->next)->vertex_index],
                          vertex_data[vtr->vertex_index], ftr->face_normal);

         ftr->face_normal[3] = -dotproduct3(ftr->face_normal, vertex_data[((vertex_type *)ftr->vlist.head)->vertex_index]);

         vtr = (vertex_type *)ftr->vlist.head;
	 
         do {
            copyarray3(vtr->normal, ftr->face_normal);
            vtr = (vertex_type *)vtr->next;
         } while (vtr != (vertex_type *)ftr->vlist.head);

         final_face_data.insert(ftr, NULL);
      }
      
      else {
         mbprintf("Removing really small/thin polygon...");
         delete ftr;
      }

   }      

   if (optimize_flag) {

      // group faces by material type and planar equation...
      gtr = NULL;
      while (final_face_data.head) {
         final_face_data.remove(ftr = (face_work_type *)final_face_data.head);

         if (!gtr) {
            gtr = new group_type;
            copyarray4(gtr->group_normal, ftr->face_normal);
            gtr->material_index = ftr->material_index;
            gtr->textureflag = ftr->textureflag;
            gtr->flist.insert(ftr, NULL);
         }

         else
            gtr->mega_insert_btree(vertex_data, ftr);
      }

      if (!gtr)
         return;

      gtr->build_list(&group_data);
      
      // remove groups of "1" face from list to process
      htr = (group_type *)group_data.head;
      while (htr) {
         gtr = htr;
         htr = (group_type *)htr->next;

         if (gtr->flist.count == 1) {
            group_data.remove(gtr);
            gtr->flist.remove(ftr = (face_work_type *)gtr->flist.head);
            final_face_data.insert(ftr, NULL);
            delete gtr;
         }

      }
      
      // process special case -> groups w/ 2 faces -> must share 2 vertices or no reason to join
      overlap_count = 0;
      htr = (group_type *)group_data.head;
      while (htr) {
         gtr = htr;
         htr = (group_type *)htr->next;

         if (gtr->flist.count == 2) {
            group_data.remove(gtr);
            combine_2_triangles(gtr, &overlap_count); 
            delete gtr;
         }

      }

      if (aggressive_flag)
         complex(&group_data, &overlap_count);
      else
         simple(&group_data);

      // get rid of any "similar" vertices
      build_optimized_vertexlist();

      etr = (face_work_type *)final_face_data.head;
      while (etr) {
         ftr = etr;
         etr = (face_work_type *)etr->next;

         // streamline edges
         ftr->post_smash(vertex_data, 1);
         if (ftr->vlist.count < 3) {
            final_face_data.remove(ftr);
            delete ftr;
         }
			
         // streamline faces
         else {
            vtr = (vertex_type *)ftr->vlist.head;
            i = 1;

            do {
               if (!query_spike(vertex_data, vtr, (vertex_type *)vtr->next, (vertex_type *)vtr->next->next, 0, COS_FIVE_DEG)) {
                  ftr->vlist.make_head(vtr);
                  i = 0;
                  break;
               }

               vtr = (vertex_type *)vtr->next;
            } while (vtr != (vertex_type *)ftr->vlist.head);

            if (i && spike_flag) {
               final_face_data.remove(ftr);
               delete ftr;
            }

         }

      }

   }

   if (!final_face_data.head)
      return;

   // sort by textures and materials

   gtr = NULL;

   while (final_face_data.head) {
      final_face_data.remove(ftr = (face_work_type *)final_face_data.head);

      if (!gtr) {
         gtr = new group_type;
         gtr->material_index = ftr->material_index;
         gtr->flist.insert(ftr, NULL);
      }
      
      else
         gtr->material_insert_btree(mtl, ftr);
   }

   gtr->build_list(&group_data);
   
   while (group_data.head) {
      group_data.remove(gtr = (group_type *)group_data.head);
      
      while (gtr->flist.head) {
         gtr->flist.remove(ftr = (face_work_type *)gtr->flist.head);
         final_face_data.insert(ftr, NULL);
      }

      delete gtr;
   }

   // final tally
   sprintf(buffer, " percent reduction: v %f f %f", (100.0*(oldvcount - vertexcount))/(float)oldvcount, (100.0*(oldfcount - final_face_data.count))/(float)oldfcount);
   mbprintf(buffer);
}
