

#include <stdlib.h>
#include <errno.h>

#include "amessage.h"
#include "timer.h"
#include "global.h"

#ifdef WIN32

#define SO_REUSEPORT SO_REUSEADDR

typedef struct sockaddr FAR STRUCT_SOCKADDR;

#define ioctl ioctlsocket      
typedef int socklen_t;

#define SERVER_ADDR INADDR_ANY

#define OPTVAL_TYPE unsigned long

#define ENETUNREACH WSAENETUNREACH

#else

#include <stdio.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>

#define OPTVAL_TYPE int

#ifdef LINUX

#include <string.h>

#define SO_REUSEPORT 15

#else

typedef int socklen_t;

#endif

#define INVALID_SOCKET -1
#define SOCKET_ERROR -1
#define SD_SEND 1

#define closesocket close

#define SERVER_ADDR htonl(INADDR_ANY)

typedef struct sockaddr STRUCT_SOCKADDR;

#endif


/* ******************************************************************
****************************************************************** */
int asockudp::init(int argc, char **argv, int type) {

#ifdef WIN32
   WSADATA WSAData;
#endif

   socklen_t optval = 1;
   socklen_t size;
   struct in_addr address;
   struct hostent *host;
   int i;
   struct linger linger_data;

   dest();

   for (i = 0; i<argc; i++) {

      if (argv[i][0] != '-')
         continue;

      if (!strcmp(TOKEN_SOCKET_PORT, argv[i])) {
         i++;
         if (i >= argc)
            break;

         port = atoi(argv[i]);
         continue;
      }

      if (!strcmp(TOKEN_SOCKET_ADDRESS, argv[i])) {
         i++;
         if (i >= argc)
            break;

         socket_address.stringcpy(argv[i]);
         continue;
      }

      if (!strcmp(TOKEN_SOCKET_BROADCAST, argv[i])) {
         socket_flag |= ASOCKET_FLAG_BROADCAST;

         i++;
         if (i >= argc)
            break;

         port = atoi(argv[i]);
         continue;
      }

   }

   config_type = type;

#ifdef WIN32

   // initialize sockets
//   if (WSAStartup(MAKEWORD(1,1), &WSAData)) {
   if (WSAStartup(0x202, &WSAData)) {
      error_handler("Error: Unable to initialize sockets...");
      return 0;
   }

#endif

   socket_flag |= ASOCKET_FLAG_INIT;

   memset(&dest_sin, 0, sizeof(struct sockaddr_in));

   dest_sin.sin_family = AF_INET;
   dest_sin.sin_port = htons(port); 

   if ((socket_flag & ASOCKET_FLAG_BROADCAST) || config_type == CONFIG_TYPE_SERVER)
      dest_sin.sin_addr.s_addr = SERVER_ADDR;
   else {
      // lookup the address for the server name
      if (!inet_aton(socket_address.string, &address)) {
         host = gethostbyname(socket_address.string);
         if (!host) {
            error_handler("Error: unable to locate host...");
            return 0;
         }

         memcpy(&address.s_addr, host->h_addr, sizeof(int));
      }

      dest_sin.sin_addr.s_addr = address.s_addr;
   }

   sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);

   if (sock == INVALID_SOCKET) {
      error_handler("Error: Unable to allocate a socket...");
      return 0;
   }

   socket_flag |= ASOCKET_FLAG_CONNECT;

   // set the reuse port option to allow client and server side to run on the same host
   setsockopt(sock, SOL_SOCKET, SO_REUSEPORT, (char *)&optval, sizeof(socklen_t));

   linger_data.l_onoff = 1;
   linger_data.l_linger = 5;
   setsockopt(sock, SOL_SOCKET, SO_LINGER, (char *)&linger_data, sizeof(struct linger));

   // non-blocking mode
   ioctl(sock, FIONBIO, (OPTVAL_TYPE *)&optval);          

   if (setsockopt(sock, SOL_SOCKET, SO_BROADCAST, (char *)&optval, sizeof(socklen_t)) == SOCKET_ERROR) {
      error_handler("Error: Unable to set a broadcast socket...");
      return 0;
   }

   if ((socket_flag & ASOCKET_FLAG_BROADCAST) || config_type == CONFIG_TYPE_SERVER) {

      if (bind(sock, (STRUCT_SOCKADDR *)&dest_sin, sizeof(struct sockaddr_in)) == SOCKET_ERROR) {
         error_handler("Error: Unable to bind a socket...");
         return 0;
      }

      if (!port) {
         size = sizeof(struct sockaddr_in);
         getsockname(sock, (STRUCT_SOCKADDR *)&dest_sin, &size);
         port = htons(dest_sin.sin_port);
      }

   }

   if (socket_flag & ASOCKET_FLAG_BROADCAST)
      dest_sin.sin_addr.s_addr = htonl(INADDR_BROADCAST);

   return 1;
}


/* ******************************************************************
****************************************************************** */
int asockudp::init_connection(float timeout) {

   buffer_type buffer, temp;   
   TIME_STRUCT t;
   float oldtime, newtime;

   if (config_type == CONFIG_TYPE_CLIENT) {
      buffer.bufferlen(0);
      
      if (timeout < 0) {
         do {
            send_message(&buffer);
            SLEEP(0.03); 
            receive_message(&temp);
         } while (!(socket_flag & ASOCKET_FLAG_UDP_CONNECT));
 
         return 1;
      }

      GETTIME(&t);
      oldtime = (float)TIME2FLOAT(t);

      do {
         send_message(&buffer);
         SLEEP(0.03); 
         receive_message(&temp);
 
         if (socket_flag & ASOCKET_FLAG_UDP_CONNECT)
            return 1;

         GETTIME(&t);
         newtime = (float)TIME2FLOAT(t);
      } while (newtime >= oldtime && newtime < oldtime + timeout);

      return 0;
   }

   // config_type == CONFIG_TYPE_SERVER

   if (timeout < 0) {
      do {
         SLEEP(0.03); 
         receive_message(&temp);
      } while (!(socket_flag & ASOCKET_FLAG_UDP_CONNECT));
 
      return 1;
   }

   GETTIME(&t);
   oldtime = (float)TIME2FLOAT(t);

   do {
      SLEEP(0.03); 
      receive_message(&temp);
 
      if (socket_flag & ASOCKET_FLAG_UDP_CONNECT)
         return 1;

      GETTIME(&t);
      newtime = (float)TIME2FLOAT(t);
   } while (newtime >= oldtime && newtime < oldtime + timeout);

   return 0;
}


/* ******************************************************************
****************************************************************** */
int asockudp::query_message(buffer_type *buffer) {

   int size;
   int i;
   socklen_t j;
   buffer_type temp;
   struct sockaddr from;
   
   ioctl(sock, FIONREAD, (unsigned long *)&size);          

   if (size < sizeof(int))
      return 0;

   j = sizeof(struct sockaddr);
   if (recvfrom(sock, (char *)&i, sizeof(int), MSG_PEEK, &from, &j) < sizeof(int))
      return 0;

   i = decode(i);

   if (size < i)
      return 0;

   temp.bufferlen(size = i);

   j = sizeof(struct sockaddr);
   recvfrom(sock, temp.data, size, MSG_PEEK, (struct sockaddr *)&from, &j);
  
   buffer->bufferlen(size -= sizeof(int));
   buffer->buffercpy(temp.data+sizeof(int), size);
   return 1;
}


/* ******************************************************************
****************************************************************** */
int asockudp::receive_message(buffer_type *buffer) {

   return receive_message(buffer, (struct sockaddr *)&dest_sin);
}


/* ******************************************************************
****************************************************************** */
int asockudp::send_message(buffer_type *buffer) {

   return send_message(buffer, (struct sockaddr *)&dest_sin);
}


/* ******************************************************************
****************************************************************** */
int asockudp::receive_message(buffer_type *buffer, struct sockaddr *dest) {

   int size;
   int i;
   socklen_t j;
   buffer_type temp;
   struct sockaddr from;
   
   ioctl(sock, FIONREAD, (unsigned long *)&size);          

   if (size < sizeof(int))
      return 0;

   j = sizeof(struct sockaddr);
   if (recvfrom(sock, (char *)&i, sizeof(int), MSG_PEEK, &from, &j) < sizeof(int))
      return 0;

   i = decode(i);

   if (size < i)
      return 0;

   temp.bufferlen(size = i);

   j = sizeof(struct sockaddr);
   i = recvfrom(sock, temp.data, size, 0, &from, &j) > 0;

   if (!(socket_flag & ASOCKET_FLAG_UDP_CONNECT)) {
      socket_flag |= ASOCKET_FLAG_UDP_CONNECT;

      if (!(socket_flag & ASOCKET_FLAG_BROADCAST) && config_type == CONFIG_TYPE_SERVER)
         memcpy(dest, &from, sizeof(struct sockaddr));
   }

   if (size == sizeof(int)) {

      if (config_type == CONFIG_TYPE_SERVER) {
         // send an acknoledgement
         temp.bufferlen(0);
         send_message(&temp, dest);      
      }

      return 0;
   }
   
   buffer->bufferlen(size -= sizeof(int));
   buffer->buffercpy(temp.data+sizeof(int), size);
   
   return i;
}


/* ******************************************************************
****************************************************************** */
int asockudp::send_message(buffer_type *buffer, struct sockaddr *dest) {

   buffer_type temp;
   int i;

#ifndef WIN32
   unsigned int a, b, c;
#endif

   i = buffer->bufferlen() + sizeof(int);
   
   temp.bufferlen(i);

   *(int *)temp.data = encode(i);
   memcpy(temp.data+sizeof(int), buffer->data, buffer->bufferlen());

   if (!(socket_flag & ASOCKET_FLAG_BROADCAST)) {
      while (sendto(sock, temp.data, i, 0, dest, sizeof(struct sockaddr)) == SOCKET_ERROR);
      return 1;
   }

   // broadcast...
   while (sendto(sock, temp.data, i, 0, dest, sizeof(struct sockaddr)) == SOCKET_ERROR)

#ifdef WIN32
      errno = WSAGetLastError();
#endif

      switch (errno) {
         // try class "C" broadcast
         case ENETUNREACH:

#ifndef WIN32
            b = htonl(gethostid());
            a = (b>>8) & 0xff;
            c = (b>>24) & 0xff;
            b &= 0xff;
            dest_sin.sin_addr.s_addr = a | (b<<8) | (c<<16) | (255<<24);
            pprintf("Warning: downgrading broadcast to local class C network...\n");
#endif

            break;

         default:
            break;
      }

   return 1;
}


/* ******************************************************************
****************************************************************** */
int asockudp::query_message_raw(buffer_type *buffer, int len) {

   int size;
   socklen_t j;
   struct sockaddr from;
   
   ioctl(sock, FIONREAD, (unsigned long *)&size);          

   j = sizeof(struct sockaddr);

   if (size < len)
      return 0;

   buffer->bufferlen(len);
   recvfrom(sock, buffer->data, len, MSG_PEEK, (struct sockaddr *)&from, &j);

   return 1;
}


/* ******************************************************************
****************************************************************** */
int asockudp::receive_message_raw(buffer_type *buffer, int len) {

   return receive_message_raw(buffer, (struct sockaddr *)&dest_sin, len);
}


/* ******************************************************************
****************************************************************** */
int asockudp::send_message_raw(buffer_type *buffer) {

   return send_message_raw(buffer, (struct sockaddr *)&dest_sin);
}


/* ******************************************************************
****************************************************************** */
int asockudp::receive_message_raw(buffer_type *buffer, struct sockaddr *dest, int len) {

   int size;
   socklen_t j;
   struct sockaddr from;
   
   ioctl(sock, FIONREAD, (unsigned long *)&size);          

   j = sizeof(struct sockaddr);

   if (size < len)
      return 0;

   buffer->bufferlen(len);

   return recvfrom(sock, buffer->data, len, 0, &from, &j) > 0;
}


/* ******************************************************************
****************************************************************** */
int asockudp::send_message_raw(buffer_type *buffer, struct sockaddr *dest) {

#ifndef WIN32
   unsigned int a, b, c;
#endif

   if (!(socket_flag & ASOCKET_FLAG_BROADCAST)) {
      while (sendto(sock, buffer->data, buffer->bufferlen(), 0, dest, sizeof(struct sockaddr)) == SOCKET_ERROR);
      return 1;
   }

   // broadcast...
   while (sendto(sock, buffer->data, buffer->bufferlen(), 0, dest, sizeof(struct sockaddr)) == SOCKET_ERROR) {

#ifdef WIN32
      errno = WSAGetLastError();
#endif

      switch (errno) {
         // try class "C" broadcast
         case ENETUNREACH:

#ifndef WIN32
            b = htonl(gethostid());
            a = (b>>8) & 0xff;
            c = (b>>24) & 0xff;
            b &= 0xff;
            dest_sin.sin_addr.s_addr = a | (b<<8) | (c<<16) | (255<<24);
            pprintf("Warning: downgrading broadcast to local class C network...\n");
#endif

            break;

         default:
            break;
      }

   }

   return 1;
}
