

#include <stdlib.h>

#include "amessage.h"
#include "timer.h"
#include "global.h"

#ifdef WIN32

#define errno WSAGetLastError()

#define EISCONN WSAEISCONN
#define SO_REUSEPORT SO_REUSEADDR

typedef struct sockaddr FAR STRUCT_SOCKADDR;
typedef int socklen_t;

#define ioctl ioctlsocket      

#define SERVER_ADDR INADDR_ANY

#define OPTVAL_TYPE unsigned long


int inet_aton(char *in, struct in_addr *address) {

   *(int *)address = inet_addr(in);
   return *(int *)address != INADDR_NONE;
}


#else

#include <unistd.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <stdio.h>

#define OPTVAL_TYPE int

#ifdef LINUX

#include <string.h>

#define SO_REUSEPORT 15

#else

typedef int socklen_t;

#endif

#define INVALID_SOCKET -1
#define SOCKET_ERROR -1
#define SD_SEND 1

#define closesocket close

#define SERVER_ADDR htonl(INADDR_ANY)

typedef struct sockaddr STRUCT_SOCKADDR;

#endif


/* ******************************************************************
****************************************************************** */
asocktcp::asocktcp() {

   socket_flag = ASOCKET_FLAG_NULL;
   port = 0;
   socket_address.stringcpy("127.0.0.1");
}


/* ******************************************************************
****************************************************************** */
unsigned int asocktcp::decode(unsigned int i) {

   union {
      unsigned char cbuf[4];
      unsigned int ibuf;
   };
   
   cbuf[0] = (unsigned char)(i>>24);
   cbuf[1] = (unsigned char)(i>>16) & 0xff;
   cbuf[2] = (unsigned char)(i>>8)  & 0xff;
   cbuf[3] = (unsigned char)i       & 0xff;

   return ibuf;
}


/* ******************************************************************
****************************************************************** */
unsigned int asocktcp::encode(unsigned int i) {

   char *c = (char *)&i;
   
   return c[3] | (c[2]<<8) | (c[1]<<16) | (c[0]<<24);
}


/* ******************************************************************
****************************************************************** */
void asocktcp::error_handler(char *error_msg) {

   pprintf(error_msg);
   dest();
}


/* ******************************************************************
****************************************************************** */
int asocktcp::init(int argc, char **argv, int type) {

#ifdef WIN32
   WSADATA WSAData;
#endif

   socklen_t optval = 1;
   socklen_t size;
   struct in_addr address;
   struct hostent *host;
   int i;
   struct linger linger_data;

   dest();

   for (i = 0; i<argc; i++) {

      if (argv[i][0] != '-')
         continue;

      if (!strcmp(TOKEN_SOCKET_PORT, argv[i])) {
         i++;
         if (i >= argc)
            break;

         port = atoi(argv[i]);
         continue;
      }

      if (!strcmp(TOKEN_SOCKET_ADDRESS, argv[i])) {
         i++;
         if (i >= argc)
            break;

         socket_address.stringcpy(argv[i]);
         continue;
      }

   }

   config_type = type;

#ifdef WIN32

   // initialize sockets
//   if (WSAStartup(MAKEWORD(1,1), &WSAData)) {
   if (WSAStartup(0x202, &WSAData)) {
      error_handler("Error: Unable to initialize sockets...");
      return 0;
   }

#endif

   socket_flag |= ASOCKET_FLAG_INIT;

   memset(&dest_sin, 0, sizeof(struct sockaddr_in));

   dest_sin.sin_family = AF_INET;
   dest_sin.sin_port = htons(port); 

   if (config_type == CONFIG_TYPE_SERVER)
      dest_sin.sin_addr.s_addr = SERVER_ADDR;
   else {
      // lookup the address for the server name
      if (!inet_aton(socket_address.string, &address)) {
         host = gethostbyname(socket_address.string);
         if (!host) {
            error_handler("Error: unable to locate host...");
            return 0;
         }

         memcpy(&address.s_addr, host->h_addr, sizeof(int));
      }

      dest_sin.sin_addr.s_addr = address.s_addr;
   }

   sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);

   if (sock == INVALID_SOCKET) {
      error_handler("Error: Unable to allocate a socket...");
      return 0;
   }

   socket_flag |= ASOCKET_FLAG_CONNECT;

   if (config_type == CONFIG_TYPE_SERVER) {

      // set the reuse port option to allow client and server side to run on the same host
      setsockopt(sock, SOL_SOCKET, SO_REUSEPORT, (char *)&optval, sizeof(socklen_t));

      linger_data.l_onoff = 1;
      linger_data.l_linger = 5;

      setsockopt(sock, SOL_SOCKET, SO_LINGER, (char *)&linger_data, sizeof(struct linger));

      // non-blocking mode
      ioctl(sock, FIONBIO, (OPTVAL_TYPE *)&optval);          

      if (bind(sock, (STRUCT_SOCKADDR *)&dest_sin, sizeof(struct sockaddr_in)) == SOCKET_ERROR) {
         error_handler("Error: Unable to bind a socket...");
         return 0;
      }

      if (!port) {
         size = sizeof(struct sockaddr_in);
         getsockname(sock, (STRUCT_SOCKADDR *)&dest_sin, &size);
         port = htons(dest_sin.sin_port);
      }

      if (listen(sock, SOMAXCONN) < 0) {
         error_handler("Error: Listen(sock) failed");
         return 0;
      }

   }

   return 1;
}


/* ******************************************************************
****************************************************************** */
int asocktcp::init_connection(float timeout) {

   socklen_t size = (socklen_t)sizeof(struct sockaddr_in);
   TIME_STRUCT t;
   float oldtime, newtime;
   
   if (config_type == CONFIG_TYPE_CLIENT) {

      if (timeout < 0) {
         while (connect(sock, (STRUCT_SOCKADDR *)&dest_sin, sizeof(struct sockaddr_in)) == INVALID_SOCKET && errno != EISCONN);
         return 1;
      }

      GETTIME(&t);
      oldtime = (float)TIME2FLOAT(t);

      while (connect(sock, (STRUCT_SOCKADDR *)&dest_sin, sizeof(struct sockaddr_in)) == INVALID_SOCKET && errno != EISCONN) {
         GETTIME(&t);
         newtime = (float)TIME2FLOAT(t);
         if (newtime < oldtime || newtime >= oldtime + timeout)
            return 0;
      }

      return 1;
   }

   // config_type == CONFIG_TYPE_SERVER

   if (timeout < 0) {
      do {
         accept_sock = accept(sock, (STRUCT_SOCKADDR *)&dest_sin, &size);                  
      } while (accept_sock == INVALID_SOCKET);

      socket_flag |= ASOCKET_FLAG_ACCEPT;
      return 1;
   }

   GETTIME(&t);
   oldtime = (float)TIME2FLOAT(t);

   do {
      accept_sock = accept(sock, (STRUCT_SOCKADDR *)&dest_sin, &size);                  

      if (accept_sock != INVALID_SOCKET) {
         socket_flag |= ASOCKET_FLAG_ACCEPT;
         return 1;
      }

      GETTIME(&t);
      newtime = (float)TIME2FLOAT(t);
   } while (newtime >= oldtime && newtime < oldtime + timeout);

   return 0;
}


/* ******************************************************************
****************************************************************** */
void asocktcp::set_socket(SOCKET s) {

   sock = s;
   socket_flag |= ASOCKET_FLAG_CONNECT;
}


/* ******************************************************************
****************************************************************** */
void asocktcp::dest() {

#ifdef WIN32
   int flag = 0;
#endif

   if (socket_flag & ASOCKET_FLAG_ACCEPT) {
      while (shutdown(accept_sock, SD_SEND));

      if (closesocket(sock) == SOCKET_ERROR)
         pprintf("Warning: Unable to close a socket...");
   }

   if (socket_flag & ASOCKET_FLAG_CONNECT) {

#ifdef WIN32

      while (!flag && shutdown(sock, SD_SEND)) {

         switch (errno) {
            case WSANOTINITIALISED:
            case WSAENETDOWN:
            case WSAEINVAL:
            case WSAENOTCONN:
            case WSAENOTSOCK:
               flag = 1;
               break;

//            case WSAEINPROGRESS:
            default:
               break;
         }

      }

#else
     shutdown(sock, SD_SEND);
#endif

      if (closesocket(sock) == SOCKET_ERROR)
         pprintf("Warning: Unable to close a socket...");
   }

#ifdef WIN32
   if (socket_flag & ASOCKET_FLAG_INIT)
      WSACleanup();
#endif

   socket_flag = ASOCKET_FLAG_NULL;
   port = 0;
}


/* ******************************************************************
****************************************************************** */
int asocktcp::query_message(buffer_type *buffer) {

   unsigned int size;
   int i;
   buffer_type temp;
   
   ioctl(sock, FIONREAD, (unsigned long *)&size);          

   if (size < sizeof(int))
      return 0;

   if (recv(sock, (char *)&i, sizeof(int), MSG_PEEK) < sizeof(int))
      return 0;

   i = decode(i);

   if (size < (unsigned int)i)
      return 0;

  temp.bufferlen(size = i);

   recv(sock, temp.data, size, MSG_PEEK);
  
   buffer->bufferlen(size -= sizeof(int));
   buffer->buffercpy(temp.data+sizeof(int), size);
   
   return 1;
}


/* ******************************************************************
****************************************************************** */
int asocktcp::receive_message(buffer_type *buffer) {

   unsigned int size;
   int i;
   buffer_type temp;
   
   ioctl(sock, FIONREAD, (unsigned long *)&size);          

   if (size < sizeof(int))
      return 0;

   if (recv(sock, (char *)&i, sizeof(int), MSG_PEEK) < sizeof(int))
      return 0;

   i = decode(i);

   if (size < (unsigned int)i)
      return 0;

   temp.bufferlen(size = i);

   i = recv(sock, temp.data, size, 0) > 0;
  
   buffer->bufferlen(size -= sizeof(int));
   buffer->buffercpy(temp.data+sizeof(int), size);
   
   return i;
}


/* ******************************************************************
****************************************************************** */
int asocktcp::send_message(buffer_type *buffer) {

   buffer_type temp;
   int i;
   
   i = buffer->bufferlen() + sizeof(int);
   
   temp.bufferlen(i);

   *(int *)temp.data = encode(i);
   memcpy(temp.data+sizeof(int), buffer->data, buffer->bufferlen());

   while (send(sock, temp.data, i, 0) == SOCKET_ERROR);

   return 1;
}


/* ******************************************************************
****************************************************************** */
int asocktcp::query_message_raw(buffer_type *buffer, int len) {

   unsigned int size;
   
   ioctl(sock, FIONREAD, (unsigned long *)&size);          

   if (size < (unsigned int)len)
      return 0;

   buffer->bufferlen(len);

   recv(sock, buffer->data, len, MSG_PEEK);
   return 1;
}


/* ******************************************************************
****************************************************************** */
int asocktcp::receive_message_raw(buffer_type *buffer, int len) {

   unsigned int size;
   
   ioctl(sock, FIONREAD, (unsigned long *)&size);          

   if (size < (unsigned int)len)
      return 0;

   buffer->bufferlen(len);

   return recv(sock, buffer->data, len, 0) > 0;
}


/* ******************************************************************
****************************************************************** */
int asocktcp::send_message_raw(buffer_type *buffer) {

   while (send(sock, buffer->data, buffer->bufferlen(), 0) == SOCKET_ERROR);
   return 1;
}
