

#include <stdlib.h>
#include <stdio.h>

#include "amessage.h"
#include "ashmem.h"


#ifdef WIN32

#define mbprintf(xxx)    MessageBox( NULL, xxx, "", MB_OK )

#else

#include <sys/shm.h>
#include <unistd.h>

#ifdef LINUX
#include <string.h>
#endif

#define mbprintf(xxx)    printf("%s\n", xxx)

#endif

#define SHMEM_SIZE(xxx) (((xxx) + 7) & 0xfffffff8)


/* ******************************************************************
****************************************************************** */
ashmem::ashmem() {

   shmem_buffer = NULL;
   shmem_size = 1024;
   shmem_key = 0xdead;
   config_type = CONFIG_TYPE_CLIENT;
}


/* ******************************************************************
****************************************************************** */
int ashmem::get_buffer_size() {

   return shmem_buffer ? shmem_size : 0;
}


/* ******************************************************************
****************************************************************** */
int ashmem::init(int argc, char **argv, int type) {

   int i;

   dest();

   config_type = type;
   
   for (i = 0; i<argc; i++) {

      if (argv[i][0] != '-')
         continue;

      if (!strcmp(TOKEN_SHMEM_KEY, argv[i])) {
         i++;
         if (i >= argc)
            break;

         shmem_key = atoi(argv[i]);
         continue;
      }

      if (!strcmp(TOKEN_SHMEM_SIZE, argv[i])) {
         i++;
         if (i >= argc)
            break;

         shmem_size = SHMEM_SIZE(atoi(argv[i]));
         continue;
      }

   }

           
   return shmem_size != 0;
}


      
/* ******************************************************************
****************************************************************** */
int ashmem::reset() {

   if (!master_block)
      return 0;
   
   memset(master_block, 0, sizeof(master_block_type));
   master_block->freespace_index = master_block->header.block_size = SHMEM_SIZE(sizeof(master_block_type));
   return 1;
}
        
        
/* ******************************************************************
****************************************************************** */
void ashmem::insert_dirty_block(dirty_block *etr) {

   dirty_block *ptr;
   int index;

   if (!master_block->dirty_block_index) {
      master_block->dirty_block_index = master_block->dirty_block_tail = etr->header.index;
      etr->back_index = etr->next_index = 0;
      return;
   }
        
   for (index = master_block->dirty_block_index; index; index = ptr->next_index) {
      ptr = (dirty_block *)&shmem_buffer[index];
      if (ptr->header.block_size <= etr->header.block_size) {
         etr->next_index = ptr->header.index;
         etr->back_index = ptr->back_index;
        
         if (ptr->back_index)
            ((dirty_block *)&shmem_buffer[ptr->back_index])->next_index = etr->header.index;
         else
            master_block->dirty_block_index = etr->header.index;
        
         ptr->back_index = etr->header.index;
         return;
      }

   }

   etr->back_index = master_block->dirty_block_tail;
   etr->next_index = 0;
   ((dirty_block *)&shmem_buffer[master_block->dirty_block_tail])->next_index = etr->header.index;
   master_block->dirty_block_tail = etr->header.index;
}


/* ******************************************************************
****************************************************************** */
void ashmem::remove_dirty_block(dirty_block *dtr) {

   // remove node from dirty list
   if (dtr->next_index)
      ((dirty_block *)&shmem_buffer[dtr->next_index])->back_index = dtr->back_index;
   else
      master_block->dirty_block_tail = dtr->back_index;
        
   if (dtr->back_index)
      ((dirty_block *)&shmem_buffer[dtr->back_index])->next_index = dtr->next_index;
   else
      master_block->dirty_block_index = dtr->next_index;
}


/* ******************************************************************
****************************************************************** */
int ashmem::allocate_block(int bytecount, int clean_flag) {

   int ret, index;
   dirty_block *dtr, *etr;
           
   bytecount += sizeof(block_header_type); // extra for size of block
        
   ret = SHMEM_SIZE(sizeof(dirty_block));
        
   if (bytecount < ret)
      bytecount = ret;
        
   bytecount = SHMEM_SIZE(bytecount);
        
   // look at the dirty list
   for (index = master_block->dirty_block_index; index; index = dtr->next_index) {
      dtr = (dirty_block *)&shmem_buffer[index];
        
      if (dtr->header.block_size >= (unsigned int)bytecount) {
        
         // remove node from dirty list
         remove_dirty_block(dtr);
        
         // if block is too big, break it up, stick on dirty list
         if (dtr->header.block_size > (unsigned int)(bytecount+ret)) {
            etr = (dirty_block *)(((char *)dtr)+bytecount);
            etr->header.index = dtr->header.index + bytecount;
            etr->header.block_size = dtr->header.block_size - bytecount;
            insert_dirty_block(etr);
        
            dtr->header.block_size = bytecount;
         }
        
         ret = dtr->header.index+sizeof(block_header_type);

         if (clean_flag)
            memset(&shmem_buffer[ret], 0, bytecount-sizeof(block_header_type));

         return ret;  // return index after block_size
      }
        
   }
        
   if (master_block->freespace_index+bytecount > (unsigned int)(shmem_size))
      return 0;
        
   ret = master_block->freespace_index;
        
   master_block->freespace_index += shmem_buffer[ret] = bytecount;
   ret += sizeof(block_header_type);

   if (clean_flag)
      memset(&shmem_buffer[ret], 0, bytecount-sizeof(block_header_type));

   return ret;  // return index after block_size
}


/* ******************************************************************
****************************************************************** */
int ashmem::remove_block(int delete_index) {

   dirty_block *dtr, *etr;
   int index, pindex;
        
   if (!delete_index)
      return 0;
        
   // true index of block - gets back "size" field
   delete_index -= sizeof(block_header_type);
         
   dtr = (dirty_block *)&shmem_buffer[delete_index];
   dtr->header.index = delete_index;
        
   // merge if block is in "front" of another block
   pindex = (int)(dtr->header.index + dtr->header.block_size);
   for (index = master_block->dirty_block_index; index; index = etr->next_index) {
      etr = (dirty_block *)&shmem_buffer[index];
      if (pindex == index) {
         remove_dirty_block(etr);
         dtr->header.block_size += etr->header.block_size;
         break;
      }
        
   }
        
   // see if block can be merged w/ free list
   pindex = dtr->header.index + dtr->header.block_size;
   if (pindex == (int)master_block->freespace_index) {
      master_block->freespace_index = dtr->header.index;
      return 1;
   }
        
   // merge if block is in "behind" of another block
   for (index = master_block->dirty_block_index; index; index = etr->next_index) {
      etr = (dirty_block *)&shmem_buffer[index];
      pindex = etr->header.index + etr->header.block_size;
      if (pindex == (int)(dtr->header.index)) {
         remove_dirty_block(etr);
         etr->header.block_size += dtr->header.block_size;
         dtr = etr;
         break;
      }
        
   }
        
   // add block to dirty list
   insert_dirty_block(dtr);
   return 1;
}




#ifdef WIN32

/* ******************************************************************
****************************************************************** */
int ashmem_win::init(int argc, char **argv, int type) {
           
   char buffer[16];
   
   if (!ashmem::init(argc, argv, type))
      return 0;
      
   // open file map
   sprintf(buffer, "%d", shmem_key);
   hMap = CreateFileMapping((HANDLE)0xFFFFFFFF, NULL,PAGE_READWRITE, 0, shmem_size, buffer);
        
   if (!hMap) {
      mbprintf("ERROR: Shared memory mapping error...");
      return 0;
   }
        
   // convert handle to pointer
   shmem_buffer = (char *)MapViewOfFile(hMap, FILE_MAP_ALL_ACCESS, 0, 0, 0);

   if (config_type == CONFIG_TYPE_SERVER)
      reset();
      
   return 1;
}
        
#else
        
/* ******************************************************************
****************************************************************** */
int ashmem_x::init(int argc, char **argv, int type) {
           
   if (!ashmem::init(argc, argv, type))
      return 0;
      
   shmem_id = shmget(getuid() + shmem_key, shmem_size, 0666 | IPC_CREAT);
        
   if (shmem_id < 0) {
      mbprintf("ERROR: Shared memory mapping error...");
      return 0;
   }
        
   shmem_buffer = (char *)shmat(shmem_id, 0, 0);
        
   if (config_type == CONFIG_TYPE_SERVER)
      reset();
      
   return 1;
}

#endif

        
#ifdef WIN32

/* *******************************************************
******************************************************* */
void ashmem_win::dest() {

   if (shmem_buffer) {
      UnmapViewOfFile(shmem_buffer); 
      CloseHandle(hMap);
      shmem_buffer = NULL;
   }

}

#else
        
/* *******************************************************
******************************************************* */
void ashmem_x::dest() {
        
   if (shmem_buffer) {
      shmdt(shmem_buffer);
      shmem_buffer = NULL;
      
      if (config_type == CONFIG_TYPE_SERVER)
         shmctl(shmem_id, IPC_RMID, NULL);
   }
        
}

#endif 

