



/* *************************************************************
   This file handles all the matrix/vector/math operations

   Reminder: Spherical Linear intERPolation (SLERP) can be
             done by splining the quaternions

************************************************************* */

#include "matrix.h"

vector4f MCR[4]  = { -0.5,      1.5,     -1.5,      0.5,
                      1.0,     -2.5,      2.0,     -0.5,
                     -0.5,      0.0,      0.5,      0.0,
                      0.0,      1.0,      0.0,      0.0 };     // catmull-rom

vector4f MCRT[4] = { -0.5,      1.0,     -0.5,      0.0,       // Tcatmull-rom
                      1.5,     -2.5,      0.0,      1.0,
                     -1.5,      2.0,      0.5,      0.0,
                      0.5,     -0.5,      0.0,      0.0 };

vector4f MB[4]   = { -1.0f/6.0f,  0.5,     -0.5,      1.0f/6.0f,   // B-spline
                      0.5,     -1.0,      0.5,      0.0,
                     -0.5,      0.0,      0.5,      0.0,
                      1.0f/6.0f,  2.0f/3.0f,  1.0f/6.0f,  0.0 };

vector4f MBT[4]  = { -1.0f/6.0f,   0.5,    -0.5,       1.0f/6.0f,   // TB-spline
                     0.5,      -1.0,     0.0,       2.0f/3.0f,
                    -0.5,       0.5,     0.5,       1.0f/6.0f,
                     1.0f/6.0f,   0.0,     0.0,       0.0 };

/* *************************************************************
   this function calculates the current value of a surface patch
   spline - s is dy, t is dx
************************************************************* */
void sppatch(float s, float t, float in[][4], float out[], int ptr[], int stype) {

   vector4f T, S, temp;
   vector4f Q[4][4];
   int  i, j, k, l;

   for (j=0; j<4; j++)
      for (k=0; k<4; k++)
         for (l=0, i=ptr[k]; l<4; l++, i+=4)
            Q[j][k][l] = in[i][j];

   T[3] = 1;
   T[2] = t;
   T[1] = T[2]*T[2];
   T[0] = T[1]*T[2];

   S[3] = 1;
   S[2] = s;
   S[1] = S[2]*S[2];
   S[0] = S[1]*S[2];

   if (stype == M_CR) {
      matvecmult(MCRT, T);
      vecmatmult(S, MCR);
   }

   else {
      matvecmult(MBT, T);
      vecmatmult(S, MB);
   }

   for (j=0; j<4; j++) {
      temp[0] = dotproduct4(Q[j][0], T);
      temp[1] = dotproduct4(Q[j][1], T);
      temp[2] = dotproduct4(Q[j][2], T);
      temp[3] = dotproduct4(Q[j][3], T);
      out[j] = dotproduct4(S, temp);
   }

}


/* *************************************************************
   this function calculates the current value of a surface patch
   spline - s is dy, t is dx
************************************************************* */
void sppatch(float s, float t, vector4f **in, float *out, int stype) {

   vector4f T, S, temp;
   vector4f Q[4][4];
   int  j, k, l;

   for (j=0; j<4; j++)
      for (k=0; k<4; k++)
         for (l=0; l<4; l++)
            Q[j][k][l] = in[k][l][j];

   T[3] = 1;
   T[2] = t;
   T[1] = T[2]*T[2];
   T[0] = T[1]*T[2];

   S[3] = 1;
   S[2] = s;
   S[1] = S[2]*S[2];
   S[0] = S[1]*S[2];

   if (stype == M_CR) {
      matvecmult(MCRT, T);
      vecmatmult(S, MCR);
   }

   else {
      matvecmult(MBT, T);
      vecmatmult(S, MB);
   }

   for (j=0; j<4; j++) {
      temp[0] = dotproduct4(Q[j][0], T);
      temp[1] = dotproduct4(Q[j][1], T);
      temp[2] = dotproduct4(Q[j][2], T);
      temp[3] = dotproduct4(Q[j][3], T);
      out[j] = dotproduct4(S, temp);
   }

}


/* *************************************************************
   this function calculates the current value of a surface patch
   spline - s is dy, t is dx
************************************************************* */
void sppatch(float s, float t, float Q[][4][4], float *out, int stype) {

   vector4f T, S, temp;
   int  j;

   T[3] = 1;
   T[2] = t;
   T[1] = T[2]*T[2];
   T[0] = T[1]*T[2];

   S[3] = 1;
   S[2] = s;
   S[1] = S[2]*S[2];
   S[0] = S[1]*S[2];

   if (stype == M_CR) {
      matvecmult(MCRT, T);
      vecmatmult(S, MCR);
   }

   else {
      matvecmult(MBT, T);
      vecmatmult(S, MB);
   }

   for (j=0; j<4; j++) {
      temp[0] = dotproduct4(Q[j][0], T);
      temp[1] = dotproduct4(Q[j][1], T);
      temp[2] = dotproduct4(Q[j][2], T);
      temp[3] = dotproduct4(Q[j][3], T);
      out[j] = dotproduct4(S, temp);
   }

}
//


/* *************************************************************
   This function calculates the patch normals
************************************************************* */
void spnormal(float s, float t, float in[][4], float out[], int ptr[], int stype) {

   vector4f ds, dt;
   vector4f S, T;
   vector4f Q[4][4];
   vector4f temp1, temp2;
   vector3f pt1, pt2;
   int   i, j, k, l;

   for (j=0; j<4; j++)
      for (k=0; k<4; k++)
         for (l=0, i=ptr[k]; l<4; l++, i+=4)
            Q[j][k][l] = in[i][j];

   T[3] = 1;
   T[2] = t;
   T[1] = T[2]*T[2];
   T[0] = T[1]*T[2];

   S[3] = 1;
   S[2] = s;
   S[1] = S[2]*S[2];
   S[0] = S[1]*S[2];

   ds[3] = 0;
   ds[2] = 1;
   ds[1] = s+s;
   ds[0] = 3*s*s;

   dt[3] = 0;
   dt[2] = 1;
   dt[1] = t+t;
   dt[0] = 3*t*t;

   if (stype == M_CR) {
      matvecmult(MCRT, T);
      vecmatmult(S, MCR);

      matvecmult(MCRT, dt);
      vecmatmult(ds, MCR);
   }

   else {
      matvecmult(MBT, T);
      vecmatmult(S, MB);

      matvecmult(MBT, dt);
      vecmatmult(ds, MB);
   }

   for (i=0; i<3; i++) {
      temp1[0] = dotproduct4(Q[i][0], T);
      temp1[1] = dotproduct4(Q[i][1], T);
      temp1[2] = dotproduct4(Q[i][2], T);
      temp1[3] = dotproduct4(Q[i][3], T);
   
      temp2[0] = dotproduct4(Q[i][0], dt);
      temp2[1] = dotproduct4(Q[i][1], dt);
      temp2[2] = dotproduct4(Q[i][2], dt);
      temp2[3] = dotproduct4(Q[i][3], dt);

      pt1[i] = dotproduct4(ds, temp1);
      pt2[i] = dotproduct4(S, temp2);
   }
      
   xproduct(out, pt1, pt2);
   normalize3(out);
}


/* *************************************************************
   This function calculates the patch normals
************************************************************* */
void spnormal(float s, float t, vector4f **in, float *out, int stype) {

   vector4f ds, dt;
   vector4f S, T;
   vector4f Q[4][4];
   vector4f temp1, temp2;
   vector3f pt1, pt2;
   int j, k, l;

   for (j=0; j<4; j++)
      for (k=0; k<4; k++)
         for (l=0; l<4; l++)
            Q[j][k][l] = in[k][l][j];

   T[3] = 1;
   T[2] = t;
   T[1] = t*t;
   T[0] = T[1]*T[2];

   S[3] = 1;
   S[2] = s;
   S[1] = s*s;
   S[0] = S[1]*S[2];

   ds[3] = 0;
   ds[2] = 1;
   ds[1] = s+s;
   ds[0] = 3*S[1];

   dt[3] = 0;
   dt[2] = 1;
   dt[1] = t+t;
   dt[0] = 3*T[1];

   if (stype == M_CR) {
      matvecmult(MCRT, T);
      vecmatmult(S, MCR);

      matvecmult(MCRT, dt);
      vecmatmult(ds, MCR);
   }

   else {
      matvecmult(MBT, T);
      vecmatmult(S, MB);

      matvecmult(MBT, dt);
      vecmatmult(ds, MB);
   }

   for (k=0; k<3; k++) {
      temp1[0] = dotproduct4(Q[k][0], T);
      temp1[1] = dotproduct4(Q[k][1], T);
      temp1[2] = dotproduct4(Q[k][2], T);
      temp1[3] = dotproduct4(Q[k][3], T);
   
      temp2[0] = dotproduct4(Q[k][0], dt);
      temp2[1] = dotproduct4(Q[k][1], dt);
      temp2[2] = dotproduct4(Q[k][2], dt);
      temp2[3] = dotproduct4(Q[k][3], dt);
   
      pt1[k] = dotproduct4(ds, temp1);
      pt2[k] = dotproduct4(S, temp2);
   }
   
   xproduct(out, pt1, pt2);
   normalize3(out);
}


/* *************************************************************
   This function calculates the patch normals
************************************************************* */
void spnormal(float s, float t, float Q[][4][4], float *out, int stype) {

   vector4f ds, dt;
   vector4f S, T;
   vector4f temp1, temp2;
   vector3f pt1, pt2;
   int k;

   T[3] = 1;
   T[2] = t;
   T[1] = t*t;
   T[0] = T[1]*T[2];

   S[3] = 1;
   S[2] = s;
   S[1] = s*s;
   S[0] = S[1]*S[2];

   ds[3] = 0;
   ds[2] = 1;
   ds[1] = s+s;
   ds[0] = 3*S[1];

   dt[3] = 0;
   dt[2] = 1;
   dt[1] = t+t;
   dt[0] = 3*T[1];

   if (stype == M_CR) {
      matvecmult(MCRT, T);
      vecmatmult(S, MCR);

      matvecmult(MCRT, dt);
      vecmatmult(ds, MCR);
   }

   else {
      matvecmult(MBT, T);
      vecmatmult(S, MB);

      matvecmult(MBT, dt);
      vecmatmult(ds, MB);
   }

   for (k=0; k<3; k++) {
      temp1[0] = dotproduct4(Q[k][0], T);
      temp1[1] = dotproduct4(Q[k][1], T);
      temp1[2] = dotproduct4(Q[k][2], T);
      temp1[3] = dotproduct4(Q[k][3], T);
   
      temp2[0] = dotproduct4(Q[k][0], dt);
      temp2[1] = dotproduct4(Q[k][1], dt);
      temp2[2] = dotproduct4(Q[k][2], dt);
      temp2[3] = dotproduct4(Q[k][3], dt);
   
      pt1[k] = dotproduct4(ds, temp1);
      pt2[k] = dotproduct4(S, temp2);
   }
   
   xproduct(out, pt1, pt2);
   normalize3(out);
}


/* *************************************************************
   This function calculates the normal at s = t = 0
************************************************************* */
void spnormal_base(vector4f **in, float out[], int stype) {

   vector4f S, T;
   vector4f Q[4][4];
   vector4f temp1, temp2;
   vector3f pt1, pt2;
   int   i, j, k, l;

   if (stype == M_CR) {
      subeqarray3(pt1, in[2][1], in[0][1]);
      subeqarray3(pt2, in[1][2], in[1][0]);
   }

   else {
      for (j=0; j<4; j++)
         for (k=0; k<4; k++)
            for (l=0; l<4; l++)
               Q[j][k][l] = in[k][l][j];

      S[0] = T[0] = MB[3][0];
      S[1] = T[1] = MB[3][1];
      S[2] = T[2] = MB[3][2];

      for (i=0; i<3; i++) {
         temp1[0] = dotproduct3(Q[i][0], T);
         temp1[2] = dotproduct3(Q[i][2], T);

         temp2[0] = 0.5f*(-Q[i][0][0] + Q[i][0][2]);
         temp2[1] = 0.5f*(-Q[i][1][0] + Q[i][1][2]);
         temp2[2] = 0.5f*(-Q[i][2][0] + Q[i][2][2]);

         pt1[i] = 0.5f*(-temp1[0] + temp1[2]);
         pt2[i] = dotproduct3(S, temp2);
      }

   }

   xproduct(out, pt1, pt2);
   normalize3(out);
}


/* *************************************************************
   This procedure calculates the current placement of objects
   using Catmull-Rom or B splines
************************************************************* */
void spline(float s, float in[][4], float out[], int i, int stype) {

   vector4f Q[4];
   vector4f T;
   int j;

/*
   for (j=0; j<4; j++)
      for (k=0; k<4; k++)
         Q[j][k] = in[i+k][j];

   if (stype == M_CR) {                           // Catmull-Rom spline
      matvecmult(MCR, Q[0]);
      matvecmult(MCR, Q[1]);
      matvecmult(MCR, Q[2]);
      matvecmult(MCR, Q[3]);
   }

   else {
      matvecmult(MB, Q[0]);                                                // B - Spline
      matvecmult(MB, Q[1]);
      matvecmult(MB, Q[2]);
      matvecmult(MB, Q[3]);
   }
*/
   
   for (j=0; j<4; j++, i++)
      copyarray4(Q[j], in[i]);

   if (stype == M_CR)                           // Catmull-Rom spline
      matmatmult(MCR, Q);
   else
      matmatmult(MB, Q);

   T[3] = 1;
   T[2] = s;
   T[1] = T[2]*T[2];
   T[0] = T[1]*T[2];

   out[0] = T[0]*Q[0][0] + T[1]*Q[1][0] + T[2]*Q[2][0] + T[3]*Q[3][0];
   out[1] = T[0]*Q[0][1] + T[1]*Q[1][1] + T[2]*Q[2][1] + T[3]*Q[3][1];
   out[2] = T[0]*Q[0][2] + T[1]*Q[1][2] + T[2]*Q[2][2] + T[3]*Q[3][2];
   out[3] = T[0]*Q[0][3] + T[1]*Q[1][3] + T[2]*Q[2][3] + T[3]*Q[3][3];
}


/* *************************************************************
   This procedure calculates the current placement of objects
   using Catmull-Rom or B splines
************************************************************* */
void splinet(float s, vector4f *in, float *out, int stype) {

   vector4f Q[4];
   vector4f T;

/*
   transpose(Q, in);

   if (stype == M_CR) {                           // Catmull-Rom spline
      matvecmult(MCR, Q[0]);
      matvecmult(MCR, Q[1]);
      matvecmult(MCR, Q[2]);
      matvecmult(MCR, Q[3]);
   }

   else {
      matvecmult(MB, Q[0]);                                                // B-Spline
      matvecmult(MB, Q[1]);
      matvecmult(MB, Q[2]);
      matvecmult(MB, Q[3]);
   }
*/

   copymx4x4(Q, in);

   if (stype == M_CR)                            // Catmull-Rom spline
      matmatmult(MCR, Q);
   else
      matmatmult(MB, Q);                                                // B-Spline
   
   T[3] = 1;
   T[2] = s;
   T[1] = T[2]*T[2];
   T[0] = T[1]*T[2];

   out[0] = T[0]*Q[0][0] + T[1]*Q[1][0] + T[2]*Q[2][0] + T[3]*Q[3][0];
   out[1] = T[0]*Q[0][1] + T[1]*Q[1][1] + T[2]*Q[2][1] + T[3]*Q[3][1];
   out[2] = T[0]*Q[0][2] + T[1]*Q[1][2] + T[2]*Q[2][2] + T[3]*Q[3][2];
   out[3] = T[0]*Q[0][3] + T[1]*Q[1][3] + T[2]*Q[2][3] + T[3]*Q[3][3];
}


/* *************************************************************
   This procedure calculates the current placement of objects
   using Catmull-Rom or B splines
************************************************************* */
void spline(float s, float Q[][4], float *out, int stype) {

   vector4f T;
   vector4f temp[4];

/*
   copymx4x4(temp, Q);

   if (stype == M_CR) {                           // Catmull-Rom spline
      matvecmult(MCR, temp[0]);
      matvecmult(MCR, temp[1]);
      matvecmult(MCR, temp[2]);
      matvecmult(MCR, temp[3]);
   }

   else {
      matvecmult(MB, temp[0]);                                                // B-Spline
      matvecmult(MB, temp[1]);
      matvecmult(MB, temp[2]);
      matvecmult(MB, temp[3]);
   }
*/
   
   transpose(temp, Q);

   if (stype == M_CR)                            // Catmull-Rom spline
      matmatmult(MCR, temp);
   else
      matmatmult(MB, temp);                                                // B-Spline

   T[3] = 1;
   T[2] = s;
   T[1] = T[2]*T[2];
   T[0] = T[1]*T[2];

   out[0] = T[0]*temp[0][0] + T[1]*temp[1][0] + T[2]*temp[2][0] + T[3]*temp[3][0];
   out[1] = T[0]*temp[0][1] + T[1]*temp[1][1] + T[2]*temp[2][1] + T[3]*temp[3][1];
   out[2] = T[0]*temp[0][2] + T[1]*temp[1][2] + T[2]*temp[2][2] + T[3]*temp[3][2];
   out[3] = T[0]*temp[0][3] + T[1]*temp[1][3] + T[2]*temp[2][3] + T[3]*temp[3][3];
}

