   

#include "matrix.h"


/* *************************************************************
   Quaternion multiplication
************************************************************* */
void quatquatmult(vector4f y, vector4f x) {

   float *x24, *y24;               // complex part of quaternion
   vector4f ans;

   x24 = &x[1];
   y24 = &y[1];

   ans[0] = y[0]*x[0] - dotproduct3(y24, x24);         // calc real part

   xproduct(&ans[1], y24, x24);                   // calc complex part

   ans[1] += y[0]*x24[0] + x[0]*y24[0];
   ans[2] += y[0]*x24[1] + x[0]*y24[1];
   ans[3] += y[0]*x24[2] + x[0]*y24[2];

   copyarray4(x, ans);
}


/* *************************************************************
   convert euler to quaternion
************************************************************* */
void euler2quat(vector3f angle, vector4f quat) {

   vector3f s, c;
   float temp;
   float c01, s01, c0s1, c1s0;

   temp = -angle[0]*0.5f;
   s[0] = SIN(temp);
   c[0] = COS(temp);

   temp = -angle[1]*0.5f;
   s[1] = SIN(temp);
   c[1] = COS(temp);

   temp = -angle[2]*0.5f;
   s[2] = SIN(temp);
   c[2] = COS(temp);

   c01 = c[0]*c[1];
   s01 = s[0]*s[1];
   c0s1 = c[0]*s[1];
   c1s0 = c[1]*s[0];

   quat[0] = c01*c[2] + s01*s[2];
   quat[1] = c1s0*c[2] - c0s1*s[2];
   quat[2] = c0s1*c[2] + c1s0*s[2];
   quat[3] = c01*s[2] - s01*c[2];
}


/* *************************************************************
   convert matrix to quaternion
************************************************************* */
void mat2quat(vector4f *mx, vector4f quat) {

   float trace, s;
   int i, j, k;

   int nxt[3] = {1, 2, 0};

   trace = mx[0][0] + mx[1][1] + mx[2][2];

   // check the diagonal
   if (trace > 0.0) {
      s = (float)sqrt(trace+1.0f);
      quat[0] = s * 0.5f;
      s = 0.5f/s;
      quat[1] = (mx[1][2] - mx[2][1])*s;
      quat[2] = (mx[2][0] - mx[0][2])*s;
      quat[3] = (mx[0][1] - mx[1][0])*s;
      return;
   }

   i = (mx[1][1] > mx[0][0]);
   i = (mx[2][2] > mx[i][i]) ? 2 : 1;

   j = nxt[i];
   k = nxt[j];

   s = (float)sqrt((mx[i][i] - (mx[j][j] + mx[k][k])) + 1.0f);

   quat[i+1] = s*0.5f;

   if (s)
      s = 0.5f/s;

   quat[0]   = (mx[j][k] - mx[k][j])*s;
   quat[j+1] = (mx[i][j] + mx[j][i])*s;
   quat[k+1] = (mx[i][k] + mx[k][i])*s;
}


/* *************************************************************
   calculate euler rotation

	czcy	-szcx+czsysx	szsx+czsycx
	szcy	czcx+szsysx		-czsx+szsycx
	-sy		cysx			cycx

   mx = [Z] * [Y] * [X]
************************************************************* */
void calc_euler(float mx[][4], float current[]) {

	float cx, sx, cy, sy, cz, sz;
	float szcx, szsx, czcx, czsx;

	float *t;

	t = &mx[0][0];

	cx = COS(current[0]);
	sx = SIN(current[0]);

	cy = COS(current[1]);
	sy = SIN(current[1]);

	cz = COS(current[2]);
	sz = SIN(current[2]);

	szsx = sz*sx;
	czcx = cz*cx;
	szcx = sz*cx;
	czsx = cz*sx;

	t[0] = cz*cy;
	t[1] = -szcx + czsx*sy;
	t[2] = szsx + czcx*sy;

	t[4] = sz*cy;
	t[5] = czcx + szsx*sy;
	t[6] = -czsx + szcx*sy;

	t[8] = -sy;
	t[9] = cy*sx;
	t[10] = cy*cx;

	t[3] = t[7] = t[11] = t[12] = t[13] = t[14] = 0.0f;
	t[15] = 1.0f;
}


/* *************************************************************
   calculate quaternion mx
************************************************************* */
void calc_quat(float mx[][4], float current[]) {

   vector4f vect;
   register float t1, t2, v1, v2;

   copyarray4(vect, current);
   normalize4(vect);

   v1 = vect[1]*vect[2];
   v2 = vect[0]*vect[3];
   t1 = v1 + v2;
   t2 = v1 - v2;
   mx[0][1] = t1 + t1;
   mx[1][0] = t2 + t2;

   v1 = vect[1]*vect[3];
   v2 = vect[0]*vect[2];
   t1 = v1 - v2;
   t2 = v1 + v2;
   mx[0][2] = t1 + t1;
   mx[2][0] = t2 + t2;

   mx[0][3] = mx[1][3] = mx[2][3] = 0;

   v1 = vect[2]*vect[3];
   v2 = vect[0]*vect[1];
   t1 = v1 + v2;
   t2 = v1 - v2;
   mx[1][2] = t1 + t1;
   mx[2][1] = t2 + t2;

   vect[1] *= vect[1];
   vect[2] *= vect[2];
   vect[3] *= vect[3];

   t1 = vect[2] + vect[3];
   t2 = vect[1] + vect[3];
   mx[0][0] = 1 - (t1 + t1);
   mx[1][1] = 1 - (t2 + t2);

   t1 = vect[1] + vect[2];
   mx[2][2] = 1 - (t1 + t1);

   mx[3][0] = mx[3][1] = mx[3][2] = 0;
   mx[3][3] = 1;
}


/* *************************************************************
   Assumption: matrix is a pure rotation matrix

        mx = [X] * [Y] * [Z]

           = [   cycz,          -cysz,           sy
                 sxsycz+cysz    -sxsysz+cycz    -sxcy
                -cxsycz+sysz     cxsysz+sycz     cxcy   ]

************************************************************* */
void decompose_xyz(vector4f mx[4], float *x, float *y, float *z) {

   float temp, tempc, temps;

   if (fabs(mx[0][2]) > ICORRECT) {

      *y = (mx[0][2] > 0) ? (float)HALFPI : (float)(-HALFPI);
      *z = 0.0f;

      if ((tempc = mx[1][1]) > ICORRECT)
         *x = 0.0f;
      else if (tempc < -ICORRECT)
         *x = (float)PI;
      else if ((temps = mx[2][1]) > ICORRECT)
         *x = (float)HALFPI;
      else if (temps < -ICORRECT)
         *x = (float)(-HALFPI);
      else
         *x = (float)atan2(temps, tempc);

      return;
   }

   *y = ASIN(mx[0][2]);
   temp = 1.0f/COS(*y);

   if ((tempc = mx[0][0]*temp) > ICORRECT)
      *z = 0.0f;
   else if  (tempc < -ICORRECT)
      *z = (float)PI;
   else if ((temps = -mx[0][1]*temp) > ICORRECT)
      *z = (float)HALFPI;
   else if (temps < -ICORRECT)
      *z = (float)-HALFPI;
   else
      *z = (float)atan2(temps, tempc);

   if ((tempc = mx[2][2]*temp) > ICORRECT)
      *x = 0.0f;
   else if (tempc < -ICORRECT)
      *x = (float)PI;
   else if ((temps = -mx[1][2]*temp) > ICORRECT)
      *x = (float)HALFPI;
   else if (temps < -ICORRECT)
      *x = (float)(-HALFPI);
   else
      *x = (float)atan2(temps, tempc);
}


/* *************************************************************
   Assumption: matrix is a pure rotation matrix

        mx = [Y] * [X] * [Z]
	
           = [    cycz+sysxsz  -cysz+sysxcz    sycx
                  cxsz          cxcz          -sx
                 -sycz+cysxsz   sysz+cysxcz    cycx    ]

************************************************************* */
void decompose_yxz(vector4f mx[4], float *x, float *y, float *z) {

   float temp, tempc, temps;

   if (fabs(mx[1][2]) > ICORRECT) {

      *x =  (mx[1][2] < 0) ? (float)HALFPI : (float)(-HALFPI);
      *z = 0.0f;

      if ((tempc = mx[0][0]) > ICORRECT)
         *y = 0.0f;
      else if (tempc < -ICORRECT)
         *y = (float)PI;
      else if ((temps = -mx[2][0]) > ICORRECT)
         *y = (float)HALFPI;
      else if (temps < -ICORRECT)
         *y = (float)(-HALFPI);
      else
         *y = (float)atan2(temps, tempc);

      return;
   }

   *x = ASIN(-mx[1][2]);
   temp = 1.0f/COS(*x);

   if ((tempc = mx[1][1]*temp) > ICORRECT)
      *z = 0.0f;
   else if  (tempc < -ICORRECT)
      *z = (float)PI;
   else if ((temps = mx[1][0]*temp) > ICORRECT)
      *z = (float)HALFPI;
   else if (temps < -ICORRECT)
      *z = (float)(-HALFPI);
   else
      *z = (float)atan2(temps, tempc);

   if ((tempc = mx[2][2]*temp) > ICORRECT)
      *y = 0.0f;
   else if (tempc < -ICORRECT)
      *y = (float)PI;
   else if ((temps = mx[0][2]*temp) > ICORRECT)
      *y = (float)HALFPI;
   else if (temps < -ICORRECT)
      *y = (float)(-HALFPI);
   else
      *y = (float)atan2(temps, tempc);
}


/* *************************************************************
   Assumption: matrix is a pure rotation matrix

        mx = [Z] * [X] * [Y]

           = [  czcy-szsxsy    -szcx        czsy+szsxcy
                szcy+czsxsy     czcx        szsy-czsxcy
               -cxsy            sx          cxcy        ]

************************************************************* */
void decompose_zxy(vector4f mx[4], float *x, float *y, float *z) {

   float temp, tempc, temps;

   if (fabs(mx[2][1]) > ICORRECT) {
      *x = (mx[2][1] > 0) ? (float)HALFPI : (float)(-HALFPI);
      *z = 0.0f;

      if ((tempc = mx[0][0]) > ICORRECT)
         *y = 0.0f;
      else if (tempc < -ICORRECT)
         *y = (float)PI;
      else if ((temps = mx[0][2]) > ICORRECT)
         *y = (float)HALFPI;
      else if (temps < -ICORRECT)
         *y = (float)(-HALFPI);
      else
         *y = (float)atan2(temps, tempc);

      return;
   }

   *x = ASIN(mx[2][1]);
   temp = 1.0f/COS(*x);

   if ((tempc = mx[1][1]*temp) > ICORRECT)
      *z = 0.0f;
   else if  (tempc < -ICORRECT)
      *z = (float)PI;
   else if ((temps = -mx[0][1]*temp) > ICORRECT)
      *z = (float)HALFPI;
   else if (temps < -ICORRECT)
      *z = (float)-HALFPI;
   else
      *z = (float)atan2(temps, tempc);

   if ((tempc = mx[2][2]*temp) > ICORRECT)
      *y = 0.0f;
   else if (tempc < -ICORRECT)
      *y = (float)PI;
   else if ((temps = -mx[2][0]*temp) > ICORRECT)
      *y = (float)HALFPI;
   else if (temps < -ICORRECT)
      *y = (float)(-HALFPI);
   else
      *y = (float)atan2(temps, tempc);
}


/* *************************************************************
   Assumption: matrix is a pure rotation matrix

        mx = [Z] * [X] * [Y]

           = [  czcy-szsxsy    -szcx        czsy+szsxcy
                szcy+czsxsy     czcx        szsy-czsxcy
               -cxsy            sx          cxcy        ]

************************************************************* */
void decompose_zxyd(vector4d mx[4], double *x, double *y, double *z) {

   double temp, tempc, temps;

   if (fabs(mx[2][1]) > ICORRECT) {
      *x = (mx[2][1] > 0) ? HALFPI : -HALFPI;
      *z = 0.0;

      if ((tempc = mx[0][0]) > ICORRECT)
         *y = 0.0;
      else if (tempc < -ICORRECT)
         *y = PI;
      else if ((temps = mx[0][2]) > ICORRECT)
         *y = HALFPI;
      else if (temps < -ICORRECT)
         *y = -HALFPI;
      else
         *y = atan2(temps, tempc);

      return;
   }

   *x = ASIN(mx[2][1]);
   temp = 1.0/COS(*x);

   if ((tempc = mx[1][1]*temp) > ICORRECT)
      *z = 0.0;
   else if  (tempc < -ICORRECT)
      *z = PI;
   else if ((temps = -mx[0][1]*temp) > ICORRECT)
      *z = HALFPI;
   else if (temps < -ICORRECT)
      *z = -HALFPI;
   else
      *z = atan2(temps, tempc);

   if ((tempc = mx[2][2]*temp) > ICORRECT)
      *y = 0.0;
   else if (tempc < -ICORRECT)
      *y = PI;
   else if ((temps = -mx[2][0]*temp) > ICORRECT)
      *y = HALFPI;
   else if (temps < -ICORRECT)
      *y = -HALFPI;
   else
      *y = atan2(temps, tempc);
}


/* *************************************************************
   Assumption: matrix is a pure rotation matrix

        mx = [Z] * [Y] * [X]

           = [   cycz    czsxsy - szcx   czcxsy + szsx
                 szcy    szsxsy + czcx   szcxsy - czsx
                -sy      sxcy            cxcy            ]

************************************************************* */
void decompose_zyx(vector4f mx[4], float *x, float *y, float *z) {

   float temp, tempc, temps;

   if (fabs(mx[2][0]) > ICORRECT) {

      *y = (-mx[2][0] > 0) ? (float)HALFPI : (float)(-HALFPI);
      *z = 0.0f;

      if ((tempc = mx[1][1]) > ICORRECT)
         *x = 0.0f;
      else if (tempc < -ICORRECT)
         *x = (float)PI;
      else if ((temps = -mx[1][2]) > ICORRECT)
         *x = (float)HALFPI;
      else if (temps < -ICORRECT)
         *x = (float)(-HALFPI);
      else
         *x = (float)atan2(temps, tempc);

      return;
   }

   *y = ASIN(-mx[2][0]);
   temp = 1.0f/COS(*y);

   if ((tempc = mx[0][0]*temp) > ICORRECT)
      *z = 0.0f;
   else if  (tempc < -ICORRECT)
      *z = (float)PI;
   else if ((temps = mx[1][0]*temp) > ICORRECT)
      *z = (float)HALFPI;
   else if (temps < -ICORRECT)
      *z = (float)(-HALFPI);
   else
      *z = (float)atan2(temps, tempc);

   if ((tempc = mx[2][2]*temp) > ICORRECT)
      *x = 0.0f;
   else if (tempc < -ICORRECT)
      *x = (float)PI;
   else if ((temps = mx[2][1]*temp) > ICORRECT)
      *x = (float)HALFPI;
   else if (temps < -ICORRECT)
      *x = (float)(-HALFPI);
   else
      *x = (float)atan2(temps, tempc);
}


/* *************************************************************
   Assumption: matrix is a pure rotation matrix

        mx = [Z] * [Y] * [X]

           = [   cycz    czsxsy - szcx   czcxsy + szsx
                 szcy    szsxsy + czcx   szcxsy - czsx
                -sy      sxcy            cxcy            ]

************************************************************* */
void decompose_zyxd(vector4d mx[4], double *x, double *y, double *z) {

   double temp, tempc, temps;

   if (fabs(mx[2][0]) > ICORRECT) {

      *y = (-mx[2][0] > 0) ? HALFPI : -HALFPI;
      *z = 0.0;

      if ((tempc = mx[1][1]) > ICORRECT)
         *x = 0.0;
      else if (tempc < -ICORRECT)
         *x = PI;
      else if ((temps = -mx[1][2]) > ICORRECT)
         *x = HALFPI;
      else if (temps < -ICORRECT)
         *x = -HALFPI;
      else
         *x = atan2(temps, tempc);

      return;
   }

   *y = ASIN(-mx[2][0]);
   temp = 1.0/COS(*y);

   if ((tempc = mx[0][0]*temp) > ICORRECT)
      *z = 0.0;
   else if  (tempc < -ICORRECT)
      *z = PI;
   else if ((temps = mx[1][0]*temp) > ICORRECT)
      *z = HALFPI;
   else if (temps < -ICORRECT)
      *z = -HALFPI;
   else
      *z = atan2(temps, tempc);

   if ((tempc = mx[2][2]*temp) > ICORRECT)
      *x = 0.0;
   else if (tempc < -ICORRECT)
      *x = PI;
   else if ((temps = mx[2][1]*temp) > ICORRECT)
      *x = HALFPI;
   else if (temps < -ICORRECT)
      *x = -HALFPI;
   else
      *x = atan2(temps, tempc);
}


/* *************************************************************
************************************************************* */
void limit_pitch(float *roll, float *pitch, float *yaw) {

   if (*pitch < HALFPI + CORRECT) {
      if (*pitch > -HALFPI - CORRECT)
         return;

      *pitch = -PI - *pitch;
   }

   else
      *pitch = PI - *pitch;

   *yaw += PI;
   *roll += PI;
}


/* *************************************************************
************************************************************* */
void limit_pitchd(double *roll, double *pitch, double *yaw) {

   if (*pitch < HALFPI + CORRECT) {
      if (*pitch > -HALFPI - CORRECT)
         return;

      *pitch = -PI - *pitch;
   }

   else
      *pitch = PI - *pitch;

   *yaw += PI;
   *roll += PI;
}


/* *************************************************************
   Assumption: matrix is a pure rotation matrix

        roll -  x
        pitch - y
        yaw -   z

************************************************************* */
void decompose_xrypzy(vector4f mx[4], float *roll, float *pitch, float *yaw) {

   decompose_xyz(mx, roll, pitch, yaw);
   limit_pitch(roll, pitch, yaw);
}


/* *************************************************************
   Assumption: matrix is a pure rotation matrix

        roll -  y
        pitch - x
        yaw -   z

************************************************************* */
void decompose_yrxpzy(vector4f mx[4], float *roll, float *pitch, float *yaw) {

   decompose_yxz(mx, pitch, roll, yaw);
   limit_pitch(roll, pitch, yaw);
}


/* *************************************************************
   Assumption: matrix is a pure rotation matrix

        roll -  x
        pitch - y
        yaw -   z

************************************************************* */
void decompose_ypxrzy(vector4f mx[4], float *roll, float *pitch, float *yaw) {

   decompose_yxz(mx, roll, pitch, yaw);
   limit_pitch(roll, pitch, yaw);
}


/* *************************************************************
   Assumption: matrix is a pure rotation matrix

        pitch - x
        yaw   - y
        roll  - z

************************************************************* */
void decompose_yyxpzr(vector4f mx[4], float *roll, float *pitch, float *yaw) {

   decompose_yxz(mx, pitch, yaw, roll);
   limit_pitch(roll, pitch, yaw);
}


/* *************************************************************
   Assumption: matrix is a pure rotation matrix

        roll -  x
        pitch - y
        yaw -   z

************************************************************* */
void decompose_zyxryp(vector4f mx[4], float *roll, float *pitch, float *yaw) {

   decompose_zxy(mx, roll, pitch, yaw);
   limit_pitch(roll, pitch, yaw);
}


/* *************************************************************
   Assumption: matrix is a pure rotation matrix

        pitch - x       axis from back -> front
        roll  - y       axis from 9 oclock -> 3 oclock (looking from top)
        yaw   - z       axis represents -height of object

************************************************************* */
void decompose_zyxpyr(vector4f mx[4], float *roll, float *pitch, float *yaw) {

   decompose_zxy(mx, pitch, roll, yaw);
   limit_pitch(roll, pitch, yaw);
}


/* *************************************************************
   Assumption: matrix is a pure rotation matrix

        pitch - x       axis from back -> front
        roll  - y       axis from 9 oclock -> 3 oclock (looking from top)
        yaw   - z       axis represents -height of object

************************************************************* */
void decompose_zyxpyrd(vector4d mx[4], double *roll, double *pitch, double *yaw) {

   decompose_zxyd(mx, pitch, roll, yaw);
   limit_pitchd(roll, pitch, yaw);
}


/* *************************************************************
   Assumption: matrix is a pure rotation matrix

        roll -  x       axis from back -> front
        pitch - y       axis from 9 oclock -> 3 oclock (looking from top)
        yaw -   z       axis represents -height of object

************************************************************* */
void decompose_zyypxr(vector4f mx[4], float *roll, float *pitch, float *yaw) {

   decompose_zyx(mx, roll, pitch, yaw);
   limit_pitch(roll, pitch, yaw);
}


/* *************************************************************
   Assumption: matrix is a pure rotation matrix

        roll -  x       axis from back -> front
        pitch - y       axis from 9 oclock -> 3 oclock (looking from top)
        yaw -   z       axis represents -height of object

************************************************************* */
void decompose_zyypxrd(vector4d mx[4], double *roll, double *pitch, double *yaw) {

   decompose_zyxd(mx, roll, pitch, yaw);
   limit_pitchd(roll, pitch, yaw);
}


/* *************************************************************
        [x] * mx
************************************************************* */
void rotate_mx_x(vector4f mx[4], float x) {

   register float cx, sx;
   vector4f temp1, temp2;

   copyarray4(temp1, mx[1]);
   copyarray4(temp2, mx[2]);

#ifdef REAL_FAST
   cx = COS(x);
   sx = SIN(x);
#else
   cx = (float)cos(x);
   sx = (float)sin(x);
#endif

   mx[1][0] = cx*temp1[0] - sx*temp2[0];
   mx[1][1] = cx*temp1[1] - sx*temp2[1];
   mx[1][2] = cx*temp1[2] - sx*temp2[2];
   mx[1][3] = cx*temp1[3] - sx*temp2[3];

   mx[2][0] = sx*temp1[0] + cx*temp2[0];
   mx[2][1] = sx*temp1[1] + cx*temp2[1];
   mx[2][2] = sx*temp1[2] + cx*temp2[2];
   mx[2][3] = sx*temp1[3] + cx*temp2[3];
}


/* *************************************************************
        [x] * mx
************************************************************* */
void rotate_mx_xd(vector4d mx[4], double x) {

   double cx, sx;
   vector4d temp1, temp2;

   copyarray4(temp1, mx[1]);
   copyarray4(temp2, mx[2]);

#ifdef REAL_FAST
   cx = COS(x);
   sx = SIN(x);
#else
   cx = cos(x);
   sx = sin(x);
#endif

   mx[1][0] = cx*temp1[0] - sx*temp2[0];
   mx[1][1] = cx*temp1[1] - sx*temp2[1];
   mx[1][2] = cx*temp1[2] - sx*temp2[2];
   mx[1][3] = cx*temp1[3] - sx*temp2[3];

   mx[2][0] = sx*temp1[0] + cx*temp2[0];
   mx[2][1] = sx*temp1[1] + cx*temp2[1];
   mx[2][2] = sx*temp1[2] + cx*temp2[2];
   mx[2][3] = sx*temp1[3] + cx*temp2[3];
}


/* *************************************************************
        mx * [x]
************************************************************* */
void arotate_mx_x(vector4f mx[4], float x) {

   vector4f r1, r2;
   float cx, sx;

   r1[0] = mx[0][1]; r2[0] = mx[0][2];
   r1[1] = mx[1][1]; r2[1] = mx[1][2];
   r1[2] = mx[2][1]; r2[2] = mx[2][2];
   r1[3] = mx[3][1]; r2[3] = mx[3][2];

#ifdef REAL_FAST
   cx = COS(x);
   sx = SIN(x);
#else
   cx = (float)cos(x);
   sx = (float)sin(x);
#endif

   mx[0][1] =  cx*r1[0] + sx*r2[0];
   mx[0][2] = -sx*r1[0] + cx*r2[0];

   mx[1][1] =  cx*r1[1] + sx*r2[1];
   mx[1][2] = -sx*r1[1] + cx*r2[1];

   mx[2][1] =  cx*r1[2] + sx*r2[2];
   mx[2][2] = -sx*r1[2] + cx*r2[2];

   mx[3][1] =  cx*r1[3] + sx*r2[3];
   mx[3][2] = -sx*r1[3] + cx*r2[3];
}


/* *************************************************************
************************************************************* */
void rotate_mx_y(vector4f mx[4], float y) {

   vector4f temp0, temp2;
   register float cy, sy;

   copyarray4(temp0, mx[0]);
   copyarray4(temp2, mx[2]);

#ifdef REAL_FAST
   cy = COS(y);
   sx = SIN(y);
#else
   cy = (float)cos(y);
   sy = (float)sin(y);
#endif

   mx[0][0] = cy*temp0[0] + sy*temp2[0];
   mx[0][1] = cy*temp0[1] + sy*temp2[1];
   mx[0][2] = cy*temp0[2] + sy*temp2[2];
   mx[0][3] = cy*temp0[3] + sy*temp2[3];

   mx[2][0] = -sy*temp0[0] + cy*temp2[0];
   mx[2][1] = -sy*temp0[1] + cy*temp2[1];
   mx[2][2] = -sy*temp0[2] + cy*temp2[2];
   mx[2][3] = -sy*temp0[3] + cy*temp2[3];
}


/* *************************************************************
************************************************************* */
void rotate_mx_yd(vector4d mx[4], double y) {

   vector4d temp0, temp2;
   double cy, sy;

   copyarray4(temp0, mx[0]);
   copyarray4(temp2, mx[2]);

#ifdef REAL_FAST
   cy = COS(y);
   sx = SIN(y);
#else
   cy = cos(y);
   sy = sin(y);
#endif

   mx[0][0] = cy*temp0[0] + sy*temp2[0];
   mx[0][1] = cy*temp0[1] + sy*temp2[1];
   mx[0][2] = cy*temp0[2] + sy*temp2[2];
   mx[0][3] = cy*temp0[3] + sy*temp2[3];

   mx[2][0] = -sy*temp0[0] + cy*temp2[0];
   mx[2][1] = -sy*temp0[1] + cy*temp2[1];
   mx[2][2] = -sy*temp0[2] + cy*temp2[2];
   mx[2][3] = -sy*temp0[3] + cy*temp2[3];
}


/* *************************************************************
************************************************************* */
void arotate_mx_y(vector4f mx[4], float y) {

   vector4f r0, r2;
   float cy, sy;

   r0[0] = mx[0][0]; r2[0] = mx[0][2];
   r0[1] = mx[1][0]; r2[1] = mx[1][2];
   r0[2] = mx[2][0]; r2[2] = mx[2][2];
   r0[3] = mx[3][0]; r2[3] = mx[3][2];

#ifdef REAL_FAST
   cy = COS(y);
   sx = SIN(y);
#else
   cy = (float)cos(y);
   sy = (float)sin(y);
#endif

   mx[0][0] =  cy*r0[0] - sy*r2[0];
   mx[0][2] =  sy*r0[0] + cy*r2[0];

   mx[1][0] =  cy*r0[1] - sy*r2[1];
   mx[1][2] =  sy*r0[1] + cy*r2[1];

   mx[2][0] =  cy*r0[2] - sy*r2[2];
   mx[2][2] =  sy*r0[2] + cy*r2[2];

   mx[3][0] =  cy*r0[3] - sy*r2[3];
   mx[3][2] =  sy*r0[3] + cy*r2[3];
}


/* *************************************************************
************************************************************* */
void rotate_mx_z(vector4f mx[4], float z) {

   vector4f temp0, temp1;
   register float cz, sz;

   copyarray4(temp0, mx[0]);
   copyarray4(temp1, mx[1]);

#ifdef REAL_FAST
   cz = COS(z);
   sz = SIN(z);
#else
   cz = (float)cos(z);
   sz = (float)sin(z);
#endif

   mx[0][0] = cz*temp0[0] - sz*temp1[0];
   mx[0][1] = cz*temp0[1] - sz*temp1[1];
   mx[0][2] = cz*temp0[2] - sz*temp1[2];
   mx[0][3] = cz*temp0[3] - sz*temp1[3];

   mx[1][0] = sz*temp0[0] + cz*temp1[0];
   mx[1][1] = sz*temp0[1] + cz*temp1[1];
   mx[1][2] = sz*temp0[2] + cz*temp1[2];
   mx[1][3] = sz*temp0[3] + cz*temp1[3];
}


/* *************************************************************
************************************************************* */
void rotate_mx_zd(vector4d mx[4], double z) {

   vector4d temp0, temp1;
   double cz, sz;

   copyarray4(temp0, mx[0]);
   copyarray4(temp1, mx[1]);

#ifdef REAL_FAST
   cz = COS(z);
   sz = SIN(z);
#else
   cz = cos(z);
   sz = sin(z);
#endif

   mx[0][0] = cz*temp0[0] - sz*temp1[0];
   mx[0][1] = cz*temp0[1] - sz*temp1[1];
   mx[0][2] = cz*temp0[2] - sz*temp1[2];
   mx[0][3] = cz*temp0[3] - sz*temp1[3];

   mx[1][0] = sz*temp0[0] + cz*temp1[0];
   mx[1][1] = sz*temp0[1] + cz*temp1[1];
   mx[1][2] = sz*temp0[2] + cz*temp1[2];
   mx[1][3] = sz*temp0[3] + cz*temp1[3];
}


/* *************************************************************
************************************************************* */
void arotate_mx_z(vector4f mx[4], float z) {

   vector4f r0, r1;
   float cz, sz;

   r0[0] = mx[0][0]; r1[0] = mx[0][1];
   r0[1] = mx[1][0]; r1[1] = mx[1][1];
   r0[2] = mx[2][0]; r1[2] = mx[2][1];
   r0[3] = mx[3][0]; r1[3] = mx[3][1];

#ifdef REAL_FAST
   cz = COS(z);
   sz = SIN(z);
#else
   cz = (float)cos(z);
   sz = (float)sin(z);
#endif

   mx[0][0] =  cz*r0[0] + sz*r1[0];
   mx[0][1] = -sz*r0[0] + cz*r1[0];

   mx[1][0] =  cz*r0[1] + sz*r1[1];
   mx[1][1] = -sz*r0[1] + cz*r1[1];

   mx[2][0] =  cz*r0[2] + sz*r1[2];
   mx[2][1] = -sz*r0[2] + cz*r1[2];

   mx[3][0] =  cz*r0[3] + sz*r1[3];
   mx[3][1] = -sz*r0[3] + cz*r1[3];
}


/* *************************************************************
   Assume "axis" is normalized

        axis  = z axis in world coords
        start = x axis in local coords
************************************************************* */
void rotate_vector(float *start, float *axis, float rad, float *sol) {

/*
   vector4f mx[4], r[4];

   init_mx(mx);

   xproduct(mx[1], axis, start);
   normalize3(mx[1]);
   copyarray3(mx[2], axis);
   normalize3(mx[2]);
   xproduct(mx[0], mx[1], mx[2]);
   transpose(mx);

   init_mx(r);
   rotate_mx_z(r, rad);

   matmatmulto(mx, r);

   sol[0] = r[0][0];
   sol[1] = r[1][0];
   sol[2] = r[2][0];
*/

   vector4f mx[4];
   vector4f pos = { 0.0, 0.0, 0.0, 1.0 };

   calc_axis_rotate_mx(pos, axis, rad, mx);
   matvecmultv(mx, start, sol);
}


/* *************************************************************
   NOTE: this fails at 180 degrees (quaternion generated
         quat[1] = quat[2] = quat[3] = 0)
   Possible Solution: examine matrix to figure out which rotation axis
************************************************************* */
void scale_rotation(vector4f *mx, float scale) {

   vector4f quat;
   float angle;

   mat2quat(mx, quat);

   // little or no rotation
   if (fabs(quat[0]) > 1-CORRECT)   // angle = 0;
      return;

   angle = ACOS(quat[0]) * scale;

   quat[0] = COS(angle);

   angle = (float)(SIN(angle)/sqrt(quat[1]*quat[1] + quat[2]*quat[2] + quat[3]*quat[3]));
   quat[1] *= angle;
   quat[2] *= angle;
   quat[3] *= angle;

   calc_quat(mx, quat);
}


/* *************************************************************

I = 3x3 Identity mx

w = unit arbitrary axis/vector

w x w = 3x3 matrix where mx[i][j] = w[i]*w[j]

W = { 0    -w[2]  w[1],
      w[2]  0    -w[0],
     -w[1]  w[0]  0 }

M = cos(@)*I + (1-cos(@))*(w x w) + sin(@) * W

************************************************************* */
void calc_axis_rotate_mx(float *start, float *axis, float radian, vector4f *mx) {

   float c     = (float)cos(radian);
   float s     = (float)sin(radian);
   float ic    = 1.0f - c;
   vector3f sa = {s*axis[0], s*axis[1], s*axis[2]};

   mx[0][0] = ic*axis[0]*axis[0];
   mx[1][0] = mx[0][1] = ic*axis[0]*axis[1];
   mx[2][0] = mx[0][2] = ic*axis[0]*axis[2];

   mx[1][1] = ic*axis[1]*axis[1];
   mx[2][1] = mx[1][2] = ic*axis[1]*axis[2];

   mx[2][2] = ic*axis[2]*axis[2];

   mx[0][0] += c;
   mx[0][1] -= sa[2];
   mx[0][2] += sa[1];

   mx[1][0] += sa[2];
   mx[1][1] += c;
   mx[1][2] -= sa[0];

   mx[2][0] -= sa[1];
   mx[2][1] += sa[0];
   mx[2][2] += c;

   mx[0][3] = start[0] - dotproduct3(mx[0], start);
   mx[1][3] = start[1] - dotproduct3(mx[1], start);
   mx[2][3] = start[2] - dotproduct3(mx[2], start);

   mx[3][0] = mx[3][1] = mx[3][2] = 0;
   mx[3][3] = 1;
}
