

/* *************************************************************
   This file handles all the matrix/vector/math operations

   Reminder: Spherical Linear intERPolation (SLERP) can be
             done by splining the quaternions

************************************************************* */


#include "matrix.h"

// used for float to int conversion
int flt2int[2] = { 0, 0x43380000 };
int flt2fx[2]  = { 0, 0x42380000 };


/* *************************************************************
************************************************************* */
int similar3(float *x, float *y) {

   vector3f t;
   
   subeqarray3(t, x, y);
   return dotproduct3(t, t) < CORRECT;
}


/* *************************************************************
************************************************************* */
int similar2(float *x, float *y) {

   vector2f t;
   
   subeqarray2(t, x, y);
   return dotproduct2(t, t) < CORRECT;
}


/* *************************************************************
   This procedure normalizes a vector and returns its magnitude
************************************************************* */
float normalize2(float *x) {

   register float mag, imag;

   mag = (float)magnitude2(x);

   if (mag) {
      imag = 1.0f/mag;
      x[0] *= imag;
      x[1] *= imag;
   }

   return mag;
}


/* *************************************************************
   This procedure normalizes a vector and returns its magnitude
************************************************************* */
float normalize3(float *x) {

   register float mag, imag;

   mag = (float)magnitude3(x);

   if (mag) {
      imag = 1.0f/mag;
      smultarray3(x, imag);
   }

   return mag;
}


/* *************************************************************
   This procedure normalizes a 4D vector and returns its magnitude
************************************************************* */
float normalize4(float *x) {

   register float mag, imag;

   mag = (float)magnitude4(x);

   if (mag) {
      imag = 1.0f/mag;
      smultarray4(x, imag);
   }

   return mag;
}


/* *************************************************************
   This procedure normalizes a vector and returns its magnitude
************************************************************* */
double normalize3d(double *x) {

   double mag, imag;

   mag = magnitude3(x);

   if (mag) {
      imag = 1.0f/mag;
      smultarray3(x, imag);
   }

   return mag;
}


/* *************************************************************
   This procedure normalizes a vector and returns its magnitude
************************************************************* */
void xproduct(float *a, float *x, float *y) {

   a[0] = x[1]*y[2] - x[2]*y[1];
   a[1] = x[2]*y[0] - x[0]*y[2];
   a[2] = x[0]*y[1] - x[1]*y[0];
}


/* *************************************************************
   This procedure normalizes a vector and returns its magnitude
************************************************************* */
void xproductd(double *a, double *x, double *y) {

   a[0] = x[1]*y[2] - x[2]*y[1];
   a[1] = x[2]*y[0] - x[0]*y[2];
   a[2] = x[0]*y[1] - x[1]*y[0];
}


/* *************************************************************
   this function calculates the transpose of a matrix
************************************************************* */
void transpose(vector4f *r, vector4f *s) {

   register int i;

   for (i=0; i<4; i++) {
      r[0][i] = s[i][0];
      r[1][i] = s[i][1];
      r[2][i] = s[i][2];
      r[3][i] = s[i][3];
   }
   
}


/* *************************************************************
   this function calculates the transpose of a matrix
************************************************************* */
void transposed(vector4d *r, vector4d *s) {

   register int i;

   for (i=0; i<4; i++) {
      r[0][i] = s[i][0];
      r[1][i] = s[i][1];
      r[2][i] = s[i][2];
      r[3][i] = s[i][3];
   }
   
}


/* *************************************************************
   this function calculates the transpose of a matrix
************************************************************* */
void transpose(vector4f *r) {

   register int   i, j;
   register float s;

   for (i=0; i<4; i++)
      for (j=0; j<i; j++) {
         s = r[j][i];
         r[j][i] = r[i][j];
         r[i][j] = s;
      }

}


/* *************************************************************
   this function calculates the transpose of a matrix
************************************************************* */
void transposed(vector4d *r) {

   register int   i, j;
   register double s;

   for (i=0; i<4; i++)
      for (j=0; j<i; j++) {
         s = r[j][i];
         r[j][i] = r[i][j];
         r[i][j] = s;
      }

}


/* *************************************************************
   This procedure multiplies 2 4X4 matricies
************************************************************* */
void matmatmult(vector4f *y, vector4f *x) {

   vector4f z[4];
   register int i, j;

   copymx4x4(z, x);

   for (i=0; i<4; i++)
      for (j=0; j<4; j++)
         x[i][j] = y[i][0]*z[0][j] + y[i][1]*z[1][j] + y[i][2]*z[2][j] + y[i][3]*z[3][j];
}


/* *************************************************************
   This procedure multiplies 2 4X4 matricies
************************************************************* */
void matmatmultd(vector4d *y, vector4d *x) {

   vector4d z[4];
   register int i, j;

   copymx4x4(z, x);

   for (i=0; i<4; i++)
      for (j=0; j<4; j++)
         x[i][j] = y[i][0]*z[0][j] + y[i][1]*z[1][j] + y[i][2]*z[2][j] + y[i][3]*z[3][j];
}


/* *************************************************************
   This procedure multiplies 2 4X4 matricies
************************************************************* */
void matmatmulto(vector4f *y, vector4f *x) {

   vector3f z0, z1, z2, z3;
   int i;

   z0[0] = x[0][0]; z0[1] = x[1][0]; z0[2] = x[2][0];
   z1[0] = x[0][1]; z1[1] = x[1][1]; z1[2] = x[2][1];
   z2[0] = x[0][2]; z2[1] = x[1][2]; z2[2] = x[2][2];
   z3[0] = x[0][3]; z3[1] = x[1][3]; z3[2] = x[2][3];
   
   for (i=0; i<3; i++) {
      x[i][0] = dotproduct3(y[i], z0);
      x[i][1] = dotproduct3(y[i], z1);
      x[i][2] = dotproduct3(y[i], z2);
      x[i][3] = dotproduct3(y[i], z3) + y[i][3];
   }

}


/* *************************************************************
   This procedure multiplies 2 4X4 matricies
************************************************************* */
void matmatmultod(vector4d *y, vector4d *x) {

   vector3d z0, z1, z2, z3;
   int i;

   z0[0] = x[0][0]; z0[1] = x[1][0]; z0[2] = x[2][0];
   z1[0] = x[0][1]; z1[1] = x[1][1]; z1[2] = x[2][1];
   z2[0] = x[0][2]; z2[1] = x[1][2]; z2[2] = x[2][2];
   z3[0] = x[0][3]; z3[1] = x[1][3]; z3[2] = x[2][3];
   
   for (i=0; i<3; i++) {
      x[i][0] = dotproduct3(y[i], z0);
      x[i][1] = dotproduct3(y[i], z1);
      x[i][2] = dotproduct3(y[i], z2);
      x[i][3] = dotproduct3(y[i], z3) + y[i][3];
   }

}


/* *************************************************************
   This procedure multiplies 2 4X4 matricies
************************************************************* */
void matmatmulto(vector4f *y, vector4f *x, vector4f *dst) {

   int i;

   for (i=0; i<3; i++) {
      dst[i][0] = y[i][0]*x[0][0] + y[i][1]*x[1][0] + y[i][2]*x[2][0];
      dst[i][1] = y[i][0]*x[0][1] + y[i][1]*x[1][1] + y[i][2]*x[2][1];
      dst[i][2] = y[i][0]*x[0][2] + y[i][1]*x[1][2] + y[i][2]*x[2][2];
      dst[i][3] = y[i][0]*x[0][3] + y[i][1]*x[1][3] + y[i][2]*x[2][3] + y[i][3];
   }

   dst[3][0] = dst[3][1] = dst[3][2] = 0.0;
   dst[3][3] = 1.0;
}


/* *************************************************************
   This procedure multiplies 2 4X4 matricies
************************************************************* */
void matmatmultod(vector4d *y, vector4d *x, vector4d *dst) {

   int i;

   for (i=0; i<3; i++) {
      dst[i][0] = y[i][0]*x[0][0] + y[i][1]*x[1][0] + y[i][2]*x[2][0];
      dst[i][1] = y[i][0]*x[0][1] + y[i][1]*x[1][1] + y[i][2]*x[2][1];
      dst[i][2] = y[i][0]*x[0][2] + y[i][1]*x[1][2] + y[i][2]*x[2][2];
      dst[i][3] = y[i][0]*x[0][3] + y[i][1]*x[1][3] + y[i][2]*x[2][3] + y[i][3];
   }

   dst[3][0] = dst[3][1] = dst[3][2] = 0.0;
   dst[3][3] = 1.0;
}


/* *************************************************************
   This procedure only multiplies orientations...
************************************************************* */
void matmatmultv(vector4f *y, vector4f *x) {

   vector3f z[3];
   register int i;

   copyarray3(z[0], x[0]);
   copyarray3(z[1], x[1]);
   copyarray3(z[2], x[2]);

   for (i=0; i<3; i++) {
      x[i][0] = y[i][0]*z[0][0] + y[i][1]*z[1][0] + y[i][2]*z[2][0];
      x[i][1] = y[i][0]*z[0][1] + y[i][1]*z[1][1] + y[i][2]*z[2][1];
      x[i][2] = y[i][0]*z[0][2] + y[i][1]*z[1][2] + y[i][2]*z[2][2];
   }

}


/* *************************************************************
   This procedure only multiplies orientations...
************************************************************* */
void matmatmultvd(vector4d *y, vector4d *x) {

   vector3d z[3];
   register int i;

   copyarray3(z[0], x[0]);
   copyarray3(z[1], x[1]);
   copyarray3(z[2], x[2]);

   for (i=0; i<3; i++) {
      x[i][0] = y[i][0]*z[0][0] + y[i][1]*z[1][0] + y[i][2]*z[2][0];
      x[i][1] = y[i][0]*z[0][1] + y[i][1]*z[1][1] + y[i][2]*z[2][1];
      x[i][2] = y[i][0]*z[0][2] + y[i][1]*z[1][2] + y[i][2]*z[2][2];
   }

}


/* *************************************************************
   This procedure multiplies a 4X4 matrix with a vector
************************************************************* */
void matvecmult(vector4f *y, float *x) {

   vector4f z;

   copyarray4(z, x);

   x[0] = dotproduct4(y[0], z);
   x[1] = dotproduct4(y[1], z);
   x[2] = dotproduct4(y[2], z);
   x[3] = dotproduct4(y[3], z);
}


/* *************************************************************
   This procedure multiplies a 4X4 matrix with a vector
************************************************************* */
void matvecmulto(vector4f *y, float *x) {

   vector3f z;

   copyarray3(z, x);

   x[0] = dotproduct3(y[0], x) + y[0][3];
   x[1] = dotproduct3(y[1], z) + y[1][3];
   x[2] = dotproduct3(y[2], z) + y[2][3];
}


/* *************************************************************
   This procedure multiplies a 4X4 matrix with a vector
************************************************************* */
void matvecmulto(vector4f *y, float *src, float *dst) {

   dst[0] = dotproduct3(y[0], src) + y[0][3];
   dst[1] = dotproduct3(y[1], src) + y[1][3];
   dst[2] = dotproduct3(y[2], src) + y[2][3];
   dst[3] = 1.0;
}


/* *************************************************************
   This procedure multiplies a 4X4 matrix with a vector
************************************************************* */
void matvecmultv(vector4f *y, float *x) {

   vector3f z;

   copyarray3(z, x);

   x[0] = dotproduct3(y[0], z);
   x[1] = dotproduct3(y[1], z);
   x[2] = dotproduct3(y[2], z);
}


/* *************************************************************
   This procedure multiplies a 4X4 matrix with a vector
************************************************************* */
void matvecmultvd(vector4d *y, double *x) {

   vector3d z;

   copyarray3(z, x);

   x[0] = dotproduct3(y[0], z);
   x[1] = dotproduct3(y[1], z);
   x[2] = dotproduct3(y[2], z);
}


/* *************************************************************
   This procedure multiplies a 4X4 matrix with a vector
************************************************************* */
void matvecmultv(vector4f *y, float *src, float *dst) {

   dst[0] = dotproduct3(y[0], src);
   dst[1] = dotproduct3(y[1], src);
   dst[2] = dotproduct3(y[2], src);
}


/* *************************************************************
   This procedure multiplies a 4X4 matrix with a vector
************************************************************* */
void matvecmultvd(vector4d *y, double *src, double *dst) {

   dst[0] = dotproduct3(y[0], src);
   dst[1] = dotproduct3(y[1], src);
   dst[2] = dotproduct3(y[2], src);
}


/* *************************************************************
   This function multiplies a vector . matrix
************************************************************* */
void vecmatmult(float *x, vector4f *y) {

   vector4f z;

   copyarray4(z, x);

   x[0] = y[0][0]*z[0] + y[1][0]*z[1] + y[2][0]*z[2] + y[3][0]*z[3];
   x[1] = y[0][1]*z[0] + y[1][1]*z[1] + y[2][1]*z[2] + y[3][1]*z[3];
   x[2] = y[0][2]*z[0] + y[1][2]*z[1] + y[2][2]*z[2] + y[3][2]*z[3];
   x[3] = y[0][3]*z[0] + y[1][3]*z[1] + y[2][3]*z[2] + y[3][3]*z[3];
}


/* *************************************************************
   This function multiplies a vector . matrix
************************************************************* */
void vecmatmulto(vector4f *y, float *x) {

   vector3f z;

   copyarray3(z, x);

   x[0] = y[0][0]*z[0] + y[1][0]*z[1] + y[2][0]*z[2];
   x[1] = y[0][1]*z[0] + y[1][1]*z[1] + y[2][1]*z[2];
   x[2] = y[0][2]*z[0] + y[1][2]*z[1] + y[2][2]*z[2];
   x[3] = y[0][3]*z[0] + y[1][3]*z[1] + y[2][3]*z[2] + y[3][3];
}


/* *************************************************************
   This function multiplies a vector . matrix
************************************************************* */
void vecmatmultv(vector4f *y, float *x) {

   vector3f z;

   copyarray3(z, x);

   x[0] = y[0][0]*z[0] + y[1][0]*z[1] + y[2][0]*z[2];
   x[1] = y[0][1]*z[0] + y[1][1]*z[1] + y[2][1]*z[2];
   x[2] = y[0][2]*z[0] + y[1][2]*z[1] + y[2][2]*z[2];
}


/* *************************************************************
   this procedure swaps two vectors
************************************************************* */
void swap4(float *mx1, float *mx2) {

   float temp;
   register int i;

   for (i=0; i<4; i++) {
      temp   = mx1[i];
      mx1[i] = mx2[i];
      mx2[i] = temp;
   }

}


/* *************************************************************
   this procedure calculates the inverse of a matrix
************************************************************* */
void inversemxrt(vector4f *mx, vector4f *mxinverse) {

   vector4f temp;
   int   i;

   for (i=0; i<3; i++) {
      mxinverse[i][0] = mx[0][i];
      mxinverse[i][1] = mx[1][i];
      mxinverse[i][2] = mx[2][i];
   }

   mxinverse[3][0] = mxinverse[3][1] = mxinverse[3][2] = 0.0;
   mxinverse[3][3] = 1.0f;

   temp[0] = mx[0][3];
   temp[1] = mx[1][3];
   temp[2] = mx[2][3];

   matvecmultv(mxinverse, temp);

   mxinverse[0][3] = -temp[0];
   mxinverse[1][3] = -temp[1];
   mxinverse[2][3] = -temp[2];
}


/* *************************************************************
   this procedure calculates the inverse of a matrix
************************************************************* */
void inversemxrtd(vector4d *mx, vector4d *mxinverse) {

   vector4d temp;
   int   i;

   for (i=0; i<3; i++) {
      mxinverse[i][0] = mx[0][i];
      mxinverse[i][1] = mx[1][i];
      mxinverse[i][2] = mx[2][i];
   }

   mxinverse[3][0] = mxinverse[3][1] = mxinverse[3][2] = 0.0;
   mxinverse[3][3] = 1.0;

   temp[0] = mx[0][3];
   temp[1] = mx[1][3];
   temp[2] = mx[2][3];

   matvecmultvd(mxinverse, temp);

   mxinverse[0][3] = -temp[0];
   mxinverse[1][3] = -temp[1];
   mxinverse[2][3] = -temp[2];
}


/* *************************************************************
   this procedure calculates the inverse of a matrix
************************************************************* */
int inversemx(vector4f *a, vector4f *b) {

   register float det, idet;

   b[0][0] = a[1][1]*a[2][2] - a[2][1]*a[1][2];
   b[0][1] = a[0][1]*a[2][2] - a[2][1]*a[0][2];
   b[0][2] = a[0][1]*a[1][2] - a[1][1]*a[0][2];

   b[1][0] = a[1][0]*a[2][2] - a[2][0]*a[1][2];
   b[1][1] = a[0][0]*a[2][2] - a[2][0]*a[0][2];
   b[1][2] = a[0][0]*a[1][2] - a[1][0]*a[0][2];

   b[2][0] = a[1][0]*a[2][1] - a[2][0]*a[1][1];
   b[2][1] = a[0][0]*a[2][1] - a[2][0]*a[0][1];
   b[2][2] = a[0][0]*a[1][1] - a[1][0]*a[0][1];

   det = a[0][0]*b[0][0] - a[1][0]*b[0][1] + a[2][0]*b[0][2];

   if (fabs(det) < CORRECT)
      return inversemx2(a, b);

   det = 1.0f/det;
   idet = -det;

   b[0][0] *= det;
   b[0][1] *= idet;
   b[0][2] *= det;

   b[1][0] *= idet;
   b[1][1] *= det;
   b[1][2] *= idet;

   b[2][0] *= det;
   b[2][1] *= idet;
   b[2][2] *= det;

   b[0][3] = -(b[0][0]*a[0][3] + b[0][1]*a[1][3] + b[0][2]*a[2][3]);
   b[1][3] = -(b[1][0]*a[0][3] + b[1][1]*a[1][3] + b[1][2]*a[2][3]);
   b[2][3] = -(b[2][0]*a[0][3] + b[2][1]*a[1][3] + b[2][2]*a[2][3]);

   b[3][0] = b[3][1] = b[3][2] = 0.0;
   b[3][3] = 1;

   return 1;
}


/* *************************************************************
   this procedure calculates the inverse of a matrix
************************************************************* */
int inversemx2(vector4f *mx, vector4f *mxinverse) {

   vector4f tempmx[4];
   register int i, j, k;
   register float temp;

   init_mx(mxinverse);

   copymx4x4o(tempmx, mx);

   for (i=0; i<4; i++) {
      if (!not_zero(tempmx[i][i]))
         for (j=i+1; (j<4 && !not_zero(tempmx[i][i])); j++)
            if (tempmx[j][i]) {
               swap4(tempmx[i], tempmx[j]);
               swap4(mxinverse[i], mxinverse[j]);
            }

      if (!not_zero(tempmx[i][i]))
         return 0;

      temp = 1.0f/tempmx[i][i];

      for (j=i+1; j<4; j++)
         tempmx[i][j] *= temp;

      smultarray4(mxinverse[i], temp);

      for (j=i+1; j<4; j++) {
         for (k=i+1; k<4; k++)
            tempmx[j][k] -= tempmx[j][i]*tempmx[i][k];

         mxinverse[j][0] -= tempmx[j][i]*mxinverse[i][0];
         mxinverse[j][1] -= tempmx[j][i]*mxinverse[i][1];
         mxinverse[j][2] -= tempmx[j][i]*mxinverse[i][2];
         mxinverse[j][3] -= tempmx[j][i]*mxinverse[i][3];
      }

   }

   for (i=3; i>-1; i--)
      for (j=i-1; j>-1; j--) {
         mxinverse[j][0] -= tempmx[j][i]*mxinverse[i][0];
         mxinverse[j][1] -= tempmx[j][i]*mxinverse[i][1];
         mxinverse[j][2] -= tempmx[j][i]*mxinverse[i][2];
         mxinverse[j][3] -= tempmx[j][i]*mxinverse[i][3];
      }

   return 1;
}


/* *************************************************************
	fails at 180 degrees...
************************************************************* */
int vector_interp_setup(float *v1, float *v2, int count, float idelta, float *delta, float *theta) {

   switch (count) {

      case 3:
         *theta = dotproduct3(v1, v2);
         break;

      default:		//case 4:
         *theta = dotproduct4(v1, v2);
         break;
   }

   if (*theta > 1-CORRECT) {	// the vectors are essentially the same - no rotation
      delta[0] = 0;             // theta - no rotation
      delta[1] = HALFPI;        // psi - instant swap
      delta[2] = 1;             // 1.0/sin(theta);
      delta[3] = 0;             // dpsi
      return 1;
   }

   if (*theta < CORRECT-1) {	// vectors are essentially the opposite, so delta = halfpi (no interp)

      delta[0] = HALFPI;        // theta - total rotation
      delta[1] = 0.0;           // psi
      delta[2] = 1;             // 1.0/sin(theta);
      delta[3] = 0;             // dpsi
      delta[3] = idelta*delta[0];        // dpsi
      return 0;
   }

   delta[0] = (float)ACOS(*theta);    // theta
   delta[1] = 0.0;                    // psi
   delta[2] = 1.0f/SIN(delta[0]);     // 1.0/sin(theta);
   delta[3] = idelta*delta[0];        // dpsi
   return 1;
}


/* *************************************************************
	fails at 180 degrees...
************************************************************* */
void vector_interp_eval(float *v1, float *v2, int count, float *d, float *out) {

   float s, is;

   s = SIN(d[1]);
   is = SIN(d[0]);

   switch (count) {

      case 4:
         out[3] = d[2] * (is*v1[3] + s*v2[3]);

      default:		//case 3:
         out[2] = d[2] * (is*v1[2] + s*v2[2]);
         out[1] = d[2] * (is*v1[1] + s*v2[1]);
         out[0] = d[2] * (is*v1[0] + s*v2[0]);
   }

   return;
}


/* *************************************************************
	map from a 3D plane to a 2D plane
************************************************************* */
void map_3D_2D(float *vertex0, float *vertex1, float *vertex2, float *uv0, float *uv1, float *uv2, float *mxu, float *mxv) {

   vector2f u, v, ov1, ov2;
   vector3f v1, v2;
   float a, s, t;

   subeqarray2(u, uv1, uv0);
   subeqarray2(v, uv2, uv1);

   subeqarray3(v1, vertex1, vertex0);
   subeqarray3(v2, vertex2, vertex1);

   a = 1.0f/dotproduct3(v1, v1);
   t = dotproduct3(v1, v2)*a;

   mxu[0] = v2[0] - v1[0]*t;
   mxu[1] = v2[1] - v1[1]*t;
   mxu[2] = v2[2] - v1[2]*t;

   s = 1.0f/dotproduct3(mxu, mxu);

   ov1[1] = (v[0] - t*u[0])*s;
   ov2[1] = (v[1] - t*u[1])*s;

   ov1[0] = u[0]*a - ov1[1]*t;
   ov2[0] = u[1]*a - ov2[1]*t;

   mxu[0] = ov1[0]*v1[0] + ov1[1] * v2[0];
   mxu[1] = ov1[0]*v1[1] + ov1[1] * v2[1];
   mxu[2] = ov1[0]*v1[2] + ov1[1] * v2[2];

   mxv[0] = ov2[0]*v1[0] + ov2[1] * v2[0];
   mxv[1] = ov2[0]*v1[1] + ov2[1] * v2[1];
   mxv[2] = ov2[0]*v1[2] + ov2[1] * v2[2];
}


/* *************************************************************
************************************************************* */
void calculate_normal(float *v1, float *v2, float *v3, float *n) {

   vector3f p1, p2;

   subeqarray3(p1, v2, v1);
   subeqarray3(p2, v3, v2);

   xproduct(n, p2, p1);
   normalize3(n);
}


/* ******************************************************************************************
****************************************************************************************** */
int query_calculate_normal(float *v1, float *v2, float *v3) {

   vector3f p1, p2;
   float m1, m2;
   float ans;

   subeqarray3(p1, v1, v2);
   subeqarray3(p2, v2, v3);

   m1 = dotproduct3(p1, p1);
   if (m1 < CORRECT)
      return 0;
      
   m2 = dotproduct3(p2, p2);
   if (m2 < CORRECT)
      return 0;

   ans = dotproduct3(p1, p2);
   return (ans*ans < ICORRECT*ICORRECT*m1*m2);
}


