

#include "matrix.h"
#include "global.h"


/* **********************************************************************
							P O I N T
*********************************************************************** */


/* **********************************************************************
*********************************************************************** */
int query_point_poly_intersect(float *pt, float *normal, int count, vector4f *vertex) {

   double angle, work;
   int  i;
   vector3f v0, v1, v2, n;
   float l0, l1, l2;

   // is point on plane?
   if (!query_point_plane_intersect(pt, normal))
      return 0;

   subeqarray3(v2, vertex[0], pt);
   l2 = dotproduct3(v2, v2);

   copyarray3(v0, v2);
   l0 = l2;

   for (i=1, angle=0; i<count; i++) {
      copyarray3(v1, v2);
      l1 = l2;

      subeqarray3(v2, vertex[i], pt);
      l2 = dotproduct3(v2, v2);

      work = ACOS(dotproduct3(v1, v2)/sqrt(l1*l2));
      calculate_normal(vertex[i-1], vertex[i], pt, n);

      if (dotproduct3(n, normal) > 0)
         angle += work;
      else
         angle -= work;
   }

   work = ACOS(dotproduct3(v0, v2)/sqrt(l0*l2));
   calculate_normal(vertex[count-1], vertex[0], pt, n);

   if (dotproduct3(n, normal) > 0)
      angle += work;
   else
      angle -= work;

   return (angle > TWOPI2D-CORRECT2 && angle < TWOPI2D+CORRECT2);
}


/* **********************************************************************
     guarenteed convex polygon :)
********************************************************************** */
int query_point_cpoly_intersect(float *pt, float *normal, int count, vector4f *vertex) {

   int i;
   vector3f v0, v1, v2, n;

   // is point on plane?
   if (!query_point_plane_intersect(pt, normal))
      return 0;

   subeqarray3(v2, vertex[0], pt);
   copyarray3(v0, v2);

   for (i=1; i<count; i++) {
      copyarray3(v1, v2);
      subeqarray3(v2, vertex[i], pt);

      xproduct(n, v2, v1);

      if (dotproduct3(n, normal) < 0)
         return 0;
   }

   xproduct(n, v0, v2);

   return dotproduct3(n, normal) >= 0;
}


/* **********************************************************************
    guarenteed convex polygon :)
    assumes that point is on plane
    points in clockwise order
********************************************************************** */
int query_point_in_cpoly(float *pt, float *normal, int count, vector4f *vertex, int *index) {

   int i;
   vector3f v0, v1, v2, n;

   subeqarray3(v2, vertex[index[0]], pt);
   copyarray3(v0, v2);

   for (i=1; i<count; i++) {
      copyarray3(v1, v2);
      subeqarray3(v2, vertex[index[i]], pt);

      xproduct(n, v2, v1);

      if (dotproduct3(n, normal) < 0)
         return 0;
   }

   xproduct(n, v0, v2);

   return dotproduct3(n, normal) >= 0;
}


/* **********************************************************************
********************************************************************** */
int query_point_sphere_intersect(float *pt, float *center, float r) {

   vector3f sum;

   subeqarray3(sum, pt, center);

   return (dotproduct3(sum, sum) < r*r + CORRECT);
}


/* ***********************************************************************
*********************************************************************** */
int query_point_rect_intersect(float *pt, vector4f *n) {

   float t;
   
   if (!line_plane_intersect(n[0], pt, n[0], NULL, &t) || t < -CORRECT)
      return 0;

   if (!line_plane_intersect(n[1], pt, n[1], NULL, &t) || t < -CORRECT)
      return 0;

   if (!line_plane_intersect(n[2], pt, n[2], NULL, &t) || t < -CORRECT)
      return 0;

   if (!line_plane_intersect(n[3], pt, n[3], NULL, &t) || t < -CORRECT)
      return 0;

   if (!line_plane_intersect(n[4], pt, n[4], NULL, &t) || t < -CORRECT)
      return 0;

   if (!line_plane_intersect(n[5], pt, n[5], NULL, &t) || t < -CORRECT)
      return 0;

   return 1;
}


/* ******************************************************************************************
   r2 is a unit vector
****************************************************************************************** */
int query_point_line_intersect(float *pt1, float *pt2, float *r2) {

   vector3f work;
   float m, n;
   
   subeqarray3(work, pt1, pt2);
   m = dotproduct3(work, work);
   
   // if same point...
   if (m < CORRECT)
      return 1;

   n = dotproduct3(work, r2);
      
   return n*n > m*ICORRECT;
}
   


/* **********************************************************************
							S P H E R E
*********************************************************************** */


/* **********************************************************************
   vector   Rd
   radius   SQR(Sr)
*********************************************************************** */
int query_line_sphere_intersect(float radius, float *sphere, float *pt, float *vector) {

   vector3f x;
   float a, b, c;

   a = dotproduct3(vector, vector);

   if (!a)
      return 0;

   // OC
   subeqarray3(x, sphere, pt);

   // L2OC
   c = dotproduct3(x, x);
   c -= radius*radius;

   // ray starts inside sphere
   if (c <= 0)
      return 1;

   // tca
   b = dotproduct3(vector, x);

   // going away from sphere
   if (b <= 0)
      return 0;

   // check if ray goes away from object
   return b*b > a*c;
}


/* **********************************************************************
   vector   Rd - normalized
   radius   SQR(Sr)
*********************************************************************** */
int query_line_sphere_intersectn(float radius, float *sphere, float *pt, float *vector) {

   vector3f x;
   float b, c;

   // OC
   subeqarray3(x, sphere, pt);

   // L2OC
   c = dotproduct3(x, x);
   c -= radius*radius;

   // ray starts inside sphere
   if (c <= 0)
      return 1;

   // tca
   b = dotproduct3(vector, x);

   // going away from sphere
   if (b <= 0)
      return 0;

   // check if ray goes away from object
   return b*b > c;
}


/* **********************************************************************
   Note: "vector" is not normalized, its magnitude is length of line
         end pts are not checked here

   vector   Rd
   radius   SQR(Sr)
*********************************************************************** */
int query_lineseg_sphere_intersect(float radius, float *sphere, float *pt, float *vector) {

   vector3f x;
   float a, b, c;

   a = dotproduct3(vector, vector);

   if (!a)
      return 0;

   // OC
   subeqarray3(x, sphere, pt);

   // L2OC
   c = dotproduct3(x, x);
   radius *= radius;

   // ray starts inside sphere
   if (c <= radius)
      return 1;

   // should not happen ??? ( distance between pt and sphere center longer than hypotenus)
   if (c+radius > a)
      return 0;

   // tca
   b = dotproduct3(vector, x);

    // going away from sphere
   if (b < 0)
      return 0;

   // check if ray goes away from object
   return b*b > a*(c - radius);
}


/* **********************************************************************
*********************************************************************** */
int query_lineseg_sphere_intersect2(float radius, float *sphere, float *start, float *end) {

   vector3f vector, x;
   float a, b, c;
   float check;
   float ans1, ans2;

   subeqarray3(vector, end, start);

   a = dotproduct3(vector, vector);

   if (!a)
      return 0;

   subeqarray3(x, sphere, start);
   b = dotproduct3(vector, x);
   c = dotproduct3(x,x) - radius*radius;

   check = b*b - a*c;

   if (check < 0)
      return 0;

   check = (float)sqrt(check);

   if (a < 0) {
      a = -a;
      ans1 = -b + check;
      ans2 = -b - check;
   }

   else {
      ans1 = b - check;
      ans2 = b + check;
   }

   return ((ans1 >= 0 && ans1 <= a) || (ans2 >= 0 && ans2 <= a));
}


/* **********************************************************************
        sphere -
        pt     - point on line
        vector - line vector
        NOTE:   if inside sphere, return positive t
*********************************************************************** */
int line_sphere_intersect(float radius, float *sphere, float *pt, float *vector, float *t) {

   float a, b, c, check;
   vector3f x;

   a = dotproduct3(vector, vector);

   // no vector
   if (!a)
      return 0;

   subeqarray3(x, sphere, pt);

   b = dotproduct3(x, vector);
   c = dotproduct3(x, x) - radius*radius;

   check = b*b - a*c;

   if (check < 0) {
      *t = 0;
      return 0;
   }

   // inside circle want t >= 0
   if (c <= 0) {
      *t = (float)((b + sqrt(check))/a);
      return -1;
   }

   // backwards
   if (b < 0)
      *t = (float)((b + sqrt(check))/a);
   // forwards
   else
      *t = (float)((b - sqrt(check))/a);

   return 1;
}


/* **********************************************************************
	same as line_sphere_intersect, but ray is assumed to be normalized
*********************************************************************** */
int line_sphere_intersectn(float radius, float *sphere, float *pt, float *vector, float *t) {

   float b, c, check;
   vector3f x;

   subeqarray3(x, sphere, pt);

   b = dotproduct3(x, vector);
   c = dotproduct3(x, x) - radius*radius;

   check = b*b - c;

   if (check < 0) {
      *t = 0;
      return 0;
   }

   // inside circle want t >= 0
   if (c <= 0) {
      *t = (float)(b + sqrt(check));
      return -1;
   }

 	// outside circle
   // backwards
   if (b < 0)
      *t = (float)(b + sqrt(check));
   // forwards
   else
      *t = (float)(b - sqrt(check));

   return 1;
}


/* **********************************************************************
*********************************************************************** */
int line_ellipse_intersect(float *scale, float *pt, float *vector, float *t) {

   float a, b, c, check;
   vector3f x, iscale;

   iscale[0] = 1.0f/(scale[0]*scale[0]);
   iscale[1] = 1.0f/(scale[1]*scale[1]);
   iscale[2] = 1.0f/(scale[2]*scale[2]);

   x[0] = iscale[0]*vector[0];
   x[1] = iscale[1]*vector[1];
   x[2] = iscale[2]*vector[2];

   b = dotproduct3(pt, x);
   a = dotproduct3(vector, x);
   c = pt[0]*pt[0]*iscale[0] + pt[1]*pt[1]*iscale[1] + pt[2]*pt[2]*iscale[2] - 1.0f;

   check = b*b - a*c;

   if (check < 0) {
      *t = 0.0f;
      return 0;
   }

   // inside circle want t >= 0
   if (c <= 0) {
      *t = (float)((-b + sqrt(check))/a);
      return -1;
   }

   // outside circle
   // backwards
   if (b > 0)
      *t = (float)((-b + sqrt(check))/a);
   // forwards
   else
      *t = (float)((-b - sqrt(check))/a);

   return 1;
}


/* **********************************************************************
	iscale[] precalculated
*********************************************************************** */
int line_ellipse_intersecti(float *iscale, float *pt, float *vector, float *t) {

   float a, b, c, check;
   vector3f x;

   x[0] = iscale[0]*vector[0];
   x[1] = iscale[1]*vector[1];
   x[2] = iscale[2]*vector[2];

   b = dotproduct3(pt, x);
   a = dotproduct3(vector, x);

   if (!a)
      return 0;

   c = pt[0]*pt[0]*iscale[0] + pt[1]*pt[1]*iscale[1] + pt[2]*pt[2]*iscale[2] - 1;

   check = b*b - a*c;

   if (check < 0) {
      *t = 0;
      return 0;
   }

   // inside circle want t >= 0
   if (c <= 0) {
      *t = (float)((-b + sqrt(check))/a);
      return -1;
   }

   // outside circle
   // backwards
   if (b > 0)
      *t = (float)((-b + sqrt(check))/a);
   // forwards
   else
      *t = (float)((-b - sqrt(check))/a);

   return 1;
}


/* **********************************************************************
********************************************************************** */
int query_sphere_sphere_intersect(float *c1, float r1, float *c2, float r2) {

   float sum[3];

   r2 += r1;
   subeqarray3(sum, c1, c2);

   return (dotproduct3(sum, sum) < r2*r2 + CORRECT);
}


/* ***********************************************************************
*********************************************************************** */
int query_sphere_rect_intersect(float *c, float r, float *rect_center, vector4f *n) {

   vector3f ray;
   vector3f start;
   int i;
   float t;
   
   copyarray3(start, c);
   subeqarray3(ray, rect_center, c);

   for (i=0; i<6; i++)
      if (line_plane_intersect(n[i], start, ray, NULL, &t))
         if (fabs(t) < r)
            return 1;

   return 0;
}


/* **********************************************************************
                                PLANE
*********************************************************************** */


/* **********************************************************************
   This function calculates the distance between an objects center, and
   a plane.
*********************************************************************** */
int line_plane_intersect(float *plane, float *pt, float *vector, float *intersect, float *t) {

   float x;

   x = dotproduct3(plane, vector);

   if (!not_zero(x))
      return 0;

   *t = -(plane[3] + dotproduct3(pt, plane))/x;

   if (intersect) {
      intersect[0] = pt[0] + *t*vector[0];
      intersect[1] = pt[1] + *t*vector[1];
      intersect[2] = pt[2] + *t*vector[2];
   }
   
   return 1;
}


/* **********************************************************************
   This function calculates the distance between an objects center, and
   a plane.
*********************************************************************** */
int lineseg_plane_intersect(float *plane, float *start, float *end, float *vector, float *intersect, float *t) {

   float denom;
   float d;

   d = dotproduct3(start, plane) + plane[3];

   // backface cull
   if (d < 0)
      return 0;

   denom = dotproduct3(end, plane) + plane[3];

   // start/end are on same side of poly
   if (denom > 0)
      return 0;

   denom = dotproduct3(vector, plane);

   if (denom > CORRECT)
      return 0;

   *t = -d / denom;

   if (intersect) {
      intersect[0] = start[0] + *t*vector[0];
      intersect[1] = start[1] + *t*vector[1];
      intersect[2] = start[2] + *t*vector[2];
   }
   
   return 1;
}


/* **********************************************************************
*********************************************************************** */
int line_poly_intersect(float *pt, float *vector, float *plane, vector4f *vertex, int count, float *intersect, float *t) {

   return line_plane_intersect(plane, pt, vector, intersect, t) &&
          query_point_poly_intersect(intersect, plane, count, vertex);
}


/* **********************************************************************
*********************************************************************** */
int line_cpoly_intersect(float *pt, float *vector, float *plane, vector4f *vertex, int count, float *intersect, float *t) {

   return line_plane_intersect(plane, pt, vector, intersect, t) &&
          query_point_cpoly_intersect(intersect, plane, count, vertex);
}


/* **********************************************************************
*********************************************************************** */
int query_poly_plane_intersect(vector4f *pt, int len, float *n) {

   int i;
   float t;

   t = dotproduct3(pt[0], n) + n[3];

   for (i=0; i<len; i++)
      if (t * (dotproduct3(pt[i], n) + n[3]) < CORRECT)
         return 1;

   return 0;
}


/* **********************************************************************
   Theory - if a plane intersects a rectangle, then it must intersect at
   LEAST 3 edges, therefore we can stop if we find one intersection.
   order of vertices - 1-4 one side, 5-8 opposite side where 1,5 match,
   2,6 match, etc. - 12 sets of compares

   new theory - if a plane intersects a rectangle, it must intersect with
   at least1 of the 4 cross-rectangle diagonals -
   only need to check 3 (2 creates a plane, 3 creates a space). (1,7 2,8 3,5)
********************************************************************** */
int query_plane_rect_intersect(float *n, vector4f *p) {

   vector3f v;
   float t;
   
   subeqarray3(v, p[6], p[0]);

   if (line_plane_intersect(n, p[0], v, NULL, &t))
      return !(t < 0 || t > 1);

   subeqarray3(v, p[7], p[1]);

   if (line_plane_intersect(n, p[1], v, NULL, &t))
      return !(t < 0 || t > 1);

   subeqarray3(v, p[4], p[2]);

   if (line_plane_intersect(n, p[2], v, NULL, &t))
      return !(t < 0 || t > 1);

   // theoretically, the program should never get here!!!
   return 0;
}


/* **********************************************************************
********************************************************************** */
int query_plane_plane_intersect(float *n1, float *n2) {

   float temp;

   temp = dotproduct3(n1, n2);
   if (fabs(temp) < 1-CORRECT)
      return 1;

   return (n1[3]*n1[3] - n2[3]*n2[3] < CORRECT);
}


/* **********************************************************************
					    OTHER
*********************************************************************** */


/* **********************************************************************
********************************************************************** */
void build_rect_index(int index[6][4]) {

   index[0][0] = 0; index[0][1] = 1; index[0][2] = 2; index[0][3] = 3;
   index[1][0] = 4; index[1][1] = 5; index[1][2] = 6; index[1][3] = 7;
   index[2][0] = 0; index[2][1] = 4; index[2][2] = 5; index[2][3] = 1;
   index[3][0] = 1; index[3][1] = 5; index[3][2] = 6; index[3][3] = 2;
   index[4][0] = 2; index[4][1] = 6; index[4][2] = 7; index[4][3] = 3;
   index[5][0] = 3; index[5][1] = 7; index[5][2] = 4; index[5][3] = 0;
}


/* ***********************************************************************
                normal orders front, back left, top, right, bottom
                polygon orders : 0-3 front, 4-7 back, 0-4-5-1 left, etc
*********************************************************************** */
int line_rect_intersect(float *pt, float *v, vector4f *p, vector4f *n, float *flag) {

   vector4f ptemp[4];
   float t;
   float intersect[2];
   int ptr = 0;
   int index[6][4];
   int i;
   vector4f pos;
   
   build_rect_index(index);

   for (i=0; i<6; i++) {
      copyarray4(ptemp[0], p[index[i][0]]);
      copyarray4(ptemp[1], p[index[i][0]]);
      copyarray4(ptemp[2], p[index[i][0]]);
      copyarray4(ptemp[3], p[index[i][0]]);

      if (line_poly_intersect(pt, v, n[i], ptemp, 4, pos, &t)) {
         intersect[ptr] = t;
         ptr++;

         if (ptr == 2) {
            if (intersect[0]*intersect[1] > 0)       // outside the box
               if (intersect[0] > 0) {                     // face to box
                  if (intersect[0] < intersect[1]) {
                     *flag = intersect[0];
                     return 1;
                  }

                  *flag = intersect[1];
                  return 1;
               }

            if (intersect[0] > intersect[1]) {               // inside box or face aw         if (intersect[0] > intersect[1])
               *flag = intersect[0];
               return 1;
            }

            *flag = intersect[1];
            return 1;
         }

      }

   }

   if (ptr == 1) {                              // should never happen unless I screwed the pooch
      *flag = intersect[0];
      pprintf("FUCKUP in LINE_RECT!!!\n");
      return 1;
   }

   return 0;
}


/* **********************************************************************
Cube Theory - if 2 cubes intersect each other, then at least one corner
of one cube is within another cube

Rectangle Theory 1: if 2 rectangles intersect, then there exists at
least 1 plane from each rectangle st the intersection of the planes (a line)
intersects both polygons that represent those planes.

*Rectangle Theory 2: if bounding spheres that bound the rectangles
do not intersect, the rectangles do not intersect.

*Rectangle Theory 3: if a corner of a rectangle is w/in the other,
there is an intersection

Rectangle Theory 4: if 2 rectangles intersect, then the edge of one
must intersect the face of another unless one completely encloses the
other. (caught by Theory 3)

*Rectangle Theory 5:  A line between the center of the rectangles
 will intersect faces of both rectangles - if the intersections are w/in the other rectangele, then
they intersect.  Otherwise, if their centers are w/in the other rectangle then they intersect.
 Otherwise they dont!

********************************************************************** */
int query_rect_rect_intersect(vector4f *n1, vector4f *p1, float *c1, float r1,
                              vector4f *n2, vector4f *p2, float *c2, float r2) {

   float dist;
   vector3f ray;
   int i;
   vector3f work;

   if (!query_sphere_sphere_intersect(c1, r1, c2, r2))
      return 0;

    // check sphere/rect
   if (!query_sphere_rect_intersect(c1, r1, c2, n2))
      return 0;

    // check sphere/rect
   if (!query_sphere_rect_intersect(c2, r2, c1, n1))
      return 0;

   // *** bounding spheres/rects intersect ***
   // now see if a vertex(pt) of one intersects the other
   for (i=0; i<8; i++)
      if (query_point_rect_intersect(p1[i], n2) || query_point_rect_intersect(p2[i], n1))
         return 1;

    // check to see if centers intersect
   if (query_point_rect_intersect(c1, n2) || query_point_rect_intersect(c2, n1))
      return 1;

    subeqarray3(ray, c2, c1);

    if (line_rect_intersect(c1, ray, p2, n2, &dist)) {
       work[0] = c1[0] + dist*ray[0];
       work[1] = c1[1] + dist*ray[1];
       work[2] = c1[2] + dist*ray[2];

       if (query_point_rect_intersect(work, n2))
          return 1;
    }

    if (line_rect_intersect(c1, ray, p1, n1, &dist)) {
       work[0] = c1[0] + dist*ray[0];
       work[1] = c1[1] + dist*ray[1];
       work[2] = c1[2] + dist*ray[2];

       if (query_point_rect_intersect(work, n1))
          return 1;
    }

    return 0;
}


/* **********************************************************************

        Algebraic Eq:     Ax + By + C = 0

*********************************************************************** */
int line2d_line2d_intersect(float *v1, float *v2, float *pt) {

   float t1, t2;

   if (fabs(v1[0]) < CORRECT) {
      if (fabs(v1[1]) < CORRECT || fabs(v2[0]) < CORRECT)
         return 0;

      pt[1] = -v1[2]/v1[1];
      pt[0] = -(v2[1]*pt[1] + v2[2])/v2[0];

      return 1;
   }

   if (fabs(v1[1]) < CORRECT) {
      if (fabs(v1[0]) < CORRECT || fabs(v2[1]) < CORRECT)
         return 0;

      pt[0] = -v1[2]/v1[0];
      pt[1] = -(v2[0]*pt[0] + v2[2])/v2[1];
      return 1;
   }

   if (fabs(v2[0]) < CORRECT) {
      if (fabs(v2[1]) < CORRECT || fabs(v1[0]) < CORRECT)
         return 0;

      pt[1] = -v2[2]/v2[1];
      pt[0] = -(v1[1]*pt[1] + v1[2])/v1[0];
      return 1;
   }

   if (fabs(v2[1]) < CORRECT) {
      if (fabs(v2[0]) < CORRECT || fabs(v1[1]) < CORRECT)
         return 0;

      pt[0] = -v2[2]/v2[0];
      pt[1] = -(v1[0]*pt[0] + v1[2])/v1[1];
      return 1;
   }

   t1 = v1[1]*v2[0] - v2[1]*v1[0];

   if (fabs(t1) < CORRECT)
      return 0;

   t2 = v2[1]*v1[2] - v1[1]*v2[2];
   pt[0] = t2/t1;
   pt[1] = -(v2[0]*pt[0] + v2[2])/v2[1];

   t1 = v1[0]*pt[0] + v1[1]*pt[1] + v2[2];
   return (fabs(t1) < CORRECT);
}


/* ******************************************************************************************
****************************************************************************************** */
int find_full_intersect(int i, int j, float *p1, float *r1, float *p2, float *r2, float *intersect) {

   float udenom, tdenom, unum, tnum;
   float px, py;
   float u, t;
   vector3f diff, point;

   tdenom = r1[i];

   if (fabs(r1[i]) < CORRECT)
      return -1;
      
   udenom = r2[i]*r1[j] - r2[j]*r1[i];

   if (fabs(udenom) < CORRECT)
      return -1;

   px = p2[i] - p1[i];
   py = p2[j] - p1[j];

   unum = py*r1[i] - px*r1[j];
   u = unum / udenom;

   tnum = px + u * r2[i];

   t = tnum / tdenom;

   point[0] = p1[0] + t*r1[0];
   point[1] = p1[1] + t*r1[1];
   point[2] = p1[2] + t*r1[2];

   diff[0] = point[0] - (p2[0] + u*r2[0]);
   diff[1] = point[1] - (p2[1] + u*r2[1]);
   diff[2] = point[2] - (p2[2] + u*r2[2]);

   if (dotproduct3(diff, diff) > CORRECT)
      return 0;

   copyarray3(intersect, point);
   return 1;
}


/* ******************************************************************************************

        Parametric Eq:     P + t*V = 0

   asdf optimization - fast reject, create plane eq w/ 3 points - (2 from one line, 1 from other)
   			and check if 2nd pt from other line is on/near plane
****************************************************************************************** */
int line3d_line3d_intersect(float *pt1, float *v1, float *pt2, float *v2, float *intersect) {

   vector3f nv1, nv2;
   int i, j, res;
   float lm1, lm2;

   copyarray3(nv1, v1);
   lm1 = normalize3(nv1);

   copyarray3(nv2, v2);
   lm2 = normalize3(nv2);

   if (lm1 < CORRECT && lm2 < CORRECT) {		// points
      if (similar3(pt1, pt2)) {
         copyarray3(intersect, pt1);
         return 1;
      }

      return 0;
   }

   if (lm1 < CORRECT)
      if (query_point_line_intersect(pt1, pt2, nv2)) {
         copyarray3(intersect, pt1);
         return 1;
      }
      
      else
         return 0;
	 
   if (lm2 < CORRECT)
      if (query_point_line_intersect(pt2, pt1, nv1)) {
         copyarray3(intersect, pt2);
         return 1;
      }
      
      else
         return 0;
	 
   // parallel -> if same line, if so, then all points intersect
   // else no intersect
   lm1 = dotproduct3(nv1, nv2);
   if (lm1*lm1 > ICORRECT) {
// check if lines are the same
//   subeqarray3(nv2, pt1, pt2);
//   if (fabs(dotproduct3(nv1, nv2))) > CORRECT*magnitude3(nv2)) {
//      copyarray3(intersect, pt2); // arbitrary pt
//      return 1;
//   }

      return 0;
   }

   for (i=0; i<3; i++)
      for (j=0; j<3; j++) {
         if (i == j)
            continue;

         res = find_full_intersect(i, j, pt1, nv1, pt2, nv2, intersect);
         if (res != -1)
	    return res;
	 
         res = find_full_intersect(i, j, pt2, nv2, pt1, nv1, intersect);
         if (res != -1)
	    return res;
      }

   return 0;
}


/* **********************************************************************
					    REFLECTION
*********************************************************************** */

/* **********************************************************************
   This function reflects an vector about a normal to a plane
*********************************************************************** */
void reflect(float normal[], float vector[]) {

   float mag;
   int    i;

   mag = (float)fabs(dotproduct3(normal, vector));
   mag += mag;

   for (i=0; i<3; i++)
      vector[i] += mag*normal[i];

   normalize3(vector);
}


/* **********************************************************************
    reflection "normal"

    src - object to "bounce" off of
    dest - moving object
********************************************************************** */
int point_point_reflectn3vf(float *src, float *dest, float *normal) {

   subeqarray3(normal, dest, src);

   return (normalize3(normal) >= CORRECT);
}


/* **********************************************************************
    reflection normal of a plane - no mater what is the plane normal
********************************************************************** */
int plane_reflectn3vf(float *n, float *normal) {

   copyarray3(normal, n);
   normalize3(normal);

   return 1;
}


/* **********************************************************************

    given a point and a rectangle, choose a face normal from the rectangle
    to represent the reflection normal of the point reflecting/bouncing of the
    rectangle

    normal orders front, back left, top, right, bottom
    polygon orders : 0-3 front, 4-7 back, 0-4-5-1 left, etc
********************************************************************** */
int point_rect_reflectn3vf(float *center, float *rect_center, vector4f *n, float *normal) {

   vector3f ray;
   vector3f start;
   int i;
   float t;
   float ct = -1;
   
   // calculate ray from box to sphere
   copyarray3(start, rect_center);
   subeqarray3(ray, center, rect_center);

   // find closest polygon that intersect
   for (i=0; i<6; i++)
      if (line_plane_intersect(n[i], start, ray, NULL, &t)) { // back face
          t = (float)fabs(t);

          if (ct < 0 || t < ct) {
             ct = t;
             copyarray3(normal, n[i]);
          }

      }

   // should always be true
   return (ct > -1);
}


/* **********************************************************************

    plane reflecting off of a rectangle

    put plane normal at center of rectangle, draw ray, find closest side hit

********************************************************************** */
int plane_rect_reflectn3vf(float *n, float *rect_center, vector4f *surfacen, float *normal) {

   vector3f ray;
   vector3f start;
   int i;
   float t;
   float ct = -1;
   
   copyarray3(start, rect_center);
   copyarray3(ray, n);

   // find closest polygon that intersect
   for (i=0; i<6; i++)

      // back face
      if (line_plane_intersect(surfacen[i], start, ray, NULL, &t)) {
          t = (float)fabs(t);

          if (ct < 0 || t < ct) {
             ct = t;
             copyarray3(normal, surfacen[i]);
          }

      }

   // should always be true
   return (ct > -1);
}


/* **********************************************************************
        NOTE:   if inside, return positive
********************************************************************** */
int line_cylinder_intersect(vector4f pt, vector4f v, float length, float radius, float *t, int *side) {

   float a, b, c, check, t1, t2;
   float topt, bottomt;
   int tflag, bflag, t1flag, t2flag, iflag;
   vector4f plane;
   float rsquare;

   rsquare = radius*radius;

   plane[0] = plane[2] = 0;
   plane[1] = 1;
   plane[3] = length;
   bflag = line_plane_intersect(plane, pt, v, NULL, &bottomt);

   if (bflag) {
      a = pt[0] + bottomt*v[0];
      b = pt[2] + bottomt*v[2];

      if (a*a + b*b > rsquare)
         bflag = 0;
   }

   plane[3] = -length;
   tflag = line_plane_intersect(plane, pt, v, NULL, &topt);

   if (tflag) {
      a = pt[0] + topt*v[0];
      b = pt[2] + topt*v[2];

      if (a*a + b*b > rsquare)
         tflag = 0;
   }

   if (tflag && bflag) {
      *t = topt;
      *side = CYL_TOP;

      if (topt < 0) {
         if (bottomt < 0) {
            if (topt < bottomt) {
               *t = bottomt;
               *side = CYL_BOTTOM;
            }

            return 1;
         }

         *t = bottomt;          // signs are different
         *side = CYL_BOTTOM;
         return -1;
      }

      if (bottomt > 0) {
         if (topt > bottomt) {
            *t = bottomt;
            *side = CYL_BOTTOM;
         }

         return 1;
      }

      return -1;
   }

   b = pt[0]*v[0] + pt[2]*v[2];
   b += b;
   c = pt[0]*pt[0] + pt[2]*pt[2] - radius*radius;
   a = v[0]*v[0] + v[2]*v[2];

   check = b*b - 4*a*c;

   if (check < 0) {
      *t = 0;
      return 0;
   }

   check = (float)sqrt(check);
   a += a;

   t1 = (-b - check)/a;
   t2 = (-b + check)/a;

   a = pt[1] + t1*v[1];
   t1flag = (a < length && a > -length);

   a = pt[1] + t2*v[1];
   t2flag = (a < length && a > -length);

   if (!(t1flag || t2flag)) {
      *t = 0;
      return 0;
   }

   iflag = -1;

   *side = CYL_SIDE;

   if (t1flag) {
      iflag = (t1 > 0);
      *t = t1;
   }

   // note : some cases "*t" isnt set because it should be set by "if (t1flag) {" code
   if (t2flag) {

      if (iflag > -1) {
         if ((t2>0) == iflag) {
            if (*t < 0) {
               if (t2 > *t)
                  *t = t2;
            }

            else if (t2 < *t)
               *t = t2;

            return 1;
         }

         if (t2 > *t)   // signs are different
            *t = t2;

         return -1;
      }

      iflag = (t2 > 0);
      *t = t2;
   }

   if (tflag) {
      if ((topt>0) == iflag) {
         if (*t < 0) {
            if (topt > *t) {
               *side = CYL_TOP;
               *t = topt;
            }

         }

         else if (topt < *t) {
            *side = CYL_TOP;
            *t = topt;
         }

         return 1;
      }

      if (topt > *t) {  // different signs
         *side = CYL_TOP;
         *t = topt;
      }

      return -1;
   }

   if (bflag) {
      if ((bottomt>0) == iflag) {
         if (*t < 0) {
            if (bottomt > *t) {
               *side = CYL_BOTTOM;
               *t = bottomt;
            }

         }

         else if (bottomt < *t) {
            *side = CYL_BOTTOM;
            *t = bottomt;
         }

         return 1;
      }

      if (bottomt > *t) {               // different signs
         *side = CYL_BOTTOM;
         *t = bottomt;
      }

      return -1;
   }

   *t = 0;
   return 0;
}


/* **********************************************************************
   0 - top
   1 - bottom
********************************************************************** */
int line_cone_intersect(vector4f pt, vector4f v, float length, float *radius, float *t, int *side) {

   float a, b, c, d, e, f, check, t1, t2;
   float topt, bottomt;
   int tflag, bflag, t1flag, t2flag, iflag;
   vector4f plane;

   plane[0] = plane[2] = 0;
   plane[1] = 1;
   plane[3] = length;
   bflag = line_plane_intersect(plane, pt, v, NULL, &bottomt);

   if (bflag) {
      a = pt[0] + bottomt*v[0];
      b = pt[2] + bottomt*v[2];

      if (a*a + b*b > radius[1]*radius[1])
         bflag = 0;
   }

   plane[3] = -length;
   tflag = line_plane_intersect(plane, pt, v, NULL, &topt);

   if (tflag) {
      a = pt[0] + topt*v[0];
      b = pt[2] + topt*v[2];

      if (a*a + b*b > radius[0]*radius[0])
         tflag = 0;
   }

   if (tflag && bflag) {
      *t = topt;
      *side = CONE_TOP;

      if (topt < 0) {
         if (bottomt < 0) {
            if (topt < bottomt) {
               *t = bottomt;
               *side = CONE_BOTTOM;
            }

            return 1;
         }

         *t = bottomt;          // signs are different
         *side = CONE_BOTTOM;
         return -1;
      }

      if (bottomt > 0) {
         if (topt > bottomt) {
            *t = bottomt;
            *side = CONE_BOTTOM;
         }

         return 1;
      }

      return -1;                // signs are different
   }

   d = (radius[0] - radius[1])/(2*length);
   e = 0.5f*(radius[1] + radius[0]);
   f = d*d;

   check = d*e;
   c = f*pt[1];
   b = pt[0]*v[0] + pt[2]*v[2] - v[1]*(c + check);
   b += b;
   c = pt[0]*pt[0] + pt[2]*pt[2] - (pt[1]*(c + check + check) + e*e);
   a = v[0]*v[0] + v[2]*v[2] - v[1]*v[1]*f;

   check = b*b - 4*a*c;

   if (check < 0) {
      *t = 0;
      return 0;
   }

   check = (float)sqrt(check);
   a += a;

   t1 = (-b - check)/a;
   t2 = (-b + check)/a;

   a = pt[1] + t1*v[1];
   t1flag = (a < length && a > -length);

   a = pt[1] + t2*v[1];
   t2flag = (a < length && a > -length);

   if (!(t1flag || t2flag)) {
      *t = 0;
      return 0;
   }

   iflag = -1;

   *side = CONE_SIDE;

   if (t1flag) {
      iflag = (t1 > 0);
      *t = t1;
   }

   if (t2flag) {
      if (iflag > -1) {
         if ((t2>0) == iflag) {
            if (*t < 0) {
               if (t2 > *t)
                  *t = t2;
            }

            else if (t2 < *t)
               *t = t2;

            return 1;
         }

         if (t2 > *t)   // signs are different
            *t = t2;

         return -1;
      }

      iflag = (t2 > 0);
      *t = t2;
   }

   if (tflag) {
      if ((topt>0) == iflag) {
         if (*t < 0) {
            if (topt > *t) {
               *side = CONE_TOP;
               *t = topt;
            }

         }

         else if (topt < *t) {
            *side = CONE_TOP;
            *t = topt;
         }

         return 1;
      }

      if (topt > *t) {  // different signs
         *side = CONE_TOP;
         *t = topt;
      }

      return -1;
   }

   if (bflag) {
      if ((bottomt>0) == iflag) {
         if (*t < 0) {
            if (bottomt > *t) {
               *side = CONE_BOTTOM;
               *t = bottomt;
            }

         }

         else if (bottomt < *t) {
            *side = CONE_BOTTOM;
            *t = bottomt;
         }

         return 1;
      }

      if (bottomt > *t) {               // different signs
         *side = CONE_BOTTOM;
         *t = bottomt;
      }

      return -1;
   }

   *t = 0;
   return 0;
}
