

#include <stdlib.h>
#include <string.h>

#include "global.h"
#include "lw_load.h"
#include "pstring.h"


/* *******************************************************************
******************************************************************* */
class lwface_type : public dbl_llist {

   public:
      int *edgelist;
      int countedge;
      material_type *ilm;
      
      lwface_type() { edgelist = NULL; ilm = NULL; }
      virtual ~lwface_type() { if (edgelist) delete [] edgelist; }
};


/* *******************************************************************
******************************************************************* */
void lightwave_loader::internal_cleanup() {

   ob = NULL;
   current = NULL;

   if (vertex)
      delete [] vertex;

   vertex = NULL;

   i_manager.dest();
   f_manager.dest();
}


/* *******************************************************************
******************************************************************* */
void lightwave_loader::cleanup() {

   file_loader::cleanup();

   internal_cleanup();
}


/* *******************************************************************
******************************************************************* */
basic_loader *lightwave_loader::find_loader(sfile *data) {

   char buffer[8];
   int i;

   data->sseek(0);

   buffer[4] = 0;

   for (i=0; i<4; i++)
      buffer[i] = data->scan_uchar();

   if (!strcmp(buffer, "FORM")) {
      data->scan_uint();

      for (i=0; i<4; i++)
         buffer[i] = data->scan_uchar();

      if (!strcmp(buffer, "LWOB")) {
         sinfile = data;
         return this;
      }

   }
   
   return next ? ((basic_loader *)next)->find_loader(data) : NULL;
}


/* *******************************************************************
   negate z to get "pixcon" coord system
******************************************************************* */
int lightwave_loader::scan_pnts(int *bytesize) {

   int i;

   countvertex = sinfile->scan_uint();
   (*bytesize) -= 4 + countvertex;

   if (*bytesize < 0)
      return 0;

   countvertex /= 12;

   vertex = new vector3f[countvertex];
   
   for (i=0; i < countvertex; i++) {
      vertex[i][0] = sinfile->scan_float();
      vertex[i][1] = sinfile->scan_float();
      vertex[i][2] = sinfile->scan_float();
   }

   return 1;
}


/* *******************************************************************
******************************************************************* */
int lightwave_loader::scan_srfs(int *bytesize) {

   int i, j;
   char buffer[MAXSTRLEN];
   material_type *mtr;
   
   j = sinfile->scan_uint();
   (*bytesize) -= 4 + j;

   if (*bytesize < 0)
      return 0;
      
   while (j > 0) {

      i_manager.append(mtr = new material_type, NULL);

      i = sinfile->scanstring(buffer, MAXSTRLEN);

      if (i < 0)
         return 0;

      mtr->name.stringcpy(buffer);
      mtr->material_id = i_manager.count-1;

      j -= i;

      if ((i & 0x01)) {		// if (odd # of bytes, read 1 more)
         sinfile->skip(1);
         j--;
      }

   }

   return 1;
}


/* *******************************************************************
******************************************************************* */
int lightwave_loader::scan_pols(int *bytesize) {

   int chunk_size;
   lwface_type *ftr;
   int i, j, k;
   
   chunk_size = sinfile->scan_uint();
   (*bytesize) -= 4 + chunk_size;

   if (*bytesize < 0)
      return 0;

   while (chunk_size > 0) {
      ftr = new lwface_type;

      ftr->countedge = sinfile->scan_ushort();

      chunk_size -= (1 + ftr->countedge + 1) << 1;

      ftr->edgelist = new int[ftr->countedge];
      
      for (i=0; i<ftr->countedge; i++)
         ftr->edgelist[i] = sinfile->scan_ushort();

      k = (short)sinfile->scan_ushort();

      if (ftr->countedge > 2) {
         i = abs(k) - 1;
         for (ftr->ilm = (material_type *)i_manager.head; ftr->ilm && i; ftr->ilm = (material_type *)ftr->ilm->next, i--);
         f_manager.append(ftr, NULL);
         if (ftr->ilm)
            ftr->ilm->facecount++;
      }

      else
         delete ftr;

      // detail polygons...
      if (k < 0) {
         k = sinfile->scan_ushort();
         chunk_size -= 2;

         for (j=0; j<k; j++) {
            ftr = new lwface_type;

            ftr->countedge = sinfile->scan_ushort();

            chunk_size -= (1 + ftr->countedge + 1) << 1;

            ftr->edgelist = new int[ftr->countedge];
      
            for (i=0; i<ftr->countedge; i++)
               ftr->edgelist[i] = sinfile->scan_ushort();

            i = (short)sinfile->scan_ushort();

            if (ftr->countedge > 2) {
               f_manager.append(ftr, NULL);
               for (ftr->ilm = (material_type *)i_manager.head; ftr->ilm && i; ftr->ilm = (material_type *)ftr->ilm->next, i--);
               if (ftr->ilm)
                  ftr->ilm->facecount++;
            }

            else
               delete ftr;
         }

      }

   }

   return chunk_size ? 0 : 1;
}


/* *******************************************************************
******************************************************************* */
int lightwave_loader::scan_surf(int *bytesize) {

   int chunk_size;
   int sub_chunk_size;
   int i;
   char buffer[MAXSTRLEN];
   material_type *mtr;
   
   chunk_size = sinfile->scan_uint();
   (*bytesize) -= 4 + chunk_size;

   if (*bytesize < 0)
      return 0;
   
   i = sinfile->scanstring(buffer, MAXSTRLEN);

   if (i < 1)
      return 0;

   chunk_size -= i;

   // if (odd # of bytes, read 1 more)
   if (i & 0x01) {
      sinfile->skip(1);
      chunk_size--;
   }

   for (mtr=(material_type *)i_manager.head; mtr && mtr->name.stringcmp(buffer); mtr=(material_type *)mtr->next);

   if (!mtr)
      return 0;

   while (chunk_size > 0) {

      buffer[4] = 0;
      for (i=0; i<4; i++)
         buffer[i] = sinfile->scan_uchar();

      sub_chunk_size = sinfile->scan_ushort();
      chunk_size -= 6 + sub_chunk_size;

	// read 4 bytes - rgb {extra}  surface color
      if (!strcmp(buffer, "COLR")) {

         if (sub_chunk_size != 4)
            return 0;

         mtr->shade[0] = sinfile->scan_uchar()/255.0f;
         mtr->shade[1] = sinfile->scan_uchar()/255.0f;
         mtr->shade[2] = sinfile->scan_uchar()/255.0f;
         sinfile->skip(1);
      }

      else if (!strcmp(buffer, "FLAG")) {

         if (sub_chunk_size != 2)
            return 0;

         i = sinfile->scan_ushort();

         if (i & 0x01)
            mtr->lum = 1.0;

         mtr->flag_dbl_side = (unsigned char)((i >> 8) & 0x01);
      }

      else if (!strcmp(buffer, "LUMI")) {
         
         if (sub_chunk_size != 2)
            return 0;

         mtr->lum = sinfile->scan_ushort()/256.0f;
      }

      else if (!strcmp(buffer, "DIFF")) {
         
         if (sub_chunk_size != 2)
            return 0;

         mtr->diffuse = sinfile->scan_ushort()/256.0f;
      }

      else if (!strcmp(buffer, "SPEC")) {

         if (sub_chunk_size != 2)
            return 0;

         mtr->specular = sinfile->scan_ushort()/256.0f;
      }

      else if (!strcmp(buffer, "GLOS")) {

         if (sub_chunk_size != 2)
            return 0;

         i = sinfile->scan_ushort();

         switch(i) {
            case 1024:
               mtr->specn = 7.0;
               break;
            case 256:
               mtr->specn = 5.0;
               break;
            case 64:
               mtr->specn = 3.0;
               break;
            default:
               mtr->specn = 1.0;
               break;
         }

      }

/*
   int texflag = MASK_NULL;


      else if (!strcmp(buffer, "DTEX")) {
         printf("DTEX...\n");

         texflag = MASK_DTEX;
         ptr->tob.init(MASK_DTEX);
         
         i = getstring(infile, buffer, MAXSTRLEN);

         if (i < 1) {
            printf("Underestimated string size (%d:%d)... Aborting...\n", i, MAXSTRLEN);
            fclose(infile);
            exit(1);
         }
 
         if ((i & 0x01)) 		// if (odd # of bytes, read 1 more)
            getuchar(infile);
      }

      else if (!strcmp(buffer, "STEX")) {
         printf("STEX...\n");

         texflag = MASK_STEX;
         if (ptr->tob.id != MASK_DTEX)
            ptr->tob.init(MASK_STEX);

         i = getstring(infile, buffer, MAXSTRLEN);

         if (i < 1) {
            printf("Underestimated string size (%d:%d)... Aborting...\n", i, MAXSTRLEN);
            fclose(infile);
            exit(1);
         }
 
         if ((i & 0x01)) 		// if (odd # of bytes, read 1 more)
            getuchar(infile);
      }

      else if (!strcmp(buffer, "CTEX")) {
         printf("CTEX...\n");

         texflag = MASK_CTEX;

         i = getstring(infile, buffer, MAXSTRLEN);

         if (i < 1) {
            printf("Underestimated string size (%d:%d)... Aborting...\n", i, MAXSTRLEN);
            fclose(infile);
            exit(1);
         }
 
         if ((i & 0x01)) 		// if (odd # of bytes, read 1 more)
            getuchar(infile);
      }

      else if (!strcmp(buffer, "RTEX")) {
         printf("RTEX...\n");

         texflag = MASK_RTEX;

         i = getstring(infile, buffer, MAXSTRLEN);

         if (i < 1) {
            printf("Underestimated string size (%d:%d)... Aborting...\n", i, MAXSTRLEN);
            fclose(infile);
            exit(1);
         }
 
         if ((i & 0x01)) 		// if (odd # of bytes, read 1 more)
            getuchar(infile);
      }

      else if (!strcmp(buffer, "TTEX")) {
         printf("TTEX...\n");

         texflag = MASK_TTEX;

         i = getstring(infile, buffer, MAXSTRLEN);

         if (i < 1) {
            printf("Underestimated string size (%d:%d)... Aborting...\n", i, MAXSTRLEN);
            fclose(infile);
            exit(1);
         }
 
         if ((i & 0x01)) 		// if (odd # of bytes, read 1 more)
            getuchar(infile);
      }

      else if (!strcmp(buffer, "BTEX")) {
         printf("BTEX...\n");

         texflag = MASK_BTEX;

         i = getstring(infile, buffer, MAXSTRLEN);

         if (i < 1) {
            printf("Underestimated string size (%d:%d)... Aborting...\n", i, MAXSTRLEN);
            fclose(infile);
            exit(1);
         }
 
         if ((i & 0x01)) 		// if (odd # of bytes, read 1 more)
            getuchar(infile);
      }

      else if (!strcmp(buffer, "LTEX")) {
         printf("LTEX...\n");

         texflag = MASK_LTEX;

         i = getstring(infile, buffer, MAXSTRLEN);

         if (i < 1) {
            printf("Underestimated string size (%d:%d)... Aborting...\n", i, MAXSTRLEN);
            fclose(infile);
            exit(1);
         }
 
         if ((i & 0x01)) 		// if (odd # of bytes, read 1 more)
            getuchar(infile);
      }

      else if (!strcmp(buffer, "TIMG")) {
         printf("TIMG...\n");

         if (texflag == ptr->tob.id) {
            i = ptr->tob.read_texname(infile);

            if (i < 1) {
               printf("Underestimated string size (%d:%d)... Aborting...\n", i, MAXSTRLEN);
               printf("Invalid bytesize (%d:%d)... Aborting...\n", i, sub_chunk_size);
               fclose(infile);
               exit(1);
            }

            if ((i & 0x01)) 		// if (odd # of bytes, read 1 more)
               getuchar(infile);
         }

         else {
            i = getstring(infile, buffer, MAXSTRLEN);

            if (i < 1) {
               printf("Underestimated string size (%d:%d)... Aborting...\n", i, MAXSTRLEN);
               printf("Invalid bytesize (%d:%d)... Aborting...\n", i, sub_chunk_size);
               fclose(infile);
               exit(1);
            }

            if ((i & 0x01)) 		// if (odd # of bytes, read 1 more)
               getuchar(infile);
         }

      }
*/

      else
         sinfile->skip(sub_chunk_size);
   }

   return 1;
}


/* ***********************************************************************
  LightWave 3D objects are stored as IFF files with a FORM type of
  LWOB.  Currently, a FORM LWOB must contain a PNTS chunk, a SRFS
  chunk, and a POLS chunk (in that order).  These may be followed by
  zero or more SURF chunks.  LightWave 2.0 also has the ability to
  save and load surface descriptions as FORM LWOB files that contain
  only a SURF chunk.
*********************************************************************** */
int lightwave_loader::read_data() {

   char buffer[8];
   int i;
   int  bytesize;

   if (!sinfile)
      return 0;

   ob = (polytype *)global_resource_manager->find_resource_object(RESOURCE_OBJECT, sinfile->filename.string, NULL);

   sinfile->sseek(0);

   buffer[4] = 0;

   for (i=0; i<4; i++)
      buffer[i] = sinfile->scan_uchar();

   if (strcmp(buffer, "FORM"))
      return 0;
      
   bytesize = sinfile->scan_uint();

   for (i=0; i<4; i++)
      buffer[i] = sinfile->scan_uchar();

   bytesize -= 4;

   if (strcmp(buffer, "LWOB"))
      return 0;

   for (i=0; i<4; i++)
      buffer[i] = sinfile->scan_uchar();

   bytesize -= 4;

   if (strcmp(buffer, "PNTS") || !scan_pnts(&bytesize)) {
      internal_cleanup();
      return 0;
   }
      
   for (i=0; i<4; i++)
      buffer[i] = sinfile->scan_uchar();

   bytesize -= 4;

   if (strcmp(buffer, "SRFS") || !scan_srfs(&bytesize)) {
      internal_cleanup();
      return 0;
   }

   for (i=0; i<4; i++)
      buffer[i] = sinfile->scan_uchar();

   bytesize -= 4;

   if (strcmp(buffer, "POLS") || !scan_pols(&bytesize)) {
      internal_cleanup();
      return 0;
   }
   
   while (bytesize > 0) {

      for (i=0; i<4; i++)
         buffer[i] = sinfile->scan_uchar();

      bytesize -= 4;

      if (strcmp(buffer, "SURF") || !scan_surf(&bytesize)) {
         internal_cleanup();
         return 0;
      }

   }

   extract(FILETYPE_SPG, sinfile->filename.string);
   return 1;
}


/* *******************************************************************
******************************************************************* */
int lightwave_loader::extract_object() {

   int countedge = 0;
   polygon_object_type *pot;
   int i, j;
   lwface_type *ftr;
   
   if (!vertex || !f_manager.head)
      return 0;

   if (!ob) {
      ob = new polytype;
      ob->dataname.stringcpy(&sinfile->filename);
   }

   ob->statusflag = STATUSFLAG_VOID;

   for (ftr = (lwface_type *)f_manager.head; ftr; ftr = (lwface_type *)ftr->next) {
      countedge += ftr->countedge;
      if (ftr->ilm && ftr->ilm->flag_dbl_side)
         countedge += ftr->countedge;
   } 
        
   ob->build(countvertex, f_manager.count, countedge);

   pot = (polygon_object_type *)ob->query_data();

   for (i=0; i<countvertex; i++) {
      copyarray3(pot->vlist[i], vertex[i]);
      pot->vlist[i][3] = 1.0f;
   }

   countedge = 0;
   
   for (i=0, ftr = (lwface_type *)f_manager.head; ftr; i++, ftr = (lwface_type *)ftr->next) {
      pot->flist[i].polynum = ftr->countedge;
      pot->flist[i].edgeptr = &pot->vindex[countedge];

      countedge += ftr->countedge;

      for (j=0; j<ftr->countedge; j++)
         pot->flist[i].edgeptr[j] = ftr->edgelist[ftr->countedge - (j+1)];

      if (ftr->ilm && ftr->ilm->flag_dbl_side) {
         i++;
         pot->flist[i].polynum = ftr->countedge;
         pot->flist[i].edgeptr = &pot->vindex[countedge];

         countedge += ftr->countedge;

         for (j=0; j<ftr->countedge; j++)
            pot->flist[i].edgeptr[j] = ftr->edgelist[j];
      }

   }

   ob->preprocess();
   ob->statusflag = STATUSFLAG_LOADABLE | STATUSFLAG_LOADED;
   return 1;
}


/* *******************************************************************
******************************************************************* */
void lightwave_loader::extract_shade(material_type *mtr, shadetype *ilm) {

   copyarray3(ilm->fka, mtr->shade);
   
   ilm->fkp[0] = mtr->shade[0]*mtr->diffuse;
   ilm->fkp[1] = mtr->shade[1]*mtr->diffuse;
   ilm->fkp[2] = mtr->shade[2]*mtr->diffuse;

   ilm->fks[0] = mtr->shade[0]*mtr->specular;
   ilm->fks[1] = mtr->shade[1]*mtr->specular;
   ilm->fks[2] = mtr->shade[2]*mtr->specular;

   ilm->flum[0] = mtr->shade[0]*mtr->lum;
   ilm->flum[1] = mtr->shade[1]*mtr->lum;
   ilm->flum[2] = mtr->shade[2]*mtr->lum;

   ilm->specn = mtr->specn;

   ilm->update_int();
}


/* *******************************************************************
******************************************************************* */
int lightwave_loader::extract_material() {

   material_type *mtr, *btr;
   int i;
   shade_block *sblock;
   lwface_type *ftr;
   
   if (!ob)
      return 0;

   if (!current) {
      current = new shadelist;
      current->dataname.stringcpy(&sinfile->filename);
      current->altname.stringcpy(&sinfile->filename);
   }

   current->replace_data(sblock = new shade_block);

   // find most commonly used material

   btr = (material_type *)i_manager.head;
   if (btr->flag_dbl_side)
      btr->facecount += btr->facecount;

   sblock->index_count = btr->facecount;

   for (mtr = (material_type *)i_manager.head->next; mtr; mtr = (material_type *)mtr->next) {
      if (mtr->flag_dbl_side)
         mtr->facecount += mtr->facecount;

      sblock->index_count += mtr->facecount;
      if (btr->facecount < mtr->facecount)
         btr = mtr;
   }

   // copy material palette
   sblock->shade_palette = new shadetype[sblock->shade_count = i_manager.count];

   extract_shade(btr, &sblock->shade_palette[0]);
   memcpy(&((shadelist *)current)->base, &sblock->shade_palette[0], sizeof(shadetype));

   for (mtr = (material_type *)i_manager.head, i=1; mtr; mtr = (material_type *)mtr->next)
      if (mtr != btr) {
         extract_shade(mtr, &sblock->shade_palette[i]);
         i++;
      }

   // copy palette indicies
   sblock->index_palette = new int[sblock->index_count];
   memset(sblock->index_palette, 0, sblock->index_count * sizeof(int));

   for (i=0, ftr = (lwface_type *)f_manager.head; ftr; ftr=(lwface_type *)ftr->next) {
      if (ftr->ilm == btr)
         sblock->index_palette[i] = 0;
      else if (ftr->ilm->material_id < btr->material_id)
         sblock->index_palette[i] = ftr->ilm->material_id + 1;
      else
         sblock->index_palette[i] = ftr->ilm->material_id;

      i++;

      if (ftr->ilm->flag_dbl_side) {
         sblock->index_palette[i] = sblock->index_palette[i-1];
         i++;
      }

   }

   ((shadelist *)current)->postprocess(ob->countobject);
   current->statusflag = STATUSFLAG_LOADABLE | STATUSFLAG_LOADED;
   return 1;
}


/* ***********************************************************************
*********************************************************************** */
void *lightwave_loader::extract(unsigned int type, char *altname, int level) {

   switch (type) {

      case FILETYPE_SPG:

         if (ob) {
            // Note: this could cause the same object to be submitted more than once
            if (!(ob->statusflag & STATUSFLAG_LOADED))
               extract_object();

            return ob;
         }

         extract_object();

         if (ob)
            global_resource_manager->register_resource_object(RESOURCE_OBJECT, ob);

         return ob;

      case FILETYPE_ILM:
         current = (resource_type *)global_resource_manager->find_resource_object(RESOURCE_MATERIAL, sinfile->filename.string, sinfile->filename.string);

         if (current) {
            if (!(current->statusflag & STATUSFLAG_LOADED))
               extract_material();

            return current;
         }

         extract_material();

         if (current) {
            global_resource_manager->register_resource_object(RESOURCE_MATERIAL, current);
            return current;
         }

         // try processing .ilm file
         current = (resource_type *)((frame_manager *)global_resource_manager)->read_ilm(altname, ob->countobject);

         if (current)
            current->dataname.stringcpy(&ob->dataname);

         return current;

      default:
         return NULL;
   }

}

