

#include <stdlib.h>
#include <string.h>

#include "hpx_load.h"
#include "global.h"
#include "pstring.h"

#define FLAG_HPX_WRAPU    0x001
#define FLAG_HPX_WRAPV    0x010
#define FLAG_HPX_BACKFACE 0x100


/* *******************************************************************
******************************************************************* */
class hpx_texture : public dbl_llist {

   public:
      string_type name;
      texbase *tob;
      int flags;

      hpx_texture() { flags = FLAG_HPX_WRAPU | FLAG_HPX_WRAPV; tob = NULL; }

      virtual ~hpx_texture() {}
};


/* *******************************************************************
******************************************************************* */
class hpx_triangle : public dbl_llist {

   public:
      int texture_index;
      int material_index;
      vector3i vindex;

      virtual ~hpx_triangle() {}
};


/* *******************************************************************
******************************************************************* */
void hpx_loader::cleanup() {
   
   file_loader::cleanup(); 

   texture_palette.dest();
   material_palette.dest();
   base_material_palette.dest();
   manager.obj_manager.dest();
   manager.parent_manager.dest();
   manager.switch_manager.dest();

   geo_manager.geometry_manager.dest();
   if (geo_manager.vertex_palette) {
      delete [] geo_manager.vertex_palette;
      geo_manager.vertex_palette = NULL;
   }

}


/* *******************************************************************
******************************************************************* */
int hpx_loader::query_levels() {

   return 1;
}


/* *******************************************************************
******************************************************************* */
basic_loader *hpx_loader::find_loader(sfile *data) {

   char token[MAXSTRLEN];

   sinfile = data;

   data->sseek(0);
   data->scan_token(token, MAXSTRLEN);

   if (!strcmp(token, HPX_VERSION_STR)) {

      sinfile->scan_token(token, MAXSTRLEN);
      sinfile->scan_token(token, MAXSTRLEN);

      // at least 1 material
      if (atoi(token))
         return this;
   }

   sinfile = NULL;
   return next ? ((basic_loader *)next)->find_loader(data) : NULL;
}


/* *******************************************************************
******************************************************************* */
int hpx_loader::read_textures(char *token, int texture_count, int *overread) {

   int i;
   int scan_flag = 0;
   hpx_texture *ptr;

   for (i=0; i<texture_count; i++) {
      texture_palette.append(ptr = new hpx_texture, NULL);

      if (!scan_flag)
         sinfile->scan_token(token, MAXSTRLEN);
      else
         scan_flag = 0;

      ptr->name.stringcpy(token);

      sinfile->scan_token(token, MAXSTRLEN);
      if (!strcmp(token, "0"))
         ptr->flags &= ~FLAG_HPX_WRAPU;
      else if (!strcmp(token, "1"))
         ptr->flags |= FLAG_HPX_WRAPU;
      else {
         scan_flag = 1;
         continue;
      }

      sinfile->scan_token(token, MAXSTRLEN);
      if (!strcmp(token, "0"))
         ptr->flags &= ~FLAG_HPX_WRAPV;
      else if (!strcmp(token, "1"))
         ptr->flags |= FLAG_HPX_WRAPV;
      else {
         scan_flag = 1;
         continue;
      }

      // note: documentation flakey - either this is a dbl sided face flag
      //       or a texture compression flag
      sinfile->scan_token(token, MAXSTRLEN);
      if (!strcmp(token, "0"))
         ptr->flags &= ~FLAG_HPX_BACKFACE;
      else if (!strcmp(token, "1"))
         ptr->flags |= FLAG_HPX_BACKFACE;
      else
         scan_flag = 1;

      ptr->tob = (texbase *)((frame_manager *)global_resource_manager)->read_tex(ptr->name.string);
      if (!ptr->tob) {
         sprintf(perror_buffer, "Warning: cannot locate \"%s\"...\n", ptr->name.string);
         pprintf(perror_buffer);
      }

   }

   *overread = scan_flag;
   return 1;
}


/* *******************************************************************
******************************************************************* */
int hpx_loader::read_materials(char *token, int material_count, int *overread) {

   int i, j;
   hpx_material *ptr;

   if (!material_count) {
      base_material_palette.append(ptr = new hpx_material, NULL);

      ptr->ka[0] = ptr->ka[1] = ptr->ka[2] = 0.5;
      ptr->kd[0] = ptr->kd[1] = ptr->kd[2] = 0.7;
      ptr->ks[0] = ptr->ks[1] = ptr->ks[2] = 0.3;
      ptr->ke[0] = ptr->ke[1] = ptr->ke[2] = 0;
      ptr->specn = 10.0;
      ptr->alpha = 255;
      return 1;
   }

   for (i=0; i<material_count; i++) {
      base_material_palette.append(ptr = new hpx_material, NULL);

      for (j=0; j<3; j++) {
         if (*overread)
            *overread = 0;
         else
            sinfile->scan_token(token, MAXSTRLEN);

         ptr->ka[j] = atof(token);
      }

      for (j=0; j<3; j++) {
         sinfile->scan_token(token, MAXSTRLEN);
         ptr->kd[j] = atof(token);
      }

      for (j=0; j<3; j++) {
         sinfile->scan_token(token, MAXSTRLEN);
         ptr->ks[j] = atof(token);
      }

      for (j=0; j<3; j++) {
         sinfile->scan_token(token, MAXSTRLEN);
         ptr->ke[j] = atof(token);
      }

      sinfile->scan_token(token, MAXSTRLEN);
      ptr->specn = atof(token);

      sinfile->scan_token(token, MAXSTRLEN);
      ptr->alpha = atof(token);
   }

   return 1;
}


/* *******************************************************************
******************************************************************* */
int hpx_loader::read_animation(dbl_llist_manager *manager, char *token, float x, float y, float z) {

   int i;
   int found = 0;
   dbl_llist *ptr;

   sinfile->scan_token(token, MAXSTRLEN); // frame count
   i = atoi(token);

   sinfile->scan_token(token, MAXSTRLEN); // some kind of time(r)

   while (i && !sinfile->seof()) {
      sinfile->scan_token(token, MAXSTRLEN);

      if (!strcmp(token, "Geometry")) {
         i--;

         if (read_geometry(manager, token, x, y, z)) {
            if (!found)
               found = 1;
            else {
               manager->remove(ptr = manager->tail);
               delete ptr;
            }

         }

         continue;
      }

   }

   return found;
}


/* *******************************************************************
******************************************************************* */
int hpx_loader::read_geometry(dbl_llist_manager *manager, char *token, float x, float y, float z) {

   int i, j;
   hpx_triangle *ttr;
   hpx_geometry *gtr;
   hpx_material *mtr, *ntr;
   hpx_material base;
   int vertex_count, triangle_count;
   int index;
   vector4i color;
   union {
      vector4uc ccolor;
      unsigned int icolor;
   };

   base.ka[0] = base.ka[1] = base.ka[2] = 5.0;
   base.kd[0] = base.kd[1] = base.kd[2] = 7.0;
   base.ks[0] = base.ks[1] = base.ks[2] = 3.0;
   base.ke[0] = base.ke[1] = base.ke[2] = 0.0;
   base.specn = 10.0;
   base.alpha = 255;

   // unknown number
   sinfile->scan_token(token, MAXSTRLEN);

   sinfile->scan_token(token, MAXSTRLEN);
   vertex_count = atoi(token);

   sinfile->scan_token(token, MAXSTRLEN);
   triangle_count = atoi(token);

   if (!vertex_count) {
      for (i=0; i<triangle_count; i++)
         for (j=0; j<5; j++)
            sinfile->scan_token(token, MAXSTRLEN);

      return 0;
   }

   if (!triangle_count) {
      for (i=0; i<vertex_count; i++)
         for (j=0; j<12; j++)
            sinfile->scan_token(token, MAXSTRLEN);

      return 0;
   }

   manager->append(gtr = new hpx_geometry, NULL);

   gtr->center[0] = x;
   gtr->center[1] = y;
   gtr->center[2] = z;

   gtr->vertex_palette = new hpx_vertex[gtr->vertex_count = vertex_count];

   // scan vertex data
   for (i=0; i<vertex_count; i++) {

      for (j=0; j<3; j++) {
         sinfile->scan_token(token, MAXSTRLEN);
         gtr->vertex_palette[i].pt[j] = atof(token);
      }

      for (j=0; j<2; j++) {
         sinfile->scan_token(token, MAXSTRLEN);
         gtr->vertex_palette[i].uv[j] = atof(token);
      }

      // vertex color (multiplier?) - 0..255
      for (j=0; j<3; j++) {
         sinfile->scan_token(token, MAXSTRLEN);
         gtr->vertex_palette[i].color[j] = atoi(token);
      }

      // floats - vertex normals?
      for (j=0; j<3; j++)
         sinfile->scan_token(token, MAXSTRLEN);

      // unknown
      sinfile->scan_token(token, MAXSTRLEN);
   }

   // scan triangle data
   for (i=0; i<triangle_count; i++) {
      gtr->geometry_manager.append(ttr = new hpx_triangle, NULL);

      sinfile->scan_token(token, MAXSTRLEN);
      ttr->texture_index = atoi(token);
      
      sinfile->scan_token(token, MAXSTRLEN);
      ttr->material_index = (ttr->texture_index == -1) ? -1 : atoi(token);

      for (j=2; j>-1; j--) {
         sinfile->scan_token(token, MAXSTRLEN);
         ttr->vindex[j] = atoi(token);
      }

      // no face color - average vertex colors
      if (ttr->material_index == -1) {
         index = 0;
         mtr = &base;
      }

      else {
         for (index=0, mtr = (hpx_material *)base_material_palette.head; mtr && index<ttr->material_index; index++)
            mtr = (hpx_material *)mtr->next;

         if (!mtr) {
            mtr = &base;
            index = base_material_palette.count;
         }

      }

      color[0] = color[1] = color[2] = 0;

      for (j=0; j<3; j++)
         addarray3(color, gtr->vertex_palette[ttr->vindex[j]].color);

      for (j=0; j<3; j++)
         ccolor[j] = color[j] / 3;
      ccolor[3] = 0;

      for (j=0, ntr = (hpx_material *)material_palette.tail; ntr && (ntr->color != icolor || ntr->source != index); j++, ntr = (hpx_material *)ntr->next);

      if (ntr)
         ttr->material_index = j;
      else {
         ttr->material_index = material_palette.count;
         material_palette.append(ntr = new hpx_material, NULL);

         for (j=0; j<3; j++) {
            ntr->ka[j] = (ccolor[j]*mtr->ka[j])/255.0;
            ntr->kd[j] = (ccolor[j]*mtr->kd[j])/255.0;
            ntr->ks[j] = (ccolor[j]*mtr->ks[j])/255.0;
            ntr->ke[j] = (ccolor[j]*mtr->ke[j])/255.0;
         }

         ntr->specn = mtr->specn;
         ntr->alpha = mtr->alpha;

         ntr->source = index;
         ntr->color = icolor;
      }

   }

   return 1;
}


/* *******************************************************************
******************************************************************* */
int hpx_loader::read_subface(dbl_llist_manager *manager, char *token, int argi, float x, float y, float z) {

   int found = 0;

   while (argi && !sinfile->seof()) {
      sinfile->scan_token(token, MAXSTRLEN);

      if (!strcmp(token, "Animation")) {
         argi--;
         found |= read_animation(manager, token, x, y, z);
         continue;
      }

      if (!strcmp(token, "Geometry")) {
         argi--;
         found |= read_geometry(manager, token, x, y, z);
         continue;
      }

   }

   return found;
}


/* *******************************************************************
******************************************************************* */
int hpx_loader::read_choice(choice_type *choice, char *token, int argi, float px, float py, float pz) {

   float x, y, z;
   choice_type *ctr;
   int found = 0;
   int i, j;
   int start;
   float near_dist;
   dbl_llist *ptr;

   for (start = 0; start != argi && !sinfile->seof(); start++) {
      sinfile->scan_token(token, MAXSTRLEN);

      if (!strcmp(token, "Animation")) {
         if (read_animation(&choice->obj_manager, token, px, py, pz)) {
            found++;
            ((index_type *)choice->obj_manager.tail)->id = start;
         }

         continue;
      }

      if (!strcmp(token, "Geometry")) {
         if (read_geometry(&choice->obj_manager, token, px, py, pz)) {
            found++;
            ((index_type *)choice->obj_manager.tail)->id = start;
         }

         continue;
      }

      if (!strcmp(token, "Subface")) {
         argi--;

         sinfile->scan_token(token, MAXSTRLEN);
         i = atoi(token);

         ptr = choice->obj_manager.tail;

         if (read_subface(&choice->obj_manager, token, i, px, py, pz)) {
            found++;
            for (ptr = ptr ? ptr->next : choice->obj_manager.head; ptr; ptr = ptr->next)
               ((index_type *)ptr)->id = start;
         }

         continue;
      }

      if (!strcmp(token, "DOF")) {
         sinfile->scan_token(token, MAXSTRLEN);  // # of objects
         i = atoi(token);

         // 12 floats - 3x1 translation + 3x3 rotation?
         for (j=0; j<12; j++)
            sinfile->scan_token(token, MAXSTRLEN); 

         sinfile->scan_token(token, MAXSTRLEN);  // ? <int>

         ctr = new choice_type;
         if (read_choice(ctr, token, i, px, py, pz)) {
            choice->parent_manager.append(ctr, NULL);
            found++;
            ctr->id = start;
         }

         else
            delete ctr;

         continue;
      }

      if (!strcmp(token, "Transform")) {
         sinfile->scan_token(token, MAXSTRLEN);  // # of objects
         i = atoi(token);

         sinfile->scan_token(token, MAXSTRLEN);  // x
         x = atof(token) + px;
         sinfile->scan_token(token, MAXSTRLEN);  // y
         y = atof(token) + py;
         sinfile->scan_token(token, MAXSTRLEN);  // z
         z = atof(token) + pz;

         sinfile->scan_token(token, MAXSTRLEN);  // ? <int>

         sinfile->scan_token(token, MAXSTRLEN);  // ? "Z", "Y"

         ctr = new choice_type;
         if (read_choice(ctr, token, i, x, y, z)) {
            choice->parent_manager.append(ctr, NULL);
            found++;
            ctr->id = start;
         }

         else
            delete ctr;

         continue;
      }

      if (!strcmp(token, "RotatingAnimation")) {
         sinfile->scan_token(token, MAXSTRLEN);  // # of objects
         i = atoi(token);

         sinfile->scan_token(token, MAXSTRLEN);  // ? <float>

         sinfile->scan_token(token, MAXSTRLEN);  // x ?
         x = atof(token) + px;
         sinfile->scan_token(token, MAXSTRLEN);  // y ?
         y = atof(token) + py;
         sinfile->scan_token(token, MAXSTRLEN);  // z ?
         z = atof(token) + pz;

         sinfile->scan_token(token, MAXSTRLEN);  // ? "y"

         ctr = new choice_type;
         if (read_choice(ctr, token, i, x, y, z)) {
            choice->parent_manager.append(ctr, NULL);
            found++;
            ctr->id = start;
         }

         else
            delete ctr;

         continue;
      }

      if (!strcmp(token, "LOD")) {
         sinfile->scan_token(token, MAXSTRLEN);  // # of objects
         i = atoi(token);

         sinfile->scan_token(token, MAXSTRLEN);  // near distance
         near_dist = atof(token);

         sinfile->scan_token(token, MAXSTRLEN);  // far distance

         ctr = new choice_type;
         if (read_choice(ctr, token, i, px, py, pz) && near_dist < CORRECT) {
            choice->parent_manager.append(ctr, NULL);
            found++;
            ctr->id = start;
         }

         else
            delete ctr;

         continue;
      }

      if (!strcmp(token, "LOD2")) {
         sinfile->scan_token(token, MAXSTRLEN);  // # of objects
         i = atoi(token);

         sinfile->scan_token(token, MAXSTRLEN);  // near distance - int/float ?
         near_dist = atof(token);

         sinfile->scan_token(token, MAXSTRLEN);  // far distance - int/float ?

         sinfile->scan_token(token, MAXSTRLEN);
         x = atof(token) + px;

         sinfile->scan_token(token, MAXSTRLEN);
         y = atof(token) + py;

         sinfile->scan_token(token, MAXSTRLEN);
         z = atof(token) + pz;

         ctr = new choice_type;
         if (read_choice(ctr, token, i, px, py, pz) && near_dist < CORRECT) {
            choice->parent_manager.append(ctr, NULL);
            found++;
            ctr->id = start;
         }

         else
            delete ctr;

         continue;
      }

      if (!strcmp(token, "Switch")) {
         sinfile->scan_token(token, MAXSTRLEN);
         i = atoi(token);

         for (j=0; j<=i; j++) // one extra
            sinfile->scan_token(token, MAXSTRLEN);

         ctr = new choice_type;
         if (read_choice(ctr, token, i, px, py, pz)) {
            choice->switch_manager.append(ctr, NULL);
            found++;
            ctr->id = start;
         }

         else
            delete ctr;

         continue;
      }

      if (!strcmp(token, "Parent")) {
         sinfile->scan_token(token, MAXSTRLEN);
         i = atoi(token);

         ctr = new choice_type;
         if (read_choice(ctr, token, i, px, py, pz)) {
            choice->parent_manager.append(ctr, NULL);
            found++;
            ctr->id = start;
         }

         else
            delete ctr;

         continue;
      }

      if (!strcmp(token, "LightString")) {

         sinfile->scan_token(token, MAXSTRLEN);  // # of children
         i = atoi(token);

         sinfile->scan_token(token, MAXSTRLEN);  // # of verts
         j = atoi(token);

         sinfile->scan_token(token, MAXSTRLEN);  // period - animation cycle time (seconds)
         sinfile->scan_token(token, MAXSTRLEN);  // time on (seconds)
         sinfile->scan_token(token, MAXSTRLEN);  // size (meters)
         sinfile->scan_token(token, MAXSTRLEN);  // min-size (meters)
         sinfile->scan_token(token, MAXSTRLEN);  // max-dist (meters)
         sinfile->scan_token(token, MAXSTRLEN);  // az (deg)
         sinfile->scan_token(token, MAXSTRLEN);  // el (deg)
         sinfile->scan_token(token, MAXSTRLEN);  // vfov (deg - 0..180)
         sinfile->scan_token(token, MAXSTRLEN);  // hfov (deg - 0..180)
         sinfile->scan_token(token, MAXSTRLEN);  // rot-rate (deg/sec)
         sinfile->scan_token(token, MAXSTRLEN);  // flags

//         for (; i; i--) {
//            //scan in children
//         }

         //scan in vertices
         for (; i; i--) {
            sinfile->scan_token(token, MAXSTRLEN);  // X
            sinfile->scan_token(token, MAXSTRLEN);  // Y
            sinfile->scan_token(token, MAXSTRLEN);  // Z
            sinfile->scan_token(token, MAXSTRLEN);  // Phase shift (seconds)
            sinfile->scan_token(token, MAXSTRLEN);  // A - intensity (0..1)
            sinfile->scan_token(token, MAXSTRLEN);  // R - 0..255
            sinfile->scan_token(token, MAXSTRLEN);  // G - 0..255
            sinfile->scan_token(token, MAXSTRLEN);  // B - 0..255
         }

         continue;
      }

   }

   return found;
}


/* *******************************************************************
******************************************************************* */
int hpx_loader::read_data() {

   int texture_count, material_count;
   char token[MAXSTRLEN];
   int overread = 0;

   if (!sinfile)
      return 0;

   sinfile->sseek(0);
   sinfile->scan_token(token, MAXSTRLEN);

   sinfile->scan_token(token, MAXSTRLEN);
   texture_count = atoi(token);

   sinfile->scan_token(token, MAXSTRLEN);
   material_count = atoi(token);

   read_textures(token, texture_count, &overread);
   read_materials(token, material_count, &overread);

   return read_choice(&manager, token, -1, 0, 0, 0);
}


/* *******************************************************************
   Note: currently should only have one hpx_geometry object at most!
******************************************************************* */
int hpx_loader::count_triangle_vertex(dbl_llist_manager *manager) {

   hpx_geometry *gtr;
   int count = 0;

   // absorb geometry
   for (gtr = (hpx_geometry *)manager->head; gtr; gtr = (hpx_geometry *)gtr->next) {

      if (!gtr->vertex_count || !gtr->geometry_manager.head)
         continue;

      count += gtr->vertex_count;
   }

   return count;
}


/* *******************************************************************
******************************************************************* */
int hpx_loader::count_parent_vertex(dbl_llist_manager *manager) {

   choice_type *ptr;
   int count = 0;

   for (ptr = (choice_type *)manager->head; ptr; ptr = (choice_type *)ptr->next)
      count += count_vertex(ptr);

   return count;
}


/* *******************************************************************
******************************************************************* */
int hpx_loader::count_single_switch_vertex(choice_type *choice, int *id) {

   hpx_geometry *gtr;
   choice_type *ptr;
   int switchid = -1;
   int count = 0;

   for (gtr = (hpx_geometry *)choice->obj_manager.head; gtr; gtr = (hpx_geometry *)gtr->next) {
      if (!gtr->vertex_count || !gtr->geometry_manager.head)
         continue;

      if (switchid == -1 || switchid > gtr->id) {
         switchid = gtr->id;
         count = gtr->vertex_count;
      }

      else if (switchid == gtr->id)
         count += gtr->vertex_count;
   }

   for (ptr = (choice_type *)choice->parent_manager.head; ptr; ptr = (choice_type *)ptr->next)
      if (switchid == -1 || switchid > ptr->id) {
         switchid = ptr->id;
         count = count_vertex(ptr);
      }

   for (ptr = (choice_type *)choice->switch_manager.head; ptr; ptr = (choice_type *)ptr->next)
      if (switchid == -1 || switchid > ptr->id) {
         switchid = ptr->id;
         count = count_vertex(ptr);
      }

   if (id)
      *id = switchid;

   return count;
}


/* *******************************************************************
******************************************************************* */
int hpx_loader::count_switch_vertex(dbl_llist_manager *manager) {

   choice_type *ptr;
   int count = 0;

   for (ptr = (choice_type *)manager->head; ptr; ptr = (choice_type *)ptr->next)
      count += count_single_switch_vertex(ptr, NULL);

   return count;
}


/* *******************************************************************
******************************************************************* */
int hpx_loader::count_vertex(choice_type *manager) {

   int count;

   count = count_triangle_vertex(&manager->obj_manager);
   count += count_parent_vertex(&manager->parent_manager);
   return count + count_switch_vertex(&manager->switch_manager);
}


/* *******************************************************************
   Note: currently should only have one hpx_geometry object at most!
******************************************************************* */
int hpx_loader::extract_triangle(hpx_geometry *geo_manager, hpx_geometry *gtr) {

   int i;
   int offset;
   hpx_triangle *ttr, *utr;

   // absorb geometry
   if (!gtr->vertex_count || !gtr->geometry_manager.head)
      return 0;

   offset = geo_manager->vertex_count;

   // copy vertex info
   for (i=0; i<gtr->vertex_count; i++, geo_manager->vertex_count++) {
      addeqarray3(geo_manager->vertex_palette[geo_manager->vertex_count].pt, gtr->vertex_palette[i].pt, gtr->center);
      copyarray2(geo_manager->vertex_palette[geo_manager->vertex_count].uv, gtr->vertex_palette[i].uv);
   }

   // copy face info
   for (ttr = (hpx_triangle *)gtr->geometry_manager.head; ttr; ttr = (hpx_triangle *)ttr->next) {
      geo_manager->geometry_manager.append(utr = new hpx_triangle, NULL);
      utr->texture_index = ttr->texture_index;
      utr->material_index = ttr->material_index;
      for (i=0; i<3; i++)
         utr->vindex[i] = ttr->vindex[i]+offset;
   }

   return 1;  
}


/* *******************************************************************
******************************************************************* */
int hpx_loader::extract_parent(hpx_geometry *geo_manager, choice_type *ptr) {

   return extract_geo(geo_manager, ptr);
}


/* *******************************************************************
******************************************************************* */
int hpx_loader::extract_switch(hpx_geometry *geo_manager, choice_type *ptr) {

   int id;
   int found = 0;
   hpx_geometry *gtr;
   choice_type *qtr;

   if (!count_single_switch_vertex(ptr, &id))
      return 0;

   for (gtr = (hpx_geometry *)ptr->obj_manager.head; gtr; gtr = (hpx_geometry *)gtr->next) {
      if (id == gtr->id && gtr->vertex_count && gtr->geometry_manager.head) {
         found = 1;
         extract_triangle(geo_manager, gtr);
      }

   }

   if (found)
      return 1;

   for (qtr = (choice_type *)ptr->parent_manager.head; qtr; qtr = (choice_type *)qtr->next)
      if (id == qtr->id)
         return extract_parent(geo_manager, qtr);
         
   for (qtr = (choice_type *)ptr->switch_manager.head; qtr; qtr = (choice_type *)qtr->next)
      if (id == qtr->id)
         return extract_parent(geo_manager, qtr);
         
   return 0;
}


/* *******************************************************************
******************************************************************* */
int hpx_loader::extract_geo(hpx_geometry *geo_manager, choice_type *manager) {

   int found = 0;
   hpx_geometry *gtr;
   choice_type *ptr;

   for (gtr = (hpx_geometry *)manager->obj_manager.head; gtr; gtr = (hpx_geometry *)gtr->next)
      found |= extract_triangle(geo_manager, gtr);

   for (ptr = (choice_type *)manager->parent_manager.head; ptr; ptr = (choice_type *)ptr->next)
      found |= extract_parent(geo_manager, ptr);

   for (ptr = (choice_type *)manager->switch_manager.head; ptr; ptr = (choice_type *)ptr->next)
      found |= extract_switch(geo_manager, ptr);

   return found;
}


/* *******************************************************************
******************************************************************* */
int hpx_loader::export_spg(polytype *ob) {

   int i, j;
   polygon_object_type *pot;
   hpx_triangle *ttr;

   geo_manager.geometry_manager.dest();
   if (geo_manager.vertex_palette)
      delete [] geo_manager.vertex_palette;

   i = count_vertex(&manager);

   if (!i)
      return 0;

   geo_manager.vertex_palette = new hpx_vertex[i];
   geo_manager.vertex_count = 0;

   if (!extract_geo(&geo_manager, &manager) || !geo_manager.vertex_count || !geo_manager.geometry_manager.count)
      return 0;

   ob->statusflag = STATUSFLAG_VOID;
   ob->build(geo_manager.vertex_count, geo_manager.geometry_manager.count, geo_manager.geometry_manager.count*3);
   pot = (polygon_object_type *)ob->query_data();

   for (i=0; i<geo_manager.vertex_count; i++) {
      copyarray3(pot->vlist[i], geo_manager.vertex_palette[i].pt);
      pot->vlist[i][3] = 1.0f;
   }

   ob->countedge = 0;
   for (i=0, ttr = (hpx_triangle *)geo_manager.geometry_manager.head; ttr; ttr = (hpx_triangle *)ttr->next, i++) {
      pot->flist[i].polynum = 3;
      pot->flist[i].edgeptr = pot->vindex+ob->countedge;
      ob->countedge += 3;

      for (j=0; j<3; j++)
         pot->flist[i].edgeptr[j] = ttr->vindex[j];
   }

   ob->preprocess();
   ob->statusflag = STATUSFLAG_LOADABLE | STATUSFLAG_LOADED;
   return 1;
}


/* *******************************************************************
******************************************************************* */
void hpx_loader::extract_material(hpx_material *src, shadetype *ilm) {

   int i;

   for (i=0; i<3; i++) {
      ilm->fka[i] = src->ka[i];
      ilm->fkp[i] = src->kd[i];
      ilm->fks[i] = src->ks[i];
      ilm->flum[i] = src->ke[i];
   }

   ilm->specn = src->specn;

   ilm->update_int();
}


/* *******************************************************************
******************************************************************* */
int hpx_loader::export_ilm(shadelist *ilm) {

   shade_block *sblock;
   hpx_triangle *ttr;
   int *temp;
   int base;
   int i;
   hpx_material *mbase, *mtr;

   if (!material_palette.head || !geo_manager.vertex_count || !geo_manager.geometry_manager.count)
      return 0;

   ilm->replace_data(sblock = new shade_block);

   // find most common used material
   temp = new int[material_palette.count];
   memset(temp, 0, material_palette.count*sizeof(int));

   for (ttr = (hpx_triangle *)geo_manager.geometry_manager.head; ttr; ttr = (hpx_triangle *)ttr->next)
      temp[ttr->material_index]++;

   base = 0;
   for (i=1; i<material_palette.count; i++)
      if (temp[base] < temp[i])
         base = i;

   for (i=0, mbase = (hpx_material *)material_palette.head; i < base; i++, mbase = (hpx_material *)mbase->next);

   // build block
   sblock->shade_palette = new shadetype[sblock->shade_count = material_palette.count];

   extract_material(mbase, &sblock->shade_palette[0]);
   memcpy(&ilm->base, &sblock->shade_palette[0], sizeof(shadetype));

   for (mtr=(hpx_material *)material_palette.head, i=1; mtr; mtr=(hpx_material *)mtr->next)
      if (mtr != mbase) {
         extract_material(mtr, &sblock->shade_palette[i]);
         i++;
      }

   sblock->index_palette = new int[sblock->index_count = geo_manager.geometry_manager.count];

   for (ttr = (hpx_triangle *)geo_manager.geometry_manager.head, i=0; ttr; ttr = (hpx_triangle *)ttr->next, i++)
      if (ttr->material_index == base)
         sblock->index_palette[i] = 0;
      else if (ttr->material_index < base)
         sblock->index_palette[i] = ttr->material_index+1;
      else
         sblock->index_palette[i] = ttr->material_index;

   ilm->postprocess(geo_manager.geometry_manager.count);
   ilm->statusflag = STATUSFLAG_LOADABLE | STATUSFLAG_LOADED;
   return 1;
}



/* *******************************************************************
******************************************************************* */
int hpx_loader::export_tex(texpolygon *texpoly) {

   int i, j;
   texface *tblock;
   hpx_triangle *ttr;
   hpx_texture *tex;
   int found = 0;

   if (!texture_palette.head || !geo_manager.vertex_count || !geo_manager.geometry_manager.count)
      return 0;

   // check for at least 1 texture
   for (ttr=(hpx_triangle *)geo_manager.geometry_manager.head; ttr && !found; ttr=(hpx_triangle *)ttr->next) {
      for (i=0, tex = (hpx_texture *)texture_palette.head; tex && i<ttr->texture_index; tex = (hpx_texture *)tex->next);

      if (tex)
         found = (int)tex->tob;
   }

   if (!found)
      return 0;

   texpoly->statusflag = STATUSFLAG_VOID;
   texpoly->setup(geo_manager.geometry_manager.count);
   tblock = (texface *)texpoly->query_data();

   for (i=0, ttr=(hpx_triangle *)geo_manager.geometry_manager.head; ttr; ttr=(hpx_triangle *)ttr->next, i++) {

//      if (!ttr->texture_index == -1)
      if (ttr->texture_index == -1)
         continue;

      for (j=0, tex = (hpx_texture *)texture_palette.head; tex && j<ttr->texture_index; tex = (hpx_texture *)tex->next);

      // invalid texture id;
      if (!tex || !tex->tob)
         continue;

      tblock[i].setup(4);
      tblock[i].ob = tex->tob;

      for (j=0; j<3; j++)
         copyarray2(tblock[i].uv[j], geo_manager.vertex_palette[ttr->vindex[j]].uv);

      copyarray2(tblock[i].uv[j], tblock[i].uv[0]);

      tblock[i].calc_area();
   }

   texpoly->statusflag = STATUSFLAG_LOADABLE | STATUSFLAG_LOADED;
   return 1;
}


/* *******************************************************************
******************************************************************* */
void *hpx_loader::extract(unsigned int type, char *altname, int level) {

   resource_type *current;

   switch (type) {

      case FILETYPE_SPG:
         current = (resource_type *)global_resource_manager->find_resource_object(RESOURCE_OBJECT, sinfile->filename.string, sinfile->filename.string);

         if (current) {
            if (!(current->statusflag & STATUSFLAG_LOADED))
               export_spg((polytype *)current);

            return current;
         }

         current = new polytype;

         if (!export_spg((polytype *)current)) {
            delete current;
            return NULL;
         }

         current->dataname.stringcpy(&sinfile->filename);
         global_resource_manager->register_resource_object(RESOURCE_OBJECT, current);
         return current;

      case FILETYPE_ILM:
         current = (resource_type *)global_resource_manager->find_resource_object(RESOURCE_MATERIAL, sinfile->filename.string, sinfile->filename.string);

         if (current) {
            if (!(current->statusflag & STATUSFLAG_LOADED))
               export_ilm((shadelist *)current);

            return current;
         }

         current = new shadelist;

         if (!export_ilm((shadelist *)current)) {
            delete current;
            return NULL;
         }

         current->dataname.stringcpy(&sinfile->filename);
         current->altname.stringcpy(&sinfile->filename);
         global_resource_manager->register_resource_object(RESOURCE_MATERIAL, current);
         return current;

      case FILETYPE_TEX:
         current = (resource_type *)global_resource_manager->find_resource_object(RESOURCE_UVCOORD, sinfile->filename.string, sinfile->filename.string);

         if (current) {
            if (!(current->statusflag & STATUSFLAG_LOADED))
               export_tex((texpolygon *)current);

            return current;
         }

         current = new texpolygon;

         if (!export_tex((texpolygon *)current)) {
            delete current;
            return NULL;
         }

         current->dataname.stringcpy(&sinfile->filename);
         current->altname.stringcpy(&sinfile->filename);
         global_resource_manager->register_resource_object(RESOURCE_UVCOORD, current);
         return current;

      default:
         return NULL;
   }

}


/* *******************************************************************
   Auto acceptance....
******************************************************************* */
basic_loader *hps_loader::find_loader(sfile *data) {

   sinfile = data;
   return this;
}


/* *******************************************************************
******************************************************************* */
int hps_loader::read_data() {

   int texture_count;
   char token[MAXSTRLEN];
   int overread = 0;
   hpx_material *mtr;

   if (!sinfile)
      return 0;

   sinfile->sseek(0);

   // textures
   sinfile->scan_token(token, MAXSTRLEN);
   texture_count = atoi(token);

   read_textures(token, texture_count, &overread);

   // materials
   material_palette.append(mtr = new hpx_material, NULL);
   mtr->ka[0] = mtr->ka[1] = mtr->ka[2] = 0.5;
   mtr->kd[0] = mtr->kd[1] = mtr->kd[2] = 0.7;
   mtr->ks[0] = mtr->ks[1] = mtr->ks[2] = 0.3;
   mtr->ke[0] = mtr->ke[1] = mtr->ke[2] = 0.0;
   mtr->specn = 10.0;
   mtr->alpha = 255;

   return read_choice(&manager, token, -1, 0, 0, 0);
}


/* *******************************************************************
******************************************************************* */
int hps_loader::read_textures(char *token, int texture_count, int *overread) {

   int i;
   hpx_texture *ptr;

   for (i=0; i<texture_count; i++) {
      texture_palette.append(ptr = new hpx_texture, NULL);

      sinfile->scan_token(token, MAXSTRLEN);
      ptr->name.stringcpy(token);

      ptr->tob = (texbase *)((frame_manager *)global_resource_manager)->read_tex(ptr->name.string);
      if (!ptr->tob) {
         sprintf(perror_buffer, "Warning: cannot locate \"%s\"...\n", ptr->name.string);
         pprintf(perror_buffer);
      }

   }

   *overread = 0;
   return 1;
}

/* *******************************************************************
******************************************************************* */
int hps_loader::read_geometry(dbl_llist_manager *manager, char *token, float x, float y, float z) {

   int i, j;
   hpx_triangle *ttr;
   hpx_geometry *gtr;
   int vertex_count, triangle_count;

   // unknown number
   sinfile->scan_token(token, MAXSTRLEN);

   sinfile->scan_token(token, MAXSTRLEN);
   vertex_count = atoi(token);

   sinfile->scan_token(token, MAXSTRLEN);
   triangle_count = atoi(token);

   if (!vertex_count) {
      for (i=0; i<triangle_count; i++)
         for (j=0; j<4; j++)
            sinfile->scan_token(token, MAXSTRLEN);

      return 0;
   }

   if (!triangle_count) {
      for (i=0; i<vertex_count; i++)
         for (j=0; j<8; j++)
            sinfile->scan_token(token, MAXSTRLEN);

      return 0;
   }

   manager->append(gtr = new hpx_geometry, NULL);

   gtr->center[0] = x;
   gtr->center[1] = y;
   gtr->center[2] = z;

   gtr->vertex_palette = new hpx_vertex[gtr->vertex_count = vertex_count];

   // scan vertex data
   for (i=0; i<vertex_count; i++) {

      for (j=0; j<3; j++) {
         sinfile->scan_token(token, MAXSTRLEN);
         gtr->vertex_palette[i].pt[j] = atof(token);
      }

      for (j=0; j<2; j++) {
         sinfile->scan_token(token, MAXSTRLEN);
         gtr->vertex_palette[i].uv[j] = atof(token);
      }

      // vertex color (multiplier/index?) - 0..255
      for (j=0; j<3; j++) {
         sinfile->scan_token(token, MAXSTRLEN);
         gtr->vertex_palette[i].color[j] = atoi(token);
      }

   }

   // scan triangle data
   for (i=0; i<triangle_count; i++) {
      gtr->geometry_manager.append(ttr = new hpx_triangle, NULL);

      sinfile->scan_token(token, MAXSTRLEN);
      ttr->texture_index = atoi(token);
      
      ttr->material_index = 0;

      for (j=2; j>-1; j--) {
         sinfile->scan_token(token, MAXSTRLEN);
         ttr->vindex[j] = atoi(token);
      }

   }

   return 1;
}
