

#include <string.h>

#include "global.h"
#include "flt_load.h"


#ifdef WIN32
#include <float.h>

#define finite(xxx) _finite(xxx)
#endif


/* *******************************************************************
******************************************************************* */
class dbl_llist_manager_list_type : public dbl_llist {

   public:
      dbl_llist_manager flist;
      int countedge;

      virtual ~dbl_llist_manager_list_type() { }
};


/* *******************************************************************
******************************************************************* */
class matrix_type : public dbl_llist {

   public:
      vector4f mx[4];

      virtual ~matrix_type() { }
};


/* *******************************************************************
******************************************************************* */
class tex : public dbl_llist {

   public:
      int index;
      string_type texname;

      virtual ~tex() { }
};


/* *******************************************************************
******************************************************************* */
class edge_type : public dbl_llist {

   public:
      int vertex_index;
      virtual ~edge_type() { }
};


/* *******************************************************************
******************************************************************* */
class mgface_type : public dbl_llist {

   public:
      int material_index;
      unsigned int flags;
      vector4uc color;
      unsigned int color_index;
      unsigned int color_name;
      int texture_index;
      work_material *ilm;

      dbl_llist_manager vlist;
   
      virtual ~mgface_type() { }
};


/* *******************************************************************
******************************************************************* */
material::material() {

   ambient[0] = ambient[1] = ambient[2] = 1.0f;
   diffuse[0] = diffuse[1] = diffuse[2] = 0.7f;
   specular[0] = specular[1] = specular[2] = 0.3f;
   lum[0] = lum[1] = lum[2] = 0;
   specn = 1;
   alpha = 1;
}


/* *******************************************************************
******************************************************************* */
void flt_loader::init() {

   vlist = NULL;
   clist = NULL;

   current_type = TYPE_NULL;
   countedge = countvertex = 0;
}


/* *******************************************************************
******************************************************************* */
void flt_loader::parse_subhead() {

   sub_header.opcode = sinfile->scan_short();
   sub_header.size = sinfile->scan_ushort();
   sinfile->sread((char *)sub_header.id, 8);
   sub_header.formatrev = sinfile->scan_uint();
   sub_header.rev = sinfile->scan_uint();
   sinfile->sread((char *)sub_header.date, 32);
   sub_header.groupid = sinfile->scan_short();
   sub_header.lodid = sinfile->scan_short();
   sub_header.obj_id = sinfile->scan_short();
   sub_header.polygon_id = sinfile->scan_short();
   sub_header.scale = sinfile->scan_short();
   sub_header.unit = (char)sinfile->scan_uchar();
   sub_header.texwhite = (char)sinfile->scan_uchar();
   sub_header.flags = sinfile->scan_int();
   sinfile->skip(120);
   sub_header.swlatlon[0] = sinfile->scan_double();
   sub_header.swlatlon[1] = sinfile->scan_double();
   sub_header.nelatlon[0] = sinfile->scan_double();
   sub_header.nelatlon[1] = sinfile->scan_double();
}


/* *******************************************************************
******************************************************************* */
void flt_loader::parse_lod(int size) {

   double in, out;
   int level = 0;
   int opcode;

   sinfile->skip(12); size -= 12;

   in = sinfile->scan_double(); size -= 8;
   out = sinfile->scan_double(); size -= 8;
   sinfile->skip(size);

   if (out-CORRECT < 0 && in > 0)
      return;

   while (!sinfile->seof()) {

      opcode = sinfile->scan_short();

      switch (opcode) {

         case OP_540_PUSH_LEVEL:
            opcode = sinfile->scan_short() - 4;
            sinfile->skip(opcode);
            level++;
            break;

         case OP_540_POP_LEVEL:
            if (level == -1) {  // should have "pushed" at least once
               sinfile->skip(-2);
               return;
            }

            opcode = sinfile->scan_short() - 4;
            sinfile->skip(opcode);
            level--;
            if (!level)
               return;

            break;

         case OP_540_OBJECT:
         case OP_540_GROUP:
         case OP_540_TEXT_COMMENT:
         case OP_540_PUSH_SUBFACE:
         case OP_540_POP_SUBFACE:
         case OP_540_BVOLUME_CENTER:
         case OP_540_BVOLUME_ORIENT:
         case OP_540_BBOX:
         case OP_540_LONG_ID:
         case OP_540_EYETRACK_PALETTE:
         case OP_540_FACE:
         case OP_540_LIGHT_POINT:
         case OP_540_DOF:
         case OP_540_VLIST:
         case OP_540_MORPH_VLIST:
         case OP_540_BSP:
         case OP_540_EXTERNAL_REF:
         case OP_540_LOD:
         case OP_540_SOUND:
         case OP_540_LIGHT:
         case OP_540_ROAD_SEGMENT:
         case OP_540_ROAD_PATH:
         case OP_540_CLIP_REGION:
         case OP_540_TEXT:
         case OP_540_SWITCH:
         case OP_540_CAT:
         case OP_540_EXTENSION:
         case OP_540_RESERVED124:
         case OP_540_BSPHERE:
         case OP_540_BCYLINDER:
         case OP_540_MATRIX:
         case OP_540_TRANSLATE:
            if (!level) {  // should have "pushed" at least once
               sinfile->skip(-2);
               return;
            }

            opcode = sinfile->scan_short() - 4;
            sinfile->skip(opcode);
            break;

         default:
            if (!level) {  // should have "pushed" at least once
               sinfile->skip(-2);
               return;
            }

            opcode = sinfile->scan_short() - 4;
            sinfile->skip(opcode);
            printf("WARNING: Indeterminate opcode %d... Skipping...\n", opcode);
            break;
      }

   }

}


/* *******************************************************************
******************************************************************* */
void flt_loader::parse_vertex_index(int size) {

   int count;
   edge_type *edgeptr;
   int i, j;

   if (current_type != TYPE_FACE) {
      sinfile->skip(size);
      return;
   }

   for (count = size>>2; count; count--) {
      ((mgface_type *)flist.head)->vlist.insert(edgeptr = new edge_type, NULL);

      j = sinfile->scan_int(); size -= 4;
      edgeptr->vertex_index = 0;

      for (i=0; i<countvertex && vlist[i].index != (unsigned int)j; i++);
      if (i != countvertex)
         edgeptr->vertex_index = i;
   }

}


/* *******************************************************************
******************************************************************* */
void flt_loader::parse_morph_vertex_index(int size) {

   int count;
   edge_type *edgeptr;
   int i;

   if (current_type != TYPE_FACE) {
      sinfile->skip(size);
      return;
   }

   for (count = size>>3; count; count--) {
      ((mgface_type *)flist.head)->vlist.insert(edgeptr = new edge_type, NULL);

      edgeptr->vertex_index = sinfile->scan_int(); size -= 4;
      sinfile->scan_int(); size -= 4; // ignore morph...

      for (i=0; i<countvertex; i++)
         if (vlist[i].index == (unsigned int)edgeptr->vertex_index) {
            edgeptr->vertex_index = i;
            break;
         }

      if (i == countvertex)
         edgeptr->vertex_index = 0;
   }

}


/* *******************************************************************
******************************************************************* */
void flt_loader::parse_face(int size, vector4f *mx) {

   mgface_type *ptr;
   unsigned int flagbits;
   int t1, t2;

// asdf put code to apply matrix to face

   if (flist.head)
      if (((mgface_type *)flist.head)->vlist.count > 2)
         countedge += ((mgface_type *)flist.head)->vlist.count;
      else {
         flist.remove(ptr = (mgface_type *)flist.head);
         delete ptr;
      }

   flist.insert(ptr = new mgface_type, NULL);

   sinfile->skip(16); size -= 16;

   ptr->color_name = sinfile->scan_ushort(); size -= 2;

   sinfile->skip(4); size -= 4;

   t1 = sinfile->scan_short(); size -= 2;
   t2 = sinfile->scan_short(); size -= 2;

   ptr->texture_index = (t2 != -1) ? t2 : t1;

   ptr->material_index = sinfile->scan_short(); size -= 2;

   sinfile->skip(12); size -= 12;

   flagbits = sinfile->scan_uint(); size -= 4;

   ptr->flags = VERTEX_NULL;

   if (!(flagbits & 0x4000000))
      ptr->flags |= (flagbits & 0x10000000) ? VERTEX_PACKED : VERTEX_COLOR;

   sinfile->skip(8); size -= 8;

   *(unsigned int *)ptr->color = sinfile->scan_uint(); size -= 4;

   sinfile->skip(8); size -= 8;

   ptr->color_index = sinfile->scan_uint(); size -= 4;

   sinfile->skip(size);
}


/* *******************************************************************
******************************************************************* */
void flt_loader::parse_vertex(int size, float scale) {

   int blocksize, bksize;
   int opcode;
   unsigned int flagbits;

   bksize = blocksize = sinfile->scan_int() - 8; size -= 4;

   sinfile->skip(size);

   if (vlist) {
      printf("WARNING: Detect multiple vertex palettes... Skipping...\n");
      sinfile->skip(blocksize);
      return;
   }

   for (countvertex=0; bksize > 0; countvertex++) {
      opcode = sinfile->scan_ushort();
      size = sinfile->scan_ushort();
      bksize -= size;
      size -= 4;
      sinfile->skip(size);
   }

   vlist = new vertex_type[countvertex];
   sinfile->skip(-blocksize);
   bksize = 8;

   for (countvertex=0; blocksize > 0; countvertex++) {
      opcode = sinfile->scan_ushort();
      size = sinfile->scan_ushort();

      vlist[countvertex].index = bksize;
      bksize += size;
      blocksize -= size;
      size -= 4;

      sinfile->skip(2); size -= 2;
      flagbits = sinfile->scan_ushort(); size -= 2;

      vlist[countvertex].flags =  (flagbits & 0x2000) ? VERTEX_NULL : VERTEX_COLOR;
      vlist[countvertex].flags |= (flagbits & 0x1000) ? VERTEX_PACKED : VERTEX_NULL;

      vlist[countvertex].pt[0] = (float)(sinfile->scan_double()*scale); size -= 8;
      vlist[countvertex].pt[1] = (float)(sinfile->scan_double()*scale); size -= 8;
      vlist[countvertex].pt[2] = (float)(sinfile->scan_double()*scale); size -= 8;

      switch (opcode) {

	 case OP_540_CVERTEX:
            *(unsigned int *)&vlist[countvertex].color = sinfile->scan_uint(); size -= 4;
            vlist[countvertex].color_index = sinfile->scan_uint(); size -= 4;
            break;

	 case OP_540_CNVERTEX:
            vlist[countvertex].normal[0] = sinfile->scan_float(); size -= 4;
            vlist[countvertex].normal[1] = sinfile->scan_float(); size -= 4;
            vlist[countvertex].normal[2] = sinfile->scan_float(); size -= 4;

            *(unsigned int *)&vlist[countvertex].color = sinfile->scan_uint(); size -= 4;
            vlist[countvertex].color_index = sinfile->scan_uint(); size -= 4;
            vlist[countvertex].flags |= VERTEX_NORMAL;
            break;

	 case OP_540_CTVERTEX:
            vlist[countvertex].uv[0] = sinfile->scan_float(); size -= 4;
            vlist[countvertex].uv[1] = sinfile->scan_float(); size -= 4;

            *(unsigned int *)&vlist[countvertex].color = sinfile->scan_uint(); size -= 4;
            vlist[countvertex].color_index = sinfile->scan_uint(); size -= 4;

            if (!finite(vlist[countvertex].uv[0]))
               vlist[countvertex].uv[0] = 0;
            if (!finite(vlist[countvertex].uv[1]))
               vlist[countvertex].uv[1] = 0;

            vlist[countvertex].flags |= VERTEX_UV;

            break;

	 case OP_540_CNTVERTEX:
            vlist[countvertex].normal[0] = sinfile->scan_float(); size -= 4;
            vlist[countvertex].normal[1] = sinfile->scan_float(); size -= 4;
            vlist[countvertex].normal[2] = sinfile->scan_float(); size -= 4;

            vlist[countvertex].uv[0] = sinfile->scan_float(); size -= 4;
            vlist[countvertex].uv[1] = sinfile->scan_float(); size -= 4;

            *(unsigned int *)&vlist[countvertex].color = sinfile->scan_uint(); size -= 4;
            vlist[countvertex].color_index = sinfile->scan_uint(); size -= 4;

            if (!finite(vlist[countvertex].uv[0]))
               vlist[countvertex].uv[0] = 0;
            if (!finite(vlist[countvertex].uv[1]))
               vlist[countvertex].uv[1] = 0;

            vlist[countvertex].flags |= VERTEX_NORMAL | VERTEX_UV;
            break;

         default:
            printf("No vertex handler %d... Skipping...\n", opcode);
            sinfile->skip(size);
            size = 0;
            break;
      }

      sinfile->skip(size);
   }

   sinfile->skip(blocksize);
}


/* *******************************************************************
******************************************************************* */
void flt_loader::parse_tex(int size) {

   tex *ptr;
   char buffer[200];
   
   tlist.insert(ptr = new tex, NULL);

   sinfile->sread(buffer, 200); size -= 200;
   ptr->texname.stringcpy(buffer);
   ptr->index = sinfile->scan_int(); size -= 4;

   sinfile->skip(size);
}


/* *******************************************************************
******************************************************************* */
void flt_loader::parse_material(int size) {

   material *ptr;

   mlist.insert(ptr = new material, NULL);

   ptr->index = sinfile->scan_int(); size -= 4;

   sinfile->skip(16); size -= 16;

   ptr->ambient[0] = sinfile->scan_float(); size -= 4;
   ptr->ambient[1] = sinfile->scan_float(); size -= 4;
   ptr->ambient[2] = sinfile->scan_float(); size -= 4;

   ptr->diffuse[0] = sinfile->scan_float(); size -= 4;
   ptr->diffuse[1] = sinfile->scan_float(); size -= 4;
   ptr->diffuse[2] = sinfile->scan_float(); size -= 4;

   ptr->specular[0] = sinfile->scan_float(); size -= 4;
   ptr->specular[1] = sinfile->scan_float(); size -= 4;
   ptr->specular[2] = sinfile->scan_float(); size -= 4;

   ptr->lum[0] = sinfile->scan_float(); size -= 4;
   ptr->lum[1] = sinfile->scan_float(); size -= 4;
   ptr->lum[2] = sinfile->scan_float(); size -= 4;

   ptr->specn = sinfile->scan_float(); size -= 4;
   ptr->alpha = sinfile->scan_float(); size -= 4;

   sinfile->skip(size);
}


/* *******************************************************************
******************************************************************* */
void flt_loader::parse_color_palette(int size) {

   if (clist) {
      printf("Warning: Multiple Color Palettes Detected...\n");
      sinfile->skip(size);
      return;
   }

   clist = new vector4uc[1024];

   sinfile->skip(128); size -= 128;
   sinfile->sread((char *)clist, 4096); size -= 4096;
   sinfile->skip(size);
}


/* *******************************************************************
******************************************************************* */
void flt_loader::parse_matrix(int size, vector4f *mx) {

   vector4f r[4];
   int i, j;

   for (i=0; i<4; i++)
      for (j=0; j<4; j++) {
         r[i][j] = sinfile->scan_float(); size -= 4;
      }

   sinfile->skip(size);

   // asdf need to check the order of the multiply - a*b or b*a
   matmatmulto(r, mx);
}


/* *******************************************************************
******************************************************************* */
void flt_loader::parse_pushsface(int size) {

   int opcode;

   sinfile->skip(size);

   do {
      opcode = sinfile->scan_ushort();
      size = sinfile->scan_ushort() - 4;
      if (opcode == OP_540_PUSH_SUBFACE)
         parse_pushsface(size);
      else
         sinfile->skip(size);

   } while (opcode != OP_540_POP_SUBFACE);

}



/* *******************************************************************
******************************************************************* */
void flt_loader::post_object() {

   mgface_type *ptr;
   dbl_llist_manager_list_type *mtr;

   if (flist.head)
      if (((mgface_type *)flist.head)->vlist.count > 2)
         countedge += ((mgface_type *)flist.head)->vlist.count;
      else {
         flist.remove(ptr = (mgface_type *)flist.head);
         delete ptr;
      }

   if (flist.head) {
      master_flist.append(mtr = new dbl_llist_manager_list_type, NULL);
      while (flist.head) {
         flist.remove(ptr = (mgface_type *)flist.head);
         mtr->flist.append(ptr, NULL);
      }

      mtr->countedge = countedge;
   }

}


/* *******************************************************************
******************************************************************* */
int flt_loader::export_spg(polytype *ob, int level) {

   int i, j;
   mgface_type *fptr;
   edge_type *edgeptr;
   polygon_object_type *pot;
   dbl_llist_manager_list_type *mtr;

   for (mtr = (dbl_llist_manager_list_type *)master_flist.head, i=0; mtr && i<level; mtr = (dbl_llist_manager_list_type *)mtr->next, i++);

   if (!mtr)
      return 0;

   ob->statusflag = STATUSFLAG_VOID;
   ob->build(countvertex, mtr->flist.count, mtr->countedge);
   pot = (polygon_object_type *)ob->query_data();

   for (i=0; i<countvertex; i++) {
      copyarray3(pot->vlist[i], vlist[i].pt);
      pot->vlist[i][3] = 1.0f;
   }

   ob->countedge = 0;

   for (i=0, fptr = (mgface_type *)mtr->flist.head; fptr; i++, fptr=(mgface_type *)fptr->next) {
      pot->flist[i].polynum = fptr->vlist.count;

      pot->flist[i].edgeptr = pot->vindex+ob->countedge;
      ob->countedge += fptr->vlist.count;

      for (j=0, edgeptr=(edge_type *)fptr->vlist.head; edgeptr; j++, edgeptr=(edge_type *)edgeptr->next)
         pot->flist[i].edgeptr[j] = edgeptr->vertex_index;
   }

   ob->preprocess();
   ob->statusflag = level ? STATUSFLAG_LOADED : (STATUSFLAG_LOADABLE | STATUSFLAG_LOADED);

   return 1;
}


/* *******************************************************************
******************************************************************* */
void flt_loader::extract_material(work_material *src, shadetype *ilm) {

   ilm->ka[0] = (int)(src->ilm->ambient[0] * src->facecolor[0]);
   ilm->ka[1] = (int)(src->ilm->ambient[1] * src->facecolor[1]);
   ilm->ka[2] = (int)(src->ilm->ambient[2] * src->facecolor[2]);

   ilm->kp[0] = (int)(src->ilm->diffuse[0] * src->facecolor[0]);
   ilm->kp[1] = (int)(src->ilm->diffuse[1] * src->facecolor[1]);
   ilm->kp[2] = (int)(src->ilm->diffuse[2] * src->facecolor[2]);

   ilm->ks[0] = (int)(src->ilm->specular[0] * src->facecolor[0]);
   ilm->ks[1] = (int)(src->ilm->specular[1] * src->facecolor[1]);
   ilm->ks[2] = (int)(src->ilm->specular[2] * src->facecolor[2]);

   ilm->lum[0] = (int)(src->ilm->lum[0] * src->facecolor[0]);
   ilm->lum[1] = (int)(src->ilm->lum[1] * src->facecolor[1]);
   ilm->lum[2] = (int)(src->ilm->lum[2] * src->facecolor[2]);

   ilm->specn = src->ilm->specn;

   ilm->update_float();
}


/* *******************************************************************
******************************************************************* */
int flt_loader::export_ilm(shadelist *ilm, int level) {

   dbl_llist_manager head;
   work_material *hptr, *base;
   material default_mtl;
   material *mptr;
   mgface_type *fptr;
   vector4uc color;
   unsigned int index;
   double div;
   int i;
   shade_block *sblock;
   dbl_llist_manager_list_type *mtr;

   for (mtr = (dbl_llist_manager_list_type *)master_flist.head, i=0; mtr && i<level; mtr = (dbl_llist_manager_list_type *)mtr->next, i++);

   if (!mtr)
      return 0;

   default_mtl.index = -1;

   for (fptr=(mgface_type *)mtr->flist.head; fptr; fptr=(mgface_type *)fptr->next) {
      if (fptr->material_index == -1)
         mptr = &default_mtl;
      else {
         for (mptr=(material *)mlist.head; mptr; mptr=(material *)mptr->next)
            if (mptr->index == fptr->material_index)
               break;

         if (!mptr)
            mptr = &default_mtl;
      }

      if (fptr->flags & VERTEX_PACKED) {
         color[0] = fptr->color[3];
         color[1] = fptr->color[2];
         color[2] = fptr->color[1];
         color[3] = fptr->color[0];
      }

      else if (fptr->flags & VERTEX_COLOR) {
         if (!clist) {
            printf("WARNING: Index color faces but no colortable...\n");
            color[0] = color[1] = color[2] = color[3] = 255;
         }

         else {
            index = fptr->color_index>>7;
            if (index > 1024) {
               printf("WARNING: Color index %d exceeds color entries (1024)\n", index);
               color[0] = color[1] = color[2] = color[3] = 255;
            }

            else {
               div = (fptr->color_index & 0x7f)/127.0;
               color[0] = (unsigned char)(clist[index][3]*div);
               color[1] = (unsigned char)(clist[index][2]*div);
               color[2] = (unsigned char)(clist[index][1]*div);
               color[3] = clist[index][0];
            }

         }

      }

      else
         color[0] = color[1] = color[2] = color[3] = 255;

      for (hptr=(work_material *)head.head; hptr; hptr=(work_material *)hptr->next)
         if (hptr->ilm == mptr && hptr->facecolor[0] == color[0] && hptr->facecolor[1] == color[1] && hptr->facecolor[2] == color[2] && hptr->facecolor[3] == color[3])
            break;

      if (!hptr) {
         head.insert(hptr = new work_material, NULL);
         hptr->ilm = mptr;
         hptr->material_id = head.count-1;
         copyarray4(hptr->facecolor, color);
      }

      hptr->count++;
      fptr->ilm = hptr;
   }

   ilm->replace_data(sblock = new shade_block);

   // find most common used material
   base = (work_material *)head.head;
   sblock->index_count = base->count;

   // note: head.head is "base", head.head->next is the start of the rest
   for (hptr=(work_material *)head.head->next; hptr; hptr=(work_material *)hptr->next) {
      sblock->index_count += hptr->count;
      if (base->count < hptr->count)
         base = hptr;
   }

   sblock->shade_palette = new shadetype[sblock->shade_count = head.count];

   extract_material(base, &sblock->shade_palette[0]);
   memcpy(&ilm->base, &sblock->shade_palette[0], sizeof(shadetype));

   for (hptr=(work_material *)head.head->next, i=1; hptr; hptr=(work_material *)hptr->next)
      if (hptr != base) {
         extract_material(hptr, &sblock->shade_palette[i]);
         i++;
      }

   sblock->index_palette = new int[sblock->index_count];

   for (fptr=(mgface_type *)mtr->flist.head, i = 0; fptr; fptr=(mgface_type *)fptr->next, i++)
      if (fptr->ilm == base)
         sblock->index_palette[i] = 0;
      else if (fptr->ilm->material_id < base->material_id)
         sblock->index_palette[i] = fptr->ilm->material_id+1;
      else
         sblock->index_palette[i] = fptr->ilm->material_id;

   ilm->postprocess(mtr->flist.count);
   ilm->statusflag = level ? STATUSFLAG_LOADED : (STATUSFLAG_LOADABLE | STATUSFLAG_LOADED);
   return 1;
}


/* *******************************************************************
******************************************************************* */
int flt_loader::export_tex(texpolygon *texpoly, int level) {

   mgface_type *fptr;
   edge_type *edgeptr;
   tex *tptr;
   int i, j;
   texbase *tob;
   texface *tblock;
   dbl_llist_manager_list_type *mtr;

   for (mtr = (dbl_llist_manager_list_type *)master_flist.head, i=0; mtr && i<level; mtr = (dbl_llist_manager_list_type *)mtr->next, i++);

   if (!mtr)
      return 0;

   for (i=0, fptr=(mgface_type *)mtr->flist.head; fptr; fptr=(mgface_type *)fptr->next) {
      if (fptr->texture_index == -1)
         continue;

      for (tptr=(tex *)tlist.head; tptr; tptr=(tex *)tptr->next)
        if (tptr->index == fptr->texture_index)
           if (find_file(tptr->texname.string, "r", TEXTURE_PATH.string, (char)PLATFORM_SLASH, NULL, NULL)) {
               i = 1;
               break;
            }

            else {
               printf("Cannot locate \"%s\"...\n", tptr->texname.string);
               tptr->index = -1;
            }

      if (i)
         break;
   }

   if (!i)
      return 0;

   texpoly->statusflag = STATUSFLAG_VOID;
   texpoly->setup(mtr->flist.count);
   tblock = (texface *)texpoly->query_data();

   for (i=0, fptr=(mgface_type *)mtr->flist.head; fptr; fptr=(mgface_type *)fptr->next, i++) {
      if (fptr->texture_index == -1)
         continue;

      for (tptr=(tex *)tlist.head; tptr; tptr=(tex *)tptr->next)
         if (tptr->index == fptr->texture_index) {
            tob = (texbase *)((frame_manager *)global_resource_manager)->read_tex(tptr->texname.string);
            if (tob)
               break;
            else {
               printf("Cannot locate \"%s\"...\n", tptr->texname.string);
               tptr->index = -1;
            }

         }

      if (!tptr)
         continue;

      tblock[i].setup(fptr->vlist.count+1);
      tblock[i].ob = tob;

      for (j=0,edgeptr=(edge_type *)fptr->vlist.head; edgeptr; edgeptr=(edge_type *)edgeptr->next, j++)
         copyarray2(tblock[i].uv[j], vlist[edgeptr->vertex_index].uv);

      copyarray2(tblock[i].uv[j], tblock[i].uv[0]);

      tblock[i].calc_area();
   }

   texpoly->statusflag = level ? STATUSFLAG_LOADED : (STATUSFLAG_LOADABLE | STATUSFLAG_LOADED);
   return 1;
}


/* *******************************************************************
******************************************************************* */
basic_loader *flt_loader::find_loader(sfile *data) {

   sinfile = data;
   data->sseek(0);

   parse_subhead();

   if (sub_header.opcode == OP_540_HEADER)
      return this;

   sinfile = NULL;
   return next ? ((basic_loader *)next)->find_loader(data) : NULL;
}


/* *******************************************************************
******************************************************************* */
int flt_loader::read_data() {

   int opcode;
   int size;
   dbl_llist_manager stacklist;
   dbl_llist_manager locallist;
   matrix_type *mx;
   float scale = 1;
   char buffer[8];

   if (!sinfile)
      return 0;

   sinfile->sseek(0);

   parse_subhead();

//   sinfile->skip(sub_header.size - sizeof(flt_sub_header)); - sizeof() is padded
   sinfile->skip(sub_header.size - 220);

   switch (sub_header.unit) {

      case FLT_METER:
         scale = 1.0f;
         break;

      case FLT_KILOMETER:
         scale = 1000.0f;
         break;

      case FLT_FEET:
         scale = 0.3048f;
         break;

      case FLT_INCH:
         scale = 0.0254f;
         break;

      default: // case FLT_KNOT:
         scale = 1852.0f;
         break;
   }

   stacklist.insert(mx = new matrix_type, NULL);
   init_mx(mx->mx);

   locallist.insert(mx = new matrix_type, NULL);
   init_mx(mx->mx);

   while (!sinfile->seof()) {

      opcode = sinfile->scan_ushort();
      size = sinfile->scan_ushort() - 4;

      switch (opcode) {

         case OP_540_LOD:
            parse_lod(size);
            break;

         case OP_540_VLIST:
            parse_vertex_index(size);
            break;

         case OP_540_MORPH_VLIST:
            parse_morph_vertex_index(size);
            break;

         case OP_540_FACE:
            current_type = TYPE_FACE;
            parse_face(size, ((matrix_type *)stacklist.head)->mx);
            break;

         case OP_540_VERTEX_PALETTE:
            parse_vertex(size, scale);
            break;

         case OP_540_COLOR_PALETTE:
            parse_color_palette(size);
            break;

         case OP_540_MATERIAL_PALETTE:
            parse_material(size);
            break;

         case OP_540_TEXTURE_PALETTE:
            parse_tex(size);
            break;

         case OP_540_PUSH_LEVEL:
            mx = new matrix_type;

            copymx4x4(mx->mx, ((matrix_type *)locallist.head)->mx);
            matmatmulto(((matrix_type *)stacklist.head)->mx, mx->mx);
            stacklist.insert(mx, NULL);

            locallist.insert(mx = new matrix_type, NULL);
            break;

         case OP_540_POP_LEVEL:
            stacklist.remove(mx = (matrix_type *)stacklist.head);
            delete mx;

            locallist.remove(mx = (matrix_type *)locallist.head);
            delete mx;

            break;

         case OP_540_MATRIX:
            parse_matrix(size, ((matrix_type *)locallist.head)->mx);
            break;

         case OP_540_PUSH_SUBFACE:
            parse_pushsface(size);
            break;

         case OP_540_OBJECT:
            sinfile->sread(buffer, 8);
            sinfile->skip(size-8);
            init_mx(((matrix_type *)locallist.head)->mx);

            post_object();

            current_type = TYPE_NULL;
            countedge = 0;
            break;

         case OP_540_GROUP:
            sinfile->sread(buffer, 8);
            sinfile->skip(size-8);
            init_mx(((matrix_type *)locallist.head)->mx);
            break;

         case OP_540_LIGHT_POINT:
            current_type = TYPE_LIGHT;
            sinfile->skip(size);
            break;

         // handled by OP_540_MATRIX
         case OP_540_ROTATE_EDGE:
         case OP_540_TRANSLATE:
         case OP_540_SCALE:
         case OP_540_ROTATE_POINT:
         case OP_540_ROTATESCALE_POINT:
         case OP_540_PUT_TRANSFORM:
         case OP_540_GENERAL_MATRIX:
            sinfile->skip(size);
            break;

         case OP_540_RESERVED124:
         case OP_540_SOUND:
         case OP_540_BVOLUME_CENTER:
         case OP_540_BVOLUME_ORIENT:
         case OP_540_BBOX:
         case OP_540_LONG_ID:
         case OP_540_EYETRACK_PALETTE:
         case OP_540_TEXT_COMMENT:
         case OP_540_LIGHT_PALETTE:
         case OP_540_TEXT:
         case OP_540_SWITCH:
         case OP_540_ROAD_SEGMENT:
         case OP_540_ROAD_PATH:
         case OP_540_BSPHERE:
         case OP_540_BCYLINDER:
            sinfile->skip(size);
            break;

         default:
            printf("No handler %d:%d... Skipping...\n", opcode, size);
            sinfile->skip(size);
            break;
      }

   }

   post_object();

   return countvertex && 
          master_flist.head &&
          ((dbl_llist_manager_list_type *)master_flist.head)->flist.head && 
          ((dbl_llist_manager_list_type *)master_flist.head)->countedge;
}


/* *******************************************************************
******************************************************************* */
void *flt_loader::extract(unsigned int type, char *altname, int level) {

   resource_type *current;
   string_type workname;
   char buffer[32];

   workname.stringcpy(&sinfile->filename);

   if (level) {
      sprintf(buffer, "__%d", level);
      workname.stringcat(buffer);
   }

   switch (type) {

      case FILETYPE_SPG:
         current = (resource_type *)global_resource_manager->find_resource_object(RESOURCE_OBJECT, workname.string, workname.string);

         if (current) {
            if (!(current->statusflag & STATUSFLAG_LOADED))
               export_spg((polytype *)current, level);

            return current;
         }

         current = new polytype;

         if (!export_spg((polytype *)current, level)) {
            delete current;
            return NULL;
         }

         current->dataname.stringcpy(&workname);
         global_resource_manager->register_resource_object(RESOURCE_OBJECT, current);
         return current;

      case FILETYPE_ILM:
         current = (resource_type *)global_resource_manager->find_resource_object(RESOURCE_MATERIAL, workname.string, workname.string);

         if (current) {
            if (!(current->statusflag & STATUSFLAG_LOADED))
               export_ilm((shadelist *)current, level);

            return current;
         }

         current = new shadelist;

         if (!export_ilm((shadelist *)current, level)) {
            delete current;
            return NULL;
         }

         current->dataname.stringcpy(&workname);
         current->altname.stringcpy(&workname);
         global_resource_manager->register_resource_object(RESOURCE_MATERIAL, current);
         return current;

      case FILETYPE_TEX:
         current = (resource_type *)global_resource_manager->find_resource_object(RESOURCE_UVCOORD, workname.string, workname.string);

         if (current) {
            if (!(current->statusflag & STATUSFLAG_LOADED))
               export_tex((texpolygon *)current, level);

            return current;
         }

         current = new texpolygon;

         if (!export_tex((texpolygon *)current, level)) {
            delete current;
            return NULL;
         }

         current->dataname.stringcpy(&workname);
         current->altname.stringcpy(&workname);
         global_resource_manager->register_resource_object(RESOURCE_UVCOORD, current);
         return current;

      default:
         return NULL;
   }

}


/* *******************************************************************
******************************************************************* */
int flt_loader::query_levels() {

   return master_flist.count;
}


/* *******************************************************************
******************************************************************* */
void flt_loader::cleanup() {
   
   file_loader::cleanup(); 

   if (vlist) delete [] vlist;
   if (clist) delete [] clist;

   mlist.dest();
   flist.dest();
   tlist.dest();
   master_flist.dest();
   
   init();
}
