

#include <string.h>

#include "pstring.h"
#include "polygon.h"

#include "3ds_load.h"


#define ID_MAIN3DS         0x4d4d
#define ID_EDIT3DS         0x3d3d
#define ID_MATERIAL        0xafff
#define ID_OBJECT          0x4000
#define ID_OBJECT_MESH     0x4100

#define ID_OBJECT_VERTICES 0x4110
#define ID_OBJECT_FACES    0x4120
#define ID_OBJECT_MATERIAL 0x4130
#define ID_OBJECT_UV       0x4140

#define ID_MATNAME         0xa000
#define ID_MATAMBIENT      0xa010
#define ID_MATDIFFUSE      0xa020
#define ID_MATSPECULAR     0xa030
#define ID_MATSHININESS    0xa040
#define ID_MATSHINYSTR     0xa041
#define ID_MATTRANSPARENCY 0xa050
#define ID_MATLUM          0xa084
#define ID_MATMAP          0xa200

#define ID_RGB1            0x0011
#define ID_RGB2            0x0012

#define ID_AMOUNT          0x0030

#define ID_MAP_FILE        0xa300
#define ID_MAP_USCALE      0xa354
#define ID_MAP_VSCALE      0xa356
#define ID_MAP_UOFFSET     0xa358
#define ID_MAP_VOFFSET     0xa35a
#define ID_MAP_ROTATE      0xa35c


/* *******************************************************************
******************************************************************* */
ilm3ds::ilm3ds() {

   ambient[0] = ambient[1] = ambient[2] = 0;
   diffuse[0] = diffuse[1] = diffuse[2] = 0;
   specular[0] = specular[1] = specular[2] = 0;

   tob = NULL;
   uvscale[0] = uvscale[1] = 1.0f;
   uvoffset[0] = uvoffset[1] = 0.0f;
   diff[0] = diff[1] = 0.0f;
   angle[0] = 1.0f;
   angle[1] = 0.0f;

   facecount = 0;
}

/* *******************************************************************
******************************************************************* */
void ilm3ds::convert(float *in, float *out) {

   vector2f st;
   
   st[0] = in[0] - uvoffset[0] - 0.5f;
   st[1] = in[1] - uvoffset[1] - 0.5f;

   // rotate texture
   out[0] = st[0] * angle[0] - st[1] * angle[1] + 0.5f;
   out[1] = st[0] * angle[1] + st[1] * angle[0] + 0.5f;

   out[0] = out[0]*uvscale[0] + diff[0];
   out[1] = out[1]*uvscale[1] + diff[1];
}


/* *******************************************************************
******************************************************************* */
void sd3_loader::cleanup() {

   file_loader::cleanup();

   ob = NULL;
   current = NULL;
   
   o_manager.dest();
   i_manager.dest();
}


/* *******************************************************************
******************************************************************* */
basic_loader *sd3_loader::find_loader(sfile *data) {

   data->sseek(0);

   if (data->scan_rushort() == ID_MAIN3DS) {
      sinfile = data;
      return this;
   }
   
   return next ? ((basic_loader *)next)->find_loader(data) : NULL;
}


/* *******************************************************************
******************************************************************* */
void sd3_loader::scan_string(string_type *string) {

   int i;
   
   for (i = 0; sinfile->ptr[i]; i++);

   string->stringcpy((char *)sinfile->ptr);
   sinfile->skip(i+1);
}


/* *******************************************************************
******************************************************************* */
int sd3_loader::extract_face_block(int length) {

   sd3_block_header block;
   object3ds *otr;
   int i;
   int ilm;
   string_type string;
   ilm3ds *itr;
   
   otr = (object3ds *)o_manager.tail;
   
   if (otr->face)
      delete [] otr->face;
	       
   otr->countobject = sinfile->scan_rushort();
   otr->face = new face3ds[otr->countobject];

   // Go through all of the faces in this object
   for (i=0; i<otr->countobject; i++) {
      otr->face[i].edgelist[0] = sinfile->scan_rushort();
      otr->face[i].edgelist[1] = sinfile->scan_rushort();
      otr->face[i].edgelist[2] = sinfile->scan_rushort();
      sinfile->scan_rushort();
   }         

   length -= 2 + (otr->countobject<<3);
            
   while (length > 0) {
      block.id = sinfile->scan_rushort();
      block.length = sinfile->scan_ruint();
      length -= block.length;

      switch (block.id) {

         case ID_OBJECT_MATERIAL:
            scan_string(&string);

            for (ilm = 0, itr = (ilm3ds *)i_manager.head; itr; ilm++, itr = (ilm3ds *)itr->next)
               if (!itr->name.stringcmp(&string))
	          break;
		  
            if (!itr) {
               i = sinfile->scan_rushort();
               sinfile->skip(2+i+i);
            }               

            else
               for (itr->facecount += (i = sinfile->scan_rushort()); i; i--)
                  otr->face[sinfile->scan_rushort()].ilm = ilm;

            break;

         default:
            sinfile->skip(block.length-6);
            break;
      }

   }

   return 1;
}


/* *******************************************************************
******************************************************************* */
int sd3_loader::extract_mesh_block(int length) {

   sd3_block_header block;
   unsigned int i;
   string_type string;
   object3ds *otr;
   
   o_manager.append(otr = new object3ds, NULL);
   
   while (length > 0) {
      block.id = sinfile->scan_rushort();
      block.length = sinfile->scan_ruint();
      length -= block.length;

      switch (block.id) {

         // This is the objects vertices
         case ID_OBJECT_VERTICES:
            if (otr->vertex) {
               i = sinfile->scan_rushort();
               if (i != (unsigned int)otr->countvertex) {
                  sprintf(perror_buffer, "Vertex count(%d) != UV count (%d)\n", i, otr->countvertex);
                  pprintf(perror_buffer);
               }

            }

            else {
               otr->countvertex = sinfile->scan_rushort();
               otr->vertex = new vertex3ds[otr->countvertex];
            }

            // Read in the number of vertices (int)
            for (i = 0; i < (unsigned int)otr->countvertex; i++) {
               otr->vertex[i].v[0] = sinfile->scan_rfloat();
               otr->vertex[i].v[1] = sinfile->scan_rfloat();
               otr->vertex[i].v[2] = sinfile->scan_rfloat();
            }
   
            break;	   

         // This is the objects face information
         case ID_OBJECT_FACES:
            extract_face_block(block.length-6);
            break;

         // This holds the UV texture coordinates for the object
         case ID_OBJECT_UV:
            if (otr->vertex) {
               i = sinfile->scan_rushort();
               if (i != (unsigned int)otr->countvertex) {
                  sprintf(perror_buffer, "Vertex count(%d) != UV count (%d)\n", otr->countvertex, i);
                  pprintf(perror_buffer);
               }

            }

            else {
               otr->countvertex = sinfile->scan_rushort();
               otr->vertex = new vertex3ds[otr->countvertex];
            }

            // Read in the number of vertices (int)
            for (i = 0; i < (unsigned int)otr->countvertex; i++) {
               otr->vertex[i].uv[0] = sinfile->scan_rfloat();
               otr->vertex[i].uv[1] = sinfile->scan_rfloat();
            }
   
            break;

         default:
            sinfile->skip(block.length-6);
            break;
      }

   }

   return 1;
}


/* *******************************************************************
******************************************************************* */
int sd3_loader::extract_object_block(int length) {

   sd3_block_header block;
   string_type string; 

   scan_string(&string);
   length -= string.stringlen() + 1;

   while (length > 0) {
      block.id = sinfile->scan_rushort();
      block.length = sinfile->scan_ruint();
      length -= block.length;

      switch (block.id) {

         // This lets us know that we are reading a new object
         case ID_OBJECT_MESH:
            extract_mesh_block(block.length-6);
	    break;

         default:
            sinfile->skip(block.length-6);
            break;
      }

   }

   return 1;
}


/* *******************************************************************
******************************************************************* */
int sd3_loader::extract_color_block(int length, unsigned char *color) {

   sd3_block_header block;

   while (length > 0) {
      block.id = sinfile->scan_rushort();
      block.length = sinfile->scan_ruint();
      length -= block.length;

      switch (block.id) {
         case ID_RGB1:
         case ID_RGB2:
            color[0] = sinfile->scan_uchar();
            color[1] = sinfile->scan_uchar();
            color[2] = sinfile->scan_uchar();
            break;

         default:
            sinfile->skip(block.length-6);
            break;
      }

   }

   return 1;
}


/* *******************************************************************
******************************************************************* */
int sd3_loader::extract_amount_block(int length) {

   sd3_block_header block;
   int i = 0;
   
   while (length > 0) {
      block.id = sinfile->scan_rushort();
      block.length = sinfile->scan_ruint();
      length -= block.length;

      switch (block.id) {
         case ID_AMOUNT:
            i = sinfile->scan_rushort();
            break;

         default:
            sinfile->skip(block.length-6);
            break;
      }

   }

   return i;
}


/* *******************************************************************
******************************************************************* */
int sd3_loader::extract_texture_block(int length) {

   sd3_block_header block;
   ilm3ds *itr;
   float temp;
   
   itr = (ilm3ds *)i_manager.tail;
      
   while (length > 0) {
      block.id = sinfile->scan_rushort();
      block.length = sinfile->scan_ruint();
      length -= block.length;

      switch (block.id) {

         // This stores the file name of the material
         case ID_MAP_FILE:
            scan_string(&itr->texname);
            sinfile->skip(block.length-(6 + itr->texname.stringlen() + 1));
            break;

         case ID_MAP_USCALE:
            itr->uvscale[0] = sinfile->scan_rfloat();
            itr->diff[0] = 0.5f*(itr->uvscale[0] - 1.0f);
            break;

         case ID_MAP_VSCALE:
            itr->uvscale[1] = sinfile->scan_rfloat();
            itr->diff[1] = 0.5f*(itr->uvscale[1] - 1.0f);
            break;

         case ID_MAP_UOFFSET:
            itr->uvoffset[0] = sinfile->scan_rfloat();
            break;

         case ID_MAP_VOFFSET:
            itr->uvoffset[1] = sinfile->scan_rfloat();
            break;

         case ID_MAP_ROTATE:
            temp = sinfile->scan_rfloat();
	    itr->angle[0] = (float)cos(temp);
	    itr->angle[1] = (float)sin(temp);
	    
            break;

         default:
            sinfile->skip(block.length-6);
            break;
      }

   }

   return 1;
}


/* *******************************************************************
******************************************************************* */
int sd3_loader::extract_material_block(int length) {

   sd3_block_header block;
   ilm3ds *itr;
   
   i_manager.append(itr = new ilm3ds, NULL);

   while (length > 0) {
      block.id = sinfile->scan_rushort();
      block.length = sinfile->scan_ruint();
      length -= block.length;

      switch (block.id) {

         // This chunk holds the name of the material
         case ID_MATNAME:
            scan_string(&itr->name);
            break;
         case ID_MATAMBIENT:
            extract_color_block(block.length-6, itr->diffuse);
            break;

         case ID_MATDIFFUSE:
            extract_color_block(block.length-6, itr->diffuse);
            break;

         case ID_MATSPECULAR:
            extract_color_block(block.length-6, itr->specular);
            break;

/* asdf
         case ID_MATSHININESS:
// 25	 
extract_amount_block(block.length-6);
            break;
		
         case ID_MATSHINYSTR:
// 5
extract_amount_block(block.length-6);
            break;
		
         case ID_MATLUM:
extract_amount_block(block.length-6);
// 0
            break;
*/

         // This is the header for the texture info
         case ID_MATMAP:
            extract_texture_block(block.length-6);
            break;

         default:
            sinfile->skip(block.length-6);
            break;
      }

   }

   return 1;
}


/* *******************************************************************
******************************************************************* */
int sd3_loader::extract_edit_block(int length) {

   sd3_block_header block;
   
   while (length > 0) {
      block.id = sinfile->scan_rushort();
      block.length = sinfile->scan_ruint();
      length -= block.length;

      switch (block.id) {

         case ID_OBJECT:
            extract_object_block(block.length-6);
            break;

         case ID_MATERIAL:
            extract_material_block(block.length-6);
            break;

	 default:
            sinfile->skip(block.length-6);
            break;
      }

   }

   return 1;
}


/* *******************************************************************
******************************************************************* */
int sd3_loader::extract_master_block(int length) {

   sd3_block_header block;

   while (length > 0) {
      block.id = sinfile->scan_rushort();
      block.length = sinfile->scan_ruint();
      length -= block.length;
      
      switch (block.id) {
      
         case ID_EDIT3DS:
            extract_edit_block(block.length-6);
            break;
 
	 default:
            sinfile->skip(block.length-6);
            break;
      }

   }

   return 1;
}


/* *******************************************************************
******************************************************************* */
int sd3_loader::read_data() {

   if (!sinfile)
      return 0;

   ob = (polytype *)global_resource_manager->find_resource_object(RESOURCE_OBJECT, sinfile->filename.string, NULL);

   sinfile->sseek(0);

   master_block.id = sinfile->scan_rushort();

   if (master_block.id != ID_MAIN3DS)
      return 0;

   master_block.length = sinfile->scan_ruint();

   extract_master_block(master_block.length-6);

   extract(FILETYPE_SPG, sinfile->filename.string);
   
   return 1;
}


/* *******************************************************************
******************************************************************* */
int sd3_loader::extract_object() {

   int countvertex, countobject, countedge;
   object3ds *otr;
   polygon_object_type *pot;
   int i;
   
   if (master_block.id != ID_MAIN3DS)
      return 0;

   countvertex = countobject = 0;

   for (otr = (object3ds *)o_manager.head; otr; otr = (object3ds *)otr->next) {
      countvertex += otr->countvertex;
      countobject += otr->countobject;
   }
   
   countedge = countobject * 3;

   if (!ob) {
      ob = new polytype;
      ob->dataname.stringcpy(&sinfile->filename);
   }

   ob->statusflag = STATUSFLAG_VOID;
   ob->build(countvertex, countobject, countedge);
   pot = (polygon_object_type *)ob->query_data();

   countvertex = countobject = countedge = 0;

   for (otr = (object3ds *)o_manager.head; otr; otr = (object3ds *)otr->next) {
      
      for (i=0; i<otr->countvertex; i++) {
         copyarray3(pot->vlist[countvertex + i], otr->vertex[i].v);
         pot->vlist[countvertex + i][3] = 1.0f;
      }

      countvertex += otr->countvertex;

      for (i=0; i<otr->countobject; i++) {
         pot->flist[countobject + i].polynum = 3;
         pot->flist[countobject + i].edgeptr = &pot->vindex[countedge];

         countedge += 3;

         pot->flist[countobject + i].edgeptr[0] = otr->face[i].edgelist[2];
         pot->flist[countobject + i].edgeptr[1] = otr->face[i].edgelist[1];
         pot->flist[countobject + i].edgeptr[2] = otr->face[i].edgelist[0];
      }

      countobject += otr->countobject;
   }

   ob->countedge = countedge;

   ob->preprocess();
   ob->statusflag = STATUSFLAG_LOADABLE | STATUSFLAG_LOADED;
   return 1;
}


/* *************************************************************
************************************************************* */
void sd3_loader::extract_shade(ilm3ds *itr, shadetype *ilm) {

   copyarray3(ilm->ka, itr->ambient);
   copyarray3(ilm->kp, itr->diffuse);
   copyarray3(ilm->ks, itr->specular);

   ilm->specn = 1;
   ilm->lum[0] = ilm->lum[1] = ilm->lum[2] = 0;
   ilm->update_float();
}


/* *******************************************************************
******************************************************************* */
int sd3_loader::extract_material() {

   shade_block *sblock;
   int i, j, base;
   ilm3ds *btr, *itr;
   object3ds *otr;
   
   if (master_block.id != ID_MAIN3DS || !i_manager.head) {
      current = NULL;
      return 0;
   }

   if (!current) {
      current = new shadelist;
      current->dataname.stringcpy(&sinfile->filename);
      current->altname.stringcpy(&sinfile->filename);
   }

   current->replace_data(sblock = new shade_block);

   // find most commonly used material
   base = 0;
   i = 1;
   btr = (ilm3ds *)i_manager.head;
   sblock->index_count = btr->facecount;

   for (itr = (ilm3ds *)i_manager.head->next; itr; i++, itr = (ilm3ds *)itr->next) {
      sblock->index_count += itr->facecount;
      if (btr->facecount < itr->facecount) {
         btr = itr;
         base = i;
      }

   }

   // copy material palette
   sblock->shade_palette = new shadetype[sblock->shade_count = i_manager.count];

   extract_shade(btr, &sblock->shade_palette[0]);
   memcpy(&((shadelist *)current)->base, &sblock->shade_palette[0], sizeof(shadetype));

   for (itr = (ilm3ds *)i_manager.head, i=1; itr; itr = (ilm3ds *)itr->next)
      if (itr != btr) {
         extract_shade(itr, &sblock->shade_palette[i]);
         i++;
      }

   // copy palette indicies
   sblock->index_palette = new int[sblock->index_count];
   memset(sblock->index_palette, 0, sblock->index_count * sizeof(int));

   for (i=0, otr = (object3ds *)o_manager.head; otr; otr = (object3ds *)otr->next)
      for (j=0; j<otr->countobject; j++, i++)
         if (otr->face[j].ilm > -1)
            if (otr->face[j].ilm == base)
               sblock->index_palette[i] = 0;
            else if (otr->face[j].ilm < base)
               sblock->index_palette[i] = otr->face[j].ilm + 1;
            else
               sblock->index_palette[i] = otr->face[j].ilm;

   ((shadelist *)current)->postprocess(ob->countobject);
   current->statusflag = STATUSFLAG_LOADABLE | STATUSFLAG_LOADED;
   return 1;
}


/* *******************************************************************
******************************************************************* */
int sd3_loader::extract_uv() {

   ilm3ds **bucket;
   int i, j;
   ilm3ds *itr;
   int count;
   vector2f uv[3];
   object3ds *otr;
    
   if (master_block.id != ID_MAIN3DS || !i_manager.head) {
      current = NULL;
      return 0;
   }

   bucket = new ilm3ds *[i_manager.count];

   for (i=0, count=0, itr=(ilm3ds *)i_manager.head; itr; i++, itr=(ilm3ds *)itr->next) {
      bucket[i] = itr;
      if (itr->texname.string[0]) {
         count += itr->facecount;

	 if (!bucket[i]->tob)
            bucket[i]->tob = (texbase *)((frame_manager *)global_resource_manager)->read_tex(bucket[i]->texname.string);
      }

   }

   if (!count) {
      current = NULL;
      delete [] bucket;
      return 0;
   }

   if (!current) {
      current = new texpolygon;
      current->dataname.stringcpy(&sinfile->filename);
      current->altname.stringcpy(&sinfile->filename);
   }

   ((texpolygon *)current)->setup(ob->countobject);

   for (i=0, otr=(object3ds *)o_manager.head; otr; otr=(object3ds *)otr->next)
      for (j=0; j<otr->countobject; j++, i++)
         if (otr->face[j].ilm > -1 && bucket[otr->face[j].ilm]->tob) {
            bucket[otr->face[j].ilm]->convert(otr->vertex[otr->face[j].edgelist[0]].uv, uv[2]);
            bucket[otr->face[j].ilm]->convert(otr->vertex[otr->face[j].edgelist[1]].uv, uv[1]);
            bucket[otr->face[j].ilm]->convert(otr->vertex[otr->face[j].edgelist[2]].uv, uv[0]);

            ((texpolygon *)current)->direct2Dmap(i, uv, bucket[otr->face[j].ilm]->tob, 3, 0);
         }
   
   current->statusflag = STATUSFLAG_LOADABLE | STATUSFLAG_LOADED;
   delete [] bucket;
   return 1; 
}


/* *******************************************************************
******************************************************************* */
void *sd3_loader::extract(unsigned int type, char *altname, int level) {

   switch (type) {

      case FILETYPE_SPG:

         if (ob) {
            if (!(ob->statusflag & STATUSFLAG_LOADED))
               extract_object();

            return ob;
         }

         extract_object();

         if (ob)
            global_resource_manager->register_resource_object(RESOURCE_OBJECT, ob);

         return ob;

      case FILETYPE_ILM:
         current = (resource_type *)global_resource_manager->find_resource_object(RESOURCE_MATERIAL, sinfile->filename.string, sinfile->filename.string);

         if (current) {
            if (!(current->statusflag & STATUSFLAG_LOADED))
               extract_material();

            return current;
         }

         extract_material();

         if (current) {
            global_resource_manager->register_resource_object(RESOURCE_MATERIAL, current);
            return current;
         }

         // try processing .ilm file
         current = (resource_type *)((frame_manager *)global_resource_manager)->read_ilm(altname, ob->countobject);

         if (current)
            current->dataname.stringcpy(&ob->dataname);

         return current;

      case FILETYPE_TEX:
         current = (resource_type *)global_resource_manager->find_resource_object(RESOURCE_UVCOORD, sinfile->filename.string, sinfile->filename.string);

         if (current) {
            if (!(current->statusflag & STATUSFLAG_LOADED))
               extract_uv();

            return current;
         }

         extract_uv();

         if (current)
            global_resource_manager->register_resource_object(RESOURCE_UVCOORD, current);

         return current;

      default:
         return NULL;
   }

}

