

#include "base.h"

#if !defined __QUARK_H
#define __QUARK_H


#define QUARK_FLAG_ACTIVE  0x00000001
#define QUARK_FLAG_RECALC  0x00000002
#define QUARK_FLAG_VISIBLE 0x00000004

#define EVENT_STANDARD_ANITROLL 0x01

#define STATE_FLAG_NULL        0x0000

#define STATE_FLAG_BNONE       0x0000
#define STATE_FLAG_BPOINT      0x0001
#define STATE_FLAG_BSPHERE     0x0002
#define STATE_FLAG_BPLANE      0x0003
#define STATE_FLAG_BRECT       0x0004
#define STATE_FLAG_BALL        0x0005
#define STATE_MASK_BOUND       0x0007

#define STATE_FLAG_TNONE       0x0000
#define STATE_FLAG_TSPHERE     0x0008
#define STATE_FLAG_TALL        0x0010
#define STATE_MASK_TREE        0x0018

#define STATE_FLAG_AUTO_UPDATE 0x0020
#define STATE_FLAG_INVERSE     0x0040


#define QUAT_SPECIAL_CASE(inverseflag, t, delta, old, last, range)      \
{                                           \
   vector_interp_setup(old, last, 4, 1.0f/range, delta, &t); \
   inverseflag = (t < -1+CORRECT);          \
                                            \
   if (inverseflag) {                                           \
      delta[1] = -last[2];                                      \
      delta[2] =  last[1];                                      \
      delta[3] = -last[0];                                      \
      delta[0] =  last[3];                                      \
   }                                        \
                                            \
   else if (t < 0) {                                            \
      old[0] = -old[0];                     \
      old[1] = -old[1];                     \
      old[2] = -old[2];                     \
      old[3] = -old[3];                                         \
                                            \
      delta[0] = -delta[0];                                     \
      delta[2] = -delta[2];                                     \
      delta[3] = -delta[3];                                     \
   }                                        \
                                            \
   delta[4] = HALFPI/range;                                     \
}


class quark;
class atom;


/* ****************************************************************
**************************************************************** */
class keyframe {

   public:
      int    status;                              // flag for transition
      int    key;                                 // ptr to current knots
      float  frame;                               // local frame #
      float  framesperspline, iframesperspline;   // # of frames/spline
      int    count;                               // # of angles
      int    auto_flag;                           // flag for repetition or not
      vector4f current;                           // current value
      vector4f old;

      float  catchup, icatchup;                   // # of frames to

      vector4f *key_data;

      keyframe();
      virtual ~keyframe();
};


/* ****************************************************************
	list of children
**************************************************************** */
class linktype : public dbl_llist {

   public:
      quark *link;

      virtual ~linktype() {}
};


/* ****************************************************************
**************************************************************** */
class kinematic {

   protected:
      keyframe orient;                          // quaternion orientation
      keyframe pos;                             // kinematic position

      void update_localsp(float deltat);
      void calc_localsp(float *v, float deltat, int liveflag);

      int  update_preparesp(float deltat);
      void update_localquat(float deltat);

   public:

      // orientation data
      float    delta[5];
      float    range;
      int      inverseflag;

      // position data
      vector4f Q[4];                            // knot/spline matrix

      int  query_motion() { return orient.key_data || pos.key_data; }
      
      void read_spline(FILE *infile, float timefactor);
      void read_quaternion(FILE *infile, float timefactor);

      void calc_localquat(vector4f *mx, float deltat = 1.0f, int liveflag = 0);
      void calc_localspline(float *v, float deltat = 1.0f, int liveflag = 0);
};


/* ****************************************************************
**************************************************************** */
class state_type {

   protected:
      vector4f xmx_buffer[4], node_buffer[4], ixmx_buffer[4]; // xformation mx
   
   public:
      vector4f center;                     // location
      vector4f *xmx, *node, *ixmx;         // xformation mx

      unsigned int state_flags;

      float    bradius;                    // for bounding sphere
      
      union {
         vector4f bcenter;                 // for bounding sphere
         vector4f bdata[8];                // bounding box verticies
      };

      union {
         vector4f bnormal[6];              // old bounding box normals
         vector4f bplane;
      };

      float tree_radius;
      vector4f tree_center;
      vector3f tree_min, tree_max;
      
      state_type();
      
      void calc_bound_data();
      void calc_tree_bound_data();
      void copy_critical(state_type *src);
      int  read_bound(FILE *infile);
};


/* *************************************************************
   This is the basic class from which all classes are inherited
************************************************************* */
class quark : public superclass {

   protected:
      kinematic motion;
      frameclass *ob;
      vector4f previous_local_motion[4];    // last calc of calc_initmx()
      union { vector4uc frustum_fail_data; unsigned int frustum_fail_flag; };

      virtual void calc_initmx(vector4f *in, vector4f *out);

   public:
      // superclass
      int query_category() { return CLASS_ANIMATION; }
      int query_whatami() { return OBJECT_QUARK; }
      int query_whatwasi(int type);
      int parse(FILE *infile, char *token);
      void preprocess(void *data);

      // quark
      string_type name;                         // name of object

      vector4f initxform[4];                    // initial world xformation
      vector4f localmx[4];                      // xformation that doesnt affect children
      vector3f origin;                          // offset from parent

      float scale, iscale;                      // global scale/inverse scale
      unsigned int flags;

      state_type state, old_state;              // state data

      dbl_llist_manager edge;                   // ptr to children

      quark();
      virtual ~quark();

      superclass *get_render_object() { return ob; }
      quark *find(quark *parent, char *part, quark **found_parent = NULL);
      quark *find(quark *parent, int cat, quark *last, quark **found_parent = NULL);

      int   remove_link(quark *child);
      void  create_link(quark *child);

      quark *read_quark(FILE *infile, int parent_flag);
      int   write_data(FILE *outfile, quark *parent, int frame);

      virtual superclass *find_ob(quark *parent, int typecode, quark **source = NULL);
      virtual void render_object(engine *proc, quark *parent, vector4f *frustum, unsigned int frustum_flag);
      virtual void update(dbl_llist_manager *hiearchy_manager, quark *parent);
      virtual void setup();
      virtual void begin(dbl_llist_manager *hiearchy_manager, quark *parent, vector4f *mx);
      virtual void whereami(dbl_llist_manager *hiearchy_manager, quark *parent, vector4f *mx);
      virtual void new_action(FILE *infile, float  timefactor, char *buffer = NULL);
      virtual int  query_specific_data(unsigned int type, void *data);
      virtual int  set_specific_data(unsigned int type, void *data);
      virtual void apply_scale(float scale);
};


/* ****************************************************************
**************************************************************** */
class quark_loader : public loader {

   protected:
      char *alias_name;

   public:
      quark_loader() { object_name = alias_name = TOKEN_QUARK_STR; }

      virtual ~quark_loader() {}
      
      superclass *make_object() { return new quark; }
      superclass *parse(FILE *infile, char *token);
      char *query_alias() { return alias_name; }
};


/* ****************************************************************
**************************************************************** */
class basic_event : public dbl_llist {

   public:
      // event time
      union { int event; float timer; };

      virtual ~basic_event() {}

      virtual int query_whatami() = 0;
      virtual void evaluate(int frameno, atom *source, dbl_llist_manager *hiearchy_manager) = 0;
};


/* ****************************************************************
**************************************************************** */
class eventtype : public basic_event {

   public:

      int query_whatami() { return EVENT_STANDARD_ANITROLL; }
      void evaluate(int frameno, atom *source, dbl_llist_manager *hiearchy_manager);

      float timefactor;       // time multiplier
      string_type efilename;  // event file

      virtual ~eventtype() {}
};


/* ****************************************************************
   This is the animated object class
**************************************************************** */
class atom : public quark {

   friend class eventtype;

   protected:
      int scale_quark(float s, char *part);
      int move_quark(char *child, char *parent);
      int drop_quark(int frameno, char *child, dbl_llist_manager *hiearchy_manager, FILE *infile);
      int join_atom(char *part, char *parent, dbl_llist_manager *hiearchy_manager);
      int take_quark(char *dest, char *parent, char *part, dbl_llist_manager *hiearchy_manager);

   public:
      // superclass
      int query_whatami() { return OBJECT_ATOM; }
      int query_whatwasi(int type);
      int parse(FILE *infile, char *token);

      // quark
      void new_action(FILE *infile, float  timefactor, char *buffer);

      // atom
      dbl_llist_manager event_manager;

      virtual ~atom() {}

      void render_object(engine *proc, quark *parent, vector4f *frustum, unsigned int frustum_flag);
      int read_data(char *filename);

      virtual basic_event *read_event(FILE *infile, char *token, int frame_offset);
      virtual void new_action(int frameno, dbl_llist_manager *hiearchy_manager);
};


/* ****************************************************************
**************************************************************** */
class atom_loader : public quark_loader {

   public:
      atom_loader() { object_name = alias_name = TOKEN_ATOM_STR; }

      virtual ~atom_loader() {}
      superclass *make_object() { return new atom; }
};


/* ****************************************************************
**************************************************************** */
class atom_list_type : public dbl_llist {

   public:
      atom *htree;

      atom_list_type() { htree = NULL; }
      virtual ~atom_list_type() { if (htree) delete htree; }
};


typedef struct {
   quark *parent;        // the parent of the current animation node
   atom_list_type *tree; // hiearchy that contains the current animation node
} geneological_type;


int query_frustum_clip(float radius, float *center, vector4f *frustum, unsigned int *frustum_flag, unsigned char *fail_data);

extern int WF;

#endif

