

/* *************************************************************
   This file handles all the matrix/vector/math operations
************************************************************* */

#if !defined (__MATRIX_H)
#define __MATRIX_H

#include <math.h>

#define CORRECT         0.00001              // accuracy limit
#define CORRECT2        0.01                 // accuracy limit
#define CORRECT3        0.001                // accuracy limit
#define CORRECT4        0.0001               // accuracy limit
#define ICORRECT        0.99999
#define PCORRECT        1.00001

#define TWOPI2D    6.28

#ifdef PI
#undef PI
#endif

#ifdef TWOPI
#undef TWOPI
#endif

#ifdef HALFPI
#undef HALFPI
#endif

#define PI         3.14159265358979323846
#define TWOPI      6.28318530717958647692
#define HALFPI     1.57079632679489661923

#define LOG2E      1.4426950408889634074
#define LOG2(x)    (LOG2E * log(x))     // nat. log to log 2
#define rnd(x) ((x)*(rand() / 32767.0))
#define randomizer(x, y) lerp(rand() / 32767.0, x, y)
#define round(x) ((int)((x) + 0.5))
#define round0(x) (((x)<0) ? (-(int)(-(x))) : ((int)(x)))
#define not_zero(x) (fabs(x) > CORRECT)



#ifndef sgn
#define sgn(x)   ( ((x)<0) ? -1 : ((x) ? 1:0) )
#endif


#ifndef powf
#define powf(x, y) pow(x, y)
#endif


#ifdef WIN32
double erf(double arg);
#define random rand
#endif


#define s_curve(t) ( t*t * (3. - ((t) + (t))) )

#define lerp(t, a, b) ( (a) + (t) * ((b) - (a)) )
#define smoothstep(t, a, b)  ( (t) <= 0 ? (a) : ((t) >= 1 ? (b) : ((a) + s_curve(t)*((b)-(a))) ) )
#define step(a, x) ((x) >= (a))
#define clamp(x, a, b) ( (x) < (a) ? (a) : ((x) > (b) ? (b) : (x) ) )

#ifdef min
#undef min
#endif

#define min(a, b) ( (a) < (b) ? (a) : (b))

#ifdef max
#undef max
#endif

#define max(a, b) ( (a) > (b) ? (a) : (b))

#define getgamma(gamma) (1.0/(gamma))
#define gammacorrect(gamma, x) (powf((x), (gamma)))

// ********************************************************************
// init masks

#define MASK_SIN_COS    0x00000001
//#define MASK_SIN        0x00000002
#define MASK_ASIN_ACOS  0x00000004
//#define MASK_ASIN       0x00000008
#define MASK_0_255_0_1  0x00000010
#define MASK_SQRT       0x00000040
#define MASK_BYTE       0x00000080
#define MASK_ALL        0xffffffff

void init_lut(int fmask = MASK_ALL);
void dest_lut();

// trig tables

#define MAX_TRIG_SIZE           0x0ffff
#define MAX_TRIG_SIZE_PLUS      0x10000
#define MAX_SIN_COS_TBL_SIZE    0x14000
#define COS_START               0x04000

extern float *cos_table, *sin_table, *asin_table;
extern double trigfactor, atrigfactor;

#define COS(x) (cos_table[((int)((x)*trigfactor)) & MAX_TRIG_SIZE])
#define SIN(x) (sin_table[((int)((x)*trigfactor)) & MAX_TRIG_SIZE])
#define ACOS(x) (HALFPI - asin_table[((int)(((x)+1)*atrigfactor)) & MAX_TRIG_SIZE])
#define ASIN(x) (asin_table[((int)(((x)+1)*atrigfactor)) & MAX_TRIG_SIZE])

// ( 0 .. 255 ) - ( 0.0 .. 1.0 ) conversions

#define MAX_0_1_0_255           0x3ff
#define MAX_0_1_0_255_PLUS      0x400

extern float *table_0_255_0_1;

#define CONVERT_0_255_0_1(x) table_0_255_0_1[x]
#define CONVERT_0_1_0_255(x) ((int)((x)*255))

// sqrt function - x is an element of [0..4]

typedef union {
   float f;
   int bits;
   unsigned int ubits;
} conversion_type;


extern int *sqrt_mantissa_lut;

/* ***********************************************************
   NOTE: high bit is zero (sign bit)
*********************************************************** */
inline float SQRT(float x) {

   conversion_type b;

   b.f = x;

   //                      mantissa                                    |       exponent
//   if (b.bits & 0x00800000)
//      b.bits = sqrt_mantissa_lut[ (b.bits >> 8) & 0x7FFF]              | ((((b.bits >> 23) + 127) & 0xfffffffe) << 22);
//   else
//      b.bits = sqrt_mantissa_lut[((b.bits >> 8) & 0x7FFF) | 0x8000]    | ((((b.bits >> 23) + 127) & 0xfffffffe) << 22);

   b.bits = sqrt_mantissa_lut[ (b.bits >> 8) & 0xFFFF] | (((b.bits >> 1) + 0x1FC00000) & 0x0ff800000);

   return b.f;
}


extern int *isqrt_mantissa_lut;

/* ***********************************************************
   NOTE: high bit is zero (sign bit)
*********************************************************** */
inline float ISQRT(float x) {

   conversion_type b;
   conversion_type ex;

/*
   ex    = (b.bits >> 23) - 127;

   //               mantissa
   if (b.bits & 0x00800000)
      b.bits = isqrt_mantissa_lut[ (b.bits >> 8) & 0x7FFF];
   else
      b.bits = isqrt_mantissa_lut[((b.bits >> 8) & 0x7FFF) | 0x8000];
 
   //               exponent
   if (b.bits)
      b.bits |= ((253 - ex) & 0xfffffffe) << 22;
   else
      b.bits  = ((254 - ex) & 0xfffffffe) << 22;
*/

   ex.f = b.f = x;

   b.bits = isqrt_mantissa_lut[ (b.bits >> 8) & 0xFFFF];

   if (b.bits)
      b.bits |= (0x5F000000 - ((ex.bits >> 1) & 0xffc00000)) & 0xff800000;
   else
      b.bits  = (0x5F400000 - ((ex.bits >> 1) & 0xffc00000)) & 0xff800000;

   return b.f;
}


extern unsigned char    *pbyte_lut;
extern unsigned char    byte_lut[1024];
#define BYTE_LUT(x)     pbyte_lut[x]

// ********************************************************************

typedef double vector2d[2];
typedef vector2d *pvector2d;
typedef double vector3d[3];
typedef vector3d *pvector3d;
typedef double vector4d[4];
typedef vector4d *pvector4d;

typedef float vector2f[2];
typedef vector2f *pvector2f;
typedef float vector3f[3];
typedef vector3f *pvector3f;
typedef float vector4f[4];
typedef vector4f *pvector4f;

typedef unsigned char vector2uc[2];
typedef vector2uc *pvector2uc;
typedef unsigned char vector3uc[3];
typedef vector3uc *pvector3uc;
typedef unsigned char vector4uc[4];
typedef vector4uc *pvector4uc;

typedef int vector2i[2];
typedef int *pvector2i;
typedef int vector3i[3];
typedef int *pvector3i;
typedef int vector4i[4];
typedef int *pvector4i;

typedef int fixed_pt;
#define FIXED_PRECISION	16
#define FIXED_TO_INT(xxx) ((xxx)>>FIXED_PRECISION)
#define INT_TO_FIXED(xxx) ((xxx)<<FIXED_PRECISION)
#define FIXED_REMAINDER(xxx) ((xxx) & 0xffff)
#define FIXED_TO_FLOAT(xxx) ((xxx)*(1.525864936539E-5))
#define FLOAT_TO_FIXED(xxx) ((int)((xxx)*65536.0))

#define NORMALIZED_FIXED_PRECISION	24
#define NORMALIZED_FIXED_TO_INT(xxx) ((xxx)>>NORMALIZED_FIXED_PRECISION)
#define INT_TO_NORMALIZED_FIXED(xxx) ((xxx)<<FIXED_NORMALIZED_PRECISION)
#define NORMALIZED_FIXED_REMAINDER(xxx) ((xxx) & 0x00ffffff)
#define NORMALIZED_FIXED_TO_FLOAT(xxx) ((xxx)*(5.960464477539E-8))
#define FLOAT_TO_NORMALIZED_FIXED(xxx) ((int)((xxx)*16777216))

#define dotproduct2(x, y) (x[0]*y[0] + x[1]*y[1])
#define dotproduct3(x, y) (x[0]*y[0] + x[1]*y[1] + x[2]*y[2])
#define dotproduct4(x, y) (x[0]*y[0] + x[1]*y[1] + x[2]*y[2] + x[3]*y[3])

#define copyarray2(x, y) {x[0] = y[0]; x[1] = y[1];}
#define copyarray3(x, y) {copyarray2(x, y); x[2] = y[2];}
#define copyarray4(x, y) {copyarray3(x, y); x[3] = y[3];}

#define addarray2(x, y) {x[0] += y[0]; x[1] += y[1];}
#define addarray3(x, y) {x[0] += y[0]; x[1] += y[1]; x[2] += y[2];}
#define addarray4(x, y) {x[0] += y[0]; x[1] += y[1]; x[2] += y[2]; x[3] += y[3];}

#define addeqarray3(a, b, c) {a[0] = b[0] + c[0]; a[1] = b[1] + c[1]; a[2] = b[2] + c[2]; }

#define subarray2(x, y) {x[0] -= y[0]; x[1] -= y[1]; }
#define subarray3(x, y) {x[0] -= y[0]; x[1] -= y[1]; x[2] -= y[2];}
#define subarray4(x, y) {x[0] -= y[0]; x[1] -= y[1]; x[2] -= y[2]; x[3] -= y[3]; }

#define subeqarray2(a, b, c) { a[0] = b[0] - c[0]; a[1] = b[1] - c[1]; }
#define subeqarray3(a, b, c) { a[0] = b[0] - c[0]; a[1] = b[1] - c[1]; a[2] = b[2] - c[2]; }

#define multarray3(x, y) {x[0] *= y[0]; x[1] *= y[1]; x[2] *= y[2];}

#define divarray3(x, y) {x[0] /= y[0]; x[1] /= y[1]; x[2] /= y[2];}

#define saddarray2(x,y)  {x[0] += y; x[1] += y;}
#define saddarray3(x,y)  {x[0] += y; x[1] += y; x[2] += y;}

#define ssubarray3(x,y)  {x[0] -= y; x[1] -= y; x[2] -= y;}

#define smultarray3(x,y) {x[0] *= y; x[1] *= y; x[2] *= y;}
#define smultarray4(x,y) {x[0] *= y; x[1] *= y; x[2] *= y; x[3] *= y;}

#define sdivarray3(x,y)  {x[0] /= y; x[1] /= y; x[2] /= y;}
#define sdivarray4(x,y)  {x[0] /= y; x[1] /= y; x[2] /= y; x[3] /= y;}

#define nsame3(x, y) (x[0]!=y[0] || x[1]!=y[1] || x[2]!=y[2])

#define copymx4x4(x, y) { \
                      \
   x[0][0] = y[0][0]; \
   x[0][1] = y[0][1]; \
   x[0][2] = y[0][2]; \
   x[0][3] = y[0][3]; \
                      \
   x[1][0] = y[1][0]; \
   x[1][1] = y[1][1]; \
   x[1][2] = y[1][2]; \
   x[1][3] = y[1][3]; \
                      \
   x[2][0] = y[2][0]; \
   x[2][1] = y[2][1]; \
   x[2][2] = y[2][2]; \
   x[2][3] = y[2][3]; \
                      \
   x[3][0] = y[3][0]; \
   x[3][1] = y[3][1]; \
   x[3][2] = y[3][2]; \
   x[3][3] = y[3][3]; \
}


#define copymx4x4o(x, y) { \
                      \
   x[0][0] = y[0][0]; \
   x[0][1] = y[0][1]; \
   x[0][2] = y[0][2]; \
   x[0][3] = y[0][3]; \
                      \
   x[1][0] = y[1][0]; \
   x[1][1] = y[1][1]; \
   x[1][2] = y[1][2]; \
   x[1][3] = y[1][3]; \
                      \
   x[2][0] = y[2][0]; \
   x[2][1] = y[2][1]; \
   x[2][2] = y[2][2]; \
   x[2][3] = y[2][3]; \
                      \
   x[3][0] = x[3][1] = x[3][2] = 0; \
   x[3][3] = 1.0;     \
}


#define init_mx(x) { x[0][1] = x[0][2] = x[0][3] =    \
                     x[1][0] = x[1][2] = x[1][3] =    \
                     x[2][0] = x[2][1] = x[2][3] =    \
                     x[3][0] = x[3][1] = x[3][2] = 0; \
                     x[0][0] = x[1][1] = x[2][2] = x[3][3] = 1.0; }

#define print_v3(x) printf("%f %f %f\n", x[0], x[1], x[2])
#define print_v4(x) printf("%f %f %f %f\n", x[0], x[1], x[2], x[3])
#define print_mx(i, x) for (i=0; i<4; i++) printf("%f %f %f %f\n", x[i][0], x[i][1], x[i][2], x[i][3])


#define magnitude2(x) sqrt(dotproduct2(x, x))
#define magnitude3(x) sqrt(dotproduct3(x, x))
#define magnitude4(x) sqrt(dotproduct4(x, x))

#define fmagnitude3(x) SQRT(dotproduct3(x, x))

inline float distance3(float *x, float *y) {
        vector3f d;
        subeqarray3(d, x, y);
        return (float)magnitude3(d);
}

int    similar2(float *x, float *y);
int    similar3(float *x, float *y);

float  normalize2(float *x);
float  normalize3(float *x);
float  normalize4(float *x);
double normalize3d(double *x);

inline float fnormalize3(float *x) {

   register float mag, imag;

   mag = fmagnitude3(x);

   if (mag) {
      imag = 1.0f/mag;
      smultarray3(x, imag);
   }

   return mag;
}

void   swap4(float *mx1, float *mx2);

void   xproduct(float *a, float *x, float *y);
void   xproductd(double *a, double *x, double *y);
#define xproduct2(x, y) (x[0]*y[1] - x[1]*y[0])

void   transpose(vector4f *r, vector4f *s);
void   transposed(vector4d *r, vector4d *s);
void   transpose(vector4f *r);
void   transposed(vector4d *r);

void   matmatmult(vector4f *y, vector4f *x);
void   matmatmultd(vector4d *y, vector4d *x);
void   matmatmulto(vector4f *y, vector4f *x);
void   matmatmultod(vector4d *y, vector4d *x);
void   matmatmulto(vector4f *y, vector4f *x, vector4f *dst);
void   matmatmultod(vector4d *y, vector4d *x, vector4d *dst);
void   matmatmultv(vector4f *y, vector4f *x);
void   matmatmultvd(vector4d *y, vector4d *x);

void   matvecmult(vector4f *y, float *x);
void   matvecmulto(vector4f *y, float *x);
void   matvecmulto(vector4f *y, float *src, float *dst);
void   matvecmultv(vector4f *y, float *x);
void   matvecmultvd(vector4d *y, double *x);
void   matvecmultv(vector4f *y, float *src, float *dst);
void   matvecmultvd(vector4d *y, double *src, double *dst);

void   vecmatmult(float *x, vector4f *y);
void   vecmatmulto(float *x, vector4f *y);
void   vecmatmultv(float *x, vector4f *y);

int    inversemx(vector4f *a,vector4f *b);
int    inversemx2(vector4f *mx, vector4f *mxinverse);
void   inversemxrt(vector4f *mx, vector4f *mxinverse);
void   inversemxrtd(vector4d *mx, vector4d *mxinverse);

int vector_interp_setup(float *v1, float *v2, int count, float idelta, float *delta, float *theta);
void vector_interp_eval(float *v1, float *v2, int count, float *d, float *out);

void map_3D_2D(float *vertex0, float *vertex1, float *vertex2, float *uv0, float *uv1, float *uv2, float *mxu, float *mxv);
void calculate_normal(float *v1, float *v2, float *v3, float *n);
int query_calculate_normal(float *v1, float *v2, float *v3);

// ********************************************************************
//                                              orientation
// ********************************************************************


#define ANGLE2RAD  0.01745329251994329577
#define rad2deg(x) ((x)*57.295779513082323)    // times 180.0/PI
#define deg2rad(x) ((x)*ANGLE2RAD)      // times 180.0/PI

void calc_quat(float mx[][4], float current[]);
void calc_euler(float mx[][4], float current[]);
void euler2quat(float angle[], float quat[]);
void mat2quat(vector4f *mx, vector4f quat);
void quatquatmult(float y[], float x[]);

void decompose_xyz(vector4f mx[4], float *x, float *y, float *z);
void decompose_yxz(vector4f mx[4], float *x, float *y, float *z);
void decompose_zxy(vector4f mx[4], float *x, float *y, float *z);
void decompose_zxyd(vector4d mx[4], double *x, double *y, double *z);
void decompose_zyx(vector4f mx[4], float *x, float *y, float *z);

void decompose_xrypzy(vector4f mx[4], float *roll, float *pitch, float *yaw);
void decompose_yrxpzy(vector4f mx[4], float *roll, float *pitch, float *yaw);
void decompose_ypxrzy(vector4f mx[4], float *roll, float *pitch, float *yaw);
void decompose_yyxpzr(vector4f mx[4], float *roll, float *pitch, float *yaw);
void decompose_zyxryp(vector4f mx[4], float *roll, float *pitch, float *yaw);
void decompose_zyxpyr(vector4f mx[4], float *roll, float *pitch, float *yaw);
void decompose_zyxpyrd(vector4d mx[4], double *roll, double *pitch, double *yaw);
void decompose_zyypxr(vector4f mx[4], float *roll, float *pitch, float *yaw);
void decompose_zyypxrd(vector4d mx[4], double *roll, double *pitch, double *yaw);

void rotate_mx_x(vector4f mx[4], float x);
void rotate_mx_xd(vector4d mx[4], double x);
void arotate_mx_x(vector4f mx[4], float x);

void rotate_mx_y(vector4f mx[4], float y);
void rotate_mx_yd(vector4d mx[4], double y);
void arotate_mx_y(vector4f mx[4], float y);

void rotate_mx_z(vector4f mx[4], float z);
void rotate_mx_zd(vector4d mx[4], double z);
void arotate_mx_z(vector4f mx[4], float z);

void rotate_vector(float *start, float *axis, float rad, float *sol);
void calc_axis_rotate_mx(float *start, float *axis, float radian, vector4f *mx);

void scale_rotation(vector4f *mx, float scale);

// ********************************************************************
//                                                      splines
// ********************************************************************

#define M_CR    0
#define M_B     1

void spline(float s, float in[][4], float out[], int i, int stype);
void spline(float s, vector4f *in, float *out, int stype);
void splinet(float s, float Q[][4], float *out, int stype);

void spnormal(float s, float t, float in[][4], float out[], int ptr[], int stype);
void spnormal(float s, float t, vector4f **in, float *out, int stype);
void spnormal(float s, float t, float Q[][4][4], float *out, int stype);

void spnormal_base(vector4f **in, float out[], int stype);

void sppatch(float s, float t, float **in, float out[], int stype);
void sppatch(float s, float t, vector4f **in, float *out, int stype);
void sppatch(float s, float t, float Q[][4][4], float *out, int stype);

// ********************************************************************
//                                                      geometric
// ********************************************************************

#define cone_area(radius, length) (PI*(radius)*sqrt((radius)*(radius) + (length)*(length)))
float cylinder_area(float radius, float length);
#define sphere_area(radius) (4*PI*(radius)*(radius))

void make_box(vector4f *in, vector4f *out);

// ********************************************************************
//                                                      collision stuff
// ********************************************************************

#define CYL_TOP         0
#define CYL_BOTTOM      1
#define CYL_SIDE        3

#define CONE_TOP         0
#define CONE_BOTTOM      1
#define CONE_SIDE        3

// **************************** POINT *********************************


#define dist_pt2plane(pt, plane) (dotproduct3(pt, plane) + plane[3])

#define query_point_point_intersect(pt1, pt2) (fabs(pt1[0]-pt2[0])<CORRECT && fabs(pt1[1]-pt2[1])<CORRECT && fabs(pt1[2]-pt2[2])<CORRECT)
int query_point_sphere_intersect(float *pt, float *center, float r);
int query_point_cpoly_intersect(float *pt, float *normal, int count, vector4f *vertex);
int query_point_in_cpoly(float *pt, float *normal, int count, vector4f *vertex, int *index);
int query_point_poly_intersect(float *pt, float *normal, int count, vector4f *vertex);
#define query_point_plane_intersect(pt, n) (fabs(dist_pt2plane(pt, n)) <= CORRECT)
int query_point_rect_intersect(float *pt, vector4f *n);
int query_point_line_intersect(float *pt1, float *pt2, float *r2);
   
// **************************** SPHERE ********************************

int query_sphere_sphere_intersect(float *c1, float r1, float *c2, float r2);

int query_line_sphere_intersect(float radius, float *sphere, float *pt, float *vector);
int query_line_sphere_intersectn(float radius, float *sphere, float *pt, float *vector);
int query_lineseg_sphere_intersect(float radius, float *sphere, float *pt, float *vector);
int query_lineseg_sphere_intersect2(float radius, float *sphere, float *start, float *end);

#define query_sphere_plane_intersect(cc, rr, nn)   (fabs(dotproduct3(nn, cc) + nn[3]) <= (rr))

int query_sphere_rect_intersect(float *c, float r, float *rect_center, vector4f *n);

int line_sphere_intersect(float radius, float *sphere, float *pt, float *vector, float *t);
int line_sphere_intersectn(float radius, float *sphere, float *pt, float *vector, float *t);
int line_ellipse_intersect(float *scale, float *pt, float *vector, float *t);

// asdf undocumented ...
int line_ellipse_intersecti(float *iscale, float *pt, float *vector, float *t);

// **************************** PLANE ********************************

int query_poly_plane_intersect(vector4f *pt, int len, float *n);
int query_plane_plane_intersect(float *n1, float *n2);
int query_plane_rect_intersect(float *n, vector4f *p);

int lineseg_plane_intersect(float *plane, float *start, float *end, float *vector, float *intersect, float *t);
int line_plane_intersect(float *plane, float *pt, float *vector, float *intersect, float *t);
int line_poly_intersect(float *pt, float *vector, float *plane, vector4f *vertex, int count, float *intersect, float *t);
int line_cpoly_intersect(float *pt, float *vector, float *plane, vector4f *vertex, int count, float *intersect, float *t);

// **************************** OTHER ********************************

int query_rect_rect_intersect(vector4f *n1, vector4f *p1, float *c1, float r1,
                              vector4f *n2, vector4f *p2, float *c2, float r2);

int line_rect_intersect(float *pt, float *v, vector4f *p, vector4f *n, float *flag);
int line2d_line2d_intersect(float *v1, float *v2, float *pt);
int line3d_line3d_intersect(float *pt1, float *r1, float *pt2, float *r2, float *intersect);
int line_cylinder_intersect(vector4f pt, vector4f v, float length, float radius, float *t, int *side);
int line_cone_intersect(vector4f pt, vector4f v, float length, float *radius, float *t, int *side);

// ************************* REFLECTION ******************************

void reflect(float normal[], float vector[]);
int point_point_reflectn3vf(float *c1, float *c2, float *normal);
int plane_reflectn3vf(float *n, float *normal);
int point_rect_reflectn3vf(float *center, float *rect_center, vector4f *n, float *normal);
int plane_rect_reflectn3vf(float *n, float *rect_center, vector4f *surfacen, float *normal);

// ********************************************************************
//                                                      fractal

class fractal_params {

   public:
      float octaves, H, lacunarity;
      float *freq;

      fractal_params() { freq = 0; }
      fractal_params(float o, float h, float l) { octaves = o; H = h; lacunarity = l; freq = 0; }
      virtual ~fractal_params();
      void init();
};


#define bias(a, b) (pow((a), log(b) / (-0.693147181)))

void  init_noise();
float gain(float a, float b);
float noise1(float arg);
float noise2(float *vec);
float noise3(float *vec);
float noise(float *vec, int len);
float turbulence(float *v, float freq, float ifreq);

int bound_fBm(fractal_params *params, float offset, float *rmin, float *rmax);
float fBm(float *point, fractal_params *params);

int bound_fBm2(fractal_params *params, float offset, float *rmin, float *rmax);
float fBm2(float *point, fractal_params *params);

int bound_HeteroTerrain(fractal_params *params, float offset, float *rmin, float *rmax);
float Hetero_Terrain( float *point, fractal_params *params, float offset);

int bound_HybridMultifractal(fractal_params *params, float offset, float *rmin, float *rmax);
float HybridMultifractal(float *point, fractal_params *params, float offset, float gain);

int bound_RidgedMultifractal(fractal_params *params, float offset, float *rmin, float *rmax);
float RidgedMultifractal(float *point, fractal_params *params, float offset, float gain);

int bound_RidgedMultifractal2(fractal_params *params, float offset, float *rmin, float *rmax);
float RidgedMultifractal2(float *point, fractal_params *params, float offset, float gain);

/* ****************************************************************************************
**************************************************************************************** */

typedef union {
   double dblflt2int;
   int iflt2int[2];
} flt2int_type;

extern int flt2int[2];
extern int flt2fx[2];

#ifdef SGI

#define F2I(xxx, yyy) (xxx.iflt2int[0] = ((int)(yyy)))
//#define F2FX(xxx, yyy) (xxx.iflt2int[0] = ((int)((yyy)*65536))>>16)

#else

#define F2I(xxx, yyy) (xxx.dblflt2int = (yyy) + *(double *)&flt2int)
//#define F2FX(xxx, yyy) (xxx.dblflt2int = (yyy) + *(double *)&flt2fx)

#endif


#endif
