

#if !defined __IMAGE_H
#define __IMAGE_H


#include "datatype.h"


/* ************************************************************************************
                                    RGB
************************************************************************************ */

typedef enum storageenum {

   VERBATIM = 0,
   RLE      = 1
} storagetype;


typedef struct rgbheadstruct {

   unsigned int   xsize, ysize, zsize;
   int            bpc;
   storagetype    storage;
   unsigned int   dimension;
   unsigned int   pixmin, pixmax;
   char           name[80];
   unsigned int   colormap;
} rgbheadtype;


/* ************************************************************************************
************************************************************************************ */
class rgb : public image_coder {

   protected:
      // rgb
      rgbheadtype head;

      int scan_header();

   public:
      // basic_loader
      basic_loader *find_loader(sfile *data);

      // image_coder

      int write_data(char *fname, mapul *tob, int flip);
      int write_compress(char *fname, mapul *tob, int flip);

      int scan_data(mapul *tob, int flip);
      int skim_data(int *maxx, int *maxy, int flip);

      // rgb
      virtual ~rgb() {}
};


/* ************************************************************************************
                                    GIF
************************************************************************************ */

                // max gif codes + 1
#define GIF_MAX_CODES   4096

typedef struct gifheadstruct {

   unsigned short xsize, ysize;
   char signature[4];
   char version[4];
   char gctable_flag;
   int  gctable_size;
   char color_resolution;
   char sort_flag;
   char bc_index;
   char pixel_aspect_ratio;
} gifheadtype;


class gifimagedata {

   public:
      unsigned char image_separator;       // = 0x2c
      unsigned short left;
      unsigned short top;
      unsigned short width;
      unsigned short height;

      unsigned char  lctable_flag;
      unsigned char  interlace_flag;
      unsigned char  sort_flag;
      unsigned char  reserved;
      int            lctable_size;

      unsigned char  pass;
      unsigned char  slinc;

      unsigned char *lctable;
      unsigned char **plctable;

      unsigned char transparentflag;
      unsigned char transparentindex;
      unsigned char renderflag;

      gifimagedata() {
         plctable = NULL;
         lctable = NULL;
         transparentflag = renderflag = 0;
      }

      virtual ~gifimagedata() {
         if (plctable) {
            delete [] plctable;
            delete [] lctable;
         }

      }

};


// The following variables are used for seperating out codes
typedef struct gifdecodestruct {

   unsigned char bytes_left;                    // # bytes left in block
   unsigned char bits_left;                     // # bits left in current byte
   unsigned char buffer[257];                   // Current block
   unsigned char *nextcptr;                     // Pointer to next byte in block
   unsigned char cptr;                          // Current byte
   int           codesize;                      // The current code size
} gifdecodetype;


/* ************************************************************************************
************************************************************************************ */
class gif : public image_coder {

   protected:
      // gif
      unsigned char  stack[GIF_MAX_CODES];         // Stack for storing pixels
      unsigned char  suffix[GIF_MAX_CODES];        // Suffix table
      unsigned short prefix[GIF_MAX_CODES];        // Prefix linked list

      unsigned short clear;                        // Value for a clear code
      unsigned short ending;                       // Value for a ending code
      unsigned short newcodes;                     // First available code
      unsigned short top_slot;                     // Highest code for current size
      unsigned short slot;                         // Last read code 

      unsigned char **pgctable;
      unsigned char *gctable;

      unsigned char **color_table;

      gifimagedata  gimage;
      gifheadtype   head;

      int  scan_header();
      int  get_next_code(gifdecodetype *stuff);
      void out_line(unsigned char *buffer, mapul *image, int width, int scany);
      void decoder(mapul *image);

   public:
      // basic_loader
      basic_loader *find_loader(sfile *data);

      // image_coder
      int scan_data(mapul *tob, int flip);
      int skim_data(int *maxx, int *maxy, int flip);

      // gif
      gif() {
         pgctable = NULL;
         gctable = NULL;
      }

      virtual ~gif() {

         if (pgctable) {
            delete [] pgctable;
            delete [] gctable;
         }

      }

};


/* ************************************************************************************
************************************************************************************ */
class ilbm : public image_coder {

   protected:
      // ilbm
      int decomprle(unsigned char *dest, int size);

   public:
      // basic_loader
      basic_loader *find_loader(sfile *data);

      // image_coder
      int scan_data(mapul *tob, int flip);
      int skim_data(int *maxx, int *maxy, int flip);

      // ilbm
      virtual ~ilbm() {}
};


/* ************************************************************************************
************************************************************************************ */
class jpeg : public image_coder {

   public:
      // basic_loader
      basic_loader *find_loader(sfile *data);

      // image_coder
      int scan_data(mapul *tob, int flip);
      int skim_data(int *maxx, int *maxy, int flip);

      int write_data(char *fname, mapul *tob, int flip);

      // jpeg
      virtual ~jpeg() {}
};


/* ************************************************************************************
************************************************************************************ */
class tiff : public image_coder {

   public:
      // basic_loader
      basic_loader *find_loader(sfile *data);

      // image_coder
      int scan_data(mapul *tob, int flip);
      int skim_data(int *maxx, int *maxy, int flip);

      // tiff
      virtual ~tiff() {}
};


/* ************************************************************************************
************************************************************************************ */
class ppm : public image_coder {

   public:
      // basic_loader
      basic_loader *find_loader(sfile *data);

      // image_coder
      int scan_data(mapul *tob, int flip);
      int skim_data(int *maxx, int *maxy, int flip);

      int write_data(char *fname, mapul *tob, int flip);

      int extract(unsigned int type, void *data);

      // ppm
      virtual ~ppm() {}
};


/* ************************************************************************************
************************************************************************************ */

typedef struct {

   char type[2];		// "BM"
   int  size;			// file size
   int  reserved;		// zero;
   int  offset;			// offset from beginning of file to data
} bmp_file_header;

typedef struct {

   int header_size;		// size of header
   int xsize;			// x length
   int ysize;			// y width
   short plane;			// normally 1
   short bpp;			// bits per texel
   int compress_flag;
   int data_size;		// size of raw data
   int hppm;			// horizontal pixels per meter
   int vppm;			// verticle pixels per meter
   int palette_size;		// # of palette colors
   int palette_isize;		// # of important colors
} bmp_bitmap_header;


typedef struct {
   unsigned char b, g, r, a;
} bmp_quad;


/* ************************************************************************************
************************************************************************************ */
class bmp : public image_coder {

   protected:
      // bmp
      void write_verbatim256(int count, unsigned char *tbuffer, unsigned char *buffer, int *size);
   
   public:
      // basic_loader
      basic_loader *find_loader(sfile *data);

      // image_coder
      int scan_data256(mapuc *tob, unsigned int *palette, int flip);
      int scan_data(mapul *tob, int flip);
      int skim_data(int *maxx, int *maxy, int flip);
      int query_256();

      int write_data(char *fname, mapul *tob, int flip);
      int write_data256(char *fname, mapuc *tob, unsigned int *palette, int flip);
      int write_compress256(char *fname, mapuc *tob, unsigned int *palette, int flip);

      // bmp
      virtual ~bmp() {}
};


/* ************************************************************************************
************************************************************************************ */


#define MIPMAP_0   0
#define MIPMAP_8   8
#define MIPMAP_32 32

typedef struct {
   vector4uc id;
   unsigned int size;
   unsigned int channels;
   vector4uc format;
   unsigned int length;
   unsigned int width;
   unsigned int bpp;
   unsigned int palette_index;
   unsigned int image_index;
   unsigned int mipmap_index;
   unsigned int padd[6];
} rtf_file_header;


/* ************************************************************************************
************************************************************************************ */
class base_mipmap_manager : public dbl_llist_manager {

   public:
      virtual int query_whatami() { return MIPMAP_0; }
      virtual ~base_mipmap_manager() {}
};


/* ************************************************************************************
************************************************************************************ */
class mipmap8_manager : public base_mipmap_manager {

   public:
      virtual int query_whatami() { return MIPMAP_8; }
      virtual ~mipmap8_manager() {}
};


/* ************************************************************************************
************************************************************************************ */
class mipmap32_manager : public base_mipmap_manager {

   public:
      virtual int query_whatami() { return MIPMAP_32; }
      virtual ~mipmap32_manager() {}
};


/* ************************************************************************************
************************************************************************************ */
class mipmap_list_type : public dbl_llist {

   public:
      map *tob;

      virtual ~mipmap_list_type() {}
};


/* ************************************************************************************
************************************************************************************ */
class rtf : public image_coder {

   protected:
      // rtf
      base_mipmap_manager *mipmap_source;

      int  scan_header(rtf_file_header *header);
      void scan_palette(rtf_file_header *header, char *palette, int flip);

      void write_verbatim256(int count, char *tbuffer, sfile *buffer);
      void build_compress256(sfile *buffer, mapuc *tob);

      int query_mipmap_storage(int maxx, int maxy, int bpp);
      int write_mipmap8(FILE *outfile, sfile *buffer);
      int write_mipmap32(FILE *outfile);
   
   public:
      // basic_loader
      basic_loader *find_loader(sfile *data);

      // image_coder
      int scan_data(mapul *tob, int flip);
      int scan_data256(mapuc *tob, unsigned int *palette, int flip);
      int skim_data(int *maxx, int *maxy, int flip);
      int query_256();

      int write_data(char *fname, mapul *tob, int flip);
      int write_data256(char *fname, mapuc *tob, unsigned int *palette, int flip);
      int write_compress256(char *fname, mapuc *tob, unsigned int *palette, int flip);

      int perform_task(int type, void *data);
      int extract(unsigned int type, void *data, int flip);

      // rtf
      rtf() { mipmap_source = NULL; }
      virtual ~rtf() {}
};

#endif

