

#include <string.h>

#include "image.h"
#include "global.h"


#define BI_RGB  0
#define BI_RLE8 1
#define BI_RLE4 2


/* ***************************************************************
*************************************************************** */
basic_loader *rtf::find_loader(sfile *data) {

   vector4uc magic10 = { 'R', 'F', '1', '0' };
   vector4uc magic11 = { 'R', 'F', '1', '1' };
   union { vector4uc c; unsigned int i; };

   sinfile = data;
   sinfile->sseek(0);

   c[0] = sinfile->scan_uchar();
   c[1] = sinfile->scan_uchar();
   c[2] = sinfile->scan_uchar();
   c[3] = sinfile->scan_uchar();

   if (*(unsigned int *)magic10 == i || *(unsigned int *)magic11 == i)
      return this;
       
   sinfile = NULL;

   return next ? ((basic_loader *)next)->find_loader(data) : NULL;
}


/* ***************************************************************
*************************************************************** */
int rtf::query_256() {

   rtf_file_header file_header;

   sinfile->sseek(0);   

   if (!scan_header(&file_header) || file_header.bpp != 8)
      return 0;

   return file_header.palette_index != 0;
}


/* ***************************************************************
*************************************************************** */
int rtf::query_mipmap_storage(int maxx, int maxy, int bpp) {

   int i, j;
   int count;
   mipmap_list_type *ptr;

   if (!mipmap_source)
      return 0;

   for (i=1; i<maxy; i+=i);
   for (j=1; j<maxx; j+=j);

   if (!mipmap_source->head ||
       bpp != mipmap_source->query_whatami() ||
       (bpp != MIPMAP_8 && bpp != MIPMAP_32) ||
       i != maxy || j != maxx) {

      mipmap_source = NULL;
      return 0;
   }

   bpp = bpp == MIPMAP_8 ? 1 : 4;

   i = maxy;
   j = maxx;
   count = 0;
   for (ptr = (mipmap_list_type *)mipmap_source->head; ptr && (i != 1 || j != 1); ptr = (mipmap_list_type *)ptr->next) {
      i = (i == 1) ? 1 : (i >> 1);
      j = (j == 1) ? 1 : (j >> 1);

      if (i != ptr->tob->maxy || j != ptr->tob->maxx) {
         mipmap_source = NULL;
         return 0;
      }

      count += i*j*bpp;
   }

   if (i != 1 || j != 1 || ptr) {
      mipmap_source = NULL;
      return 0;
   }

   return count + 8;
}


/* ***************************************************************
*************************************************************** */
int rtf::scan_header(rtf_file_header *header) {

   vector4uc magic10 = { 'R', 'F', '1', '0' };
   vector4uc magic11 = { 'R', 'F', '1', '1' };

   // read file header
   header->id[0] = sinfile->scan_uchar();
   header->id[1] = sinfile->scan_uchar();
   header->id[2] = sinfile->scan_uchar();
   header->id[3] = sinfile->scan_uchar();

   if (*(unsigned int *)magic10 != *(unsigned int *)header->id &&
       *(unsigned int *)magic11 != *(unsigned int *)header->id)
      return 0;

   header->size = sinfile->scan_uint();
   header->channels = sinfile->scan_uint();
   header->format[0] = sinfile->scan_uchar();
   header->format[1] = sinfile->scan_uchar();
   header->format[2] = sinfile->scan_uchar();
   header->format[3] = sinfile->scan_uchar();
   header->length = sinfile->scan_uint();
   header->width = sinfile->scan_uint();
   header->bpp = sinfile->scan_uint();
   header->palette_index = sinfile->scan_uint();
   header->image_index = sinfile->scan_uint();

   if (*(unsigned int *)magic10 == *(unsigned int *)header->id) {
      header->mipmap_index = 0;
      sinfile->skip(header->size-36);
      return 1;
   }

   header->mipmap_index = sinfile->scan_uint();
   sinfile->skip(header->size-40);
   return 1;
}


/* ***************************************************************
*************************************************************** */
void rtf::scan_palette(rtf_file_header *header, char *palette, int flip) {

   int i;
   vector4uc rgba, color;
   
   sinfile->sseek(header->palette_index);

   sinfile->scan_uint(); // id
   sinfile->scan_uint(); // size
   
   for (i=0; i<4; i++)
      switch (header->format[i]) {
         case 'R':
            rgba[0] = i;
            break;

         case 'G':
            rgba[1] = i;
            break;

         case 'B':
            rgba[2] = i;
            break;

         default: // 'A'
            rgba[3] = i;
            break;
      }

   switch (flip) {

      case CBYTE_ORDER_ABGR:
         color[0] = 'A';
         color[1] = 'B';
         color[2] = 'G';
         color[3] = 'R';
         
         if (*(unsigned int *)header->format == *(unsigned int *)color) {
            sinfile->sread(palette, 1024);

            if (STANDARD_TEXTURE_PROCESSING || header->channels < 4)
               for (i=0; i<1024; i+=4)
                  palette[i] = (palette[i+1] || palette[i+2] || palette[i+3]) ? 255 : 0;

            return;
         }
	 
         for (i=0; i<1024; i+=4) {
            color[0] = sinfile->scan_uchar();
            color[1] = sinfile->scan_uchar();
            color[2] = sinfile->scan_uchar();
            color[3] = sinfile->scan_uchar();

            palette[i+1] = color[rgba[2]];
            palette[i+2] = color[rgba[1]];
            palette[i+3] = color[rgba[0]];

            if (STANDARD_TEXTURE_PROCESSING || header->channels < 4)
               palette[i] = (palette[i+1] | palette[i+2] | palette[i+3]) ? 255 : 0;
            else
               palette[i] = color[rgba[3]];
         }

         return;

      case CBYTE_ORDER_ARGB:
         color[0] = 'A';
         color[1] = 'R';
         color[2] = 'G';
         color[3] = 'B';
         
         if (*(unsigned int *)header->format == *(unsigned int *)color) {
            sinfile->sread(palette, 1024);

            if (STANDARD_TEXTURE_PROCESSING || header->channels < 4)
               for (i=0; i<1024; i+=4)
                  palette[i] = (palette[i+1] | palette[i+2] | palette[i+3]) ? 255 : 0;

            return;
         }
	 
         for (i=0; i<1024; i+=4) {
            color[0] = sinfile->scan_uchar();
            color[1] = sinfile->scan_uchar();
            color[2] = sinfile->scan_uchar();
            color[3] = sinfile->scan_uchar();

            palette[i+3] = color[rgba[2]];
            palette[i+2] = color[rgba[1]];
            palette[i+1] = color[rgba[0]];

            if (STANDARD_TEXTURE_PROCESSING || header->channels < 4)
               palette[i] = (palette[i+1] | palette[i+2] | palette[i+3]) ? 255 : 0;
            else
               palette[i] = color[rgba[3]];
         }

         return;

      case CBYTE_ORDER_BGRA:
         color[0] = 'B';
         color[1] = 'G';
         color[2] = 'R';
         color[3] = 'A';
         
         if (*(unsigned int *)header->format == *(unsigned int *)color) {
            sinfile->sread(palette, 1024);

            if (STANDARD_TEXTURE_PROCESSING || header->channels < 4)
               for (i=0; i<1024; i+=4)
                  palette[i+3] = (palette[i] | palette[i+1] | palette[i+2]) ? 255 : 0;

            return;
         }
	 
         for (i=0; i<1024; i+=4) {
            color[0] = sinfile->scan_uchar();
            color[1] = sinfile->scan_uchar();
            color[2] = sinfile->scan_uchar();
            color[3] = sinfile->scan_uchar();

            palette[i]   = color[rgba[2]];
            palette[i+1] = color[rgba[1]];
            palette[i+2] = color[rgba[0]];

            if (STANDARD_TEXTURE_PROCESSING || header->channels < 4)
               palette[i+3] = (palette[i] | palette[i+1] | palette[i+2]) ? 255 : 0;
            else
               palette[i+3] = color[rgba[3]];
         }

         return;

      default: // case CBYTE_ORDER_RGBA:
         color[0] = 'R';
         color[1] = 'G';
         color[2] = 'B';
         color[3] = 'A';
         
         if (*(unsigned int *)header->format == *(unsigned int *)color) {
            sinfile->sread(palette, 1024);

            if (STANDARD_TEXTURE_PROCESSING || header->channels < 4)
               for (i=0; i<1024; i+=4)
                  palette[i+3] = (palette[i] | palette[i+1] | palette[i+2]) ? 255 : 0;

            return;
         }
	 
         for (i=0; i<1024; i+=4) {
            color[0] = sinfile->scan_uchar();
            color[1] = sinfile->scan_uchar();
            color[2] = sinfile->scan_uchar();
            color[3] = sinfile->scan_uchar();

            palette[i]   = color[rgba[0]];
            palette[i+1] = color[rgba[1]];
            palette[i+2] = color[rgba[2]];

            if (STANDARD_TEXTURE_PROCESSING || header->channels < 4)
               palette[i+3] = (palette[i] | palette[i+1] | palette[i+2]) ? 255 : 0;
            else
               palette[i+3] = color[rgba[3]];
         }

         return;
   }
   
}


/* ***************************************************************
*************************************************************** */
int rtf::skim_data(int *maxx, int *maxy, int flip) {

   rtf_file_header file_header;

   sinfile->sseek(0);

   if (!scan_header(&file_header))
      return 0;
   
   *maxx = file_header.length;
   *maxy = file_header.width;
   
   return 1;
}


/* ***************************************************************
*************************************************************** */
int rtf::scan_data256(mapuc *tob, unsigned int *palette, int flip) {

   rtf_file_header file_header;
   unsigned char *data, *ptr, *end_file;
   int temp;
   
   sinfile->sseek(0);

   if (!scan_header(&file_header) || file_header.bpp != 8 || !file_header.palette_index)
      return 0;

   scan_palette(&file_header, (char *)palette, flip);

   sinfile->sseek(file_header.image_index);

   sinfile->scan_uint(); // id
   sinfile->scan_uint(); // size
   
   data = sinfile->ptr;

   tob->init_map(file_header.length, file_header.width);

   ptr = tob->data;
   end_file = ptr + file_header.length*file_header.width;

   while (ptr != end_file)
      // encode mode
      if (*data & 0x80) {
         temp = *data & 0x7f;
         memset(ptr, *(data+1), temp);
         data += 2;
         ptr += temp;
      }

      else {
         temp = *data;
         memcpy(ptr, data+1, temp);
         data += temp+1;
         ptr += temp;
      }

   return 1;
}


/* ***************************************************************
   assumes mipmap_source is valid
   Note: current mipmap code is uncompressed
*************************************************************** */
int rtf::write_mipmap8(FILE *outfile, sfile *buffer) {

   putuchar(outfile, 'M');
   putuchar(outfile, 'I');
   putuchar(outfile, 'P');
   putuchar(outfile, 'M');
   putuint(outfile, 8 + buffer->vsize);
   fwrite(buffer->data, 1, buffer->vsize, outfile);

   return 1;
}


/* ***************************************************************
*************************************************************** */
int rtf::write_data256(char *fname, mapuc *tob, unsigned int *palette, int flip) {

   return write_compress256(fname, tob, palette, flip);
}


/* ***************************************************************
*************************************************************** */
void rtf::write_verbatim256(int count, char *tbuffer, sfile *buffer) {

   buffer->sput_uchar(count);
   buffer->swrite(tbuffer, count);
}


/* ***************************************************************
*************************************************************** */
void rtf::build_compress256(sfile *buffer, mapuc *tob) {

   int i, j;
   unsigned char *ptr;
   int count, rle_count, key;
   char tbuffer[128];

   ptr = tob->data;

   for (i=0; i < (int)tob->maxy; i++) {
      j = 0;
      do {
         for (count = rle_count = 0; count != 127 && j != (int)tob->maxx && rle_count != 4; j++, count++, ptr++) {
            if (!count) {
               rle_count = 1;
               tbuffer[0] = *ptr;
               continue;
            }
	    
            tbuffer[count] = *ptr;
	    	    
            if (tbuffer[count-1] == *ptr)
               rle_count++;
            else
               rle_count = 1;
         }

         if (rle_count != 4) 
            write_verbatim256(count, tbuffer, buffer);
         else {
            if (count != 4)
               write_verbatim256(count-4, tbuffer, buffer);

            key = tbuffer[count-1];
            count = 4;

            while (*ptr == key && j != (int)tob->maxx && count != 127) {
               count++;
               ptr++;
               j++;
            }

            buffer->sput_uchar(0x80 | count);
            buffer->sput_uchar(key);
         }
	 
      } while (j < (int)tob->maxx);
    
   }

}


/* ***************************************************************
*************************************************************** */
int rtf::write_compress256(char *fname, mapuc *tob, unsigned int *palette, int flip) {

   FILE *outfile;
   sfile buffer;
   char tbuffer[32];
   int i;
   mipmap_list_type *ptr;

   outfile = fopen(fname,"wb");

   if (!outfile)
      return 0;

   putuchar(outfile, 'R');
   putuchar(outfile, 'F');
   putuchar(outfile, '1');
   putuchar(outfile, '1');
   putuint(outfile, 64);
   putuint(outfile, 4);

   switch (flip) {
      case CBYTE_ORDER_ABGR:
         putuchar(outfile, 'A');
         putuchar(outfile, 'B');
         putuchar(outfile, 'G');
         putuchar(outfile, 'R');
         break;

      case CBYTE_ORDER_ARGB:
         putuchar(outfile, 'A');
         putuchar(outfile, 'R');
         putuchar(outfile, 'G');
         putuchar(outfile, 'B');
         break;

      case CBYTE_ORDER_BGRA:
         putuchar(outfile, 'B');
         putuchar(outfile, 'G');
         putuchar(outfile, 'R');
         putuchar(outfile, 'A');
         break;

      default:  // CBYTE_ORDER_RGBA
         putuchar(outfile, 'R');
         putuchar(outfile, 'G');
         putuchar(outfile, 'B');
         putuchar(outfile, 'A');
         break;
   }
   
   putuint(outfile, tob->maxx);
   putuint(outfile, tob->maxy);
   putuint(outfile, 8);             // bpp
   putuint(outfile, 64);            // palette

   if (query_mipmap_storage(tob->maxx, tob->maxy, MIPMAP_8)) {
      buffer.init();

      for (ptr = (mipmap_list_type *)mipmap_source->head; ptr; ptr = (mipmap_list_type *)ptr->next)
         build_compress256(&buffer, (mapuc *)ptr->tob);

      i = 8 + buffer.vsize;
   }

   else
      i = 0;

   putuint(outfile, 1096 + i);      // image
   putuint(outfile, i ? 1096 : 0);  // mipmap

   memset(tbuffer, 0, 24);
   fwrite(tbuffer, 1, 24, outfile);

   putuchar(outfile, 'P');
   putuchar(outfile, 'A');
   putuchar(outfile, 'L');
   putuchar(outfile, 'T');
   putuint(outfile, 1032);

   fwrite(palette, 1, 1024, outfile);

   if (i)
      write_mipmap8(outfile, &buffer);

   buffer.init();
   build_compress256(&buffer, tob);
      
   putuchar(outfile, 'I');
   putuchar(outfile, 'M');
   putuchar(outfile, 'A');
   putuchar(outfile, 'G');
   putuint(outfile, buffer.vsize+8);
   fwrite(buffer.data, 1, buffer.vsize, outfile);

   fclose(outfile);
   return 1;
}


/* ***************************************************************
*************************************************************** */
int rtf::scan_data(mapul *tob, int flip) {

   rtf_file_header file_header;
   unsigned int palette[256];
   unsigned char *data;
   unsigned int *ptr, *etr, *end_file;
   unsigned int color, temp;
   union { vector4uc ucolor; unsigned int icolor; };
   vector4uc rgba;
   int i, j;
   
   sinfile->sseek(0);

   if (!scan_header(&file_header))
      return 0;

   // read in a paletted texture
   if (file_header.bpp == 8) {
      if (!file_header.palette_index)
         return 0;
	 
      scan_palette(&file_header, (char *)palette, flip);
      sinfile->sseek(file_header.image_index);
      
      sinfile->scan_uint(); // id
      sinfile->scan_uint(); // size
   
      data = sinfile->ptr;

      tob->init_map(file_header.length, file_header.width);

      ptr = tob->data;
      end_file = ptr + tob->maxx*tob->maxy;

      while (ptr != end_file)
	 
         // encode mode
         if (*data & 0x80) {
	    temp = *data & 0x7f;
	    color = palette[*(data+1)];
	    data += 2;
	    
            for (etr = ptr + temp; ptr<etr; ptr++)
	       *ptr = color;
         }

         else {
            temp = *data;
            data++;
	    
            for (etr = ptr + temp; ptr<etr; ptr++, data++)
	       *ptr = palette[*data];
         }

      return 1;
   }

   // read in a raw texture
   for (i=0; i<4; i++)
      switch (file_header.format[i]) {
         case 'R':
	    rgba[0] = i;
	    break;

	 case 'G':
	    rgba[1] = i;
	    break;

	 case 'B':
	    rgba[2] = i;
	    break;

         default: // 'A'
	    rgba[3] = i;
	    break;
      }

   sinfile->sseek(file_header.image_index);

   sinfile->scan_uint(); // id
   sinfile->scan_uint(); // size
   
   tob->init_map(file_header.length, file_header.width);
   j = tob->maxx*tob->maxy<<2;

   data = (unsigned char *)tob->data;

   switch (flip) {

      case CBYTE_ORDER_ABGR:
         ucolor[0] = 'A';
         ucolor[1] = 'B';
         ucolor[2] = 'G';
         ucolor[3] = 'R';
         
         if (*(unsigned int *)file_header.format == icolor) {
            sinfile->sread((char *)data, j);

            if (STANDARD_TEXTURE_PROCESSING || file_header.channels < 4)
               for (i=0; i<j; i+=4)
                  data[i] = (data[i+1] | data[i+2] | data[i+3]) ? 255 : 0;

            return 1;
         }

         for (i=0; i<j; i+=4) {
            ucolor[0] = sinfile->scan_uchar();
            ucolor[1] = sinfile->scan_uchar();
            ucolor[2] = sinfile->scan_uchar();
            ucolor[3] = sinfile->scan_uchar();

            data[i+1] = ucolor[rgba[2]];
            data[i+2] = ucolor[rgba[1]];
            data[i+3] = ucolor[rgba[0]];

            if (STANDARD_TEXTURE_PROCESSING || file_header.channels < 4)
               data[i] = (data[i+1] | data[i+2] | data[i+3]) ? 255 : 0;
            else
               data[i] = ucolor[rgba[3]];
         }

         return 1;

      case CBYTE_ORDER_ARGB:
         ucolor[0] = 'A';
         ucolor[1] = 'R';
         ucolor[2] = 'G';
         ucolor[3] = 'B';
         
         if (*(unsigned int *)file_header.format == icolor) {
            sinfile->sread((char *)data, j);

            if (STANDARD_TEXTURE_PROCESSING || file_header.channels < 4)
               for (i=0; i<j; i+=4)
                  data[i] = (data[i+1] | data[i+2] | data[i+3]) ? 255 : 0;

            return 1;
         }

         for (i=0; i<j; i+=4) {
            ucolor[0] = sinfile->scan_uchar();
            ucolor[1] = sinfile->scan_uchar();
            ucolor[2] = sinfile->scan_uchar();
            ucolor[3] = sinfile->scan_uchar();

            data[i+1] = ucolor[rgba[0]];
            data[i+2] = ucolor[rgba[1]];
            data[i+3] = ucolor[rgba[2]];

            if (STANDARD_TEXTURE_PROCESSING || file_header.channels < 4)
               data[i] = (data[i+1] | data[i+2] | data[i+3]) ? 255 : 0;
            else
               data[i] = ucolor[rgba[3]];
         }

         return 1;

      case CBYTE_ORDER_BGRA:
         ucolor[0] = 'B';
         ucolor[1] = 'G';
         ucolor[2] = 'R';
         ucolor[3] = 'A';
         
         if (*(unsigned int *)file_header.format == icolor) {
            sinfile->sread((char *)data, j);

            if (STANDARD_TEXTURE_PROCESSING || file_header.channels < 4)
               for (i=0; i<j; i+=4)
                  data[i+3] = (data[i] | data[i+1] | data[i+2]) ? 255 : 0;

            return 1;
         }
	 
         for (i=0; i<j; i+=4) {
            ucolor[0] = sinfile->scan_uchar();
            ucolor[1] = sinfile->scan_uchar();
            ucolor[2] = sinfile->scan_uchar();
            ucolor[3] = sinfile->scan_uchar();

            data[i]   = ucolor[rgba[2]];
            data[i+1] = ucolor[rgba[1]];
            data[i+2] = ucolor[rgba[0]];

            if (STANDARD_TEXTURE_PROCESSING || file_header.channels < 4)
               data[i+3] = (data[i] | data[i+1] | data[i+2]) ? 255 : 0;
            else
                data[i+3] = ucolor[rgba[3]];
         }

         return 1;

      default: // case CBYTE_ORDER_RGBA:
         ucolor[0] = 'R';
         ucolor[1] = 'G';
         ucolor[2] = 'B';
         ucolor[3] = 'A';
         
         if (*(unsigned int *)file_header.format == icolor) {
            sinfile->sread((char *)data, j);

            if (STANDARD_TEXTURE_PROCESSING || file_header.channels < 4)
               for (i=0; i<j; i+=4)
                  data[i+3] = (data[i] | data[i+1] | data[i+2]) ? 255 : 0;

            return 1;
         }
	 
         for (i=0; i<j; i+=4) {
            ucolor[0] = sinfile->scan_uchar();
            ucolor[1] = sinfile->scan_uchar();
            ucolor[2] = sinfile->scan_uchar();
            ucolor[3] = sinfile->scan_uchar();

            data[i]   = ucolor[rgba[0]];
            data[i+1] = ucolor[rgba[1]];
            data[i+2] = ucolor[rgba[2]];

            if (STANDARD_TEXTURE_PROCESSING || file_header.channels < 4)
               data[i+3] = (data[i] | data[i+1] | data[i+2]) ? 255 : 0;
            else
               data[i+3] = ucolor[rgba[3]];
         }

         return 1;
   }
   
   return 1;
}


/* ***************************************************************
   assumes mipmap_source is valid
*************************************************************** */
int rtf::write_mipmap32(FILE *outfile) {

   mipmap_list_type *ptr;
   int count;

   count = 8;
   for (ptr = (mipmap_list_type *)mipmap_source->head; ptr; ptr = (mipmap_list_type *)ptr->next)
      count += ptr->tob->maxx * ptr->tob->maxy;

   putuchar(outfile, 'M');
   putuchar(outfile, 'I');
   putuchar(outfile, 'P');
   putuchar(outfile, 'M');
   putuint(outfile, count);

   for (ptr = (mipmap_list_type *)mipmap_source->head; ptr; ptr = (mipmap_list_type *)ptr->next)
      fwrite(((mapul *)ptr->tob)->data, 1, ptr->tob->maxx*ptr->tob->maxy, outfile);

   return 1;
}


/* ***************************************************************
*************************************************************** */
int rtf::write_data(char *fname, mapul *tob, int flip) {

   FILE *outfile;
   char tbuffer[32];
   int i;
   
   outfile = fopen(fname,"wb");

   if (!outfile)
      return 0;

   putuchar(outfile, 'R');
   putuchar(outfile, 'F');
   putuchar(outfile, '1');
   putuchar(outfile, '1');
   putuint(outfile, 64);
   putuint(outfile, 4);

   switch (flip) {
      case CBYTE_ORDER_ABGR:
         putuchar(outfile, 'A');
         putuchar(outfile, 'B');
         putuchar(outfile, 'G');
         putuchar(outfile, 'R');
         break;

      case CBYTE_ORDER_ARGB:
         putuchar(outfile, 'A');
         putuchar(outfile, 'R');
         putuchar(outfile, 'G');
         putuchar(outfile, 'B');
         break;

      case CBYTE_ORDER_BGRA:
         putuchar(outfile, 'B');
         putuchar(outfile, 'G');
         putuchar(outfile, 'R');
         putuchar(outfile, 'A');
         break;

      default:  // CBYTE_ORDER_RGBA
         putuchar(outfile, 'R');
         putuchar(outfile, 'G');
         putuchar(outfile, 'B');
         putuchar(outfile, 'A');
         break;
   }
   
   putuint(outfile, tob->maxx);
   putuint(outfile, tob->maxy);
   putuint(outfile, 32);         // bpp
   putuint(outfile, 0);          // palette

   i = query_mipmap_storage(tob->maxx, tob->maxy, MIPMAP_32);
   putuint(outfile, 64+i);       // image
   putuint(outfile, i ? 64 : 0); // mipmap

   memset(tbuffer, 0, 24);
   fwrite(tbuffer, 1, 24, outfile);

   if (i)
      write_mipmap32(outfile);

   i = tob->maxx*tob->maxy<<2;
   putuchar(outfile, 'I');
   putuchar(outfile, 'M');
   putuchar(outfile, 'A');
   putuchar(outfile, 'G');
   putuint(outfile, i);
   
   fwrite(tob->data, 1, i, outfile);

   fclose(outfile);
   return 1;
}


/* ***************************************************************
*************************************************************** */
int rtf::extract(unsigned int type, void *data, int flip) {

   rtf_file_header file_header;
   int i, j, k, count, size;
   vector4uc magic = { 'M', 'I', 'P', 'M' };
   union { vector4uc uc; unsigned int ui; };
   mipmap_list_type *ptr;
   dbl_llist_manager *manager;

   switch (type) {

      case IMAGE_GET_MIPMAP32:
         sinfile->sseek(0);
         if (!scan_header(&file_header) || !file_header.mipmap_index || file_header.bpp != 32)
            return 0;

         sinfile->sseek(file_header.mipmap_index);

         uc[0] = sinfile->scan_uchar();
         uc[1] = sinfile->scan_uchar();
         uc[2] = sinfile->scan_uchar();
         uc[3] = sinfile->scan_uchar();

         if (ui != *(unsigned int *)magic)
            return 0;

         size = sinfile->scan_uint() - 8;

         i = file_header.length > file_header.width ? file_header.length : file_header.width;
         for (count=1, j=1; j<i; j+=j, count++);
         
         manager = (dbl_llist_manager *)data;
         manager->dest(); 

         // "0" entry is for main image
         for (k=1, i = file_header.width, j = file_header.length; k<count; k++) {
            i  = i > 1 ? (i>>1) : 1;
            j  = j > 1 ? (j>>1) : 1;

            manager->append(ptr = new mipmap_list_type, NULL);
            ptr->tob = new mapul;
            ((mapul *)ptr->tob)->init_map(j, i);

            sinfile->sread((char *)((mapul *)ptr->tob)->data, i*j*4);
         }

         return count;

      case IMAGE_GET_MIPMAP8:
         sinfile->sseek(0);
         if (!scan_header(&file_header) || !file_header.mipmap_index || file_header.bpp != 8)
            return 0;

         sinfile->sseek(file_header.mipmap_index);

         uc[0] = sinfile->scan_uchar();
         uc[1] = sinfile->scan_uchar();
         uc[2] = sinfile->scan_uchar();
         uc[3] = sinfile->scan_uchar();

         if (ui != *(unsigned int *)magic)
            return 0;

         size = sinfile->scan_uint() - 8;

         i = file_header.length > file_header.width ? file_header.length : file_header.width;
         for (count=1, j=1; j<i; j+=j, count++);

         manager = (dbl_llist_manager *)data;
         manager->dest(); 

         // "0" entry is for main image
         for (k=1, i = file_header.width, j = file_header.length; k<count; k++) {
            i  = i > 1 ? (i>>1) : 1;
            j  = j > 1 ? (j>>1) : 1;

            manager->append(ptr = new mipmap_list_type, NULL);
            ptr->tob = new mapuc;
            ((mapuc *)ptr->tob)->init_map(j, i);

            sinfile->sread((char *)((mapuc *)ptr->tob)->data, i*j);
         }

         return count;

      default:
         break;
   }

   return image_coder::extract(type, data, flip);
}


/* ***************************************************************
*************************************************************** */
int rtf::perform_task(int type, void *data) {

   switch (type) {

      case IMAGE_SET_MIPMAP8:
      case IMAGE_SET_MIPMAP32:
         mipmap_source = (base_mipmap_manager *)data;
         return 1;

      default:
         break;
   }

   return 0;
}

