

#include "image.h"
#include "global.h"


#define MAGIC   474
#define MAXCOUNT 0x80

#define   NORMAL    0
#define   DITHERED  1
#define   SCREEN    2
#define   COLORMAP  3


/* ***************************************************************
*************************************************************** */
basic_loader *rgb::find_loader(sfile *data) {

   sinfile = data;
   sinfile->sseek(0);

   if (sinfile->scan_ushort() == MAGIC)
      return this;

   sinfile = NULL;

   return next ? ((basic_loader *)next)->find_loader(data) : NULL;
}


/* ***********************************************************************
        This procedure will write out an rgb file in Verbatim format
*********************************************************************** */
int rgb::write_data(char *fname, mapul *tob, int flip) {

   FILE *outfile;
   unsigned char *buffer;
   int i, x, y;
   unsigned char *base;

   if (tob->data == (unsigned int *)NULL)
      return 0;

   if (!(outfile = fopen(fname,"wb")))
      return 0;
 
   head.dimension = 3;
   head.xsize = (unsigned short)tob->maxx;
   head.ysize = (unsigned short)tob->maxy;
   head.zsize = 4;

   buffer = new unsigned char[head.xsize > 404 ? head.xsize : 404];
   buffer[0] = 0;

   putushort(outfile, MAGIC);    // MAGIC
   putuchar(outfile, VERBATIM);  // STORAGE is VERBATIM
   putuchar(outfile, 1);         // BPC is 1 (# of bytes per pixel component/channel (1 or 2))
   putushort(outfile, head.dimension);// DIMENSION (1 - single row BW, 2 - x*y BW, 3 - x*y color)
   putushort(outfile, head.xsize);    // width on xaxis
   putushort(outfile, head.ysize);    // length on yaxis
   putushort(outfile, head.zsize);    // ZSIZE/# of channels ( 1 - BW, 3 - RGB, 4 - RGBA)
   putuint(outfile, 0);         // PIXMIN is 0
   putuint(outfile, 255);       // PIXMAX is 255

   buffer[0] = 0;

   fwrite(buffer,4,1,outfile);   // dummy 4 bytes

   fwrite(buffer,80,1,outfile);  // IMAGENAME
   putuint(outfile,NORMAL);     // COLORMAP is 0 - NORMAL

   fwrite(buffer,404,1,outfile);   // dummy 404 bytes

   for (i=0; i < (int)head.zsize; i++) {

      switch (flip) {
         case CBYTE_ORDER_ABGR:
            base = (unsigned char *)tob->data + 3 - i;
            break;

         case CBYTE_ORDER_ARGB:
            base = (unsigned char *)tob->data + 1 + i;
            break;

         case CBYTE_ORDER_BGRA:
            base = (unsigned char *)tob->data + (i==3 ? 3 : 2-i);
            break;

         default:  // CBYTE_ORDER_RGBA
            base = (unsigned char *)tob->data + i;
            break;
      }

      for (y=0; y < (int)head.ysize; y++) {
         for (x=0; x < (int)head.xsize; x++, base+=4)
            buffer[x] = *base;

         fwrite(buffer,head.xsize, 1, outfile);
      }

   }

   fclose(outfile);
   delete [] buffer;
   return 1;
}


/* ***********************************************************************
        This procedure will write out an rgb file in RLE format
*********************************************************************** */
int rgb::write_compress(char *fname, mapul *tob, int flip) {

   FILE *outfile;
   unsigned int i, j, k;
   unsigned char *buffer;
   unsigned char count;
   unsigned char ctype;
   unsigned int  qtr;
   unsigned int  *starttab, *lengthtab;
   unsigned int  tablem;
   unsigned char *ptr;

   if (tob->data == (unsigned int *)NULL)
      return 0;

   if (!(outfile = fopen(fname,"wb")))
      return 0;

   head.dimension = 3;
   head.xsize = (unsigned short)tob->maxx;
   head.ysize = (unsigned short)tob->maxy;
   head.zsize = 4;

   putushort(outfile, MAGIC);    // MAGIC
   putuchar(outfile, RLE);       // STORAGE is RLE
   putuchar(outfile, 1);         // BPC is 1 (# of bytes per pixel component/channel (1 or 2))
   putushort(outfile, head.dimension);// DIMENSION (1 - single row BW, 2 - x*y BW, 3 - x*y color)
   putushort(outfile, head.xsize);    // width on xaxis
   putushort(outfile, head.ysize);    // length on yaxis
   putushort(outfile, head.zsize);    // ZSIZE/# of channels ( 1 - BW, 3 - RGB, 4 - RGBA)
   putuint(outfile, 0);         // PIXMIN is 0
   putuint(outfile, 255);       // PIXMAX is 255

   j = head.xsize+head.xsize;
   tablem = head.ysize*head.zsize;
   i = tablem<<3;
   if (j < i)
      j = i;
   if (j < 404)
      j = 404;

   buffer = new unsigned char[j];
   buffer[0] = 0;

   fwrite(buffer,4,1,outfile);   // dummy 4 bytes

   fwrite(buffer,80,1,outfile); // IMAGENAME
   putuint(outfile,NORMAL);    // COLORMAP is 0 - NORMAL

   fwrite(buffer,404,1,outfile);// dummy 404 bytes
   starttab  = new unsigned int[tablem];
   lengthtab = new unsigned int[tablem];
   starttab[0] = 512+i;
   lengthtab[0] = 0;

   fwrite(buffer,(unsigned int)i,1,outfile);  // blank tables

   qtr = 0;
   for (i=0; i<head.zsize; i++) {

      switch (flip) {
         case CBYTE_ORDER_ABGR:
            ptr = (unsigned char *)tob->data + 3 - i;
            break;

         case CBYTE_ORDER_ARGB:
            ptr = (unsigned char *)tob->data + 1 + i;
            break;

         case CBYTE_ORDER_BGRA:
            ptr = (unsigned char *)tob->data + ( (i==3) ? 3 : 2-i);
            break;

         default:  // CBYTE_ORDER_RGBA
            ptr = (unsigned char *)tob->data + i;
            break;
      }

      for (j=0; j<head.ysize; j++) {
         count = 1;
         buffer[0] = *ptr;
         ptr+=4;

         for (k=1; k<head.xsize; k++, ptr+=4) {
            if (count == 1) {
               if (buffer[0] == *ptr)
                  ctype = 0;
               else {
                  ctype = 1;
                  buffer[1] = *ptr;
               }

               count = 2;
            }

            else
               if (ctype)
                  if (buffer[count-1] == *ptr) {
                     count--;
                     lengthtab[qtr] += count+1;
                     ctype = count | MAXCOUNT;
                     fwrite(&ctype, 1, 1, outfile);
                     fwrite(buffer, count, 1, outfile);
                     buffer[0] = *ptr;
                     count = 2;
                     ctype = 0;
                  }

                  else {
                     if (count < MAXCOUNT-1) {
                        buffer[count] = *ptr;
                        count++;
                     }

                     else {
                        lengthtab[qtr] += count+1;
                        ctype = count | MAXCOUNT;
                        fwrite(&ctype, 1, 1, outfile);
                        fwrite(buffer, count, 1, outfile);
                        buffer[0] = *ptr;
                        count = 1;
                     }

                  }

               else
                  if (buffer[0] == *ptr)
                     if (count < MAXCOUNT-1)
                        count++;
                     else {
                        lengthtab[qtr] += 2;
                        fwrite(&count, 1, 1, outfile);
                        fwrite(buffer, 1, 1, outfile);
                        count = 1;
                     }

                  else {
                     lengthtab[qtr] += 2;
                     fwrite(&count, 1, 1, outfile);
                     fwrite(buffer, 1, 1, outfile);
                     buffer[0] = *ptr;
                     count = 1;
                  }

         }

         if (ctype) {
            lengthtab[qtr] += count+1;
            ctype = MAXCOUNT | count;
            fwrite(&ctype, 1, 1, outfile);
            fwrite(buffer, count, 1, outfile);
         }

         else {
            lengthtab[qtr] += 2;
            fwrite(&count, 1, 1, outfile);
            fwrite(buffer, 1, 1, outfile);
         }

         count = 0;
         fwrite(&count, 1, 1, outfile);
         lengthtab[qtr]++;

         if (qtr+1 < tablem) {
            starttab[qtr+1] = starttab[qtr] + lengthtab[qtr];
            qtr++;
            lengthtab[qtr] = 0;
         }

      }

   }

   fseek(outfile,512,SEEK_SET);

   for (i=0; i<tablem; i++)
      putuint(outfile, starttab[i]);

   for (i=0; i<tablem; i++)
      putuint(outfile, lengthtab[i]);

   delete [] buffer;
   delete [] starttab;
   delete [] lengthtab;

   fclose(outfile);

   return 1;
}


/* ***********************************************************************
*********************************************************************** */
int rgb::scan_header() {

   if (sinfile->scan_ushort() != MAGIC)
      return 0;

   head.storage = (storagetype)sinfile->scan_uchar();
   head.bpc = sinfile->scan_uchar();
   head.dimension = sinfile->scan_ushort();
   head.xsize = sinfile->scan_ushort();
   head.ysize = sinfile->scan_ushort();
   head.zsize = sinfile->scan_ushort();

   sinfile->skip(12);
   sinfile->sread((char *)head.name, 80);

   head.colormap = sinfile->scan_uint();
   return 1;
}


/* ***********************************************************************
*********************************************************************** */
int rgb::scan_data(mapul *tob, int flip) {

   unsigned int *starttab, *lengthtab, *pstarttab, *plengthtab;
   int i, j;
   int ptr;
   unsigned char  *charbuffer;
   unsigned short *shortbuffer;
   unsigned char  *base, *buffer, *map, *limit;
   unsigned char  b;
   unsigned char  ccount;

   sinfile->sseek(0);

   if (!scan_header())
      return 0;

   sinfile->sseek(512);

   tob->init_map(head.xsize, head.ysize);
   buffer = new unsigned char[head.xsize];

   if (head.storage == VERBATIM)                        // read in verbatim
      for (ptr=0; ptr < (int)head.zsize; ptr++) {

         switch (flip) {
            case CBYTE_ORDER_ABGR:
               base = (unsigned char *)tob->data + 3 - ptr;
               break;

            case CBYTE_ORDER_ARGB:
               base = (unsigned char *)tob->data + 1 + ptr;
               break;

            case CBYTE_ORDER_BGRA:
               base = (unsigned char *)tob->data + (ptr==3 ? 3 : 2-ptr);
               break;

            default:  // CBYTE_ORDER_RGBA
               base = (unsigned char *)tob->data + ptr;
               break;
         }

         for (i=0; i < (int)head.ysize; i++) {
            sinfile->sread((char *)buffer, head.xsize);
            for (j=0; j < (int)head.xsize; j++, base+=4)
               *base = buffer[j];
         }

      }

   else {                                               // read in rle
      j = head.ysize*head.zsize;

      pstarttab = starttab  = new unsigned int[j];
      plengthtab = lengthtab = new unsigned int[j];

      for (i=0; i<j; i++)
         starttab[i] = sinfile->scan_uint();

      for (i=0; i<j; i++)
         lengthtab[i] = sinfile->scan_uint();

      if (head.bpc == 1)
         charbuffer = new unsigned char[MAXCOUNT];
      else
         shortbuffer = new unsigned short[MAXCOUNT];

      ptr = 0;
      for (i=0; i < (int)head.zsize; i++) {

         switch (flip) {
            case CBYTE_ORDER_ABGR:
               base = (unsigned char *)tob->data + 3 - i;
               break;

            case CBYTE_ORDER_ARGB:
               base = (unsigned char *)tob->data + 1 + i;
               break;

            case CBYTE_ORDER_BGRA:
               base = (unsigned char *)tob->data + (i==3 ? 3 : 2 - i);
               break;

            default:  // CBYTE_ORDER_RGBA
               base = (unsigned char *)tob->data + i;
               break;
         }

         for (j=0; j < (int)head.ysize; j++, pstarttab++, plengthtab++) {

            sinfile->sseek(*pstarttab);

            if (head.bpc == 1)
               while (*plengthtab) {
                  ccount = sinfile->scan_uchar();
                  *plengthtab -= 1;

                  if (!ccount)
                     break;

                  if (ccount & 0x80) {
                     ccount &= 0x7f;
                     sinfile->sread((char *)buffer, ccount);
                     *plengthtab -= ccount;
                     for (map=buffer, limit = map + ccount; map < limit; map++, base+=4)
                        *base = *map;
                  }

                  else {
                     b = sinfile->scan_uchar();
                     *plengthtab -= 1;
                     for (limit = base + (ccount<<2); base<limit; base+=4)
                        *base = b;
                  }

               }

            else
               while (*plengthtab) {
                  sinfile->skip(1);
                  ccount = sinfile->scan_uchar();
                  *plengthtab -= 2;

                  if (!ccount)
                     break;

                  if (ccount & 0x80) {
                     ccount &= 0x7f;
                     *plengthtab -= ccount<<1;

                     while (ccount--) {
                        *base = sinfile->scan_uchar();
                        sinfile->skip(1);
                        base += 4;
                     }

                  }

                  else {
                     b = sinfile->scan_uchar();
                     sinfile->skip(1);
                     *plengthtab -= 2;

                     while (ccount--) {
                        *base = b;
                        base += 4;
                     }

                  }

               }

         }

      }

      delete [] starttab;
      delete [] lengthtab;

      if (head.bpc == 1)
         delete [] charbuffer;
      else
         delete [] shortbuffer;
   }

   map = (unsigned char *)tob->data;

   j = (head.xsize*head.ysize)<<2;

   switch (flip) {
      case CBYTE_ORDER_ABGR:
         if (head.zsize < 3)
            for (i=0; i<j; i+=4)
               map[i+1] = map[i+2] = map[i+3];

         if (STANDARD_TEXTURE_PROCESSING || head.zsize < 4)
            for (i=0; i<j; i+=4)
               map[i] = (map[i+1] | map[i+2] | map[i+3]) ? 255 : 0;
 
         break;

      case CBYTE_ORDER_ARGB:
         if (head.zsize < 3)
            for (i=0; i<j; i+=4)
               map[i+3] = map[i+2] = map[i+1];

         if (STANDARD_TEXTURE_PROCESSING || head.zsize < 4)
            for (i=0; i<j; i+=4)
               map[i] = (map[i+1] | map[i+2] | map[i+3]) ? 255 : 0;
 
         break;

      case CBYTE_ORDER_BGRA:
         if (head.zsize < 3)
            for (i=0; i<j; i+=4)
               map[i] = map[i+1] = map[i+2];

         if (STANDARD_TEXTURE_PROCESSING || head.zsize < 4)
            for (i=0; i<j; i+=4)
               map[i+3] = (map[i] | map[i+1] | map[i+2]) ? 255 : 0;
 
         break;

      default:  // CBYTE_ORDER_RGBA
         if (head.zsize < 3)
            for (i=0; i<j; i+=4)
               map[i+1] = map[i+2] = map[i];

         if (STANDARD_TEXTURE_PROCESSING || head.zsize < 4)
            for (i=0; i<j; i+=4)
               map[i+3] = (map[i] | map[i+1] | map[i+2]) ? 255 : 0;
 
         break;
   }

   delete [] buffer;
   return 1;
}


/* ***********************************************************************
*********************************************************************** */
int rgb::skim_data(int *maxx, int *maxy, int flip) {

   sinfile->sseek(0);

   if (!scan_header())
      return 0;

   *maxx = head.xsize;
   *maxy = head.ysize;

   return 1;
}

