

#include <string.h>

#include "pstring.h"
#include "image.h"
#include "global.h"


/* Define internal ILBM types */
#define ILBM_NORMAL     0
#define ILBM_EHB        1
#define ILBM_HAM        2
#define ILBM_HAM8       3
#define ILBM_24BIT      4


typedef unsigned char ilbm_colortype[3];


/* ***************************************************************
*************************************************************** */
basic_loader *ilbm::find_loader(sfile *data) {

   int i;
   char buffer[8];

   sinfile = data;
   sinfile->sseek(0);

   for (i=0; i<4; i++)
      buffer[i] = sinfile->scan_uchar();

   buffer[4] = 0;

   if (!strcmp(buffer, "FORM"))
      return this;

   sinfile = NULL;

   return next ? ((basic_loader *)next)->find_loader(data) : NULL;
}


/* *********************************************************
********************************************************* */
int ilbm::scan_data(mapul *tob, int flip) {

   int i, j, k;
   char buffer[MAXSTRLEN];
   unsigned char *cbuffer;
   char bmhdflag = 0, camgflag = 0;

   unsigned int depth, compression;
   unsigned int viewmode;

   int bytesize, chunksize;

   ilbm_colortype *cmap = NULL;
   ilbm_colortype *icptr;
   int colorcount, memcount;

   char format;

   unsigned int lineskip;
   unsigned char bitmask;

   vector4uc curr;
   unsigned char *pic;
   unsigned char *ucptr, *ucptr2, *ucptr3;
   unsigned int ultemp;
   int col, colbit;

   sinfile->sseek(0);

   for (i=0; i<4; i++)
      buffer[i] = sinfile->scan_uchar();

   buffer[4] = 0;

   if (strcmp(buffer, "FORM"))
      return 0;

   bytesize = sinfile->scan_uint();

   for (i=0; i<4; i++)
      buffer[i] = sinfile->scan_uchar();

   buffer[4] = 0;
   bytesize -= 4;

   if (strcmp(buffer, "ILBM"))
      return 0;

   while (bytesize > 0) {

      for (i=0; i<4; i++)
         buffer[i] = sinfile->scan_uchar();

      buffer[4] = 0;
      chunksize = sinfile->scan_uint();
      bytesize -= 8 + chunksize;

      if (bytesize < 0) {
         if (cmap)
            delete [] cmap;
         return 0;
      }

      if (!strcmp(buffer, "BMHD")) {
         bmhdflag = 1;

         i = sinfile->scan_ushort();
         j = sinfile->scan_ushort();
         tob->init_map(i, j);

         sinfile->skip(4);                 // dunno what this is...

         depth = sinfile->scan_uchar();
         //mask = sinfile->scan_uchar();
         sinfile->scan_uchar();
         compression = sinfile->scan_uchar();

         sinfile->skip(1);                 // dunno what this is...

         sinfile->skip(2);                 // transcol
         chunksize -= 14;

         if (chunksize)                 // ignore rest if any
            sinfile->skip(chunksize);
      }

      else if (!strcmp(buffer, "CMAP")) {
         memcount = colorcount = chunksize / 3;
         if (cmap)
            delete [] cmap;

         cmap = new ilbm_colortype[colorcount];

         for (i=0; i<colorcount; i++) {
            cmap[i][0] = sinfile->scan_uchar();              // r
            cmap[i][1] = sinfile->scan_uchar();              // g
            cmap[i][2] = sinfile->scan_uchar();              // b
         }

      }

      else if (!strcmp(buffer, "CAMG")) {
         camgflag = 1;
         viewmode = sinfile->scan_uint();
      }

      else if (!strcmp(buffer, "BODY")) {

         if (!cmap)                                     // BMHD found?
            return 0;

         if (!bmhdflag) {                               // BMHD found?
            delete [] cmap;
            return 0;
         }

         if (compression == 1) {                // rle compressed
            i = (((tob->maxx + 15)>>4)<<1)*tob->maxy*depth; // uncompressed size
            cbuffer = new unsigned char[i];

            if (decomprle(cbuffer, i) == -1) {  // RLE chunk corrupt
               delete [] cmap;
               delete [] cbuffer;
               return 0;
            }

         }

         else {
            cbuffer = new unsigned char[chunksize];
            for (i=0; i<chunksize; i++)
               cbuffer[i] = sinfile->scan_uchar();
         }

         /* the following determines the type of the ILBM file.
           it's either NORMAL, EHB, HAM, HAM8 or 24BIT */

         format = ILBM_NORMAL;                        /* assume normal ILBM */

         if (depth == 24)
            format = ILBM_24BIT;

         if (camgflag)
            if (depth == 8) {
               if (viewmode & 0x800)
                  format = ILBM_HAM8;
            }

            else if (depth > 5) {
               if (viewmode & 0x80)
                  format = ILBM_EHB;
               else if (viewmode & 0x800)
                  format = ILBM_HAM;
            }

         if (format != ILBM_24BIT || format != ILBM_HAM8) {
            switch(format) {
               case ILBM_NORMAL:
                  if (colorcount < (i = (1<<depth))) {
                     delete [] cmap;
                     delete [] cbuffer;
                     return 0;
                  }

                  colorcount =  i;
                  break;

               case ILBM_EHB:
                  if (colorcount < 32) {
                     delete [] cmap;
                     delete [] cbuffer;
                     return 0;
                  }

                  colorcount = 32;
                  break;

               case ILBM_HAM:
                  if (colorcount < 16) {
                     delete [] cmap;
                     delete [] cbuffer;
                     return 0;
                  }

                  colorcount = 16;
                  break;
            }

            for (i=0; i<colorcount; i++) {
               cmap[i][0] = (cmap[i][0]>>4)*17;
               cmap[i][1] = (cmap[i][1]>>4)*17;
               cmap[i][2] = (cmap[i][2]>>4)*17;
            }

         }

         ucptr = cbuffer;
         lineskip = ((tob->maxx + 15) >> 4) << 1; // # of bytes/line
         ultemp = lineskip * depth;

         if (format != ILBM_NORMAL && format != ILBM_EHB) {

            for (i=0; i < (int)tob->maxy; i++) {
               pic = (unsigned char *)tob->pdata[tob->maxy-(1+i)];
               bitmask = 0x80;
               ucptr2 = ucptr;

               // at start of each line, init RGB values to background
               curr[0] = cmap[0][0];
               curr[1] = cmap[0][1];
               curr[2] = cmap[0][2];

               for (j=0; j < (int)tob->maxx; j++) {
                  col = 0;
                  colbit = 1;
                  ucptr3 = ucptr2;

                  for (k=0; k < (int)depth; k++) {
                     if (*ucptr3 & bitmask)
                        col += colbit;

                     ucptr3 += lineskip;
                     colbit <<= 1;
                  }

                  if (format==ILBM_HAM)
                     switch (col & 0x30) {
                        case 0x00:
                           k = col & 0x0f;
                           curr[0] = cmap[k][0];
                           curr[1] = cmap[k][1];
                           curr[2] = cmap[k][2];
                           break;

                        case 0x10:
                           curr[2] = (col & 0x0f) * 17;
                           break;

                        case 0x20:
                           curr[0] = (col & 0x0f) * 17;
                           break;

                        case 0x30:
                           curr[1] = (col & 0x0f) * 17;
                     }

                  else if (format == ILBM_HAM8)
                     switch(col & 0xc0) {
                        case 0x00:
                           k = col & 0x3f;
                           curr[0] = cmap[k][0];
                           curr[1] = cmap[k][1];
                           curr[2] = cmap[k][2];
                           break;

                        case 0x40:
                           curr[2] = (curr[2] & 3) | ((col & 0x3f) << 2);
                           break;

                        case 0x80:
                           curr[0] = (curr[0] & 3) | ((col & 0x3f) << 2);
                           break;

                        case 0xc0:

                           // next line - XV BUG??? (green based on red???)
                           // curr[1] = (curr[0] & 3) | ((col & 0x3f) << 2);
                           curr[1] = (curr[1] & 3) | ((col & 0x3f) << 2);
                     }

                  else {
                     curr[0] = col & 0xff;
                     curr[1] = (col >> 8) & 0xff;
                     curr[2] = (col >> 16) & 0xff;
                  }

                  switch (flip) {
                     case CBYTE_ORDER_ABGR:
                        *pic++ = (curr[0] | curr[1] | curr[2]) ? 255 : 0;
                        *pic++ = curr[2];
                        *pic++ = curr[1];
                        *pic++ = curr[0];
                        break;

                     case CBYTE_ORDER_ARGB:
                        *pic++ = (curr[0] | curr[1] | curr[2]) ? 255 : 0;
                        *pic++ = curr[0];
                        *pic++ = curr[1];
                        *pic++ = curr[2];
                        break;

                     case CBYTE_ORDER_BGRA:
                        *pic++ = curr[2];
                        *pic++ = curr[1];
                        *pic++ = curr[0];
                        *pic++ = (curr[0] | curr[1] | curr[2]) ? 255 : 0;
                        break;

                     default:  // CBYTE_ORDER_RGBA
                        *pic++ = curr[0];
                        *pic++ = curr[1];
                        *pic++ = curr[2];
                        *pic++ = (curr[0] | curr[1] | curr[2]) ? 255 : 0;
                        break;
                  }

                  bitmask = bitmask >> 1;
                  if (!bitmask) {
                     bitmask = 0x80;
                     ucptr2++;
                  }

               }

               ucptr += ultemp;
            }

         }  /* HAM, HAM8, or 24BIT */

         else {
            if (format == ILBM_EHB) {           // double cmap for EHB mode
               if (colorcount + 32 > memcount) {
                  icptr = cmap;
                  memcount = colorcount + 32;
                  cmap = new ilbm_colortype[memcount];
                  memcpy(cmap, icptr, colorcount*3);

                  delete [] icptr;
               }

               for (i=0; i<32; i++) {
                  cmap[i+colorcount][0] = cmap[i][0]>>1;
                  cmap[i+colorcount][1] = cmap[i][1]>>1;
                  cmap[i+colorcount][2] = cmap[i][2]>>1;
               }

            }

            for (i=0; i < (int)tob->maxy; i++) {
               pic = (unsigned char *)tob->pdata[tob->maxy-(1+i)];
               bitmask = 0x80;                      /* left most bit (mask) */
               ucptr2 = ucptr;                 /* work ptr to source */

               for (j=0; j < (int)tob->maxx; j++) {
                  col = 0;
                  colbit = 1;
                  ucptr3 = ucptr2;              /* ptr to byte in 1st pln */

                  for (k=0; k < (int)depth; k++) {
                     if (*ucptr3 & bitmask)          /* if bit set in this pln */
                        col = col + colbit;           /* add bit to chunky byte */
                     ucptr3 += lineskip;           /* go to next line */
                     colbit <<= 1;                   /* shift color bit */
                  }

                  switch (flip) {
                     case CBYTE_ORDER_ABGR:
                        *pic++ = (cmap[col][0] | cmap[col][1] | cmap[col][2]) ? 255 : 0;
                        *pic++ = cmap[col][2];
                        *pic++ = cmap[col][1];
                        *pic++ = cmap[col][0];
                        break;

                     case CBYTE_ORDER_ARGB:
                        *pic++ = (cmap[col][0] | cmap[col][1] | cmap[col][2]) ? 255 : 0;
                        *pic++ = cmap[col][0];
                        *pic++ = cmap[col][1];
                        *pic++ = cmap[col][2];
                        break;

                     case CBYTE_ORDER_BGRA:
                        *pic++ = cmap[col][2];
                        *pic++ = cmap[col][1];
                        *pic++ = cmap[col][0];
                        *pic++ = (cmap[col][0] | cmap[col][1] | cmap[col][2]) ? 255 : 0;
                        break;

                     default:  // CBYTE_ORDER_RGBA
                        *pic++ = cmap[col][0];
                        *pic++ = cmap[col][1];
                        *pic++ = cmap[col][2];
                        *pic++ = (cmap[col][0] | cmap[col][1] | cmap[col][2])? 255 : 0;
                        break;
                  }

                  bitmask = bitmask >> 1;             /* shift mask to next bit */
                  if (!bitmask) {                /* if mask is zero */
                     bitmask = 0x80;                  /* reset mask */
                     ucptr2++;                     /* mv ptr to next byte */
                  }

               }  /* for j ... */

               ucptr += ultemp;
            }  /* for i ... */

         }  /* if NORMAL or EHB */

         break;
      }

      else // skip - unknown...
         sinfile->skip(chunksize);
   }

   if (cmap)
      delete [] cmap;

   if (cbuffer)
      delete [] cbuffer;

   return 1;
}


/* *********************************************************
********************************************************* */
int ilbm::skim_data(int *maxx, int *maxy, int flip) {

   int i;
   char buffer[8];
   int bytesize, chunksize;

   sinfile->sseek(0);
   
   for (i=0; i<4; i++)
      buffer[i] = sinfile->scan_uchar();

   buffer[4] = 0;

   if (strcmp(buffer, "FORM"))
      return 0;

   bytesize = sinfile->scan_uint();

   for (i=0; i<4; i++)
      buffer[i] = sinfile->scan_uchar();

   buffer[4] = 0;
   bytesize -= 4;

   if (strcmp(buffer, "ILBM"))
      return 0;

   while (bytesize > 0) {

      for (i=0; i<4; i++)
         buffer[i] = sinfile->scan_uchar();

      buffer[4] = 0;
      chunksize = sinfile->scan_uint();
      bytesize -= 8 + chunksize;

      if (bytesize < 0)
         return 0;

      if (!strcmp(buffer, "BMHD")) {
         *maxx = sinfile->scan_ushort();
         *maxy = sinfile->scan_ushort();
         return 1;
      }

      // skip - unknown...
      sinfile->skip(chunksize);
   }

   return 0;
}



/* *************************************************************************
  void decomprle(source, destination, source length, buffer size)

  Decompress run-length encoded data from source to destination. Terminates
  when source is decoded completely or destination buffer is full.

  The decruncher is as optimized as I could make it, without risking
  safety in case of corrupt BODY chunks.
************************************************************************** */
int ilbm::decomprle(unsigned char *dest, int size) {

   int bytecount = 0, count = 0;
   unsigned char len, temp;
   int index = 0;
   int i;

   while (count < size) {

      // read control byte
      len = sinfile->scan_uchar();
      bytecount++;

      if (len == 0x80)
         return bytecount;      // done ????

      else if (!(len & 0x80)) {
         len++;
         bytecount += len;
         count += len;

         if (count > size)
            break;

         for (i=index+len; index<i; index++)
            dest[index] = sinfile->scan_uchar();
      }

      else {
         len = 0x81 - (len & 0x7f);
         bytecount++;
         count += len;

         if (count > size)
            break;

         temp = sinfile->scan_uchar();

         for (i=index+len; index<i; index++)
            dest[index] = temp;
      }

   }

   if (count > size)
      return -1;

   return bytecount;
}

