

#include <string.h>

#include "image.h"
#include "global.h"


int code_mask[13] = {
     0,
     0x0001, 0x0003,
     0x0007, 0x000F,
     0x001F, 0x003F,
     0x007F, 0x00FF,
     0x01FF, 0x03FF,
     0x07FF, 0x0FFF
};


/* ***************************************************************
*************************************************************** */
basic_loader *gif::find_loader(sfile *data) {

   sinfile = data;
   sinfile->sseek(0);

   sinfile->sread(head.signature, 3);
   head.signature[3] = 0;

   if (!strcmp(head.signature, "GIF"))
      return this;

   sinfile = NULL;

   return next ? ((basic_loader *)next)->find_loader(data) : NULL;
}


/* ************************************************************************
************************************************************************ */
void gif::out_line(unsigned char *buffer, mapul *image, int width, int scany) {

   int i, j;

   j = gimage.left+width;

   if (gimage.transparentflag) {
      for (i=gimage.left; i<j; i++)
         if (buffer[i] != gimage.transparentindex)
            ((int *)image->pdata[scany])[i] = *((int *)(color_table[buffer[i]]));
      return;
   }

   for (i=gimage.left; i<j; i++)
      ((int *)image->pdata[scany])[i] = *((int *)(color_table[buffer[i]]));
}


/* ************************************************************************
 * - gets the next code from the GIF file.  Returns the code, or else
 * a negative number in case of file errors...
************************************************************************ */
int gif::get_next_code(gifdecodetype *stuff) {

   int ret;

   if (!stuff->bits_left) {
      if (!stuff->bytes_left) {

         // Out of bytes in current block, so read next block

         stuff->nextcptr = stuff->buffer;
         do {
            stuff->bytes_left = sinfile->scan_uchar();
         } while(!stuff->bytes_left);

         sinfile->sread((char *)stuff->buffer, stuff->bytes_left);
      }

      stuff->cptr = *stuff->nextcptr++;
      stuff->bits_left = 8;
      stuff->bytes_left--;
   }

   ret = stuff->cptr >> (8 - stuff->bits_left);

   while (stuff->codesize > stuff->bits_left) {
      if (!stuff->bytes_left) {

         // Out of bytes in current block, so read next block

         stuff->nextcptr = stuff->buffer;
         do {
            stuff->bytes_left = sinfile->scan_uchar();
         } while(!stuff->bytes_left);

         sinfile->sread((char *)stuff->buffer, stuff->bytes_left);
      }

      stuff->cptr = *stuff->nextcptr++;
      ret |= stuff->cptr << stuff->bits_left;
      stuff->bits_left += 8;
      --stuff->bytes_left;
   }

   stuff->bits_left -= stuff->codesize;
   return ret &= code_mask[stuff->codesize];
}


/* ************************************************************************
 *
 * - This function decodes an LZW image, according to the method used
 * in the GIF spec.  Every *linewidth* "characters" (ie. pixels) decoded
 * will generate a call to out_line(), which is a user specific function
 * to display a line of pixels.  The function gets it's codes from
 * get_next_code() which is responsible for reading blocks of data and
 * seperating them into the proper size codes.  Finally, get_byte() is
 * the global routine to read the next byte from the GIF file.
 *
 * It is generally a good idea to have linewidth correspond to the actual
 * width of a line (as specified in the Image header) to make your own
 * code a bit simpler, but it isn't absolutely necessary.
 *
************************************************************************ */
void gif::decoder(mapul *image) {

   unsigned char size;                  // # of bits of pixels
   int top_slot;                        // # of pixels to decode
   int clear, ending;                   // clear/end code
   int slot, newcodes;                  // first avail compression code value
   int oc, fc = 0;
   unsigned char *sp, *bufptr;

   unsigned char *buffer, *stack;
   int bufcnt;
   int c;

   int code;
   gifdecodetype stuff;
   int scany = head.ysize - (1 + gimage.top);

   stuff.bytes_left = 0;                // # bytes left in block
   stuff.bits_left  = 0;                // # bits left in current byte

   // Initialize for decoding a new image...
   size = sinfile->scan_uchar();

   // initializes the decoder for reading a new image.
   stuff.codesize = size + 1;

   top_slot = 1 << stuff.codesize;
   clear = 1 << size;
   ending = clear + 1;
   slot = newcodes = ending + 1;
   stuff.bytes_left = stuff.bits_left = 0;

   // Allocate space for the decode buffer
   buffer = new unsigned char[gimage.width + 1];
   stack  = new unsigned char[gimage.height*gimage.width+1];

   // Set up the stack pointer and decode buffer pointer
   sp     = stack;
   bufptr = buffer;
   bufcnt = 0;

   /* This is the main loop.  For each code we get we pass through the
    * linked list of prefix codes, pushing the corresponding "character" for
    * each code onto the stack.  When the list reaches a single "character"
    * we push that on the stack too, and then start unstacking each
    * character for output in the correct order.  Special handling is
    * included for the clear code, and the whole thing ends when we get
    * an ending code.
    */

   while ((c = get_next_code(&stuff)) != ending) {

      // If the code is a clear code, reinitialize all necessary items.

      if (c == clear) {
         stuff.codesize = size + 1;
         slot = newcodes;
         top_slot = 1 << stuff.codesize;

         // Continue reading codes until we get a non-clear code
         // (Another unlikely, but possible case...)

         while ((c = get_next_code(&stuff)) == clear);

         // If we get an ending code immediately after a clear code
         // (Yet another unlikely case), then break out of the loop.

         if (c == ending)
            break;

         /* Finally, if the code is beyond the range of already set codes,
          * (This one had better NOT happen...  I have no idea what will
          * result from this, but I doubt it will look good...) then set it
          * to color zero.
          */

         if (c >= slot)
            c = 0;

         oc = fc = c;

         /* And let us not forget to put the char into the buffer... And
          * if, on the off chance, we were exactly one pixel from the end
          * of the line, we have to send the buffer to the out_line()
          * routine...
          */

         *bufptr++ = c;
         if (++bufcnt == gimage.width) {

            if (image)
               out_line(buffer, image, bufcnt, scany);

            scany-=gimage.slinc;

            if (scany <= head.ysize - (gimage.top+gimage.height) - 1) {
               if (gimage.interlace_flag)
                  switch (gimage.pass) {

                     case 0:
                        gimage.pass = 1;
                        gimage.slinc = 8;
                        scany = head.ysize - (5 + gimage.top);
                        break;

                     case 1:
                        gimage.pass = 2;
                        gimage.slinc = 4;
                        scany = head.ysize - (3 + gimage.top);
                        break;

                     default:
                        gimage.pass = 3;
                        gimage.slinc = 2;
                        scany = head.ysize - (2 + gimage.top);
                        break;

                  }

            }

            bufptr = buffer;
            bufcnt = 0;
         }

      }

      else {
         /* In this case, it's not a clear code or an ending code, so
          * it must be a code code...  So we can now decode the code into
          * a stack of character codes. (Clear as mud, right?)
          */

         code = c;

         /* Here we go again with one of those off chances...  If, on the
          * off chance, the code we got is beyond the range of those already
          * set up (Another thing which had better NOT happen...) we trick
          * the decoder into thinking it actually got the last code read.
          * (Hmmn... I'm not sure why this works...  But it does...)
          */

         if (code >= slot) {
            code = oc;
            *sp++ = fc;
         }

         /* Here we scan back along the linked list of prefixes, pushing
          * helpless characters (ie. suffixes) onto the stack as we do so.
          */

         while (code >= newcodes) {
            *sp++ = suffix[code];
            code  = prefix[code];
         }

         /* Push the last character on the stack, and set up the new
          * prefix and suffix, and if the required slot number is greater
          * than that allowed by the current bit size, increase the bit
          * size.  (NOTE - If we are all full, we *don't* save the new
          * suffix and prefix...  I'm not certain if this is correct...
          * it might be more proper to overwrite the last code...
          */

         *sp++ = code;
         if (slot < top_slot) {
            suffix[slot] = fc = code;
            prefix[slot++] = oc;
            oc = c;
         }

         if (slot >= top_slot)
            if (stuff.codesize < 12) {
               top_slot <<= 1;
               stuff.codesize++;
            }

         /* Now that we've pushed the decoded string (in reverse order)
          * onto the stack, lets pop it off and put it into our decode
          * buffer...  And when the decode buffer is full, write another
          * line...
          */

         while (sp > stack) {
            *bufptr++ = *(--sp);
            if (++bufcnt == gimage.width) {
               if (image)
                  out_line(buffer, image, bufcnt, scany);

               scany-=gimage.slinc;

               if (scany <= head.ysize - (gimage.top+gimage.height) - 1) {
                  if (gimage.interlace_flag)
                     switch (gimage.pass) {

                        case 0:
                           gimage.pass = 1;
                           gimage.slinc = 8;
                           scany = head.ysize - (5 + gimage.top);
                           break;

                        case 1:
                           gimage.pass = 2;
                           gimage.slinc = 4;
                           scany = head.ysize - (3 + gimage.top);
                           break;

                        default:
                           gimage.pass = 3;
                           gimage.slinc = 2;
                           scany = head.ysize - (2 + gimage.top);
                           break;

                     }

               }

               bufptr = buffer;
               bufcnt = 0;
            }

         }

      }

   }

   if (bufcnt && image && scany > -1)
      out_line(buffer, image, bufcnt, scany);

   delete [] buffer;
   delete [] stack;
}


/* ************************************************************************
************************************************************************ */
int gif::scan_header() {

   unsigned char temp[4];

   // signature
   sinfile->sread(head.signature, 3);
   head.signature[3] = 0;

   if (strcmp(head.signature, "GIF"))
      return 0;

   // version
   sinfile->sread(head.version, 3);
   head.version[3] = 0;

   // logical screen descriptor
   head.xsize = sinfile->scan_rushort();
   head.ysize = sinfile->scan_rushort();

   sinfile->sread((char *)temp, 3);

   head.gctable_flag = (temp[0]>>7);                 // global color table flag
   head.color_resolution = ((temp[0]>>4) & 0x07);    // color resolution - # of bits per color
   head.sort_flag = (temp[0]>>3) & 0x01;             // sort flag
   head.gctable_size = 1 << ((temp[0] & 0x07)+1);    // global color table size 2^(x+1)

   head.bc_index = temp[1];
   head.pixel_aspect_ratio = temp[2];

   return 1;
}


/* ************************************************************************
************************************************************************ */
int gif::scan_data(mapul *tob, int flip) {

   int i, j, k, m, n;
//   int l;
   unsigned char temp;
   unsigned char is, it;
   int doneflag;

   sinfile->sseek(0);

   // read in header & logical screen descriptor
   if (!scan_header())
      return 0;

   tob->init_map(head.xsize, head.ysize);

// not needed? - found old bug (l gets altered by accident - fixed) - ok now?
//   l = 7-head.color_resolution;

   // global color table
   if (head.gctable_flag) {
      i = head.gctable_size<<2;
      pgctable = new puchar[head.gctable_size];
      gctable = new unsigned char[i];

      for (j=k=0; k<i; j++, k+=4) {
         pgctable[j] = &gctable[k];

         switch (flip) {
            case CBYTE_ORDER_ABGR:
               pgctable[j][3] = sinfile->scan_uchar();
               pgctable[j][2] = sinfile->scan_uchar();
               pgctable[j][1] = sinfile->scan_uchar();
               pgctable[j][0] = (pgctable[j][1] | pgctable[j][2] | pgctable[j][3]) ? 255 : 0;
               break;

            case CBYTE_ORDER_ARGB:
               pgctable[j][1] = sinfile->scan_uchar();
               pgctable[j][2] = sinfile->scan_uchar();
               pgctable[j][3] = sinfile->scan_uchar();
               pgctable[j][0] = (pgctable[j][1] | pgctable[j][2] | pgctable[j][3]) ? 255 : 0;
               break;

            case CBYTE_ORDER_BGRA:
               pgctable[j][2] = sinfile->scan_uchar();
               pgctable[j][1] = sinfile->scan_uchar();
               pgctable[j][0] = sinfile->scan_uchar();
               pgctable[j][3] = (pgctable[j][0] | pgctable[j][1] | pgctable[j][2]) ? 255 : 0;
               break;

            default:  // CBYTE_ORDER_RGBA
               pgctable[j][0] = sinfile->scan_uchar();
               pgctable[j][1] = sinfile->scan_uchar();
               pgctable[j][2] = sinfile->scan_uchar();
               pgctable[j][3] = (pgctable[j][0] | pgctable[j][1] | pgctable[j][2]) ? 255 : 0;
               break;
         }

      }

/* apparently not needed

      if (l)
         for (k=0; k<head.gctable_size; k++) {
            pgctable[k][3] = (unsigned char)(pgctable[k][3] << l);
            pgctable[k][2] = (unsigned char)(pgctable[k][2] << l);
            pgctable[k][1] = (unsigned char)(pgctable[k][1] << l);
         }
*/
      k = *((int *)pgctable[head.bc_index]);

      for (i=0; i<head.ysize; i++)              // init background color
         for (j=0; j<head.xsize; j++)
            ((int *)tob->pdata[i])[j] = k;
   }

   doneflag = 1;

   // image descriptors
   while (doneflag) {
 
      if (sinfile->seof())
         break;

      is = sinfile->scan_uchar(); 

      switch (is) {

         // trailer/eof
         case 0x3b:
            doneflag = 0;
            break;

         // extension
         case 0x21: 
            it = sinfile->scan_uchar();

            switch (it) {

               // graphic control extension
               case 0xf9:
                  sinfile->skip(1);             // blocksize = 4
                  temp = sinfile->scan_uchar(); // packed byte
                                                // bit 5-7 reserved
                                                // bit 2-4 disposal method
                                                // bit 1 user input flag
                                                // bit 0 transparent flag

                  gimage.transparentflag = temp & 0x01;
                  gimage.renderflag = (temp>>2) && 0x07;

                  sinfile->skip(2);             // delay time

                  gimage.transparentindex = sinfile->scan_uchar(); // transparent color index
                  sinfile->skip(1);             // block terminator = 0x00
                  break;

               case 0xff:                       // application extention
               case 0x01:                       // plain text extension
               case 0xfe:                       // comment extension
               case 0x52:                       // aspect extension
               default:
                  do {
                     temp = sinfile->scan_uchar();
                     sinfile->skip(temp);
                  } while(temp);

                  break;
            }

            break;

         case 0x2c:                     // image separator
            gimage.image_separator = is;

            gimage.left   = sinfile->scan_rushort();
            gimage.top    = sinfile->scan_rushort();
            gimage.width  = sinfile->scan_rushort();
            gimage.height = sinfile->scan_rushort();

            temp = sinfile->scan_uchar();

            gimage.lctable_flag = (temp>>7) & 0x01;
            gimage.interlace_flag = (temp>>6) & 0x01;
            gimage.sort_flag = (temp>>5) & 0x01;
            gimage.reserved = (temp>>3) & 0x03;
            gimage.lctable_size = 1<<((temp & 0x07)+1);

            if (gimage.interlace_flag) {
               gimage.pass = 0;
               gimage.slinc = 8;
            }

            else
               gimage.slinc = 1;

            if (gimage.lctable_flag) {          // local color table if not exist use gct
               if (gimage.plctable) {
                  delete [] gimage.plctable;
                  delete [] gimage.lctable;
               }

               i = gimage.lctable_size<<2;

               gimage.lctable = new unsigned char[i];
               gimage.plctable = new puchar[gimage.lctable_size];

               for (j=k=0; k<i; j++, k+=4) {
                  gimage.plctable[j] = &gimage.lctable[k];

                  switch (flip) {
                     case CBYTE_ORDER_ABGR:
                        gimage.plctable[j][3] = sinfile->scan_uchar();
                        gimage.plctable[j][2] = sinfile->scan_uchar();
                        gimage.plctable[j][1] = sinfile->scan_uchar();
                        gimage.plctable[j][0] = (gimage.plctable[j][1] | gimage.plctable[j][2] | gimage.plctable[j][3]) ? 255 : 0;
                        break;

                     case CBYTE_ORDER_ARGB:
                        gimage.plctable[j][1] = sinfile->scan_uchar();
                        gimage.plctable[j][2] = sinfile->scan_uchar();
                        gimage.plctable[j][3] = sinfile->scan_uchar();
                        gimage.plctable[j][0] = (gimage.plctable[j][1] | gimage.plctable[j][2] | gimage.plctable[j][3]) ? 255 : 0;
                        break;

                     case CBYTE_ORDER_BGRA:
                        gimage.plctable[j][2] = sinfile->scan_uchar();
                        gimage.plctable[j][1] = sinfile->scan_uchar();
                        gimage.plctable[j][0] = sinfile->scan_uchar();
                        gimage.plctable[j][3] = (gimage.plctable[j][0] | gimage.plctable[j][1] | gimage.plctable[j][2]) ? 255 : 0;
                        break;

                     default:  // CBYTE_ORDER_RGBA
                        gimage.plctable[j][0] = sinfile->scan_uchar();
                        gimage.plctable[j][1] = sinfile->scan_uchar();
                        gimage.plctable[j][2] = sinfile->scan_uchar();
                        gimage.plctable[j][3] = (gimage.plctable[j][0] | gimage.plctable[j][1] | gimage.plctable[j][2]) ? 255 : 0;
                        break;
                  }

               }

/* apparently not needed
               if (l)
                  for (k=0; k<gimage.lctable_size; k++) {
                     gimage.plctable[k][3] = (unsigned char)(gimage.plctable[k][3] << l);
                     gimage.plctable[k][2] = (unsigned char)(gimage.plctable[k][2] << l);
                     gimage.plctable[k][1] = (unsigned char)(gimage.plctable[k][1] << l);
                  }
*/

               color_table = gimage.plctable;
            }

            else
               color_table = pgctable;

            switch (gimage.renderflag) {

               case 0:
               case 1:
                  decoder(tob);
                  break;

               case 2:                          // goes through default
                  j = gimage.top-gimage.height;
                  m = gimage.left+gimage.width;
                  n = *((int *)(pgctable[head.bc_index]));

                  for (i=gimage.top; i>j; i--)
                     for (k=gimage.left; k<m; k++)
                        ((int *)tob->pdata[i])[k] = n;

               default:
                  decoder(NULL);
                  break;
            }

            gimage.transparentflag = 0;
            gimage.renderflag = 0;
            break;
      }

   }

   if (pgctable) {
      delete [] pgctable;
      delete [] gctable;
      pgctable = NULL;
   }

   if (gimage.plctable) {
      delete [] gimage.plctable;
      delete [] gimage.lctable;
      gimage.plctable = NULL;
   }

   return 1;
}


/* ************************************************************************
************************************************************************ */
int gif::skim_data(int *maxx, int *maxy, int flip) {

   sinfile->sseek(0);

   if (!scan_header())
      return 0;

   *maxx = head.xsize;
   *maxy = head.ysize;

   return 1;
}

