

#include <string.h>

#include "image.h"
#include "global.h"


#define BI_RGB  0
#define BI_RLE8 1
#define BI_RLE4 2


/* ***************************************************************
*************************************************************** */
basic_loader *bmp::find_loader(sfile *data) {

   int i, j;

   sinfile = data;
   sinfile->sseek(0);

   i = sinfile->scan_uchar();
   j = sinfile->scan_uchar();

   if (i == 'B' && j == 'M')
      return this;

   sinfile = NULL;

   return next ? ((basic_loader *)next)->find_loader(data) : NULL;
}


/* ***************************************************************
*************************************************************** */
int bmp::query_256() {

   bmp_file_header file_header;
   int size, bpp;

   sinfile->sseek(0);

   // read file header
   file_header.type[0] = sinfile->scan_uchar();
   file_header.type[1] = sinfile->scan_uchar();

   if (file_header.type[0] != 'B' || file_header.type[1] != 'M')
      return 0;

   file_header.size = sinfile->scan_ruint();
   sinfile->skip(8);

   // read bitmap header
   size = sinfile->scan_ruint();

   sinfile->skip((size==40 || size==64) ? 10 : 6);

   bpp = sinfile->scan_rushort();

   return bpp == 8;
}


/* ***************************************************************
*************************************************************** */
int bmp::scan_data256(mapuc *tob, unsigned int *palette, int flip) {

   bmp_file_header file_header;
   bmp_bitmap_header bitmap_header;
   unsigned char *data, *pdata;
   unsigned char *ptr, *end_file;
   int i, j, k;
   int bytepad;
   unsigned int temp;

   sinfile->sseek(0);
   
   // read file header
   file_header.type[0] = sinfile->scan_uchar();
   file_header.type[1] = sinfile->scan_uchar();

   if (file_header.type[0] != 'B' || file_header.type[1] != 'M')
      return 0;

   // read bitmap header
   file_header.size = sinfile->scan_ruint();
   sinfile->scan_ruint();
   file_header.offset = sinfile->scan_ruint();

   memset(&bitmap_header, 0, sizeof(bmp_bitmap_header));

   bitmap_header.header_size = sinfile->scan_ruint();

   if (bitmap_header.header_size == 40 || bitmap_header.header_size == 64 ) {
      bitmap_header.xsize = sinfile->scan_ruint();
      bitmap_header.ysize = sinfile->scan_ruint();
      bitmap_header.plane = sinfile->scan_rushort();
      bitmap_header.bpp   = sinfile->scan_rushort();
      bitmap_header.compress_flag = sinfile->scan_ruint();
      bitmap_header.data_size = sinfile->scan_ruint();
      bitmap_header.hppm = sinfile->scan_ruint();
      bitmap_header.vppm = sinfile->scan_ruint();
      bitmap_header.palette_size = sinfile->scan_ruint();
      bitmap_header.palette_isize = sinfile->scan_ruint();

      if (bitmap_header.header_size > 40)
         sinfile->skip(bitmap_header.header_size - 40);
   }

   else {
      // assume - may not be true
      bitmap_header.xsize = sinfile->scan_rushort();
      bitmap_header.ysize = sinfile->scan_rushort();
      bitmap_header.plane = sinfile->scan_rushort();
      bitmap_header.bpp   = sinfile->scan_rushort();
      bitmap_header.data_size = ((bitmap_header.plane*bitmap_header.bpp*bitmap_header.xsize+31)>>5)*bitmap_header.ysize<<2;
      bitmap_header.compress_flag = BI_RGB;
      bitmap_header.hppm = bitmap_header.vppm = bitmap_header.palette_isize = 0;
      bitmap_header.palette_size = 256;
   }

   if (bitmap_header.bpp != 8)
      return 0;

   i = (bitmap_header.palette_size) ? bitmap_header.palette_size : (1 << bitmap_header.bpp);

   switch (flip) {

      case CBYTE_ORDER_ABGR:
         for (j=0; j<i; j++) {
            ((unsigned char *)&palette[j])[1] = sinfile->scan_uchar();
            ((unsigned char *)&palette[j])[2] = sinfile->scan_uchar();
            ((unsigned char *)&palette[j])[3] = sinfile->scan_uchar();
            sinfile->scan_uchar();
            ((unsigned char *)&palette[j])[0] = (((unsigned char *)&palette[j])[1] | ((unsigned char *)&palette[j])[2] | ((unsigned char *)&palette[j])[3]) ? 255 : 0;
         }

         break;

      case CBYTE_ORDER_ARGB:
         for (j=0; j<i; j++) {
            ((unsigned char *)&palette[j])[3] = sinfile->scan_uchar();
            ((unsigned char *)&palette[j])[2] = sinfile->scan_uchar();
            ((unsigned char *)&palette[j])[1] = sinfile->scan_uchar();
            sinfile->scan_uchar();
            ((unsigned char *)&palette[j])[0] = (((unsigned char *)&palette[j])[1] | ((unsigned char *)&palette[j])[2] | ((unsigned char *)&palette[j])[3]) ? 255 : 0;
         }

         break;

      case CBYTE_ORDER_BGRA:
         for (j=0; j<i; j++) {
            ((unsigned char *)&palette[j])[0] = sinfile->scan_uchar();
            ((unsigned char *)&palette[j])[1] = sinfile->scan_uchar();
            ((unsigned char *)&palette[j])[2] = sinfile->scan_uchar();
            sinfile->scan_uchar();
            ((unsigned char *)&palette[j])[3] = (((unsigned char *)&palette[j])[0] | ((unsigned char *)&palette[j])[1] | ((unsigned char *)&palette[j])[2]) ? 255 : 0;
         }

         break;

      default:  // CBYTE_ORDER_RGBA
         for (j=0; j<i; j++) {
            ((unsigned char *)&palette[j])[2] = sinfile->scan_uchar();
            ((unsigned char *)&palette[j])[1] = sinfile->scan_uchar();
            ((unsigned char *)&palette[j])[0] = sinfile->scan_uchar();
            sinfile->scan_uchar();
            ((unsigned char *)&palette[j])[3] = (((unsigned char *)&palette[j])[0] | ((unsigned char *)&palette[j])[1] | ((unsigned char *)&palette[j])[2]) ? 255 : 0;
         }

         break;
   }

   sinfile->sseek(file_header.offset);
   data = sinfile->ptr;

   tob->init_map(bitmap_header.xsize, bitmap_header.ysize);

   pdata = data-1;				// simulates byte not read in yet...
   ptr = tob->data;

   switch (bitmap_header.compress_flag) {

      case BI_RGB:
         bytepad = ((bitmap_header.xsize + 3) & 0xfffffffc) - bitmap_header.xsize;		// pad lines to multiples of 32 bits

         for (i=0; i<bitmap_header.ysize; i++, pdata+=bytepad) {
	    memcpy(ptr, pdata+1, bitmap_header.xsize);
	    pdata += bitmap_header.xsize;
	    ptr += bitmap_header.xsize;
         }
	 
         break;

      default:		// BI_RLE8:

         tob->zero();

         end_file = ptr + bitmap_header.xsize*bitmap_header.ysize;
         k = 0;

         do {
            pdata++;
            if (*pdata) {         // encode mode
               temp = *pdata;
               memset(ptr, *(++pdata), temp);
	       ptr += temp;
               k += temp;
            }

            else {
               pdata++;
               switch (*pdata) {

                  case 0x00:                                      // end of line segment
                     ptr += bitmap_header.xsize - k;
                     k = 0;
                     break;

                  case 0x01:                                      // end of pic
                     ptr = end_file;
                     break;

                  case 0x02:                                      // delta
                     pdata++;
                     j = *pdata;
                     k += j;

                     pdata++;
                     j += *pdata*bitmap_header.xsize;
                     ptr += j;
                     break;

                  default:                                // absolute mode
                     temp = *pdata;

                     memcpy(ptr, pdata+1, temp);
                     ptr += temp;
		     pdata += temp;

                     k += temp;

                     if (temp & 0x01)            // read pad
                        pdata++;

                     break;
               }

            }

         } while (ptr != end_file);

         break;
   }

   return 1;
}


/* ***************************************************************
*************************************************************** */
int bmp::write_data256(char *fname, mapuc *tob, unsigned int *palette, int flip) {

   FILE *outfile;
   int  i, j, k;
   unsigned char *ptr;
   int pad;
   
   if (!(outfile = fopen(fname,"wb")))
      return 0;

   pad = tob->maxx & 0x03;

   if (pad)
      pad = 4 - pad;

   // calc palette size
   for (k=0, i=0, ptr = tob->data; i < (int)tob->maxy; i++)
      for (j=0; j < (int)tob->maxx; j++, ptr++)
         if (*ptr > k)
	    k = *ptr;
	    
   j = (k+1)<<2;	    
   k++;
   i = tob->maxy*(tob->maxx+pad);

   putuchar(outfile, 'B');
   putuchar(outfile, 'M');
   putruint(outfile, 14+40+j+i);
   putushort(outfile, 0);
   putushort(outfile, 0);
   putruint(outfile, 14+40+j);

   putruint(outfile, 40);
   putruint(outfile, tob->maxx);
   putruint(outfile, tob->maxy);
   putrushort(outfile, 1);
   putrushort(outfile, 8);
   putruint(outfile, BI_RGB);
   putruint(outfile, i);
   putruint(outfile, 0);
   putruint(outfile, 0);
   putruint(outfile, k);
   putruint(outfile, k);

   ptr = (unsigned char *)palette;
       
   switch (flip) {
      case CBYTE_ORDER_ABGR:
         for (j=0; j<k; j++, ptr+=4) {
            putuchar(outfile, ptr[1]);
            putuchar(outfile, ptr[2]);
            putuchar(outfile, ptr[3]);
            putuchar(outfile, 0);
         }
	 
         break;

      case CBYTE_ORDER_ARGB:
         for (j=0; j<k; j++, ptr+=4) {
            putuchar(outfile, ptr[3]);
            putuchar(outfile, ptr[2]);
            putuchar(outfile, ptr[1]);
            putuchar(outfile, 0);
         }
	 
         break;

      case CBYTE_ORDER_BGRA:
         for (j=0; j<k; j++, ptr+=4) {
            putuchar(outfile, ptr[0]);
            putuchar(outfile, ptr[1]);
            putuchar(outfile, ptr[2]);
            putuchar(outfile, 0);
         }
	 
         break;

      default:  // CBYTE_ORDER_RGBA
         for (j=0; j<k; j++, ptr+=4) {
            putuchar(outfile, ptr[2]);
            putuchar(outfile, ptr[1]);
            putuchar(outfile, ptr[0]);
            putuchar(outfile, 0);
         }
	 
         break;
   }

   for (i=0, ptr=tob->data; i < (int)tob->maxy; i++, ptr+=tob->maxx) {
      fwrite(ptr, 1, tob->maxx, outfile);
      for (j=0; j<pad; j++)
         putuchar(outfile, 0);
   }
   
   fclose(outfile);
   return 1;
}


/* ***************************************************************
*************************************************************** */
void bmp::write_verbatim256(int count, unsigned char *tbuffer, unsigned char *buffer, int *size) {

   if (count == 1) {
      buffer[*size] = 1;
      buffer[*size+1] = tbuffer[0];
      *size += 2;
      return;
   }
      
   if (count == 2) {
      buffer[*size] = 1;
      buffer[*size+1] = tbuffer[0];
      buffer[*size+2] = 1;
      buffer[*size+3] = tbuffer[1];
      *size += 4;
      return;
   }
      
   buffer[*size] = 0;
   buffer[*size+1] = count;
   *size += 2;
	 
   memcpy(&buffer[*size], tbuffer, count);
   *size += count;
   
   // padd 2-byte count	 
   *size += count & 0x01;
}


/* ***************************************************************
*************************************************************** */
int bmp::write_compress256(char *fname, mapuc *tob, unsigned int *palette, int flip) {

   FILE *outfile;
   int  i, j, k;
   unsigned char *ptr;
   int count, size;
   unsigned char *buffer;
   unsigned int key;
   unsigned char tbuffer[256];
   int rle_count;
   
   if (!(outfile = fopen(fname,"wb")))
      return 0;

   buffer = new unsigned char[4*(tob->maxx+2)*tob->maxy + 4];
   size = 0;
   
   ptr = tob->data;
   
   for (i=0; i < (int)tob->maxy; i++) {
      j = 0;
      do {
         for (count = rle_count = 0; count != 255 && j != (int)tob->maxx && rle_count != 4; j++, count++, ptr++) {
            if (!count) {
               rle_count = 1;
               tbuffer[0] = *ptr;
               continue;
            }
	    
            tbuffer[count] = *ptr;

            if (tbuffer[count-1] == *ptr)
               rle_count++;
            else
               rle_count = 1;
         }

         if (rle_count != 4) 
            write_verbatim256(count, tbuffer, buffer, &size);
         else {
            if (count != 4)
               write_verbatim256(count-4, tbuffer, buffer, &size);

            key = tbuffer[count-1];
            count = 4;

            while (*ptr == key && j != (int)tob->maxx && count != 255) {
               count++;
               ptr++;
               j++;
            }

            buffer[size] = count;
            buffer[size+1] = key;
            size += 2;	    
         }
	 
      } while (j < (int)tob->maxx);
	    
      buffer[size] = 0;	    
      buffer[size+1] = 0x0;	    
      size += 2;
   }      
      
   buffer[size] = 0;	    
   buffer[size+1] = 0x01;	    
   size += 2;

   // calc palette size
   for (k=0, i=0, ptr = tob->data; i < (int)tob->maxy; i++)
      for (j=0; j < (int)tob->maxx; j++, ptr++)
         if (*ptr > k)
	    k = *ptr;
	    
   j = (k+1)<<2;	    
   k++;

   // write compression
   putuchar(outfile, 'B');
   putuchar(outfile, 'M');
   putruint(outfile, 14+40+j+size);
   putushort(outfile, 0);
   putushort(outfile, 0);
   putruint(outfile, 14+40+j);

   putruint(outfile, 40);
   putruint(outfile, tob->maxx);
   putruint(outfile, tob->maxy);
   putrushort(outfile, 1);
   putrushort(outfile, 8);
   putruint(outfile, BI_RLE8);
   putruint(outfile, size);
   putruint(outfile, 0);
   putruint(outfile, 0);
   putruint(outfile, k);
   putruint(outfile, k);

   ptr = (unsigned char *)palette;
       
   switch (flip) {
      case CBYTE_ORDER_ABGR:
         for (j=0; j<k; j++, ptr+=4) {
            putuchar(outfile, ptr[1]);
            putuchar(outfile, ptr[2]);
            putuchar(outfile, ptr[3]);
            putuchar(outfile, 0);
         }
	 
         break;

      case CBYTE_ORDER_ARGB:
         for (j=0; j<k; j++, ptr+=4) {
            putuchar(outfile, ptr[3]);
            putuchar(outfile, ptr[2]);
            putuchar(outfile, ptr[1]);
            putuchar(outfile, 0);
         }
	 
         break;

      case CBYTE_ORDER_BGRA:
         for (j=0; j<k; j++, ptr+=4) {
            putuchar(outfile, ptr[0]);
            putuchar(outfile, ptr[1]);
            putuchar(outfile, ptr[2]);
            putuchar(outfile, 0);
         }
	 
         break;

      default:  // CBYTE_ORDER_RGBA
         for (j=0; j<k; j++, ptr+=4) {
            putuchar(outfile, ptr[2]);
            putuchar(outfile, ptr[1]);
            putuchar(outfile, ptr[0]);
            putuchar(outfile, 0);
         }
	 
         break;
   }

   fwrite(buffer, 1, size, outfile);
      
   delete [] buffer;   
   fclose(outfile);
   return 1;
}


/* ***************************************************************
*************************************************************** */
int bmp::write_data(char *fname, mapul *tob, int flip) {

   FILE *outfile;
   int  i, j;
   unsigned int *ptr;
   int pad;
   char *buffer, *pbuffer;
   
   if (!(outfile = fopen(fname,"wb")))
      return 0;

   i = tob->maxx*3;

   pad = i & 0x03;

   if (pad)
      pad = 4 - pad;

   i = tob->maxy*(i+pad);
   pbuffer = buffer = new char[i];

   putuchar(outfile, 'B');
   putuchar(outfile, 'M');
   putruint(outfile, 14+40+i);
   putushort(outfile, 0);
   putushort(outfile, 0);
   putruint(outfile, 54);

   putruint(outfile, 40);
   putruint(outfile, tob->maxx);
   putruint(outfile, tob->maxy);
   putrushort(outfile, 1);
   putrushort(outfile, 24);
   putruint(outfile, BI_RGB);
   putruint(outfile, i);
   putruint(outfile, 0);
   putruint(outfile, 0);
   putruint(outfile, 0);
   putruint(outfile, 0);

   ptr = tob->data;
   switch (flip) {

      case CBYTE_ORDER_ABGR:
         for (i=0; i<(int)tob->maxy; i++) {
            for (j=0; j<(int)tob->maxx; j++, ptr++, pbuffer += 3) {
               pbuffer[0] = ((char *)ptr)[3];
               pbuffer[1] = ((char *)ptr)[2];
               pbuffer[2] = ((char *)ptr)[1];
            }

            for (j=0; j<pad; j++, pbuffer++)
               *pbuffer = 0;
         }

         break;

      case CBYTE_ORDER_ARGB:
         for (i=0; i<(int)tob->maxy; i++) {
            for (j=0; j<(int)tob->maxx; j++, ptr++, pbuffer += 3) {
               pbuffer[0] = ((char *)ptr)[1];
               pbuffer[1] = ((char *)ptr)[2];
               pbuffer[2] = ((char *)ptr)[3];
            }

            for (j=0; j<pad; j++, pbuffer++)
               *pbuffer = 0;
         }

         break;

      case CBYTE_ORDER_BGRA:
         for (i=0; i<(int)tob->maxy; i++) {
            for (j=0; j<(int)tob->maxx; j++, ptr++, pbuffer += 3) {
               pbuffer[0] = ((char *)ptr)[0];
               pbuffer[1] = ((char *)ptr)[1];
               pbuffer[2] = ((char *)ptr)[2];
            }

            for (j=0; j<pad; j++, pbuffer++)
               *pbuffer = 0;
         }

         break;

      default:  // CBYTE_ORDER_RGBA
         for (i=0; i<(int)tob->maxy; i++) {
            for (j=0; j<(int)tob->maxx; j++, ptr++, pbuffer += 3) {
               pbuffer[0] = ((char *)ptr)[2];
               pbuffer[1] = ((char *)ptr)[1];
               pbuffer[2] = ((char *)ptr)[0];
            }

            for (j=0; j<pad; j++, pbuffer++)
               *pbuffer = 0;
         }

         break;
   }
   
   fwrite(buffer, 1, tob->maxy*((tob->maxx*3)+pad), outfile);
   delete [] buffer;
   fclose(outfile);
   return 1;
}


/* ***************************************************************
*************************************************************** */
int bmp::scan_data(mapul *tob, int flip) {

   bmp_file_header file_header;
   bmp_bitmap_header bitmap_header;
   bmp_quad *quad = NULL;
   unsigned char *data, *pdata;
   unsigned int *ptr;
   int i, j, k, z;
   int bitcount;
   int bytepad;
   unsigned int temp;

   sinfile->sseek(0);

   // read file header
   file_header.type[0] = sinfile->scan_uchar();
   file_header.type[1] = sinfile->scan_uchar();

   if (file_header.type[0] != 'B' || file_header.type[1] != 'M')
      return 0;

   // read bitmap header
   file_header.size = sinfile->scan_ruint();
   sinfile->scan_rushort();
   sinfile->scan_rushort();
   file_header.offset = sinfile->scan_ruint();

   memset(&bitmap_header, 0, sizeof(bmp_bitmap_header));

   bitmap_header.header_size = sinfile->scan_ruint();

   if (bitmap_header.header_size == 40 || bitmap_header.header_size == 64 ) {
      bitmap_header.xsize = sinfile->scan_ruint();
      bitmap_header.ysize = sinfile->scan_ruint();
      bitmap_header.plane = sinfile->scan_rushort();
      bitmap_header.bpp   = sinfile->scan_rushort();
      bitmap_header.compress_flag = sinfile->scan_ruint();
      bitmap_header.data_size = sinfile->scan_ruint();
      bitmap_header.hppm = sinfile->scan_ruint();
      bitmap_header.vppm = sinfile->scan_ruint();
      bitmap_header.palette_size = sinfile->scan_ruint();
      bitmap_header.palette_isize = sinfile->scan_ruint();

      if (bitmap_header.header_size > 40)
         sinfile->skip(bitmap_header.header_size - 40);
   }

   else {
      bitmap_header.xsize = sinfile->scan_rushort();
      bitmap_header.ysize = sinfile->scan_rushort();
      bitmap_header.plane = sinfile->scan_rushort();
      bitmap_header.bpp   = sinfile->scan_rushort();
      bitmap_header.data_size = ((bitmap_header.plane*bitmap_header.bpp*bitmap_header.xsize+31)>>5)*bitmap_header.ysize<<2;
      bitmap_header.compress_flag = BI_RGB;
      bitmap_header.hppm = bitmap_header.vppm = bitmap_header.palette_size = bitmap_header.palette_isize = 0;
   }

   // read quads
   if (bitmap_header.palette_size)
      i = bitmap_header.palette_size;
   else if (bitmap_header.bpp == 24)
      i = 0;
   else
      i = 1 << bitmap_header.bpp;

   if (i) {
      quad = new bmp_quad[i];

      switch (flip) {
         case CBYTE_ORDER_ABGR:
            for (j=0; j<i; j++) {
               quad[j].g = sinfile->scan_uchar();
               quad[j].r = sinfile->scan_uchar();
               quad[j].a = sinfile->scan_uchar();
               sinfile->scan_uchar();
               quad[j].b = (quad[j].a | quad[j].r | quad[j].g) ? 255 : 0;
            }

            break;

         case CBYTE_ORDER_ARGB:
            for (j=0; j<i; j++) {
               quad[j].a = sinfile->scan_uchar();
               quad[j].r = sinfile->scan_uchar();
               quad[j].g = sinfile->scan_uchar();
               sinfile->scan_uchar();
               quad[j].b = (quad[j].a | quad[j].r | quad[j].g) ? 255 : 0;
            }

            break;

         case CBYTE_ORDER_BGRA:
            for (j=0; j<i; j++) {
               quad[j].b = sinfile->scan_uchar();
               quad[j].g = sinfile->scan_uchar();
               quad[j].r = sinfile->scan_uchar();
               sinfile->scan_uchar();
               quad[j].a = (quad[j].r | quad[j].g | quad[j].b) ? 255 : 0;
            }

            break;

         default:       // CBYTE_ORDER_RGBA
            for (j=0; j<i; j++) {
               quad[j].r = sinfile->scan_uchar();
               quad[j].g = sinfile->scan_uchar();
               quad[j].b = sinfile->scan_uchar();
               sinfile->scan_uchar();
               quad[j].a = (quad[j].r | quad[j].g | quad[j].b) ? 255 : 0;
            }

            break;
      }

   }

   sinfile->sseek(file_header.offset);
   data = sinfile->ptr;

   tob->init_map(bitmap_header.xsize, bitmap_header.ysize);

   pdata = data-1;				// simulates byte not read in yet...
   ptr = (unsigned int *)tob->data;

   switch (bitmap_header.bpp) {

      case 1:
         bytepad = (((bitmap_header.xsize + 31)>>5) << 2) - ((bitmap_header.xsize+7)>>3);		// pad lines to multiples of 32 bits

         for (i=0; i<bitmap_header.ysize; i++, pdata+=bytepad)
            for (j = 0, bitcount = 8; j < bitmap_header.xsize; j++) {
               if (bitcount == 8) {
                  bitcount = 0;
                  pdata++;
               }

               *ptr = *(unsigned int *)&quad[(*pdata & 0x80) ? 1 : 0];
               *pdata <<= 1;
               ptr++;
               bitcount++;
            }

         break;

      case 4:

         switch (bitmap_header.compress_flag) {

            case BI_RGB:

               bytepad = (((bitmap_header.xsize + 7)>>3) << 2) - ((bitmap_header.xsize+1)>>1);		// pad lines to multiples of 32 bits

               for (i=0; i<bitmap_header.ysize; i++, pdata+=bytepad)
                  for (j=0, bitcount=2; j<bitmap_header.xsize; j++) {
                     if (bitcount == 2) {
                        bitcount = 0;
                        pdata++;
                     }

                     *ptr = *(unsigned int *)&quad[(*pdata)>>4];
                     *pdata <<= 4;
                     ptr++;
                     bitcount++;
                  }

                  break;

            default:			// BI_RLE4
               tob->zero();
               i = k = 0;

               do {
                  pdata++;
                  if (*pdata) {		// encode mode
                     temp = *pdata;
                     pdata++;
                     for (j=0; j < (int)temp; j++, ptr++)
                        *ptr = *(unsigned int *)&quad[(j & 0x01) ? ((*pdata) & 0x0f) : (((*pdata)>>4) & 0x0f)];

                     i += temp;
                     k += temp;
                  }

                  else {
                     pdata++;
                     switch (*pdata) {
                        case 0x00:					// end of line segment
                           j = bitmap_header.xsize - k;
                           ptr += j;

                           i += j;
                           k = 0;

                           break;

                        case 0x01:					// end of pic
                           i = bitmap_header.xsize*bitmap_header.ysize;
                           break;

                        case 0x02:					// delta
                           pdata++;
                           j = *pdata;
                           k += j;

                           pdata++;
                           j += *pdata*bitmap_header.xsize;

                           ptr += j;
                           i += j;

                           break;

                        default:				// absolute mode
                           temp = *pdata;
                           for (j=0; j < (int)temp; j++, ptr++)
                              if (j & 0x01)
                                 *ptr = *(unsigned int *)&quad[*pdata & 0x0f];
                              else {
                                 pdata++;
                                 *ptr = *(unsigned int *)&quad[((*pdata)>>4) & 0x0f];
                              }

                           i += temp;
                           k += temp;

                           if ((temp & 0x03)==1 || (temp & 0x03)==2)		// read pad
                              pdata++;
                           break;
                     }

                  }

               } while (i < bitmap_header.xsize*bitmap_header.ysize);

         }

         break;

      case 8:
         switch (bitmap_header.compress_flag) {

            case BI_RGB:

               bytepad = ((bitmap_header.xsize + 3) & 0xfffffffc) - bitmap_header.xsize;		// pad lines to multiples of 32 bits

               for (i=0; i<bitmap_header.ysize; i++, pdata+=bytepad)
                  for (j=0; j<bitmap_header.xsize; j++) {
                     pdata++;
                     *ptr = *(unsigned int *)&quad[*pdata];
                     ptr++;
                  }

               break;

            default:		// BI_RLE8:

               i = k = 0;
               z = bitmap_header.xsize*bitmap_header.ysize;

               do {
                  pdata++;
                  if (*pdata) {         // encode mode
                     temp = *pdata;
                     pdata++;

                     for (j=0; j < (int)temp; j++, ptr++)
                        *ptr = *(unsigned int *)&quad[*pdata];

                     k += temp;
                     i += temp;
                  }

                  else {
                     pdata++;
                     switch (*pdata) {

                        case 0x00:                                      // end of line segment
                           j = bitmap_header.xsize - k;
                           i += j;
                           k = 0;

                           for (j--; j > -1; j--, ptr++)
                              *ptr = *(unsigned int *)&quad[0];

                           break;

                        case 0x01:                                      // end of pic
                           j = z - i;
                           i = z;

                           for (j--; j > -1; j--, ptr++)
                              *ptr = *(unsigned int *)&quad[0];
                           break;

                        case 0x02:                                      // delta
                           pdata++;
                           j = *pdata;
                           k += j;

                           pdata++;
                           j += *pdata*bitmap_header.xsize;
                           i += j;

                           for (j--; j > -1; j--, ptr++)
                              *ptr = *(unsigned int *)&quad[0];

                           break;

                        default:                                // absolute mode
                           temp = *pdata;
                           for (j=0; j < (int)temp; j++, ptr++) {
                              pdata++;
                              *ptr = *(unsigned int *)&quad[*pdata];
                           }
			   
                           i += temp;
                           k += temp;

                           if (temp & 0x01)            // read pad
                              pdata++;

                           break;
                     }

                  }

               } while (i < z);

               break;
         }

         break;

      default:		// 24

         bytepad = (4-((bitmap_header.xsize*3) & 0x03)) & 0x03;           // pad lines to multiples of 32 bits
         pdata++;

         switch (flip) {
            case CBYTE_ORDER_ABGR:
               for (i=0; i<bitmap_header.ysize; i++, pdata+=bytepad)
                  for (j=0; j<bitmap_header.xsize; j++, pdata+=3, ptr++) {
                     ((unsigned char *)ptr)[3] = pdata[2];
                     ((unsigned char *)ptr)[2] = pdata[1];
                     ((unsigned char *)ptr)[1] = pdata[0];
                     ((unsigned char *)ptr)[0] = (pdata[0] | pdata[1] | pdata[2]) ? 255 : 0;
                  }

               break;

            case CBYTE_ORDER_ARGB:
               for (i=0; i<bitmap_header.ysize; i++, pdata+=bytepad)
                  for (j=0; j<bitmap_header.xsize; j++, pdata+=3, ptr++) {
                     ((unsigned char *)ptr)[3] = pdata[0];
                     ((unsigned char *)ptr)[2] = pdata[1];
                     ((unsigned char *)ptr)[1] = pdata[2];
                     ((unsigned char *)ptr)[0] = (pdata[0] | pdata[1] | pdata[2]) ? 255 : 0;
                  }

               break;

            case CBYTE_ORDER_BGRA:
               for (i=0; i<bitmap_header.ysize; i++, pdata+=bytepad)
                  for (j=0; j<bitmap_header.xsize; j++, pdata+=3, ptr++) {
                     ((unsigned char *)ptr)[0] = pdata[0];
                     ((unsigned char *)ptr)[1] = pdata[1];
                     ((unsigned char *)ptr)[2] = pdata[2];
                     ((unsigned char *)ptr)[3] = (pdata[0] | pdata[1] | pdata[2]) ? 255 : 0;
                  }

               break;

            default:      // CBYTE_ORDER_RGBA
               for (i=0; i<bitmap_header.ysize; i++, pdata+=bytepad)
                  for (j=0; j<bitmap_header.xsize; j++, pdata+=3, ptr++) {
                     ((unsigned char *)ptr)[0] = pdata[2];
                     ((unsigned char *)ptr)[1] = pdata[1];
                     ((unsigned char *)ptr)[2] = pdata[0];
                     ((unsigned char *)ptr)[3] = (pdata[0] | pdata[1] | pdata[2]) ? 255 : 0;
                  }

               break;
         }

         break;
   }

   if (quad)
      delete [] quad;

   return 1;
}


/* ***************************************************************
*************************************************************** */
int bmp::skim_data(int *maxx, int *maxy, int flip) {

   bmp_file_header file_header;
   bmp_bitmap_header bitmap_header;

   sinfile->sseek(0);

   // read file header
   file_header.type[0] = sinfile->scan_uchar();
   file_header.type[1] = sinfile->scan_uchar();

   if (file_header.type[0] != 'B' || file_header.type[1] != 'M')
      return 0;

   // read bitmap header
   file_header.size = sinfile->scan_ruint();
   sinfile->skip(4);
   file_header.offset = sinfile->scan_ruint();

   bitmap_header.header_size = sinfile->scan_ruint();

   if (bitmap_header.header_size == 40 || bitmap_header.header_size == 64 ) {
      *maxx = sinfile->scan_ruint();
      *maxy = sinfile->scan_ruint();
   }

   else {
      *maxx = sinfile->scan_rushort();
      *maxy = sinfile->scan_rushort();
   }

   return 1;
}

