

/* *************************************************************
   this file contains all the routines for the zbuffer class
************************************************************* */


#include <string.h>

#include "zbuffer.h"

#include "global.h"


#define BYTEORDER  CBYTE_ORDER_RGBA
#define MAXFILTER 7

static float  default_filter[MAXFILTER*MAXFILTER] = {
         0.001f, 0.004f, 0.008f, 0.010f, 0.008f, 0.004f, 0.001f,
         0.004f, 0.012f, 0.025f, 0.029f, 0.025f, 0.012f, 0.004f,
         0.008f, 0.025f, 0.049f, 0.058f, 0.049f, 0.025f, 0.008f,
         0.010f, 0.029f, 0.058f, 0.068f, 0.058f, 0.029f, 0.010f,
         0.008f, 0.025f, 0.049f, 0.058f, 0.049f, 0.025f, 0.008f,
         0.004f, 0.012f, 0.025f, 0.029f, 0.025f, 0.012f, 0.004f,
         0.001f, 0.004f, 0.008f, 0.010f, 0.008f, 0.004f, 0.001f,
      };


/* *************************************************************
************************************************************* */
void init_buff(unsigned int *buffer, unsigned int padd, unsigned int count) {

   int i = count & 0x0f;
   
   switch (i) {
      case 0x0f:
         buffer[0x0f] = padd;
      case 0x0e:
         buffer[0x0e] = padd;
      case 0x0d:
         buffer[0x0d] = padd;
      case 0x0c:
         buffer[0x0c] = padd;
      case 0x0b:
         buffer[0x0b] = padd;
      case 0x0a:
         buffer[0x0a] = padd;
      case 0x09:
         buffer[0x09] = padd;
      case 0x08:
         buffer[0x08] = padd;
      case 0x07:
         buffer[0x07] = padd;
      case 0x06:
         buffer[0x06] = padd;
      case 0x05:
         buffer[0x05] = padd;
      case 0x04:
         buffer[0x04] = padd;
      case 0x03:
         buffer[0x03] = padd;
      case 0x02:
         buffer[0x02] = padd;
      case 0x01:
         buffer[0x01] = padd;
      default:
         buffer += i;
	 count -= i;
         break;
   }
      
   if (!count)
      return;
      
#ifdef WIN32

   unsigned int dptr[2];

   dptr[0] = dptr[1] = padd;

   _asm {
      mov eax, padd
      mov ecx, buffer

      fld qword ptr [dptr]
      mov edi, count

loop1:
      fst qword ptr [ecx + edi*4 - 8]
      mov [ecx + edi*4 - 12], eax
      mov [ecx + edi*4 - 16], eax
      mov [ecx + edi*4 - 20], eax
      mov [ecx + edi*4 - 24], eax

      fst qword ptr [ecx + edi*4 - 32]
      mov [ecx + edi*4 - 36], eax
      mov [ecx + edi*4 - 40], eax
      mov [ecx + edi*4 - 44], eax
      mov [ecx + edi*4 - 48], eax

      fst qword ptr [ecx + edi*4 - 56]
      mov [ecx + edi*4 - 60], eax
      mov [ecx + edi*4 - 64], eax

      sub edi, 16
      jnz loop1

      fstp dptr
   }

#else

   do {
      buffer[count-1] = padd;
      buffer[count-2] = padd;

      buffer[count-3] = padd;
      buffer[count-4] = padd;

      buffer[count-5] = padd;
      buffer[count-6] = padd;

      buffer[count-7] = padd;
      buffer[count-8] = padd;

      buffer[count-9] = padd;
      buffer[count-10] = padd;

      buffer[count-11] = padd;
      buffer[count-12] = padd;

      buffer[count-13] = padd;
      buffer[count-14] = padd;

      buffer[count-15] = padd;
      buffer[count-16] = padd;

      count -= 16;
   } while (count);

#endif

}


/* *************************************************************
   this function reads in an anti-aliasing filter
************************************************************* */
int zbuffer::read_filter(char *fname, mapf *filter) {

   FILE   *infile;
   int    i, j;

   if (!find_file(fname, "r", FILTER_PATH.string, (char)PLATFORM_SLASH, NULL, &infile)) {
      pprintf("Invalid Filter... using default\n");
      return 0;
   }

   fscanf(infile, "%d", &j);

   if (!j || !(j & 0x01)) {
      pprintf("Invalid Filter... using default\n");
      return 0;
   }

   filter->init_map(j, j);

   j *= j;

   for (i=0; i<j; i++)
      fscanf(infile, "%f", &filter->data[i]);

   fclose(infile);
   return 1;
}


/* *************************************************************
************************************************************* */
void zbuffer::save_image(char *filename, image_coder *coder) {

   coder->write_compress(filename, this, BYTEORDER);
}


/* **************************************************
   this function anti-aliases the zbuffer
************************************************** */
void zbuffer::antialias(char *fname) {

   mapf      filter;
   int       defflag = 0;
   int       current;                           // center of filter
   int       h, i, j, k, l, m, n, o;                  // looping variables
   float     temp[3];                           // current pixel value
   int       cmaxx, cmaxy;

   unsigned char *tbuff;
   unsigned char **ptbuff;
   unsigned char *ctemp;

   unsigned int index, end;
   vector4uc *cdata = (vector4uc *)data;

   end = maxy*maxx;

   if (strlen(fname))                       // read filter from file
      defflag = read_filter(fname, &filter);

   if (!defflag) {
      filter.init_map(MAXFILTER, MAXFILTER);

      for (index=0, end = MAXFILTER*MAXFILTER; index<end; index++)
         filter.data[index] = default_filter[index];
   }

   k = filter.maxy*(h=maxx<<2);
   tbuff = new unsigned char[k];
   ptbuff = new puchar[filter.maxy];

   for (i=j=0; i<(int)filter.maxy; i++, j+=h)
      ptbuff[i] = &tbuff[j];

   memcpy(ptbuff[1], data, j-h);

   current = (filter.maxx-1) >> 1;
   cmaxx = maxx - current;
   cmaxy = maxy - current;

   for (i=current*maxx, o=cmaxy*maxx, n=(filter.maxy-1)*maxx; i<o; i += maxx, n+=maxx) {

      ctemp = ptbuff[0];
      for (k=0, l=filter.maxy-1; k<l; k++)
         ptbuff[k] = ptbuff[k+1];
      ptbuff[k] = ctemp;
      memcpy(ctemp, &data[n], h);

      for (j=current, m=0; j<cmaxx; j++, m+=4) {
         temp[0] = temp[1] = temp[2] = 0;

         for (k=0; k<(int)filter.maxy; k++)
            for (l=0, index=m; l<(int)filter.maxx; l++, index+=4) {
               temp[0] += filter.pdata[k][l] * ptbuff[k][index+0];
               temp[1] += filter.pdata[k][l] * ptbuff[k][index+1];
               temp[2] += filter.pdata[k][l] * ptbuff[k][index+2];
            }

         cdata[i+j][0] = pbyte_lut[(int)temp[0]];
         cdata[i+j][1] = pbyte_lut[(int)temp[1]];
         cdata[i+j][2] = pbyte_lut[(int)temp[2]];
      }

   }

   delete [] ptbuff;
   delete [] tbuff;
}


/* **************************************************
   this function anti-aliases the zbuffer
************************************************** */
void zbuffer::antialias_edge(char *fname) {

   mapf      filter;
   int       defflag = 0;
   int       current;                           // center of filter
   int       h, i, j, k, l, m, n, o, p;                  // looping variables
   float     temp;                           // current pixel value
   int       cmaxx, cmaxy;

   unsigned char *tbuff;
   unsigned char **ptbuff;
   unsigned char *ctemp;

   unsigned int index, end;
   vector4uc *cdata = (vector4uc *)data;

   end = maxy*maxx;

   if (strlen(fname))                       // read filter from file
      defflag = read_filter(fname, &filter);

   if (!defflag) {
      filter.init_map(MAXFILTER, MAXFILTER);

      for (index=0, end = MAXFILTER*MAXFILTER; index<end; index++)
         filter.data[index] = default_filter[index];
   }

   k = filter.maxy*(h=maxx<<2);
   tbuff = new unsigned char[k];
   ptbuff = new puchar[filter.maxy];

   for (i=j=0; i<(int)filter.maxy; i++, j+=h)
      ptbuff[i] = &tbuff[j];

   memcpy(ptbuff[1], data, j-h);

   current = (filter.maxx-1) >> 1;
   cmaxx = maxx - current;
   cmaxy = maxy - current;

   for (i=current*maxx, o=cmaxy*maxx, n=(filter.maxy-1)*maxx; i<o; i += maxx, n+=maxx) {

      ctemp = ptbuff[0];
      for (k=0, l=filter.maxy-1; k<l; k++)
         ptbuff[k] = ptbuff[k+1];
      ptbuff[k] = ctemp;
      memcpy(ctemp, &data[n], h);

      for (j=current, m=0; j<cmaxx; j++, m+=4) {

         for (p = 0; p<3; p++) {
            temp = (float)(-ptbuff[current-1][m+p+4] - ptbuff[current+1][m+p+4]
                   -ptbuff[current][m+p] + 4*ptbuff[current][m+p+4] - ptbuff[current][m+p+8]);
            if (temp > 0.0) {
               temp = 0.0;

               for (k=0; k<(int)filter.maxy; k++)
                  for (l=0, index=m; l<(int)filter.maxx; l++, index+=4)
                     temp += filter.pdata[k][l] * ptbuff[k][index+p];

               cdata[i+j][p] = pbyte_lut[(int)temp];
            }

         }

      }

   }

   delete [] ptbuff;
   delete [] tbuff;
}


/* **************************************************
   this function initializes the buffer
************************************************** */
void zbuffer::gammabuff(unsigned char *pgamma) {

   vector4uc *cdata = (vector4uc *)data;
   int end = maxx*maxy;
   int index;

   for (index=0; index < end; index++) {
      cdata[index][0] = pgamma[cdata[index][0]];
      cdata[index][1] = pgamma[cdata[index][1]];
      cdata[index][2] = pgamma[cdata[index][2]];
   }

}


/* **************************************************
   this function initializes the buffer
************************************************** */
int zbuffer::initbuff(unsigned int x, unsigned int y, float backplane) {

   unsigned int i;

   init_map(x, y);

   i  = x*y;

   if (zdata)
      delete [] zdata;

   zdata = new float[i];

   if (adata)
      delete [] adata;

   adata = new unsigned int[i];

   while (i) {
      i -= maxx;
      init_buff((unsigned int *)(zdata+i), *(unsigned int *)&backplane, maxx);
      init_buff(adata+i, 0, maxx);
      init_buff(data+i, 0, maxx);
   }

   return 1;
}


/* **************************************************
   this function initializes the buffer
************************************************** */
int zbuffer::initbuff(unsigned int *d, puint *pd, unsigned int *ad, float *zd, unsigned int x, unsigned int y, float backplane) {

   unsigned int i;

   if (data) {
      delete [] data;
      delete [] pdata;
   }

   data = d;
   pdata = pd;

   maxx = x;
   maxy = y;

   i = x*y;

   if (zdata)
      delete [] zdata;

   zdata = zd;

   if (adata)
      delete [] adata;

   adata = ad;

   while (i) {
      i -= maxx;
      init_buff((unsigned int *)(zdata+i), *(unsigned int *)&backplane, maxx);
      init_buff(adata+i, 0, maxx);
      init_buff(data+i, 0, maxx);
   }

   return 1;
}


/* **************************************************
   this function initializes the buffer
************************************************** */
int zbuffer::initbuff(unsigned int *d, puint *pd, float *zd, unsigned int x, unsigned int y, float backplane) {

   unsigned int i;

   if (data) {
      delete [] data;
      delete [] pdata;
   }

   data = d;
   pdata = pd;

   maxx = x;
   maxy = y;

   i = x*y;

   if (zdata)
      delete [] zdata;

   zdata = zd;

   while (i) {
      i -= maxx;
      init_buff((unsigned int *)(zdata+i), *(unsigned int *)&backplane, maxx);
      init_buff(data+i, 0, maxx);
   }

   return 1;
}


/* *************************************************************
   this function puts an XImage on the screen
************************************************************* */
void zbuffer::render(mapul *mcanvas) {

   memcpy(mcanvas->data, data, maxx*maxy<<2);
}
