

/* *************************************************************
   This file scans surface patch data into the buffer
************************************************************* */


#include <string.h>

#include "spatch.h"


/* *************************************************************
************************************************************* */
int spatch::bound_box(eye *parm) {

   int i, j;
   
   copyarray3(bbox[0], ob->data[0]);
   copyarray3(bbox[1], ob->data[0]);

   for (i=0; i<ob->vsize; i++)
      for (j=0; j<3; j++)
         if (bbox[0][j] > ob->data[i][j])
            bbox[0][j] = ob->data[i][j];
         else if (bbox[1][j] < ob->data[i][j])
            bbox[1][j] = ob->data[i][j];

   return bbox[0][2] <= front;
}


/* *************************************************************
************************************************************* */
int spatch::clip(eye *parm, int maxx, int maxy) {

   if (bbox[0][2] > front ||
        parm->zscale(bbox[0][0], bbox[0][2]) > parm->vrc[1] ||     // outside boundaries
        parm->zscale(bbox[1][0], bbox[0][2]) < parm->vrc[0] ||
        parm->zscale(bbox[0][1], bbox[0][2]) > parm->vrc[3] ||
        parm->zscale(bbox[1][1], bbox[0][2]) < parm->vrc[2])
      return 0;

   return 1;
}


/* *************************************************************
   This function converts surface patch data into polygonal data
************************************************************* */
int spatch::spline2poly(spatchtype *sob, texpolygon *stob, int xflag) {

   int    i, j, k, l, m;
   int    ptr[4];
   float  d[3];
   float  temp;
   int    rflag;                        // variable to keep track of splitting of patch
   pvector4f vtr[4];
   int    patch;
   vector2f uv[4];
   polygon_object_type *pot;
   shade_block *sblock;
   texface *uvblock, *tuvblock;
   int bflag = 0;
   
   pot = (polygon_object_type *)pob->query_data();
   if (xflag) {
      uvblock = (texface *)stob->query_data();
      tuvblock = (texface *)tpob->query_data();
   }
   
   pob->countobject = pob->countvertex = pob->countedge = 0;

   for (l=0, i=1; i<=sob->srow; i++)
      for (j=1, k=0; j<=sob->scol; k=j, j++, l++) {
         copyarray4(pot->vlist[l], sob->mvertex[i][j]);

         if (!bflag) {
            bflag = 1;
            copyarray3(bbox[0], pot->vlist[l]);
            copyarray3(bbox[1], pot->vlist[l]);
         }

         else
            for (m=0; m<3; m++)
               if (bbox[0][m] > pot->vlist[l][m])
                  bbox[0][m] = pot->vlist[l][m];
               else if (bbox[1][m] < pot->vlist[l][m])
                  bbox[1][m] = pot->vlist[l][m];
      }		  

   if (bbox[0][2] > front)
      return 0;

   if (mctype.model > PWFBW || (mctype.model > WFBW && mctype.model <= PHONG))
      for (l=0, i=1; i<=sob->srow; i++)
         for (j=1, k=0; j<=sob->scol; k=j, j++, l++)
            if (mctype.model > PWFBW) {
               copyarray3(pot->nlist[l], sob->gnormal[i][j]);
            }

            else {
               vtr[0] = &ob->mvertex[i-1][k];
               vtr[1] = &ob->mvertex[i][k];
               vtr[2] = &ob->mvertex[i+1][k];
               vtr[3] = &ob->mvertex[i+2][k];

               spnormal_base(vtr, (float *)pot->nlist[l], M_CR);
            }

   pob->countvertex = l;

   if (mctype.model > PBW || (mctype.model > BW && mctype.model <= PHONG))
      sblock = (shade_block *)plob->query_data();

   for (i=patch=0, m=0; i<sob->prow; i++, m++)
      for (j=0; j<sob->pcol; j++, m++, patch++) {
         ptr[0] = m;
         ptr[1] = ptr[0] + 1;
         ptr[3] = ptr[0] + sob->scol;
         ptr[2] = ptr[3] + 1;

         k = 0;
         for (l=0; l<4; l++)
            if (!k || !similar3(pot->vlist[pot->vindex[pob->countedge+k-1]], pot->vlist[ptr[l]])) {
               pot->vindex[pob->countedge+k] = ptr[l];
               k++;
            }

         if (k == 3) {
            if (mctype.model > PBW || (mctype.model > BW && mctype.model <= PHONG))
               sblock->facelist[pob->countobject] = sob->mshade[i][j];
	    
            pot->flist[pob->countobject].edgeptr = &pot->vindex[pob->countedge];

            if (xflag && stob->query_tflag(patch)) {

               pot->flist[pob->countobject].polynum = 4;

               pot->vindex[pob->countedge] = ptr[0];
               pob->countedge++;
               pot->vindex[pob->countedge] = ptr[1];
               pob->countedge++;
               pot->vindex[pob->countedge] = ptr[2];
               pob->countedge++;
               pot->vindex[pob->countedge] = ptr[3];
               pob->countedge++;

               uv[3][0] = uv[0][0] = uvblock[patch].uv[0][0];
               uv[1][1] = uv[0][1] = uvblock[patch].uv[0][1];
               uv[2][0] = uv[1][0] = uvblock[patch].uv[1][0];
               uv[3][1] = uv[2][1] = uvblock[patch].uv[1][1];

               tpob->direct2Dmap(pob->countobject, uv, uvblock[patch].ob, 4, 1);
            }

            else {
               pob->countedge += 3;
               pot->flist[pob->countobject].polynum = 3;
            }

            pob->countobject++;
         }

         else
            if (k == 4) {
               subeqarray3(d, pot->vlist[ptr[0]], pot->vlist[ptr[2]]);
               temp = dotproduct3(d, d);

               subeqarray3(d, pot->vlist[ptr[1]], pot->vlist[ptr[3]]);

               if ((rflag = (dotproduct3(d, d) < temp))) {
                  k = ptr[0];
                  ptr[0] = ptr[1];
                  ptr[1] = ptr[2];
                  ptr[2] = ptr[3];
                  ptr[3] = k;
               }

               pot->flist[pob->countobject].edgeptr = &pot->vindex[pob->countedge];

               if (mctype.model > PBW || (mctype.model > BW && mctype.model <= PHONG)) {
                  sblock->facelist[pob->countobject]   = sob->mshade[i][j];
                  sblock->facelist[pob->countobject+1] = sob->mshade[i][j];
               }

               pot->vindex[pob->countedge] = ptr[0];
               pob->countedge++;
               pot->vindex[pob->countedge] = ptr[1];
               pob->countedge++;
               pot->vindex[pob->countedge] = ptr[2];
               pob->countedge++;

               if (xflag && stob->query_tflag(patch)) {

                  if (rflag) {
                     uv[2][0] = uvblock[patch].uv[0][0];
                     uv[0][1] = uvblock[patch].uv[0][1];
                     uv[1][0] = uv[0][0] = uvblock[patch].uv[1][0];
                     uv[2][1] = uv[1][1] = uvblock[patch].uv[1][1];
                  }

                  else {
                     uv[0][0] = uvblock[patch].uv[0][0];
                     uv[1][1] = uv[0][1] = uvblock[patch].uv[0][1];
                     uv[2][0] = uv[1][0] = uvblock[patch].uv[1][0];
                     uv[2][1] = uvblock[patch].uv[1][1];
                  }

                  tpob->direct2Dmap(pob->countobject, uv, uvblock[patch].ob, 3, 1);
               }

               pot->flist[pob->countobject].polynum = 3;
               pob->countobject++;

               pot->flist[pob->countobject].edgeptr = &pot->vindex[pob->countedge];

               pot->vindex[pob->countedge] = ptr[2];
               pob->countedge++;
               pot->vindex[pob->countedge] = ptr[3];
               pob->countedge++;
               pot->vindex[pob->countedge] = ptr[0];
               pob->countedge++;

               if (xflag && stob->query_tflag(patch)) {
                  tuvblock[pob->countobject].setup(3);
                  tuvblock[pob->countobject].ob = uvblock[patch].ob;

                  if (rflag) {
                     uv[1][0] = uv[0][0] = uvblock[patch].uv[0][0];
                     uv[2][1] = uv[1][1] = uvblock[patch].uv[0][1];
                     uv[0][1] = uvblock[patch].uv[1][1];
                     uv[2][0] = uvblock[patch].uv[1][0];
                  }

                  else {
                     uv[2][0] = uv[1][0] = uvblock[patch].uv[0][0];
                     uv[1][1] = uv[0][1] = uvblock[patch].uv[1][1];
                     uv[0][0] = uvblock[patch].uv[1][0];
                     uv[2][1] = uvblock[patch].uv[0][1];
                  }

                  tpob->direct2Dmap(pob->countobject, uv, uvblock[patch].ob, 3, 1);
               }

               pot->flist[pob->countobject].polynum = 3;
               pob->countobject++;
            }

      }

   if (xflag)
      tpob->countobject = pob->countobject;

   return 1;
}


/* **************************************************
   This procedure scans 2D grids into surface patches
   or polygonal structures.
************************************************** */
int spatch::convert(int i, int j, spatchtype *sob, texpolygon *stob, shaderlisttype *ssob) {

   int h, k, l, m, n;
   float u, v, t;
   pvector4f ptr[4];
   float SPLQ[4][4], SPLQ2[4][4];
   float SPPQ[4][4][4];
   float dt[2], start[2];
   float pt[2][2];
   float delta = 1.0f/SPDELTA;
   int patch = i * ob->pcol + j;
   int subpatch;
   texface *uvblock, *tuvblock;
   
   ptr[0] = ob->mvertex[i]   + j;
   ptr[1] = ob->mvertex[i+1] + j;
   ptr[2] = ob->mvertex[i+2] + j;
   ptr[3] = ob->mvertex[i+3] + j;

   for (n=0; n<4; n++)
      for (m=0; m<4; m++)
         for (h=0; h<4; h++)
              SPPQ[n][m][h] = ptr[m][h][n];

                                                // top row
   for (n=0; n<4; n++)
      for (h=0; h<4; h++)
         SPLQ[n][h] = ptr[1][h][n];

   k=i+1;
   l=j+1;
   copyarray4(sob->mvertex[1][1], ob->mvertex[k][l]);

   if (mctype.model > PWFBW)
      spnormal(0, 0, SPPQ, (float *)&sob->gnormal[1][1], M_CR);

   for (n=1, m=2, t=delta; n<SPDELTA; n++, m++, t+=delta) {
      spline(t, SPLQ, (float *)&sob->mvertex[1][m], M_CR);
      sob->mvertex[1][m][3] = 1;

      if (mctype.model > PWFBW)
         spnormal(0, t, SPPQ, (float *)&sob->gnormal[1][m], M_CR);
   }

   l++;
   copyarray4(sob->mvertex[1][m], ob->mvertex[k][l]);
   if (mctype.model > PWFBW)
      spnormal(0, 1, SPPQ, (float *)&sob->gnormal[1][m], M_CR);

                                                // last row
   for (n=0; n<4; n++)
      for (h=0; h<4; h++)
         SPLQ[n][h] = ptr[2][h][n];

   k=i+2;
   l=j+1;
   h = 1+SPDELTA;
   copyarray4(sob->mvertex[h][1], ob->mvertex[k][l]);

   if (mctype.model > PWFBW)
      spnormal(1, 0, SPPQ, (float *)&sob->gnormal[h][1], M_CR);

   for (n=1, m=2, t=delta; n<SPDELTA; n++, m++, t+=delta) {
      spline(t, SPLQ, (float *)&sob->mvertex[h][m], M_CR);
      sob->mvertex[1][m][3] = 1;

      if (mctype.model > PWFBW)
         spnormal(1, t, SPPQ, (float *)&sob->gnormal[h][m], M_CR);
   }

   l++;
   copyarray4(sob->mvertex[h][m], ob->mvertex[k][l]);
   if (mctype.model > PWFBW)
      spnormal(1, 1, SPPQ, (float *)&sob->gnormal[h][m], M_CR);

                                        // rest
   for (n=0; n<4; n++)
      for (h=0; h<4; h++)
         SPLQ[n][h] = ptr[h][1][n];

   for (n=0; n<4; n++)
      for (h=0; h<4; h++)
         SPLQ2[n][h] = ptr[h][2][n];

   for (h=2, k=1, u=delta; k<SPDELTA; k++, u+=delta, h++) {
      spline(u, SPLQ, (float *)&sob->mvertex[h][1], M_CR);
      sob->mvertex[h][1][3] = 1;

      if (mctype.model > PWFBW)
         spnormal(u, 0, SPPQ, (float *)&sob->gnormal[h][1], M_CR);

      for (l=2,n=1, v=delta; n<SPDELTA; n++, v+=delta, l++) {
         sppatch(u, v, SPPQ, (float *)&sob->mvertex[h][l], M_CR);
         sob->mvertex[h][l][3] = 1;

         if (mctype.model > PWFBW)
            spnormal(u, v, SPPQ, (float *)&sob->gnormal[h][l], M_CR);
      }

      spline(u, SPLQ2, (float *)&sob->mvertex[h][l], M_CR);
      sob->mvertex[h][l][3] = 1;

      if (mctype.model > PWFBW)
         spnormal(u, 1, SPPQ, (float *)&sob->gnormal[h][l], M_CR);
   }

   if (mctype.model > PBW) {

      for (k=0; k<SPDELTA; k++)
         for (l=0; l<SPDELTA; l++)
            sob->mshade[k][l] = ob->mshade[i][j];

      if ((mcinfo.info & CITEXTURE) && tob->query_tflag(patch)) {
         uvblock = (texface *)tob->query_data();      
         tuvblock = (texface *)stob->query_data();      
         copyarray2(start, uvblock[patch].uv[0]);
         dt[0] = (uvblock[patch].uv[1][0]-start[0])*delta;
         dt[1] = (uvblock[patch].uv[1][1]-start[1])*delta;

         for (subpatch = k=0, pt[0][1]=start[1], pt[1][1]=start[1]+dt[1]; k<SPDELTA; k++, pt[0][1]=pt[1][1], pt[1][1]+=dt[1])
            for (l=0, pt[0][0]=start[0], pt[1][0]=start[0]+dt[0]; l<SPDELTA; l++, subpatch++, pt[0][0]=pt[1][0], pt[1][0]+=dt[0]) {
               tuvblock[subpatch].setup(SPDELTA);
               tuvblock[subpatch].ob = uvblock[patch].ob;
               copyarray2(tuvblock[subpatch].uv[0], pt[0]);
               copyarray2(tuvblock[subpatch].uv[1], pt[1]);
            }

      }

      if (mcinfo.info & CI3DTEX) {
         m = sob->psize<<1;

         for (k=0; k<m; k++) {
            ssob->shade[k].scount = shaderlist->shade[patch].scount;
            ssob->shade[k].s = shaderlist->shade[patch].s;
            ssob->shade[k].stype = shaderlist->shade[patch].stype;
         }

      }

   }

   return spline2poly(sob, stob, (mcinfo.info & CITEXTURE));
}


/* *************************************************************
   This procedure puts the object in the edge table
************************************************************* */
void spatch::transform(eye *parm) {

   int   i;                          // looping vars

   matmatmulto(parm->transform, rotate, world);

   world[3][0] = world[3][1] = world[3][2] = 0;
   world[3][3] = 1;
   
   inversemx(world, iworld);

   for (i=0; i<ob->vsize; i++)
      matvecmulto(world, ob->data[i]);
}


/* *************************************************************
************************************************************* */
void spatch::end_scan() {

   if (ob)
      delete ob;

   if (tpob)
      delete tpob;

   if (slist.shade) {
      delete [] slist.shade;
      slist.shade  = NULL;
   }

   if (plob) {
      delete plob;
      plob = NULL;
   }
   
   polyob.end_scan();
}


/* *************************************************************
   This function copies the basic object data into a composite
   structure for manipulation
************************************************************* */
void spatch::datacopy() {

   int i;                          // looping variables
   shade_block *sblock;
   
   mctype.datacopy();
   mcinfo.datacopy();

   i = (mctype.model > BW && mctype.model <= PHONG) || mctype.model > PBW;

   ob->build(dob->vrow, dob->vcol, 1, i);

   memcpy(ob->data, dob->data, ob->vsize*sizeof(vector4f));

   if (i) {
      base_color = &lob->base;
      sblock = (shade_block *)lob->query_data();
      
      for (i=0; i<ob->psize; i++)
         ob->shade[i] = sblock->facelist[i];
   }

   pob = NULL;
   plob = NULL;
}


/* **************************************************
   This procedure scans 2D grids into surface patches
   or polygonal structures.
************************************************** */
int spatch::scan(camera *cparm, light *lmain, engine *proc) {

   int       i, l;
   int       texflag = mcinfo.info & CITEXTURE;
   int       flag3d = mcinfo.info & CI3DTEX;
   shadertype *str;

   renderflag = 0;

   transform(cparm);

   if (!ob->psize || !bound_box(cparm))
      return 0;

   if (!clip(cparm, proc->zbuff.maxx, proc->zbuff.maxy))
      return 0;

   if (mctype.model <= PHONG) {                // display data as polygons
      setup_transfer(ob->srow*ob->scol, ob->psize);

      if (!spline2poly(ob, tob, texflag))
         return 0;

      polyob.setup_patch(pob, plob, mctype.model, mcinfo.info, id, sflag, splane, world, iworld, bbox, base_color);

      if (texflag)
         polyob.setup_texture(tpob);

      if (flag3d) {
         i = ob->psize<<1;
         slist.scount = i;
         slist.scale = shaderlist->scale;

         slist.shade = new shadertype[i];

         for (i=0, str = slist.shade; i<ob->psize; i++, str += 2)
           *(str+1) =  *str = shaderlist->shade[i];

         polyob.setup_texture3d(&slist);
      }

      renderflag = SPATPOLY;
      i = polyob.scan(cparm, lmain, proc);
      shadptr = polyob.shadptr;
      polyob.shadptr = (pc *)NULL;
      return i;
   }

   if (sflag && lmain && !NOSHADOW && !(mcinfo.info & CITRANSPARENT))
      shadptr = new basenull;

   cptr = cparm;
   lptr = lmain;

   ptr.build(SPDELTA + 3, SPDELTA + 3, mctype.model > PWFBW, mctype.model > PBW);
   setup_transfer(ptr.vsize, ptr.psize);

   if (texflag) {
      ttr.setup(ptr.psize);
      polyob.setup_texture(tpob);
   }

   if (flag3d) {
      l = ptr.psize<<1;

      slist.scount = 0;
      slist.shade = new shadertype[l];

      polyob.setup_texture3d(&slist);
   }

   renderflag = SPATREND;
   return 1;
}


/* **************************************************
************************************************** */
void spatch::render(camera *cparm, light *lmain, light *spot, engine *proc) {

   int     i, j, k, l;
   int     texflag = mcinfo.info & CITEXTURE;
   int     flag3d = mcinfo.info & CI3DTEX;
   texface *uvblock;
   
   switch (renderflag) {

      case SPATPOLY:
         polyob.render(cparm, lmain, spot, proc);
         return;

      case SPATREND:

         l = ptr.psize<<1;

         for (i=0; i<ob->prow; i++)
            for (j=0; j<ob->pcol; j++) {
               if (texflag) {
                  if (ttr.query_tflag(0)) {
                     uvblock = (texface *)ttr.query_data();
                     for (k=0; k<ttr.countobject; k++) {
                        delete [] uvblock[k].uv;
                        uvblock[k].uv = NULL;
                        uvblock[k].count = 0;
                     }

                  }

                  if (tpob->query_tflag(0)) {
                     uvblock = (texface *)tpob->query_data();
                     for (k=0; k<tpob->countobject; k++) {
                        delete [] uvblock[k].uv;
                        uvblock[k].uv = NULL;
                        uvblock[k].count = 0;
                     }

                  }

               }

               if (!convert(i, j, &ptr, &ttr, &slist))
                  continue;

               polyob.setup_patch(pob, plob, mctype.model, mcinfo.info, id, sflag, splane, world, iworld, bbox, base_color);

               if (polyob.scan(cparm, lmain, proc))
                  polyob.render(cparm, lmain, spot, proc);

               if (polyob.shadptr) {
                  if (shadptr) {
                     polyob.shadptr->next = shadptr->next;
                     shadptr->next = polyob.shadptr;
                  }

                  else
                     delete polyob.shadptr;

                  polyob.shadptr = (pc *)NULL;
               }

            }

         if (flag3d)
            for (l--; l > -1; l--) {
               slist.shade[l].s = NULL;
               slist.shade[l].stype = NULL;
            }

         return;
   }

}


/* **************************************************
   This procedure scans 2D grids into surface patches
   or polygonal structures.
************************************************** */
void spatch::prender(engine *proc) {

   int i, j;

   switch (renderflag) {

      case SPATPOLY:
         polyob.prender(proc);
         return;

      case SPATREND:

         for (i=0; i<ob->prow; i++)
            for (j=0; j<ob->pcol; j++) {
               if (!convert(i, j, &ptr, (texpolygon *)NULL, (shaderlisttype *)NULL))
                  continue;

               polyob.setup_patch(pob, plob, mctype.model, mcinfo.info, id, sflag, splane, world, iworld, bbox, base_color);

               if (polyob.scan(cptr, lptr, proc))
                  polyob.prender(proc);

               if (polyob.shadptr != (pc *)NULL) {
                  if (shadptr) {
                     polyob.shadptr->next = shadptr->next;
                     shadptr->next = polyob.shadptr;
                  }

                  else
                     delete polyob.shadptr;

                  polyob.shadptr = (pc *)NULL;
               }

            }

         return;
   }

}


/* **************************************************
************************************************** */
int spatch::beamscan(spotlight *lparm, engine *proc) {

   renderflag = 0;

   if (mctype.model < BW || (mctype.model > PHONG && mctype.model < PBW))
      return 0;

   mcinfo.info = CINULL;
   mctype.model = (mctype.model <= PHONG) ? BW : PBW;

   transform(lparm);

   if (!ob->psize || bound_box(lparm))
      return 0;

   if (!clip(lparm, lparm->maxx, lparm->maxy))
      return 0;

   if (mctype.model == BW) {                // display data as polygons
      setup_transfer(ob->srow*ob->scol, ob->psize);

      if (!spline2poly(ob, (texpolygon *)NULL, 0))
         return 0;

      polyob.setup_patch(pob, plob, mctype.model, CINULL, id, sflag, splane, world, iworld, bbox, base_color);

      renderflag = SPATPOLY;
      return polyob.beamscan(lparm, proc);
   }

   ptr.build(SPDELTA + 3, SPDELTA + 3, mctype.model > PWFBW, 0);
   setup_transfer(ptr.vsize, ptr.psize);

   renderflag = SPATREND;
   return 1;
}


/* **************************************************
************************************************** */
void spatch::beamrender(spotlight *lparm, engine *proc) {

   int i, j;

   switch (renderflag) {

      case SPATPOLY:
         polyob.beamrender(lparm, proc);
         return;

      case SPATREND:

         for (i=0; i<ob->prow; i++)
            for (j=0; j<ob->pcol; j++) {
               if (!convert(i, j, &ptr, (texpolygon *)NULL, (shaderlisttype *)NULL))
                  continue;

               polyob.setup_patch(pob, plob, mctype.model, CINULL, id, sflag, splane, world, iworld, bbox, base_color);

               if (polyob.beamscan(lparm, proc))
                  polyob.beamrender(lparm, proc);
            }

         return;

      default:
         return;
   }

}
