

/* *************************************************************
   This file scans surface patch data into the buffer
************************************************************* */

#include <stdlib.h>
#include <string.h>

#include "pstring.h"
#include "spatch.h"


/* *************************************************************
************************************************************* */
void spatchtype::init() {

   statusflag = STATUSFLAG_VOID;

   bs_radius = 0;
   bs_center[0] = bs_center[1] = bs_center[2] = 0; bs_center[3] = 1.0;

   ndata = NULL;
   gnormal = NULL;
   data = NULL;
   mvertex = NULL;
   mshade = NULL;
}


/* *************************************************************
************************************************************* */
void spatchtype::dest() {

   if (mvertex) {
      delete [] data;
      delete [] mvertex;
   }

   if (gnormal) {
      delete [] ndata;
      delete [] gnormal;
   }

   if (mshade)
      delete [] mshade;

   init();
}

/* *************************************************************
************************************************************* */
int spatchtype::build(int crow, int ccol, int nflag, int sflag) {

   int i, j;

   if (!crow || !ccol)
      return 0;

   dest();

   vrow = crow;
   vcol = ccol;

   srow = vrow - 2;
   scol = vcol - 2;

   prow = vrow - 3;
   pcol = vcol - 3;

   vsize = vrow*vcol;
   psize = prow*pcol;

   data = new vector4f[vsize];
   mvertex = new pvector4f[vrow];

   for (i=j=0; i<vrow; i++, j+=vcol)
      mvertex[i] = &data[j];

   if (nflag) {
      ndata = new vector3f[vsize];
      gnormal = new pvector3f[vrow];

      for (i=j=0; i<vrow; i++, j+=vcol)
         gnormal[i] = &ndata[j];
   }

   if (sflag) {
      shade  = new pshadetype[psize];
      mshade = new ppshadetype[prow];

      for (i=j=0; i<prow; i++, j+=pcol)
         mshade[i] = &shade[j];
   }

   return 1;
}


/* *************************************************************
************************************************************* */
int spatchtype::read_data(char *filename) {

   FILE *infile;
   int i, j, size;
   vector3f min, max;
   int init_flag = 1;
   
   if (!find_file(filename, "r", OBJECT_PATH.string, (char)PLATFORM_SLASH, NULL, &infile)) {
      sprintf(perror_buffer, "Could not access %s.\n", filename);
      pprintf(perror_buffer);
      return 0;
   }

   dataname.stringcpy(filename);

   fscanf(infile, "%d %d", &vrow, &vcol);

   srow = vrow - 2;
   scol = vcol - 2;

   prow = vrow - 3;
   pcol = vcol - 3;

   vsize = vrow*vcol;
   psize = prow*pcol;

   size = vrow * vcol;

   data = new vector4f[size];

   for (i=0; i<size; i++) {
      fscanf(infile, "%f %f %f", &data[i][0], &data[i][1], &data[i][2]);
      data[i][3] = 1;

      if (init_flag) {
         init_flag = 0;
         copyarray3(min, data[i]);
         copyarray3(max, data[i]);
      }

      else
         for (j=0; j<3; j++)
	    if (data[i][j] < min[j])
	       min[j] = data[i][j];
	    else if (data[i][j] > max[j])
	       max[j] = data[i][j];
   }

   bs_center[0] = (float)(0.5 * (min[0] + max[0]));
   bs_center[1] = (float)(0.5 * (min[1] + max[1]));
   bs_center[2] = (float)(0.5 * (min[2] + max[2]));

   subarray3(min, max);
   bs_radius = dotproduct3(min, min);

   bs_radius = (float)((bs_radius < CORRECT) ? 0 : 0.5*sqrt(bs_radius));

   fclose(infile);

   statusflag |= STATUSFLAG_LOADABLE | STATUSFLAG_LOADED;

   return (vrow-3)*(vcol-3);
}


/* *************************************************************
************************************************************* */
void spatchtype::update(float current_time) {

}


/* *************************************************************
************************************************************* */
void *spatchtype::query_data() {

   return NULL;
}


/* *************************************************************
************************************************************* */
void spatchtype::replace_data(void *subresource) {

}


/* *************************************************************
************************************************************* */
int spatch::query_whatwasi(int type) {

   if (spatch::query_whatami() == type)
      return 1;

   return pc::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int spatch::parse(FILE *infile, char *token) {

   switch (token[0]) {

      case '2':
         if (!strcmp(token, TOKEN_2D_TEXTURE_STR)) {
            get_token(infile, token);
            texname2.stringcpy(token);
            mcinfo.mask_or(CITEXTURE);
            return 1;
         }

         break;

      case 'f':
         if (!strcmp(token, TOKEN_FILE_STR)) {
            get_token(infile, token);
            filename.stringcpy(token);
            return 1;
         }

         break;

      default:
         break;
   }

   return make_poly::parse(infile, token);
}


/* *************************************************************
************************************************************* */
void spatch::preprocess(void *data) {

   colortype ilm;
   int i;

   make_poly::preprocess(data);

   smultarray3(rotate[0], size);
   smultarray3(rotate[1], size);
   smultarray3(rotate[2], size);

   rotate[0][3] += center[0];
   rotate[1][3] += center[1];
   rotate[2][3] += center[2];

   if (!((frame_manager *)global_resource_manager)->query_render())
      return;

   i = read_data(filename.string);
   
   if (!i) {
      sprintf(perror_buffer, "Error: Unable to access \"%s\"... Aborting...\n", filename.string);
      pprintf(perror_buffer);
      exit(0);
   }

   if (colorname.string[0])
      lob = (shadelist *)((frame_manager *)global_resource_manager)->read_ilm(colorname.string, i);
      
   ilm = mctype.query_master();

   if ((ilm > BW && ilm <= PHONG) || ilm > PBW) {

      if (mcinfo.query_master() & CITEXTURE)
         read_tex(texname2.string, filename.string);
      if (mcinfo.query_master() & CI3DTEX)
         read_3dtex(texname3.string, filename.string);
   }

}


/* *************************************************************
************************************************************* */
void spatch::bound_sphere() {

   float mag, temp;
   vector4f r[4];

   matvecmulto(rotate, dob->bs_center, bcenter);

   transpose(r, rotate);

   mag = dotproduct3(r[0], r[0]);

   temp = dotproduct3(r[1], r[1]);
   if (temp > mag)
      mag = temp;

   temp = dotproduct3(r[2], r[2]);
   if (temp > mag)
      mag = temp;

   bradius = (float)(dob->bs_radius * sqrt(mag));

   bvalid_flag = 1;
}


/* *************************************************************
************************************************************* */
int spatch::read_3dtex(char *filename, char *dfilename) {

   shaderlist = (shaderlisttype *)((frame_manager *)global_resource_manager)->read_tex3d(filename, dfilename, dob->psize);

   if (!shaderlist) {
      mcinfo.mask_and(~CI3DTEX);
      return 0;
   }

   return 1;
}

/* *************************************************************
************************************************************* */
int spatch::read_tex(char *filename, char *dfilename) {

   FILE *infile;                        // file pointer
   int i, j, k, l, m;
   int coord[2][2];
   float corner[4][2];
   int count;
   char buffer[MAXSTRLEN], tempstr[MAXSTRLEN];             // generic string
   int patch;
   texbase *ptr;
   vector2f *pointer;
   int   start[2], end[2];
   double pt[2], d[2], current[2];
   int   vtr;
   texface *uvblock;
   
   int prow, pcol, srow, scol;

   prow = dob->prow;
   pcol = dob->pcol;
   srow = dob->srow;
   scol = dob->scol;

   tob = (texpolygon *)global_resource_manager->find_resource_object(RESOURCE_UVCOORD, dfilename, filename);
   
   if (tob)
      return 1;

   if (!find_file(filename, "r", OBJECT_PATH.string, (char)PLATFORM_SLASH, NULL, &infile)) {
      sprintf(perror_buffer, "Could not access %s.\n", filename);
      pprintf(perror_buffer);
      mcinfo.mask_and(~CITEXTURE);
      return 0;
   }

   tob = new texpolygon;
   tob->altname.stringcpy(filename);
   tob->dataname.stringcpy(dfilename);

   tob->setup(prow*pcol);
   uvblock = (texface *)tob->query_data();
   
   global_resource_manager->register_resource_object(RESOURCE_UVCOORD, tob);
   
   pointer = new vector2f[scol*srow];

   fscanf(infile, "%d", &count);

   for (; count>0; count--) {

      fscanf(infile, "%s", buffer);
      lower_case(buffer);

      fscanf(infile, "%d %d %d %d", &coord[0][0], &coord[0][1], &coord[1][0], &coord[1][1]);

      if (!strcmp(buffer, "coord"))
         fscanf(infile, "%f %f %f %f", &corner[0][0], &corner[0][1], &corner[2][0], &corner[2][1]);

      fscanf(infile, "%s", tempstr);               // texture name

      ptr = (texbase *)((frame_manager *)global_resource_manager)->read_tex(tempstr);

      if (!ptr) {
         mcinfo.mask_and(~CITEXTURE);
         fclose(infile);
         return 0;
      }

      coord[0][0]--;
      coord[0][1]--;
      coord[1][0]--;
      coord[1][1]--;

      if (!strcmp(buffer, "corner")) {
         if (coord[0][0] != coord[1][0] && coord[0][1] != coord[1][1]) {

            if (coord[0][0] > coord[1][0]) {
               start[0] = coord[1][0];
               end[0]   = coord[0][0];
            }

            else {
               start[0] = coord[0][0];
               end[0]   = coord[1][0];
            }

            if (coord[0][1] > coord[1][1]) {
               start[1] = coord[1][1];
               end[1]   = coord[0][1];
            }

            else {
               start[1] = coord[0][1];
               end[1]   = coord[1][1];
            }

            corner[1][0] = corner[0][0] = corner[3][1] = corner[0][1] = 0.0;
            corner[2][1] = corner[1][1] = (float)(ptr->maxy-1);
            corner[3][0] = corner[2][0] = (float)(ptr->maxx-1);

            pt[0] = corner[0][0];
            pt[1] = corner[0][1];
            d[0]  = (corner[2][0]-corner[0][0]) / ((double)(coord[1][0]-coord[0][0]));
            d[1]  = (corner[2][1]-corner[0][1]) / ((double)(coord[1][1]-coord[0][1]));

            for (j=start[1], current[1]=pt[1], i=j*scol; j<=end[1]; j++, current[1]+=d[1], i+=scol)
               for (k=start[0], current[0]=pt[0], vtr=i+k; k<=end[0]; k++, current[0]+=d[0], vtr++)
                   copyarray2(pointer[vtr], (float)current);

            l = end[1]+start[1]-1;
            for (j=start[1], i=j*scol; j<end[1]; j++, i+=scol)
               for (k=start[0], vtr=i+k, m=vtr+scol+1, patch=(l-j)*pcol; k<end[0]; k++, vtr++, m++) {
                  uvblock[patch+k].setup(2);
                  uvblock[patch+k].ob = ptr;

                  uvblock[patch+k].uv[0][0] = pointer[vtr][0];
                  uvblock[patch+k].uv[0][1] = pointer[m][1];

                  uvblock[patch+k].uv[1][0] = pointer[m][0];
                  uvblock[patch+k].uv[1][1] = pointer[vtr][1];
               }

         }

      }

      else {
         corner[0][0]--;
         corner[0][1]--;
         corner[2][0]--;
         corner[2][1]--;

         if (coord[0][0] > coord[1][0]) {
            start[0] = coord[1][0];
            end[0]   = coord[0][0];
         }

         else {
            start[0] = coord[0][0];
            end[0]   = coord[1][0];
         }

         if (coord[0][1] > coord[1][1]) {
            start[1] = coord[1][1];
            end[1]   = coord[0][1];
         }

         else {
            start[1] = coord[0][1];
            end[1]   = coord[1][1];
         }

         pt[0] = corner[0][0];
         pt[1] = corner[0][1];
         d[0]  = (corner[2][0]-corner[0][0]) / ((double)(coord[1][0]-coord[0][0]));
         d[1]  = (corner[2][1]-corner[0][1]) / ((double)(coord[1][1]-coord[0][1]));

         for (j=start[1], current[1]=pt[1], i=j*scol; j<=end[1]; j++, current[1]+=d[1], i+=scol)
            for (k=start[0], current[0]=pt[0], vtr=i+k; k<=end[0]; k++, current[0]+=d[0], vtr++)
                copyarray2(pointer[vtr], (float)current);

         l = end[1]+start[1]-1;
         for (j=start[1], i=j*scol; j<end[1]; j++, i+=scol)
            for (k=start[0], vtr=i+k, m=vtr+scol+1, patch=(l-j)*pcol; k<end[0]; k++, vtr++, m++) {
               uvblock[patch+k].setup(2);
               uvblock[patch+k].ob = ptr;
               uvblock[patch+k].uv[0][0] = pointer[vtr][0];
               uvblock[patch+k].uv[0][1] = pointer[m][1];
               uvblock[patch+k].uv[1][0] = pointer[m][0];
               uvblock[patch+k].uv[1][1] = pointer[vtr][1];
            }

      }

   }

   fclose(infile);
   delete [] pointer;
   return 1;
}


/* *************************************************************
   This function reads in the surface patch data
************************************************************* */
int spatch::read_data(char *filename) {

   dob = (spatchtype *)((frame_manager *)global_resource_manager)->read_spatch(filename);
   return (dob) ? dob->psize : 0;
}


/* *************************************************************
************************************************************* */
int spatch::dump_frame(FILE *outfile) {

   char token[4][MAXSTRLEN];
   int i, j;
   unsigned int master;

   master = mcinfo.query_master();

   fprintf(outfile, "%s {\n", TOKEN_SPATCH_STR);

   sprintf(token[0], "\t%s ", TOKEN_SHADE_STR);
   sprintf(token[1], " %s\n", colorname.string);
   mctype.write_model(outfile, token[0], token[1]);

   if (master & CITEXTURE)
      fprintf(outfile, "\t%s %s\n", TOKEN_2D_TEXTURE_STR, texname2.string);
   if (master & CI3DTEX)
      fprintf(outfile, "\t%s %s\n", TOKEN_3D_TEXTURE_STR, texname3.string);

   fprintf(outfile, "\t%s %s\n", TOKEN_FILE_STR, filename.string);

   if (master & CIRAYCAST)
      fprintf(outfile, "\t%s\n", TOKEN_RAYCAST_STR);

   if (master & CITRANSPARENT)
      fprintf(outfile, "\t%s\n", TOKEN_TRANSPARENT_STR);
   else if (sflag && !NOSHADOW) {
      float2char(splane[0], token[0]);
      float2char(splane[1], token[1]);
      float2char(splane[2], token[2]);
      float2char(splane[3], token[3]);
      fprintf(outfile, "\t%s %s %s %s %s\n", TOKEN_SHADOW_STR, token[0], token[1], token[2], token[3]);
   }

   fprintf(outfile, "\t%s ", TOKEN_TR_MATRIX_STR);
   for (i=0; i<4; i++)
      for (j=0; j<4; j++) {
         float2char(rotate[i][j], token[0]);
         fprintf(outfile, "%s ", token);
      }

   fprintf(outfile, "\n");

   fprintf(outfile, "\tframe %d\n", frame);
   fprintf(outfile, "}\n");

   return 1;
}

