

#include <string.h>

#include "pstring.h"
#include "shading.h"
#include "datatype.h"
#include "global.h"
#include "dataio.h"


/* *************************************************************
************************************************************* */
shadetype::shadetype() {

   fka[0] = fka[1] = fka[2] = 1.0f;
   fkp[0] = fkp[1] = fkp[2] = 0.7f;
   fks[0] = fks[1] = fks[2] = 0.3f;
   flum[0] = flum[1] = flum[2] = 0;
   specn = 1.0;

   update_int();
}


/* *************************************************************
************************************************************* */
void shadetype::update_int() {

   ka[0] = (int)(fka[0] * 255);
   ka[1] = (int)(fka[1] * 255);
   ka[2] = (int)(fka[2] * 255);

   kp[0] = (int)(fkp[0] * 255);
   kp[1] = (int)(fkp[1] * 255);
   kp[2] = (int)(fkp[2] * 255);

   ks[0] = (int)(fks[0] * 255);
   ks[1] = (int)(fks[1] * 255);
   ks[2] = (int)(fks[2] * 255);

   lum[0] = (int)(flum[0] * 255);
   lum[1] = (int)(flum[1] * 255);
   lum[2] = (int)(flum[2] * 255);
}


/* *************************************************************
************************************************************* */
void shadetype::update_float() {

   double div = 1.0/255.0;
   
   fka[0] = (float)(ka[0] * div);
   fka[1] = (float)(ka[1] * div);
   fka[2] = (float)(ka[2] * div);

   fkp[0] = (float)(kp[0] * div);
   fkp[1] = (float)(kp[1] * div);
   fkp[2] = (float)(kp[2] * div);

   fks[0] = (float)(ks[0] * div);
   fks[1] = (float)(ks[1] * div);
   fks[2] = (float)(ks[2] * div);

   flum[0] = (float)(lum[0] * div);
   flum[1] = (float)(lum[1] * div);
   flum[2] = (float)(lum[2] * div);
}


/* *************************************************************
************************************************************* */
void shadetype::read_data(FILE *infile) {

   char junk[MAXSTRLEN];

   fscanf(infile, "%s %f %f %f", junk, &fka[0], &fka[1], &fka[2]);
   fscanf(infile, "%s %f %f %f", junk, &fkp[0], &fkp[1], &fkp[2]);
   fscanf(infile, "%s %f %f %f", junk, &fks[0], &fks[1], &fks[2]);
   fscanf(infile, "%s %f", junk, &specn);
   fscanf(infile, "%s %f %f %f", junk, &flum[0], &flum[1], &flum[2]);

   update_int();
}


/* *************************************************************
************************************************************* */
int shadetype::write_data(FILE *outfile) {

   fprintf(outfile, "KA %f %f %f\n", fka[0], fka[1], fka[2]);
   fprintf(outfile, "KP %f %f %f\n", fkp[0], fkp[1], fkp[2]);
   fprintf(outfile, "KS %f %f %f\n", fks[0], fks[1], fks[2]);
   fprintf(outfile, "SPECN %f\n", specn);
   fprintf(outfile, "LUM %f %f %f\n\n", flum[0], flum[1], flum[2]);
   return 1;
}


/* *************************************************************
************************************************************* */
int shadetype::matcmp(shadetype *x) {

   vector3i diff;
   float t;
   
   subeqarray3(diff, ka, x->ka);

   if (diff[0] < 0)
      return -1;
   else if (diff[0] > 0)
      return 1;
      
   if (diff[1] < 0)
      return -1;
   else if (diff[1] > 0)
      return 1;

   if (diff[2] < 0)
      return -1;
   else if (diff[2] > 0)
      return 1;

   subeqarray3(diff, kp, x->kp);

   if (diff[0] < 0)
      return -1;
   else if (diff[0] > 0)
      return 1;
      
   if (diff[1] < 0)
      return -1;
   else if (diff[1] > 0)
      return 1;

   if (diff[2] < 0)
      return -1;
   else if (diff[2] > 0)
      return 1;

   subeqarray3(diff, ks, x->ks);

   if (diff[0] < 0)
      return -1;
   else if (diff[0] > 0)
      return 1;
      
   if (diff[1] < 0)
      return -1;
   else if (diff[1] > 0)
      return 1;

   if (diff[2] < 0)
      return -1;
   else if (diff[2] > 0)
      return 1;

   t = x->specn - x->specn;
   
   if (t < -CORRECT)
      return -1;
   else if (t > CORRECT)
      return 1;
      
   return 0;
}


/* ***************************************************************
*************************************************************** */
void texcolortype::set(shadetype *x) {

   copyarray3(color.ka, x->ka);
   copyarray3(color.kp, x->kp);
   copyarray3(color.ks, x->ks);
   color.specn = x->specn;
   copyarray3(color.lum, x->lum);
}


/* ***************************************************************
*************************************************************** */
shade_block::shade_block() {

   shade_count = index_count = 0;
   facelist = NULL;
   shade_palette = NULL;
   index_palette = NULL;
}


/* ***************************************************************
*************************************************************** */
shade_block::~shade_block() {

   if (facelist)
      delete [] facelist;

   if (shade_palette)
      delete [] shade_palette;

   if (index_palette)
      delete [] index_palette;
}


/* ***************************************************************
*************************************************************** */
shadelist::shadelist() {

   sblock = NULL;
   maxfacecount = 0;
}


/* ***************************************************************
*************************************************************** */
shadelist::shadelist(int fcount) : resource_type() {

   maxfacecount = fcount;
   sblock = new shade_block;
   sblock->facelist = new pshadetype[fcount];
}


/* *************************************************************
************************************************************* */
shadelist::~shadelist() {

   if (sblock)
      delete sblock;
}


/* *************************************************************
************************************************************* */
void *shadelist::query_data() {

   file_loader *loader;

   if (!(statusflag & STATUSFLAG_LOADED)) {
      if (!(statusflag & STATUSFLAG_LOADABLE))
         return sblock;

      loader = (file_loader *)global_resource_manager->find_resource_object(RESOURCE_POLYGON_LOADER, dataname.string, NULL);

      if (!loader)
         return NULL;

      loader->read_data();
      loader->extract(FILETYPE_ILM, altname.string);
      loader->cleanup();
   }

   statusflag |= STATUSFLAG_TIME_UPDATE;
   return sblock;   
}


/* *************************************************************
************************************************************* */
void shadelist::update(float current_time) {

   if (!(statusflag & STATUSFLAG_LOADABLE) || !(statusflag & STATUSFLAG_LOADED))
      return;

   if (!(statusflag & STATUSFLAG_TIME_UPDATE)) {
      delete sblock;
      sblock = NULL;
      statusflag &= ~STATUSFLAG_LOADED;
      return;
   }

   global_statistic_material_count++;
   access_time = current_time;
   statusflag &= ~STATUSFLAG_TIME_UPDATE;
}


/* *************************************************************
************************************************************* */
void shadelist::replace_data(void *subresource) {

   if (sblock)
      delete sblock;

   sblock = (shade_block *)subresource;
}


/* *************************************************************
************************************************************* */
int shadelist::read_data(char *filename, int facecount) {

   FILE *infile;                        // file pointer
   char junk[MAXSTRLEN];
   int i, j, k;
   shadetype *str;
   int *itr;
   int max;

   altname.stringcpy(filename);

   if (sblock)
      delete sblock;

   sblock = new shade_block;
   statusflag = STATUSFLAG_VOID;
      
   if (!find_file(filename, "r", MATERIAL_PATH.string, (char)PLATFORM_SLASH, NULL, &infile)) {

      sblock->shade_palette = new shadetype[sblock->shade_count = 1];

      sblock->index_palette = new int[sblock->index_count = facecount];
      memset(sblock->index_palette, 0, facecount*sizeof(int));

      postprocess(facecount);
      statusflag |= STATUSFLAG_LOADABLE | STATUSFLAG_LOADED;
      sprintf(perror_buffer, "Couldn't open %s\n", filename);
      pprintf(perror_buffer);
      return 0;
   }

   fscanf(infile, "%s", junk);

   if (!strcmp(NEW_ILM_TAG, junk)) {
      fscanf(infile, "%s %d\n", junk, &sblock->shade_count);

      sblock->shade_palette = new shadetype[sblock->shade_count];

      for (i=0; i<sblock->shade_count; i++)
         sblock->shade_palette[i].read_data(infile);

      fscanf(infile, "%s %d\n", junk, &j);

      sblock->index_count = j > facecount ? j : facecount;
   
      sblock->index_palette = new int[sblock->index_count];
      memset(sblock->index_palette, 0, sblock->index_count*sizeof(int));

      for (i=0; i<j; i++) {
         fscanf(infile, "%s %d", junk, &k);
         sblock->index_palette[i] = k-1;
      }

      memcpy(&base, &sblock->shade_palette[0], sizeof(shadetype));
   }

   // old format

   else {
      fseek(infile, 0, SEEK_SET);
      base.read_data(infile);

      fscanf(infile, "%s %d", junk, &j);

      if (!j) {
         sblock->shade_palette = new shadetype[sblock->shade_count = 1];
         memcpy(&sblock->shade_palette[0], &base, sizeof(shadetype));

         sblock->index_palette = new int[sblock->index_count = facecount];
         memset(sblock->index_palette, 0, facecount*sizeof(int));
      }

      else {
         sblock->shade_palette = new shadetype[sblock->shade_count = j+1];

         memcpy(&sblock->shade_palette[0], &base, sizeof(shadetype));
         max = 1;

         if (j) {
            itr = new int[j];
            str = new shadetype[j];

            for (i = 0; i<j; i++) {
               fscanf(infile, "%s %d", junk, &k);
               itr[i] = k-1;
               str[i].read_data(infile);
               if (k > max)
                  max = k;
            }

         }

         sblock->index_count = facecount > max ? facecount : max;
         sblock->index_palette = new int[sblock->index_count];
         memset(sblock->index_palette, 0, sblock->index_count*sizeof(int));

         if (j) {
            memcpy(&sblock->shade_palette[1], str, j*sizeof(shadetype));

            for (i=0; i<j; i++)
               sblock->index_palette[itr[i]] = i+1;

            delete [] itr;
            delete [] str;
         }

      }

   }

   fclose(infile);
   postprocess(sblock->index_count);

   statusflag |= STATUSFLAG_LOADABLE | STATUSFLAG_LOADED;
   return 1;
}


/* *************************************************************
************************************************************* */
int shadelist::write_data(char *filename) {

   FILE *outfile;                        // file pointer
   int i;

   if (!sblock)
      return 0;
   
   outfile = fopen(filename, "w");
   if (!outfile) {
      sprintf(perror_buffer, "Couldn't open %s for output\n", filename);
      pprintf(perror_buffer);
      return 0;
   }

   fprintf(outfile, "%s\n", NEW_ILM_TAG);

   fprintf(outfile, "no_shades %d\n", sblock->shade_count);

   for (i=0; i<sblock->shade_count; i++)
      sblock->shade_palette[i].write_data(outfile);
   
   fprintf(outfile, "no_index %d\n", sblock->index_count);

   for (i=0; i<sblock->index_count; i++)
      fprintf(outfile, "# %d\n", sblock->index_palette[i]+1);

   fclose(outfile);
   return 1;
}


/* *************************************************************
************************************************************* */
void shadelist::postprocess(int facecount) {

   int i;

   if (facecount <= maxfacecount) {
      if (maxfacecount && sblock->facelist)
         return;
   }
   
   else 
      maxfacecount = facecount;

   if (sblock->facelist) {
      delete [] sblock->facelist;
      sblock->facelist = NULL;
   }

   sblock->facelist = new pshadetype[maxfacecount];

   for (i=0; i<sblock->index_count; i++)
      sblock->facelist[i] = &sblock->shade_palette[sblock->index_palette[i]];

   for (; i<maxfacecount; i++)
      sblock->facelist[i] = &sblock->shade_palette[0];
}
