

#include <string.h>

#include "polytype.h"
#include "global.h"


/* *************************************************************
************************************************************* */
polygon_object_type::polygon_object_type() {

   vlist  = NULL;
   nlist  = NULL;
   flist  = NULL;
   vindex = NULL;
}


/* *************************************************************
************************************************************* */
void polygon_object_type::dest() {

   if (vlist)
      delete [] vlist;

   if (nlist)
      delete [] nlist;

   if (flist)
      delete [] flist;

   if (vindex)
      delete [] vindex;
}


/* *************************************************************
************************************************************* */
int polygon_object_type::build(int cvertex, int cobject, int cedge) {

   if (!cvertex || !cobject || !cedge)
      return 0;

   dest();

   vlist  = new vector4f[cvertex];
   nlist  = new vector3f[cvertex];
   flist  = new face_type[cobject];
   vindex = new int[cedge];

   return 1;
}


/* *************************************************************
************************************************************* */
void polytype::init() {

   statusflag = STATUSFLAG_VOID;

   polygon_data = NULL;

   maxpolynum = 0;
   bs_radius = 0;
   bs_center[0] = bs_center[1] = bs_center[2] = 0; bs_center[3] = 1.0;
}


/* *************************************************************
************************************************************* */
void polytype::dest() {

   if (polygon_data)
      delete polygon_data;

   init();
}


/* *************************************************************
************************************************************* */
int polytype::build(int cvertex, int cobject, int cedge) {

   if (!cvertex || !cobject || !cedge)
      return 0;

   dest();

   countvertex = cvertex;
   countobject = cobject;
   countedge   = cedge;

   polygon_data = new polygon_object_type;
   polygon_data->build(cvertex, cobject, cedge);
   return 1;
}


/* *************************************************************
************************************************************* */
void *polytype::query_data() {

   file_loader *loader;

   if (!(statusflag & STATUSFLAG_LOADED)) {
      if (!(statusflag & STATUSFLAG_LOADABLE))
         return polygon_data;

      loader = (file_loader *)global_resource_manager->find_resource_object(RESOURCE_POLYGON_LOADER, dataname.string, NULL);

      if (!loader)
         return NULL;
	 
      loader->read_data();
      loader->extract(FILETYPE_SPG, dataname.string);
      loader->cleanup();
   }

   statusflag |= STATUSFLAG_TIME_UPDATE;
   return polygon_data;
}


/* *************************************************************
asdf - future - can use accesstime to drop object based on time...
************************************************************* */
void polytype::update(float current_time) {

   if (!(statusflag & STATUSFLAG_LOADABLE) || !(statusflag & STATUSFLAG_LOADED))
      return;

   if (!(statusflag & STATUSFLAG_TIME_UPDATE)) {
      delete polygon_data;
      polygon_data = NULL;
      statusflag &= ~STATUSFLAG_LOADED;
      return;
   }

   global_statistic_object_count++;
   access_time = current_time;
   statusflag &= ~STATUSFLAG_TIME_UPDATE;
}


/* *************************************************************
************************************************************* */
void polytype::replace_data(void *subresource) {

   if (polygon_data)
      delete polygon_data;

   polygon_data = (polygon_object_type *)subresource;
}


/* *************************************************************
************************************************************* */
void polytype::preprocess() {

   vector3f min, max;
   int init_flag = 1;
   int i, j;
   float *v;
   face_type *face;

   for (i=0, v=(float *)polygon_data->vlist; i<countvertex; i++, v+=4) {

      if (init_flag) {
         init_flag = 0;
         copyarray3(min, v);
         copyarray3(max, v);
         continue;
      }

      for (j=0; j<3; j++)
         if (v[j] < min[j])
            min[j] = v[j];
         else if (v[j] > max[j])
            max[j] = v[j];
   }

   bs_center[0] = (float)(0.5 * (min[0] + max[0]));
   bs_center[1] = (float)(0.5 * (min[1] + max[1]));
   bs_center[2] = (float)(0.5 * (min[2] + max[2]));

   subarray3(min, max);
   bs_radius = dotproduct3(min, min);
   bs_radius = (bs_radius < CORRECT) ? 0 : (float)(0.5*sqrt(bs_radius));

   maxpolynum = 0;

   for (i=0, face=polygon_data->flist; i<countobject; i++, face++)
      if (maxpolynum < face->polynum)
         maxpolynum = face->polynum;

   calc_normal();
   calc_area();
   calc_vnormal();
}


/* *************************************************************
************************************************************* */
int polytype::read_data(char *filename) {

   FILE *infile;                        // file pointer
   int i, j, k, l;                      // looping var
   float *v;
   face_type *face;
   
   if (!find_file(filename, "r", OBJECT_PATH.string, (char)PLATFORM_SLASH, NULL, &infile)) {
      sprintf(perror_buffer, "Could not access \"%s\"....\n", filename);
      pprintf(perror_buffer);
      return 0;
   }

   dataname.stringcpy(filename);

   fscanf(infile, "%d %d %d", &countvertex, &countobject, &countedge);

   if (!countvertex || !countobject || !countedge) {
      fclose(infile);
      return 0;
   }

   build(countvertex, countobject, countedge);

   for (i=0, v=(float *)polygon_data->vlist; i<countvertex; i++, v+=4) {
      fscanf(infile, "%f %f %f", &v[0], &v[1], &v[2]);
      v[3] = 1.0f;
   }

   countedge = 0;

   for (i=0, face=polygon_data->flist; i<countobject; i++, face++) {
      fscanf(infile, "%d", &l);
      face->polynum = l;

      face->edgeptr = &polygon_data->vindex[countedge];

      countedge += l;

      for (j=0; j<l; j++) {
         fscanf(infile, "%d", &k);
         face->edgeptr[j] = k-1;
      }

   }

   fclose(infile);

   preprocess();
   
   statusflag |= STATUSFLAG_LOADABLE | STATUSFLAG_LOADED;

   return countobject;
}


/* *************************************************************
************************************************************* */
int polytype::write_data(char *filename) {

   FILE *outfile;                        // file pointer
   int i, j;                           // looping var
   float *v;
   face_type *face;
   
   if (!polygon_data) {
      sprintf(perror_buffer, "No geometrical data for \"%s\"...\n", filename);
      pprintf(perror_buffer);
      return 0;
   }

   if (!(outfile = fopen(filename, "w"))) {
      sprintf(perror_buffer, "Could not output \"%s\"....\n", filename);
      pprintf(perror_buffer);
      return 0;
   }

   fprintf(outfile, "%d %d %d\n", countvertex, countobject, countedge);

   fprintf(outfile, "\n");

   for (i=0, v=(float *)polygon_data->vlist; i<countvertex; i++, v+=4)
      fprintf(outfile, "%f %f %f\n", v[0], v[1], v[2]);

   fprintf(outfile, "\n");

   for (i=0, face=polygon_data->flist; i<countobject; i++, face++) {
      fprintf(outfile, "%d ", face->polynum);

      for (j=0; j<face->polynum; j++)
         fprintf(outfile, "%d ", face->edgeptr[j]+1);

      fprintf(outfile, "\n");
   }

   fclose(outfile);
   return 1;
}


/* *************************************************************
************************************************************* */
void polytype::calc_normal() {

   int i;
   face_type *face;

   for (i=0, face=polygon_data->flist; i<countobject; i++, face++) {
      if (face->polynum < 3)
         continue;

      calculate_normal(polygon_data->vlist[face->edgeptr[0]], polygon_data->vlist[face->edgeptr[1]], polygon_data->vlist[face->edgeptr[2]], face->normal);

                                        // used for phong2  ax+by+cz+d
      face->normal[3] = -dotproduct3(face->normal, polygon_data->vlist[face->edgeptr[0]]);
   }

}


/* *************************************************************
   asdf  - can keep counts for each vertex, and use that instead of normalize
************************************************************* */
void polytype::calc_vnormal() {

   int i;
   face_type *face;
   int *sindex, *eindex;

   memset(polygon_data->nlist, 0, countvertex*sizeof(vector3f));

   for (i=0, face=polygon_data->flist; i<countobject; i++, face++)
      for (sindex=face->edgeptr, eindex=face->edgeptr+face->polynum; sindex<eindex; sindex++)
         addarray3(polygon_data->nlist[*sindex], face->normal);

   for (i=0; i<countvertex; i++)
      normalize3(polygon_data->nlist[i]);
}


/* *************************************************************
   used for software mipmapping
************************************************************* */
void polytype::calc_area() {

   int i;
   vector3f temp;
   float work;
   face_type *face;
   int *sindex, *eindex;

   for (i=0, face=polygon_data->flist; i<countobject; i++, face++) {
      work = 0;

      for (sindex=face->edgeptr, eindex=face->edgeptr+face->polynum-1; sindex<eindex; sindex++) {
         xproduct(temp, polygon_data->vlist[*sindex], polygon_data->vlist[sindex[1]]);
         work += dotproduct3(face->normal, temp);
      }

      face->iarea = (float)ISQRT((float)(0.5*fabs(work)));
   }

}


/* *************************************************************
************************************************************* */
void polytype::vtransform(vector4f *mx) {

   int i;
   float *v;

   for (i=0, v=(float *)polygon_data->vlist; i<countvertex; i++, v+=4)
      matvecmultv(mx, v);
}


/* *************************************************************
************************************************************* */
void polytype::ntransform(vector4f *mx) {

   int i;
   face_type *face;

   for (i=0, face=polygon_data->flist; i<countobject; i++, face++)
      matvecmultv(mx, face->normal);
}


/* *************************************************************
************************************************************* */
void polytype::vntransform(vector4f *mx) {

   int i;
   float *n;

   for (i=0, n=(float *)polygon_data->nlist; i<countvertex; i++, n+=3)
      matvecmultv(mx, n);
}


/* *************************************************************
************************************************************* */
void polytype::scale_area(float scale) {

   int i;
   float iscale = 1.0f/scale;
   face_type *face;

   for (i=0, face=polygon_data->flist; i<countobject; i++, polygon_data->flist)
      face->iarea *= iscale;
}
