

#include <string.h>

#include "pstring.h"
#include "spatch.h"
#include "texture.h"


/* *************************************************************
************************************************************* */
frame_manager::frame_manager() {

   frame_manager::local_init();
}


/* *************************************************************
************************************************************* */
frame_manager::~frame_manager() {

   frame_manager::local_reset();
}


/* *************************************************************
************************************************************* */
void frame_manager::init(int x) {

   resource_manager::init(x);
   frame_manager::local_init();
}


/* *************************************************************
************************************************************* */
void frame_manager::reset() {

   resource_manager::reset();
   frame_manager::local_reset();
}


/* *************************************************************
************************************************************* */
void frame_manager::local_init() {

   vis_flag = 1;
   flag256  = 1;
   flipflag = CBYTE_ORDER_ABGR;

   __postprocessCBtexture = NULL;
   __cleanupCBtexture = NULL;
}


/* *************************************************************
************************************************************* */
void frame_manager::local_reset() {

   resource_type *data;

   object_list.dest();
   material_list.dest();
   
   if (__cleanupCBtexture)
      for (data = (resource_type *)texture_list.head; data; data = (resource_type *)data->next)
         ((int (*)(texbase *))__cleanupCBtexture)((texbase *)data);

   texture_list.dest();
   uvcoord_list.dest();
   uvwcoord_list.dest();   
}


/* *************************************************************
************************************************************* */
int frame_manager::postprocessCBtexture(texbase *tob) {

   if (__postprocessCBtexture)
      return ((int (*)(texbase *))__postprocessCBtexture)(tob);

   return 0;
}


/* *************************************************************
************************************************************* */
int frame_manager::cleanupCBtexture(texbase *tob) {

   if (__cleanupCBtexture)
      return ((int (*)(texbase *))__cleanupCBtexture)(tob);

   return 0;
}


/* *************************************************************
************************************************************* */
resource_type *frame_manager::read_spatch(char *filename) {

   resource_type *data;

   data = (resource_type *)find_resource_object(RESOURCE_OBJECT, filename, NULL);

   if (data)
      return data;

   data = new spatchtype;

   if (!((datatype *)data)->read_data(filename)) {
      delete data;
      return NULL;
   }

   register_resource_object(RESOURCE_OBJECT, data);

   return (spatchtype *)data;
}


/* *************************************************************
************************************************************* */
int frame_manager::query_video(sfile *sinfile) {

   char buffer[8];

   sinfile->sseek(0);

   sinfile->scan_token(buffer, 8);
   lower_case(buffer);

   return (!strcmp(buffer, "asdf"));
}


/* *************************************************************
************************************************************* */
texbase *frame_manager::alloc_texture(sfile *sinfile, dbl_llist_manager *handler, image_coder **loader, int flag256) {

   image_coder *ptr;

   if (!handler->head) {
      *loader = NULL;
      return NULL;
   }

   if (query_video(sinfile)) {
      *loader = NULL;
      return new texvid;
   }

   ptr = (image_coder *)((basic_loader *)handler->head)->find_loader(sinfile);

   if (!ptr) {
      *loader = NULL;
      return NULL;
   }

   *loader = ptr;

   if (flag256 && ptr->query_256())
      return new texture256;

   return new texture;
}


/* *************************************************************
   Note: unlike other resources, textures are "registered" in read/skim_texture()
************************************************************* */
resource_type *frame_manager::read_tex(char *filename) {

   texbase *tmap;
   sfile sinfile;
   image_coder *loader;
   int i;

   tmap = (texbase *)find_resource_object(RESOURCE_TEXTURE, filename, NULL);

   if (tmap)
      return tmap;

   if (!sinfile.scan_data(filename, TEXTURE_PATH.string, PLATFORM_SLASH))
      return NULL;

   sinfile.filename.stringcpy(filename);  // sinfile has "true" name, we want "given" name
   tmap = alloc_texture(&sinfile, &image_loader_list, &loader, flag256);

   if (!tmap)
      return NULL;

   if (DYNAMIC_LOADER)
      i = tmap->skim_texture(&sinfile, &image_loader_list, loader, flag256, flipflag);
   else
      i = tmap->read_texture(&sinfile, &image_loader_list, loader, flag256, flipflag);

   if (loader)
      loader->cleanup();

   if (!i) {
      delete tmap;
      return NULL;
   }

   return tmap;
}


/* *************************************************************
************************************************************* */
resource_type *frame_manager::read_ilm(char *filename, int count) {

   shadelist *lob;

   lob = (shadelist *)find_resource_object(RESOURCE_MATERIAL, NULL, filename);

   if (lob) {
      lob->read_data(filename, count);
      return lob;
   }

   lob = new shadelist;

   lob->read_data(filename, count);

   register_resource_object(RESOURCE_MATERIAL, lob);

   return lob;
}



/* *************************************************************
************************************************************* */
resource_type *frame_manager::read_tex3d(char *filename, char *dfilename, int objectcount) {

   int i, count;
   char buffer[MAXSTRLEN];
   FILE *infile;
   shaderlisttype *str;

   str = (shaderlisttype *)find_resource_object(RESOURCE_UVWCOORD, dfilename, filename);

   if (str)
      return str;

   if (!find_file(filename, "r", OBJECT_PATH.string, (char)PLATFORM_SLASH, NULL, &infile)) {
      sprintf(perror_buffer, "Can't find file %s\n", filename);
      pprintf(perror_buffer);
      return NULL;
   }

   str = new shaderlisttype;
   str->altname.stringcpy(filename);
   str->dataname.stringcpy(dfilename);
   str->init(objectcount);

   register_resource_object(RESOURCE_UVWCOORD, str);

   fscanf(infile, "%s %f", buffer, &str->scale);
   lower_case(buffer);

   // ALL
   if (strcmp(buffer, "mono")) {
      str->shade[0].read_data(infile, &proc_shader_list);

      for (i=1; i<objectcount; i++)
         str->shade[i] = str->shade[0];
   }

   fscanf(infile, "%d", &count);

   for (; count>0; count--) {
      fscanf(infile, "%d", &i);

      if (i < objectcount)
         str->shade[i-1].read_data(infile, &proc_shader_list);
      else {
         get_token(infile, buffer);
         get_token(infile, buffer);
      }

   }

   fclose(infile);
   return str;
}


/* *************************************************************
************************************************************* */
void frame_manager::register_resource_object(int type, void *data) {

   binary_node *btr;
   
   if (!data)
      return;
      
   switch (type) {
      case RESOURCE_OBJECT:
         object_list.insert((resource_type *)data, NULL);
         return;

      case RESOURCE_MATERIAL:
         material_list.insert((resource_type *)data, NULL);
         return;

      case RESOURCE_TEXTURE:
         texture_list.insert((resource_type *)data, NULL);
         return;

      case RESOURCE_UVCOORD:
         uvcoord_list.insert((resource_type *)data, NULL);
         return;

      case RESOURCE_UVWCOORD:
         uvwcoord_list.insert((resource_type *)data, NULL);
         return;

      case RESOURCE_SHADER:
         btr = new binary_node;
	 btr->key = (shader *)data;
         proc_shader_list.insert(btr);
         return;

      case RESOURCE_IMAGE_LOADER:
         image_loader_list.insert((image_coder *)data, NULL);
         return;

      case RESOURCE_POLYGON_LOADER:
         poly_loader_list.insert((file_loader *)data, NULL);
         return;

      case RESOURCE_OBJECT_LOADER:
         object_loader_list.insert((loader *)data, NULL);
         return;

      default:
         resource_manager::register_resource_object(type, data);
         return;
   }

}


/* *************************************************************
************************************************************* */
void *frame_manager::find_resource_object(int type, char *id, char *alias) {

   resource_type *dob;
   string_binary_data sbd;
   binary_node *btr;

   switch (type) {

      case RESOURCE_OBJECT:
         for (dob=(resource_type *)object_list.head; dob; dob=(resource_type *)dob->next)
            if (!dob->dataname.stringcmp(id))
               return dob;

         return NULL;

      case RESOURCE_MATERIAL:
         for (dob=(resource_type *)material_list.head; dob; dob=(resource_type *)dob->next)
            if (!dob->altname.stringcmp(alias))
               return dob;

         return NULL;

      case RESOURCE_UVCOORD:
         for (dob=(resource_type *)uvcoord_list.head; dob; dob=(resource_type *)dob->next)
            if (!dob->dataname.stringcmp(id) && !dob->altname.stringcmp(alias))
               return dob;

         return NULL;

      case RESOURCE_UVWCOORD:
         for (dob=(resource_type *)uvwcoord_list.head; dob; dob=(resource_type *)dob->next)
            if (!dob->dataname.stringcmp(id) && !dob->altname.stringcmp(alias))
               return dob;

         return NULL;

      case RESOURCE_SHADER:
         sbd.string.stringcpy(id);
         btr = proc_shader_list.find(&sbd);

         return btr ? btr->key : NULL;

      case RESOURCE_POLYGON_LOADER:
         if (!sinfile.scan_data(id, OBJECT_PATH.string,(char)PLATFORM_SLASH)) {
            sprintf(perror_buffer, "Warning: \"%s\" not found...\n", OBJECT_PATH.string);
            pprintf(perror_buffer);
            return NULL;
         }

         return poly_loader_list.head ? ((file_loader *)poly_loader_list.head)->find_loader(&sinfile) : (file_loader *)NULL;

      case RESOURCE_TEXTURE:
         for (dob=(resource_type *)texture_list.head; dob; dob=(resource_type *)dob->next)
            if (!dob->dataname.stringcmp(id))
               return dob;

         return NULL;

      case RESOURCE_OBJECT_LOADER:
      case RESOURCE_IMAGE_LOADER:

      default:
         return resource_manager::find_resource_object(type, id, alias);
   }

}


/* *************************************************************
************************************************************* */
void *frame_manager::get_resource_object(int type) {

   switch (type) {
      case RESOURCE_OBJECT:
         return &object_list;

      case RESOURCE_MATERIAL:
         return &material_list;

      case RESOURCE_TEXTURE:
         return &texture_list;

      case RESOURCE_UVCOORD:
         return &uvcoord_list;

      case RESOURCE_UVWCOORD:
         return &uvwcoord_list;

      case RESOURCE_SHADER:
         return &proc_shader_list;

      case RESOURCE_IMAGE_LOADER:
         return &image_loader_list;

      case RESOURCE_POLYGON_LOADER:
         return &poly_loader_list;

      case RESOURCE_OBJECT_LOADER:
         return &object_loader_list;

      default:
         return resource_manager::get_resource_object(type);
   }

}


/* *************************************************************
************************************************************* */
void frame_manager::update(float current_time) {

   resource_type *dtr;

   resource_manager::update(current_time);

   if (!DYNAMIC_LOADER)
      return;

   global_statistic_object_count = 0;
   global_statistic_texture_count = 0;
   global_statistic_material_count = 0;
   global_statistic_uv_count = 0;

   for (dtr=(resource_type *)object_list.head; dtr; dtr=(resource_type *)dtr->next)
      dtr->update(current_time);

   for (dtr=(resource_type *)material_list.head; dtr; dtr=(resource_type *)dtr->next)
      dtr->update(current_time);

   for (dtr=(resource_type *)texture_list.head; dtr; dtr=(resource_type *)dtr->next)
      dtr->update(current_time);

   for (dtr=(resource_type *)uvcoord_list.head; dtr; dtr=(resource_type *)dtr->next)
      dtr->update(current_time);

   for (dtr=(resource_type *)uvwcoord_list.head; dtr; dtr=(resource_type *)dtr->next)
      dtr->update(current_time);
}

