

#include <string.h>

#include "sphrtype.h"
#include "gamegine.h"


/* *************************************************************
************************************************************* */
int colsphere::read_data(FILE *infile) {

   int i;

   fscanf(infile, "%f %f %f", &center[0], &center[1], &center[2]);
   fscanf(infile, "%f", &radius);
   fscanf(infile, "%d %d", &level, &count);

   if (index)
      delete [] index;

   index = new int[count];

   for (i=0; i<count; i++) {
      fscanf(infile, "%d", &index[i]);
      index[i]--;
   }

   return 1;
}


/* *************************************************************
************************************************************* */
int colsphere::write_data(FILE *outfile) {

   int i;

   fprintf(outfile, "%f %f %f\n", center[0], center[1], center[2]);
   fprintf(outfile, "%f\n", radius);
   fprintf(outfile, "%d %d ", level, count);

   for (i=0; i<count; i++)
      fprintf(outfile, "%d ", index[i]+1);

   fprintf(outfile, "\n");
   return 1;
}


/* *************************************************************
   normal[] is defaulted to reverse direction of sphr
************************************************************* */
int colsphere::collision(colsphere *sphrlist, polygon_object_type *pot, char *findex, float sphr_radius, float *sphr_center, float *pos, float *normal, int lvl) {

   float temp, temp2;
   vector3f v;
   int i, j;

   temp = radius + sphr_radius;
   subeqarray3(v, sphr_center, center);
   temp2 = dotproduct3(v,v);

   if (temp*temp < temp2)
      return 0;

   if (level < lvl) {
      if (temp2 < CORRECT) {
         copyarray3(pos, center);
         pos[3] = radius;
         return -1;
      }

      temp2 = (float)(1.0/sqrt(temp2));
      normal[0] = v[0]*temp2;
      normal[1] = v[1]*temp2;
      normal[2] = v[2]*temp2;

      pos[0] = center[0] + normal[0]*radius;
      pos[1] = center[1] + normal[1]*radius;
      pos[2] = center[2] + normal[2]*radius;
      return 1;
   }

   // subsphere checks
   if (level) {
      for (i=0; i<count; i++)
         if (sphrlist[index[i]].collision(sphrlist, pot, findex, sphr_radius, sphr_center, pos, normal, lvl))
            return 1;

      return 0;
   }

   // face checks
   for (i=0; i<count; i++) {
      j = index[i];
      if (findex[j])
         continue;

      if (sphere_poly_intersect(sphr_radius, sphr_center, j, pot, pos, normal))
         return 1;

      findex[j] = 1;
   }

   return 0;
}


/* *************************************************************
************************************************************* */
int colsphere::sphere_poly_intersect(float sphr_radius, float *sphr_center, int face, polygon_object_type *pot, float *pos, float *normal) {

   float planedotcenter;
   float *fnormal;
   int *edgelist;
   int status;
   int i, j;
   vector3f v;
   vector4f *vlist;
   face_type *flist;

   vlist = pot->vlist;
   flist = &pot->flist[face];

   fnormal = flist->normal;

   // plane intersect sphere ???
   planedotcenter = dotproduct3(fnormal, sphr_center) + fnormal[3];
   if (planedotcenter*planedotcenter > sphr_radius*sphr_radius)
      return 0;

   status = 0;

   edgelist = flist->edgeptr;

   // does sphere contain polygon verticies ?
   for (i=0; i<flist->polynum; i++)
      if (query_point_sphere_intersect(vlist[edgelist[i]], sphr_center, sphr_radius)) {
         status = 1;
         break;
      }

   if (!status) {
      // does polygon edge intersect sphere ?
      for (i=1, j=0; i<flist->polynum; i++, j++) {
         subeqarray3(v, vlist[edgelist[j]], vlist[edgelist[i]]);
         if (query_lineseg_sphere_intersect(sphr_radius, sphr_center, vlist[edgelist[i]], v)) {
            status = 1;
            break;
         }

      }

      if (!status) {
         subeqarray3(v, vlist[edgelist[0]], vlist[edgelist[j]]);
         status = query_lineseg_sphere_intersect(sphr_radius, sphr_center, vlist[edgelist[j]], v);

         if (!status) {
            // does polygon "contain" sphere?
//            t = -planedotcenter / dotproduct3(ftr->normal, ftr->normal);    // note - normal is normalized

            pos[0] = sphr_center[0] - planedotcenter * fnormal[0];
            pos[1] = sphr_center[1] - planedotcenter * fnormal[1];
            pos[2] = sphr_center[2] - planedotcenter * fnormal[2];

            if (query_point_in_cpoly(pos, fnormal, flist->polynum, vlist, edgelist)) {
               copyarray3(normal, fnormal);
               return 1;
            }

            return 0;
         }

      }

   }

   pos[0] = sphr_center[0] - planedotcenter * fnormal[0];
   pos[1] = sphr_center[1] - planedotcenter * fnormal[1];
   pos[2] = sphr_center[2] - planedotcenter * fnormal[2];

   copyarray3(normal, fnormal);
   return 1;
}


/* *************************************************************
   "ray" is not normalized - length of lineseg
************************************************************* */
int colsphere::lineseg_collision(colsphere *sphrlist, polygon_object_type *pot,
                                 char *findex, float *midpt, float mid_radius, float *ray,
                                 float *start, float *end, float *pos, float *normal, int *ret, float *t, int lvl) {

   float temp;
   vector4f v;
   float *fnormal;
   int i, j, k;
   face_type *flist;
   vector4f *vlist;

   // check sphere vs sphere
   temp = radius + mid_radius;
   subeqarray3(v, midpt, center);

   if (temp*temp < dotproduct3(v,v))
      return 0;

   // check line vs sphere
   if (!query_point_sphere_intersect(end, center, radius) &&
//       !query_point_sphere_intersect(start, center, radius) &&        // covered in query_lineseg_sphere_intersect
       !query_lineseg_sphere_intersect(radius, center, start, ray))
      return 0;

   // hack - should really find point of intersection and normal on surface
   if (level < lvl) {
      copyarray3(pos, start);
      
      normal[0] = -ray[0];
      normal[1] = -ray[1];
      normal[2] = -ray[2];
      normalize3(normal);      
      *t = 0;

      return *ret = 1;
   }

   // subsphere checks
   if (level) {
      for (i=0; i<count; i++)
         sphrlist[index[i]].lineseg_collision(sphrlist, pot, findex, midpt, mid_radius, ray, start, end, pos, normal, ret, t, lvl);

      return *ret;
   }

   // face checks
   for (i=0, k=-1, flist = pot->flist, vlist = pot->vlist; i<count; i++) {
      j = index[i];

      if (findex[j])
         continue;

      findex[j] = 1;
      fnormal = flist[j].normal;

      if (lineseg_plane_intersect(fnormal, start, end, ray, v, &temp) && (!*ret || temp < *t) &&
          query_point_in_cpoly(v, fnormal, flist[j].polynum, vlist, flist[j].edgeptr)) {

         *ret = 1;
	 *t = temp;
         k = j;
         copyarray3(pos, v);
      }

   }
   
   if (k == -1)
      return 0;
      
  copyarray3(normal, flist[k].normal);
  return 1;
}


/* *************************************************************
   "ray" is normalized
************************************************************* */
int colsphere::line_collision(colsphere *sphrlist, polygon_object_type *pot,
                              char *findex, float *pt, float *ray, float *pos, float *normal, int *ret, float *t, int lvl) {

   float temp;
   int i, j, k;
   vector4f v;
   face_type *flist;
   vector4f *vlist;
   float *fnormal;

   if (level < lvl) {
      if (line_sphere_intersectn(radius, center, pt, ray, &temp) && (!*ret || temp < *t) && temp > -CORRECT) {

         pos[0] = pt[0] + temp*ray[0];
         pos[1] = pt[1] + temp*ray[1];
         pos[2] = pt[2] + temp*ray[2];

         subeqarray3(normal, pos, center);
         normalize3(normal);
	 *t = temp;
	 return *ret = 1;
      }
      
      return 0;
   }

   // check line vs sphere
   if (!query_line_sphere_intersectn(radius, center, pt, ray))
      return 0;

   // subsphere checks
   if (level) {
      for (i=0; i<count; i++)
         sphrlist[index[i]].line_collision(sphrlist, pot, findex, pt, ray, pos, normal, ret, t, lvl);

      return *ret;
   }

   // face checks
   for (i=0, k=-1, flist = pot->flist, vlist = pot->vlist; i<count; i++) {
      j = index[i];

      if (findex[j])
         continue;

      findex[j] = 1;
      fnormal = flist[j].normal;

      if (line_plane_intersect(fnormal, pt, ray, v, &temp) && temp > -CORRECT && (!*ret || temp < *t) &&
          query_point_in_cpoly(v, fnormal, flist[j].polynum, vlist, flist[j].edgeptr)) {

         *ret = 1;
         *t = temp;
         k = j;
         copyarray3(pos, v);
      }

   }
   
   if (k == -1)
      return 0;
      
  copyarray3(normal, flist[k].normal);
  return 1;
}


/* *****************************************************************************
***************************************************************************** */
spheretype::spheretype(int count) : datatype() {

   init();

   sblock = new sphere_block;
   sblock->countsphere = count;
   sblock->slist = new colsphere[count];
}


/* *****************************************************************************
***************************************************************************** */
void spheretype::update(float current_time) {

   if (!(statusflag & STATUSFLAG_LOADABLE) || !(statusflag & STATUSFLAG_LOADED))
      return;

   if (!(statusflag & STATUSFLAG_TIME_UPDATE)) {
      delete sblock;
      sblock = NULL;
      statusflag &= ~STATUSFLAG_LOADED;
      return;
   }

   global_statistic_sphere_count++;
   access_time = current_time;
   statusflag &= ~STATUSFLAG_TIME_UPDATE;
}


/* *************************************************************
************************************************************* */
void spheretype::replace_data(void *subresource) {

   if (sblock)
      delete sblock;

   sblock = (sphere_block *)subresource;
}


/* *****************************************************************************
***************************************************************************** */
void *spheretype::query_data() {

   file_loader *loader;

   if (!(statusflag & STATUSFLAG_LOADED)) {
      if (!(statusflag & STATUSFLAG_LOADABLE))
         return sblock;

      loader = (file_loader *)global_resource_manager->find_resource_object(RESOURCE_POLYGON_LOADER, dataname.string, NULL);

      if (!loader)
         return NULL;

      loader->read_data();
      loader->extract(FILETYPE_SPHERE, altname.string);
      loader->cleanup();
   }

   statusflag |= STATUSFLAG_TIME_UPDATE;
   return sblock;
}


/* *************************************************************
************************************************************* */
float spheretype::query_radius() {

   return (!sblock || !sblock->slist) ? 0.0f : sblock->slist[0].radius;
}

/* *************************************************************
************************************************************* */
void spheretype::init() {

   sblock = NULL;
   statusflag = STATUSFLAG_VOID;
}


/* *************************************************************
************************************************************* */
void spheretype::dest() {

   if (sblock)
      delete sblock;

   init();
}


/* *************************************************************
************************************************************* */
int spheretype::read_data(char *filename) {

   FILE *infile;                        // file pointer
   int i;                               // looping var

   dest();

   if (!find_file(filename, "r", OBJECT_PATH.string, (char)PLATFORM_SLASH, NULL, &infile))
      return 0;

   altname.stringcpy(filename);

   sblock = new sphere_block;

   fscanf(infile, "%d", &sblock->countsphere);

   if (sblock->countsphere) {
      sblock->slist = new colsphere[sblock->countsphere];

      for (i=0; i<sblock->countsphere; i++)
         sblock->slist[i].read_data(infile);
   }

   fclose(infile);

   statusflag |= STATUSFLAG_LOADABLE | STATUSFLAG_LOADED;
   return 1;
}


/* *************************************************************
************************************************************* */
int spheretype::write_data(char *filename) {

   FILE *outfile;                       // file pointer
   int i;                               // looping var
   
   if (!sblock) {
      sprintf(perror_buffer, "No spherical data for \"%s\"...\n", filename);
      pprintf(perror_buffer);
      return 0;
   }

   outfile = fopen(filename, "w");

   if (!outfile) {
      sprintf(perror_buffer, "Could not access .col %s....\n", filename);
      pprintf(perror_buffer);
      return 0;
   }

   fprintf(outfile, "%d\n\n", sblock->countsphere);

   if (!sblock->countsphere) {
      fclose(outfile);
      return 1;
   }

   for (i=0; i<sblock->countsphere; i++)
      sblock->slist[i].write_data(outfile);

   fclose(outfile);
   return 1;
}
