

#include <stdlib.h>
#include <string.h>

#ifdef WIN32

#include "winsnd.h"
#include "user_io_w32.h"

#define USER_IO_TYPE user_io_win

#else

#include "soundman.h"
#include "user_io_x.h"
#define USER_IO_TYPE user_io_X

#endif

#include "gamegine.h"
#include "spb_load.h"
#include "anineutr.h"

#include "pstring.h"
#include "polygon.h"


unsigned int *hi_red  = NULL;
unsigned int *lut_red = NULL;

unsigned int *hi_green  = NULL;
unsigned int *lut_green = NULL;

unsigned int *hi_blue  = NULL;
unsigned int *lut_blue = NULL;

unsigned char *byte_mult = NULL;

vector4uc ogl_red   = { 255, 0, 0, 255 };
vector4uc ogl_green = { 0, 255, 0, 255 };
vector4uc ogl_blue  = { 0, 0, 255, 255 };

int game_fogflag[MAXGROUND] = { 0, 0, 0 };
float game_fog_start = 0;
float game_fog_end = 0;
vector4f game_fog_color = { 1.0f, 1.0f, 1.0f, 0.0f };

string_type FILENAME_FONT("");
int FONTTYPE = FONT_VECTOR;

gengine proc;
gamegod_type *complex = NULL;


/* **************************************************
************************************************** */
void dest_gamelut() {

   if (hi_red) {
      delete [] hi_red;
      hi_red = NULL;
   }

   if (hi_green) {
      delete [] hi_green;
      hi_green = NULL;
   }

   if (hi_blue) {
      hi_blue = NULL;
      delete [] hi_blue;
   }

   if (byte_mult) {
      delete [] byte_mult;
      byte_mult = NULL;
   }

}


/* **************************************************
************************************************** */
void init_gamelut() {

   int i, j;

   dest_gamelut();
   
   hi_red   = new unsigned int[1024];
   hi_green = new unsigned int[1024];
   hi_blue  = new unsigned int[1024];

   lut_red   = &hi_red[128];
   lut_green = &hi_green[128];
   lut_blue  = &hi_blue[128];

   // video lut's
   memset(hi_red, 0, 128<<2);
   memset(hi_green, 0, 128<<2);
   memset(hi_blue, 0, 128<<2);

#ifdef SGI
   for (i=0; i<256; i++) {
      lut_red[i]   = i<<24;
      lut_green[i] = i<<16;
      lut_blue[i]  = i<<8;
   }

   for (i=256; i<896; i++) {
      lut_red[i]   = 0xff000000;
      lut_green[i] = 0x00ff0000;
      lut_blue[i]  = 0x0000ff00;
   }
#else
   for (i=0; i<256; i++) {
      lut_red[i]   = i;
      lut_green[i] = i<<8;
      lut_blue[i]  = i<<16;
   }

   for (i=256; i<896; i++) {
      lut_red[i]   = 0x000000ff;
      lut_green[i] = 0x0000ff00;
      lut_blue[i]  = 0x00ff0000;
   }
#endif

   byte_mult = new unsigned char[65536];

   for (i=0; i<256; i++)
      for (j=0; j<256; j++)
        byte_mult[(i<<8) + j] = (unsigned char)((i*j)/255);
}


/* *************************************************************
************************************************************* */
void texturepostCB(texbase *ptr) {

   ptr->pre_process_texture();
}


/* **************************************************
************************************************** */
void init_gamegine(gamegod_type *god) {

   loader_list_type *ptr;

   if (logfile)
      fclose(logfile);
      
   // setup logfile
   logfile = fopen("logfile", "w");

   if (!logfile)
      pprintf("Warning: Could not create game logfile...\n");

   if (!god->boss) {
      pprintf("Error: game_manager not set... Aborting...\n");

      if (logfile) {
         fclose(logfile);
         logfile = NULL;
      }

      exit(0);
   }

   // init luts
   init_lut();
   init_gamelut();

   god->init();

   // init resource manager
   global_resource_manager = god->boss;
   global_resource_manager->init(1);
   ((frame_manager *)global_resource_manager)->set_postprocessCBtexture((void *)texturepostCB);

   // register object loaders
   global_resource_manager->register_resource_object(RESOURCE_POLYGON_LOADER, new spg2_loader);
   global_resource_manager->register_resource_object(RESOURCE_POLYGON_LOADER, new spb_loader);

   while (god->resource_loader_manager.head) {
      god->resource_loader_manager.remove(ptr = (loader_list_type *)god->resource_loader_manager.head);
      global_resource_manager->register_resource_object(ptr->type, ptr->ob);
      delete ptr;
   }

   if (god->soundmanager) {
#ifdef WIN32
      ((soundman_win *)god->soundmanager)->set_win(god->win);
#endif
      god->soundmanager->init();
   }

   if (god->userinput)
      ((USER_IO_TYPE *)god->userinput)->set_win(god->win);

   // init graphics
   if (god->gfx)
      god->gfx->gfxInit();

   god->win = NULL;
   god->boss = NULL;

   if (complex)
      delete complex;

   complex = god;

   complex->reset();
}


/* **************************************************
************************************************** */
void shutdown_gamegine() {

   if (global_resource_manager)
      global_resource_manager->reset();

   if (complex) {
      complex->dest();
      delete complex;
      complex = NULL;
   }

   if (global_resource_manager) {
      delete global_resource_manager;
      global_resource_manager = NULL;
   }

   dest_gamelut();
   dest_lut();

   if (logfile) {
      pprintf("Closing down gamegine... Have a nice day :)\n");
      fclose(logfile);
      logfile = NULL;
   }

}


/* **************************************************
************************************************** */
void parsetoken(char *buffer, string_parser *tokenizer, void *callback) {

   char token[MAXSTRLEN];
   
   switch (buffer[0]) {

      case 'a':
         if (buffer[1] == 0) {
            DEFLST = tokenizer->get_token(token, MAXSTRLEN);
            if (DEFLST)
               SDEFLST.stringcpy(token);

            return;
         }

         break;

      case 'b':
         if (buffer[1] == 0) {
            if (tokenizer->get_token(token, MAXSTRLEN))
               back = (float)-atof(token);

            return;
         }

         break;

      case 'd':
         if (buffer[1] == 0) {
            DYNAMIC_LOADER = 1;
            return;
         }

         break;

      case 'f':
         if (!strcmp(buffer, "fog")) {
            if (tokenizer->get_token(token, MAXSTRLEN)) {
               lower_case(token);

               if (!strcmp(token, "foreground"))
                  game_fogflag[FOREGROUND] = 1;
               else if (!strcmp(token, "midground"))
                  game_fogflag[MIDGROUND] = 1;
               else if (!strcmp(token, "background"))
                  game_fogflag[BACKGROUND] = 1;
            }
	    
            return;
         }
	 
         if (!strcmp(buffer, "fog_start")) {

            if (tokenizer->get_token(token, MAXSTRLEN))
               game_fog_start = (float)atof(token);

            return;
         }
	 
         if (!strcmp(buffer, "fog_end")) {

            if (tokenizer->get_token(token, MAXSTRLEN))
               game_fog_end = (float)atof(token);

            return;
         }
	 
         if (!strcmp(buffer, "fog_color")) {

            if (tokenizer->get_token(token, MAXSTRLEN))
               game_fog_color[0] = (float)(atof(token)/255.0);
            if (tokenizer->get_token(token, MAXSTRLEN))
               game_fog_color[1] = (float)(atof(token)/255.0);
            if (tokenizer->get_token(token, MAXSTRLEN))
               game_fog_color[2] = (float)(atof(token)/255.0);

            return;
         }
	 
         break;	 
	 	 
      case 'j':
         if (!strcmp(buffer, "joystick")) {
            CONTROLLER = CONTROLJOYSTICK;
            return;
         }

         break;

      case 'k':
         if (!strcmp(buffer, "keyboard_decoder")) {
            if (tokenizer->get_token(token, MAXSTRLEN))
               FILENAME_KEYBOARD_DECODER.stringcpy(token);

            return;
         }
		  
         if (!strcmp(buffer, "keyboard")) {
            CONTROLLER = CONTROLKEYBOARD;
            return;
         }

         break;

      case 'm':
         if (!strcmp(buffer, "mouse")) {
            CONTROLLER = CONTROLMOUSE;
            return;
         }

         if (!strcmp(buffer, "music_volume")) {
            tokenizer->get_token(token, MAXSTRLEN);
            global_music_volume = atoi(token);
            return;
         }
	 
         break;

      case 'o':
         if (!strcmp(buffer, "opengl")) {
            RENDER_MODE = RENDERER_OPENGL;
            return;
         }

         break;

      case 'r':
         if (!strcmp(buffer, "raster")) {
            FONTTYPE = FONT_RASTER;
            tokenizer->get_token(token, MAXSTRLEN);
            FILENAME_FONT.stringcpy(token);
            return;
         }

         break;

      case 's':                   // silent to file
         if (!buffer[1]) {
            SILENT = 1;
            return;
         }

         if (!strcmp(buffer, "sfx_volume")) {
            tokenizer->get_token(token, MAXSTRLEN);
            global_sfx_volume = atoi(token);
            return;
         }
	 
         break;

      case 'v':                   // verbose mode
         if (!strcmp(buffer, "vector")) {
            FONTTYPE = FONT_VECTOR;
            tokenizer->get_token(token, MAXSTRLEN);
            FILENAME_FONT.stringcpy(token);
            return;
         }
	 
         if (!strcmp(buffer, "voice_volume")) {
            tokenizer->get_token(token, MAXSTRLEN);
            global_voice_volume = atoi(token);
            return;
         }
	 
         if (buffer[1] == 0) {
            SILENT = 0;
            return;
         }

         break;

/*
      case 'w':
         if (!strcmp(buffer, "winx")) {
            tokenizer->get_token(token, MAXSTRLEN);
            WINX = atoi(token);
            WINY = (WINX*3)/4;
            break;
         }

         break;
*/

      default:
         break;
   }

   if (callback)
      ((void (*)(char *, string_parser *))callback)(buffer, tokenizer);
}


/* **************************************************
************************************************** */
int parseinput(char *filename, void *callback) {

   FILE *infile;
   string_parser tokenizer;
   string_type buffer;
   char token[MAXSTRLEN];

   // parse setup file
   infile = fopen(filename, "r");

   if (!infile)
      return 0;
      
   buffer.stringcat(" ");

   while (get_token(infile, token)) {
      buffer.stringcat(token);
      buffer.stringcat(" ");
   }

   fclose(infile);

   tokenizer.set_string(buffer.string);

   while (tokenizer.get_token(token, MAXSTRLEN))
      if (token[0] == '-') {
         lower_case(token);
         parsetoken(token+1, &tokenizer, callback);
      }
      
   return 1;
}


/* **************************************************
************************************************** */
void parseinput(int argc, char **argv, void *callback) {

   string_parser tokenizer;
   string_type buffer;
   char token[MAXSTRLEN];
   int i;
   
   buffer.stringcat(" ");

   // parse command line args
   for (i=0; i<argc; i++) {
      buffer.stringcat(argv[i]);
      buffer.stringcat(" ");
   }

   tokenizer.set_string(buffer.string);

   while (tokenizer.get_token(token, MAXSTRLEN))
      if (token[0] == '-') {
         lower_case(token);
         parsetoken(token+1, &tokenizer, callback);
      }
      
}


/* **************************************************
************************************************** */
int writeinput(char *filename) {

   FILE *outfile;

   outfile = fopen(filename, "w");

   if (!outfile)
      return 0;
      
   if (DEFLST)
      fprintf(outfile, "-a %s\n", SDEFLST.string);            

   fprintf(outfile, "-back %f\n", -back);

   if (DYNAMIC_LOADER)
      fprintf(outfile, "-dynamic\n");

   if (game_fogflag[FOREGROUND])
      fprintf(outfile, "-fog foreground\n");

   if (game_fogflag[MIDGROUND])
      fprintf(outfile, "-fog midground\n");

   if (game_fogflag[BACKGROUND])
      fprintf(outfile, "-fog background\n");

   fprintf(outfile, "-fog_start %f\n", game_fog_start);
   fprintf(outfile, "-fog_end %f\n", game_fog_end);
   fprintf(outfile, "-fog_color %d %d %d\n", (int)(game_fog_color[0]*255), (int)(game_fog_color[1]*255), (int)(game_fog_color[2]*255));

   if (CONTROLLER == CONTROLJOYSTICK)
      fprintf(outfile, "-joystick\n");
   else if (CONTROLLER == CONTROLKEYBOARD)
      fprintf(outfile, "-keyboard\n");
   else if (CONTROLLER == CONTROLMOUSE)
      fprintf(outfile, "-mouse\n");

   fprintf(outfile, "-keyboard_decoder %s\n", FILENAME_KEYBOARD_DECODER.string);

   if (RENDER_MODE == RENDERER_OPENGL)
      fprintf(outfile, "-opengl\n");

   if (FONTTYPE == FONT_RASTER)
      fprintf(outfile, "-raster %s\n", FILENAME_FONT.string);
   else if (FONTTYPE == FONT_VECTOR)
      fprintf(outfile, "-vector %s\n", FILENAME_FONT.string);

   if (SILENT)
      fprintf(outfile, "-silent\n");
   else
      fprintf(outfile, "-verbose\n");
   
   fprintf(outfile, "-sfx_volume %d\n", global_sfx_volume);
   fprintf(outfile, "-music_volume %d\n", global_music_volume);
   fprintf(outfile, "-voice_volume %d\n", global_voice_volume);

   fclose(outfile);
   return 1;
}


/* **************************************************************
************************************************************** */
gameatom *load_gameatom(char *filename, dbl_llist_manager *hiearchy_manager) {

   atom_list_type *atr;
   geneological_type ani_param;
   gameatom *ptr;

   ptr = new gameatom;

   if (!ptr->read_data(filename)) {
      delete ptr;
      return NULL;
   }

   hiearchy_manager->append(atr = new atom_list_type, NULL);
   atr->htree = ptr;

   ani_param.tree = atr;
   ani_param.parent = NULL;
   ptr->preprocess(&ani_param);

   return ptr;
}


/* *************************************************************
************************************************************* */
gameatom *load_game_object(char *filename, dbl_llist_manager *hiearchy_manager) {

   atom_list_type *atr;
   geneological_type ani_param;
   gameatom *ptr;
   antineutron *dtr;
   polygon *ob;

   ptr = new gameatom;
   ptr->name.stringcpy(filename);
   ptr->create_link(dtr = new antineutron);

   dtr->name.stringcpy(filename);
   ob = (polygon *)dtr->get_render_object();
   ob->filename.stringcpy(filename);
   ob->colorname.stringcpy(filename);
   ob->texname2.stringcpy(filename);
   ob->mctype.set_master(GOURAUD);
   ob->mcinfo.mask_or(CITEXTURE);

   hiearchy_manager->append(atr = new atom_list_type, NULL);
   atr->htree = ptr;

   ani_param.tree = atr;
   ani_param.parent = NULL;
   ptr->preprocess(&ani_param);

   return ptr;
}

