

#include <stdlib.h>
#include <string.h>

#include "pstring.h"
#include "light.h"
#include "global.h"


/* *************************************************************
************************************************************* */
void light::set_ambient(int *ka, float *ambient, int *lum) {

   ambient[0] = Ia[0]*ka[0] + lum[0];
   ambient[1] = Ia[1]*ka[1] + lum[1];
   ambient[2] = Ia[2]*ka[2] + lum[2];
}


/* *************************************************************
   This procedure calculates the intensity of light at a certain point
   * this function also takes into account the position of the light
     source, and shadows the appropriate sides :)

************************************************************* */
void light::lighting(float *surface, float *gnormal, float *rcolor,
                      shadetype *shade, float *mIp, float *vlight, float dist2surface) {

   int i;                                                  // looping var
   float NL;                                               // normal.light
   float fatten;                                           // light atten
   vector4f R;                                             // R vector
   float length;

   NL = dotproduct3(gnormal, vlight);   // normal . light

   if (NL <= 0)                         // backface
      return;

   for (i=0; i<3; i++) {
      R[i] = NL * gnormal[i];           // calculate reflection vector
      R[i] += R[i] - vlight[i];
   }

   length = fmagnitude3(surface);

   R[0] = -dotproduct3(R, surface)/length;    // cos() for specular

   R[3] = (R[0] < 0.0f) ? 0.0f : (float)pow(R[0], shade->specn);

   if (fattflag) {
      fatten = length + dist2surface;
      fatten = fatt[0] + (fatt[1] + fatt[2]*fatten)*fatten;    // calculate fatt
      fatten = (fatten < 1.0f) ? 1.0f : 1.0f/fatten;
      R[3] *= fatten;
      NL   *= fatten;
   }
                                                      // assign pixel color
   rcolor[0] += mIp[0]*(shade->kp[0]*NL + shade->ks[0]*R[3]);
   rcolor[1] += mIp[1]*(shade->kp[1]*NL + shade->ks[1]*R[3]);
   rcolor[2] += mIp[2]*(shade->kp[2]*NL + shade->ks[2]*R[3]);
}


/* *************************************************************
************************************************************* */
int light::parse(FILE *infile, char *token) {

   switch (token[0]) {

      case 'a':
         if (!strcmp(token, TOKEN_AMBIENT_STR)) {
            get_token(infile, token);
            Ia[0] = (float)atof(token);
            get_token(infile, token);
            Ia[1] = (float)atof(token);
            get_token(infile, token);
            Ia[2] = (float)atof(token);

            return 1;
         }

         break;

      case 'd':

         if (!strcmp(token, TOKEN_DIFFUSE_STR)) {
            get_token(infile, token);
            Ip[0] = (float)atof(token);
            get_token(infile, token);
            Ip[1] = (float)atof(token);
            get_token(infile, token);
            Ip[2] = (float)atof(token);

            return 1;
         }

         break;

      case 'f':
         if (!strcmp(token, TOKEN_FATT_STR)) {
            get_token(infile, token);
            fatt[0] = (float)atof(token);
            get_token(infile, token);
            fatt[1] = (float)atof(token);
            get_token(infile, token);
            fatt[2] = (float)atof(token);

            fattflag = 1;

            return 1;
         }

         break;

      default:
         break;
   }

   return eye::parse(infile, token);
}


/* *************************************************************
************************************************************* */
int far_light::query_whatwasi(int type) {

   if (far_light::query_whatami() == type)
      return 1;

   return eye::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
void far_light::preprocess(void *data) {

   light::preprocess(data);

   normalize3(vpn);
}


/* *************************************************************
************************************************************* */
int far_light::dump_frame(FILE *outfile) {

    char token[3][MAXSTRLEN];

    fprintf(outfile, "%s {\n", TOKEN_FAR_STR);

    float2char(-vpn[0], token[0]);
    float2char(-vpn[1], token[1]);
    float2char(-vpn[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_DIRECTION_STR, token[0], token[1], token[2]);

    float2char(Ia[0], token[0]);
    float2char(Ia[1], token[1]);
    float2char(Ia[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_AMBIENT_STR, token[0], token[1], token[2]);

    float2char(Ip[0], token[0]);
    float2char(Ip[1], token[1]);
    float2char(Ip[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_DIFFUSE_STR, token[0], token[1], token[2]);

    float2char(fatt[0], token[0]);
    float2char(fatt[1], token[1]);
    float2char(fatt[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_FATT_STR, token[0], token[1], token[2]);

    fprintf(outfile, "\t%s %d\n", TOKEN_FRAME_STR, frame);
    fprintf(outfile, "}\n");
    return 1;
}


/* *************************************************************
        parallel project... (no div by z)
************************************************************* */
void far_light::map2screen(float *v) {

   v[0]  = mscale * (v[0] - vrc[0]);
   v[1]  = mscale * (v[1] - vrc[2]);
}


/* *************************************************************
   Scale image from the window
************************************************************* */
void far_light::screen2map(float *v) {

   v[0]     = (v[0]*imscale + vrc[0]);
   v[1]     = (v[1]*imscale + vrc[2]);
}


/* *************************************************************
************************************************************* */
int far_light::parse(FILE *infile, char *token) {

   switch (token[0]) {

      case 'd':
         if (!strcmp(token, TOKEN_DIRECTION_STR)) {
            get_token(infile, token);
            vpn[0] = (float)-atof(token);
            get_token(infile, token);
            vpn[1] = (float)-atof(token);
            get_token(infile, token);
            vpn[2] = (float)-atof(token);

            return 1;
         }

         break;

      default:
         break;
   }

   return light::parse(infile, token);
}


/* *************************************************************
   This procedure calculates the intensity of light at a certain point
   * this function also takes into account the position of the light
     source, and shadows the appropriate sides :)
************************************************************* */
void far_light::intensity(float* surface, float *gnormal, float *rcolor,
                      eye *parm, shadetype *shade, int source_id) {

   lighting(surface, gnormal, rcolor, shade, Ip, transvec, 0);
}


/* *************************************************************
************************************************************* */
void far_light::xform(eye *parm) {

   matvecmultv(parm->transform, vpn, transvec);
   transvec[3] = 0;
}


/* *************************************************************
************************************************************* */
int point_light::query_whatwasi(int type) {

   if (point_light::query_whatami() == type)
      return 1;

   return eye::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int point_light::dump_frame(FILE *outfile) {

    char token[3][MAXSTRLEN];

    fprintf(outfile, "%s {\n", TOKEN_POINT_STR);

    float2char(location[0], token[0]);
    float2char(location[1], token[1]);
    float2char(location[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_LOCATION_STR, token[0], token[1], token[2]);

    float2char(Ia[0], token[0]);
    float2char(Ia[1], token[1]);
    float2char(Ia[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_AMBIENT_STR, token[0], token[1], token[2]);

    float2char(Ip[0], token[0]);
    float2char(Ip[1], token[1]);
    float2char(Ip[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_DIFFUSE_STR, token[0], token[1], token[2]);

    float2char(fatt[0], token[0]);
    float2char(fatt[1], token[1]);
    float2char(fatt[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_FATT_STR, token[0], token[1], token[2]);

    fprintf(outfile, "\t%s %d\n", TOKEN_FRAME_STR, frame);
    fprintf(outfile, "}\n");
    return 1;
}


/* *************************************************************
   Scale image to the window
************************************************************* */
void point_light::map2screen(float *v) {

   v[0]  = -mscale * (v[0]/v[2] + vrc[0]);
   v[1]  = -mscale * (v[1]/v[2] + vrc[2]);
}


/* *************************************************************
   Scale image from the window
************************************************************* */
void point_light::screen2map(float *v) {

   v[1]     = -(v[1]*imscale + vrc[2]) * v[2];
   v[0]     = -(v[0]*imscale + vrc[0]) * v[2];
}


/* *************************************************************
   This procedure calculates the intensity of light at a certain point
   * this function also takes into account the position of the light
     source, and shadows the appropriate sides :)
************************************************************* */
void point_light::intensity(float* surface, float *gnormal, float *rcolor,
                      eye *parm, shadetype *shade, int source_id) {

   vector3f vlight;
   float    dist2surface;

   subeqarray3(vlight, transloc, surface);
   dist2surface = fnormalize3(vlight);

   lighting(surface, gnormal, rcolor, shade, Ip, vlight, dist2surface);
}


/* *************************************************************
************************************************************* */
void point_light::xform(eye *parm) {

   matvecmulto(parm->transform, location, transloc);
   transloc[3] = 1;
}


/* *************************************************************
************************************************************* */
int simple_far_light::dump_frame(FILE *outfile) {

    char token[3][MAXSTRLEN];

    fprintf(outfile, "%s {\n", TOKEN_SIMPLE_FAR_STR);

    float2char(-vpn[0], token[0]);
    float2char(-vpn[1], token[1]);
    float2char(-vpn[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_DIRECTION_STR, token[0], token[1], token[2]);

    float2char(Ia[0], token[0]);
    float2char(Ia[1], token[1]);
    float2char(Ia[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_AMBIENT_STR, token[0], token[1], token[2]);

    float2char(Ip[0], token[0]);
    float2char(Ip[1], token[1]);
    float2char(Ip[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_DIFFUSE_STR, token[0], token[1], token[2]);

    fprintf(outfile, "\t%s %d\n", TOKEN_FRAME_STR, frame);
    fprintf(outfile, "}\n");

    return 1;
}


/* *************************************************************
************************************************************* */
void simple_far_light::lighting(float *surface, float *gnormal, float *rcolor,
        shadetype *shade, float *mIp, float *vlight, float dist2surface) {

   simple_light;
}


/* *************************************************************
************************************************************* */
int simple_point_light::dump_frame(FILE *outfile) {

    char token[3][MAXSTRLEN];

    fprintf(outfile, "%s {\n", TOKEN_SIMPLE_POINT_STR);

    float2char(location[0], token[0]);
    float2char(location[1], token[1]);
    float2char(location[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_LOCATION_STR, token[0], token[1], token[2]);

    float2char(Ia[0], token[0]);
    float2char(Ia[1], token[1]);
    float2char(Ia[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_AMBIENT_STR, token[0], token[1], token[2]);

    float2char(Ip[0], token[0]);
    float2char(Ip[1], token[1]);
    float2char(Ip[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_DIFFUSE_STR, token[0], token[1], token[2]);

    fprintf(outfile, "\t%s %d\n", TOKEN_FRAME_STR, frame);
    fprintf(outfile, "}\n");

    return 1;
}


/* *************************************************************
************************************************************* */
void simple_point_light::lighting(float *surface, float *gnormal, float *rcolor,
        shadetype *shade, float *mIp, float *vlight, float dist2surface) {

   simple_light;
}


/* *************************************************************
************************************************************* */
void spotlight::preprocess(void *data) {

   float temp[2];
   float i, j;
   
   light::preprocess(data);

   normalize3(vpn);

   width /= distance2plane;
   length /= distance2plane;

   tob = (texbase *)((frame_manager *)global_resource_manager)->read_tex(texname.string);

   if (!tob) {
      sprintf(perror_buffer, "Error: Cannot access light texture \"%s\"... Aborting...\n", texname.string);
      pprintf(perror_buffer);
      exit(1);
   }

   i = (float)tob->oldx;
   j = (float)tob->oldy;

   temp[0] = length/width;
   temp[1] = i/j;

   if (temp[0] > temp[1])            // reshape viewport to texture ratios;
      length = temp[1]*width;
   else
      width = length/temp[1];

   temp[0] = 0.5f*length;
   temp[1] = 0.5f*width;

   vrc[0] = -temp[0];
   vrc[1] =  temp[0];
   vrc[2] = -temp[1];
   vrc[3] =  temp[1];

   mscale = (width > length) ? MAXLDIM/width : MAXLDIM/length;
   imscale = 1.0f/mscale;

   maxx = (int)(length * mscale);
   maxy = (int)(width * mscale);

   ztscale = (i>j) ? i/(float)MAXLDIM : j/(float)MAXLDIM;      // scale from lbuff to tmap

   nper();
}


/* *************************************************************
************************************************************* */
int spotlight::parse(FILE *infile, char *token) {

   switch (token[0]) {
      case '2':
         if (!strcmp(token, TOKEN_2D_TEXTURE_STR)) {
            get_token(infile, token);
            texname.stringcpy(token);
            return 1;
         }

         break;

      case 'd':
         if (!strcmp(token, TOKEN_DISTANCE2PLANE_STR)) {
            get_token(infile, token);
            distance2plane = (float)atof(token);
            return 1;
         }

         if (!strcmp(token, TOKEN_DIRECTION_STR)) {
            get_token(infile, token);
            vpn[0] = (float)-atof(token);
            get_token(infile, token);
            vpn[1] = (float)-atof(token);
            get_token(infile, token);
            vpn[2] = (float)-atof(token);
            return 1;
         }

         break;

      case 'l':
         if (!strcmp(token, TOKEN_LENGTH_STR)) {
            get_token(infile, token);
            length = (float)atof(token);
            return 1;
         }

         break;

      case 'w':
         if (!strcmp(token, TOKEN_WIDTH_STR)) {
            get_token(infile, token);
            width = (float)atof(token);
            return 1;
         }

         break;

      default:
         break;
   }

   return light::parse(infile, token);
}


/* *************************************************************
************************************************************* */
int spotlight::init_buff(memman *control) {

   unsigned int i;

   if (lbuff == (lightbufftype *)NULL)
      lbuff = (lightbufftype *)control->pop(MM_LIGHTBUFF);

   if (!lbuff->lbuff.data)
      lbuff->lbuff.init_map(MAXLDIM, MAXLDIM);

   i = MAXLDIM*MAXLDIM;

   do {
      lbuff->lbuff.data[i-1].zdata = back;
      lbuff->lbuff.data[i-1].idbuff = 0;
      lbuff->lbuff.data[i-2].zdata = back;
      lbuff->lbuff.data[i-2].idbuff = 0;
      lbuff->lbuff.data[i-3].zdata = back;
      lbuff->lbuff.data[i-3].idbuff = 0;
      lbuff->lbuff.data[i-4].zdata = back;
      lbuff->lbuff.data[i-4].idbuff = 0;
      i -= 4;
   } while (i);

   return 1;
}


/* *************************************************************
************************************************************* */
void spotlight::xform(eye *parm) {

   vector4f temp[4];

   matvecmultv(parm->transform, vpn, transvec);
   transvec[3] = 0;

   matvecmulto(parm->transform, location, transloc);
   transloc[3] = 1;

   copymx4x4o(temp, transform);
   matmatmulto(temp, parm->Tinverse, transform);
   transform[3][0] = transform[3][1] = transform[3][2] = 0;
   transform[3][3] = 1;
   
   matmatmulto(parm->transform, Tinverse);

//   area = parm->imscale*imscale; // this->mscale = sqrt(TA/PA)

   area = (tob->maxx*tob->maxy)/(float)(maxx*maxy);
   area = parm->imscale*SQRT(area)*mscale;
}


/* *************************************************************
************************************************************* */
int beam_far::query_whatwasi(int type) {

   return (beam_far::query_whatami() == type) ? 1 : eye::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int beam_far::dump_frame(FILE *outfile) {

    char token[3][MAXSTRLEN];

    fprintf(outfile, "%s {\n", TOKEN_BEAM_FAR_STR);
    fprintf(outfile, "\t%s %s\n", TOKEN_2D_TEXTURE_STR, texname.string);

    float2char(length, token[0]);
    fprintf(outfile, "\t%s %s\n", TOKEN_LENGTH_STR, token[0]);

    float2char(width, token[0]);
    fprintf(outfile, "\t%s %s\n", TOKEN_WIDTH_STR, token[0]);

    float2char(-vpn[0], token[0]);
    float2char(-vpn[1], token[1]);
    float2char(-vpn[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_DIRECTION_STR, token[0], token[1], token[2]);

    float2char(location[0], token[0]);
    float2char(location[1], token[1]);
    float2char(location[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_LOCATION_STR, token[0], token[1], token[2]);

    float2char(Ia[0], token[0]);
    float2char(Ia[1], token[1]);
    float2char(Ia[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_AMBIENT_STR, token[0], token[1], token[2]);

    float2char(fatt[0], token[0]);
    float2char(fatt[1], token[1]);
    float2char(fatt[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_FATT_STR, token[0], token[1], token[2]);

    float2char(vup[0], token[0]);
    float2char(vup[1], token[1]);
    float2char(vup[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_UP_STR, token[0], token[1], token[2]);

    fprintf(outfile, "\t%s %d\n", TOKEN_FRAME_STR, frame);
    fprintf(outfile, "}\n");

    return 1;
}


/* *************************************************************
        parallel project... (no div by z)
************************************************************* */
void beam_far::map2screen(float *v) {

   v[0] = mscale * (v[0] - vrc[0]);
   v[1] = mscale * (v[1] - vrc[2]);
}


/* *************************************************************
   Scale image from the window
************************************************************* */
void beam_far::screen2map(float *v) {

   v[0] = (v[0]*imscale + vrc[0]);
   v[1] = (v[1]*imscale + vrc[2]);
}


/* *************************************************************
id - 0          always (transparent)
id - "+"        id of object
id - "-"        ignore (should never call... ergo never happen)

csquared can be precalculated for polygons - constant
************************************************************* */
void beam_far::intensity(float *surface, float *gnormal, float *rcolor, eye *parm,
                        shadetype *shade, int source_id) {

   vector4f pt;
   int  i, j, k, l;
   int  frac;
   int  pt2[2][2];
   float temp;
   float csquared;
   vector3f tempv;
   float cam_surf_dist;

   vector3f mIp;
   float dist2surface;

   if (!source_id)
      return;

   // from camera space to light space
   matvecmulto(transform, surface, pt);
   pt[3] = 1;

   dist2surface = -pt[2];

   if (dist2surface < CORRECT)
      return;

   if (!tob) {
      mIp[0] = mIp[1] = mIp[2] = 1.0;
      lighting(surface, gnormal, rcolor, shade, mIp, transvec, dist2surface);
      return;
   }

   map2screen(pt);

   if (pt[0] < 0 || pt[0] >= maxx || pt[1] < 0 || pt[1] >= maxy)
      return;

   csquared = dotproduct3(gnormal, transvec);

   // backface...
   if (csquared < CORRECT)
      return;

//   csquared = SQRT(csquared);

   j = (int)pt[0];
   i = (int)pt[1];

   subeqarray3(tempv, surface, parm->location);
   cam_surf_dist = fmagnitude3(tempv);

   // if spot is directly in light
   if ((unsigned int)source_id == lbuff->lbuff.pdata[i][j].idbuff) {

      tempv[0] = pt[0]*ztscale;
      tempv[1] = pt[1]*ztscale;

      temp = cam_surf_dist*area*csquared;

      tob->query_texel(tempv[0], tempv[1], mIp, frame, temp);

      lighting(surface, gnormal, rcolor, shade, mIp, transvec, dist2surface);
      return;
   }

   // if spot is adjacent to light...
   pt2[0][0] = j-1;
   pt2[1][0] = j+1;
   pt2[0][1] = i-1;
   pt2[1][1] = i+1;

   tempv[0] = pt[0]*ztscale;
   tempv[1] = pt[1]*ztscale;

   if (pt2[0][0] < 0)
      pt2[0][0] = 0;
   if (pt2[1][0] >= maxx)
      pt2[1][0] = maxx-1;
   if (pt2[0][1] < 0)
      pt2[0][1] = 0;
   if (pt2[1][1] >= maxy)
      pt2[1][1] = maxy-1;

   frac = 0;

   // box filter
   for (l=pt2[0][1]; l<pt2[1][1]; l++)
      for (k=pt2[0][0]; k<pt2[1][0]; k++)
         if ((unsigned int)source_id == lbuff->lbuff.pdata[l][k].idbuff)
            frac += 25;

   if (!frac)
      return;

   temp = cam_surf_dist*area*csquared;

   tob->query_texel(tempv[0], tempv[1], mIp, frame, temp);

   temp = (frac + 50.0f) * 0.002f;
   smultarray3(mIp, temp);

   lighting(surface, gnormal, rcolor, shade, mIp, transvec, dist2surface);
}


/* *************************************************************
************************************************************* */
int beam_point::query_whatwasi(int type) {

   if (beam_point::query_whatami() == type)
      return 1;

   return eye::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int beam_point::dump_frame(FILE *outfile) {

    char token[3][MAXSTRLEN];

    fprintf(outfile, "%s {\n", TOKEN_BEAM_POINT_STR);
    fprintf(outfile, "\t%s %s\n", TOKEN_2D_TEXTURE_STR, texname.string);

    float2char(length, token[0]);
    fprintf(outfile, "\t%s %s\n", TOKEN_LENGTH_STR, token[0]);

    float2char(width, token[0]);
    fprintf(outfile, "\t%s %s\n", TOKEN_WIDTH_STR, token[0]);

    float2char(-vpn[0], token[0]);
    float2char(-vpn[1], token[1]);
    float2char(-vpn[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_DIRECTION_STR, token[0], token[1], token[2]);

    float2char(location[0], token[0]);
    float2char(location[1], token[1]);
    float2char(location[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_LOCATION_STR, token[0], token[1], token[2]);

    float2char(Ia[0], token[0]);
    float2char(Ia[1], token[1]);
    float2char(Ia[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_AMBIENT_STR, token[0], token[1], token[2]);

    float2char(fatt[0], token[0]);
    float2char(fatt[1], token[1]);
    float2char(fatt[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_FATT_STR, token[0], token[1], token[2]);

    float2char(vup[0], token[0]);
    float2char(vup[1], token[1]);
    float2char(vup[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_UP_STR, token[0], token[1], token[2]);

    fprintf(outfile, "\t%s %d\n", TOKEN_FRAME_STR, frame);
    fprintf(outfile, "}\n");

    return 1;
}
/* *************************************************************
   Scale image to the window
************************************************************* */
void beam_point::map2screen(float *v) {

   v[0]  = -mscale * (v[0]/v[2] + vrc[0]);
   v[1]  = -mscale * (v[1]/v[2] + vrc[2]);
}


/* *************************************************************
   Scale image from the window
************************************************************* */
void beam_point::screen2map(float *v) {

   v[1]     = -(v[1]*imscale + vrc[2]) * v[2];
   v[0]     = -(v[0]*imscale + vrc[0]) * v[2];
}


/* *************************************************************
id - 0          always (transparent)
id - "+"        id of object
id - "-"        ignore (should never call... ergo never happen)
************************************************************* */
void beam_point::intensity(float *surface, float *gnormal, float *rcolor, eye *parm,
                        shadetype *shade, int source_id) {

   vector4f pt;
   int  i, j, k, l;
   int  frac;
   int  pt2[2][2];
   float temp;
   float csquared;
   vector3f tempv;
   float cam_surf_dist;

   vector3f mIp;
   float dist2surface;
   vector3f vlight;

   if (!source_id)
      return;

   // from camera space to light space
   matvecmulto(transform, surface, pt);
   pt[3] = 1;

   // distance from object to camera
   dist2surface = fmagnitude3(pt);

   if (dist2surface < CORRECT)
      return;

   if (!tob) {
      mIp[0] = mIp[1] = mIp[2] = 1.0;
      lighting(surface, gnormal, rcolor, shade, mIp, transvec, dist2surface);
      return;
   }

   map2screen(pt);

   if (pt[0] < 0 || pt[0] >= maxx || pt[1] < 0 || pt[1] >= maxy)
      return;

   temp = 1.0f/dist2surface;
   subeqarray3(vlight, transloc, surface);      // calculate surface to light
   smultarray3(vlight, temp);

   csquared = dotproduct3(gnormal, vlight);

   if (csquared < CORRECT)
      return;

//   csquared = SQRT(csquared);

   j = (int)pt[0];
   i = (int)pt[1];

   subeqarray3(tempv, surface, parm->location);  // calculate surface to camera
   cam_surf_dist = fmagnitude3(tempv);

   if ((unsigned int)source_id == lbuff->lbuff.pdata[i][j].idbuff) {

      tempv[0] = pt[0]*ztscale;
      tempv[1] = pt[1]*ztscale;

//      temp = (cam_surf_dist*area*csquared)/dist2surface;
      temp = (cam_surf_dist*area)/dist2surface;

      tob->query_texel(tempv[0], tempv[1], mIp, frame, temp);

      lighting(surface, gnormal, rcolor, shade, mIp, vlight, dist2surface);
      return;
   }

   pt2[0][0] = j-1;
   pt2[1][0] = j+1;
   pt2[0][1] = i-1;
   pt2[1][1] = i+1;

   j = (int)(tempv[0] = pt[0]*ztscale);
   i = (int)(tempv[1] = pt[1]*ztscale);

   if (pt2[0][0] < 0)
      pt2[0][0] = 0;
   if (pt2[1][0] >= maxx)
      pt2[1][0] = maxx-1;
   if (pt2[0][1] < 0)
      pt2[0][1] = 0;
   if (pt2[1][1] >= maxy)
      pt2[1][1] = maxy-1;

   frac = 0;

   for (l=pt2[0][1]; l<pt2[1][1]; l++)
      for (k=pt2[0][0]; k<pt2[1][0]; k++)
         if ((unsigned int)source_id == lbuff->lbuff.pdata[l][k].idbuff)
            frac += 25;

   if (!frac)
      return;

//   temp = (cam_surf_dist*area*csquared)/dist2surface;
   temp = (cam_surf_dist*area)/dist2surface;
							// alt - could use zbuff for z values instead of calc distance....
   tob->query_texel(tempv[0], tempv[1], mIp, frame, temp);

   temp = (frac + 50.0f) * 0.002f;
   smultarray3(mIp, temp);

   lighting(surface, gnormal, rcolor, shade, mIp, vlight, dist2surface);
}


/* *************************************************************
************************************************************* */
int simple_beam_far::dump_frame(FILE *outfile) {

    char token[3][MAXSTRLEN];

    fprintf(outfile, "%s {\n", TOKEN_SIMPLE_BEAM_FAR_STR);
    fprintf(outfile, "\t%s %s\n", TOKEN_2D_TEXTURE_STR, texname.string);

    float2char(length, token[0]);
    fprintf(outfile, "\t%s %s\n", TOKEN_LENGTH_STR, token[0]);

    float2char(width, token[0]);
    fprintf(outfile, "\t%s %s\n", TOKEN_WIDTH_STR, token[0]);

    float2char(-vpn[0], token[0]);
    float2char(-vpn[1], token[1]);
    float2char(-vpn[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_DIRECTION_STR, token[0], token[1], token[2]);

    float2char(location[0], token[0]);
    float2char(location[1], token[1]);
    float2char(location[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_LOCATION_STR, token[0], token[1], token[2]);

    float2char(Ia[0], token[0]);
    float2char(Ia[1], token[1]);
    float2char(Ia[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_AMBIENT_STR, token[0], token[1], token[2]);

    float2char(vup[0], token[0]);
    float2char(vup[1], token[1]);
    float2char(vup[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_UP_STR, token[0], token[1], token[2]);

    fprintf(outfile, "\t%s %d\n", TOKEN_FRAME_STR, frame);
    fprintf(outfile, "}\n");

    return 1;
}


/* *************************************************************
************************************************************* */
void simple_beam_far::lighting(float *surface, float *gnormal, float *rcolor,
        shadetype *shade, float *mIp, float *vlight, float dist2surface) {

   simple_light;
}


/* *************************************************************
************************************************************* */
int simple_beam_point::dump_frame(FILE *outfile) {

    char token[3][MAXSTRLEN];

    fprintf(outfile, "%s {\n", TOKEN_SIMPLE_BEAM_POINT_STR);
    fprintf(outfile, "\t%s %s\n", TOKEN_2D_TEXTURE_STR, texname.string);

    float2char(length, token[0]);
    fprintf(outfile, "\t%s %s\n", TOKEN_LENGTH_STR, token[0]);

    float2char(width, token[0]);
    fprintf(outfile, "\t%s %s\n", TOKEN_WIDTH_STR, token[0]);

    float2char(-vpn[0], token[0]);
    float2char(-vpn[1], token[1]);
    float2char(-vpn[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_DIRECTION_STR, token[0], token[1], token[2]);

    float2char(location[0], token[0]);
    float2char(location[1], token[1]);
    float2char(location[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_LOCATION_STR, token[0], token[1], token[2]);

    float2char(Ia[0], token[0]);
    float2char(Ia[1], token[1]);
    float2char(Ia[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_AMBIENT_STR, token[0], token[1], token[2]);

    float2char(vup[0], token[0]);
    float2char(vup[1], token[1]);
    float2char(vup[2], token[2]);
    fprintf(outfile, "\t%s %s %s %s\n", TOKEN_UP_STR, token[0], token[1], token[2]);

    fprintf(outfile, "\t%s %d\n", TOKEN_FRAME_STR, frame);
    fprintf(outfile, "}\n");

    return 1;
}


/* *************************************************************
************************************************************* */
void simple_beam_point::lighting(float *surface, float *gnormal, float *rcolor,
        shadetype *shade, float *mIp, float *vlight, float dist2surface) {

   simple_light;
}


/* *******************************************************************
******************************************************************* */
superclass *light_loader::parse(FILE *infile, char *token) {

   superclass *ptr, *mtr;

   if (strcmp(token, query_name()))
      return (next ? ((loader *)next)->parse(infile, token) : (superclass *)NULL);

   ptr = make_object();

   do {
      if (!get_token(infile, token) || token[0] == '}')
         break;

      lower_case(token);

      if (!strcmp(token, TOKEN_TYPE_STR)) {
         if (!get_token(infile, token) || token[0] == '}')
            break;

         mtr = ptr;

         if (!strcmp(token, TOKEN_POINT_STR))
            ptr = new point_light;
         else if (!strcmp(token, TOKEN_FAR_STR))
            ptr = new far_light;
         else if (!strcmp(token, TOKEN_BEAM_FAR_STR))
            ptr = new beam_far;
         else if (!strcmp(token, TOKEN_BEAM_POINT_STR))
            ptr = new beam_point;
         else if (!strcmp(token, TOKEN_SIMPLE_POINT_STR))
            ptr = new simple_point_light;
         else if (!strcmp(token, TOKEN_SIMPLE_FAR_STR))
            ptr = new simple_far_light;
         else if (!strcmp(token, TOKEN_SIMPLE_BEAM_FAR_STR))
            ptr = new simple_beam_far;
         else if (!strcmp(token, TOKEN_SIMPLE_BEAM_POINT_STR))
            ptr = new simple_beam_point;
         else {
            sprintf(perror_buffer, "Warning: bad light type : %s\n", token);
            pprintf(perror_buffer);
            mtr = NULL;
         }

         if (mtr)
            delete mtr;
      }

      else if (!ptr->parse(infile, token)) {
         sprintf(perror_buffer, "Warning: Invalid data format for light object - \"%s\".\n", token);
         pprintf(perror_buffer);
      }

   } while (1);

   return ptr;
}
