

/* *************************************************************
   this file contains all functions to do transforms
************************************************************* */

#include <stdlib.h>
#include <string.h>

#include "pstring.h"
#include "camera.h"
#include "global.h"


/* *************************************************************
************************************************************* */
int camera::query_whatwasi(int type) {

   return (camera::query_whatami() == type) ? 1 : eye::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int camera::parse(FILE *infile, char *token) {

   switch (token[0]) {

      case 'c':
         if (!strcmp(token, TOKEN_COA_STR)) {
            get_token(infile, token);
            coa[0] = (float)atof(token);
            get_token(infile, token);
            coa[1] = (float)atof(token);
            get_token(infile, token);
            coa[2] = (float)atof(token);

            return 1;
         }

         break;

      default:
         break;
   }

   return eye::parse(infile, token);
}


/* *************************************************************
************************************************************* */
void camera::preprocess(void *data) {

   eye::preprocess(data);

   subeqarray3(vpn, location, coa);
   normalize3(vpn);

   imscale = (float)(1.0/mscale);
   nper();
}


/* *************************************************************
************************************************************* */
void camera::set_dim(int x, int y) {

   vrc[0] = -(vrc[1] = ((float)x)/(y + y));
   vrc[2] = -0.5;
   vrc[3] =  0.5;

   mscale = (float)y;
   imscale = (float)(1.0/mscale);
}


/* *************************************************************
************************************************************* */
void camera::set_dim(float fov, int x, int y) {

   float hwidth;                       // half width of viewplane
   vector2f temp;
   float ffactor;

   if (fov > HALFPI - CORRECT)
      fov = HALFPI - CORRECT;
   else if (fov < -HALFPI + CORRECT)
      fov = -HALFPI + CORRECT;

   /* calculate viewing parameters */
   ffactor = y/(float)x;

   hwidth = tan(0.5*fov);
   temp[0] = hwidth;
   temp[1] = temp[0]*ffactor;

   vrc[0] = -temp[0];
   vrc[1] =  temp[0];
   vrc[2] = -temp[1];
   vrc[3] =  temp[1];

   mscale = (x+x)/hwidth;
   imscale = (float)(1.0/mscale);
}


/* *************************************************************
************************************************************* */
camera::camera() : eye() {

   location[0] = 0;
   location[1] = 0;
   location[2] = 0;

   coa[0] = coa[1] = 0;
   coa[2] = 1;

   vup[0] = 0;
   vup[1] = 1;
   vup[2] = 0;

   vrc[0] =  -(vrc[1] = 4.0f/3.0f);
   vrc[2] = -0.5;
   vrc[3] =  0.5;

   mscale = 1;
}


/* *************************************************************
************************************************************* */
void camera::read_def(FILE *infile, unsigned int *x, unsigned int *y) {

   float distance2plane;               // distance to view plane
   float wwidth;                       // width of viewplane
   vector2f temp;
   float ffactor;

   fscanf(infile, "%f %f %f", &location[0], &location[1], &location[2]);
   fscanf(infile, "%f %f %f", &coa[0], &coa[1], &coa[2]);
   fscanf(infile, "%f %f %f", &vup[0], &vup[1], &vup[2]);
   fscanf(infile, "%f", &distance2plane);
   fscanf(infile, "%f", &wwidth);

   /* calculate viewing parameters */
   ffactor = *x/(float)(*y);

   wwidth /= distance2plane;
   temp[1] = (float)(0.5*wwidth);
   temp[0] = temp[1]*ffactor;   // temp*4/3

   vrc[0] = -temp[0];
   vrc[1] =  temp[0];
   vrc[2] = -temp[1];
   vrc[3] =  temp[1];

   mscale = *y/wwidth;

   *y = (int)(wwidth * mscale);
   *x = (int)(*y*ffactor);

   preprocess(NULL);
}


/* *************************************************************
************************************************************* */
int camera::dump_frame(FILE *outfile) {

   char token[3][MAXSTRLEN];

   fprintf(outfile, "%s {\n", TOKEN_CAMERA_STR);

   float2char(location[0], token[0]);
   float2char(location[1], token[1]);
   float2char(location[2], token[2]);
   fprintf(outfile, "\t%s %s %s %s\n", TOKEN_LOCATION_STR, token[0], token[1], token[2]);

   float2char(coa[0], token[0]);
   float2char(coa[1], token[1]);
   float2char(coa[2], token[2]);
   fprintf(outfile, "\t%s %s %s %s\n", TOKEN_COA_STR, token[0], token[1], token[2]);

   float2char(vup[0], token[0]);
   float2char(vup[1], token[1]);
   float2char(vup[2], token[2]);
   fprintf(outfile, "\t%s %s %s %s\n", TOKEN_UP_STR, token[0], token[1], token[2]);

   fprintf(outfile, "\t%s %d\n", TOKEN_FRAME_STR, frame);
   fprintf(outfile, "}\n");

   return 1;
}

