 

#include <stdlib.h>
#include <string.h>

#include "quark.h"
#include "pstring.h"

int WF = 0;                 // default - not wireframe test mode


/* *************************************************************
************************************************************* */
int query_frustum_clip(float radius, float *center, vector4f *frustum, unsigned int *frustum_flag, unsigned char *fail_data) {

   float limit = -radius;
   float dist;
   int i, j;

   i = fail_data[0];
   j = fail_data[1];

   // check against last known failure
   if (*frustum_flag & j) {
      dist = dotproduct3(frustum[i], center) + frustum[i][3];

      // check if completely outside
      if (dist < limit)
         return 1;

      // check if completely w/in the plane
      if (dist > radius)
         *frustum_flag &= ~j;
   }

   for (i=0, j=1; i<FRUSTUM_COUNT; i++, j+=j)
      if (i != fail_data[0] && (*frustum_flag & j)) {
         dist = dotproduct3(frustum[i], center) + frustum[i][3];

         // check if completely outside
         if (dist < limit) {
            fail_data[0] = i;
            fail_data[1] = j;
            return 1;
         }

         // check if completely w/in the plane
         if (dist > radius)
            *frustum_flag &= ~j;
      }

   return 0;
}


/* *******************************************************************
******************************************************************* */
superclass *quark_loader::parse(FILE *infile, char *token) {

   superclass *ptr;
   
   if (strcmp(token, query_name()) && strcmp(token, query_alias()))
      return (next ? ((quark_loader *)next)->parse(infile, token) : (superclass *)NULL);

   ptr = make_object();

   do {
      if (!get_token(infile, token) || token[0] == '}')
         break;

      lower_case(token);

      if (!ptr->parse(infile, token)) {
         sprintf(perror_buffer, "ERROR: Invalid data format... Last read token %s... Aborting...\n", token);
         pprintf(perror_buffer);
         fclose(infile);
         exit(0);
      }

   } while (1);

   return ptr;
}


/* *************************************************************
************************************************************* */
state_type::state_type() {
 
   state_flags = STATE_FLAG_AUTO_UPDATE;

   xmx = xmx_buffer;
   node = node_buffer;
   ixmx = ixmx_buffer;
   
   bradius = 0;
   node[3][0] = node[3][1] = node[3][2] = 0;
   node[3][3] = 1.0f;
   center[0] = center[1] = center[2] = 0;
}

  
/* *************************************************************
************************************************************* */
void state_type::calc_bound_data() {

   int i;
   vector4f r[4];
   float temp, temp2;
   vector3f v1, v2;

   if (!(state_flags & STATE_FLAG_AUTO_UPDATE))
      return;
      
   switch (state_flags & STATE_MASK_BOUND) {

      case STATE_FLAG_BSPHERE:

         transpose(r, xmx);

         temp = dotproduct3(r[0], r[0]);

         temp2 = dotproduct3(r[1], r[1]);
         if (temp < temp2)
            temp = temp2;

         temp2 = dotproduct3(r[2], r[2]);
         if (temp < temp2)
            temp = temp2;

         bradius *= (float)sqrt(temp);

         matvecmulto(xmx, bcenter);
         return;

      case STATE_FLAG_BRECT:
         transpose(r, xmx);

         temp = dotproduct3(r[0], r[0]);

         temp2 = dotproduct3(r[1], r[1]);
         if (temp < temp2)
            temp = temp2;

         temp2 = dotproduct3(r[2], r[2]);
         if (temp < temp2)
            temp = temp2;

         bradius *= (float)sqrt(temp);

         for (i=0; i<8; i++)
            matvecmulto(xmx, bdata[i]);

         subeqarray3(v1, bdata[1], bdata[0]);
         subeqarray3(v2, bdata[2], bdata[1]);
         xproduct(bnormal[0], v2, v1);
         normalize3(bnormal[0]);
         bnormal[0][3] = -dotproduct3(bnormal[0], bdata[0]);

         bnormal[1][0] = -bnormal[0][0];
         bnormal[1][1] = -bnormal[0][1];
         bnormal[1][2] = -bnormal[0][2];
         bnormal[1][3] = -dotproduct3(bnormal[1], bdata[4]);

         subeqarray3(v1, bdata[5], bdata[4]);
         subeqarray3(v2, bdata[1], bdata[5]);
         xproduct(bnormal[2], v2, v1);
         normalize3(bnormal[2]);
         bnormal[2][3] = -dotproduct3(bnormal[2], bdata[0]);

         bnormal[4][0] = -bnormal[2][0];
         bnormal[4][1] = -bnormal[2][1];
         bnormal[4][2] = -bnormal[2][2];
         bnormal[4][3] = -dotproduct3(bnormal[4], bdata[2]);

         subeqarray3(v1, bdata[5], bdata[1]);
         subeqarray3(v2, bdata[6], bdata[5]);
         xproduct(bnormal[3], v2, v1);
         normalize3(bnormal[3]);
         bnormal[3][3] = -dotproduct3(bnormal[3], bdata[1]);

         bnormal[5][0] = -bnormal[3][0];
         bnormal[5][1] = -bnormal[3][1];
         bnormal[5][2] = -bnormal[3][2];
         bnormal[5][3] = -dotproduct3(bnormal[5], bdata[0]);

         return;

      case STATE_FLAG_BPLANE:
         bdata[0][3] = 1;
         matvecmulto(xmx, bdata[0]);

         matvecmultv(xmx, bplane);
         normalize3(bplane);
         bplane[3] = -dotproduct3(bplane, bdata[0]);

         return;

      default:
         return;
   }

}


/* *************************************************************
************************************************************* */
void state_type::calc_tree_bound_data() {

   vector3f v1;
   int i;
   float temp;

   state_flags &= ~STATE_MASK_TREE;

   switch (state_flags & STATE_MASK_BOUND) {

      case STATE_FLAG_BNONE:
         state_flags |= STATE_FLAG_TNONE;
//         tree_center[0] = tree_center[1] = tree_center[2] = 0;
//         tree_center[3] = 1.0f;
//         tree_radius = 0;
         return;

      case STATE_FLAG_BPOINT:
         state_flags |= STATE_FLAG_TSPHERE;
         copyarray3(tree_center, center);
         tree_center[3] = 1.0f;
         tree_radius = 0;
         return;

      case STATE_FLAG_BSPHERE:
         state_flags |= STATE_FLAG_TSPHERE;
         copyarray3(tree_center, bcenter);
         tree_center[3] = 1.0f;
         tree_radius = bradius;
         return;

      case STATE_FLAG_BRECT:
         state_flags |= STATE_FLAG_TSPHERE;
         copyarray3(tree_center, center);
         tree_center[3] = 1.0f;

         subeqarray3(v1, center, bdata[0]);
         tree_radius = dotproduct3(v1, v1);

         for (i=1; i< 8; i++) {
            subeqarray3(v1, center, bdata[i]);
            temp = dotproduct3(v1, v1);

            if (temp > tree_radius)
               tree_radius = temp;
         }

         tree_radius = (float)sqrt(tree_radius);
         return;

      default:
         state_flags |= STATE_FLAG_TALL;
         return;
   }

}


/* *************************************************************
************************************************************* */
void state_type::copy_critical(state_type *src) {

   vector4f *ptr, *qtr, *rtr;
   
   copyarray3(center, src->center);
  
   ptr = xmx;
   qtr = node;
   rtr = ixmx;
   xmx = src->xmx;
   node = src->node;
   ixmx = src->ixmx;
   src->xmx = ptr; 
   src->node = qtr; 
   src->ixmx = rtr;
   
   state_flags = src->state_flags;
   
   switch (state_flags & STATE_MASK_BOUND) {
      case STATE_FLAG_BSPHERE:
         copyarray3(bcenter, src->bcenter);
	 bcenter[3] = 1.0f;
	 bradius = src->bradius;
         break;
	 
      case STATE_FLAG_BRECT:
	 bradius = src->bradius;
         memcpy(bdata, src->bdata, 128); // sizeof(float)*4x8
         memcpy(bnormal, src->bnormal, 96); // sizeof(float)*4x6
         break;

      case STATE_FLAG_BPLANE:
         copyarray3(bdata[0], src->bdata[0]);
         bdata[0][3] = 1;
         copyarray3(bplane, src->bplane);
         break;

      default:
         break;
   }
      
//   if ((state_flags & STATE_MASK_TREE) == STATE_FLAG_TALL)
//      return;
     
//   tree_radius = src->tree_radius;
//   copyarray3(tree_center, src->tree_center);
//   tree_center[3] = 1.0f;
}
      
      
/* ********************************************************
******************************************************** */
int state_type::read_bound(FILE *infile) {

   char token[MAXSTRLEN];
   int i;
   float temp;

   get_token(infile, token);
   lower_case(token);

   if (!strcmp(token, "brect")) {       // rectangles have 8 points in local space...
      state_flags &= ~STATE_MASK_BOUND;
      state_flags |= STATE_FLAG_BRECT;

      for (i=0; i<8; i++) {
         get_token(infile, token);
         bdata[i][0] = (float)atof(token);
         get_token(infile, token);
         bdata[i][1] = (float)atof(token);
         get_token(infile, token);
         bdata[i][2] = (float)atof(token);
         bdata[i][3] = 1;
      }

      bradius = dotproduct3(bdata[0], bdata[0]);
      for (i=1; i<8; i++) {
         temp = dotproduct3(bdata[i], bdata[i]);
         if (temp < bradius)
            bradius = temp;
      }

      bradius = (float)sqrt(bradius);
      return 1;
   }

   if (!strcmp(token, "bsphere")) {     // radius around bcenter
      state_flags &= ~STATE_MASK_BOUND;
      state_flags |= STATE_FLAG_BSPHERE;
      get_token(infile, token);
      bradius = (float)atof(token);
      bcenter[0] = bcenter[1] = bcenter[2] = 0;
      bcenter[3] = 1.0f;
      return 1;
   }

   if (!strcmp(token, "bsphere_offset")) {     // change the offset of the sphere
      get_token(infile, token);
      bcenter[0] = (float)atof(token);
      get_token(infile, token);
      bcenter[1] = (float)atof(token);
      get_token(infile, token);
      bcenter[2] = (float)atof(token);
      return 1;
   }

   if (!strcmp(token, "bplane")) {      // normal + point on plane
      state_flags &= ~STATE_MASK_BOUND;
      state_flags |= STATE_FLAG_BPLANE;
      get_token(infile, token);
      bplane[0] = (float)atof(token);
      get_token(infile, token);
      bplane[1] = (float)atof(token);
      get_token(infile, token);
      bplane[2] = (float)atof(token);
      normalize3(bplane);

      get_token(infile, token);
      bdata[0][0] = (float)atof(token);
      get_token(infile, token);
      bdata[0][1] = (float)atof(token);
      get_token(infile, token);
      bdata[0][2] = (float)atof(token);
      bdata[0][3] = 1;

      return 1;
   }

   if (!strcmp(token, "bpoint")) {      // point (center)
      state_flags &= ~STATE_MASK_BOUND;
      state_flags |= STATE_FLAG_BPOINT;
      bradius = 0;
      return 1;
   }

   if (!strcmp(token, "bnone")) {       // no bound...
      state_flags &= ~STATE_MASK_BOUND;
      state_flags |= STATE_FLAG_BNONE;
      return 1;
   }

   if (!strcmp(token, "ball")) {       // undefined bound...
      state_flags &= ~STATE_MASK_BOUND;
      state_flags |= STATE_FLAG_BALL;
      return 1;
   }

   return 0;
}


/* *************************************************************
************************************************************* */
int quark::query_whatwasi(int type) {

   return (quark::query_whatami() == type) ? 1 : superclass::query_whatwasi(type);
}


/* *************************************************************
   This is the constructor for this class
************************************************************* */
quark::quark() {

   init_mx(initxform);
   init_mx(localmx);
   origin[0] = origin[1] = origin[2] = 0;

   iscale = 1.0;
   flags = QUARK_FLAG_ACTIVE | QUARK_FLAG_RECALC;

   frustum_fail_data[0] = 0;

   ob = NULL;

   id = object_counter;
   object_counter++;
   frustum_fail_data[1] = 1;
}


/* *************************************************************
   This is the destructor for this class
************************************************************* */
quark::~quark() {

   linktype *ptr;

   if (ob)
      delete ob;

   for (ptr = (linktype *)edge.head; ptr; ptr = (linktype *)ptr->next) {
      ptr->link->remove_link(this);
      delete ptr->link;
   }

}


/* *************************************************************
************************************************************* */
int quark::parse(FILE *infile, char *token) {

   vector4f tmx[4];
   char     flag;
   int      count;

   switch (token[0]) {

      case 'b':
         if (!strcmp(token, TOKEN_BOUND_STR)) {
            state.read_bound(infile);
            return 1;
         }

         break;

      case 'c':
         if (!strcmp(token, TOKEN_CHILDREN_STR)) {
            get_token(infile, token);

            for (count = atoi(token); count > 0; count--) 	// read in quark's children
               read_quark(infile, 1);

            return 1;
         }

         break;

      case 'n':
         if (!strcmp(token, TOKEN_NAME_STR)) {
            get_token(infile, token);
            name.stringcpy(token);
            return 1;
         }

         break;

      case 'o':
         if (!strcmp(token, TOKEN_ORIGIN_STR)) {
            get_token(infile, token);
            initxform[0][3] += (float)atof(token);
            get_token(infile, token);
            initxform[1][3] += (float)atof(token);
            get_token(infile, token);
            initxform[2][3] += (float)atof(token);

            return 1;
         }

         if (!strcmp(token, "off")) {
            flags &= ~QUARK_FLAG_ACTIVE;
            return 1;
         }

         if (!strcmp(token, "on")) {
            flags |= QUARK_FLAG_ACTIVE;
            return 1;
         }

         break;

      case 'p':
         if (!strcmp(token, TOKEN_PIVOT_STR)) {
            get_token(infile, token);
            origin[0] = (float)atof(token);
            get_token(infile, token);
            origin[1] = (float)atof(token);
            get_token(infile, token);
            origin[2] = (float)atof(token);
            return 1;
         }

         break;

      case 'r':
         if (strlen(token) < 8)
            break;

         if (token[7] == 'l') {

            if (strlen(token) < 14)
               break;

            switch(token[13]) {

               case 'x':
                  if (!strcmp(token, TOKEN_ROTATE_LOCAL_X_STR)) {
                     get_token(infile, token);
                     rotate_mx_x(localmx, (float)deg2rad(atof(token)));
                     return 1;
                  }

                  break;

               case 'y':
                  if (!strcmp(token, TOKEN_ROTATE_LOCAL_Y_STR)) {
                     get_token(infile, token);
                     rotate_mx_y(localmx, (float)deg2rad(atof(token)));
                     return 1;
                  }

                  break;

               case 'z':
                  if (!strcmp(token, TOKEN_ROTATE_LOCAL_Z_STR)) {
                     get_token(infile, token);
                     rotate_mx_z(localmx, (float)deg2rad(atof(token)));
                     return 1;
                  }

                  break;

               default:
                  break;
            }

            break;
         }

         switch(token[7]) {

            case 'x':
               if (!strcmp(token, TOKEN_ROTATE_X_STR)) {
                  get_token(infile, token);
                  rotate_mx_x(initxform, (float)deg2rad(atof(token)));
                  return 1;
               }

               break;

            case 'y':
               if (!strcmp(token, TOKEN_ROTATE_Y_STR)) {
                  get_token(infile, token);
                  rotate_mx_y(initxform, (float)deg2rad(atof(token)));
                  return 1;
               }

               break;

            case 'z':
               if (!strcmp(token, TOKEN_ROTATE_Z_STR)) {
                  get_token(infile, token);
                  rotate_mx_z(initxform, (float)deg2rad(atof(token)));
                  return 1;
               }

               break;

            default:
               break;
         }

         break;

      case 's':
         if (!strcmp(token, TOKEN_SCALE_STR)) {
            get_token(infile, token);
            iscale = (float)atof(token);
            return 1;
         }

         if (!strcmp(token, TOKEN_SHADOW_STR)) {            // setup rest of data in first frame...
            if (ob->query_category() == CLASS_OBJECT)
                ((pc *)ob)->sflag = 1;
            return 1;
         }

         break;

      case 't':
         if (strlen(token) < 6)
            return 0;

         flag = !strcmp(token, TOKEN_TR_MATRIX_STR);
         if (flag || !strcmp(token, TOKEN_TR_LOCAL_MATRIX_STR)) {
            get_token(infile, token);
            tmx[0][0] = (float)atof(token);
            get_token(infile, token);
            tmx[0][1] = (float)atof(token);
            get_token(infile, token);
            tmx[0][2] = (float)atof(token);
            get_token(infile, token);
            tmx[0][3] = (float)atof(token);

            get_token(infile, token);
            tmx[1][0] = (float)atof(token);
            get_token(infile, token);
            tmx[1][1] = (float)atof(token);
            get_token(infile, token);
            tmx[1][2] = (float)atof(token);
            get_token(infile, token);
            tmx[1][3] = (float)atof(token);

            get_token(infile, token);
            tmx[2][0] = (float)atof(token);
            get_token(infile, token);
            tmx[2][1] = (float)atof(token);
            get_token(infile, token);
            tmx[2][2] = (float)atof(token);
            get_token(infile, token);
            tmx[2][3] = (float)atof(token);

            get_token(infile, token);
            tmx[3][0] = (float)atof(token);
            get_token(infile, token);
            tmx[3][1] = (float)atof(token);
            get_token(infile, token);
            tmx[3][2] = (float)atof(token);
            get_token(infile, token);
            tmx[3][3] = (float)atof(token);

            if (flag)
               matmatmulto(tmx, initxform);
            else
               matmatmulto(tmx, localmx);

            return 1;
         }

         if (token[5] == 'l') {
            if (!strcmp(token, TOKEN_TRANSLATE_STR)) {
               get_token(infile, token);
               initxform[0][3] += (float)atof(token);
               get_token(infile, token);
               initxform[1][3] += (float)atof(token);
               get_token(infile, token);
               initxform[2][3] += (float)atof(token);

               return 1;
            }

            if (!strcmp(token, TOKEN_TRANSLATE_LOCAL_STR)) {
               get_token(infile, token);
               localmx[0][3] += (float)atof(token);
               get_token(infile, token);
               localmx[1][3] += (float)atof(token);
               get_token(infile, token);
               localmx[2][3] += (float)atof(token);

               return 1;
            }

            break;
         }

         break;

      default:
         break;
   }

   if (ob)
      return ob->parse(infile, token);

   return 0;
}


/* ********************************************************
******************************************************** */
void quark::preprocess(void *data) {

   geneological_type *param = (geneological_type *)data;
   quark *parent;
   linktype *ptr;

   smultarray3(localmx[0], iscale);
   smultarray3(localmx[1], iscale);
   smultarray3(localmx[2], iscale);
   iscale = 1;

   if (ob) {
      if (ob->query_category() == CLASS_OBJECT) {
         copyarray4(((pc *)ob)->splane, SHADOW_PLANE);

         if (WF) {
            ((pc *)ob)->mctype.set_master(WFBW);
            ((pc *)ob)->mcinfo.mask_and(CINULL);
            ((pc *)ob)->colorname.stringcpy("");
         }

      }

      ob->preprocess(NULL);
   }

   parent = param->parent;
   param->parent = this;

   for (ptr=(linktype *)edge.head; ptr; ptr=(linktype *)ptr->next)
      if (ptr->link != parent)
         ptr->link->preprocess(param);

   param->parent = parent;
}


/* ********************************************************
******************************************************** */
int quark::write_data(FILE *outfile, quark *parent, int frame) {

   int count = 0;
   linktype *ptr;

   if (ob && (flags & QUARK_FLAG_ACTIVE)) {
      ob->frame = frame;
      count = ob->dump_frame(outfile);
   }

   for (ptr=(linktype *)edge.head; ptr; ptr=(linktype *)ptr->next)
      if (ptr->link != parent)
         count += ptr->link->write_data(outfile, this, frame);

    return count;
}


/* *************************************************************
************************************************************* */
void quark::render_object(engine *proc, quark *parent, vector4f *frustum, unsigned int frustum_flag) {

   linktype *ptr;
   
   // cull node w/ view frustum
   if ((old_state.state_flags & STATE_MASK_TREE) == STATE_FLAG_TNONE)
      frustum_flag = 0;
      
   if (frustum_flag && (old_state.state_flags & STATE_MASK_TREE) == STATE_FLAG_TSPHERE &&
       query_frustum_clip(old_state.tree_radius, old_state.tree_center, frustum, &frustum_flag, frustum_fail_data))
      return;

   if (ob && (flags & QUARK_FLAG_ACTIVE))
      proc->engine_submit(ob);

   for (ptr=(linktype *)edge.head; ptr; ptr=(linktype *)ptr->next)
      if (ptr->link != parent)
         ptr->link->render_object(proc, this, frustum, frustum_flag);
}


/* *************************************************************
   Read animation data for hiearchical parts
************************************************************* */
quark *quark::read_quark(FILE *infile, int parent_flag) {

   quark *header;
   char  token[MAXSTRLEN];
   string_type buffer;
   int counter;
   dbl_llist_manager *obj_loader;

   if (!get_token(infile, token))
      return NULL;

   lower_case(token);
   buffer.stringcpy(token);
   get_token(infile, token);           // "{"
   strcpy(token, buffer.string);
      
   obj_loader = (dbl_llist_manager *)global_resource_manager->get_resource_object(RESOURCE_OBJECT_LOADER);

   if (!obj_loader->head)
      header = NULL;
   else
      header = (quark *)((loader *)obj_loader->head)->parse(infile, token);

   if (!header) {
      sprintf(perror_buffer, "ERROR: quark::read_quark() - Invalid object type \"%s\"... Ignoring...\n", buffer.string);
      pprintf(perror_buffer);
      
//      get_token(infile, token);     // "{"
      counter = 1;

      while (get_token(infile, token)) {

         if (!strcmp(token, "{")) {
            counter++;
            continue;
         }

         if (!strcmp(token, "}")) {
            counter--;

            if (!counter)
               break;
         }

      }

      return NULL;
   }

   if (parent_flag)                 // insert quark in hiearchy
      header->create_link(this);

   create_link(header);
   return header;
}


/* *************************************************************
	1) origin   (translate)
	2) quat     (rotate around origin)
    3) init_mx  (translate/rotate offset from parent)
    4) scale    (scale all)
    5) spline   (translate - movement)
    6) xmx      (attach to parent)
************************************************************* */
void quark::calc_initmx(vector4f *in, vector4f *out) {

   vector4f r[4];
   vector3f v;

   if (!(flags & QUARK_FLAG_RECALC)) {
      matmatmulto(in, previous_local_motion, out);
      return;
   }
      
   init_mx(out);

   out[0][3] = -origin[0];                      // move to origin/pivot point
   out[1][3] = -origin[1];
   out[2][3] = -origin[2];

   motion.calc_localquat(r);
   matmatmulto(r, out);                         // move back...

   out[0][3] += origin[0];
   out[1][3] += origin[1];
   out[2][3] += origin[2];

   matmatmulto(initxform, out);                 // combine w/ global xform

   smultarray4(out[0], iscale);                 // scale global
   smultarray4(out[1], iscale);
   smultarray4(out[2], iscale);

   motion.calc_localspline(v);                  // calc spline

   out[0][3] += v[0];
   out[1][3] += v[1];
   out[2][3] += v[2];
   
   copymx4x4o(previous_local_motion, out);

   if (!motion.query_motion())
      flags &= ~QUARK_FLAG_RECALC;
                                                // concatinate to parent
   matmatmulto(in, out);
}


/* *******************************************************************
******************************************************************* */
superclass *quark::find_ob(quark *parent, int typecode, quark **source) {

   linktype   *ptr;
   superclass *qtr;

   if ((flags & QUARK_FLAG_ACTIVE) && ob && ob->query_whatami() == typecode) {
      if (source)
         *source = this;
      return ob;
   }

   for (ptr=(linktype *)edge.head; ptr; ptr=(linktype *)ptr->next)
      if (parent != ptr->link) {
         qtr = ptr->link->find_ob(this, typecode, source);
         if (qtr)
            return qtr;
      }

   return NULL;
}


/* *************************************************************
   This function finds a specific child in the hiearchy

   returns child
************************************************************* */
quark *quark::find(quark *parent, char *part, quark **found_parent) {

   linktype *ptr;
   quark    *qtr;

   if (!name.stringcmp(part)) {
      if (found_parent)
         *found_parent = parent;

      return this;
   }

   for (ptr=(linktype *)edge.head; ptr; ptr=(linktype *)ptr->next) {
      if (parent == ptr->link)
         continue;

      qtr = ptr->link->find(this, part);
      if (qtr)
         return qtr;
   }

   return NULL;
}


/* *************************************************************
   This function finds quarks based on type

   returns first quark of that type if "last" == NULL, or the one
   	following "last"
************************************************************* */
quark *quark::find(quark *parent, int cat, quark *last, quark **found_parent) {

   linktype *ptr;
   quark    *qtr;

   if (query_whatwasi(cat)) {
      if (!last) {
         if (found_parent)
            *found_parent = parent;
	    
         return this;
      }

      if (this == last)
         last = NULL;
   }

   for (ptr=(linktype *)edge.head; ptr; ptr=(linktype *)ptr->next) {
      if (parent == ptr->link)
         continue;

      qtr = ptr->link->find(this, cat, last, found_parent);
      if (qtr)
         return qtr;
   }

   return NULL;
}


/* *************************************************************
   This function removes a child from the hiearchy
   and deletes the child
************************************************************* */
int quark::remove_link(quark *child) {

   linktype *ptr;

   if (!child)
      return 0;

   for (ptr=(linktype *)edge.head; ptr; ptr=(linktype *)ptr->next) {
      if (ptr->link != child)
         continue;

      edge.remove(ptr);
      delete ptr;
      return 1;
   }

   return 0;
}

/* *************************************************************
   this function connects a child to the hiearchy
************************************************************* */
void quark::create_link(quark *child) {

   linktype *ptr;

   edge.insert(ptr = new linktype, NULL);
   ptr->link = child;
}


/* ********************************************************
   update data with new event
******************************************************** */
void quark::new_action(FILE *infile, float  timefactor, char *buffer) {

   char token[MAXSTRLEN];                     // type of event

   if (buffer)
      strcpy(token, buffer);
   else {
      get_token(infile, token);
      lower_case(token);
   }

   if (!strcmp(token, "off")) {
      flags &= ~QUARK_FLAG_ACTIVE;
      return;
   }

   if (!strcmp(token, "on")) {
      flags |= QUARK_FLAG_ACTIVE;
      return;
   }

   if (!strcmp(token, "spline")) {               // spline event
      motion.read_spline(infile, timefactor);
      flags |= QUARK_FLAG_RECALC;
      return;
   }

   // else must need to read quat event
   motion.read_quaternion(infile, timefactor);
   flags |= QUARK_FLAG_RECALC;
}


/* *************************************************************
************************************************************* */
void quark::setup() {

}


/* *************************************************************
   place object based on local and parent xforms
************************************************************* */
void quark::begin(dbl_llist_manager *hiearchy_manager, quark *parent, vector4f *mx) {

   linktype *ptr;			

   calc_initmx(mx, state.node);

   for (ptr=(linktype *)edge.head; ptr; ptr=(linktype *)ptr->next) 
      if (ptr->link != parent)                           
         ptr->link->begin(hiearchy_manager, this, state.node);

   matmatmulto(state.node, localmx, state.xmx);

   state.center[0] = state.xmx[0][3];
   state.center[1] = state.xmx[1][3];
   state.center[2] = state.xmx[2][3];

   if (flags & QUARK_FLAG_ACTIVE)
      setup();
}


/* *************************************************************
   place object based on local and parent xforms
************************************************************* */
void quark::whereami(dbl_llist_manager *hiearchy_manager, quark *parent, vector4f *mx) {

   linktype *ptr;			

   calc_initmx(mx, state.node);

   for (ptr=(linktype *)edge.head; ptr; ptr=(linktype *)ptr->next)  
      if (ptr->link != parent)                          
         ptr->link->whereami(hiearchy_manager, this, state.node);

   matmatmulto(state.node, localmx, state.xmx);

   state.center[0] = state.xmx[0][3];
   state.center[1] = state.xmx[1][3];
   state.center[2] = state.xmx[2][3];

   if (flags & QUARK_FLAG_ACTIVE)
      setup();
}


/* *************************************************************
   tree data - remember even though object not active, it's children might be
************************************************************* */
void quark::update(dbl_llist_manager *hiearchy_manager, quark *parent) {

   linktype *ptr;
   int start_flag = 0;
   quark *qtr;
   int i;
   float temp, temp2;
   vector3f work;
   
   for (ptr=(linktype *)edge.head; ptr; ptr=(linktype *)ptr->next)
      if (ptr->link != parent)
         ptr->link->update(hiearchy_manager, this);

   if (!(flags & QUARK_FLAG_ACTIVE)) {
      old_state.state_flags &= ~STATE_MASK_TREE;
      old_state.state_flags |= STATE_FLAG_TNONE;
   }

   else {
      old_state.copy_critical(&state);
      state.state_flags &= ~STATE_FLAG_INVERSE;
      old_state.calc_bound_data();
      old_state.calc_tree_bound_data();

      if ((old_state.state_flags & STATE_MASK_TREE) == STATE_FLAG_TALL)
         return;

      // calc diagonals
      if ((old_state.state_flags & STATE_MASK_TREE) == STATE_FLAG_TSPHERE) {
         start_flag = 1;
         for (i=0; i<3; i++) {
            old_state.tree_min[i] = old_state.tree_center[i] - old_state.tree_radius;
            old_state.tree_max[i] = old_state.tree_center[i] + old_state.tree_radius;
         }
      
      }

   }
   
   for (ptr=(linktype *)edge.head; ptr; ptr=(linktype *)ptr->next) {
      qtr = ptr->link;

      if (qtr == parent || (qtr->old_state.state_flags & STATE_MASK_TREE) == STATE_FLAG_TNONE)
         continue;

      if ((qtr->old_state.state_flags & STATE_MASK_TREE) == STATE_FLAG_TALL) {
         old_state.state_flags &= ~STATE_MASK_TREE;
         old_state.state_flags |= STATE_FLAG_TALL;
         return;
      }

      if (start_flag) {
         start_flag++;
         for (i=0; i<3; i++) {
            if (qtr->old_state.tree_min[i] < old_state.tree_min[i])
               old_state.tree_min[i] = qtr->old_state.tree_min[i];
	       
            if (qtr->old_state.tree_max[i] > old_state.tree_max[i])
               old_state.tree_max[i] = qtr->old_state.tree_max[i];
         }

         continue;	 
      }

      start_flag = 1;
      copyarray3(old_state.tree_min, qtr->old_state.tree_min);
      copyarray3(old_state.tree_max, qtr->old_state.tree_max);

      copyarray3(old_state.tree_center, qtr->old_state.tree_center);
      old_state.tree_radius = qtr->old_state.tree_radius;
   }

   if (!start_flag)
      return;

   old_state.state_flags &= ~STATE_MASK_TREE;
   old_state.state_flags |= STATE_FLAG_TSPHERE;

   if (start_flag == 1)
      return;

   old_state.tree_center[0] = 0.5f * (old_state.tree_max[0] + old_state.tree_min[0]);
   old_state.tree_center[1] = 0.5f * (old_state.tree_max[1] + old_state.tree_min[1]);
   old_state.tree_center[2] = 0.5f * (old_state.tree_max[2] + old_state.tree_min[2]);

   subeqarray3(work, old_state.tree_max, old_state.tree_min);
   temp = dotproduct3(work, work);

   // sqr(half diameter) / 3
   old_state.tree_radius = (float)sqrt(temp*0.0833);

   for (ptr=(linktype *)edge.head; ptr; ptr=(linktype *)ptr->next) {
      if (ptr->link == parent || (ptr->link->old_state.state_flags & STATE_MASK_TREE) == STATE_FLAG_TNONE)
         continue;

      temp2 = old_state.tree_radius - ptr->link->old_state.tree_radius;

      subeqarray3(work, old_state.tree_center, ptr->link->old_state.tree_center);
      temp = dotproduct3(work, work);

      if (temp2 <= 0 || temp >= temp2*temp2)
         old_state.tree_radius = (float)(ptr->link->old_state.tree_radius + (temp ? sqrt(temp) : 0));
   }

   old_state.tree_radius += (float)CORRECT;
}


/* *************************************************************
************************************************************* */
int quark::query_specific_data(unsigned int type, void *data) {

   return 0;
}


/* *************************************************************
************************************************************* */
int quark::set_specific_data(unsigned int type, void *data) {

   return 0;
}


/* *************************************************************
************************************************************* */
void quark::apply_scale(float scale) {

   iscale *= scale;
}

