

#include <stdlib.h>
#include <string.h>

#include "boid.h"
#include "pstring.h"

#define query_sphere_not_above_plane(cc, rr, nn)   (dotproduct3(nn, cc) + nn[3] <= (rr))


/* *************************************************************
************************************************************* */
int boidbait::query_whatwasi(int type) {

   return (boidbait::query_whatami() == type) ? 1 : quark::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int boid::query_whatwasi(int type) {

   return (boid::query_whatami() == type) ? 1 : wwdigo::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
boid::boid() {

   gflag = 0;
   gravity[0] = gravity[2] = 0; gravity[1] = -1;
   max_accel = maxspeed = minspeed = velocity = 1;
   ra_radius = ya_radius = 1;
   ya_weight = 0.5;
   ifps = 1;
   icl = 0.5;
   nb_min_radius = nb_max_radius = nl_min_radius = nl_max_radius = 1;
   max_turn_rad = PI;
}


/* *************************************************************
************************************************************* */
int boid::parse(FILE *infile, char *token) {

   float gravity_vel;

   switch (token[0]) {

      case 'b':
         if (!strcmp(token, TOKEN_BOID_RANGE_STR)) {
            get_token(infile, token);
            nb_min_radius = (float)atof(token);

            get_token(infile, token);
            nb_max_radius = (float)atof(token);
            return 1;
         }

         break;

      case 'c':
         if (!strcmp(token, TOKEN_COLLISION_LOSS_STR)) {
            icl = (float)(1 - get_token(infile, token));
            return 1;
         }

         break;

      case 'f':
         if (!strcmp(token, TOKEN_FRICTION_PER_SEC_STR)) {
            get_token(infile, token);
            ifps = (float)(1.0-atof(token));
            return 1;
         }

         break;

      case 'g':
         if (!strcmp(token, TOKEN_GRAVITY_STR)) {
            gflag = 1;
            get_token(infile, token);
            gravity_vel = (float)atof(token);
            get_token(infile, token);
            gravity[0] = (float)atof(token);
            get_token(infile, token);
            gravity[1] = (float)atof(token);
            get_token(infile, token);
            gravity[2] = (float)atof(token);

            gravity_vel /= (float)magnitude3(gravity);

            smultarray3(gravity, gravity_vel);
            return 1;
         }

         break;

      case 'l':
         if (!strcmp(token, TOKEN_LEADER_RANGE_STR)) {
            get_token(infile, token);
            nl_min_radius = (float)atof(token);

            get_token(infile, token);
            nl_max_radius = (float)atof(token);
            return 1;
         }

         break;

      case 'm':
         if (!strcmp(token, TOKEN_MAX_TURN_RADIUS_STR)) {
            get_token(infile, token);
            max_turn_rad = (float)deg2rad(atof(token));
            return 1;
         }

         if (!strcmp(token, TOKEN_MAX_ACCELERATION_STR)) {
            get_token(infile, token);
            max_accel = (float)atof(token);
            return 1;
         }

         break;

      case 'r':
         if (!strcmp(token, TOKEN_RED_ALERT_STR)) {
            get_token(infile, token);
            ra_radius = (float)atof(token);
            return 1;
         }

         break;

      case 'v':
         if (!strcmp(token, TOKEN_VELOCITY_STR)) {
            get_token(infile, token);
            velocity = (float)atof(token);
            return 1;
         }

         if (!strcmp(token, TOKEN_VELOCITY_RANGE_STR)) {
            get_token(infile, token);
            minspeed = (float)atof(token);
            get_token(infile, token);
            maxspeed = (float)atof(token);
            return 1;
         }

         break;

      case 'y':
         if (!strcmp(token, TOKEN_YELLOW_ALERT_STR)) {
            get_token(infile, token);
            ya_radius = (float)atof(token);

            get_token(infile, token);
            ya_weight = (float)atof(token);

            return 1;
         }

         break;

      default:
         break;
   }

   return wwdigo::parse(infile, token);
}


/* *************************************************************
************************************************************* */
void boid::update(dbl_llist_manager *hiearchy_manager, quark *parent) {

   quark::update(hiearchy_manager, parent);
   calc_bound();             // calc bound sphere/rect/plane
}


/* *************************************************************
************************************************************* */
void boid::box_normals(vector4f *in_data, vector4f *out_normal) {

   vector3f v1, v2;

   subeqarray3(v1, in_data[1], in_data[0]); // 0 1 2 3
   subeqarray3(v2, in_data[2], in_data[1]);
   xproduct(out_normal[0], v2, v1);
   normalize3(out_normal[0]);
   out_normal[0][3] = -dotproduct3(out_normal[0], in_data[0]);

   out_normal[1][0] = -out_normal[0][0];
   out_normal[1][1] = -out_normal[0][1];
   out_normal[1][2] = -out_normal[0][2];
   out_normal[1][3] = -dotproduct3(out_normal[1], in_data[4]);

   subeqarray3(v1, in_data[5], in_data[4]);
   subeqarray3(v2, in_data[1], in_data[5]);
   xproduct(out_normal[2], v2, v1);
   normalize3(out_normal[2]);
   out_normal[2][3] = -dotproduct3(out_normal[2], in_data[0]);

   out_normal[4][0] = -out_normal[2][0];
   out_normal[4][1] = -out_normal[2][1];
   out_normal[4][2] = -out_normal[2][2];
   out_normal[4][3] = -dotproduct3(out_normal[4], in_data[2]);

   subeqarray3(v1, in_data[5], in_data[1]);
   subeqarray3(v2, in_data[6], in_data[5]);
   xproduct(out_normal[3], v2, v1);
   normalize3(out_normal[3]);
   out_normal[3][3] = -dotproduct3(out_normal[3], in_data[1]);

   out_normal[5][0] = -out_normal[3][0];
   out_normal[5][1] = -out_normal[3][1];
   out_normal[5][2] = -out_normal[3][2];
   out_normal[5][3] = -dotproduct3(out_normal[5], in_data[0]);
}


/* *************************************************************
************************************************************* */
void boid::bound_rect(vector4f *in_data, float radius, vector4f *out_data) {

   int i;
   vector3f work;
   float temp;

   for (i=0; i<8; i++)
      copyarray4(out_data[i], in_data[i]);

   subeqarray3(work, out_data[0], out_data[6]);
   temp = (float)(radius/magnitude3(work));
   smultarray3(work, temp);
   addarray3(out_data[0], work);
   subarray3(out_data[6], work);

   subeqarray3(work, out_data[1], out_data[7]);
   temp = (float)(radius/magnitude3(work));
   smultarray3(work, temp);
   addarray3(out_data[1], work);
   subarray3(out_data[7], work);

   subeqarray3(work, out_data[2], out_data[4]);
   temp = (float)(radius/magnitude3(work));
   smultarray3(work, temp);
   addarray3(out_data[2], work);
   subarray3(out_data[4], work);

   subeqarray3(work, out_data[3], out_data[5]);
   temp = (float)(radius/magnitude3(work));
   smultarray3(work, temp);
   addarray3(out_data[3], work);
   subarray3(out_data[5], work);

   for (i=0; i<8; i++)
      matvecmulto(state.xmx, out_data[i]);
}


/* *************************************************************
************************************************************* */
void boid::add2path(float *target, float *me, float dist, float maxdist, float *weight, float *want) {

    vector3f work;
    float temp;

    subeqarray3(work, target, me);
    temp = (dist - maxdist)/maxdist;

    if (temp > *weight) {
        temp = *weight;
        *weight = 0;
    }

    else
        *weight -= temp;

    temp = lerp(temp, minspeed, maxspeed);

    want[0] += work[0] * temp;
    want[1] += work[1] * temp;
    want[2] += work[2] * temp;
}


/* *************************************************************
************************************************************* */
void boid::sub2path(float *target, float *me, float dist, float maxdist, float *weight, float *want) {

    vector3f work;
    float temp;

    subeqarray3(work, me, target);
    temp = (maxdist - dist)/maxdist;

    if (temp > *weight) {
        temp = *weight;
        *weight = 0;
    }

    else
        *weight -= temp;

    temp = lerp(temp, minspeed, maxspeed);

    want[0] += work[0] * temp;
    want[1] += work[1] * temp;
    want[2] += work[2] * temp;
}


/* *************************************************************
    calculate bounds for yellow and red alerts
************************************************************* */
void boid::calc_bound() {

   switch (old_state.state_flags & STATE_MASK_BOUND) {
      case STATE_FLAG_BPOINT:
         yradius = ya_radius;
         rradius = ra_radius;
         break;

      case STATE_FLAG_BSPHERE:
         yradius = ya_radius + old_state.bradius;
         rradius = ra_radius + old_state.bradius;
         break;

      case STATE_FLAG_BRECT:
         bound_rect(old_state.bdata, ya_radius, ybdata);
         box_normals(ybdata, ybnormal);

         bound_rect(old_state.bdata, ra_radius, rbdata);
         box_normals(rbdata, rbnormal);

         yradius = ya_radius + old_state.bradius;
         rradius = ra_radius + old_state.bradius;
         break;

      case STATE_FLAG_BPLANE:
         // move and rotate point
         ybdata[0][0] = old_state.bdata[0][0] + ya_radius*old_state.bdata[1][0];
         ybdata[0][1] = old_state.bdata[0][1] + ya_radius*old_state.bdata[1][1];
         ybdata[0][2] = old_state.bdata[0][2] + ya_radius*old_state.bdata[1][2];
         ybdata[0][3] = 1;
         matvecmulto(state.xmx, ybdata[0]);
	 
         // rotate normal
         matvecmultv(state.xmx, old_state.bdata[1], ybplane);
         normalize3(ybplane);
         ybplane[3] = -dotproduct3(ybplane, ybdata[0]);
	 
         // move and rotate point
         rbdata[0][0] = old_state.bdata[0][0] + ra_radius*old_state.bdata[1][0];
         rbdata[0][1] = old_state.bdata[0][1] + ra_radius*old_state.bdata[1][1];
         rbdata[0][2] = old_state.bdata[0][2] + ra_radius*old_state.bdata[1][2];
         rbdata[0][3] = 1;
         matvecmulto(state.xmx, rbdata[0]);
	 
         // rotate normal
         matvecmultv(state.xmx, old_state.bdata[1], rbplane);
         normalize3(rbplane);
         rbplane[3] = -dotproduct3(rbplane, rbdata[0]);
         break;
   }

}


/* *************************************************************
************************************************************* */
void boid::calc_reflect(quark *drunk, float *dir, float *ref) {

   vector3f n;
   int flag = 0;
   float temp, temp2;

   switch (old_state.state_flags & STATE_MASK_BOUND) {

      case STATE_FLAG_BPOINT:
         switch (drunk->old_state.state_flags & STATE_MASK_BOUND) {              // calc reflect normal

            case STATE_FLAG_BPOINT:
               flag = point_point_reflectn3vf(drunk->old_state.center, old_state.center, n);
               break;

            case STATE_FLAG_BSPHERE:
               flag = point_point_reflectn3vf(drunk->old_state.bcenter, old_state.center, n);
               break;

            case STATE_FLAG_BPLANE:
               flag = plane_reflectn3vf(drunk->old_state.bplane, n);
               break;

            case STATE_FLAG_BRECT:
               flag = point_rect_reflectn3vf(old_state.center, drunk->old_state.center, drunk->old_state.bnormal, n);
               break;

            default:            //case STATE_FLAG_BALL:
               break;
         }

         break;

      case STATE_FLAG_BSPHERE:

         switch (drunk->old_state.state_flags & STATE_MASK_BOUND) {              // calc reflect normal

            case STATE_FLAG_BPOINT:
               flag = point_point_reflectn3vf(drunk->old_state.center, old_state.bcenter, n);
               break;

            case STATE_FLAG_BSPHERE:
               flag = point_point_reflectn3vf(drunk->old_state.bcenter, old_state.bcenter, n);
               break;

            case STATE_FLAG_BPLANE:
               flag = plane_reflectn3vf(drunk->old_state.bplane, n);
               break;

            case STATE_FLAG_BRECT:
               flag = point_rect_reflectn3vf(old_state.bcenter, drunk->old_state.center, drunk->old_state.bnormal, n);
               break;

            default:            //case STATE_FLAG_BALL:
               break;
         }

         break;

      case STATE_FLAG_BRECT:
         switch (drunk->old_state.state_flags & STATE_MASK_BOUND) {
            case STATE_FLAG_BPOINT:
               flag = point_rect_reflectn3vf(drunk->old_state.center, old_state.center, old_state.bnormal, n);
               n[0] = -n[0];
               n[1] = -n[1];
               n[2] = -n[2];
               break;

            case STATE_FLAG_BSPHERE:
               flag = point_rect_reflectn3vf(drunk->old_state.bcenter, old_state.center, old_state.bnormal, n);
               n[0] = -n[0];
               n[1] = -n[1];
               n[2] = -n[2];
               break;

            case STATE_FLAG_BPLANE:
               flag = plane_reflectn3vf(drunk->old_state.bplane, n);
               break;

            case STATE_FLAG_BRECT:
               flag = point_rect_reflectn3vf(old_state.center, drunk->old_state.center, drunk->old_state.bnormal, n);
               break;

            default:            //case STATE_FLAG_BALL:
               break;
         }

         break;

      //case STATE_FLAG_BALL:
      //case STATE_FLAG_BPLANE:        // a boid a plane? never happen....
      default:
         return;
   }

   if (!flag) {
      n[0] = -dir[0];
      n[1] = -dir[1];
      n[2] = -dir[2];
      normalize3(n);
   }

   if (dotproduct3(n, dir) < -CORRECT) {    // if above surface, reflect
      ref[0] = -dir[0];
      ref[1] = -dir[1];
      ref[2] = -dir[2];
      reflect(n,ref);

      if (dotproduct3(ref, dir) < CORRECT-1) {		// perfect reflection
         temp = (float)fabs(ref[0]);
         flag = 0;

         temp2 = (float)fabs(ref[1]);
         if (temp > temp2) {
            temp = temp2;
            flag = 1;
         }

         if (temp > fabs(ref[2]))
            flag = 2;

         ref[2] += 0.01f;
         normalize3(ref);
      }

   }

//   else						// else travel through
//      copyarray3(ref, dir);

   // perfect reflection
   else {
      copyarray3(ref, n);
      normalize3(ref);
   }
   
}


/* *************************************************************
************************************************************* */
int boid::query_collision(quark *drunk) {

   if ((old_state.state_flags & STATE_MASK_BOUND) == STATE_FLAG_BNONE ||
       (drunk->old_state.state_flags & STATE_MASK_BOUND) == STATE_FLAG_BNONE)
      return NO_COLLISION;

   switch (old_state.state_flags & STATE_MASK_BOUND) {

      case STATE_FLAG_BPOINT:
         switch (drunk->old_state.state_flags & STATE_MASK_BOUND) {

            case STATE_FLAG_BALL:
               return NO_COLLISION;
	       
            case STATE_FLAG_BPOINT:
               if (!query_point_sphere_intersect(drunk->old_state.center, old_state.center, yradius))
                  return NO_COLLISION;
               if (query_point_sphere_intersect(drunk->old_state.center, old_state.center, rradius))
                  return RED_ALERT;

               return YELLOW_ALERT;

            case STATE_FLAG_BSPHERE:
               if (!query_sphere_sphere_intersect(drunk->old_state.bcenter, drunk->old_state.bradius, old_state.center, yradius))
                  return NO_COLLISION;

               if (query_sphere_sphere_intersect(drunk->old_state.bcenter, drunk->old_state.bradius, old_state.center, rradius))
                  return RED_ALERT;

               return YELLOW_ALERT;

            case STATE_FLAG_BRECT:
               if (!query_sphere_rect_intersect(old_state.center, yradius, drunk->old_state.center, drunk->old_state.bnormal))
                  return NO_COLLISION;
               if (query_sphere_rect_intersect(old_state.center, rradius, drunk->old_state.center,  drunk->old_state.bnormal))
                  return RED_ALERT;

               return YELLOW_ALERT;

            default:            // case STATE_FLAG_BPLANE:

               if (!query_sphere_not_above_plane(old_state.center, yradius, drunk->old_state.bplane))
                  return NO_COLLISION;

               if (query_sphere_not_above_plane(old_state.center, rradius, drunk->old_state.bplane))
                  return RED_ALERT;

               return YELLOW_ALERT;
         }

         // should never be here...
         return NO_COLLISION;

      case STATE_FLAG_BSPHERE:
         switch (drunk->old_state.state_flags & STATE_MASK_BOUND) {

            case STATE_FLAG_BALL:
               return NO_COLLISION;
	       
            case STATE_FLAG_BPOINT:
               if (!query_point_sphere_intersect(drunk->old_state.center, old_state.bcenter, yradius))
                  return NO_COLLISION;
               if (query_point_sphere_intersect(drunk->old_state.center, old_state.bcenter, rradius))
                  return RED_ALERT;

               return YELLOW_ALERT;

            case STATE_FLAG_BSPHERE:
               if (!query_sphere_sphere_intersect(drunk->old_state.bcenter, drunk->old_state.bradius, old_state.bcenter, yradius))
                  return NO_COLLISION;

               if (query_sphere_sphere_intersect(drunk->old_state.bcenter, drunk->old_state.bradius, old_state.bcenter, rradius))
                  return RED_ALERT;

               return YELLOW_ALERT;

            case STATE_FLAG_BRECT:
               if (!query_sphere_rect_intersect(old_state.bcenter, yradius, drunk->old_state.center, drunk->old_state.bnormal))
                  return NO_COLLISION;
               if (query_sphere_rect_intersect(old_state.bcenter, rradius, drunk->old_state.center,  drunk->old_state.bnormal))
                  return RED_ALERT;

               return YELLOW_ALERT;

            default:		// case STATE_FLAG_BPLANE:
               if (!query_sphere_not_above_plane(old_state.bcenter, yradius, drunk->old_state.bplane))
                  return NO_COLLISION;
               if (query_sphere_not_above_plane(old_state.bcenter, rradius, drunk->old_state.bplane))
                  return RED_ALERT;

               return YELLOW_ALERT;
         }

         // should never be here...
         return NO_COLLISION;

      case STATE_FLAG_BRECT:

         switch (drunk->old_state.state_flags & STATE_MASK_BOUND) {

            case STATE_FLAG_BALL:
               return NO_COLLISION;
	       
            case STATE_FLAG_BPOINT:
               if (!query_point_rect_intersect(drunk->old_state.center, ybnormal))
                  return NO_COLLISION;
               if (query_point_rect_intersect(drunk->old_state.center, rbnormal))
                  return RED_ALERT;

               return YELLOW_ALERT;

            case STATE_FLAG_BSPHERE:
               if (!query_sphere_rect_intersect(drunk->old_state.bcenter, drunk->old_state.bradius, old_state.center, ybnormal))
                  return NO_COLLISION;
               if (query_sphere_rect_intersect(drunk->old_state.bcenter, drunk->old_state.bradius, old_state.center, rbnormal))
                  return RED_ALERT;

               return YELLOW_ALERT;

            case STATE_FLAG_BRECT:
               if (!query_rect_rect_intersect(ybnormal, ybdata, old_state.center, old_state.bradius,
                   drunk->old_state.bnormal, drunk->old_state.bdata, drunk->old_state.center, drunk->old_state.bradius))
                  return NO_COLLISION;
               if (!query_rect_rect_intersect(rbnormal, rbdata, old_state.center, old_state.bradius,
                   drunk->old_state.bnormal, drunk->old_state.bdata, drunk->old_state.center, drunk->old_state.bradius))
                  return RED_ALERT;

               return YELLOW_ALERT;

            default:        // case STATE_FLAG_BPLANE:
//asdf need to update this - this only checks intersections - doesnt check if rect-gon is BELOW plane
               if (!query_plane_rect_intersect(drunk->old_state.bplane, ybdata))
                  return NO_COLLISION;
//asdf need to update this - this only checks intersections - doesnt check if rect-gon is BELOW plane
               if (query_plane_rect_intersect(drunk->old_state.bplane, rbdata))
                  return RED_ALERT;

               return YELLOW_ALERT;
         }

      default:      // STATE_FLAG_BPLANE/BALL - boid is a plane??? get real!
         return NO_COLLISION;
   }

}


/* *************************************************************
************************************************************* */
int boid::calc_avoid(float *avoid, quark *qtr, int *aflag) {

   vector3f reflect;

   switch (query_collision(qtr)) {
      case NO_COLLISION:
         return 0;

      case YELLOW_ALERT:
         if (*aflag == RED_ALERT)
            return 0;

         calc_reflect(qtr, global_dir, reflect);

         if (*aflag != YELLOW_ALERT) {
            copyarray3(avoid, reflect);
            *aflag = YELLOW_ALERT;
            return 1;
	 }
	 
         addarray3(avoid, reflect);	 
         return 0;

      default:	//case RED_ALERT:

         calc_reflect(qtr, global_dir, reflect);

         if (*aflag != RED_ALERT) {
            copyarray3(avoid, reflect);
            *aflag = RED_ALERT;
            return 1;
         }

         addarray3(avoid, reflect);
         return 0;
   }

}


/* *************************************************************
    nbd2 = SQR(nearest boid)
    nld2 = SQR(nearest leader)
************************************************************* */
void boid::calc_dist_data(boid_avoid_type *data) {

   linktype *ptr;
   vector3f reflect;
   float    dist;
   float    temp;
   boid_avoid_type subdata;
   int      sphereflag, old_alert;
   float    *bc1, *bc2;

   if (data->target == this || (old_state.state_flags & STATE_MASK_TREE) == STATE_FLAG_TNONE || (data->target->old_state.state_flags & STATE_MASK_TREE) == STATE_FLAG_TNONE)
      return;

   sphereflag = ((old_state.state_flags & STATE_MASK_TREE) == STATE_FLAG_TSPHERE) & ((data->target->old_state.state_flags & STATE_MASK_TREE) == STATE_FLAG_TSPHERE);

   if (sphereflag) {
      subeqarray3(reflect, data->target->old_state.tree_center, old_state.tree_center);
      dist = (float)(sqrt(dotproduct3(reflect, reflect)) - data->target->old_state.tree_radius);
   }
   
   // calc alert vs tree, then object
   if (data->flags & FLAG_BOID_ALERT) {
      if (!sphereflag)
         subdata.flags = FLAG_BOID_ALERT;
      else if (alert == RED_ALERT)
         subdata.flags = (dist <= rradius) ? FLAG_BOID_ALERT : FLAG_BOID_NULL;
      else
         subdata.flags = (dist <= yradius) ? FLAG_BOID_ALERT : FLAG_BOID_NULL;

      if ((subdata.flags & FLAG_BOID_ALERT) && calc_avoid(avoid_dir, data->target, &alert) && alert == RED_ALERT && dist > rradius)
         subdata.flags = FLAG_BOID_NULL;
   }

   else
      subdata.flags = FLAG_BOID_NULL;

   // set nearest object and leader flags
   if (alert != RED_ALERT)
      if (sphereflag) {

         dist -= old_state.tree_radius;

         if ((data->flags & FLAG_BOID_LEAD) && (!nearest_leader || dist <= nl_dist))
            subdata.flags |= FLAG_BOID_LEAD;
         if ((data->flags & FLAG_BOID_OBJECT) && (!nearest_boid || dist <= nb_dist))
            subdata.flags |= FLAG_BOID_OBJECT;
      }

      else
         subdata.flags |= data->flags & (FLAG_BOID_LEAD | FLAG_BOID_OBJECT);

   if (!subdata.flags)
      return;

   if ((data->target->old_state.state_flags & STATE_MASK_BOUND) != STATE_FLAG_BPLANE && (old_state.state_flags & STATE_MASK_BOUND) != STATE_FLAG_BPLANE && (subdata.flags & (FLAG_BOID_LEAD | FLAG_BOID_OBJECT))) {

      bc1 = ((old_state.state_flags & STATE_MASK_BOUND) == STATE_FLAG_BSPHERE) ? old_state.bcenter : old_state.center;
      bc2 = ((data->target->old_state.state_flags & STATE_MASK_BOUND) == STATE_FLAG_BSPHERE) ? data->target->old_state.bcenter : data->target->old_state.center;
	  
      subeqarray3(reflect, bc2, bc1);
      temp = (float)(sqrt(dotproduct3(reflect, reflect)) - (data->target->old_state.bradius + old_state.bradius));
	 
      sphereflag = data->target->query_whatami();

      if (sphereflag == OBJECT_BOIDBAIT && (subdata.flags & FLAG_BOID_LEAD) && (!nearest_leader || temp < nl_dist)) {
         nearest_leader = data->target;
         nl_dist = temp;
      }

      if ((sphereflag == OBJECT_BOIDBAIT || sphereflag == OBJECT_BOID) && (subdata.flags & FLAG_BOID_OBJECT) && (!nearest_boid || temp < nb_dist)) {
         nearest_boid = data->target;
         nb_dist = temp;
      }

   }

   subdata.parent = data->target;
   old_alert = alert;
   
   for (ptr = (linktype *)data->target->edge.head; ptr; ptr=(linktype *)ptr->next)
      if (ptr->link != data->parent) {
         subdata.target = ptr->link;
         calc_dist_data(&subdata);

         if (old_alert != alert) {
            old_alert = alert;
            if (alert == RED_ALERT)
               subdata.flags = FLAG_BOID_ALERT;
         }

      }

}


/* *************************************************************
************************************************************* */
void boid::begin(dbl_llist_manager *hiearchy_manager, quark *parent, vector4f *mx) {

   linktype *ptr;

   copyarray3(local_dir, old_vz);
   normalize3(local_dir);

   inversemx(mx, old_imx);				// backup global inverse mx
   old_local_pos[0] = initxform[0][3];
   old_local_pos[1] = initxform[1][3];
   old_local_pos[2] = initxform[2][3];

   copyarray3(state.xmx[0], old_vx);
   state.xmx[0][3] = 0;
   copyarray3(state.xmx[1], old_vy);
   state.xmx[1][3] = 0;
   copyarray3(state.xmx[2], old_vz);
   state.xmx[2][3] = 0;
   copyarray3(state.xmx[3], old_local_pos);
   state.xmx[3][3] = 1;

   transpose(state.xmx);
   matmatmulto(mx, state.xmx);

   global_dir[0] = state.xmx[0][2];
   global_dir[1] = state.xmx[1][2];
   global_dir[2] = state.xmx[2][2];
   normalize3(global_dir);

   copymx4x4o(state.node, state.xmx);

   for (ptr=(linktype *)edge.head; ptr; ptr=(linktype *)ptr->next)             
      if (ptr->link != parent)                     
         ptr->link->begin(hiearchy_manager, this, state.node);

   state.center[0] = state.xmx[0][3];
   state.center[1] = state.xmx[1][3];
   state.center[2] = state.xmx[2][3];
}


/* *************************************************************
    NOTE: boid class is independent on external xforms....
          and this object is not actually drawn - its an
          animation device, not a 3d object (its child is though :) )

// FUTURE:
//6) note: for animals - if going down, glide
//                        if going up, flap

************************************************************* */
void boid::whereami(dbl_llist_manager *hiearchy_manager, quark *parent, vector4f *mx) {

   linktype *ptr;
   atom_list_type *atr;
   
   boid_avoid_type avoid_data;

   vector3f want, work;
   vector4f stuff;
   float mag_work, mag_want;

   float weight;
   float temp;
   int old_alert;
   
   nl_dist = nb_dist = 0.0;
   nearest_leader = nearest_boid = NULL;
   avoid_dir[0] = avoid_dir[1] = avoid_dir[2] = 0.0;

   avoid_data.flags = (FLAG_BOID_ALERT | FLAG_BOID_LEAD | FLAG_BOID_OBJECT);
   avoid_data.parent = NULL;
   old_alert = alert = NO_COLLISION;

   for (atr = (atom_list_type *)hiearchy_manager->head; atr; atr = (atom_list_type *)atr->next)
      for (ptr=(linktype *)atr->htree->edge.head; ptr; ptr = (linktype *)ptr->next) {
         avoid_data.target = ptr->link;
         calc_dist_data(&avoid_data);

         if (old_alert != alert) {
	    old_alert = alert;
	    if (alert == RED_ALERT)
	       avoid_data.flags = FLAG_BOID_ALERT;
	 }
	 
      }

   copyarray3(want, avoid_dir);

   temp = dotproduct3(want, want);

   if (alert == RED_ALERT) {                    // collision eminent
      if (temp) {				
         temp = (float)(maxspeed/sqrt(temp));
         smultarray3(want, temp);
         velocity *= icl;			// if collision put in, remove this...
      }

      // undecided - continue present course at average speed...
      else {
         want[0] = old_state.xmx[0][2];
         want[1] = old_state.xmx[1][2];
         want[2] = old_state.xmx[2][2];
         temp = (float)(((maxspeed+minspeed)*0.5)/magnitude3(want));
         smultarray3(want, temp);
      }

   }

   else {
      weight = 1;

      if (alert == YELLOW_ALERT) {                              	// possible collisions...
         if (temp) {							// undecided - continue present course at average speed...
            temp = (float)(lerp(ya_weight, minspeed, maxspeed)/sqrt(temp)); 	// future - figure percentage based on how close
            smultarray3(want, temp);
            weight -= ya_weight;
         }
      }

      else
         want[0] = want[1] = want[2] = 0;

      if (weight > CORRECT) {
         if (nearest_leader)           // nearest flock leader
            if (nl_dist > nl_max_radius)
               add2path(nearest_leader->old_state.center, old_state.center, nl_dist, nl_max_radius, &weight, want);
            else if (nl_dist < nl_min_radius)
               sub2path(nearest_leader->old_state.center, old_state.center, nl_dist, nl_min_radius, &weight, want);

         if (weight > CORRECT) {
            if (nearest_boid)          // nearest boid
               if (nb_dist > nb_max_radius)
                  add2path(nearest_boid->old_state.center, old_state.center, nb_dist, nb_max_radius, &weight, want);
               else if (nb_dist < nb_min_radius)
                  sub2path(nearest_boid->old_state.center, old_state.center, nb_dist, nb_min_radius, &weight, want);

            if (weight > CORRECT) {                // part of previous path at average speed
               copyarray3(work, global_dir);
               temp = weight * 0.5f;
               temp = lerp(temp, minspeed, maxspeed);
               weight = 0;

               want[0] += work[0] * temp;
               want[1] += work[1] * temp;
               want[2] += work[2] * temp;
            }

         }

      }

   }

   // at this stage, want == direction of where want to be in global frame

   matvecmultv(old_imx, want);   // want = dir in local frame

   temp = velocity*ifps;     // slow down - inverse friction
   work[0] = local_dir[0] * temp;
   work[1] = local_dir[1] * temp;
   work[2] = local_dir[2] * temp;

   if (gflag)               // gravity
      addarray3(work, gravity);

      // at this stage, work == direction of where going in global frame
      //    		want == where we want to go

   mag_want = normalize3(want);
   mag_work = normalize3(work);

   temp = ACOS(dotproduct3(want, work));

   // turn as much as possible
   if (temp > max_turn_rad) {

      vector_interp_setup(work, want, 3, 1.0f/temp, stuff, &temp);

      stuff[3] *= max_turn_rad;
      stuff[0] -= stuff[3];
      stuff[1] += stuff[3];

      vector_interp_eval(work, want, 3, stuff, want);
   }

   else
      copyarray3(work, want);

   copyarray3(local_dir, want);

   // now shift speed as much as posible

   temp = mag_want - mag_work;

   if (temp > max_accel)
      velocity = mag_work + max_accel;
   else if (-temp > max_accel)
      velocity = mag_work - max_accel;
   else
      velocity = mag_want;

   if (velocity > maxspeed)
      velocity = maxspeed;
   else if (velocity < minspeed)
      velocity = minspeed;

   calc_orientation(local_dir);

   inversemx(mx, old_imx);				// backup global inverse mx
   old_local_pos[0] += local_dir[0] * velocity;
   old_local_pos[1] += local_dir[1] * velocity;
   old_local_pos[2] += local_dir[2] * velocity;

   copyarray3(state.xmx[0], old_vx);
   state.xmx[0][3] = 0;
   copyarray3(state.xmx[1], old_vy);
   state.xmx[1][3] = 0;
   copyarray3(state.xmx[2], old_vz);
   state.xmx[2][3] = 0;
   copyarray3(state.xmx[3], old_local_pos);
   state.xmx[3][3] = 1;
   
   transpose(state.xmx);
   matmatmulto(mx, state.xmx);

   global_dir[0] = state.xmx[0][2];
   global_dir[1] = state.xmx[1][2];
   global_dir[2] = state.xmx[2][2];
   normalize3(global_dir);

   copymx4x4o(state.node, state.xmx);

   for (ptr=(linktype *)edge.head; ptr; ptr=(linktype *)ptr->next)  
      if (ptr->link != parent) 
         ptr->link->whereami(hiearchy_manager, this, state.node);

   state.center[0] = state.xmx[0][3];
   state.center[1] = state.xmx[1][3];
   state.center[2] = state.xmx[2][3];
}

