

#include <stdlib.h>
#include <string.h>

#include "quark.h"
#include "pstring.h"


/* *************************************************************
************************************************************* */
void eventtype::evaluate(int frameno, atom *source, dbl_llist_manager *hiearchy_manager) {

   FILE      *infile;                           // input file
   char      token[MAXSTRLEN];
   char      parent[MAXSTRLEN], dest[MAXSTRLEN];// temp string
   int       number;                            // # of changes during an event
   int       j;                                 // loop vars
   float     temp;                              // temp array
   quark     *qtr;                              // quark pointer
   
   if (!find_file(efilename.string, "r", EVENT_PATH.string, (char)PLATFORM_SLASH, NULL, &infile)) {
      sprintf(perror_buffer, "Warning: can't access %s...\n", efilename.string);
      pprintf(perror_buffer);
      return;
   }

   get_token(infile, token);

   for (number=atoi(token); number>0; number--) {
      get_token(infile, token);
      strcpy(dest, token);                   // backup in case of quark name
      lower_case(token);

      if (!strcmp(token, "composite")) {     // the following event affects atom
         get_token(infile, token);
         lower_case(token);

         if (!strcmp(token, "spline"))       // xyz dof change
            source->motion.read_spline(infile, timefactor);
         else                                // quaternion dof
            source->motion.read_quaternion(infile, timefactor);
      }

      else if (!strcmp(token, "move")) {       // change parent of a child
         get_token(infile, token);
         get_token(infile, parent);
         source->move_quark(token, parent);
      }

      else if (!strcmp(token, "drop")) {     // make a child indep. object
         get_token(infile, token);
         source->drop_quark(frameno, token, hiearchy_manager, infile);
      }

      // absorb another object
      else if (!strcmp(token, "absorb")) {
         get_token(infile, token);
         get_token(infile, parent);
         if (!source->name.stringcmp(token))
            source->join_atom(token, parent, hiearchy_manager);
      }

      else if (!strcmp(token, "take")) {// take child from object
         get_token(infile, token);
         get_token(infile, parent);
         get_token(infile, dest);

         source->take_quark(dest, parent, token, hiearchy_manager);
      }

      else if (!strcmp(token, "scale")) { // global scale of object
         get_token(infile, token);
         temp = (float)atof(token);
         get_token(infile, token);
         source->scale_quark(temp, token);
      }

      else {                     // child event
         qtr = source->find((quark *)NULL, dest);
	    
         if (qtr)
            qtr->new_action(infile, timefactor);
         else {                          // cant find item, so junk event
            sprintf(perror_buffer, "Can't find %s\n", dest);
            pprintf(perror_buffer);
	       
            get_token(infile, token);
            lower_case(token);

            if (!strcmp(token, "spline") || !strcmp(token, "quaternion")) {
               get_token(infile, token);
               j = atoi(token);
               get_token(infile, token);
               get_token(infile, token);
               get_token(infile, token);

               for (; j>-1; j--) {
                  get_token(infile, token);
                  get_token(infile, token);
                  get_token(infile, token);
               }

            }

         }

      }

   }

   fclose(infile);
}


/* *************************************************************
************************************************************* */
void atom::render_object(engine *proc, quark *parent, vector4f *frustum, unsigned int frustum_flag) {

   linktype *ptr;

   // cull node w/ view frustum
   if ((old_state.state_flags & STATE_MASK_TREE) == STATE_FLAG_TNONE)
      frustum_flag = 0;

   if (frustum_flag && (old_state.state_flags & STATE_MASK_TREE) == STATE_FLAG_TSPHERE && 
       query_frustum_clip(old_state.tree_radius, old_state.tree_center, frustum, &frustum_flag, frustum_fail_data))  {  
      flags &= ~QUARK_FLAG_VISIBLE;
      return;
   }

   flags |= QUARK_FLAG_VISIBLE;

   for (ptr=(linktype *)edge.head; ptr; ptr=(linktype *)ptr->next)
      ptr->link->render_object(proc, this, frustum, frustum_flag);
}


/* *************************************************************
************************************************************* */
void atom::new_action(FILE *infile, float  timefactor, char *buffer) {
}


/* *************************************************************
************************************************************* */
int atom::query_whatwasi(int type) {

   return (atom::query_whatami() == type) ? 1 : quark::query_whatwasi(type);
}


/* *************************************************************
   This is the destructor for this class
************************************************************* */
int atom::parse(FILE *infile, char *token) {

   if (!quark::parse(infile, token))
      read_event(infile, token, 0);

   return 1;
}


/* *************************************************************
   Read in an object's data
************************************************************* */
int atom::read_data(char *filename) {

   FILE *infile;
   
   if (!find_file(filename, "r", ATOM_PATH.string, (char)PLATFORM_SLASH, NULL, &infile))
      return 0;

   name.stringcpy(filename);

   while (read_quark(infile, 0));

   fclose(infile);
   return 1;
}


/* *************************************************************
************************************************************* */
basic_event *atom::read_event(FILE *infile, char *token, int frame_offset) {

   eventtype *ptr;
   basic_event *qtr;

   ptr = new eventtype;

   get_token(infile, token);
   ptr->event = atoi(token) + frame_offset;

   get_token(infile, token);
   get_token(infile, token);
   ptr->timefactor = (float)atof(token);

   get_token(infile, token);
   get_token(infile, token);
   ptr->efilename.stringcpy(token);

   for (qtr = (basic_event *)event_manager.head; qtr && qtr->event < ptr->event; qtr = (basic_event *)qtr->next);
   
   if (qtr)
      event_manager.insert(ptr, qtr);
   else
      event_manager.append(ptr, NULL);
   
   return ptr;
}


/* *************************************************************
   This function implements new events
************************************************************* */
void atom::new_action(int frameno, dbl_llist_manager *hiearchy_manager) {

   basic_event *ptr;                            // event pointer
   
   for (ptr=(basic_event *)event_manager.head; ptr && frameno >= ptr->event; ptr = (basic_event *)event_manager.head) {
      ptr->evaluate(frameno, this, hiearchy_manager);
      event_manager.remove(ptr);
      delete ptr;
   }
      
}


/* ********************************************************
******************************************************** */
int atom::scale_quark(float s, char *part) {

   quark *target, *parent;
   
   target = find(NULL, part, &parent);

   if (target) {
      target->apply_scale(s); 
      return 1;
   }
   	 
   sprintf(perror_buffer, "Could not scale %s\n", part);
   pprintf(perror_buffer);
   return 0;
}


/* ********************************************************
   This function changes the parent of a child - moves it w/in its own atom tree
******************************************************** */
int atom::move_quark(char *child, char *parent) {

   quark *ptr;                         // temp pointers
   quark *old_parent = NULL;           // temp pointer
   quark *new_parent;
   
   // find "child" and make sure not to assign a child to a node in the child's tree
   ptr = find(NULL, child, &old_parent);

   if (!ptr || ptr->find(NULL, parent)) {
      sprintf(perror_buffer, "Could not move %s to %s\n", child, parent);
      pprintf(perror_buffer);
      return 0;
   }

   // find new parent
   new_parent = !strcmp(parent, TOKEN_NULL_STR) ? this : find(NULL, parent);
   
   if (!new_parent) {
      sprintf(perror_buffer, "Could not move %s to %s\n", child, parent);
      pprintf(perror_buffer);
      return 0;
   }
   
   if (old_parent) {
      old_parent->remove_link(ptr);
      ptr->remove_link(old_parent);
   }
   
   else
      remove_link(ptr);
      

   ptr->create_link(new_parent);
   new_parent->create_link(ptr);

   return 1;
}


/* ********************************************************
   This function makes a child an independent object
******************************************************** */
int atom::drop_quark(int frameno, char *child, dbl_llist_manager *hiearchy_manager, FILE *infile) {

   atom_list_type *tree;
   atom      *atr;                              // new object
   quark     *old_child;                        // child to drop
   quark     *old_parent = (quark *)NULL;       // parent of child
   char      token[MAXSTRLEN];                  // temp string
   linktype  *ptr;

   old_child = find(NULL, child, &old_parent);

   if (!old_child) {  // cant find child....
      get_token(infile, token);
      get_token(infile, token);
      get_token(infile, token);

      do {
         if (!get_token(infile, token) || token[0] == '}')
            break;
      } while (1);

      return 0;
   }

   if (!old_parent) {  // since root node is to be dropped, reuse old atom
      event_manager.dest();
      atr = this;
   }

   else {
      old_parent->remove_link(old_child);
      old_child->remove_link(old_parent);

      // make new object
      hiearchy_manager->append(tree = new atom_list_type, NULL);
      tree->htree = atr = new atom;
      atr->edge.insert(ptr = new linktype, NULL);
      ptr->link = old_child;
   }

   // read object events
   get_token(infile, token);
   atr->name.stringcpy(token);
   get_token(infile, token);                // "{"

   do {
      if (!get_token(infile, token) || token[0] == '}')
         break;

      atr->read_event(infile, token, frameno);
   } while (1);

   // update object
   atr->new_action(frameno, hiearchy_manager);

   return 1;
}


/* ********************************************************
   This function takes another object, and makes it
   a part of it.

   NOTE: events from one does not transfer over to the new object
******************************************************** */
int atom::join_atom(char *part, char *parent, dbl_llist_manager *hiearchy_manager) {

   atom     *atr, *btr;                 // atom pointers
   quark    *ptr;                       // quark pointers
   char     token[MAXSTRLEN];
   quark    *qtr;
   linktype *rtr;
   atom_list_type *tree;

   strcpy(token, parent);
   lower_case(token);

   // move to a child
   if (strcmp(token, "link")) {
      ptr = find((quark *)NULL, parent);
      if (!ptr)
         return 0;
   }

   // coexists w/ current children
   else 
      ptr = this;

   for (tree = (atom_list_type *)hiearchy_manager->head; tree && tree->htree->name.stringcmp(part); tree = (atom_list_type *)tree->next);

   if (!tree)
      return 0;

   // remove atom from the master list
   hiearchy_manager->remove(tree);
   btr = tree->htree;
   tree->htree = NULL;
   delete tree;

   for (rtr = (linktype *)btr->edge.head; rtr; rtr = (linktype *)rtr->next) {
      qtr = rtr->link;
      rtr->link = NULL;

      ptr->create_link(qtr);
      qtr->create_link(ptr);
   }
   
   // delete old atom
   delete btr;

   return 1;
}


/* ********************************************************
   This function takes part of another object, and makes it
   a part of itself
******************************************************** */
int atom::take_quark(char *dest, char *parent, char *part, dbl_llist_manager *hiearchy_manager) {

   atom_list_type *atr;               // target pointer
   quark *child, *new_parent;         // quark pointers
   quark *old_parent = NULL;          // quark pointer
   char buffer[MAXSTRLEN];

   for (atr = (atom_list_type *)hiearchy_manager->head; atr && atr->htree->name.stringcmp(parent); atr=(atom_list_type *)atr->next);

   if (!atr || !(child = atr->htree->find(NULL, part, &old_parent))) {
      sprintf(buffer, "Could not take %s from %s to %s\n", part, parent, dest);
      pprintf(buffer);
      return 0;
   }

   new_parent = !strcmp(dest, TOKEN_NULL_STR) ? this : find(NULL, dest);
   
   if (!new_parent) {
      sprintf(buffer, "Could not take %s from %s to %s\n", part, parent, dest);
      pprintf(buffer);
      return 0;
   }

   if (old_parent) {
      old_parent->remove_link(child);
      child->remove_link(old_parent);
   }

   else {
      atr->htree->remove_link(child);

      // wipe out empty node
      if (!atr->htree->edge.head) {
         hiearchy_manager->remove(atr);
         delete atr;
      }

   }
   
   new_parent->create_link(child);
   child->create_link(new_parent);

   return 1;
}

