

#include "winsnd.h"
#include "global.h"


#define WAV_LIMIT 64


/* **************************************************************************
************************************************************************** */
int wav_sample_type::parse(char *filename) {

   unsigned int riff;
   unsigned int length;
   unsigned int type;
   unsigned char *end;

   wav_info = NULL;
   wav_raw = NULL;

   if (!wav_file.scan_data(filename))
      return 0;
      
   wav_file.sseek(0);

   riff   = wav_file.scan_ruint();
   if (riff != CHAR2INT('R', 'I', 'F', 'F'))
      return 0;

   length = wav_file.scan_ruint();

   type   = wav_file.scan_ruint();
   if (type != CHAR2INT('W', 'A', 'V', 'E'))
      return 0;

   // Get the pointer to the end of our wave memory
   end = wav_file.data + length - 4;

   while ((unsigned char *)wav_file.ptr < end) {

      type   = wav_file.scan_ruint();
      length = wav_file.scan_ruint();

      // Located the format portion
      if (type == CHAR2INT('f', 'm', 't', ' ')) {

         if (!wav_info) {
            if (length < sizeof(WAVEFORMAT))
               return 0;                          // Drop out because this is not a WAV

            // Set the lplpWaveHeader parameter to point to this piece of the memory
            wav_info = (wav_header *)wav_file.ptr;

            if (wav_raw)
              return 1;
         }

      }

      // We are at the the samples - fill the values to return for the sample and size
      else if (type == CHAR2INT('d', 'a', 't', 'a')) {

         if (!wav_raw) {
            wav_raw = (char *)wav_file.ptr;
            wav_length = length;

            // If the header pointer is filled, we can return happy..
            if (wav_info)
               return 1;
         }

      }

      // Set the pointer to the next portion of memory
      wav_file.skip((length+1) & 0xfffffffe);
   }

   return 0;
}


/* **************************************************************************
************************************************************************** */
int wav_sample_type::update(IDirectSoundBuffer *device) {

   VOID *wavebits1;                             // A pointer to the first block of sound data
   VOID *wavebits2;                             // A pointer to the second block of sound data
   DWORD length1, length2;                      // Length of the first/second block of data
   
   if (!device)
      return 0;

   // We can copy blocks of sound data into the buffer
   if (device->Lock (
       0,                // The offset into the buffer where we will start writing
       wav_length,       // The size of the wave file to copy in
       &wavebits1,       // The first block of sound data
       &length1,         // The length of the first block of data
       &wavebits2,       // The second block of sound data
       &length2,         // The length of the second block of data
       0) != DS_OK) {    // Flags

      sprintf(perror_buffer, "WARNING: Unable to lock and update wav \"%s\"...\n", wav_file.filename);
      pprintf(perror_buffer);
      return 0;
   }

   // Copy the first block of data from the wave file
   memcpy(wavebits1, wav_raw, length1);

   // Copy the second block of data from the wave file
   if (length2)
      memcpy(wavebits2, wav_raw + length1, length2);

   // Unlock our buffer
   if (device->Unlock(wavebits1, length1, wavebits2, length2) != DS_OK) {
      sprintf(perror_buffer, "WARNING: Unable to unlock and update wav \"%s\"...\n", wav_file.filename);
      pprintf(perror_buffer);
      return 0;
   }

   return 1;
}


/* **************************************************************************
************************************************************************** */
IDirectSoundBuffer *wav_sample_type::init(IDirectSound *driver, unsigned int flags) {

   DSBUFFERDESC dsbd;
   HRESULT hresult;
   IDirectSoundBuffer *device;
      
   // Set up our direct sound buffer.
   memset(&dsbd, 0, sizeof(DSBUFFERDESC));

   // Now set up a buffer on the sound card's memory (DSBCAPS_STATIC)
//   dsbd.dwFlags        = DSBCAPS_STATIC | DSBCAPS_CTRLVOLUME | DSBCAPS_CTRLFREQUENCY;
   dsbd.dwFlags        = (flags & FLAG_SOUND_3D) ? DSBCAPS_CTRL3D | DSBCAPS_CTRLVOLUME | DSBCAPS_MUTE3DATMAXDISTANCE :
                         DSBCAPS_STATIC | DSBCAPS_CTRLVOLUME;
   dsbd.dwSize         = sizeof(DSBUFFERDESC);
   dsbd.dwBufferBytes  = wav_length;
   dsbd.lpwfxFormat    = (LPWAVEFORMATEX)wav_info;

   // Now, create our sound buffer
   hresult = driver->CreateSoundBuffer(&dsbd, &device, NULL);

   if (hresult != DS_OK) {
      sprintf(perror_buffer, "Cannot Create sound buffer \"%s\"... ", wav_file.filename);
      pprintf(perror_buffer);
      
      switch (hresult) {
         case DSERR_ALLOCATED:
            pprintf("DSERR_ALLOCATED\n");
            break;
         case DSERR_BADFORMAT:
            pprintf("DSERR_BADFORMAT\n");
            break;
         case DSERR_INVALIDPARAM:
            pprintf("DSERR_INVALIDPARAM\n");
            break;
         case DSERR_NOAGGREGATION:
            pprintf("DSERR_NOAGGREGATION\n");
            break;
         case DSERR_OUTOFMEMORY:
            pprintf("DSERR_OUTOFMEMORY\n");
            break;
         case DSERR_UNINITIALIZED:
            pprintf("DSERR_UNINITIALIZED\n");
            break;
         case DSERR_UNSUPPORTED:
            pprintf("DSERR_UNSUPPORTED\n");
            break;
//         case DSERR_GENERIC:
         default:
            pprintf("DSERR_GENERIC\n");
            break;
      }

      return NULL;
   }

   if (update(device))
      return device;

   sprintf(perror_buffer, "WARNING: Unable to create soundbuffer \"%s\"... Ignoring...\n", wav_file.filename);
   pprintf(perror_buffer);
   device->Release();
   return NULL;
}


/* **************************************************************************
************************************************************************** */
wav_play_type::~wav_play_type() {

   if (device)
      device->Release();
}


/* **************************************************************************
************************************************************************** */
void wav_play_type::volume(int v) {

   device->SetVolume((int)(DSBVOLUME_MIN + v**volumex*0.0001*(DSBVOLUME_MAX-DSBVOLUME_MIN)));
}


/* **************************************************************************
************************************************************************** */
void wav_play_type::update() {

   DWORD status;

   if (!(flags & FLAG_SOUND_PLAYING))
      return;
      
   device->GetStatus(&status);

   if (flags & FLAG_SOUND_LOOPING) {
      
      if (status & DSBSTATUS_BUFFERLOST) {
         device->Restore();
         ((wav_sample_type *)id)->update(device);
         if (!(status & DSBSTATUS_PLAYING))
            device->Play(0, 0, DSBPLAY_LOOPING);
      }
      	 
   }
   
   else if (!(status & DSBSTATUS_PLAYING))
      flags &= ~FLAG_SOUND_PLAYING;
}


/* **************************************************************************
************************************************************************** */
void wav_play_type::stop() {

   device->Stop();
   flags &= ~(FLAG_SOUND_LOCK | FLAG_SOUND_PLAYING);
}


/* **************************************************************************
************************************************************************** */
wav_play3d_type::~wav_play3d_type() {

   if (device3d)
      device3d->Release();

   if (device)
      device->Release();
}


/* **************************************************************************
************************************************************************** */
void wav_play3d_type::volume(int v) {

   device->SetVolume((int)(DSBVOLUME_MIN + v**volumex*0.0001*(DSBVOLUME_MAX-DSBVOLUME_MIN)));
}


/* **************************************************************************
************************************************************************** */
void wav_play3d_type::update() {

   DWORD status;

   if (!(flags & FLAG_SOUND_PLAYING))
      return;
      
   device->GetStatus(&status);

   if (flags & FLAG_SOUND_LOOPING) {
      
      if (status & DSBSTATUS_BUFFERLOST) {
         device->Restore();
         ((wav_sample_type *)id)->update(device);
         if (!(status & DSBSTATUS_PLAYING))
            device->Play(0, 0, DSBPLAY_LOOPING);
      }
      	 
   }
   
   else if (!(status & DSBSTATUS_PLAYING))
      flags &= ~FLAG_SOUND_PLAYING;
}


/* **************************************************************************
************************************************************************** */
void wav_play3d_type::location(float *pos) {

   device3d->SetPosition(pos[0], pos[1], pos[2], DS3D_DEFERRED);
}


/* **************************************************************************
************************************************************************** */
void wav_play3d_type::stop() {

   device->Stop();
   flags &= ~(FLAG_SOUND_LOCK | FLAG_SOUND_PLAYING);
}


/* **************************************************************************
************************************************************************** */
void wav_manager::dest() {

   limit = 0;
   sample_manager.dest();
   play_manager.dest();
}


/* **************************************************************************
************************************************************************** */
int wav_manager::find(sound_id_type *sound_id) {

   wav_sample_type *wtr;

   for (wtr = (wav_sample_type *)sample_manager.head; wtr; wtr = (wav_sample_type *)wtr->next)
      if (!wtr->wav_file.filename.stringcmp(sound_id->name.string)) {
         sound_id->id = wtr;
         return 1;
      }

   sample_manager.append(wtr = new wav_sample_type, NULL);
   if (!wtr->parse(sound_id->name.string)) {
      sample_manager.remove(wtr);
      delete wtr;
      return 0;
   }

   sound_id->id = wtr;
   return 1;
}


/* **************************************************************************
************************************************************************** */
sound_type *wav_manager::play(sound_id_type *sound_id, IDirectSound *driver, int *volumex) {

   IDirectSoundBuffer *device;
   IDirectSound3DBuffer *device3d;
   wav_play_type *wtr;
   wav_play3d_type *wtr3d;

   if (!sound_id->id || limit >= WAV_LIMIT)
      return NULL;
      
   device = ((wav_sample_type *)sound_id->id)->init(driver, sound_id->flags);

   if (!device)
      return NULL;

   device->SetCurrentPosition(0);
   device->Play(0,0, (sound_id->flags & FLAG_SOUND_LOOPING) ? DSBPLAY_LOOPING : 0);

   if (sound_id->flags & FLAG_SOUND_3D) {
      if (device->QueryInterface(IID_IDirectSound3DBuffer, (LPVOID *)&device3d) != DS_OK) {
         device->Release();
         return NULL;
      }

      device3d->SetMode(DS3DMODE_NORMAL, DS3D_DEFERRED);

      device3d->SetMinDistance(1000, DS3D_DEFERRED);  // distance at which volume is halved
      device3d->SetMaxDistance(1001, DS3D_DEFERRED);  // distance at which sound is no longer reduced

      play_manager.insert(wtr3d = new wav_play3d_type, NULL);
      wtr3d->id = sound_id->id;

      wtr3d->flags |= FLAG_SOUND_PLAYING | sound_id->flags;
      wtr3d->device = device;
      wtr3d->device3d = device3d;
      wtr3d->volumex = volumex;
      limit++;
      wtr3d->volume(sound_id->db);
      return wtr3d;
   }

   else {
      play_manager.insert(wtr = new wav_play_type, NULL);
      wtr->id = sound_id->id;
      wtr->flags |= FLAG_SOUND_PLAYING | sound_id->flags;
      wtr->device = device;
      wtr->volumex = volumex;
      limit++;
      wtr->volume(sound_id->db);
      return wtr;
   }

}


/* **************************************************************************
************************************************************************** */
void wav_manager::update() {

   sound_type *ptr, *qtr;

   limit = 0;
   ptr = (sound_type *)play_manager.head;
   while (ptr) {
      ptr->update();
      if (ptr->flags & (FLAG_SOUND_PLAYING | FLAG_SOUND_LOCK)) {
         ptr = (sound_type *)ptr->next;
         limit++;
         continue;
      }

      qtr = ptr;
      ptr = (sound_type *)ptr->next;
      play_manager.remove(qtr);
      delete qtr;
   }
      
}

