

#include "WIN_W32.h"
#include "winsnd.h"
#include "global.h"


/* **************************************************************************
************************************************************************** */
soundman_win::soundman_win() {

   window = NULL;
   driver = NULL;
   listener = NULL;

   midi_performer = NULL;
   midi_loader = NULL;
   
   midi.parent = this;
}


/* **************************************************************************
   problem if window is closed before this is called
************************************************************************** */
soundman_win::~soundman_win() {

   dest();
}


/* **************************************************************************
************************************************************************** */
void soundman_win::set_win(VIRTUALwindow *win) {
   
   window = ((WIN32window *)win)->mwindow;
}


/* **************************************************************************
************************************************************************** */
int soundman_win::init() {

   DSBUFFERDESC dsbd;
   WAVEFORMATEX wfm;
   IDirectSoundBuffer *primary;
   
   dest();

   // setup Direct Sound
   if (FAILED(DirectSoundCreate8(NULL, &driver, NULL))) {
      pprintf("Could not create primary sound driver... Aborting...\n");
      return 0;
   }

   driver->SetCooperativeLevel(window, DSSCL_PRIORITY);

   // Now set up primary buffer on the sound card's memory
   memset(&dsbd, 0, sizeof(DSBUFFERDESC));
   dsbd.dwFlags = DSBCAPS_CTRL3D | DSBCAPS_PRIMARYBUFFER;
   dsbd.dwSize = sizeof(DSBUFFERDESC);
   dsbd.dwBufferBytes = 0;  // must be zero for primary buffer..

   // Now, create our primary 3D sound buffer
   if (FAILED(driver->CreateSoundBuffer(&dsbd, &primary, NULL))) {
      pprintf("Could not create primary sound buffer... Aborting...\n");
      dest();
      return 0;
   }

   // setup .wav format
   memset(&wfm, 0, sizeof(WAVEFORMATEX));
   wfm.wFormatTag = WAVE_FORMAT_PCM;
   wfm.nChannels = 2;
   wfm.nSamplesPerSec = 22050;	// 44100;
   wfm.wBitsPerSample = 16;
   wfm.nBlockAlign = 4;		// (wfm.wBitsPerSample*wfm.nCChannels) / 8;
   wfm.nAvgBytesPerSec = 88200; // wfm.nSamplesPerSec*wfm.nBlockAlign;
   
   primary->SetFormat(&wfm);

   // get 3d Listener
   if (FAILED(primary->QueryInterface(IID_IDirectSound3DListener, (void**)&listener))) {
      pprintf("Cannot allocate 3D listener... Aborting\n");
      primary->Release();
      dest();
      return 0;
   }

   primary->Release();

   // setup DirectMusic
   CoInitialize(NULL);

   // Create loader object
   if (FAILED(CoCreateInstance(CLSID_DirectMusicLoader, NULL, CLSCTX_INPROC, IID_IDirectMusicLoader8, (void**)&midi_loader ))) {
      pprintf("Cannot create midi loader... Aborting\n");
      dest();
      return 0;
   }


   // Create performance object
   if (FAILED(CoCreateInstance(CLSID_DirectMusicPerformance, NULL, CLSCTX_INPROC, IID_IDirectMusicPerformance8, (void**)&midi_performer))) {
      pprintf("Cannot allocate DirectMusic performer... Aborting\n");
      dest();
      return 0;
   }

   if (FAILED(midi_performer->InitAudio(NULL, (IDirectSound **)&driver, window, DMUS_APATH_SHARED_STEREOPLUSREVERB, 64, DMUS_AUDIOF_ALL, NULL))) {
      pprintf("Cannot init DirectMusic... Aborting\n");
      dest();
      return 0;
   }

   volume[SOUND_SFX]   = global_sfx_volume;
   volume[SOUND_MUSIC] = global_music_volume;
   volume[SOUND_VOICE] = global_voice_volume;
   return 1;
}


/* **************************************************************************
************************************************************************** */
void soundman_win::reset() {

   midi.dest();
   wav.dest();   

   soundman::reset();
}


/* **************************************************************************
************************************************************************** */
int soundman_win::dest() {

   reset();  // dont matter in destructor - same effect
   
   if (midi_loader) {
      midi_loader->Release();
      midi_loader = NULL;
   
      if (midi_performer) {
         midi_performer->CloseDown();
         midi_performer->Release();
         midi_performer = NULL;
      }
   
      CoUninitialize();
   }

   if (listener) {
      listener->Release();
      listener = NULL;
   }

   if (driver) {
      driver->Release();
      driver = NULL;
   }

   return 1;
}


/* **************************************************************************
************************************************************************** */
sound_type *soundman_win::play(sound_id_type *sound_id) {

   int level;
   
   if (sound_id->flags & FLAG_SOUND_MUSIC)
      level = SOUND_MUSIC;
   else if (sound_id->flags & FLAG_SOUND_VOICE)
      level = SOUND_VOICE;
   else
      level = SOUND_SFX;
      
   if (sound_id->flags & FLAG_SOUND_MIDI)
      return midi.play(sound_id, &volume[level]);
      
   if (sound_id->flags & FLAG_SOUND_WAV)
      return wav.play(sound_id, driver, &volume[level]);
      
   return NULL;
}


/* **************************************************************************
************************************************************************** */
void soundman_win::update(float *pos, float *dir, float *up) {

   midi.update();
   wav.update();

   if (!listener)
      return;

   listener->SetOrientation(dir[0], dir[1], dir[2], up[0], up[1], up[2], DS3D_DEFERRED);
   listener->SetPosition(pos[0], pos[1], pos[2], DS3D_DEFERRED);
   listener->CommitDeferredSettings();
}


/* **************************************************************************
************************************************************************** */
int soundman_win::find(sound_id_type *sound_id) {

   if (sound_id->flags & FLAG_SOUND_MIDI)
      return midi.find(sound_id);
      
   if (sound_id->flags & FLAG_SOUND_WAV)
      return wav.find(sound_id);
      
   return 0;
}

