

#include "WIN_W32.h"
#include "user_io_w32.h"
#include "global.h"


//-----------------------------------------------------------------------------
// Name: EnumJoysticksCallback()
// Desc: Called once for each enumerated joystick. If we find one, create a
//       device interface on it so we can play with it.
//-----------------------------------------------------------------------------
BOOL CALLBACK EnumJoysticksCallback(const DIDEVICEINSTANCE* pdidInstance, void *userinput) {

   HRESULT hr;

   // Obtain an interface to the enumerated joystick.
   hr = ((user_io_win *)userinput)->dio->CreateDevice( pdidInstance->guidInstance, &((user_io_win *)userinput)->joystick, NULL);

   // If it failed, then we can't use this joystick. (Maybe the user unplugged
   // it while we were in the middle of enumerating it.)
   if (FAILED(hr)) 
      return DIENUM_CONTINUE;

    // Stop enumeration. Note: we're just taking the first joystick we get. You
    // could store all the enumerated joysticks and let the user pick.
    return DIENUM_STOP;
}


/* **********************************************************************
********************************************************************** */
user_io_win::~user_io_win() {

   if (mouse) {
      mouse->Unacquire();
      mouse->Release();
   }

   if (keyboard) {
      keyboard->Unacquire();
      keyboard->Release();
   }

   if (joystick) {
      joystick->Unacquire();
      joystick->Release();
   }

   if (dio)
      dio->Release();
}


/* **********************************************************************
********************************************************************** */
void user_io_win::set_win(VIRTUALwindow *win) {

   DIPROPRANGE diprg;
   DIPROPDWORD dipdw;
   unsigned int cflags;

   if (win && ((WIN32window *)win)->mwindow) {
      cflags = DISCL_NONEXCLUSIVE | DISCL_FOREGROUND;
      winhandle = ((WIN32window *)win)->mwindow;
   }
   
   else {
      cflags = DISCL_NONEXCLUSIVE | DISCL_BACKGROUND;
      winhandle = GetDesktopWindow();
   }
   
   if (dio) {

      if (keyboard)
         keyboard->SetCooperativeLevel(winhandle, cflags);

      if (mouse)
         mouse->SetCooperativeLevel(winhandle, cflags);

      if (joystick)
         joystick->SetCooperativeLevel(winhandle, cflags);

      return;
   }

   DirectInput8Create(GetModuleHandle(NULL), DIRECTINPUT_VERSION, IID_IDirectInput8, (void **)&dio, NULL);

   if (!dio)
      return;

   // keyboard
   if (SUCCEEDED(dio->CreateDevice(GUID_SysKeyboard, &keyboard, NULL))) {
      keyboard->SetDataFormat(&c_dfDIKeyboard);
      keyboard->SetCooperativeLevel(winhandle, cflags);
   }

   // mouse
   if (SUCCEEDED(dio->CreateDevice(GUID_SysMouse, &mouse, NULL))) {
      mouse->SetDataFormat(&c_dfDIMouse);
      mouse->SetCooperativeLevel(winhandle, cflags);

      // Set absolute axis mode.
      dipdw.diph.dwSize       = sizeof(DIPROPDWORD);
      dipdw.diph.dwHeaderSize = sizeof(DIPROPHEADER);
      dipdw.diph.dwObj        = 0;
      dipdw.diph.dwHow        = DIPH_DEVICE;
      dipdw.dwData            = DIPROPAXISMODE_ABS;

      mouse->SetProperty(DIPROP_AXISMODE, &dipdw.diph);

      flags |= FLAG_IO_MOUSE_RECENTER;
   }

   // joystick
   if (SUCCEEDED(dio->EnumDevices(DI8DEVCLASS_GAMECTRL, EnumJoysticksCallback, this, DIEDFL_ATTACHEDONLY)) && joystick) {
      joystick->SetDataFormat(&c_dfDIJoystick);
      joystick->SetCooperativeLevel(winhandle, cflags);

      // Set range.
      diprg.diph.dwSize       = sizeof(DIPROPRANGE);
      diprg.diph.dwHeaderSize = sizeof(DIPROPHEADER);
      diprg.diph.dwObj        = 0;
      diprg.diph.dwHow        = DIPH_DEVICE;
      diprg.lMin              = JOYMIN;
      diprg.lMax              = JOYMAX;
      joystick->SetProperty(DIPROP_RANGE, &diprg.diph);

      // Set deadzone.
      dipdw.diph.dwSize       = sizeof(DIPROPDWORD);
      dipdw.diph.dwHeaderSize = sizeof(DIPROPHEADER);
      dipdw.diph.dwObj        = 0;
      dipdw.diph.dwHow        = DIPH_DEVICE;
      dipdw.dwData            = JOYDEAD;
      joystick->SetProperty(DIPROP_DEADZONE, &dipdw.diph);

      // Set saturation.
//      dipdw.dwData            = JOYSAT;
//      joystick->SetProperty(DIPROP_SATURATION, &dipdw.diph);

      povdegree = POV_DEFAULT;
   }

}


/* **********************************************************************
********************************************************************** */
void user_io_win::poll_mouse() {

   DIMOUSESTATE mousestate;
   RECT rect;
   int winx, winy;

   if (!mouse)
      return;
      
   memset(&mousestate, 0, sizeof(DIMOUSESTATE));
   mouse->Acquire();
   mouse->Poll();
   mouse->GetDeviceState(sizeof(DIMOUSESTATE), &mousestate);

   if (mousestate.rgbButtons[0] & 0x80)
      status_buttonpress |= BUTTONLEFT;

   if (mousestate.rgbButtons[2] & 0x80)
      status_buttonpress |= BUTTONMIDDLE;

   if (mousestate.rgbButtons[1] & 0x80)
      status_buttonpress |= BUTTONRIGHT;

   if (flags & FLAG_IO_MOUSE_RECENTER) {
      flags &= ~FLAG_IO_MOUSE_RECENTER;
      center[0] = mousestate.lX;
      center[1] = mousestate.lY;
   }

   GetWindowRect(winhandle, &rect);

   winx = rect.right - rect.left;
   winy = rect.bottom - rect.top;

   joyaxis[AXIS_MOUSE_X] = ((mousestate.lX - center[0])*2.0f)/winx;
   joyaxis[AXIS_MOUSE_Y] = ((mousestate.lY - center[1])*2.0f)/winy;

   tether_mouse(mousestate.lX, mousestate.lY, winx, winy);
}


/* **********************************************************************
********************************************************************** */
void user_io_win::poll_joystick() {

   DIJOYSTATE joystate;
   unsigned int j, mask;

   if (!joystick)
      return;
   
   memset(&joystate, 0, sizeof(DIJOYSTATE));
   joystick->Acquire();
   joystick->Poll();
   joystick->GetDeviceState(sizeof(DIJOYSTATE), &joystate);

   for (j=0, mask = BUTTON1; j<MAXBUTTON; j++, mask += mask)
      if (joystate.rgbButtons[j] & 0x80)
         status_buttonpress |= mask;

   joyaxis[AXIS_X] = (float)(-joystate.lX*JOYNORMALIZE);
   joyaxis[AXIS_Y] = (float)(joystate.lY*JOYNORMALIZE);
   joyaxis[AXIS_Z] = (float)(joystate.lRz*JOYNORMALIZE);
   if (joystate.lZ)
      joyaxis[AXIS_THROTTLE] = (float)(-joystate.lZ*JOYNORMALIZE);  // original value is backwards...
   else if (joystate.rglSlider[0])
      joyaxis[AXIS_THROTTLE] = (float)(-joystate.rglSlider[0]*JOYNORMALIZE);  // original value is backwards...
   else
      joyaxis[AXIS_THROTTLE] = 0.0f;

   if ((joystate.rgdwPOV[0] & 0xffff) != 0xffff)
      povdegree = joystate.rgdwPOV[0] * 0.01f;
}


/* **********************************************************************
********************************************************************** */
void user_io_win::poll_keyboard() {

   MSG message;

   // pole keyboard
   if (keyboard) {
      keyboard->Acquire();
      keyboard->GetDeviceState(MAXKEYS, status_keypress);
   }

   // Check message queue
   while (PeekMessage(&message, NULL, 0, 0, PM_REMOVE) && message.message != WM_QUIT)
      DispatchMessage(&message);
}


/* **********************************************************************
********************************************************************** */
int user_io_win::get_keymap(char *map) {

   if (!keyboard)
      return 0;

   keyboard->Acquire();
   keyboard->GetDeviceState(MAXKEYS, map);
   return 1;
}

