

/* *************************************************************
   This file contains some of the file I/O routines
************************************************************** */


#include <stdlib.h>
#include <string.h>

#include "pstring.h"

#include "particle.h"
#include "spatch.h"
#include "cylinder.h"
#include "cone.h"
#include "sbfast.h"
#include "line.h"
#include "sphere.h"
#include "radiant.h"
#include "image.h"
#include "shaders.h"
#include "spg_load.h"

#include "pixcon.h"


/* **************************************************
   this function parses command line to find user defined files,
   or else use the default files.
************************************************** */
void parseinput(int argc, char *argv[]) {

   int i, j;
   float tgamma;

   for (i=1; i<argc; i++)
      if (*(argv[i]) == '-')
         switch (*(argv[i]+1)) {
            case 'a':                   // aliasing

               if (argv[i][2] == 'e')
                  ANTIALIAS = 2;
               else
                  ANTIALIAS = 1;

               if (i+1 < argc && argv[i+1][0] != '-') {
                  i++;
                  SANTIALIAS.stringcpy(argv[i]);
               }

               else
                  SANTIALIAS.stringcpy("");

               break;

            case 'b':                   // bmp file
               i++;
               if (i < argc) {
                  RGBFILE = IMAGE_BMP;
                  SRGBFILE.stringcpy(argv[i]);
               }

               break;

            case 'g':
               i++;
               if (i < argc) {
                  GAMMA = 1;
                  tgamma = (float)getgamma(atof(argv[i]));

                  memset(GAMMALUT, 0, 256);

                  for (j=0; j<256; j++)
                     PGAMMA[j] = BYTE_LUT((int)(255.0*gammacorrect(tgamma, j/255.0)));

                  memset(&GAMMALUT[512], PGAMMA[255], 512);
               }

               break;

            case 'l':                   // user defined .lst
               i++;
               if (i < argc) {
                  SDEFLST.stringcpy(argv[i]);
                  DEFLST = 1;
               }

               break;

            case 'n':                   // no shadow
               if (strlen(argv[i]) > 4 && argv[i][4] == 'c') {
                  NOSCREEN = 1;
                  PENCILTEST = 0;
               }

               else
                  NOSHADOW = 1;

               break;

            case 'p':                   // pencil test
               PENCILTEST = 1;
               NOSCREEN = 0;
               break;

            case 'r':                   // image file
               i++;
               if (i < argc) {
                  RGBFILE = IMAGE_RGB;
                  SRGBFILE.stringcpy(argv[i]);
               }

               break;

            case 's':                   // silent to file
               SILENT = 1;
               break;

            case 'v':                   // verbose mode
               SILENT = 0;
               break;

            case 'x':
               i++;
               winx = atoi(argv[i]);
               break;

            case 'y':
               i++;
               winy = atoi(argv[i]);
               break;

            default:
               break;
         }

}


/* **************************************************
   // build mipmaps for 2D textures
************************************************** */
void texture_postprocessCB(texbase *tptr) {

   tptr->pre_process_texture();
   tptr->build_mipmaps((dbl_llist_manager *)global_resource_manager->get_resource_object(RESOURCE_IMAGE_LOADER), ((frame_manager *)global_resource_manager)->query_color_byte_order());
}


/***************************************************************************
****************************************************************************/
void insert_shader(char *str, int fcn(void *), void icn() = NULL) {

   shader *ptr;

   ptr = new shader;
   ptr->set(str, (void *)fcn, (void *)icn);
   global_resource_manager->register_resource_object(RESOURCE_SHADER, ptr);
}


/* **************************************************
************************************************** */
void init_shader() {

   insert_shader("marble", marble);                                     // surface: marble texture
   insert_shader("wood", wood);                                         // surface: plywood texture
   insert_shader("wave", wave);                                         // surface: normal manipulator
   insert_shader("checker", checker);                                   // surface: 3D checker pattern
   insert_shader("wood_panel", wood_panel);                             // surface: plywood panels

   insert_shader("moon", moon, moon_init);                              // surface: fractal moon
   insert_shader("planet", planet, planet_init);                        // surface: fractal planet - variation of moon
   insert_shader("cloud", cloud, cloud_init);                           // surface: cloud texture
   insert_shader("stars", stars, stars_init);                           // surface: star texture
   insert_shader("warp", warp, warp_init);                              // surface: displaces xyz for color calculations
   insert_shader("terrain", terrain, terrain_init);                     // surface: color base on altitude
   insert_shader("planet2", planet2, planet2_init);                     // surface: another planet
   insert_shader("cloud2", cloud2, cloud2_init);                        // surface: another cloud texture
   insert_shader("terrain2", terrain2, terrain2_init);                  // surface: another terrain texture
   insert_shader("terrain3", terrain3, terrain3_init);                  // surface: yet another terrain
   insert_shader("cave_ceiling", cave_ceiling, cave_ceiling_init);      // surface: (stalagmite?) shader
   insert_shader("cave_floor", cave_floor, cave_floor_init);            // surface: (stalagtite?) shader

   insert_shader("fog_mus", fog_mus);                                   // atmosphere: musgrave's fog shader
   insert_shader("fog", fog);                                           // atmosphere: another fog shader
   insert_shader("fog2", fog2);                                         // atmosphere: yet another fog shader
   insert_shader("nightfog2", nightfog2);                               // atmosphere: yet another fog shader
   insert_shader("energy_beam", energy_beam);                           // atmosphere: "Vandagraph" electron bolts

   insert_shader("planetclouds", planetclouds);                         // surface: better than cloud2()
   insert_shader("contour", contour);                                   // surface: a weird contour map thingy
   insert_shader("carpet", carpet);                                     // surface: carpet texture

   insert_shader("default_alt", default_alt);                           // sbfast: a height field
   insert_shader("x_alt", x_alt);                                       // sbfast: default_alt w/o clamping
   insert_shader("mnb_alt", mnb_alt);                                   // sbfast: another height field
   insert_shader("barfland_alt", barfland_alt);                         // sbfast: another height field
}


/* **************************************************
************************************************** */
void init_loader(int framecount) {

   if (global_resource_manager)
      delete global_resource_manager;

   global_resource_manager = new render_manager;
   global_resource_manager->init(framecount);
   ((frame_manager *)global_resource_manager)->set_postprocessCBtexture((void *)texture_postprocessCB);
   ((frame_manager *)global_resource_manager)->set_color_byte_order(CBYTE_ORDER_ABGR);

   init_shader();

   global_resource_manager->register_resource_object(RESOURCE_IMAGE_LOADER, new tiff);
   global_resource_manager->register_resource_object(RESOURCE_IMAGE_LOADER, new ppm);
   global_resource_manager->register_resource_object(RESOURCE_IMAGE_LOADER, new ilbm);
   global_resource_manager->register_resource_object(RESOURCE_IMAGE_LOADER, new bmp);
   global_resource_manager->register_resource_object(RESOURCE_IMAGE_LOADER, new jpeg);
   global_resource_manager->register_resource_object(RESOURCE_IMAGE_LOADER, new gif);
   global_resource_manager->register_resource_object(RESOURCE_IMAGE_LOADER, new rgb);
   global_resource_manager->register_resource_object(RESOURCE_IMAGE_LOADER, new rtf);

   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new camera_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new light_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new far_light_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new point_light_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new beam_far_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new beam_point_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new simple_far_light_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new simple_point_light_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new simple_beam_far_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new simple_beam_point_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new polygon_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new spatch_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new line_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new point_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new sphere_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new cone_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new cylinder_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new radiant_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new sbfast_loader);

   global_resource_manager->register_resource_object(RESOURCE_POLYGON_LOADER, new spg_loader);
}


/* *************************************************************
************************************************************* */
int parse_list(char *filename, int *mapcount) {

   FILE  *infile;                // input file name
   frameclass *ptr;                // pointers to objects
   char  token[MAXSTRLEN], buffer[MAXSTRLEN];
   dbl_llist_manager *object_loader;

   int   i;
   int   objectcount;            // # of objects

   if (!(infile = fopen(filename, "r"))) {
      printf("WARNING: Cannot access .list file %s...\n", filename);
      return 0;
   }

   get_token(infile, token);
   lower_case(token);

   if (strcmp(token, TOKEN_LIST_STR)) {
      printf("%s is not a .list file...\n", filename);
      fclose(infile);
      return 0;
   }

   get_token(infile, token);
   get_token(infile, token);
   *mapcount = atoi(token);

   get_token(infile, token);
   get_token(infile, token);
   objectcount = atoi(token);

   init_loader(*mapcount);
   object_loader = (dbl_llist_manager *)global_resource_manager->get_resource_object(RESOURCE_OBJECT_LOADER);

   if (!object_loader->head) {
      printf("WARNING: No object loaders registered...\n");
      fclose(infile);
      return 0;
   }

   for (i=0; i<objectcount; i++) {
      get_token(infile, token);
      lower_case(token);
      get_token(infile, buffer);            // "{"

      if (!(ptr = (frameclass *)((loader *)object_loader->head)->parse(infile, token))) {
         printf("Invalid object type \"%s\"\n", token);
         fclose(infile);
         exit(1);
      }

      if (ptr->frame >= *mapcount) {
         delete ptr;
         continue;
      }

      ptr->preprocess(NULL);

      switch (ptr->query_category()) {

         case CLASS_OBJECT:
            ((pc *)ptr)->id = i+1;

//         case CLASS_LIGHT:
//         case CLASS_CAMERA:
         default:
            global_resource_manager->register_resource_object(RESOURCE_FRAME_OBJECT, ptr);
            break;
      }

   }

   fclose(infile);
   return 1;
}

