

#include <stdlib.h>
#include <string.h>

#include "pstring.h"

#include "darkgine.h"
#include "darkfx.h"
#include "fighter.h"
#include "thruster.h"


#define FLAG_SPEED_SETTING_NULL   0
#define FLAG_SPEED_SETTING_STOP   1
#define FLAG_SPEED_SETTING_CHANGE 2


/* *************************************************************
************************************************************* */
class sound_node : public dbl_llist {

   public:
      sound_type *sndfx;
      vector4f offset;
    
      sound_node() { sndfx = NULL; }
      virtual ~sound_node() { if (sndfx) sndfx->stop(); }
};
   

/* *************************************************************
************************************************************* */
player::player() {

   jump_sndfx = NULL;
   global_player = this;
}


/* *************************************************************
************************************************************* */
int player::query_whatwasi(int type) {

   return (player::query_whatami() == type) ? 1 : fighter::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int player::parse(FILE *infile, char *token) {

   if (!strcmp(token, TOKEN_COLLISION_SFX_STR)) {
      get_token(infile, token);
      collision_sndfx_id.name.stringcpy(token);
      collision_sndfx_id.flags = FLAG_SOUND_WAV | FLAG_SOUND_3D;
      return 1;
   }
   
   if (!strcmp(token, TOKEN_COLLISION_SFX_VOLUME_STR)) {
      get_token(infile, token);
      collision_sndfx_id.volume(atoi(token));
      return 1;
   }
      
   if (!strcmp(token, TOKEN_JUMP_SFX_STR)) {
      get_token(infile, token);
      jump_sndfx_id.name.stringcpy(token);
      jump_sndfx_id.flags = FLAG_SOUND_WAV;
      return 1;
   }
   
   if (!strcmp(token, TOKEN_JUMP_SFX_VOLUME_STR)) {
      get_token(infile, token);
      jump_sndfx_id.volume(atoi(token));
      return 1;
   }
      
   return fighter::parse(infile, token);
}


/* *************************************************************
************************************************************* */
void player::preprocess(void *data) {

   fighter::preprocess(data);

   stop_toggle = throttle_toggle = 0;
   status.set_primary_mode(PRIMARY_MODE_SINGLE);
   complex->soundmanager->find(&collision_sndfx_id);
   complex->soundmanager->find(&jump_sndfx_id);
   jump_sndfx = NULL;
}


/* *************************************************************
************************************************************* */
void player::whereami(dbl_llist_manager *hiearchy_manager, quark *parent, vector4f *mx) {

   fighter::whereami(hiearchy_manager, parent, mx);
   update_sound();
}


/* *************************************************************
************************************************************* */
void player::update_engines() {

   int changeflag = FLAG_SPEED_SETTING_NULL;
   int toggleflag = 0;
   float scale, temp;
   float speed_ratio = 0;
   vector4f r[4], s[4];
   equip_inertial_compensator *ic;
   equip_targeting_computer *tc;
   vector3f v;
   float maxspeed;

   // - check for inertial compensator
   //      - project speed of current orientation vs velocity direction
   ic = (equip_inertial_compensator *)status.find_category(EQUIPCAT_INERTIA);
   tc = (equip_targeting_computer *)status.find_specific(EQUIP_TARGETING_COMPUTER);
   
   // update throttle
   if (complex->userinput->translate_stroke(KEYSTROKE_THROTTLE_TOGGLE))
      throttle_toggle = !throttle_toggle;

   // velocity changes
   status.afterburner_toggle = complex->userinput->translate(KEYSTROKE_AFTERBURNER);
   maxspeed = status.query_maxspeed();

   if (status.afterburner_toggle) {
      changeflag = FLAG_SPEED_SETTING_CHANGE;
      toggleflag = 1;
      speed_ratio = 1.0f;
   }

   else if (complex->userinput->translate(KEYSTROKE_STOP)) {
      changeflag = FLAG_SPEED_SETTING_STOP;
      toggleflag = 1;
      speed_ratio = 0;
   }
      
   else if (throttle_toggle && fabs(complex->userinput->control[AXIS_THROTTLE]) > CORRECT) {
      changeflag = FLAG_SPEED_SETTING_CHANGE;
      toggleflag = 1;
      speed_ratio = complex->userinput->control[AXIS_THROTTLE];
   }

   else if (complex->userinput->translate(KEYSTROKE_FORWARD)) { 
      changeflag = FLAG_SPEED_SETTING_CHANGE;
      toggleflag = 1;
      speed_ratio = 1.0f;
   }

   else if (complex->userinput->translate(KEYSTROKE_BACKWARD)) {
      changeflag = FLAG_SPEED_SETTING_CHANGE;
      toggleflag = 1;
      speed_ratio = -1.0f;
   }
   
   else if (tc && tc->match_velocity_flag) {
      changeflag = FLAG_SPEED_SETTING_CHANGE;
      speed_ratio = (tc->query_match_velocity()-status.speed)/maxspeed;
      if (speed_ratio > 1.0f)
         speed_ratio = 1.0f;
   }
      
   if (toggleflag && tc)
      tc->match_velocity_flag = 0;

   thruster_flag = THRUSTERFLAG_NULL;

   // compensate for direction
   if (ic && ic->toggle) {
      stop_toggle = 0;
      if (changeflag == FLAG_SPEED_SETTING_STOP)
         status.set_speed_ratio = 0;
      else {
         status.set_speed_ratio += complex->timer.speedscale*speed_ratio;

      	 if (status.set_speed_ratio > 1.0)
            status.set_speed_ratio = 1.0;
      	 else if (status.set_speed_ratio < -1.0)
            status.set_speed_ratio = -1.0;
      }

      if (status.set_speed_ratio < 0) {
         v[0] = -initxform[0][2];
         v[1] = -initxform[1][2];
         v[2] = -initxform[2][2];
      }
      
      else {
         v[0] = initxform[0][2];
         v[1] = initxform[1][2];
         v[2] = initxform[2][2];
      }
            
      if (dotproduct3(status.vel, v) < ICORRECT) {
         scale = status.query_accel() * complex->timer.speedscale;

         status.vel[0] = status.speed*status.vel[0] + scale*v[0];
         status.vel[1] = status.speed*status.vel[1] + scale*v[1];
         status.vel[2] = status.speed*status.vel[2] + scale*v[2];

         temp = dotproduct3(status.vel, status.vel);

         if (temp < CORRECT) {
            copyarray3(status.vel, v);
            status.speed = 0;
         }

         else {
            scale = (float)sqrt(temp);
            temp = 1.0f/scale;
            smultarray3(status.vel, temp);

            temp = (float)(fabs(status.set_speed_ratio)*maxspeed);
            if (temp < scale)
               scale = temp;

            temp = scale - status.speed;

            if (temp < -CORRECT)
               thruster_flag |= THRUSTERFLAG_ENGINE_BACKWARD;
            else if (temp > CORRECT)
               thruster_flag |= THRUSTERFLAG_ENGINE_FORWARD;

            status.speed = scale;
         }

      }

      else {
         scale = status.query_accel() * complex->timer.speedscale;
         copyarray3(status.vel, v);

         temp = (float)(fabs(status.set_speed_ratio)*maxspeed - status.speed);

         if (temp < 0) {
            scale = -scale;
            if (temp < -CORRECT)
               thruster_flag |= THRUSTERFLAG_ENGINE_BACKWARD;
            if (temp > scale)
               scale = temp;
         }
	 
         else {
            if (temp > CORRECT)
               thruster_flag |= THRUSTERFLAG_ENGINE_FORWARD;
            if (temp < scale)
               scale = temp;
         }

         status.speed += scale;
      }

   }	    

   // freefly
   else if (changeflag || stop_toggle) {
      scale = status.query_accel() * complex->timer.speedscale;

      if (changeflag == FLAG_SPEED_SETTING_STOP || (!changeflag && stop_toggle)) {
         stop_toggle = 1;
         if (status.speed < scale)
            status.speed = 0;
         else
            status.speed -= scale;
      }
      
      else {	 
         stop_toggle = 0;
         scale *= speed_ratio;
	 
         if (scale > CORRECT)
            thruster_flag |= THRUSTERFLAG_ENGINE_FORWARD;
         else if (scale < -CORRECT)
            thruster_flag |= THRUSTERFLAG_ENGINE_BACKWARD;
	    
         status.vel[0] = status.speed*status.vel[0] + scale*initxform[0][2];
         status.vel[1] = status.speed*status.vel[1] + scale*initxform[1][2];
         status.vel[2] = status.speed*status.vel[2] + scale*initxform[2][2];

         status.speed = dotproduct3(status.vel, status.vel);

         if (status.speed > CORRECT) {
            status.speed = (float)sqrt(status.speed);
            scale = 1.0f/status.speed;
            smultarray3(status.vel, scale);

            if (status.speed > maxspeed)
               status.speed = maxspeed;
         }

         else {
            status.speed = 0;
            status.vel[0] = initxform[0][2];
            status.vel[1] = initxform[1][2];
            status.vel[2] = initxform[2][2];
	    normalize3(status.vel);
         }
	 
      }
      
   }

   else if (status.speed > maxspeed)
      status.speed = maxspeed;

   if (complex->userinput->control[AXIS_ROLL] < -CORRECT)
      thruster_flag |= THRUSTERFLAG_ROLL_CLOCK;
   else if (complex->userinput->control[AXIS_ROLL] > CORRECT)
      thruster_flag |= THRUSTERFLAG_ROLL_CCLOCK;

   if (complex->userinput->control[AXIS_PITCH] < -CORRECT)
      thruster_flag |= THRUSTERFLAG_PITCH_UP;
   else if (complex->userinput->control[AXIS_PITCH] > CORRECT)
      thruster_flag |= THRUSTERFLAG_PITCH_DOWN;

   if (complex->userinput->control[AXIS_YAW] < -CORRECT)
      thruster_flag |= THRUSTERFLAG_YAW_RIGHT;
   else if (complex->userinput->control[AXIS_YAW] > CORRECT)
      thruster_flag |= THRUSTERFLAG_YAW_LEFT;

   // speedscale angular velocity
   copymx4x4o(r, initxform);

   init_mx(r);

   temp = status.dorient*complex->timer.speedscale;
   arotate_mx_z(r, complex->userinput->control[AXIS_ROLL]*temp);
   arotate_mx_x(r, complex->userinput->control[AXIS_PITCH]*temp);
   arotate_mx_y(r, complex->userinput->control[AXIS_YAW]*temp);

   copymx4x4o(s, initxform);
   matmatmulto(s, r, initxform);

   scale = status.speed * complex->timer.speedscale;
   initxform[0][3] += scale * status.vel[0];
   initxform[1][3] += scale * status.vel[1];
   initxform[2][3] += scale * status.vel[2];
}


/* *************************************************************
************************************************************* */
void player::update_instruments() {

   equip_targeting_computer *tc;
    
   tc = (equip_targeting_computer *)status.find_specific(EQUIP_TARGETING_COMPUTER);

   if (tc) {
      // targeting...  
      if (complex->userinput->translate_stroke(KEYSTROKE_TARGET_NEXT)) {
         tc->set_target_type(0, GAMEFLAG_ALLIANCE_MASK);
         tc->set_target_mode(TARGET_MODE_NEXT);
         tc->update_target(TARGET_MASK_FIND_TARGET);
      }

      else if (complex->userinput->translate_stroke(KEYSTROKE_TARGET_NEXT_ENEMY)) {
         tc->set_target_type(0, calc_enemy(flags));
         tc->set_target_mode(TARGET_MODE_NEXT);
         tc->update_target(TARGET_MASK_FIND_TARGET);
      }

      else if (complex->userinput->translate_stroke(KEYSTROKE_TARGET_NEAREST_ENEMY)) {
         tc->set_target_type(0, calc_enemy(flags));
         tc->set_target_mode(TARGET_MODE_NEAREST);
         tc->update_target(TARGET_MASK_FIND_TARGET);
      }

      else if (complex->userinput->translate_stroke(KEYSTROKE_TARGET_CENTER)) {
         tc->set_target_type(0, GAMEFLAG_ALLIANCE_MASK);
         tc->set_target_mode(TARGET_MODE_CENTER);
         tc->update_target(TARGET_MASK_FIND_TARGET);
      }

      else if (complex->userinput->translate_stroke(KEYSTROKE_SUBTARGET_NEXT))
         tc->update_target(TARGET_MASK_FIND_SUBTARGET);

      // match target speed
      if (complex->userinput->translate_stroke(KEYSTROKE_MATCH_SPEED))
         tc->match_velocity_flag = !tc->match_velocity_flag;
   }

   // cycle status readout
   if (complex->userinput->translate_stroke(KEYSTROKE_CYCLE_STATUS))
      status.cycle_status();
}


/* *************************************************************
************************************************************* */
void player::process_intel(dbl_llist_manager *hiearchy_manager, vector4f *mx) {

   update_instruments();
   update_engines();
}


/* *************************************************************
************************************************************* */
void player::process_fire_control(dbl_llist_manager *hiearchy_manager) {

   target_type target;
   equip_targeting_computer *tc;

   if (complex->userinput->translate_stroke(KEYSTROKE_ROUND_ROBIN_PRIMARY))
      status.set_primary_mode(PRIMARY_MODE_ROBIN);
   else if (complex->userinput->translate_stroke(KEYSTROKE_FULL_PRIMARY))
      status.set_primary_mode(PRIMARY_MODE_FULL);
   else if (complex->userinput->translate_stroke(KEYSTROKE_SELECT_NEXT_PRIMARY))
      status.select_next_weapon(EQUIPCAT_PRIMARY_WEAPON);

   if (complex->userinput->translate_stroke(KEYSTROKE_SELECT_NEXT_SECONDARY))
      status.select_next_weapon(EQUIPCAT_SECONDARY_WEAPON);

   tc = (equip_targeting_computer *)status.find_specific(EQUIP_TARGETING_COMPUTER);

   if (tc) {
      target.object = tc->target.object;
      target.specific_parent = tc->target.specific_parent;
      target.specific = tc->target.specific;
   }

   if (complex->userinput->translate(KEYSTROKE_FIRE_PRIMARY))
      status.fire_weapon(&target, state.xmx, EQUIPCAT_PRIMARY_WEAPON);

   if (complex->userinput->translate_stroke(KEYSTROKE_FIRE_SECONDARY))
      status.fire_weapon(&target, state.xmx, EQUIPCAT_SECONDARY_WEAPON);
}


/* *************************************************************
************************************************************* */
void player::process_collision(vector4f *mx) {

   fighter::process_collision(mx);

   attach_sound(&collision_sndfx_id, coldata.pos);
}


/* *************************************************************
************************************************************* */
void player::display_hud(dbl_llist_manager *hiearchy_manager, engine *proc, mapul *mapbuffer) {

   sim_task_node *ptr;
   float rowsize;
   char msg_buffer[] = "S.T.Fold drive now online";
   
   status.display_hud(hiearchy_manager, this, proc, mapbuffer);

   for (ptr = (sim_task_node *)task_manager.head; ptr && ptr->node->query_whatami() != DARKSIM_JUMP; ptr = (sim_task_node *)ptr->next);
   
   if (ptr) {
      rowsize = (float)(mapbuffer->maxy/64.0);
      complex->font->set_color((unsigned char)255, (unsigned char)96, (unsigned char)96);
      complex->font->set_scale((float)(rowsize*0.85));
      complex->font->print((proc->zbuff.maxx - complex->font->pixel_length((unsigned char *)msg_buffer))>>1, (int)((proc->zbuff.maxy-(rowsize + rowsize))), (unsigned char *)msg_buffer, mapbuffer);
   }
      
}


/* *************************************************************
  Note: assumes this is done after whereami/begin using state
        instead of old_state
************************************************************* */
void player::attach_sound(sound_id_type *sndfx_id, float *pos) {

   sound_node *str;
   sound_type *sndfx;
     
   sndfx = complex->soundmanager->play(sndfx_id);

   if (!sndfx)
      return;

   sndfx->flags |= FLAG_SOUND_LOCK;
   ((sound_wav3d_type *)sndfx)->location(pos);

   sndfx_manager.insert(str = new sound_node, NULL);
   str->sndfx = sndfx;

   if (!(state.state_flags & STATE_FLAG_INVERSE)) {
      inversemx(state.xmx, state.ixmx);
      state.state_flags |= STATE_FLAG_INVERSE;
   }
   
   inversemx(state.xmx, state.ixmx);
   matvecmulto(state.ixmx, pos, str->offset);
}


/* *************************************************************
************************************************************* */
void player::update_sound() {

   vector4f pos;
   sound_node *str, *ttr;
   
   str = (sound_node *)sndfx_manager.head;
   while (str) {
      ttr = (sound_node *)str->next;
      
      if (str->sndfx->flags & FLAG_SOUND_PLAYING) {
         matvecmulto(state.xmx, str->offset, pos);
         ((sound_wav3d_type *)str->sndfx)->location(pos);
      }
      
      else {
         str->sndfx->flags &= ~FLAG_SOUND_LOCK;
         str->sndfx = NULL;
         sndfx_manager.remove(str);
         delete str;
      }

      str = (sound_node *)ttr;
   }
      
}


/* *************************************************************
************************************************************* */
void player::update(dbl_llist_manager *hiearchy_manager, quark *parent) {

   sim_task_node *ptr;
   shockwave *sfx;
   explosion *efx;
   int jump_flag = 0;
   
   if (jump_sndfx) {
      if (!(jump_sndfx->flags & FLAG_SOUND_PLAYING)) {
         jump_sndfx->flags &= ~FLAG_SOUND_LOCK;
         jump_sndfx = NULL;
         jump_flag = 1;
      }
         
   }
      
   else if (complex->userinput->translate_stroke(KEYSTROKE_JUMP))
      for (ptr = (sim_task_node *)task_manager.head; ptr; ptr = (sim_task_node *)ptr->next)
         if (ptr->node->query_whatami() == DARKSIM_JUMP) {
            jump_sndfx = complex->soundmanager->play(&jump_sndfx_id);
       
            if (!jump_sndfx)
               jump_flag = 1;

            break;
         }

   if (jump_flag) {
      for (ptr = (sim_task_node *)task_manager.head; ptr; ptr = (sim_task_node *)ptr->next)
         if (ptr->node->query_whatami() == DARKSIM_JUMP) {
            ptr->node->owner->flags |= DARKSIM_FLAG_JUMP;
            break;
         }
	       
      sfx = (shockwave *)complex->teacher.issue(OBJECT_SHOCKWAVE);
      sfx->init(old_state.xmx, 5, 5000);

      efx = (explosion *)complex->teacher.issue(OBJECT_EXPLOSION);
      efx->init(old_state.xmx, 5, 5000);
      
      flags &= ~QUARK_FLAG_ACTIVE;
      sheader->flags |= DARKSIM_FLAG_JUMP;
   }
   
   fighter::update(hiearchy_manager, parent);
}


/* *************************************************************
************************************************************* */
int player::set_specific_data(unsigned int type, void *data) {

   unsigned int enemy_mask;
   
   switch (type) {

      case DATAFLAG_KILL_NOTICE:
         if (!sheader)
            return 0;
	    
         if (flags & GAMEFLAG_ALLIANCE_MASK & ((gameatom *)data)->flags) {
            if (((gameatom *)data)->flags & (GAMEFLAG_FIGHTER | GAMEFLAG_BOMBER))
               sheader->player_ally_fighter_kill_count++;
            else
               sheader->player_ally_cap_kill_count++;

            return 1;
         }
	 
         enemy_mask = calc_enemy(global_player->flags);
	 
         if (enemy_mask & GAMEFLAG_ALLIANCE_MASK & ((gameatom *)data)->flags)
            if (((gameatom *)data)->flags & (GAMEFLAG_FIGHTER | GAMEFLAG_BOMBER))
               sheader->player_enemy_fighter_kill_count++;
            else	 
               sheader->player_enemy_cap_kill_count++;
	
         return 1;

      default:
         break;	    
   }
   
   return fighter::set_specific_data(type, data);
}
