

#include <stdlib.h>
#include <string.h>

#include "pstring.h"

#include "fighter.h"
#include "darkfx.h"
#include "darkgine.h"


#define MISSILE_LOCK_MAX_TIME     1.0f
#define SYSTEM_COMPUTER_UPDATE_DELAY 0.75f

#define RADAR_RADAR_ALARM_DELAY   0.4f
#define RADAR_MISSILE_ALARM_DELAY 1.0f
#define RADAR_SCAN_RATE           2.0f


/* *************************************************************
************************************************************* */
equip_engine::equip_engine() {

   speed = accel = 1.0f;
   sndfx = NULL;
}


/* *************************************************************
************************************************************* */
equip_engine::~equip_engine() {

   if (sndfx)
      sndfx->stop();
}


/* *************************************************************
************************************************************* */
int equip_engine::query_whatwasi(int type) {

   return equip_engine::query_whatami() == type ? 1 : equipment::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int equip_engine::parse(FILE *infile, char *token) {

   if (!strcmp(token, TOKEN_SPEED_STR)) {
      get_token(infile, token);
      speed = (float)atof(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_ACCEL_STR)) {
      get_token(infile, token);
      accel = (float)atof(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_SFX_STR)) {
      get_token(infile, token);
      sndfx_id[SOUND_EQUIP_ID_ENGINE].name.stringcpy(token);
      get_token(infile, token);
      sndfx_id[SOUND_EQUIP_ID_AFTERBURNER].name.stringcpy(token);
      sndfx_id[SOUND_EQUIP_ID_ENGINE].flags = sndfx_id[SOUND_EQUIP_ID_AFTERBURNER].flags = FLAG_SOUND_WAV | FLAG_SOUND_3D | FLAG_SOUND_LOOPING;
      return 1;
   }

   return equipment::parse(infile, token);
}


/* *************************************************************
************************************************************* */
equipment *equip_engine::copy(equipment *item) {

   int i;
   equip_engine *obj = (equip_engine *)item;

   equipment::copy(item);

   for (i=0; i<SOUND_EQUIP_ID_ENGINE_MAX; i++) {
      obj->sndfx_id[i].name.stringcpy(&sndfx_id[i].name);
      obj->sndfx_id[i].flags = sndfx_id[i].flags;
   }
      
   obj->speed = speed;
   obj->accel = accel;

   return item;
}


/* *************************************************************
************************************************************* */
void equip_engine::reset(gameatom *source, object_desc *ob) {

   int i;
   
   equipment::reset(source, ob);

   for (i=0; i<SOUND_EQUIP_ID_ENGINE_MAX; i++)
      sndfx_id[i].id = NULL;

   sndfx = NULL;

   if (source == global_player) {

      for (i=0; i<SOUND_EQUIP_ID_ENGINE_MAX; i++)
         complex->soundmanager->find(&sndfx_id[i]);
	 
      sndfx = complex->soundmanager->play(&sndfx_id[SOUND_EQUIP_ID_ENGINE]);

      if (sndfx)
         sndfx->flags |= FLAG_SOUND_LOCK;
   }

   ((fighter_desc *)gear)->maxspeed += speed;
   ((fighter_desc *)gear)->accel    += accel;
}


/* *************************************************************
   update equipment after movement
************************************************************* */
void equip_engine::update_fx(vector4f *mx) {

   vector4f pos;

   equipment::update_fx(mx);

   if (sndfx) {
      if (((fighter_desc *)gear)->afterburner_toggle && ((fighter_desc *)gear)->afterburner_status == 1) {
         if (sndfx->id != sndfx_id[SOUND_EQUIP_ID_AFTERBURNER].id) {
            sndfx->flags &= ~FLAG_SOUND_LOCK;
            sndfx->stop();
            sndfx = complex->soundmanager->play(&sndfx_id[SOUND_EQUIP_ID_AFTERBURNER]);
         }
	 
      }
      
      else {
         if (sndfx->id != sndfx_id[SOUND_EQUIP_ID_ENGINE].id) {
            sndfx->flags &= ~FLAG_SOUND_LOCK;
            sndfx->stop();
            sndfx = complex->soundmanager->play(&sndfx_id[SOUND_EQUIP_ID_ENGINE]);
         }
	 
// asdf not the best - still calculating "maxspeed"
         sndfx->volume((int)((100*((fighter_desc *)gear)->speed)/((fighter_desc *)gear)->query_maxspeed()));
      }
            
      pos[0] = mx[0][3];
      pos[1] = mx[1][3];
      pos[2] = mx[2][3];

      ((sound_wav3d_type *)sndfx)->location(pos);
   }

}


/* *************************************************************
************************************************************* */
void equip_engine::apply_damage(int *damage) {

   float factor;
   int   temp;

   temp = *damage;

   equipment::apply_damage(damage);

   factor = (temp-*damage)*imaxhp;

   ((fighter_desc *)gear)->accel    -= accel*factor;
   ((fighter_desc *)gear)->maxspeed -= speed*factor;

   if (((fighter_desc *)gear)->maxspeed < 1)
      ((fighter_desc *)gear)->maxspeed = 1;
}


/* *************************************************************
************************************************************* */
int equip_thruster::query_whatwasi(int type) {

   if (equip_thruster::query_whatami() == type)
      return 1;

   return equipment::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
void equip_thruster::reset(gameatom *source, object_desc *ob) {

   equipment::reset(source, ob);

   ((fighter_desc *)gear)->dorient += dorient;
}


/* *************************************************************
************************************************************* */
int equip_thruster::parse(FILE *infile, char *token) {

   if (!strcmp(token, TOKEN_DORIENT_STR)) {
      get_token(infile, token);
      dorient = (float)atof(token);
      dorient = (float)deg2rad(dorient);
      return 1;
   }

   return equipment::parse(infile, token);
}


/* *************************************************************
************************************************************* */
equipment *equip_thruster::copy(equipment *item) {

   equip_thruster *obj = (equip_thruster *)item;

   equipment::copy(item);

   obj->dorient = dorient;
   return item;
}


/* *************************************************************
************************************************************* */
void equip_thruster::apply_damage(int *damage) {

   float factor;
   int temp;

   temp = *damage;

   equipment::apply_damage(damage);

   factor = (temp - *damage)*imaxhp;

   ((fighter_desc *)gear)->dorient -= dorient * factor;
}


/* *************************************************************
************************************************************* */
equip_weapon::equip_weapon() {

   offset[0] = offset[1] = offset[2] = 0;
   max_range = speed = max_time = 1.0f;
   current_flag = 0;
   max_delay = link_delay = delay = 0;
   speed = 1.0f;
   max_damage = 1;
}


/* *************************************************************
************************************************************* */
int equip_weapon::parse(FILE *infile, char *token) {

   if (!strcmp(token, TOKEN_DAMAGE_STR)) {
      get_token(infile, token);
      max_damage = atoi(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_DELAY_STR)) {
      get_token(infile, token);
      max_delay = (float)atof(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_MAX_RANGE_STR)) {
      get_token(infile, token);
      max_range = (float)atof(token);
      return 1;
   }

   // NOTE: usually set by the fighter inventory list instead of here
   if (!strcmp(token, TOKEN_OFFSET_STR)) {
      get_token(infile, token);
      offset[0] = (float)atof(token);
      get_token(infile, token);
      offset[1] = (float)atof(token);
      get_token(infile, token);
      offset[2] = (float)atof(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_SPEED_STR)) {
      get_token(infile, token);
      speed = (float)atof(token);
      return 1;
   }

   return equipment::parse(infile, token);
}


/* *************************************************************
************************************************************* */
equipment *equip_weapon::copy(equipment *item) {

   equip_weapon *obj = (equip_weapon *)item;

   equipment::copy(item);

   copyarray3(obj->offset, offset);
   obj->max_range = max_range;
   obj->speed = speed;
   obj->max_time = max_time;
   obj->max_delay = max_delay;
   obj->delay = delay;
   obj->link_delay = link_delay;
   obj->max_damage = max_damage;

   return item;
}


/* *************************************************************
************************************************************* */
void equip_weapon::reset(gameatom *source, object_desc *ob) {

   equipment::reset(source, ob);

   current_flag = 0;
   delay = 0;
   link_delay = 0;
   max_time = max_range/speed;
}


/* *************************************************************
************************************************************* */
int equip_lazer::query_whatwasi(int type) {

   if (equip_lazer::query_whatami() == type)
      return 1;

   return equipment::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
equip_lazer::equip_lazer() {

   recharge_rate = 1.0f;
   discharge_rate = 0;
   capacitor = 0;
   max_capacitor_charge = capacitor_charge = 0;
   sndfx = NULL;

   death_tob = NULL;
}


/* *************************************************************
************************************************************* */
equip_lazer::~equip_lazer() {

   if (sndfx)
      sndfx->stop();
}


/* *************************************************************
************************************************************* */
int equip_lazer::parse(FILE *infile, char *token) {

   if (!strcmp(token, TOKEN_DEATHNAME_STR)) {
      get_token(infile, token);
      deathname.stringcpy(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_DISCHARGE_STR)) {
      get_token(infile, token);
      discharge_rate = (float)atof(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_MAX_CAPACITOR_CHARGE_STR)) {
      get_token(infile, token);
      max_capacitor_charge = (float)atof(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_RECHARGE_STR)) {
      get_token(infile, token);
      recharge_rate = (float)atof(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_SFX_STR)) {
      get_token(infile, token);
      sndfx_id.name.stringcpy(token);
      sndfx_id.flags = FLAG_SOUND_WAV | FLAG_SOUND_3D;
      return 1;
   }

   if (!strcmp(token, TOKEN_SFX_VOLUME_STR)) {
      get_token(infile, token);
      sndfx_id.volume(atoi(token));
      return 1;
   }

   return equip_weapon::parse(infile, token);
}


/* *************************************************************
************************************************************* */
void equip_lazer::preprocess(void *data) {

   equipment::preprocess(data);
   death_tob = (texbase *)((frame_manager *)global_resource_manager)->read_tex(deathname.string);
}


/* *************************************************************
************************************************************* */
equipment *equip_lazer::copy(equipment *item) {

   equip_lazer *obj = (equip_lazer *)item;

   equip_weapon::copy(item);

   obj->capacitor = capacitor;
   obj->discharge_rate = discharge_rate;
   obj->max_capacitor_charge = max_capacitor_charge;
   obj->capacitor_charge = capacitor_charge;
   obj->recharge_rate = recharge_rate;
   obj->death_tob = death_tob;
   obj->deathname.stringcpy(&deathname);
   obj->sndfx_id.name.stringcpy(&sndfx_id.name);
   obj->sndfx_id.flags = sndfx_id.flags;
   obj->sndfx_id.db = sndfx_id.db;

   return item;
}


/* *************************************************************
************************************************************* */
void equip_lazer::launch(atom_list_type *source, target_type *target, vector4f *mx) {

   lazer *ammo;
   vector3f dir;

   if (!query_fire())
      return;

   ammo = (lazer *)complex->teacher.issue(query_gfx());

   dir[0] = mx[0][2];
   dir[1] = mx[1][2];
   dir[2] = mx[2][2];

   ammo->init(source, max_range, speed, max_damage, dir, offset, mx, death_tob, target);

   capacitor -= discharge_rate;
   delay = max_delay;

   if (sndfx)
      sndfx->stop();

   sndfx = complex->soundmanager->play(&sndfx_id);
   if (sndfx) {
      sndfx->flags |= FLAG_SOUND_LOCK;

      dir[0] = ammo->initxform[0][3];
      dir[1] = ammo->initxform[1][3];
      dir[2] = ammo->initxform[2][3];
      ((sound_wav3d_type *)sndfx)->location(dir);
   }

}


/* *************************************************************
   update equipment befor movement
************************************************************* */
void equip_lazer::update(dbl_llist_manager *hiearchy_manager) {

   capacitor += complex->timer.speedscale*recharge_rate;
   delay -= complex->timer.speedscale;

   if (capacitor > capacitor_charge)
      capacitor = capacitor_charge;

   equipment::update(hiearchy_manager);
}


/* *************************************************************
   update equipment after movement
************************************************************* */
void equip_lazer::update_fx(vector4f *mx) {

   vector4f pos;

   equipment::update_fx(mx);

   if (sndfx)
      if (sndfx->flags & FLAG_SOUND_PLAYING) {
         matvecmulto(mx, offset, pos);
         ((sound_wav3d_type *)sndfx)->location(pos);
      }

      else {
         sndfx->flags &= ~FLAG_SOUND_LOCK;
         sndfx = NULL;
      }

}


/* *************************************************************
************************************************************* */
void equip_lazer::display_hud(dbl_llist_manager *hiearchy_manager, gameatom *source, engine *proc, mapul *mapbuffer) {

   if (current_flag)
      hud_display_primary_weapon(mapbuffer, this, (fighter_desc *)gear, capacitor/capacitor_charge);
}


/* *************************************************************
************************************************************* */
void equip_lazer::reset(gameatom *source, object_desc *ob) {

   equip_weapon::reset(source, ob);
   capacitor = capacitor_charge = max_capacitor_charge;

   sndfx_id.id = NULL;
   sndfx = NULL;
   if (source == global_player)
      complex->soundmanager->find(&sndfx_id);
}


/* *************************************************************
************************************************************* */
equip_weapon *equip_lazer::query_fire() {

   return (capacitor > 0 && delay <= 0 && link_delay <= 0) ? this : (equip_weapon *)NULL;
}


/* *************************************************************
************************************************************* */
void equip_lazer::apply_damage(int *damage) {

   equipment::apply_damage(damage);

   capacitor_charge = (max_capacitor_charge*hp)*imaxhp;
}


/* *************************************************************
************************************************************* */
int equip_lazer2blu::query_whatwasi(int type) {

   return (equip_lazer2blu::query_whatami() == type) ? 1 : equip_lazer::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int equip_lazer2red::query_whatwasi(int type) {

   return (equip_lazer2red::query_whatami() == type) ? 1 : equip_lazer::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int equip_dual_lazer::query_whatwasi(int type) {

   return (equip_dual_lazer::query_whatami() == type) ? 1 : equip_lazer::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int equip_cone_lazer::query_whatwasi(int type) {

   return (equip_cone_lazer::query_whatami() == type) ? 1 : equip_lazer::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int equip_rail_gun::query_whatwasi(int type) {

   return (equip_rail_gun::query_whatami() == type) ? 1 : equip_lazer::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
void equip_rail_gun::reset(gameatom *source, object_desc *ob) {

   recharge_rate = 0;
   equip_lazer::reset(source, ob);
}


/* *************************************************************
************************************************************* */
int equip_launcher::query_whatwasi(int type) {

   return (equip_launcher::query_whatami() == type) ? 1 : equipment::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
equip_launcher::equip_launcher() {

   count = maxcount = 0;
   death_tob = trail_tob = NULL;

   accel = 1;
   launch_speed = 0;
   dorient = HALFPI;

   launch_dir[0] =  0;
   launch_dir[1] = -1;
   launch_dir[2] =  0;

   sndfx = NULL;
}


/* *************************************************************
************************************************************* */
equip_launcher::~equip_launcher() {

   if (sndfx)
      sndfx->stop();
}


/* *************************************************************
************************************************************* */
int equip_launcher::parse(FILE *infile, char *token) {

   if (!strcmp(token, TOKEN_COUNT_STR)) {
      get_token(infile, token);
      maxcount = atoi(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_DEATHNAME_STR)) {
      get_token(infile, token);
      deathname.stringcpy(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_TRAILNAME_STR)) {
      get_token(infile, token);
      trailname.stringcpy(token);
      
      get_token(infile, token);
      smokename.stringcpy(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_ACCEL_STR)) {
      get_token(infile, token);
      accel = (float)atof(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_DORIENT_STR)) {
      get_token(infile, token);
      dorient = (float)deg2rad(atof(token));
      return 1;
   }

   if (!strcmp(token, TOKEN_LAUNCH_SPEED_STR)) {
      get_token(infile, token);
      launch_speed = (float)atof(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_LAUNCH_DIR_STR)) {
      get_token(infile, token);
      launch_dir[0] = (float)atof(token);
      get_token(infile, token);
      launch_dir[1] = (float)atof(token);
      get_token(infile, token);
      launch_dir[2] = (float)atof(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_SFX_STR)) {
      get_token(infile, token);
      sndfx_id.name.stringcpy(token);
      sndfx_id.flags = FLAG_SOUND_WAV | FLAG_SOUND_3D;
      return 1;
   }

   if (!strcmp(token, TOKEN_SFX_VOLUME_STR)) {
      get_token(infile, token);
      sndfx_id.volume(atoi(token));
      return 1;
   }

   if (!strcmp(token, TOKEN_DETONATE_SFX_STR)) {
      get_token(infile, token);
      detonate_sndfx_id.name.stringcpy(token);
      detonate_sndfx_id.flags = FLAG_SOUND_WAV | FLAG_SOUND_3D;
      return 1;
   }

   if (!strcmp(token, TOKEN_DETONATE_SFX_VOLUME_STR)) {
      get_token(infile, token);
      detonate_sndfx_id.volume(atoi(token));
      return 1;
   }

   return equip_weapon::parse(infile, token);
}


/* *************************************************************
************************************************************* */
void equip_launcher::preprocess(void *data) {

   equipment::preprocess(data);

   death_tob = (texbase *)((frame_manager *)global_resource_manager)->read_tex(deathname.string);
   trail_tob = (texbase *)((frame_manager *)global_resource_manager)->read_tex(trailname.string);
   smoke_tob = (texbase *)((frame_manager *)global_resource_manager)->read_tex(smokename.string);
}


/* *************************************************************
************************************************************* */
equipment *equip_launcher::copy(equipment *item) {

   equip_launcher *obj = (equip_launcher *)item;

   equip_weapon::copy(item);

   obj->accel = accel;
   obj->dorient = dorient;
   obj->launch_speed = launch_speed;
   copyarray3(obj->launch_dir, launch_dir);

   obj->maxcount = maxcount;
   obj->count = count;
   obj->death_tob = death_tob;
   obj->trail_tob = trail_tob;
   obj->smoke_tob = smoke_tob;
   obj->deathname.stringcpy(&deathname);
   obj->trailname.stringcpy(&trailname);
   obj->smokename.stringcpy(&smokename);
   obj->sndfx_id.name.stringcpy(&sndfx_id.name);
   obj->sndfx_id.flags = sndfx_id.flags;
   obj->sndfx_id.db = sndfx_id.db;
   obj->detonate_sndfx_id.name.stringcpy(&detonate_sndfx_id.name);
   obj->detonate_sndfx_id.flags = detonate_sndfx_id.flags;
   obj->detonate_sndfx_id.db = detonate_sndfx_id.db;

   return item;
}


/* *************************************************************
************************************************************* */
void equip_launcher::launch(atom_list_type *source, target_type *target, vector4f *mx) {

   missile *ammo;
   vector4f v;
   vector4f r[4];

   if (!query_fire())
      return;

   ammo = (missile *)complex->teacher.issue(query_gfx());

   copymx4x4o(r, mx);

   matvecmultv(mx, offset, v);

   r[0][3] += v[0];
   r[1][3] += v[1];
   r[2][3] += v[2];

   matvecmultv(mx, launch_dir, v);

   ammo->init(source, max_range, max_damage, r, death_tob, trail_tob, smoke_tob, speed, accel, dorient, launch_speed, v, target, &detonate_sndfx_id);

   count--;
   delay = max_delay;

   if (sndfx)
      sndfx->stop();

   sndfx = complex->soundmanager->play(&sndfx_id);
   if (sndfx) {
      sndfx->flags |= FLAG_SOUND_LOCK;

      v[0] = r[0][3];
      v[1] = r[1][3];
      v[2] = r[2][3];
      ((sound_wav3d_type *)sndfx)->location(v);
   }

}


/* *************************************************************
   update equipment before movement
************************************************************* */
void equip_launcher::update(dbl_llist_manager *hiearchy_manager) {

   delay -= complex->timer.speedscale;

   equipment::update(hiearchy_manager);
}


/* *************************************************************
   update equipment after movement
************************************************************* */
void equip_launcher::update_fx(vector4f *mx) {

   vector4f pos;

   equipment::update_fx(mx);

   if (sndfx)
      if (sndfx->flags & FLAG_SOUND_PLAYING) {
         matvecmulto(mx, offset, pos);
         ((sound_wav3d_type *)sndfx)->location(pos);
      }

      else {
         sndfx->flags &= ~FLAG_SOUND_LOCK;
         sndfx = NULL;
      }

}


/* *************************************************************
************************************************************* */
void equip_launcher::display_hud(dbl_llist_manager *hiearchy_manager, gameatom *source, engine *proc, mapul *mapbuffer) {

   if (current_flag && count)
      hud_display_secondary_weapon(mapbuffer, this, count, maxcount);
}


/* *************************************************************
************************************************************* */
void equip_launcher::reset(gameatom *source, object_desc *ob) {

   equip_weapon::reset(source, ob);
   count = maxcount;

   sndfx_id.id = NULL;
   detonate_sndfx_id.id = NULL;
   sndfx = NULL;
   if (source == global_player)
      complex->soundmanager->find(&sndfx_id);

   complex->soundmanager->find(&detonate_sndfx_id);
}


/* *************************************************************
************************************************************* */
equip_weapon *equip_launcher::query_fire() {

   return (count > 0 && delay <= 0 && link_delay <= 0) ? this : (equip_weapon *)NULL;
}


/* *************************************************************
************************************************************* */
int equip_zuni_launcher::query_whatwasi(int type) {

   return (equip_zuni_launcher::query_whatami() == type) ? 1 : equip_launcher::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int equip_ssm_launcher::query_whatwasi(int type) {

   return (equip_ssm_launcher::query_whatami() == type) ? 1 : equip_launcher::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int equip_ssm_launcher::parse(FILE *infile, char *token) {

   if (!strcmp(token, TOKEN_LOCKTIME_STR)) {
      get_token(infile, token);
      locktime = atoi(token);
      return 1;
   }

   return equip_launcher::parse(infile, token);
}


/* *************************************************************
************************************************************* */
equipment *equip_ssm_launcher::copy(equipment *item) {

   equip_ssm_launcher *obj = (equip_ssm_launcher *)item;

   equip_launcher::copy(item);

   obj->locktime = locktime;
   return item;
}


/* *************************************************************
************************************************************* */
void equip_ssm_launcher::launch(atom_list_type *source, target_type *target, vector4f *mx) {

   missile *ammo;
   vector4f v;
   vector4f r[4];
   equip_targeting_computer *comp;

   if (!query_fire())
      return;

   ammo = (missile *)complex->teacher.issue(query_gfx());

   copymx4x4o(r, mx);

   matvecmultv(mx, offset, v);

   r[0][3] += v[0];
   r[1][3] += v[1];
   r[2][3] += v[2];

   matvecmultv(mx, launch_dir, v);

   ammo->init(source, max_range, max_damage, r, death_tob, trail_tob, smoke_tob, speed, accel, dorient, launch_speed, v, target, &detonate_sndfx_id);

   // calc missile lock
   comp = (equip_targeting_computer *)gear->find_specific(EQUIP_TARGETING_COMPUTER);
   ((ssm_missile *)ammo)->lockflag = comp && comp->missilelock_countdown < 0;

   count--;
   delay = max_delay;

   if (sndfx)
      sndfx->stop();

   sndfx = complex->soundmanager->play(&sndfx_id);
   if (sndfx) {
      sndfx->flags |= FLAG_SOUND_LOCK;

      v[0] = r[0][3];
      v[1] = r[1][3];
      v[2] = r[2][3];
      ((sound_wav3d_type *)sndfx)->location(v);
   }

}


/* *************************************************************
************************************************************* */
void equip_status_computer::reset(gameatom *source, object_desc *ob) {

   equipment::reset(source, ob);

   current_flag = 0;
};



/* *************************************************************
************************************************************* */
int equip_status_computer::cycle_display() {

   if (current_flag) {
      current_flag = 0;
      return 1;
   }
   
   current_flag = 1;
   return 0;
}


/* *************************************************************
************************************************************* */
equip_targeting_computer::~equip_targeting_computer() {

   if (sndfx)
      sndfx->stop();
}


/* *************************************************************
************************************************************* */
int equip_targeting_computer::query_whatwasi(int type) {

   return (equip_targeting_computer::query_whatami() == type) ? 1 : equipment::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
equip_targeting_computer::equip_targeting_computer() {

   missilelock_time = missilelock_countdown = 0;
   missilelock_active = 0;
   target_flag = 0;
   match_velocity_flag = 0;
   target_mode = TARGET_MODE_NEXT;
   sndfx = NULL;
}


/* *************************************************************
************************************************************* */
int equip_targeting_computer::parse(FILE *infile, char *token) {

   if (!strcmp(token, TOKEN_SFX_STR)) {
      get_token(infile, token);
      sndfx_id.name.stringcpy(token);
      sndfx_id.flags = FLAG_SOUND_WAV;
      return 1;
   }

   if (!strcmp(token, TOKEN_SFX_VOLUME_STR)) {
      get_token(infile, token);
      sndfx_id.volume(atoi(token));
      return 1;
   }

   return equipment::parse(infile, token);
}


/* *************************************************************
************************************************************* */
equipment *equip_targeting_computer::copy(equipment *item) {

   equip_targeting_computer *obj = (equip_targeting_computer *)item;

   equipment::copy(item);

   obj->target.object = target.object;
   obj->target.specific_parent = target.specific_parent;
   obj->target.specific = target.specific;
   obj->missilelock_countdown = missilelock_countdown;
   obj->missilelock_time = missilelock_time;
   obj->missilelock_active = missilelock_active;
   obj->targetlock = targetlock;
   obj->sublock = sublock;
   obj->target_mode = target_mode;
   obj->target_flag = target_flag;
   obj->sndfx_id.name.stringcpy(&sndfx_id.name);
   obj->sndfx_id.flags = sndfx_id.flags;
   obj->sndfx_id.db = sndfx_id.db;
   return item;
}


/* *************************************************************
************************************************************* */
void equip_targeting_computer::reset(gameatom *source, object_desc *ob) {

   equip_status_computer::reset(source, ob);

   target.object = NULL;
   target.specific_parent = NULL;
   target.specific = NULL;

   targetlock.init();
   targetlock.set_mask(0, GAMEFLAG_ALLIANCE_MASK);
   targetlock.set_source(source);
   targetlock.set_range(-1);

   sublock.init();

   missilelock_countdown = missilelock_time;
   missilelock_active = 0;
   target_mode = TARGET_MODE_NEXT;
   target_flag = 0;

   sndfx_id.id = NULL;
   sndfx = NULL;
   timer = 0;

   match_velocity_flag = 0;

   if (source == global_player)
      complex->soundmanager->find(&sndfx_id);
      
   target_substate_manager.dest();
}


/* *************************************************************
   update equipment after movement
************************************************************* */
void equip_targeting_computer::display_hud(dbl_llist_manager *hiearchy_manager, gameatom *source, engine *proc, mapul *mapbuffer) {

   camera *cam;
   
   if (!target.object)
      return;

   cam = proc->query_camera();

   if (!cam)
      return;

   hud_display_intercept(mapbuffer, (gameatom *)target.object->htree, cam, source, gear, this);

   if (hud_display_target(mapbuffer, (gameatom *)target.object->htree, cam, current_flag)) {
      if (missilelock_active && missilelock_countdown < missilelock_time-CORRECT)
         hud_display_missile_lock(mapbuffer, (gameatom *)target.object->htree, cam, source, missilelock_countdown, missilelock_time, gear);

      if (target.specific)
         hud_display_subtarget(mapbuffer, (gamequark *)target.specific, cam);
   }
   
}


/* *************************************************************
************************************************************* */
void equip_targeting_computer::update(dbl_llist_manager *hiearchy_manager) {

   equip_launcher *weap;
   vector3f dir, v;
   float dist;
   float time2target;
   intercept_pathway prediction_comp;
   int flag;
   gameatom *source;
   substate_type *str;
   float *target_vel;
   
   equipment::update(hiearchy_manager);

   if (!GATOM_STATUS(target.object) || (target_flag & TARGET_MASK_FIND_TARGET)) {

      target_substate_manager.dest();
      target_flag &= ~TARGET_MASK_FIND_TARGET;
      
      switch (target_mode) {
         case TARGET_MODE_NEXT:
            targetlock.get_next_target(hiearchy_manager, &target);
            break;

         case TARGET_MODE_NEAREST:
            targetlock.get_nearest_target(hiearchy_manager, &target);
            break;

         default: // TARGET_MODE_CENTER
            targetlock.get_center_target(hiearchy_manager, &target);
            target_mode = TARGET_MODE_NEXT;
            break;
      }

      sublock.init();
      missilelock_countdown = missilelock_time;
      match_velocity_flag = 0;
   }

   if (!GATOM_STATUS(target.object))
      return;
      
   weap = ((fighter_desc *)gear)->equip_stat[CRADLE_SECONDARY].current ? (equip_launcher *)((fighter_desc *)gear)->equip_stat[CRADLE_SECONDARY].current->link : (equip_launcher *)NULL;

   missilelock_active = weap && weap->query_guidance();

   if (missilelock_active) {
      missilelock_countdown -= complex->timer.speedscale;

      // check if offscreen & in range
      source = targetlock.get_source();
      flag = prediction_comp.intercept_prediction(target.object->htree, source->old_state.center, weap->speed, v, &time2target);

      if (flag)
         if (time2target > weap->max_time)
            flag = 0;
         else {
            subeqarray3(v, target.object->htree->old_state.center, source->old_state.center);

            dir[0] = source->old_state.xmx[0][2];
            dir[1] = source->old_state.xmx[1][2];
            dir[2] = source->old_state.xmx[2][2];

            // if w/in ~36 degrees
            dist = dotproduct3(v, dir);
            flag &= dist > 0 && (dist*dist >= 0.64*dotproduct3(v,v)*dotproduct3(dir,dir));
         }

      if (!flag)
         missilelock_countdown = missilelock_time;
      else if (missilelock_countdown < 0) 
         target.object->htree->set_specific_data(DATAFLAG_RADAR_LOCK, source);
   }

   if (!GQUARK_STATUS(target.specific, (gameatom *)target.object->htree) || (target_flag & TARGET_MASK_FIND_SUBTARGET)) {
      target_flag &= ~TARGET_MASK_FIND_SUBTARGET;
      sublock.get_next_subtarget(target.object, &target);
   }

   if (target.object->htree->query_specific_data(DATAFLAG_PERCEIVED_VELOCITY, &target_vel)) {
      if (target_substate_manager.count < 4)
         target_substate_manager.insert(str = new substate_type, NULL);
      else {
         target_substate_manager.remove(str = (substate_type *)target_substate_manager.tail);
         target_substate_manager.insert(str, NULL);
      }

      copyarray4(str->weighted_velocity, target_vel);
   }
   
}


/* *************************************************************
************************************************************* */
void equip_targeting_computer::update_fx(vector4f *mx) {

   equipment::update_fx(mx);

   if (sndfx && !(sndfx->flags & FLAG_SOUND_PLAYING)) {
      sndfx->flags &= ~FLAG_SOUND_LOCK;
      sndfx = NULL;
   }

   timer -= complex->timer.speedscale;

   if (missilelock_active && timer < 0 && missilelock_countdown < missilelock_time) {
      if (missilelock_countdown < 0)
         timer = MISSILE_LOCK_MAX_TIME*0.1f;
      else
         timer = MISSILE_LOCK_MAX_TIME*(0.1f + 0.9f*(missilelock_countdown/missilelock_time));

      if (sndfx)
         if (sndfx->flags & FLAG_SOUND_PLAYING)
            sndfx->stop();
         else
            sndfx->flags &= ~FLAG_SOUND_LOCK;

      sndfx = complex->soundmanager->play(&sndfx_id);
      if (sndfx)
         sndfx->flags |= FLAG_SOUND_LOCK;
   }

}


/* *************************************************************
************************************************************* */
float equip_targeting_computer::query_match_velocity() {

   float *target_vel;

   if (!GATOM_STATUS(target.object))
      return -1.0f;

   if (!target.object->htree->query_specific_data(DATAFLAG_PERCEIVED_VELOCITY, &target_vel))
      return 0;

   return target_vel[3] > CORRECT ? (float)(magnitude3(target_vel)/target_vel[3]) : 0;
}


/* ************************************************************
************************************************************ */
void *radar_man::pop() {

   dbl_llist *ptr;
   
   if (man.head) {
      man.remove(ptr = man.head);
      return ptr;
   }
   
   return new radar_blip_type;
}


/* *************************************************************
************************************************************* */
equip_radar::equip_radar() {

   status_flag = FLAG_RADAR_NULL;
   radar_timer = missile_timer = scan_timer = 0;
   radar_sndfx = missile_sndfx = NULL;
   display_tob = pulse_tob = NULL;
}


/* *************************************************************
************************************************************* */
equip_radar::~equip_radar() {

   if (radar_sndfx)
      radar_sndfx->stop();

   if (missile_sndfx)
      missile_sndfx->stop();
}


/* *************************************************************
************************************************************* */
int equip_radar::query_whatwasi(int type) {

   return equip_radar::query_whatami() == type ? 1 : equipment::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
equipment *equip_radar::copy(equipment *item) {

   equip_radar *obj = (equip_radar *)item;
   
   equipment::copy(item);

   obj->radar_sndfx_id.name.stringcpy(&radar_sndfx_id.name);
   obj->radar_sndfx_id.flags = radar_sndfx_id.flags;
   obj->radar_sndfx_id.db = radar_sndfx_id.db;
   obj->missile_sndfx_id.name.stringcpy(&missile_sndfx_id.name);
   obj->missile_sndfx_id.flags = missile_sndfx_id.flags;
   obj->missile_sndfx_id.db = missile_sndfx_id.db;
   obj->display_tob = display_tob;
   obj->pulse_tob = pulse_tob;

   return item;
}


/* *************************************************************
************************************************************* */
int equip_radar::parse(FILE *infile, char *token) {

   if (!strcmp(token, TOKEN_RADAR_SFX_STR)) {
      get_token(infile, token);
      radar_sndfx_id.name.stringcpy(token);
      radar_sndfx_id.flags = FLAG_SOUND_WAV;
      return 1;
   }

   if (!strcmp(token, TOKEN_RADAR_SFX_VOLUME_STR)) {
      get_token(infile, token);
      radar_sndfx_id.volume(atoi(token));
      return 1;
   }

   if (!strcmp(token, TOKEN_MISSILE_SFX_STR)) {
      get_token(infile, token);
      missile_sndfx_id.name.stringcpy(token);
      missile_sndfx_id.flags = FLAG_SOUND_WAV;
      return 1;
   }

   if (!strcmp(token, TOKEN_MISSILE_SFX_VOLUME_STR)) {
      get_token(infile, token);
      missile_sndfx_id.volume(atoi(token));
      return 1;
   }

   return equipment::parse(infile, token);
}


/* *************************************************************
************************************************************* */
void equip_radar::preprocess(void *data) {

   dbl_llist *ptr;
   char buffer[MAXSTRLEN];
   
   equipment::preprocess(data);

   sprintf(buffer, "%s%s.rtf", FILENAME_PRETEX, FILENAME_RADAR);
   display_tob = (texbase *)((frame_manager *)global_resource_manager)->read_tex(buffer);
   sprintf(buffer, "%s%s.rtf", FILENAME_PRETEX, FILENAME_FLARE6);
   pulse_tob   = (texbase *)((frame_manager *)global_resource_manager)->read_tex(buffer);
   cam = NULL;
   parent = NULL;

   while (contact.head) {
      contact.remove(ptr = contact.head);
      rman.push(ptr);
   }

}


/* *************************************************************
   update_scan() moved to render
************************************************************* */
void equip_radar::update(dbl_llist_manager *hiearchy_manager) {

   equipment::update(hiearchy_manager);

   if (parent != global_player)
      return;

   radar_timer -= complex->timer.speedscale;
   missile_timer -= complex->timer.speedscale;

   scan_timer += complex->timer.speedscale;
   while (scan_timer >= RADAR_SCAN_RATE)
      scan_timer -= RADAR_SCAN_RATE;
}


/* *************************************************************
************************************************************* */
void equip_radar::update_fx(vector4f *mx) {

   equipment::update_fx(mx);

   if (parent != global_player)
      return;

   if ((status_flag & FLAG_RADAR_RADAR_LOCK) && radar_timer < 0) { 
      radar_timer = RADAR_RADAR_ALARM_DELAY;

      if (radar_sndfx)
         if (radar_sndfx->flags & FLAG_SOUND_PLAYING)
            radar_sndfx->stop();
         else
            radar_sndfx->flags &= ~FLAG_SOUND_LOCK;

      radar_sndfx = complex->soundmanager->play(&radar_sndfx_id);
      if (radar_sndfx)
         radar_sndfx->flags |= FLAG_SOUND_LOCK;
   }
      
   if ((status_flag & FLAG_RADAR_MISSILE_LOCK) && missile_timer < 0) { 
      missile_timer = RADAR_MISSILE_ALARM_DELAY;

      if (missile_sndfx)
         if (missile_sndfx->flags & FLAG_SOUND_PLAYING)
            missile_sndfx->stop();
         else
            missile_sndfx->flags &= ~FLAG_SOUND_LOCK;

      missile_sndfx = complex->soundmanager->play(&missile_sndfx_id);
      if (missile_sndfx)
         missile_sndfx->flags |= FLAG_SOUND_LOCK;
   }
	 
   status_flag &= ~(FLAG_RADAR_RADAR_LOCK | FLAG_RADAR_MISSILE_LOCK);
}


/* *************************************************************
************************************************************* */
void equip_radar::reset(gameatom *source, object_desc *ob) {

   dbl_llist *ptr;
   
   equipment::reset(source, ob);
   
   status_flag = FLAG_RADAR_NULL;

   radar_sndfx_id.id = NULL;
   missile_sndfx_id.id = NULL;

   if (source == global_player) {
      complex->soundmanager->find(&radar_sndfx_id);
      complex->soundmanager->find(&missile_sndfx_id);
   }

   scan_timer = radar_timer = missile_timer = 0;
   cam = NULL;
   parent = source;

   while (contact.head) {
      contact.remove(ptr = contact.head);
      rman.push(ptr);
   }

}


/* *************************************************************
************************************************************* */
void equip_radar::display_hud(dbl_llist_manager *hiearchy_manager, gameatom *source, engine *proc, mapul *mapbuffer) {

   int offset;
   float foffset;
   vector4uc c;
   vector2i ll, ur;
   radar_blip_type *ptr;

   update_scan(hiearchy_manager, mapbuffer->maxx, mapbuffer->maxy);

   offset = (int)(foffset = mapbuffer->maxx/7.0f);

   if (display_tob) {
      c[0] = 127;
      c[1] = 255;
      c[2] = 127;

      complex->gfx->gfxPaste2D(mapbuffer, display_tob, c, offset*3,0,offset*4,offset, GFX_TRANSPARENT);
   }

   if (pulse_tob) {
//      c[0] = c[1] = 127;
//      c[2] = 255;
      c[0] = c[1] = 96;
      c[2] = 170;

      ur[0] = ll[0] = mapbuffer->maxx >> 1;
      ur[1] = ll[1] = offset >> 1;
      foffset *= (float)((scan_timer*0.5)/RADAR_SCAN_RATE);
      complex->gfx->gfxPaste2D(mapbuffer, pulse_tob, c, (int)(ll[0] - foffset),(int)(ll[1] - foffset),(int)(ur[0] + foffset),(int)(ur[1] + foffset), GFX_TRANSPARENT);
   }

   complex->gfx->gfxPointSize(mapbuffer->maxx/320.0f);
   complex->gfx->gfxBegin(GFX_POINTS, mapbuffer);

      for (ptr = (radar_blip_type *)contact.head; ptr; ptr = (radar_blip_type *)ptr->next) {
         complex->gfx->gfxColor3ubv(ptr->color);
         complex->gfx->gfxVertex2iv(ptr->pos);
      }

   complex->gfx->gfxEnd();
   complex->gfx->gfxPointSize(1.0f);
}


/* *************************************************************
************************************************************* */
void equip_radar::set_missile_lock(gameatom *source) {

   status_flag |= FLAG_RADAR_MISSILE_LOCK;   
}


/* *************************************************************
************************************************************* */
void equip_radar::set_radar_lock(gameatom *source) {

   status_flag |= FLAG_RADAR_RADAR_LOCK;   
}


/* *************************************************************
************************************************************* */
int equip_radar::compute_offset(float *in, int *out, float scale, int *offset, int winx, int winy) {

   vector4f v;
   int hwinx, hwiny;
   float radius;

   // radius;
   matvecmulto(cam->transform, in, v);

   radius = ACOS(-v[2]/magnitude3(v));

   if (radius > PI-CORRECT) {
      v[0] = 0;
      v[1] = 1.0f;
      return 1;
   }

   // placement
   hwinx = winx>>1;
   hwiny = winy>>1;

   if (fabs(v[2]) < 1.0f)
      v[2] = -1.0f;
   else if (v[2] > 0)
      v[2] = -v[2];

   cam->map2screen(v);

   v[0] -= (float)hwinx;
   v[1] -= (float)hwiny;

   if (!((int)v[0]) && !((int)v[1])) {
      v[0] = v[1] = 0;
      return 1;
   }

   radius = (float)((radius*scale)/(magnitude2(v)*PI));
   out[0] = (int)(v[0]*radius + offset[0]);
   out[1] = (int)(v[1]*radius + offset[1]);
   return 1;
}


/* *************************************************************
************************************************************* */
void equip_radar::update_scan(dbl_llist_manager *hiearchy_manager, int winx, int winy) {

   atom_list_type *atr;
   atom_list_type *target = NULL;
   radar_blip_type *ptr;
   float scale;
   vector2i offset;
   equip_targeting_computer *tc;

   while (contact.head) {
      contact.remove(ptr = (radar_blip_type *)contact.head);
      rman.push(ptr);
   }

   if (!cam) {
      for (atr=(atom_list_type *)hiearchy_manager->head; atr && !cam; atr=(atom_list_type *)atr->next)
         cam = (camera *)atr->htree->find_ob(NULL, OBJECT_CAMERA, NULL);
      
      if (!cam)
         return;
   }

   offset[0] = winx>>1;
   offset[1] = (int)(scale = winx/14.0f);

   tc = (equip_targeting_computer *)gear->find_specific(EQUIP_TARGETING_COMPUTER);
   if (tc)
      target = tc->target.object;

   for (atr=(atom_list_type *)hiearchy_manager->head; atr; atr=(atom_list_type *)atr->next)
      if (atr->htree != parent && atr != target && GATOM_STATUS(atr) && ((atr->htree->old_state.state_flags & STATE_MASK_TREE) == STATE_FLAG_TSPHERE)) {
         contact.append(ptr = new radar_blip_type, NULL);

         // pos
         compute_offset(atr->htree->old_state.tree_center, ptr->pos, scale, offset, winx, winy);

         // color
         switch (atr->htree->flags & GAMEFLAG_ALLIANCE_MASK) {
            case GAMEFLAG_RED:
               copyarray3(ptr->color, ogl_red);
               break;

            case GAMEFLAG_GREEN:
               copyarray3(ptr->color, ogl_green);
               break;

            default:      // case GAMEFLAG_BLUE:
               copyarray3(ptr->color, ogl_blue);
               break;
         }

      }

   if (target) {
      contact.append(ptr = (radar_blip_type *)rman.pop(), NULL);

      // pos
      compute_offset(target->htree->old_state.tree_center, ptr->pos, scale, offset, winx, winy);
      // color
      ptr->color[0] = 255;
      ptr->color[1] = 255;
      ptr->color[2] = 127;
   }

}


/* *************************************************************
************************************************************* */
int equip_systems_computer::query_whatwasi(int type) {

   return equip_systems_computer::query_whatami() == type ? 1 : equipment::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int equip_systems_computer::parse(FILE *infile, char *token) {

   if (!strcmp(token, SHADE_GOURAUD_STR)) {
      get_token(infile, token);
      rgb[0] = atoi(token);
      get_token(infile, token);
      rgb[1] = atoi(token);
      get_token(infile, token);
      rgb[2] = atoi(token);
      return 1;
   }

   return equipment::parse(infile, token);
}


/* *************************************************************
************************************************************* */
equipment *equip_systems_computer::copy(equipment *item) {

   equip_systems_computer *obj = (equip_systems_computer *)item;

   equipment::copy(item);

   *(unsigned int *)obj->rgb = *(unsigned int *)rgb;
   return item;
}


/* *************************************************************
************************************************************* */
void equip_systems_computer::reset(gameatom *source, object_desc *ob) {

   equip_status_computer::reset(source, ob);
   update_timer = SYSTEM_COMPUTER_UPDATE_DELAY;
   current_status_index = 0;
}


/* *************************************************************
************************************************************* */
void equip_systems_computer::update(dbl_llist_manager *hiearchy_manager) {

   equipment::update(hiearchy_manager);

   update_timer -= complex->timer.speedscale;

   if (update_timer < 0) {
      update_timer += SYSTEM_COMPUTER_UPDATE_DELAY;
      current_status_index++;
   }
   
}


/* *************************************************************
************************************************************* */
void equip_systems_computer::display_hud(dbl_llist_manager *hiearchy_manager, gameatom *source, engine *proc, mapul *mapbuffer) {

   hud_display_systems_computer(mapbuffer, this);
}


/* *************************************************************
************************************************************* */
int equip_systems_computer::cycle_display() {

   current_flag = (current_flag + 1) % 3;
   return current_flag == 0;
}


/* *************************************************************
************************************************************* */
int equip_ablative_armour::query_whatwasi(int type) {

   if (equip_ablative_armour::query_whatami() == type)
      return 1;

   return equipment::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int equip_ablative_armour::parse(FILE *infile, char *token) {

   if (!strcmp(token, TOKEN_MAXHP_STR)) {
      get_token(infile, token);
      armour = atoi(token);
      return 1;
   }

   return equipment::parse(infile, token);
}


/* *************************************************************
************************************************************* */
equipment *equip_ablative_armour::copy(equipment *item) {

   equip_ablative_armour *obj = (equip_ablative_armour *)item;

   equipment::copy(item);

   obj->armour = armour;
   return item;
}


/* *************************************************************
************************************************************* */
void equip_ablative_armour::reset(gameatom *source, object_desc *ob) {

   equipment::reset(source, ob);
   ((fighter_desc *)gear)->max_armour += armour;
}


/* *************************************************************
************************************************************* */
equip_shields::equip_shields() {

   downtime = 1;
   regen_rate = max_shields = 0;
   fx_id = 0;
   duration = 0;
}


/* *************************************************************
************************************************************* */
int equip_shields::query_whatwasi(int type) {

   if (equip_shields::query_whatami() == type)
      return 1;

   return equipment::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
int equip_shields::parse(FILE *infile, char *token) {

   if (!strcmp(token, TOKEN_DURATION_STR)) {
      get_token(infile, token);
      duration = (float)atof(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_MAX_SHIELDS_STR)) {
      get_token(infile, token);
      max_shields = (float)atof(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_REGEN_RATE_STR)) {
      get_token(infile, token);
      max_regen_rate = (float)atof(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_SHIELD_FX_STR)) {
      get_token(infile, token);
      fx_id = complex->teacher.identify(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_DOWNTIME_STR)) {
      get_token(infile, token);
      downtime = (float)atof(token);
      return 1;
   }

   if (!strcmp(token, TOKEN_SFX_STR)) {
      get_token(infile, token);
      sndfx_id.name.stringcpy(token);
      sndfx_id.flags = FLAG_SOUND_WAV | FLAG_SOUND_3D;
      return 1;
   }

   if (!strcmp(token, TOKEN_SFX_VOLUME_STR)) {
      get_token(infile, token);
      sndfx_id.volume(atoi(token));
      return 1;
   }

   return equipment::parse(infile, token);
}


/* *************************************************************
************************************************************* */
equipment *equip_shields::copy(equipment *item) {

   equip_shields *obj = (equip_shields *)item;

   equipment::copy(item);

   obj->regen_rate = regen_rate;
   obj->max_regen_rate = max_regen_rate;
   obj->max_shields = max_shields;
   obj->duration = duration;
   obj->fx_id = fx_id;
   obj->downtime = downtime;
   obj->sndfx_id.name.stringcpy(&sndfx_id.name);
   obj->sndfx_id.flags = sndfx_id.flags;
   obj->sndfx_id.db = sndfx_id.db;
   return item;
}


/* *************************************************************
************************************************************* */
void equip_shields::reset(gameatom *source, object_desc *ob) {

   equipment::reset(source, ob);
   ((fighter_desc *)gear)->max_shields += max_shields;
   regen_rate = max_regen_rate;

   sndfx_id.id = NULL;
   if (source == global_player)
      complex->soundmanager->find(&sndfx_id);
}


/* *************************************************************
************************************************************* */
void equip_shields::update(dbl_llist_manager *hiearchy_manager) {

   equipment::update(hiearchy_manager);

   ((fighter_desc *)gear)->shields += regen_rate*complex->timer.speedscale;
}


/* *************************************************************
************************************************************* */
void equip_shields::apply_damage(int *damage) {

   int t = hp;

   equipment::apply_damage(damage);

   t -= hp;

   regen_rate = (max_regen_rate*hp)*imaxhp;
   ((fighter_desc *)gear)->max_shields -= (t*max_shields)*imaxhp;
}


/* *************************************************************
************************************************************* */
void equip_shields::activate(gameatom *tree, vector4f *mx, float *pos) {

   float *shield_data;
   shield_ripple *shield;

   // gfx
   tree->query_specific_data(DATAFLAG_SHIELDS, &shield_data);
   shield = (shield_ripple *)complex->teacher.issue(fx_id);

   if (shield)
      shield->init(tree, mx, pos, shield_data, duration);

   // sfx
   if (tree == global_player)
      ((player *)tree)->attach_sound(&sndfx_id, pos);
}


/* *************************************************************
************************************************************* */
int equip_inertial_compensator::query_whatwasi(int type) {

   return equip_inertial_compensator::query_whatami() == type ? 1 : equipment::query_whatwasi(type);
}


/* *************************************************************
************************************************************* */
equipment *equip_inertial_compensator::copy(equipment *item) {

   equip_inertial_compensator *obj = (equip_inertial_compensator *)item;

   equipment::copy(item);

   obj->toggle     = toggle;
   obj->playerflag = playerflag;
   return item;
}


/* *************************************************************
************************************************************* */
void equip_inertial_compensator::reset(gameatom *source, object_desc *ob) {

   equipment::reset(source, ob);
   toggle = 1;
   playerflag = source == global_player;
}


/* *************************************************************
************************************************************* */
void equip_inertial_compensator::update(dbl_llist_manager *hiearchy_manager) {

   equipment::update(hiearchy_manager);

   if (playerflag)
      if (complex->userinput->translate_stroke(KEYSTROKE_INERTIA_TOGGLE))
         toggle = !toggle;
}


/* *************************************************************
************************************************************* */
int equip_inertial_dampener::query_whatwasi(int type) {

   return (equip_inertial_dampener::query_whatami() == type) ? 1 : equip_inertial_compensator::query_whatwasi(type);
}
