

#include <stdlib.h>
#include <time.h>

#ifdef WIN32

#include "WIN_WOGL.h"

#include <GL/gl.h>

#define WINDOW_TYPE WOGLwindow

#else

#include "WIN_XOGL.h"

#define WINDOW_TYPE XOGLwindow

#endif

#include "image.h"
#include "polygon.h"
#include "pstring.h"

#include "electron.h"
#include "photon.h"
#include "anineutr.h"

#include "gamegine.h"
#include "gfx.h"

gameatom *object_tree = NULL;
gameatom *camera_tree = NULL;


#ifdef WIN32

/* **********************************************************************
   Process the messages for the main application window
********************************************************************** */
LONG FAR PASCAL game_polling(HWND window, UINT message, WPARAM wparam, LPARAM lparam) {

   switch (message) {

      case WM_DESTROY:
         PostQuitMessage( 0 );
         return 0;

      default:
         break;
   }

   return DefWindowProc(window, message, wparam, lparam);
}

#endif


/* **************************************************
   initialize the system
************************************************** */
void startup(VIRTUALwindow *win) {

   gamegod_type *ggod;

   // setup the overall manager
   ggod = new gamegod_type(&proc);

   // create the rendering window
   ggod->set_win(win);

   // setup the software/opengl rendering system
   ggod->set_gfx_manager(RENDER_MODE == RENDERER_OPENGL ? (base_gfx *)new ogl_gfx : (base_gfx *)new sw_gfx);

   // setup the  resource manager
   ggod->set_game_manager(new game_manager);

   // register the "camera" object loader
   ggod->register_resource_loader(RESOURCE_OBJECT_LOADER, new electron_loader);

   // register the "light" object loader
   ggod->register_resource_loader(RESOURCE_OBJECT_LOADER, new photon_loader);

   // register the .rtf 2D image processor
   ggod->register_resource_loader(RESOURCE_IMAGE_LOADER, new rtf);

   // register the IRIS .rgb 2D image processor
   ggod->register_resource_loader(RESOURCE_IMAGE_LOADER, new rgb);

   // initialize the system
   init_gamegine(ggod);

   // setup the camera tree
   camera_tree = load_gameatom("config/camera.atm", &complex->animation_manager);
   if (!camera_tree) {
      pprintf("Error: Cannot locate \"camera.atm\"... Aborting...\n");
      exit(0);
   }

   // load benchmark object
   object_tree = load_game_object(SDEFLST.string, &complex->animation_manager);
}


/* **************************************************
   shutdown the system
************************************************** */
void shutdown() {

   static int shutdown_flag = 0;

   if (shutdown_flag)
      return;

#ifdef __FreeBSD__
   fpresetsticky(FP_X_INV|FP_X_OFL|FP_X_UFL|FP_X_DZ|FP_X_IMP|FP_X_DNML);
#endif

   // shutdown the system
   shutdown_gamegine();

   shutdown_flag = 1;
}


/* **************************************************
   count the polygons directly controlled by the
   "qtr" animation element
************************************************** */
int count_polygons(quark *qtr, quark *parent) {

   int count = 0;
   superclass *ob;
   linktype *ptr;

   ob = qtr->get_render_object();

   if (ob && ob->query_whatwasi(OBJECT_POLYGON) && ((polygon *)ob)->get_render_object())
      count = ((polygon *)ob)->get_render_object()->countobject;

   for (ptr=(linktype *)qtr->edge.head; ptr; ptr = (linktype *)ptr->next)
      if (ptr->link != parent)
         count += count_polygons(ptr->link, qtr);

   return count;
}


/* **************************************************
   main animation loop
************************************************** */
void main_loop(BLTwindow *win) {

   vector4f mx[4];
   int framecount;
   TIME_STRUCT t;
   vector4f frustum[FRUSTUM_COUNT];
   camera *cparam;
   time_t oldtime, newtime;
   float duration, dist;
   atom_list_type *atr;
   float rowsize = (float)(win->mapbuffer.maxy/32.0);
   int x = (int)(win->mapbuffer.maxx*0.75);
   float y;
   char buffer[MAXSTRLEN];
   int count;

#ifdef WIN32
   SetWindowLong(((WINDOW_TYPE *)win)->mwindow, GWL_WNDPROC, (long)game_polling);
   SetWindowPos(((WINDOW_TYPE *)win)->mwindow, NULL, 0,0,0,0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOZORDER | SWP_FRAMECHANGED);
#endif

   // setup the text on screen characteristics
   if (complex->font) {
      complex->font->set_color((unsigned char)255, (unsigned char)0, (unsigned char)255);
      complex->font->set_scale((float)(rowsize*0.85));
   }

   // initialize the 3D engine
   proc.engine_init();

   // setup the camera
   cparam = (camera *)camera_tree->find_ob(NULL, OBJECT_CAMERA);

   if (!cparam) {
      pprintf("Error: No camera.  Aborting...\n");
      return;
   }
   
   cparam->set_dim(win->mapbuffer.maxx, win->mapbuffer.maxy);

   // setup the animation object
   init_mx(mx);

   for (atr = (atom_list_type *)complex->animation_manager.head; atr; atr = (atom_list_type *)atr->next) {
      atr->htree->begin(&complex->animation_manager, NULL, mx);
      atr->htree->update(&complex->animation_manager, NULL);
   }

   // setup the frustum cull system 
   cparam->calc_clip_plane();
   cparam->query_frustum(frustum);

   dist = object_tree->old_state.tree_radius + (float)magnitude3(object_tree->old_state.tree_center);
   dist += dist;

   // setup rendering statistics
   time(&oldtime);
   duration = 0;
   count = count_polygons(object_tree, NULL);

   // render the rest fo the frames
   for (framecount=0; framecount<10000; framecount++) {

      // rotate the object in the center of the screen
      init_mx(mx);
      rotate_mx_x(mx, PI/180.0*framecount);
      rotate_mx_y(mx, PI/180.0*framecount*2);
      rotate_mx_z(mx, PI/180.0*framecount*4);
      mx[2][3] = dist/cparam->vrc[3];

      // animate the objects
      for (atr = (atom_list_type *)complex->animation_manager.head; atr; atr = (atom_list_type *)atr->next)
         if (atr->htree != camera_tree) {
            atr->htree->whereami(&complex->animation_manager, NULL, mx);
            atr->htree->update(&complex->animation_manager, NULL);
         }

      // setup the 3D engine for rendering
      proc.engine_init();

      // register objects to the rendering engine
      for (atr = (atom_list_type *)complex->animation_manager.head; atr; atr = (atom_list_type *)atr->next)
         atr->htree->render_object(&proc, NULL, frustum, FRUSTUM_CLIP_ALL);

      // render the frame
      proc.spawn(framecount, &win->mapbuffer);

      // setup the 2D engine for rendering
      complex->gfx->gfx2DMode(win->mapbuffer.maxx, win->mapbuffer.maxy);

      // print current stats of the rendering engine to the screen
      if (complex->font && !(framecount % 20)) {
         time(&newtime);
         if (newtime > oldtime)
            duration = (framecount +1)/(float)(newtime - oldtime);
      }

      y = 0.025f*win->mapbuffer.maxy;
      sprintf(buffer, "PPS: %4.4f", count*duration);
      complex->font->print(x, (int)y, (unsigned char *)buffer, &win->mapbuffer);

      y += rowsize;
      sprintf(buffer, "FPS: %4.4f", duration);
      complex->font->print(x, (int)y, (unsigned char *)buffer, &win->mapbuffer);

      // make sure accessed objects remain resident in memory
      GETTIME(&t);
      global_resource_manager->update((float)TIME2FLOAT(t));

      // display the frame to the screen
      complex->gfx->gfxBitblt(win, &win->mapbuffer);
   }

   // dump final rendering statistics to the logfile
   if (logfile) {
      time(&newtime);
      duration = (float)(newtime - oldtime);

      fprintf(logfile, "Framecount %d\n", framecount);
      fprintf(logfile, "Framerate Average %4.4f\n", framecount/duration);
      fprintf(logfile, "Polygon Average %4.4f\n", (count * framecount)/duration);
   }

}


/* **************************************************
   main function
************************************************** */
int main(int argc, char **argv) {

   WINDOW_TYPE *win;
   int winx, winy;

   int attribs[] = {
      OGLFLAG_BPP, 32,
      OGLFLAG_DOUBLEBUFFER,
      OGLFLAG_DEPTH_SIZE, 1,
      OGLFLAG_RGBA,
      OGLFLAG_MAPBUFFER,
      OGLFLAG_NULL
   };

   // register default exit routine
   atexit(shutdown);

   // set keyboard as main input device
   CONTROLLER = CONTROLKEYBOARD;

   // set rendering engine to OpenGL
   RENDER_MODE = RENDERER_OPENGL;

   // set font mode to raster
   FONTTYPE = FONT_RASTER;

   // set font bitmaps
   FILENAME_FONT.stringcpy("config/200.fnt");

   // load objects from the default directory
   FILENAME_PRESPG[0] = 0;

   // parse command line argument
   parseinput(argc, argv);

   // setup the rendering screen
   gfx_get_screen_dim(&winx, &winy);
   win = new WINDOW_TYPE;
   if (!win->VIRTUALreset(argc, argv, winx, winy, "Pixcon", (void *)attribs))
      return 0;

   // init the game engine
   startup(win);

   // enter the rendering loop
   main_loop(win);

   // shutdown the game engine
   shutdown();

   // close the rendering screen
   win->VIRTUALcancel();
   delete win;

   return 1;
}

