

/* *************************************************************
   This program is a keyframe-hiearchical script interpretor
   designed to create an animation datafile in the PIXCON
   data ascii file format

   by: Alex Harper              12/20/93
************************************************************* */


#include <stdlib.h>
#include <time.h>


#ifdef WIN32

#include "WIN_WOGL.h"

#define WINDOW_TYPE WOGLwindow

#else

#include <string.h>

#include "WIN_XOGL.h"

#ifdef __FreeBSD__
#include <floatingpoint.h>
#endif

#define USERIO_TYPE user_io_X
#define WINDOW_TYPE XOGLwindow

#endif

#include "shaders.h"
#include "image.h"
#include "spg_load.h"
#include "pstring.h"

#include "quark.h"
#include "lock.h"
#include "flight.h"
#include "boid.h"
#include "neutron.h"
#include "electron.h"
#include "photon.h"
#include "tachyon.h"
#include "ion.h"
#include "chain.h"
#include "anicode.h"

#include "anitroll.h"

// parser results

int FRAME = 0;              // default - calc the entire animation
int FRAME_START, FRAME_END;

int CONTINUE = 0;           // default - start frame # at 1

int RENDER = 0;		    // default - output to file
int WINX = 640;
int WINY = 480;

string_type infilename;
string_type outfilename;

shader *shades = NULL;


/* **************************************************
************************************************** */
void shutdown() {

#ifdef __FreeBSD__
   fpresetsticky(FP_X_INV|FP_X_OFL|FP_X_UFL|FP_X_DZ|FP_X_IMP|FP_X_DNML);
#endif

}


/* **************************************************
   // build mipmaps for 2D textures
************************************************** */
void texture_postprocessCB(texbase *tptr) {

   tptr->pre_process_texture();
   tptr->build_mipmaps((dbl_llist_manager *)global_resource_manager->get_resource_object(RESOURCE_IMAGE_LOADER), ((frame_manager *)global_resource_manager)->query_color_byte_order());
}


/***************************************************************************
****************************************************************************/
void insert_shader(char *str, int fcn(void *), void icn() = NULL) {

   shader *ptr;

   ptr = new shader;
   ptr->set(str, (void *)fcn, (void *)icn);
   global_resource_manager->register_resource_object(RESOURCE_SHADER, ptr);
}


/* **************************************************
************************************************** */
void init_shader() {

   insert_shader("marble", marble);                                     // surface: marble texture
   insert_shader("wood", wood);                                         // surface: plywood texture
   insert_shader("wave", wave);                                         // surface: normal manipulator
   insert_shader("checker", checker);                                   // surface: 3D checker pattern
   insert_shader("wood_panel", wood_panel);                             // surface: plywood panels

   insert_shader("moon", moon, moon_init);                              // surface: fractal moon
   insert_shader("planet", planet, planet_init);                        // surface: fractal planet - variation of moon
   insert_shader("cloud", cloud, cloud_init);                           // surface: cloud texture
   insert_shader("stars", stars, stars_init);                           // surface: star texture
   insert_shader("warp", warp, warp_init);                              // surface: displaces xyz for color calculations
   insert_shader("terrain", terrain, terrain_init);                     // surface: color base on altitude
   insert_shader("planet2", planet2, planet2_init);                     // surface: another planet
   insert_shader("cloud2", cloud2, cloud2_init);                        // surface: another cloud texture
   insert_shader("terrain2", terrain2, terrain2_init);                  // surface: another terrain texture
   insert_shader("terrain3", terrain3, terrain3_init);                  // surface: yet another terrain
   insert_shader("cave_ceiling", cave_ceiling, cave_ceiling_init);      // surface: (stalagmite?) shader
   insert_shader("cave_floor", cave_floor, cave_floor_init);            // surface: (stalagtite?) shader

   insert_shader("fog_mus", fog_mus);                                   // atmosphere: musgrave's fog shader
   insert_shader("fog", fog);                                           // atmosphere: another fog shader
   insert_shader("fog2", fog2);                                         // atmosphere: yet another fog shader
   insert_shader("nightfog2", nightfog2);                               // atmosphere: yet another fog shader
   insert_shader("energy_beam", energy_beam);                           // atmosphere: "Vandagraph" electron bolts

   insert_shader("planetclouds", planetclouds);                         // surface: better than cloud2()
   insert_shader("contour", contour);                                   // surface: a weird contour map thingy
   insert_shader("carpet", carpet);                                     // surface: carpet texture

   insert_shader("default_alt", default_alt);                           // sbfast: a height field
   insert_shader("x_alt", x_alt);                                       // sbfast: default_alt w/o clamping
   insert_shader("mnb_alt", mnb_alt);                                   // sbfast: another height field
   insert_shader("barfland_alt", barfland_alt);                         // sbfast: another height field
}


/* **************************************************
************************************************** */
void init_aloader() {

   if (global_resource_manager)
      delete global_resource_manager;
               
   global_resource_manager = new frame_manager;
   global_resource_manager->init(1);
   ((frame_manager *)global_resource_manager)->set_render(RENDER);
   ((frame_manager *)global_resource_manager)->set_postprocessCBtexture((void *)texture_postprocessCB);

   init_shader();

   global_resource_manager->register_resource_object(RESOURCE_IMAGE_LOADER, new tiff);
   global_resource_manager->register_resource_object(RESOURCE_IMAGE_LOADER, new ppm);
   global_resource_manager->register_resource_object(RESOURCE_IMAGE_LOADER, new ilbm);
   global_resource_manager->register_resource_object(RESOURCE_IMAGE_LOADER, new bmp);
   global_resource_manager->register_resource_object(RESOURCE_IMAGE_LOADER, new jpeg);
   global_resource_manager->register_resource_object(RESOURCE_IMAGE_LOADER, new gif);
   global_resource_manager->register_resource_object(RESOURCE_IMAGE_LOADER, new rgb);
   global_resource_manager->register_resource_object(RESOURCE_IMAGE_LOADER, new rtf);

   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new neutron_loader);  // polygon
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new electron_loader); // electron
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new photon_loader);   //light - generic 
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new tachyon_loader);  // particle
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new shell_loader);    // spatch
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new ion_loader);      // radiant
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new helix_loader);    // sbfast
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new tube_loader);     // cylinder
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new funnel_loader);   // cone
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new chain_loader);    // line
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new buckey_loader);   // sphere
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new quark_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new lock_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new flight_loader);
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new boidbait_loader); 
   global_resource_manager->register_resource_object(RESOURCE_OBJECT_LOADER, new boid_loader);

   global_resource_manager->register_resource_object(RESOURCE_POLYGON_LOADER, new spg_loader);
}


/* **************************************************
************************************************** */
void init() {

   init_noise();
   init_lut();                          // init matrixlib

   init_aloader();              // build anitroll loader
}


/* **************************************************
   this function parses command line to find user defined files,
   or else use the default files.
************************************************** */
void parseinput(int argc, char **argv) {

   int i;

   for (i=1; i<argc; i++)
      if (*(argv[i]) == '-')
         switch (*(argv[i]+1)) {
            case 'c':			// continue numbering (must have -f)
               CONTINUE = 1;
               break;

            case 'f':                   // partial animation
               FRAME = 1;
               i++;
               FRAME_START = atoi(argv[i]);
               i++;
               FRAME_END = atoi(argv[i]);
               break;

            case 'i':
               i++;
               infilename.stringcpy(argv[i]);
               break;

            case 'o':
               i++;
               outfilename.stringcpy(argv[i]);
               break;

            case 'r':			// render mode
               RENDER = 1;
               break;

            case 's':
               SILENT = 1;
               break;

            case 't':                   // test mode
               WF = 1;
               break;

            case 'v':
               SILENT = 0;
               break;

            case 'x':
               i++;
               WINX = atoi(argv[i]);
               break;

            case 'y':
               i++;
               WINY = atoi(argv[i]);
               break;

            default:
               break;
         }

}


/* *************************************************************
   This function reads in the data for the animation (.cnt) file
************************************************************* */
void read_data(char *filename, int *framecount, dbl_llist_manager *hiearchy_manager) {

   FILE        *infile;                          // input file
   int         count;                            // # of objects
   atom        *header = (atom *)NULL;           // list of objects
   atom        *ptr;                             // temp ptr
   int         i;                                // looping var
   char        token[MAXSTRLEN], buffer[MAXSTRLEN];
   atom_loader aloader;
   geneological_type ani_param;
   atom_list_type *atr;
   
   if (!(infile = fopen(filename, "r"))) {
      sprintf(perror_buffer, "Error: Could not access %s... Aborting...\n", filename);
      pprintf(perror_buffer);
      exit(0);
   }

   get_token(infile, token);
   lower_case(token);

   if (strcmp(token, "anitroll")) {
      sprintf(perror_buffer, "Error: \"%s\" is not an Anitroll script... Aborting...\n", filename);
      pprintf(perror_buffer);
      fclose(infile);
      exit(0);
   }

   get_token(infile, token);                // frame count
   get_token(infile, token);
   *framecount = atoi(token);

   get_token(infile, token);                // shadow plane
   get_token(infile, token);
   SHADOW_PLANE[0] = (float)atof(token);
   get_token(infile, token);
   SHADOW_PLANE[1] = (float)atof(token);
   get_token(infile, token);
   SHADOW_PLANE[2] = (float)atof(token);
   get_token(infile, token);
   SHADOW_PLANE[3] = (float)atof(token);

   get_token(infile, token);                // atom count
   get_token(infile, token);
   count = atoi(token);

   for (i=0; i<count; i++) {

      get_token(infile, buffer);             // atm filename
      get_token(infile, token);              // "{"

      strcpy(token, "atom");
      if (!(ptr = (atom *)aloader.parse(infile, token))) {
         sprintf(perror_buffer, "Invalid object type \"%s\"\n", token);
         pprintf(perror_buffer);
         fclose(infile);
         exit(1);
      }

      // read atom hiearchy
      if (!ptr->read_data(buffer)) {
         sprintf(perror_buffer, "Error: atom::read_data() - Could not access %s...  Aborting...\n", buffer);
         pprintf(perror_buffer);
         exit(0);
      }

      hiearchy_manager->append(atr = new atom_list_type, NULL);
      atr->htree = ptr;

      ani_param.tree = atr;
      ani_param.parent = NULL;
      ptr->preprocess(&ani_param);
   }

   fclose(infile);
}


/* ****************************************************************
**************************************************************** */
void render(int argc, char **argv, dbl_llist_manager *hiearchy_manager) {

   aengine proc;
   WINDOW_TYPE win;
   clock_t oldtime;
   atom_list_type *atr;
   vector4f mx[4];
   int i, j;

   int attribs[] = {
      OGLFLAG_BPP, 16,
      OGLFLAG_DOUBLEBUFFER,
      OGLFLAG_DEPTH_SIZE, 1,
      OGLFLAG_RGBA,
      OGLFLAG_MAPBUFFER,
      OGLFLAG_NULL
   };

   if (!SILENT)
      pprintf("Animating...\n");

   // open window
   if (!win.VIRTUALreset(argc, argv, WINX, WINY, "Pixcon", (void *)attribs))
      exit(-1);

   proc.adata.init_map(WINX, WINY);
   proc.adata.zero();

   proc.zdata.init_map(WINX, WINY);
   proc.zdata.zero();

   oldtime = clock();

   // first frame :)
   if (FRAME_START <= 1) {

      proc.engine_init();

      for (atr=(atom_list_type *)hiearchy_manager->head; atr; atr=(atom_list_type *)atr->next) {
         atr->htree->new_action(1, hiearchy_manager);
         init_mx(mx);

         atr->htree->begin(hiearchy_manager, NULL, mx);
         atr->htree->render_object(&proc, NULL, NULL, FRUSTUM_CLIP_NONE);
         atr->htree->update(hiearchy_manager, NULL);
      }

      proc.spawn(1, &win.mapbuffer);

      win.BLTbitmap(&win.mapbuffer);

      i = 2;
   }

   else {
      for (atr=(atom_list_type *)hiearchy_manager->head; atr; atr=(atom_list_type *)atr->next) {
         atr->htree->new_action(1, hiearchy_manager);
         init_mx(mx);

         atr->htree->begin(hiearchy_manager, NULL, mx);
         atr->htree->update(hiearchy_manager, NULL);
      }

      // update animation to start frame
      for (i=2; i<FRAME_START; i++)
         for (atr=(atom_list_type *)hiearchy_manager->head; atr; atr=(atom_list_type *)atr->next) {
            atr->htree->new_action(i, hiearchy_manager);
            init_mx(mx);

            atr->htree->whereami(hiearchy_manager, NULL, mx);
            atr->htree->update(hiearchy_manager, NULL);
         }

   }

   if (CONTINUE)
      j = i;
   else
      j = (1 >= FRAME_START) ? 2 : 1;

   for (; i<=FRAME_END; i++, j++) {

      proc.engine_init();

      for (atr=(atom_list_type *)hiearchy_manager->head; atr; atr=(atom_list_type *)atr->next) {
         atr->htree->new_action(i, hiearchy_manager);
         init_mx(mx);

         atr->htree->whereami(hiearchy_manager, NULL, mx);
         atr->htree->render_object(&proc, NULL, NULL, FRUSTUM_CLIP_NONE);
         atr->htree->update(hiearchy_manager, NULL);
      }

      proc.spawn(j, &win.mapbuffer);

      win.BLTbitmap(&win.mapbuffer);
   }

   printf("Average Frame Time : %f\n", (clock()-oldtime)/(float)(CLOCKS_PER_SEC*(FRAME_END-FRAME_START+1)));

   hiearchy_manager->dest();

   win.BLTbitblt(&win.mapbuffer);

   win.VIRTUALcancel();         // independent of OGL/X
}


/* ****************************************************************
   This is the animation controller - main function
**************************************************************** */
void export_animation(dbl_llist_manager *hiearchy_manager) {

   char filename[MAXSTRLEN];
   char buffer[MAXSTRLEN];
   int  obcount;                      // # of objects
   clock_t oldtime;
   FILE   *outfile, *infile;
   atom_list_type *atr;               // object pointers
   int    i, j;                       // looping var
   vector4f mx[4];

   sprintf(filename, "%d", time(NULL) % 10000000);

   if (!SILENT)
      pprintf("Animating...\n");

   obcount = 0;

   oldtime = clock();

   if (!(outfile = fopen(filename, "w"))) {
      pprintf("Error: Could not open temporary file for output... Aborting...\n");
      exit(1);
   }

   // first frame :)
   if (FRAME_START <= 1) {
      for (atr=(atom_list_type *)hiearchy_manager->head; atr; atr=(atom_list_type *)atr->next) {
         atr->htree->new_action(1, hiearchy_manager);
         init_mx(mx);

         atr->htree->begin(hiearchy_manager, NULL, mx);
         obcount += atr->htree->write_data(outfile, NULL, 1);
         atr->htree->update(hiearchy_manager, NULL);
      }

      i = 2;
   }

   else {
      for (atr=(atom_list_type *)hiearchy_manager->head; atr; atr=(atom_list_type *)atr->next) {
         atr->htree->new_action(1, hiearchy_manager);
         init_mx(mx);

         atr->htree->begin(hiearchy_manager, NULL, mx);
         atr->htree->update(hiearchy_manager, NULL);
      }

      // update animation to start frame
      for (i=2; i<FRAME_START; i++)
         for (atr=(atom_list_type *)hiearchy_manager->head; atr; atr=(atom_list_type *)atr->next) {
            atr->htree->new_action(i, hiearchy_manager);
            init_mx(mx);

            atr->htree->whereami(hiearchy_manager, NULL, mx);
            atr->htree->update(hiearchy_manager, NULL);
         }

   }

   if (CONTINUE)
      j = i;
   else
      j = (1 >= FRAME_START) ? 2 : 1;

                                         // calculate animation
   for (; i<=FRAME_END; i++, j++) {

      for (atr=(atom_list_type *)hiearchy_manager->head; atr; atr=(atom_list_type *)atr->next) {
         atr->htree->new_action(i, hiearchy_manager);
         init_mx(mx);

         atr->htree->whereami(hiearchy_manager, NULL, mx);
         obcount += atr->htree->write_data(outfile, NULL, j);
         atr->htree->update(hiearchy_manager, NULL);
      }

   }

   fclose(outfile);

   if (!(infile = fopen(filename, "r"))) {
      pprintf("Error: Could not open temporary file for input... Aborting...\n");
      exit(1);
   }

   if (!(outfile = fopen(outfilename.string, "w"))) {
      sprintf(perror_buffer, "Error: Could not open %s for output... Aborting...\n", outfilename.string);
      pprintf(perror_buffer);
      fclose(infile);
      exit(1);
   }

                                        // print # of frames & objects
   fprintf(outfile, "list\nframe_count %d\nobject_count %d\n\n", FRAME_END-FRAME_START+1, obcount);

   while(fgets(buffer, MAXSTRLEN, infile))
      fprintf(outfile, "%s", buffer);

   fclose(outfile);
   remove(filename);

   printf("Average Frame Time : %f\n", (clock()-oldtime)/(float)(CLOCKS_PER_SEC*(FRAME_END-FRAME_START+1)));
   hiearchy_manager->dest();
}


/* ****************************************************************
   This is the animation controller - main function
**************************************************************** */
int main(int argc, char **argv) {

   int framecount;
   camera *camptr;
   atom_list_type *atr;
   dbl_llist_manager animation_manager;

#ifdef __FreeBSD__
   fpsetmask(0);
#endif

   atexit(shutdown);

   if (argc < 3) {
      printf("usage : anitroll [-in <filename>] [-out <filename>] [-test] [-frame # #] [-c] [-render]\n");
      printf("                 [-silent] [-verbose] [-x <length>] [-y <height>]\n");
      exit(0);
   }

   pprintf("Preprocessing...\n");

   parseinput(argc, argv);

   init();

   pprintf("Parsing scripts...\n");

   read_data(infilename.string, &framecount, &animation_manager);

   if (!FRAME) {
      FRAME_START = 1;
      FRAME_END = framecount;
   }

   else {
      if (FRAME_START < 1)
         FRAME_START = 1;

      if (FRAME_END < FRAME_START) {
         pprintf("Error: Invalid frame numbers... Aborting...\n");
         exit(0);
      }

   }

   // find camera
   for (camptr=NULL, atr=(atom_list_type *)animation_manager.head; atr && !camptr; atr=(atom_list_type *)atr->next)
      camptr = (camera *)atr->htree->find_ob(NULL, OBJECT_CAMERA);

   if (camptr)
      camptr->set_dim(WINX, WINY);

   if (RENDER)
      render(argc, argv, &animation_manager);
   else
      export_animation(&animation_manager);

   return 1;
}

