

#include <stdlib.h>
#include <time.h>

#ifdef WIN32

#include "WIN_WOGL.h"
#include <GL/gl.h>
#include "user_io_w32.h"

#define USERIO_TYPE user_io_win
#define WINDOW_TYPE WOGLwindow

#else

#include "WIN_XOGL.h"
#include "user_io_x.h"

#define USERIO_TYPE user_io_X
#define WINDOW_TYPE XOGLwindow

#endif

#include "csphrlib.h"

#include "electron.h"
#include "photon.h"

#include "gamegine.h"

#include "image.h"
#include "polygon.h"
#include "pstring.h"

gameatom *object_tree;
gameatom *camera_tree;
mouse_state_type mouse_state;

float RFACTOR = 1.0f;
float TFACTOR = 1.0f;


#ifdef WIN32

/* **********************************************************************
   Process the messages for the main application window
********************************************************************** */
LONG FAR PASCAL game_polling(HWND window, UINT message, WPARAM wparam, LPARAM lparam) {

   switch (message) {

      case WM_DESTROY:
         PostQuitMessage( 0 );
         return 0;

      default:
         break;
   }

   return DefWindowProc(window, message, wparam, lparam);
}

#endif


/* **************************************************
************************************************** */
void startup(VIRTUALwindow *win, game_manager *grm) {

   gamegod_type *ggod;

   // setup managers
   ggod = new gamegod_type(&proc);
   ggod->set_win(win);
   ggod->set_game_manager(grm);
   ggod->set_sound_manager(new soundman);
   ggod->set_io_manager(new USERIO_TYPE);
   ggod->set_gfx_manager(RENDER_MODE == RENDERER_OPENGL ?(base_gfx *)new ogl_gfx : (base_gfx *)new sw_gfx);

   // build anitroll loader list
   ggod->register_resource_loader(RESOURCE_OBJECT_LOADER, new electron_loader);
   ggod->register_resource_loader(RESOURCE_OBJECT_LOADER, new photon_loader);

   // register texture loader
   ggod->register_resource_loader(RESOURCE_IMAGE_LOADER, new rtf);
   ggod->register_resource_loader(RESOURCE_IMAGE_LOADER, new rgb);

   // init gamegine libs
   init_gamegine(ggod);

   // load the camera tree
   camera_tree = load_gameatom("config/camera.atm", &complex->animation_manager);
   if (!camera_tree) {
      mbprintf("ERROR: Cannot locate \"camera.atm\"... Aborting...\n");
      exit(0);
   }
   
   // init the mouse
   complex->userinput->flag_on(FLAG_IO_MOUSE_CONSTRAIN);
   mouse_state.flag = FLAG_MOUSE_DRAW;

   init_game();
}


/* **************************************************
************************************************** */
void shutdown() {

   static int shutdown_flag = 0;

   if (shutdown_flag)
      return;

#ifdef __FreeBSD__
   fpresetsticky(FP_X_INV|FP_X_OFL|FP_X_UFL|FP_X_DZ|FP_X_IMP|FP_X_DNML);
#endif

   shutdown_game();
   
   shutdown_gamegine();

   shutdown_flag = 1;
}


/* **************************************************
************************************************** */
void calc_initial_mx(camera *cparam, atom *object, vector4f *mx) {

   float r;
   
   r = object->old_state.tree_radius + (float)magnitude3(object->old_state.tree_center);

//   mx[0][3] = -object->old_state.tree_center[0];
//   mx[1][3] = -object->old_state.tree_center[1];
//   mx[2][3] = -object->old_state.tree_center[2] + object->old_state.tree_radius/cparam->vrc[3]*2.0f;
   mx[2][3] = (r+r)/cparam->vrc[3];
}


/* **************************************************
************************************************** */
void user_rotate(camera *cparam, vector4f *mx, BLTwindow *win) {

   vector4f imx[4], mrot[4], orient[4];
   float x;
   vector3f temp1, temp2;
   vector2f v;
   
   // CALC MX's to go from/to object space
   
   // horizontal movement = pitch
   temp1[0] = mouse_state.old_state[0];
   temp1[1] = mouse_state.old_state[1];
   temp1[2] = -1;

   temp2[0] = mouse_state.new_state[0];
   temp2[1] = mouse_state.new_state[1];
   temp2[2] = -1;

   cparam->screen2map(temp1);
   cparam->screen2map(temp2);

   temp1[2] = 1;
   temp2[2] = 1;

   // calc start/stop in local orientation
   normalize3(temp1);
   normalize3(temp2);

   x = dotproduct3(temp1, temp2);

   if (fabs(x) > ICORRECT)
      return;

   init_mx(mrot);

   copyarray3(mrot[2], temp1);

   xproduct(mrot[1], temp1, temp2);
   normalize3(mrot[1]);

   xproduct(mrot[0], mrot[1], mrot[2]);

   transpose(imx, mrot);

   // grab Z, calc rotation mx around Y-axis
      
   init_mx(orient);

   v[0] = (mouse_state.old_state[0]-mouse_state.new_state[0])/win->mapbuffer.maxx;
   v[1] = (mouse_state.old_state[1]-mouse_state.new_state[1])/win->mapbuffer.maxy;

   x = (float)magnitude2(v)*PI*RFACTOR*complex->timer.speedscale;

   orient[0][0] = orient[2][2] = (float)cos(x);
   orient[2][0] = -(orient[0][2] = (float)sin(x));

   matmatmultv(imx, mx);
   matmatmultv(orient, mx);
   matmatmultv(mrot,  mx);
}


/* **************************************************
************************************************** */
void user_translate(camera *cparam, atom *object, vector4f *mx, BLTwindow *win) {

   float t;
   float r;
   
   t = -(mouse_state.new_state[1]-mouse_state.old_state[1])/win->mapbuffer.maxy;
   r = object->old_state.tree_radius + (float)magnitude3(object->old_state.tree_center);

   mx[2][3] += (complex->timer.speedscale*TFACTOR*t*r)/cparam->vrc[3];
}


/* **************************************************
************************************************** */
void process_events(vector4f *mx, vector4f *tmx, camera *cparam, unsigned int *flags, BLTwindow *win) {

   complex->userinput->process_events(&complex->timer);

   mouse_state.new_state[0] = (1 + complex->userinput->joyaxis[AXIS_MOUSE_X])*win->mapbuffer.maxx*0.5f;
   mouse_state.new_state[1] = (1 - complex->userinput->joyaxis[AXIS_MOUSE_Y])*win->mapbuffer.maxy*0.5f;

   if (*flags & FLAG_QUERYQUIT) {
      if (complex->userinput->key_stroke(KEYSTROKE_Y)) {
         *flags |= FLAG_QUIT;
         return;
      }

      if (complex->userinput->key_stroke(KEYSTROKE_N) || complex->userinput->key_stroke(KEYSTROKE_ESC))
         *flags &= ~FLAG_QUERYQUIT;
   }
      
   else
      *flags |= complex->userinput->key_stroke(KEYSTROKE_ESC) ? FLAG_QUERYQUIT : FLAG_NULL;

   if (complex->userinput->status_buttonpress & BUTTONLEFT)
      if (mouse_state.button_state & BUTTONLEFT)
         user_rotate(cparam, tmx, win);
      else {
         mouse_state.button_state |= BUTTONLEFT;
         copyarray2(mouse_state.old_state, mouse_state.new_state);
      }
   
   else {
      if (mouse_state.button_state & BUTTONLEFT) {
         user_rotate(cparam, tmx, win);
         matmatmultv(tmx, mx);

         init_mx(tmx);
         mouse_state.button_state &= ~BUTTONLEFT;
      }
   
      if (complex->userinput->status_buttonpress & BUTTONRIGHT) {
         if (mouse_state.button_state & BUTTONRIGHT) {
            user_translate(cparam, object_tree, tmx, win);
            if (tmx[2][3] + mx[2][3] < 0)
               tmx[2][3] = -mx[2][3];
         }
      
         else {
            mouse_state.button_state |= BUTTONRIGHT;
            copyarray2(mouse_state.old_state, mouse_state.new_state);
         }
      
      }
   
      else
         if (mouse_state.button_state & BUTTONRIGHT) {
            user_translate(cparam, object_tree, tmx, win);
            mx[0][3] += tmx[0][3];
            mx[1][3] += tmx[1][3];
            mx[2][3] += tmx[2][3];
            init_mx(tmx);
            mouse_state.button_state &= ~BUTTONRIGHT;
         }

   }

   process_io(mx, tmx, cparam, flags);
}


/* **************************************************
************************************************** */
void render3d(vector4f *frustum, int framecount, BLTwindow *win) {

   atom_list_type *atr;
   
   for (atr = (atom_list_type *)complex->animation_manager.head; atr; atr = (atom_list_type *)atr->next)
      atr->htree->render_object(&proc, NULL, frustum, FRUSTUM_CLIP_ALL);

   // render frame
   proc.spawn(framecount, &win->mapbuffer);
}


/* **************************************************
************************************************** */
void render2d(BLTwindow *win, int flags) {

   float rowsize = (float)(win->mapbuffer.maxy/24.0);
   char msg_quit[]  = "Exit? (Y/N)";
   vector4uc g = { 0, 255, 0, 0 };

   complex->gfx->gfx2DMode(win->mapbuffer.maxx, win->mapbuffer.maxy);

   if (complex->font && (flags & FLAG_QUERYQUIT)) {
      complex->font->set_color((unsigned char)255, (unsigned char)0, (unsigned char)255);
      complex->font->set_scale((float)(rowsize*0.85));
      complex->font->print((win->mapbuffer.maxx - complex->font->pixel_length((unsigned char *)msg_quit))>>1, (int)((win->mapbuffer.maxy-rowsize)*0.5), (unsigned char *)msg_quit, &win->mapbuffer);
   }

   if (mouse_state.flag & FLAG_MOUSE_DRAW) {
      glBegin(GL_LINES);
         glColor3ubv((GLubyte *)g);
         glVertex2f(mouse_state.new_state[0]-5, mouse_state.new_state[1]);
         glVertex2f(mouse_state.new_state[0]+5, mouse_state.new_state[1]);
         glVertex2f(mouse_state.new_state[0], mouse_state.new_state[1]-5);
         glVertex2f(mouse_state.new_state[0], mouse_state.new_state[1]+5);
      glEnd();
   }

}


/* **************************************************
************************************************** */
void main_loop(BLTwindow *win) {

   static int snapcount = 0;

   vector4f mx[4], tmx[4], current_mx[4];
   unsigned int flags = FLAG_TRANSPARENT;
   int framecount = 0;
   TIME_STRUCT t;
   vector4f frustum[FRUSTUM_COUNT];
   camera *cparam;
   time_t oldtime, newtime;
   float duration;
   atom_list_type *atr;
   bmp coder;
   char buffer[MAXSTRLEN];
   
#ifdef WIN32
   SetWindowLong(((WINDOW_TYPE *)win)->mwindow, GWL_WNDPROC, (long)game_polling);
   SetWindowPos(((WINDOW_TYPE *)win)->mwindow, NULL, 0,0,0,0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOZORDER | SWP_FRAMECHANGED);
#endif

   init_mx(mx);
   init_mx(tmx);
   init_mx(current_mx);
   mouse_state.button_state = 0;
   
   complex->set_font(FILENAME_FONT.string, FONTTYPE);
   
   // first frame :)
   proc.engine_init();

   // find camera
   cparam = (camera *)camera_tree->find_ob(NULL, OBJECT_CAMERA);

   if (!cparam) {
      printf("No camera...  Aborting...\n");
      return;
   }
   
   cparam->set_dim(win->mapbuffer.maxx, win->mapbuffer.maxy);

   // process objects
   for (atr = (atom_list_type *)complex->animation_manager.head; atr; atr = (atom_list_type *)atr->next) {
      atr->htree->begin(&complex->animation_manager, NULL, mx);
      atr->htree->update(&complex->animation_manager, NULL);
   }
 
   cparam->calc_clip_plane();
   cparam->query_frustum(frustum);

   calc_initial_mx(cparam, object_tree, current_mx);

   time(&oldtime);
   
   // rest of frames ...
   while (1) {

      // process user io
      process_events(current_mx, tmx, cparam, &flags, win);

      if (flags & FLAG_QUIT)
         break;
	 
      copymx4x4(mx, current_mx);
      matmatmultv(tmx, mx);

      mx[0][3] = tmx[0][3] + current_mx[0][3];
      mx[1][3] = tmx[1][3] + current_mx[1][3];
      mx[2][3] = tmx[2][3] + current_mx[2][3];

      while (!complex->timer.tick());

      // process objects
      for (atr = (atom_list_type *)complex->animation_manager.head; atr; atr = (atom_list_type *)atr->next)
         if (atr->htree != camera_tree) {
            atr->htree->whereami(&complex->animation_manager, NULL, mx);
            atr->htree->update(&complex->animation_manager, NULL);
         }

      // send objects to rendering buffer
      proc.engine_init();

      render3d(frustum, framecount, win);
      other3d(mx, frustum, framecount, win);

      render2d(win, flags);
      other2d(win, flags);

      if (complex->userinput->translate_stroke(KEYSTROKE_SNAPSHOT)) {
         sprintf(buffer, "imag%04d.bmp",  snapcount);
         snapcount++;
         complex->gfx->gfxSnapshot(buffer, &win->mapbuffer,  &coder);
      }

      GETTIME(&t);
      global_resource_manager->update((float)TIME2FLOAT(t));
      complex->gfx->gfxBitblt(win, &win->mapbuffer);

      framecount++;
   }

   if (logfile) {
      time(&newtime);
      duration = (float)(newtime - oldtime);

      fprintf(logfile, "Framecount %d\n", framecount);
      fprintf(logfile, "Framerate Average %4.2f\n", framecount/duration);
   }

}


/* **************************************************
************************************************** */
int main_body(int argc, char **argv) {

   WINDOW_TYPE *win;
   int winx, winy;

   int attribs[] = {
      OGLFLAG_BPP, 32,
      OGLFLAG_DOUBLEBUFFER,
      OGLFLAG_DEPTH_SIZE, 1,
      OGLFLAG_RGBA,
      OGLFLAG_MAPBUFFER,
      OGLFLAG_NULL
   };

   atexit(shutdown);

   // config system
   CONTROLLER = CONTROLKEYBOARD;
   RENDER_MODE = RENDERER_OPENGL;
   FONTTYPE = FONT_RASTER;
   FILENAME_FONT.stringcpy("config/200.fnt");
   FILENAME_PRESPG[0] = 0;

   parseinput(argc, argv);
   gfx_get_screen_dim(&winx, &winy);

   win = new WINDOW_TYPE;

   // init screen
   if (!win->VIRTUALreset(argc, argv, winx, winy, "Pixcon", (void *)attribs))
      return 0;

   // init the game engine
   startup(win, new game_manager);

   // start up the game
   main_loop(win);

   shutdown();
   win->VIRTUALcancel();
   delete win;

   return 1;
}

