#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "oscript.hpp"
#include "ogreobj.hpp"
#include "o-bicone.hpp"
#include "o-block.hpp"
#include "o-checkr.hpp"
#include "o-cylndr.hpp"
#include "o-dome.hpp"
#include "o-fuzzy.hpp"
#include "o-gear.hpp"
#include "o-maze.hpp"
#include "o-panel.hpp"
#include "o-poly.hpp"
#include "o-post.hpp"
#include "o-pyramd.hpp"
#include "o-rect.hpp"
#include "o-revolv.hpp"
#include "o-soccer.hpp"
#include "o-sphere.hpp"
#include "o-torus.hpp"
#include "o-triang.hpp"
#include "o-twist.hpp"
#include "o-wedge.hpp"


#define OG_INVALID       0
#define OG_DEFINE        1
#define OG_VAR           2
#define OG_SET           3
#define OG_ADD           4
#define OG_SUB           5
#define OG_MUL           6
#define OG_DIV           7
#define OG_MOD           8
#define OG_RANDOM        9
#define OG_OBJECT       10
#define OG_FREE         11
#define OG_LOAD         12
#define OG_SAVENFF10    13
#define OG_SAVENFF17    14
#define OG_SAVEOGR      15
#define OG_SAVEPLG      16
#define OG_SAVEOBT      17
#define OG_SAVEFLOB     18
#define OG_CALL         19
#define OG_MANIP        20
#define OG_DJNZ         21
#define OG_JNZ          22
#define OG_JZ           23
#define OG_JUMP         24
#define OG_NEG          25
#define OG_COLORDEF     26

#define OG_SCALE       101
#define OG_TRANS       102
#define OG_ROTATE      103
#define OG_ROTX        104
#define OG_ROTY        105
#define OG_ROTZ        106
#define OG_COPY        107
#define OG_INVERT      108
#define OG_MERGE       109
#define OG_PAINT       110
#define OG_CENTER      111
#define OG_PAINTPOLY   112
#define OG_DOUBLESIDE  113
#define OG_FLOOR       114


#define OG_BICONE      201
#define OG_BLOCK       202
#define OG_CHECKER     203
#define OG_CYLINDER    204
#define OG_DOME        205
#define OG_FUZZY       206
#define OG_GEAR        207
#define OG_PANEL       208
#define OG_POSTFIELD   209
#define OG_POLYGON     210
#define OG_PYRAMID     211
#define OG_RECTANGLE   212
#define OG_REVOLUTION  213
#define OG_SOCCER      214
#define OG_SPHERE      215
#define OG_TORUS       216
#define OG_TRIANGLE    217
#define OG_TWIST       218
#define OG_WEDGE       219


#ifdef MFX
// avoid warnings about conversions to integer
static inline int my_floor(double x)
{
    return (int) (floor(x));
}
#undef floor
#define floor   my_floor

static char *my_strupr(char *p)
{
    unsigned char *s = (unsigned char *) p;
    for ( ; *s; s++)
        if (islower(*s))
	    *s = toupper(*s);
    return p;
}
#undef strupr
#define strupr  my_strupr

#ifdef __linux__
#undef stricmp
#define stricmp strcasecmp
#endif
#endif


int
OgreScript::line_count(char *fname)
{
    char dummy_string[256];
    int line_count;
    FILE *fp;

    line_count = -1;

    fp = fopen(fname, "rt");
    if (fp != NULL) {
        line_count = 1;
        while (fgets(dummy_string, 256, fp)) {
            line_count++;
        }

        fclose(fp);
    }

    return line_count;
}


int
OgreScript::load_script(char *fname)
{
    FILE *fp;
    int i, len;
    char tempstr[256];

    user_lines = line_count(fname);

    if (user_lines == -1) {
        return 0;
    } else {
        fp = fopen(fname, "rt");

        user_script = new char *[user_lines];

        for (i=1; i < user_lines; i++) {
#ifdef MFX
            char *p;

            tempstr[0] = 0;
            fgets(tempstr, sizeof(tempstr), fp);
            tempstr[sizeof(tempstr)-1] = 0;
            // remove trailing whitespace
            p = tempstr + strlen(tempstr);
            if (p > tempstr)
                while (--p > tempstr && isspace(((unsigned char) *p)))
                    *p = 0;
            // remove leading whitespace
            p = tempstr;
            while (*p && isspace(((unsigned char) *p)))
                p++;
            len = strlen(p) + 1;
            user_script[i] = new char[len];
            strcpy(user_script[i], p);
#else
            fgets(tempstr, 256, fp);
            len = strlen(tempstr) + 1;
            user_script[i] = new char[len];
            user_script[i][0] = '\0';
            strcat(user_script[i], tempstr);
            if (user_script[i][len-2] == '\n') {
                user_script[i][len-2] = '\0';
            }
#endif
        }

        fclose(fp);

        return 1;
    }
}


void
OgreScript::unload_script(void)
{
    int i;

    for (i=0; i < max_color_decl; i++) {
        if (SCRIPT_colors[i].varname != NULL) {
            delete SCRIPT_colors[i].varname;
            SCRIPT_colors[i].varname = NULL;
        }
    }

    for (i=0; i < max_number_decl; i++) {
        if (SCRIPT_numbers[i].varname != NULL) {
            delete SCRIPT_numbers[i].varname;
            SCRIPT_numbers[i].varname = NULL;
        }
    }

    for (i=0; i < max_object_decl; i++) {
        if (SCRIPT_objects[i].varname != NULL) {
            delete SCRIPT_objects[i].varname;
            SCRIPT_objects[i].varname = NULL;
        }

        if (SCRIPT_objects[i].obj != NULL) {
            delete SCRIPT_objects[i].obj;
            SCRIPT_objects[i].obj = NULL;
        }
    }

    if (user_lines != -1) {
#ifdef MFX
        for (i=1; i < user_lines; i++) {
#else
        for (i=0; i < user_lines; i++) {
#endif
            delete user_script[i];
        }

        delete user_script;
        user_script = NULL;
        user_lines = -1;
    }
}


float
OgreScript::get_value(char *varname, int *ok)
{
    float retval;
    int found;

    found = find_number(varname);

    *ok = 1;

    if (found == -1) {
        retval = atof(varname);

        if ((retval == 0.0) && (varname[0] != '0')) {
            // This means that the string is not a variable or a number
            *ok = 0;
            err_header();
            undecl_number_err(varname);
        }
    } else {
        retval = SCRIPT_numbers[found].value;
    }

    return retval;
}


int
OgreScript::find_color(char *varname)
{
    int i, found;

    found = -1;

    for (i=0; i < max_color_decl; i++) {
        if (!stricmp(varname, SCRIPT_colors[i].varname)) {
            // Kick out of the loop
            found = i;
            i = max_color_decl;
        }
    }

    return (found);
}


int
OgreScript::find_number(char *varname)
{
    int i, found;

    found = -1;

    for (i=0; i < max_number_decl; i++) {
        if (!stricmp(varname, SCRIPT_numbers[i].varname)) {
            // Kick out of the loop
            found = i;
            i = max_number_decl;
        }
    }

    return (found);
}


int
OgreScript::find_object(char *varname)
{
    int i, found;

    found = -1;

    for (i=0; i < max_object_decl; i++) {
        if (!stricmp(varname, SCRIPT_objects[i].varname)) {
            // Kick out of the loop
            found = i;
            i = max_object_decl;
        }
    }

    return (found);
}


OgreScript::OgreScript(void)
{
    // Do nothing
    err_flag = 0;
}


OgreScript::~OgreScript(void)
{
    // Do nothing
}


int
OgreScript::Execute(char *fname)
{
    char local_fname[129];
    int ok, jump;

#ifdef MFX
    srand((unsigned) (time(NULL)));
#else
    randomize();
#endif

    local_fname[0] = '\0';
    strcat(local_fname, fname);

    if (strchr(local_fname, '.') == NULL) {
        strcat(local_fname, ".osf");
    }

    ok = load_script(local_fname);

    if (ok) {
        max_color_decl  = 0;
        max_number_decl = 0;
        max_object_decl = 0;

        current_line = 1;

        printf("Executing script: %s\n", local_fname);

        while ((current_line < user_lines) && (current_line >= 1)) {
            jump = do_line(current_line);

            current_line += jump;
        }

        unload_script();

        return 0;
    } else {
        printf("Cannot open script: %s\n", local_fname);
        return -1;
    }
}


int
OgreScript::get_opcode(char *opname)
{
    int retval;

    if (!stricmp("DEFINE", opname)) {
        retval = OG_DEFINE;
    } else if (!stricmp("VAR", opname)) {
        retval = OG_VAR;
    } else if (!stricmp("COLORDEF", opname)) {
        retval = OG_COLORDEF;
    } else if (!stricmp("SET", opname)) {
        retval = OG_SET;
    } else if (!stricmp("ADD", opname)) {
        retval = OG_ADD;
    } else if (!stricmp("INC", opname)) {
        retval = OG_ADD;
    } else if (!stricmp("SUB", opname)) {
        retval = OG_SUB;
    } else if (!stricmp("DEC", opname)) {
        retval = OG_SUB;
    } else if (!stricmp("MUL", opname)) {
        retval = OG_MUL;
    } else if (!stricmp("DIV", opname)) {
        retval = OG_DIV;
    } else if (!stricmp("MOD", opname)) {
        retval = OG_MOD;
    } else if (!stricmp("NEG", opname)) {
        retval = OG_NEG;
    } else if (!stricmp("RANDOM", opname)) {
        retval = OG_RANDOM;
    } else if (!stricmp("DJNZ", opname)) {
        retval = OG_DJNZ;
    } else if (!stricmp("JZ", opname)) {
        retval = OG_JZ;
    } else if (!stricmp("JNZ", opname)) {
        retval = OG_JNZ;
    } else if (!stricmp("JUMP", opname)) {
        retval = OG_JUMP;
    } else if (!stricmp("OBJECT", opname)) {
        retval = OG_OBJECT;
    } else if (!stricmp("FREE", opname)) {
        retval = OG_FREE;
    } else if (!stricmp("LOAD", opname)) {
        retval = OG_LOAD;
    } else if (!stricmp("SAVENFF10", opname)) {
        retval = OG_SAVENFF10;
    } else if (!stricmp("SAVENFF17", opname)) {
        retval = OG_SAVENFF17;
    } else if (!stricmp("SAVEOGR", opname)) {
        retval = OG_SAVEOGR;
    } else if (!stricmp("SAVEPLG", opname)) {
        retval = OG_SAVEPLG;
    } else if (!stricmp("SAVEOBT", opname)) {
        retval = OG_SAVEOBT;
    } else if (!stricmp("SAVEFLOB", opname)) {
        retval = OG_SAVEFLOB;
    } else if (!stricmp("CALL", opname)) {
        retval = OG_CALL;
    } else if (find_object(opname) != -1) {
        // It must be an object manipulation
        retval = OG_MANIP;
    } else {
        // Bad opcode
        retval = OG_INVALID;
    }

    return retval;
}


int
OgreScript::get_objcmd(char *opname)
{
    int retval;

    if (!stricmp("SCALE", opname)) {
        retval = OG_SCALE;
    } else if (!stricmp("TRANS", opname)) {
        retval = OG_TRANS;
    } else if (!stricmp("ROTATE", opname)) {
        retval = OG_ROTATE;
    } else if (!stricmp("ROTX", opname)) {
        retval = OG_ROTX;
    } else if (!stricmp("ROTY", opname)) {
        retval = OG_ROTY;
    } else if (!stricmp("ROTZ", opname)) {
        retval = OG_ROTZ;
    } else if (!stricmp("COPY", opname)) {
        retval = OG_COPY;
    } else if (!stricmp("INVERT", opname)) {
        retval = OG_INVERT;
    } else if (!stricmp("DOUBLESIDE", opname)) {
        retval = OG_DOUBLESIDE;
    } else if (!stricmp("MERGE", opname)) {
        retval = OG_MERGE;
    } else if (!stricmp("PAINT", opname)) {
        retval = OG_PAINT;
    } else if (!stricmp("CENTER", opname)) {
        retval = OG_CENTER;
    } else if (!stricmp("FLOOR", opname)) {
        retval = OG_FLOOR;
    } else if (!stricmp("PAINTPOLY", opname)) {
        retval = OG_PAINTPOLY;
    } else if (!stricmp("BICONE", opname)) {
        retval = OG_BICONE;
    } else if (!stricmp("BLOCK", opname)) {
        retval = OG_BLOCK;
    } else if (!stricmp("CHECKER", opname)) {
        retval = OG_CHECKER;
    } else if (!stricmp("CONE", opname)) {
        retval = OG_PYRAMID;
    } else if (!stricmp("CYLINDER", opname)) {
        retval = OG_CYLINDER;
    } else if (!stricmp("DOME", opname)) {
        retval = OG_DOME;
    } else if (!stricmp("FUZZY", opname)) {
        retval = OG_FUZZY;
    } else if (!stricmp("GEAR", opname)) {
        retval = OG_GEAR;
    } else if (!stricmp("PANEL", opname)) {
        retval = OG_PANEL;
    } else if (!stricmp("POSTFIELD", opname)) {
        retval = OG_POSTFIELD;
    } else if (!stricmp("POLYGON", opname)) {
        retval = OG_POLYGON;
    } else if (!stricmp("PYRAMID", opname)) {
        retval = OG_PYRAMID;
    } else if (!stricmp("RECTANGLE", opname)) {
        retval = OG_RECTANGLE;
    } else if (!stricmp("REVOLUTION", opname)) {
        retval = OG_REVOLUTION;
    } else if (!stricmp("SOCCER", opname)) {
        retval = OG_SOCCER;
    } else if (!stricmp("SPHERE", opname)) {
        retval = OG_SPHERE;
    } else if (!stricmp("TORUS", opname)) {
        retval = OG_TORUS;
    } else if (!stricmp("TRIANGLE", opname)) {
        retval = OG_TRIANGLE;
    } else if (!stricmp("TWIST", opname)) {
        retval = OG_TWIST;
    } else if (!stricmp("WEDGE", opname)) {
        retval = OG_WEDGE;
    } else {
        // Bad opcode
        retval = OG_INVALID;
    }

    return retval;
}


int
OgreScript::creation_op(int opcode)
{
    if ((opcode >= 201) && (opcode <= 299)) {
        return 1;
    } else {
        return 0;
    }
}


void
OgreScript::err_header(void)
{
    err_flag = 1;
    printf("Death and destruction on line %d\n", current_line);
}


int
OgreScript::get_errflag(void)
{
    return err_flag;
}


void
OgreScript::undecl_color_err(char *varname)
{
    printf("Undeclared Color: %s\n", varname);
}


void
OgreScript::undecl_number_err(char *varname)
{
    printf("Undeclared Variable: %s\n", varname);
}


void
OgreScript::undecl_object_err(char *varname)
{
    printf("Undeclared Object: %s\n", varname);
}


void
OgreScript::missing_file_err(char *fname)
{
    printf("Cannot open file: %s\n", fname);
}


#ifdef MFX
#define color_decl      _color_decl
#endif
int
OgreScript::color_decl(char *varname)
{
    int index, retval;

    strupr(varname);

    retval = -1;

    if (max_color_decl == 256) {
        err_header();
        printf("Too Many Colors (Max = 256): %s\n", varname);
    } else {
        index = find_color(varname);
        if (index != -1) {
            err_header();
            printf("Redeclaration of Color: %s\n", varname);
        } else {
            SCRIPT_colors[max_color_decl].varname = new
                char[strlen(varname)+1];
            SCRIPT_colors[max_color_decl].red   = 0;
            SCRIPT_colors[max_color_decl].green = 0;
            SCRIPT_colors[max_color_decl].blue  = 0;
            SCRIPT_colors[max_color_decl].varname[0] = '\0';
            strcat(SCRIPT_colors[max_color_decl].varname,varname);
            retval = max_color_decl;
            max_color_decl++;
        }
    }

    return retval;
}


#ifdef MFX
#define number_decl     _number_decl
#endif
int
OgreScript::number_decl(char *varname)
{
    int index, retval;

    strupr(varname);

    retval = -1;

    if (max_number_decl == 256) {
        err_header();
        printf("Too Many Variables (Max = 256): %s\n", varname);
    } else {
        index = find_number(varname);
        if (index != -1) {
            err_header();
            printf("Redeclaration of Variable: %s\n", varname);
        } else {
            SCRIPT_numbers[max_number_decl].varname = new
                char[strlen(varname)+1];
            SCRIPT_numbers[max_number_decl].value = 0.0;
            SCRIPT_numbers[max_number_decl].varname[0] = '\0';
            strcat(SCRIPT_numbers[max_number_decl].varname,varname);
            retval = max_number_decl;
            max_number_decl++;
        }
    }

    return retval;
}


#ifdef MFX
#define object_decl     _object_decl
#endif
int
OgreScript::object_decl(char *varname)
{
    int index, retval;

    if (varname[0] == '\0') {
        return -1;
    }

    strupr(varname);

    retval = -1;

    if (max_object_decl == 256) {
        err_header();
        printf("Too Many Objects (Max = 256): %s\n", varname);
    } else {
        index = find_object(varname);

        if (index != -1) {
            err_header();
            printf("Redeclaration of Object: %s\n", varname);
        } else {
            SCRIPT_objects[max_object_decl].varname = new
                char[strlen(varname)+1];
            SCRIPT_objects[max_object_decl].obj = NULL;
            SCRIPT_objects[max_object_decl].varname[0] = '\0';
            strcat(SCRIPT_objects[max_object_decl].varname, varname);
            retval = max_object_decl;
            max_object_decl++;
        }
    }

    return retval;
}


int
OgreScript::do_line(int line_num)
{
    FILE *fp;
    OgreObject *tempobj;
    OgreScript *tempscript;
    char *bufptr;
    char tempchar;
    float *data;
    int i, retval, count, source, source2, dest, jump, ok;
    int opcode, operand1, operand2, operand3, objcmd;
    int intval1, intval2, intval3, intval4, intval5, intval6, intval7,
        intval8, intval9, intval10, intval11, intval12;
    float value1, value2, value3, value4, value5, value6, value7,
          value8, value9, value10, value11, value12;
    char opcode_string[32],
         op_string1[32], op_string2[32], op_string3[32], op_string4[32],
         op_string5[32], op_string6[32], op_string7[32], op_string8[32],
         op_string9[32], op_string10[32], op_string11[32];

    opcode_string[0] = '\0';
    op_string1[0] =  op_string2[0] =  op_string3[0] =  op_string4[0] = '\0';
    op_string5[0] =  op_string6[0] =  op_string7[0] =  op_string8[0] = '\0';
    op_string9[0] = op_string10[0] = op_string11[0] = '\0';


    retval = 1;

    bufptr = user_script[line_num];

    sscanf(bufptr, "%s%n", opcode_string, &count);

    if ((*bufptr == '#') || (*bufptr == '\0') || (*bufptr == '\n')) {
        return 1;
    }

    opcode = get_opcode(opcode_string);

    switch (opcode) {
        case OG_DEFINE:
            bufptr += count;
            sscanf(bufptr, "%s %s", op_string1, op_string2);
            strupr(op_string2);

            operand1 = number_decl(op_string1);
            value2 = get_value(op_string2, &ok);

            if ((ok) && (operand1 != -1)) {
                SCRIPT_numbers[operand1].value = value2;
            }
            break;

        case OG_VAR:
            bufptr += count;
            ok = sscanf(bufptr, "%s%n", op_string1, &count);
            bufptr += count;

            while (ok == 1) {
                number_decl(op_string1);
                op_string1[0] = '\0';
                ok = sscanf(bufptr, "%s%n", op_string1, &count);
                bufptr += count;
            }
            break;

        case OG_COLORDEF:
            bufptr += count;
            sscanf(bufptr, "%s %s %s %s", op_string1, op_string2,
                                          op_string3, op_string4);
            strupr(op_string2);

            operand1 = color_decl(op_string1);
            intval2 = floor(get_value(op_string2, &ok));
            intval3 = floor(get_value(op_string3, &ok));
            intval4 = floor(get_value(op_string4, &ok));

            if ((ok) && (operand1 != -1)) {
                SCRIPT_colors[operand1].red   = intval2;
                SCRIPT_colors[operand1].green = intval3;
                SCRIPT_colors[operand1].blue  = intval4;
            }
            break;

        case OG_SET:
            bufptr += count;
            sscanf(bufptr, "%s %s", op_string1, op_string2);
            strupr(op_string1);
            strupr(op_string2);

            operand1 = find_number(op_string1);

            value2 = get_value(op_string2, &ok);

            if (operand1 == -1) {
                err_header();
                undecl_number_err(op_string1);
            } else {
                if (ok) {
                    SCRIPT_numbers[operand1].value = value2;
                }
            }
            break;

        case OG_ADD:
            bufptr += count;
            sscanf(bufptr, "%s %s", op_string1, op_string2);
            strupr(op_string1);
            strupr(op_string2);

            operand1 = find_number(op_string1);

            value2 = get_value(op_string2, &ok);

            if (operand1 == -1) {
                err_header();
                undecl_number_err(op_string1);
            } else {
                if (ok) {
                    SCRIPT_numbers[operand1].value += value2;
                }
            }
            break;

        case OG_SUB:
            bufptr += count;
            sscanf(bufptr, "%s %s", op_string1, op_string2);
            strupr(op_string1);
            strupr(op_string2);

            operand1 = find_number(op_string1);

            value2 = get_value(op_string2, &ok);

            if (operand1 == -1) {
                err_header();
                undecl_number_err(op_string1);
            } else {
                if (ok) {
                    SCRIPT_numbers[operand1].value -= value2;
                }
            }
            break;

        case OG_MUL:
            bufptr += count;
            sscanf(bufptr, "%s %s", op_string1, op_string2);
            strupr(op_string1);
            strupr(op_string2);

            operand1 = find_number(op_string1);

            value2 = get_value(op_string2, &ok);

            if (operand1 == -1) {
                err_header();
                undecl_number_err(op_string1);
            } else {
                if (ok) {
                    SCRIPT_numbers[operand1].value *= value2;
                }
            }
            break;

        case OG_DIV:
            bufptr += count;
            sscanf(bufptr, "%s %s", op_string1, op_string2);
            strupr(op_string1);
            strupr(op_string2);

            operand1 = find_number(op_string1);

            value2 = get_value(op_string2, &ok);

            if (operand1 == -1) {
                err_header();
                undecl_number_err(op_string1);
            } else {
                if (ok) {
                    if (value2 == 0.0) {
                        err_header();
                        printf("DIV by zero is not kosher.\n");
                    } else {
                        SCRIPT_numbers[operand1].value /= value2;
                    }
                }
            }
            break;

        case OG_MOD:
            bufptr += count;
            sscanf(bufptr, "%s %s", op_string1, op_string2);
            strupr(op_string1);
            strupr(op_string2);

            operand1 = find_number(op_string1);

            value2 = get_value(op_string2, &ok);

            if (operand1 == -1) {
                err_header();
                undecl_number_err(op_string1);
            } else {
                if (ok) {
                    if (value2 == 0.0) {
                        err_header();
                        printf("MOD by zero is not kosher.\n");
                    } else {
                        value1 = get_value(op_string1, &ok);

                        intval1 = floor(value1);
                        intval2 = floor(value2);

                        SCRIPT_numbers[operand1].value = (intval1 % intval2);
                    }
                }
            }
            break;

        case OG_NEG:
            bufptr += count;
            sscanf(bufptr, "%s", op_string1);
            strupr(op_string1);

            operand1 = find_number(op_string1);

            if (operand1 == -1) {
                err_header();
                undecl_number_err(op_string1);
            } else {
                SCRIPT_numbers[operand1].value *= -1;
            }
            break;

        case OG_RANDOM:
            bufptr += count;
            sscanf(bufptr, "%s %s", op_string1, op_string2);
            strupr(op_string1);
            strupr(op_string2);

            operand1 = find_number(op_string1);

            intval2 = floor(get_value(op_string2, &ok));

            if (operand1 == -1) {
                err_header();
                undecl_number_err(op_string1);
            } else {
                if (ok) {
#ifdef MFX
                    if (intval2 < 1) {
                        err_header();
                        printf("RANDOM %d is not kosher.\n", intval2);
                    } else {
                        SCRIPT_numbers[operand1].value = rand() % intval2;
                    }
#else
                    SCRIPT_numbers[operand1].value = random(intval2);
#endif
                }
            }
            break;

        case OG_DJNZ:
            /* Decrement, Jump if Not Zero */
            bufptr += count;
            sscanf(bufptr, "%s %d", op_string1, &jump);
            strupr(op_string1);

            operand1 = find_number(op_string1);

            if (operand1 == -1) {
                err_header();
                undecl_number_err(op_string1);
            } else {
                SCRIPT_numbers[operand1].value--;

                if (SCRIPT_numbers[operand1].value != 0.0) {
                    retval = jump;
                }
            }
            break;

        case OG_JNZ:
            /* Jump if Not Zero */
            bufptr += count;
            sscanf(bufptr, "%s %d", op_string1, &jump);
            strupr(op_string1);

            operand1 = find_number(op_string1);

            if (operand1 == -1) {
                err_header();
                undecl_number_err(op_string1);
            } else {
                if (SCRIPT_numbers[operand1].value != 0.0) {
                    retval = jump;
                }
            }
            break;

        case OG_JZ:
            /* Jump if Zero */
            bufptr += count;
            sscanf(bufptr, "%s %d", op_string1, &jump);
            strupr(op_string1);

            operand1 = find_number(op_string1);

            if (operand1 == -1) {
                err_header();
                undecl_number_err(op_string1);
            } else {
                if (SCRIPT_numbers[operand1].value == 0.0) {
                    retval = jump;
                }
            }
            break;

        case OG_JUMP:
            /* Jump Unconditionally */
            bufptr += count;
            sscanf(bufptr, "%d", &jump);

            retval = jump;
            break;

        case OG_OBJECT:
            bufptr += count;
            ok = sscanf(bufptr, "%s%n", op_string1, &count);
            bufptr += count;

            while (ok == 1) {
                object_decl(op_string1);
                op_string1[0] = '\0';
                ok = sscanf(bufptr, "%s%n", op_string1, &count);
                bufptr += count;
            }
            break;

        case OG_FREE:
            bufptr += count;
            sscanf(bufptr, "%s", op_string1);
            strupr(op_string1);

            source = find_object(op_string1);

            if (source == -1) {
                err_header();
                undecl_object_err(op_string1);
            } else {
                if (SCRIPT_objects[source].obj == NULL) {
                    // Don't need to free NULL Object
                } else {
                    delete SCRIPT_objects[source].obj;
                    SCRIPT_objects[source].obj = NULL;
                }
            }
            break;

        case OG_LOAD:
            bufptr += count;
            sscanf(bufptr, "%s %s", op_string1, op_string2);
            strupr(op_string1);
#ifndef MFX
            strupr(op_string2);
#endif

            dest = find_object(op_string1);

            if (dest == -1) {
                err_header();
                undecl_object_err(op_string1);
            } else {
                if (SCRIPT_objects[dest].obj != NULL) {
                    delete SCRIPT_objects[dest].obj;
                }
                SCRIPT_objects[dest].obj = new OgreObject(op_string2);
            }
            break;

        case OG_SAVENFF10:
            bufptr += count;
            sscanf(bufptr, "%s %s", op_string1, op_string2);
            strupr(op_string1);
#ifndef MFX
            strupr(op_string2);
#endif

            source = find_object(op_string1);

            if (source == -1) {
                err_header();
                undecl_object_err(op_string1);
            } else {
                if (SCRIPT_objects[source].obj == NULL) {
                    err_header();
                    printf("Cannot save NULL object: %s\n", op_string1);
                } else {
                    SCRIPT_objects[source].obj->saveNFF10(op_string2);
                }
            }
            break;

        case OG_SAVENFF17:
            bufptr += count;
            sscanf(bufptr, "%s %s", op_string1, op_string2);
            strupr(op_string1);
#ifndef MFX
            strupr(op_string2);
#endif

            source = find_object(op_string1);

            if (source == -1) {
                err_header();
                undecl_object_err(op_string1);
            } else {
                if (SCRIPT_objects[source].obj == NULL) {
                    err_header();
                    printf("Cannot save NULL object: %s\n", op_string1);
                } else {
                    SCRIPT_objects[source].obj->saveNFF17(op_string2);
                }
            }
            break;

        case OG_SAVEOGR:
            bufptr += count;
            sscanf(bufptr, "%s %s", op_string1, op_string2);
            strupr(op_string1);
#ifndef MFX
            strupr(op_string2);
#endif

            source = find_object(op_string1);

            if (source == -1) {
                err_header();
                undecl_object_err(op_string1);
            } else {
                if (SCRIPT_objects[source].obj == NULL) {
                    err_header();
                    printf("Cannot save NULL object: %s\n", op_string1);
                } else {
                    SCRIPT_objects[source].obj->saveOGR(op_string2);
                }
            }
            break;

        case OG_SAVEPLG:
            bufptr += count;
            sscanf(bufptr, "%s %s", op_string1, op_string2);
            strupr(op_string1);
#ifndef MFX
            strupr(op_string2);
#endif

            source = find_object(op_string1);

            if (source == -1) {
                err_header();
                undecl_object_err(op_string1);
            } else {
                if (SCRIPT_objects[source].obj == NULL) {
                    err_header();
                    printf("Cannot save NULL object: %s\n", op_string1);
                } else {
                    SCRIPT_objects[source].obj->savePLG(op_string2);
                }
            }
            break;

        case OG_SAVEOBT:
            bufptr += count;
            sscanf(bufptr, "%s %s", op_string1, op_string2);
            strupr(op_string1);
#ifndef MFX
            strupr(op_string2);
#endif

            source = find_object(op_string1);

            if (source == -1) {
                err_header();
                undecl_object_err(op_string1);
            } else {
                if (SCRIPT_objects[source].obj == NULL) {
                    err_header();
                    printf("Cannot save NULL object: %s\n", op_string1);
                } else {
                    SCRIPT_objects[source].obj->saveOBT(op_string2);
                }
            }
            break;

        case OG_SAVEFLOB:
            bufptr += count;
            sscanf(bufptr, "%s %s", op_string1, op_string2);
            strupr(op_string1);
#ifndef MFX
            strupr(op_string2);
#endif

            source = find_object(op_string1);

            if (source == -1) {
                err_header();
                undecl_object_err(op_string1);
            } else {
                if (SCRIPT_objects[source].obj == NULL) {
                    err_header();
                    printf("Cannot save NULL object: %s\n", op_string1);
                } else {
                    SCRIPT_objects[source].obj->saveFLOB(op_string2);
                }
            }
            break;

        case OG_CALL:
            bufptr += count;
            sscanf(bufptr, "%s", op_string1);

            tempscript = new OgreScript();
            tempscript->Execute(op_string1);
            delete tempscript;
            break;

        case OG_MANIP:
            dest = find_object(opcode_string);

            if (dest == -1) {
                err_header();
                undecl_object_err(opcode_string);
            } else {
                bufptr += count;
                sscanf(bufptr, " = %s%n", opcode_string, &count);
                bufptr += count;

                objcmd = get_objcmd(opcode_string);

                if (creation_op(objcmd)) {
                    if (SCRIPT_objects[dest].obj != NULL) {
                        delete SCRIPT_objects[dest].obj;
                        SCRIPT_objects[dest].obj = NULL;
                    }

                    switch (objcmd) {
                        case OG_BICONE:
                            sscanf(bufptr, "%s %s %s %s",
                                op_string1, op_string2, op_string3, op_string4);

                            value1  = get_value(op_string1, &ok);
                            value2  = get_value(op_string2, &ok);
                            value3  = get_value(op_string3, &ok);
                            intval4 = floor(get_value(op_string4, &ok));

                            SCRIPT_objects[dest].obj = new
                                OgreBiCone(value1, value2, value3, intval4);
                            break;

                        case OG_BLOCK:
                            sscanf(bufptr, "%s %s %s",
                                op_string1, op_string2, op_string3);

                            value1 = get_value(op_string1, &ok);
                            value2 = get_value(op_string2, &ok);
                            value3 = get_value(op_string3, &ok);

                            SCRIPT_objects[dest].obj = new
                                OgreBlock(value1, value2, value3);
                            break;

                        case OG_CHECKER:
                            sscanf(bufptr, "%s %s %s %s %s %s %s %s %s %s %s",
                                   op_string1, op_string2, op_string3,
                                   op_string4, op_string5, op_string6,
                                   op_string7, op_string8, op_string9,
                                   op_string10, op_string11);

                            intval1  = floor(get_value(op_string1, &ok));
                            intval2  = floor(get_value(op_string2, &ok));
                            value3   = get_value(op_string3, &ok);
                            value4   = get_value(op_string4, &ok);
                            value5   = get_value(op_string5, &ok);
                            intval6  = floor(get_value(op_string6, &ok));
                            intval7  = floor(get_value(op_string7, &ok));
                            intval8  = floor(get_value(op_string8, &ok));
                            intval9  = floor(get_value(op_string9, &ok));
                            intval10 = floor(get_value(op_string10, &ok));
                            intval11 = floor(get_value(op_string11, &ok));

                            SCRIPT_objects[dest].obj = new
                                OgreCheckerBoard(
                                    intval1, intval2, value3, value4,
                                    value5, intval6, intval7, intval8,
                                    intval9, intval10, intval11);
                            break;

                        case OG_CYLINDER:
                            sscanf(bufptr, "%s %s %s %s",
                                op_string1, op_string2, op_string3, op_string4);

                            value1  = get_value(op_string1, &ok);
                            value2  = get_value(op_string2, &ok);
                            value3  = get_value(op_string3, &ok);
                            intval4 = floor(get_value(op_string4, &ok));

                            SCRIPT_objects[dest].obj = new
                                OgreCylinder(value1, value2, value3, intval4);
                            break;

                        case OG_DOME:
                            sscanf(bufptr, "%s %s %s",
                                op_string1, op_string2, op_string3);

                            value1  = get_value(op_string1, &ok);
                            intval2 = floor(get_value(op_string2, &ok));
                            intval3 = floor(get_value(op_string3, &ok));

                            SCRIPT_objects[dest].obj = new
                                OgreDome(value1, intval2, intval3);
                            break;

                        case OG_FUZZY:
                            ok = sscanf(bufptr, "%s %s %s %s",
                                op_string1, op_string2, op_string3, op_string4);

                            if (ok < 4) {
                                fp = fopen(op_string1, "rt");
                                if (fp == NULL) {
                                    err_header();
                                    missing_file_err(op_string1);
                                } else {
                                    fclose(fp);
                                    SCRIPT_objects[dest].obj = new
                                        OgreFuzzy(op_string1);
                                }
                            } else {
                                value1  = get_value(op_string1, &ok);
                                intval2 = floor(get_value(op_string2, &ok));
                                intval3 = floor(get_value(op_string3, &ok));
                                intval4 = floor(get_value(op_string4, &ok));

                                SCRIPT_objects[dest].obj = new
                                    OgreFuzzy(value1, intval2, intval3, intval4);
                            }
                            break;

                        case OG_GEAR:
                            sscanf(bufptr, "%s %s %s %s",
                                op_string1, op_string2, op_string3, op_string4);

                            value1  = get_value(op_string1, &ok);
                            value2  = get_value(op_string2, &ok);
                            value3  = get_value(op_string3, &ok);
                            intval4 = floor(get_value(op_string4, &ok));

                            if (value1 <= value2) {
                                err_header();
                                printf("Major radius should be larger than minor radius\n");
                                SCRIPT_objects[dest].obj = new
                                    OgreGear(value2, value1, value3, intval4);
                            } else {
                                SCRIPT_objects[dest].obj = new
                                    OgreGear(value1, value2, value3, intval4);
                            }
                            break;

                        case OG_PANEL:
                            sscanf(bufptr, "%s %s %s %s [%n",
                                op_string1, op_string2, op_string3,
                                op_string4, &count);

                            bufptr += count;

                            value1  = get_value(op_string1, &ok);
                            intval2 = floor(get_value(op_string2, &ok));
                            value3  = get_value(op_string3, &ok);
                            value4  = get_value(op_string4, &ok);

                            data = new float[intval2];

                            for (i=0; i < intval2; i++) {
                                sscanf(bufptr, "%s%n", op_string1, &count);
                                data[i] = get_value(op_string1, &ok);
                                bufptr += count;
                            }

                            SCRIPT_objects[dest].obj = new
                                OgrePanel(intval2, value1, value3,
                                value4, data);

                            delete data;
                            break;

                        case OG_POLYGON:
                            sscanf(bufptr, "%s %s",
                                op_string1, op_string2);

                            intval1 = floor(get_value(op_string1, &ok));
                            value2  = get_value(op_string2, &ok);

                            SCRIPT_objects[dest].obj = new
                                OgrePolygon(intval1, value2);
                            break;

                        case OG_POSTFIELD:
                            ok = sscanf(bufptr, "%s %s %s %s %s",
                                op_string1, op_string2, op_string3,
                                op_string4, op_string5);

                            if (ok < 5) {
                                fp = fopen(op_string1, "rt");
                                if (fp == NULL) {
                                    err_header();
                                    missing_file_err(op_string1);
                                } else {
                                    fclose(fp);
                                    SCRIPT_objects[dest].obj = new
                                        OgrePostField(op_string1);
                                }
                            } else {
                                intval1 = floor(get_value(op_string1, &ok));
                                intval2 = floor(get_value(op_string2, &ok));
                                value3  = get_value(op_string3, &ok);
                                value4  = get_value(op_string4, &ok);
                                intval5 = floor(get_value(op_string5, &ok));

                                SCRIPT_objects[dest].obj = new
                                    OgrePostField(intval1, intval2, value3,
                                    value4, intval5);
                            }
                            break;

                        case OG_PYRAMID:
                            sscanf(bufptr, "%s %s %s",
                                op_string1, op_string2, op_string3);

                            value1  = get_value(op_string1, &ok);
                            value2  = get_value(op_string2, &ok);
                            intval3 = floor(get_value(op_string3, &ok));

                            SCRIPT_objects[dest].obj = new
                                OgrePyramid(value1, value2, intval3);
                            break;

                        case OG_RECTANGLE:
                            sscanf(bufptr, "%s %s", op_string1, op_string2);

                            value1 = get_value(op_string1, &ok);
                            value2 = get_value(op_string2, &ok);

                            SCRIPT_objects[dest].obj = new
                                OgreRectangle(value1, value2);
                            break;

                        case OG_REVOLUTION:
                            sscanf(bufptr, "%s %s %s %s [%n",
                                op_string1, op_string2, op_string3,
                                op_string4, &count);

                            bufptr += count;

                            value1  = get_value(op_string1, &ok);
                            intval2 = floor(get_value(op_string2, &ok));
                            intval3 = floor(get_value(op_string3, &ok));
                            value4  = get_value(op_string4, &ok);

                            data = new float[intval2];

                            for (i=0; i < intval2; i++) {
                                sscanf(bufptr, "%s%n", op_string1, &count);
                                data[i] = get_value(op_string1, &ok);
                                bufptr += count;
                            }

                            SCRIPT_objects[dest].obj = new
                                OgreRevolution(value1, intval2, intval3,
                                value4, data);

                            delete data;
                            break;

                        case OG_SOCCER:
                            sscanf(bufptr, "%s", op_string1);

                            value1 = get_value(op_string1, &ok);

                            SCRIPT_objects[dest].obj = new
                                OgreSoccer(value1);
                            break;

                        case OG_SPHERE:
                            sscanf(bufptr, "%s %s %s",
                                op_string1, op_string2, op_string3);

                            value1  = get_value(op_string1, &ok);
                            intval2 = floor(get_value(op_string2, &ok));
                            intval3 = floor(get_value(op_string3, &ok));

                            SCRIPT_objects[dest].obj = new
                                OgreSphere(value1, intval2, intval3);
                            break;

                        case OG_TORUS:
                            sscanf(bufptr, "%s %s %s %s",
                                op_string1, op_string2, op_string3, op_string4);

                            value1  = get_value(op_string1, &ok);
                            value2  = get_value(op_string2, &ok);
                            intval3 = floor(get_value(op_string3, &ok));
                            intval4 = floor(get_value(op_string4, &ok));

                            SCRIPT_objects[dest].obj = new
                                OgreTorus(value1, value2, intval3, intval4);
                            break;

                        case OG_TRIANGLE:
                            sscanf(bufptr, "%s %s %s", op_string1, op_string2,
                                op_string3);

                            value1 = get_value(op_string1, &ok);
                            value2 = get_value(op_string2, &ok);
                            value3 = get_value(op_string3, &ok);

                            SCRIPT_objects[dest].obj = new
                                OgreTriangle(value1, value2, value3);
                            break;

                        case OG_TWIST:
                            sscanf(bufptr, "%s", op_string1);

                            fp = fopen(op_string1, "rt");
                            if (fp == NULL) {
                                err_header();
                                missing_file_err(op_string1);
                            } else {
                                fclose(fp);
                                SCRIPT_objects[dest].obj = new
                                    OgreTwist(op_string1);
                            }
                            break;

                        case OG_WEDGE:
                            sscanf(bufptr, "%s %s %s %s",
                                op_string1, op_string2, op_string3, op_string4);

                            value1  = get_value(op_string1, &ok);
                            value2  = get_value(op_string2, &ok);
                            value3  = get_value(op_string3, &ok);
                            intval4 = floor(get_value(op_string4, &ok));

                            SCRIPT_objects[dest].obj = new
                                OgreWedge(value1, value2, value3, intval4);
                            break;
                    }
                } else if (objcmd == OG_MERGE) {
                    sscanf(bufptr, "%s %s", op_string1, op_string2);

                    source  = find_object(op_string1);
                    source2 = find_object(op_string2);

                    tempchar = 1;
                    if (source  == -1) {
                        tempchar = 0;
                        err_header();
                        undecl_object_err(op_string1);
                    }
                    if (source2 == -1) {
                        tempchar = 0;
                        err_header();
                        undecl_object_err(op_string2);
                    }

                    if (tempchar) {
                        if (source == source2) {
                            err_header();
                            printf("Please don't merge objects with themselves\n");
                        } else if ((SCRIPT_objects[source ].obj == NULL) &&
                                   (SCRIPT_objects[source2].obj == NULL)) {
                            if (SCRIPT_objects[dest].obj != NULL) {
                                delete SCRIPT_objects[dest].obj;
                                SCRIPT_objects[dest].obj = NULL;
                            }
                        } else if (SCRIPT_objects[source ].obj == NULL) {
                            // Copy source2 to dest
                            if (dest != source2) {
                                tempobj =
                                    new OgreObject(SCRIPT_objects[source2].obj);

                                if (SCRIPT_objects[dest].obj != NULL) {
                                    delete SCRIPT_objects[dest].obj;
                                }

                                SCRIPT_objects[dest].obj = tempobj;
                            }
                        } else if (SCRIPT_objects[source2].obj == NULL) {
                            // Copy source to dest
                            if (dest != source ) {
                                tempobj =
                                    new OgreObject(SCRIPT_objects[source].obj);

                                if (SCRIPT_objects[dest].obj != NULL) {
                                    delete SCRIPT_objects[dest].obj;
                                }

                                SCRIPT_objects[dest].obj = tempobj;
                            }
                        } else if (dest == source ) {
                            tempobj = new OgreObject(SCRIPT_objects[dest].obj);
                            delete SCRIPT_objects[dest].obj;

                            SCRIPT_objects[dest].obj = new
                                OgreObject(tempobj, SCRIPT_objects[source2].obj);

                            delete tempobj;
                        } else if (dest == source2) {
                            tempobj = new OgreObject(SCRIPT_objects[dest].obj);
                            delete SCRIPT_objects[dest].obj;

                            SCRIPT_objects[dest].obj = new
                                OgreObject(tempobj, SCRIPT_objects[source].obj);

                            delete tempobj;
                        } else {
                            if (SCRIPT_objects[dest].obj != NULL) {
                                delete SCRIPT_objects[dest].obj;
                            }

                            SCRIPT_objects[dest].obj = new
                                OgreObject(SCRIPT_objects[source ].obj,
                                           SCRIPT_objects[source2].obj);
                        }
                    }
                } else {
                    sscanf(bufptr, "%s%n", opcode_string, &count);
                    bufptr += count;

                    source = find_object(opcode_string);

                    if (source == -1) {
                        err_header();
                        undecl_object_err(op_string1);
                    } else if (SCRIPT_objects[source].obj == NULL) {
                        err_header();
                        printf("NULL object: %s\n", opcode_string);
                    } else {
                        if (source != dest) {
                            if (SCRIPT_objects[dest].obj != NULL) {
                                delete SCRIPT_objects[dest].obj;
                            }

                            SCRIPT_objects[dest].obj = new
                                OgreObject(SCRIPT_objects[source].obj);
                        }

                        switch (objcmd) {
                            case OG_SCALE:
                                sscanf(bufptr, "%s %s %s",
                                       op_string1, op_string2, op_string3);

                                value1 = get_value(op_string1, &ok);
                                value2 = get_value(op_string2, &ok);
                                value3 = get_value(op_string3, &ok);

                                SCRIPT_objects[dest].obj->scale(value1,
                                    value2, value3);
                                break;

                            case OG_TRANS:
                                sscanf(bufptr, "%s %s %s",
                                       op_string1, op_string2, op_string3);

                                value1 = get_value(op_string1, &ok);
                                value2 = get_value(op_string2, &ok);
                                value3 = get_value(op_string3, &ok);

                                SCRIPT_objects[dest].obj->trans(value1,
                                    value2, value3);
                                break;

                            case OG_ROTATE:
                                sscanf(bufptr, "%s %s", op_string1, op_string2);

                                tempchar = toupper(op_string1[0]);
                                value2 = get_value(op_string2, &ok);

                                switch (tempchar) {
                                    case 'X':
                                        SCRIPT_objects[dest].obj->rotx(value2);
                                        break;

                                    case 'Y':
                                        SCRIPT_objects[dest].obj->roty(value2);
                                        break;

                                    case 'Z':
                                        SCRIPT_objects[dest].obj->rotz(value2);
                                        break;

                                    default:
                                        err_header();
                                        printf("Invalid rotation axis %c\n",
                                            tempchar);
                                        break;
                                }
                                break;

                            case OG_ROTX:
                                sscanf(bufptr, "%s", op_string1);

                                value1 = get_value(op_string1, &ok);

                                SCRIPT_objects[dest].obj->rotx(value1);
                                break;

                            case OG_ROTY:
                                sscanf(bufptr, "%s", op_string1);

                                value1 = get_value(op_string1, &ok);

                                SCRIPT_objects[dest].obj->roty(value1);
                                break;

                            case OG_ROTZ:
                                sscanf(bufptr, "%s", op_string1);

                                value1 = get_value(op_string1, &ok);

                                SCRIPT_objects[dest].obj->rotz(value1);
                                break;

                            case OG_COPY:
                                // Do nothing, it's already done.
                                break;

                            case OG_INVERT:
                                SCRIPT_objects[dest].obj->invert();
                                break;

                            case OG_DOUBLESIDE:
                                SCRIPT_objects[dest].obj->doubleside();
                                break;

                            case OG_PAINT:
                                ok = sscanf(bufptr, "%s %s %s",
                                            op_string1, op_string2, op_string3);

                                if (ok < 3) {
                                    operand1 = find_color(op_string1);
                                    if (operand1 == -1) {
                                        err_header();
                                        undecl_color_err(op_string1);
                                    } else {
                                        SCRIPT_objects[dest].obj->paint(
                                            SCRIPT_colors[operand1].red,
                                            SCRIPT_colors[operand1].green,
                                            SCRIPT_colors[operand1].blue);
                                    }
                                } else {
                                    value1 = floor(get_value(op_string1, &ok));
                                    value2 = floor(get_value(op_string2, &ok));
                                    value3 = floor(get_value(op_string3, &ok));

#ifdef MFX
                                    SCRIPT_objects[dest].obj->paint(
                                        (unsigned char) value1,
                                        (unsigned char) value2,
                                        (unsigned char) value3);
#else
                                    SCRIPT_objects[dest].obj->paint(value1,
                                        value2, value3);
#endif
                                }
                                break;

                            case OG_PAINTPOLY:
#ifdef MFX
                                ok = sscanf(bufptr, "%s %s %s %s",
#else
                                ok = sscanf(bufptr, "%d %s %s %s",
#endif
                                            op_string4, op_string1,
                                            op_string2, op_string3);

                                intval4 = floor(get_value(op_string4, &ok));

                                if (ok < 4) {
                                    operand1 = find_color(op_string1);
                                    if (operand1 == -1) {
                                        err_header();
                                        undecl_color_err(op_string1);
                                    } else {
                                        SCRIPT_objects[dest].obj->paintpoly(
                                            intval4,
                                            SCRIPT_colors[operand1].red,
                                            SCRIPT_colors[operand1].green,
                                            SCRIPT_colors[operand1].blue);
                                    }
                                } else {
                                    value1 = floor(get_value(op_string1, &ok));
                                    value2 = floor(get_value(op_string2, &ok));
                                    value3 = floor(get_value(op_string3, &ok));

#ifdef MFX
                                    SCRIPT_objects[dest].obj->paintpoly(
                                        intval4,
                                        (unsigned char) value1,
                                        (unsigned char) value2,
                                        (unsigned char) value3);
#else
                                    SCRIPT_objects[dest].obj->paintpoly(
                                        intval4, value1, value2, value3);
#endif
                                }
                                break;

                            case OG_CENTER:
                                SCRIPT_objects[dest].obj->re_center();
                                break;

                            case OG_FLOOR:
                                SCRIPT_objects[dest].obj->re_floor();
                                break;

                            case OG_INVALID:
                            default:
                                err_header();
                                printf("Invalid opcode: %s\n", opcode_string);
                                break;
                        }
                    }
                }
            }
            break;

        case OG_INVALID:
        default:
            err_header();
            printf("Invalid opcode: %s\n", opcode_string);
            break;
    }

    return retval;
}

