#include <math.h>
#include <stdio.h>

#include "ogreobj.hpp"

#define FLOAT_TO_FIXED(x) ((long) ((x * 65536.0) + 0.5))

#define SQUARE(x) ((x) * (x))

#ifdef MFX
static const
#endif
long int PLG_colortab[240][3] = { {  64,   0,   0 },  // RED
                                  {  72,   0,   0 },
                                  {  80,   0,   0 },
                                  {  88,   0,   0 },
                                  {  96,   0,   0 },
                                  { 104,   0,   0 },
                                  { 112,   0,   0 },
                                  { 120,   0,   0 },
                                  { 132,   0,   0 },
                                  { 144,   0,   0 },
                                  { 156,   0,   0 },
                                  { 168,   0,   0 },
                                  { 184,   0,   0 },
                                  { 204,   0,   0 },
                                  { 224,   0,   0 },
                                  { 248,   0,   0 },

                                  {  64,  32,   0 },  // ORANGE
                                  {  72,  36,   0 },
                                  {  80,  40,   0 },
                                  {  88,  40,   0 },
                                  {  96,  44,   0 },
                                  { 104,  48,   0 },
                                  { 112,  56,   0 },
                                  { 120,  60,   0 },
                                  { 132,  64,   0 },
                                  { 144,  68,   0 },
                                  { 156,  76,   0 },
                                  { 168,  80,   0 },
                                  { 184,  88,   0 },
                                  { 204, 100,   0 },
                                  { 224, 108,   0 },
                                  { 248, 120,   0 },

                                  {  64,  32,  32 },  // SALMON
                                  {  72,  36,  36 },
                                  {  80,  40,  40 },
                                  {  88,  40,  40 },
                                  {  96,  44,  44 },
                                  { 104,  48,  48 },
                                  { 112,  56,  56 },
                                  { 120,  60,  60 },
                                  { 132,  64,  64 },
                                  { 144,  68,  68 },
                                  { 156,  76,  76 },
                                  { 168,  80,  80 },
                                  { 184,  88,  88 },
                                  { 204, 100, 100 },
                                  { 224, 108, 108 },
                                  { 248, 120, 120 },

                                  {  64,  48,  32 },  // TAN
                                  {  72,  56,  36 },
                                  {  80,  60,  40 },
                                  {  88,  64,  40 },
                                  {  96,  72,  44 },
                                  { 104,  76,  48 },
                                  { 112,  84,  56 },
                                  { 120,  88,  60 },
                                  { 132,  96,  64 },
                                  { 144, 108,  68 },
                                  { 156, 116,  76 },
                                  { 168, 124,  80 },
                                  { 184, 136,  88 },
                                  { 204, 152, 100 },
                                  { 224, 168, 108 },
                                  { 248, 184, 120 },

                                  {  64,  64,  16 },  // LIGHT YELLOW
                                  {  72,  72,  18 },
                                  {  80,  80,  20 },
                                  {  88,  88,  20 },
                                  {  96,  96,  22 },
                                  { 104, 104,  24 },
                                  { 112, 112,  28 },
                                  { 120, 120,  30 },
                                  { 132, 132,  32 },
                                  { 144, 144,  34 },
                                  { 156, 156,  38 },
                                  { 168, 168,  40 },
                                  { 184, 184,  44 },
                                  { 204, 204,  50 },
                                  { 224, 224,  54 },
                                  { 248, 248,  60 },

                                  {  64,  64,   0 },  // YELLOW
                                  {  72,  72,   0 },
                                  {  80,  80,   0 },
                                  {  88,  88,   0 },
                                  {  96,  96,   0 },
                                  { 104, 104,   0 },
                                  { 112, 112,   0 },
                                  { 120, 120,   0 },
                                  { 132, 132,   0 },
                                  { 144, 144,   0 },
                                  { 156, 156,   0 },
                                  { 168, 168,   0 },
                                  { 184, 184,   0 },
                                  { 204, 204,   0 },
                                  { 224, 224,   0 },
                                  { 248, 248,   0 },

                                  {   0,  64,  16 },  // LIGHT GREEN
                                  {   0,  72,  18 },
                                  {   0,  80,  20 },
                                  {   0,  88,  20 },
                                  {   0,  96,  22 },
                                  {   0, 104,  24 },
                                  {   0, 112,  28 },
                                  {   0, 120,  30 },
                                  {   0, 132,  32 },
                                  {   0, 144,  34 },
                                  {   0, 156,  38 },
                                  {   0, 168,  40 },
                                  {   0, 184,  44 },
                                  {   0, 204,  50 },
                                  {   0, 224,  54 },
                                  {   0, 248,  60 },

                                  {   0,  64,   0 },  // GREEN
                                  {   0,  72,   0 },
                                  {   0,  80,   0 },
                                  {   0,  88,   0 },
                                  {   0,  96,   0 },
                                  {   0, 104,   0 },
                                  {   0, 112,   0 },
                                  {   0, 120,   0 },
                                  {   0, 132,   0 },
                                  {   0, 144,   0 },
                                  {   0, 156,   0 },
                                  {   0, 168,   0 },
                                  {   0, 184,   0 },
                                  {   0, 204,   0 },
                                  {   0, 224,   0 },
                                  {   0, 248,   0 },

                                  {   0,  64,  48 },  // AQUA
                                  {   0,  72,  56 },
                                  {   0,  80,  60 },
                                  {   0,  88,  64 },
                                  {   0,  96,  72 },
                                  {   0, 104,  76 },
                                  {   0, 112,  84 },
                                  {   0, 120,  88 },
                                  {   0, 132,  96 },
                                  {   0, 144, 108 },
                                  {   0, 156, 116 },
                                  {   0, 168, 124 },
                                  {   0, 184, 136 },
                                  {   0, 204, 152 },
                                  {   0, 224, 168 },
                                  {   0, 248, 184 },

                                  {   0,  48,  64 },  // LIGHT BLUE
                                  {   0,  56,  72 },
                                  {   0,  60,  80 },
                                  {   0,  64,  88 },
                                  {   0,  72,  96 },
                                  {   0,  76, 104 },
                                  {   0,  84, 112 },
                                  {   0,  88, 120 },
                                  {   0,  96, 132 },
                                  {   0, 108, 144 },
                                  {   0, 116, 156 },
                                  {   0, 124, 168 },
                                  {   0, 136, 184 },
                                  {   0, 152, 204 },
                                  {   0, 168, 224 },
                                  {   0, 184, 248 },

                                  {   0,   0,  64 },  // BLUE
                                  {   0,   0,  72 },
                                  {   0,   0,  80 },
                                  {   0,   0,  88 },
                                  {   0,   0,  96 },
                                  {   0,   0, 104 },
                                  {   0,   0, 112 },
                                  {   0,   0, 120 },
                                  {   0,   0, 132 },
                                  {   0,   0, 144 },
                                  {   0,   0, 156 },
                                  {   0,   0, 168 },
                                  {   0,   0, 184 },
                                  {   0,   0, 204 },
                                  {   0,   0, 224 },
                                  {   0,   0, 248 },

                                  {  64,   0,  64 },  // PURPLE
                                  {  72,   0,  72 },
                                  {  80,   0,  80 },
                                  {  88,   0,  88 },
                                  {  96,   0,  96 },
                                  { 104,   0, 104 },
                                  { 112,   0, 112 },
                                  { 120,   0, 120 },
                                  { 132,   0, 132 },
                                  { 144,   0, 144 },
                                  { 156,   0, 156 },
                                  { 168,   0, 168 },
                                  { 184,   0, 184 },
                                  { 204,   0, 204 },
                                  { 224,   0, 224 },
                                  { 248,   0, 248 },

                                  {  64,   0,  48 },  // PINK
                                  {  72,   0,  56 },
                                  {  80,   0,  60 },
                                  {  88,   0,  64 },
                                  {  96,   0,  72 },
                                  { 104,   0,  76 },
                                  { 112,   0,  84 },
                                  { 120,   0,  88 },
                                  { 132,   0,  96 },
                                  { 144,   0, 108 },
                                  { 156,   0, 116 },
                                  { 168,   0, 124 },
                                  { 184,   0, 136 },
                                  { 204,   0, 152 },
                                  { 224,   0, 168 },
                                  { 248,   0, 184 },

                                  {  32,  32,  32 },  // GREY
                                  {  36,  36,  36 },
                                  {  40,  40,  40 },
                                  {  44,  44,  44 },
                                  {  48,  48,  48 },
                                  {  52,  52,  52 },
                                  {  56,  56,  56 },
                                  {  60,  60,  60 },
                                  {  64,  64,  64 },
                                  {  72,  72,  72 },
                                  {  76,  76,  76 },
                                  {  84,  84,  84 },
                                  {  92,  92,  92 },
                                  { 104, 104, 104 },
                                  { 112, 112, 112 },
                                  { 124, 124, 124 },

                                  {  64,  64,  64 },  // WHITE
                                  {  72,  72,  72 },
                                  {  80,  80,  80 },
                                  {  88,  88,  88 },
                                  {  96,  96,  96 },
                                  { 104, 104, 104 },
                                  { 112, 112, 112 },
                                  { 120, 120, 120 },
                                  { 132, 132, 132 },
                                  { 144, 144, 144 },
                                  { 156, 156, 156 },
                                  { 168, 168, 168 },
                                  { 184, 184, 184 },
                                  { 204, 204, 204 },
                                  { 224, 224, 224 },
                                  { 252, 252, 252 },
                               };


char *
OgreObject::get_next_line(char *str, int len, FILE *fp)
{
    char *retval;

    do {
        retval = fgets(str, len, fp);
    } while (((str[0] == '#') || (str[0] == '\n')) && (retval != NULL));

    return (retval);
}


OgreObject::OgreObject(void)
{
    num_verts = 0;
    num_polys = 0;

    bsphere = 0.0;

    verts = NULL;
    polys = NULL;
}


OgreObject::OgreObject(char *fname)
{
    FILE *fp;
    int i, j, count, count2;
    char tempstr[256];

    fp = fopen(fname, "rt");

    if (fp != NULL) {
        get_next_line(tempstr, 256, fp);
        get_next_line(tempstr, 256, fp);

        sscanf(tempstr, "%d %d %f", &num_verts, &num_polys, &bsphere);

        verts = new Vertex[num_verts];
        polys = new Poly[num_polys];

        for (i=0; i < num_verts; i++) {
            get_next_line(tempstr, 256, fp);
            sscanf(tempstr, "%f %f %f", &verts[i].x, &verts[i].y, &verts[i].z);
        }

        for (i=0; i < num_polys; i++) {
            get_next_line(tempstr, 256, fp);
#ifdef MFX
            sscanf(tempstr, "r%c g%c b%c %f %f %f %d %n",
#else
            sscanf(tempstr, "r%d g%d b%d %f %f %f %d %n",
#endif
                            &polys[i].color.red,
                            &polys[i].color.green,
                            &polys[i].color.blue,
                            &polys[i].norm_vect.x,
                            &polys[i].norm_vect.y,
                            &polys[i].norm_vect.z,
                            &polys[i].num_points,
                            &count);

            polys[i].points = new int[polys[i].num_points];

            for (j=0; j < polys[i].num_points; j++) {
                sscanf((tempstr+count), "%d %n", &polys[i].points[j], &count2);
                count = count + count2;
            }
        }

        fclose(fp);
    }
}


OgreObject::OgreObject(OgreObject *oldobj)
{
    int i, j;

    num_verts = oldobj->num_verts;
    num_polys = oldobj->num_polys;

    verts = new Vertex[num_verts];
    polys = new Poly[num_polys];

    bsphere = oldobj->bsphere;

    for (i=0; i < num_verts; i++) {
        verts[i].x = oldobj->verts[i].x;
        verts[i].y = oldobj->verts[i].y;
        verts[i].z = oldobj->verts[i].z;
    }

    for (i = 0; i < num_polys; i++) {
        polys[i].num_points  = oldobj->polys[i].num_points;
        polys[i].color.red   = oldobj->polys[i].color.red;
        polys[i].color.green = oldobj->polys[i].color.green;
        polys[i].color.blue  = oldobj->polys[i].color.blue;
        polys[i].norm_vect.x = oldobj->polys[i].norm_vect.x;
        polys[i].norm_vect.y = oldobj->polys[i].norm_vect.y;
        polys[i].norm_vect.z = oldobj->polys[i].norm_vect.z;

        polys[i].points = new int[polys[i].num_points];
        for (j=0; j < polys[i].num_points; j++) {
            polys[i].points[j] = oldobj->polys[i].points[j];
        }
    }
}


OgreObject::OgreObject(OgreObject *obj1, OgreObject *obj2)
{
    int i, j, v1, v2, p1, p2, temp;

    v1 = obj1->num_verts;
    v2 = obj2->num_verts;
    p1 = obj1->num_polys;
    p2 = obj2->num_polys;

    num_verts = v1 + v2;
    num_polys = p1 + p2;

    verts = new Vertex[num_verts];
    polys = new Poly[num_polys];

    if (obj1->bsphere > obj2->bsphere) {
        bsphere = obj1->bsphere;
    } else {
        bsphere = obj2->bsphere;
    }

    for (i=0; i < v1; i++) {
        verts[i].x = obj1->verts[i].x;
        verts[i].y = obj1->verts[i].y;
        verts[i].z = obj1->verts[i].z;
    }

    for (i=0; i < v2; i++) {
        verts[i+v1].x = obj2->verts[i].x;
        verts[i+v1].y = obj2->verts[i].y;
        verts[i+v1].z = obj2->verts[i].z;
    }

    for (i=0; i < p1; i++) {
        temp = obj1->polys[i].num_points;

        polys[i].num_points  = temp;
        polys[i].color.red   = obj1->polys[i].color.red;
        polys[i].color.green = obj1->polys[i].color.green;
        polys[i].color.blue  = obj1->polys[i].color.blue;
        polys[i].norm_vect.x = obj1->polys[i].norm_vect.x;
        polys[i].norm_vect.y = obj1->polys[i].norm_vect.y;
        polys[i].norm_vect.z = obj1->polys[i].norm_vect.z;
        polys[i].points = new int[temp];

        for (j=0; j < temp; j++) {
            polys[i].points[j] = obj1->polys[i].points[j];
        }
    }

    for (i=0; i < p2; i++) {
        temp = obj2->polys[i].num_points;

        polys[i+p1].num_points  = temp;
        polys[i+p1].color.red   = obj2->polys[i].color.red;
        polys[i+p1].color.green = obj2->polys[i].color.green;
        polys[i+p1].color.blue  = obj2->polys[i].color.blue;
        polys[i+p1].norm_vect.x = obj2->polys[i].norm_vect.x;
        polys[i+p1].norm_vect.y = obj2->polys[i].norm_vect.y;
        polys[i+p1].norm_vect.z = obj2->polys[i].norm_vect.z;
        polys[i+p1].points = new int[temp];

        for (j=0; j < temp; j++) {
            polys[i+p1].points[j] = obj2->polys[i].points[j] + v1;
        }
    }
}


OgreObject::~OgreObject(void)
{
    int i;

    delete verts;
    for (i=0; i < num_polys; i++) {
        delete polys[i].points;
    }

    delete polys;
}


void
OgreObject::scale(float x, float y, float z)
{
    int i;

    for (i=0; i < num_verts; i++) {
        verts[i].x *= x;
        verts[i].y *= y;
        verts[i].z *= z;
    }

    calc_normals();
    calc_bsphere();
}


void
OgreObject::trans(float x, float y, float z)
{
    int i;

    for (i=0; i < num_verts; i++) {
        verts[i].x += x;
        verts[i].y += y;
        verts[i].z += z;
    }

    calc_normals();
    calc_bsphere();
}


void
OgreObject::rotx(float angle)
{
    int i;
    float angle_cos, angle_sin;
    float tempy, tempz;

    angle = (angle / 180) * M_PI;

    angle_cos = cos(angle);
    angle_sin = sin(angle);

    for (i=0; i < num_verts; i++) {
        tempy = (verts[i].y * angle_cos) - (verts[i].z * angle_sin);
        tempz = (verts[i].y * angle_sin) + (verts[i].z * angle_cos);

        verts[i].y = tempy;
        verts[i].z = tempz;
    }

    for (i=0; i < num_polys; i++) {
        tempy = (polys[i].norm_vect.y * angle_cos) -
                (polys[i].norm_vect.z * angle_sin);
        tempz = (polys[i].norm_vect.y * angle_sin) +
                (polys[i].norm_vect.z * angle_cos);

        polys[i].norm_vect.y = tempy;
        polys[i].norm_vect.z = tempz;
    }

    calc_bsphere();
}


void
OgreObject::roty(float angle)
{
    int i;
    float angle_cos, angle_sin;
    float tempx, tempz;

    angle = (angle / 180) * M_PI;

    angle_cos = cos(angle);
    angle_sin = sin(angle);

    for (i=0; i < num_verts; i++) {
        tempx = (verts[i].x * angle_cos) + (verts[i].z * angle_sin);
        tempz = (verts[i].z * angle_cos) - (verts[i].x * angle_sin);

        verts[i].x = tempx;
        verts[i].z = tempz;
    }

    for (i=0; i < num_polys; i++) {
        tempx = (polys[i].norm_vect.x * angle_cos) +
                (polys[i].norm_vect.z * angle_sin);
        tempz = (polys[i].norm_vect.z * angle_cos) -
                (polys[i].norm_vect.x * angle_sin);

        polys[i].norm_vect.x = tempx;
        polys[i].norm_vect.z = tempz;
    }

    calc_bsphere();
}


void
OgreObject::rotz(float angle)
{
    int i;
    float angle_cos, angle_sin;
    float tempx, tempy;

    angle = (angle / 180) * M_PI;

    angle_cos = cos(angle);
    angle_sin = sin(angle);

    for (i=0; i < num_verts; i++) {
        tempx = (verts[i].x * angle_cos) - (verts[i].y * angle_sin);
        tempy = (verts[i].x * angle_sin) + (verts[i].y * angle_cos);

        verts[i].x = tempx;
        verts[i].y = tempy;
    }

    for (i=0; i < num_polys; i++) {
        tempx = (polys[i].norm_vect.x * angle_cos) -
                (polys[i].norm_vect.y * angle_sin);
        tempy = (polys[i].norm_vect.x * angle_sin) +
                (polys[i].norm_vect.y * angle_cos);

        polys[i].norm_vect.x = tempx;
        polys[i].norm_vect.y = tempy;
    }

    calc_bsphere();
}


void
OgreObject::paint(unsigned char r, unsigned char g, unsigned char b)
{
    int i;

    for (i=0; i < num_polys; i++) {
        polys[i].color.red   = r;
        polys[i].color.green = g;
        polys[i].color.blue  = b;
    }
}


void
OgreObject::paintpoly(int pnum, unsigned char r, unsigned char g, unsigned char b)
{
    polys[pnum].color.red   = r;
    polys[pnum].color.green = g;
    polys[pnum].color.blue  = b;
}


void
OgreObject::invert(void)
{
    int i, j, temp, swap;

    for (i=0; i < num_polys; i++) {
        temp = polys[i].num_points;

        for (j=0; j < (temp / 2); j++) {
            swap = polys[i].points[j];
            polys[i].points[j] = polys[i].points[temp-j-1];
            polys[i].points[temp-j-1] = swap;
        }
        polys[i].norm_vect.x = -polys[i].norm_vect.x;
        polys[i].norm_vect.y = -polys[i].norm_vect.y;
        polys[i].norm_vect.z = -polys[i].norm_vect.z;
    }
}


void
OgreObject::doubleside(void)
{
    Poly *newpolys;
    int i, j, temp, num;

    temp = num_polys;
    newpolys = new Poly[temp * 2];

    for (i=0; i < temp; i++) {
        num = polys[i].num_points;

        newpolys[i].num_points  = num;
        newpolys[i].color.red   = polys[i].color.red;
        newpolys[i].color.green = polys[i].color.green;
        newpolys[i].color.blue  = polys[i].color.blue;
        newpolys[i].norm_vect.x = polys[i].norm_vect.x;
        newpolys[i].norm_vect.y = polys[i].norm_vect.y;
        newpolys[i].norm_vect.z = polys[i].norm_vect.z;
        newpolys[i].points      = new int[num];

        newpolys[i+temp].num_points  =  num;
        newpolys[i+temp].color.red   =  polys[i].color.red;
        newpolys[i+temp].color.green =  polys[i].color.green;
        newpolys[i+temp].color.blue  =  polys[i].color.blue;
        newpolys[i+temp].norm_vect.x = -polys[i].norm_vect.x;
        newpolys[i+temp].norm_vect.y = -polys[i].norm_vect.y;
        newpolys[i+temp].norm_vect.z = -polys[i].norm_vect.z;
        newpolys[i+temp].points      =  new int[num];

        for (j=0; j < num; j++) {
            newpolys[i].points[j]      = polys[i].points[j];
            newpolys[i+temp].points[j] = polys[i].points[num-j-1];
        }
        delete polys[i].points;
    }

    num_polys *= 2;
    delete polys;
    polys = newpolys;
}


int
OgreObject::save(char *fname)
{
    return (saveOGR(fname));
}


int
OgreObject::saveOGR(char *fname)
{
    int i, j;
    FILE *fp;


    fp = fopen(fname, "wt");

    if (fp != NULL) {
        fprintf(fp, "OGRE\n");
        fprintf(fp, "%d %d %f\n\n", num_verts, num_polys, bsphere);

        for (i=0; i < num_verts; i++) {
            fprintf(fp, "%f %f %f\n", verts[i].x, verts[i].y, verts[i].z);
        }

        fprintf(fp, "\n");

        for (i=0; i < num_polys; i++) {
            fprintf(fp, "r%03d g%03d b%03d %9f %9f %9f %3d",
                        polys[i].color.red,
                        polys[i].color.green,
                        polys[i].color.blue,
                        polys[i].norm_vect.x,
                        polys[i].norm_vect.y,
                        polys[i].norm_vect.z,
                        polys[i].num_points);

            for (j=0; j < polys[i].num_points; j++) {
                fprintf(fp, " %d", polys[i].points[j]);
            }

            fprintf(fp, "\n");
        }

        fprintf(fp, "\n");

        fclose(fp);
        return (0);
    } else {
        return (-1);
    }
}


int
OgreObject::savePLG(char *fname)
{
    long int best_dist, dist;
    int      best_color;
    int i, j;
    unsigned int plgcolor, basecolor;
    FILE *fp;


    fp = fopen(fname, "wt");

    if (fp != NULL) {
        fprintf(fp, "OGRE %d %d\n\n", num_verts, num_polys);

        for (i=0; i < num_verts; i++) {
            fprintf(fp, "%d %d %d\n", (int) floor(verts[i].x + 0.5),
                                      (int) floor(verts[i].y + 0.5),
                                      (int) floor(verts[i].z + 0.5));
        }

        fprintf(fp, "\n");

        for (i=0; i < num_polys; i++) {
            best_color = 0;
            best_dist  = (long)((PLG_colortab[0][0] - polys[i].color.red) *
                                (PLG_colortab[0][0] - polys[i].color.red)) +
                         (long)((PLG_colortab[0][1] - polys[i].color.green) *
                                (PLG_colortab[0][1] - polys[i].color.green)) +
                         (long)((PLG_colortab[0][2] - polys[i].color.blue) *
                                (PLG_colortab[0][2] - polys[i].color.blue));

            for (j=1; j < 240; j++) {
                dist  = (long)((PLG_colortab[j][0] - polys[i].color.red) *
                               (PLG_colortab[j][0] - polys[i].color.red)) +
                        (long)((PLG_colortab[j][1] - polys[i].color.green) *
                               (PLG_colortab[j][1] - polys[i].color.green)) +
                        (long)((PLG_colortab[j][2] - polys[i].color.blue) *
                               (PLG_colortab[j][2] - polys[i].color.blue));

                if (dist < best_dist) {
                    best_dist = dist;
                    best_color = j;
                }
            }

            basecolor = (best_color / 16) + 1;
            plgcolor  = 4096 + (basecolor * 256) +
                        ((best_color % 16) * 16);

            fprintf(fp, "0x%X %d", plgcolor, polys[i].num_points);

            for (j=0; j < polys[i].num_points; j++) {
                fprintf(fp, " %d", polys[i].points[j]);
            }

            fprintf(fp, "\n");
        }

        fprintf(fp, "\n");

        fclose(fp);
        return (0);
    } else {
        return (-1);
    }
}


int
OgreObject::saveNFF10(char *fname)
{
    int i, j;
    unsigned int plgcolor;
    FILE *fp;


    fp = fopen(fname, "wt");

    if (fp != NULL) {
        fprintf(fp, "s8\n");
        fprintf(fp, "OGRE\n");
        fprintf(fp, "%d\n", num_verts);

        for (i=0; i < num_verts; i++) {
            fprintf(fp, "%f %f %f\n", verts[i].x, verts[i].y, verts[i].z);
        }

        fprintf(fp, "%d\n", num_polys);

        for (i=0; i < num_polys; i++) {
            fprintf(fp, "%d ", polys[i].num_points);

            for (j = (polys[i].num_points-1); j >= 0; j--) {
                fprintf(fp, "%d ", polys[i].points[j]);
            }

            plgcolor = ((polys[i].color.red   >> 4) << 8) |
                       ((polys[i].color.green >> 4) << 4) |
                        (polys[i].color.blue  >> 4);

            fprintf(fp, "0x%03x\n", plgcolor);
        }

        fprintf(fp, "\n");

        fclose(fp);
        return (0);
    } else {
        return (-1);
    }
}


int
OgreObject::saveNFF17(char *fname)
{
    int i, j;
    unsigned int plgcolor;
    FILE *fp;


    fp = fopen(fname, "wt");

    if (fp != NULL) {
        fprintf(fp, "nff\n");
        fprintf(fp, "version 1.7\n");
        fprintf(fp, "OGRE\n");
        fprintf(fp, "%d\n", num_verts);

        for (i=0; i < num_verts; i++) {
            fprintf(fp, "%f %f %f\n", verts[i].x, verts[i].y, verts[i].z);
        }

        fprintf(fp, "%d\n", num_polys);

        for (i=0; i < num_polys; i++) {
            fprintf(fp, "%d ", polys[i].num_points);

            for (j = (polys[i].num_points-1); j >= 0; j--) {
                fprintf(fp, "%d ", polys[i].points[j]);
            }

            plgcolor = ((polys[i].color.red   >> 4) << 8) |
                       ((polys[i].color.green >> 4) << 4) |
                        (polys[i].color.blue  >> 4);

            fprintf(fp, "0x%03x\n", plgcolor);
        }

        fprintf(fp, "\n");

        fclose(fp);
        return (0);
    } else {
        return (-1);
    }
}


int
OgreObject::saveOBT(char *fname)
{
    Vector v1, v2;
    float normal_x, normal_y, normal_z, normal_len;
    int i, j, vert0, vert1, vertN;
    FILE *fp;


    fp = fopen(fname, "wt");

    if (fp != NULL) {
        fprintf(fp, "# Object created by OGRE 4.0\n");
        fprintf(fp, "# Written by David Boeren at the\n");
        fprintf(fp, "# University of Central Florida\n");
        fprintf(fp, "# boeren@ist.ucf.edu\n");

        for (i=0; i < num_verts; i++) {
            // PCVR Uses inverse Z
            verts[i].z = -verts[i].z;
        }

        fprintf(fp, "name OGREOBJ %d %d %d\n", (num_polys+num_verts),
            num_polys, num_verts);

        for (i=0; i < num_verts; i++) {
            fprintf(fp, "%ld %ld %ld\n",
                FLOAT_TO_FIXED(verts[i].x),
                FLOAT_TO_FIXED(verts[i].y),
                FLOAT_TO_FIXED(verts[i].z));
        }

        fprintf(fp, "# Start of normal vectors\n");
        for (i=0; i < num_polys; i++) {
            // Compute cross-product to get poly normal
            vert0 = polys[i].points[0];
            vert1 = polys[i].points[1];
            vertN = polys[i].points[polys[i].num_points - 1];

            v1.x = verts[vert1].x - verts[vert0].x;
            v1.y = verts[vert1].y - verts[vert0].y;
            v1.z = verts[vert1].z - verts[vert0].z;

            v2.x = verts[vertN].x - verts[vert0].x;
            v2.y = verts[vertN].y - verts[vert0].y;
            v2.z = verts[vertN].z - verts[vert0].z;

            normal_x = ((v1.y * v2.z) - (v1.z * v2.y));
            normal_y = ((v1.z * v2.x) - (v1.x * v2.z));
            normal_z = ((v1.x * v2.y) - (v1.y * v2.x));

            normal_len = (normal_x * normal_x) +
                         (normal_y * normal_y) +
                         (normal_z * normal_z);

            normal_len = sqrt(normal_len);

            normal_x = (normal_x / normal_len);
            normal_y = (normal_y / normal_len);
            normal_z = (normal_z / normal_len);

            fprintf(fp, "%ld %ld %ld\n",
                FLOAT_TO_FIXED(normal_x),
                FLOAT_TO_FIXED(normal_y),
                FLOAT_TO_FIXED(normal_z));
        }
        fprintf(fp, "# End of normal vectors\n");


        for (i=0; i < num_polys; i++) {
            fprintf(fp, "%d %d %d diffuse|ambient %d %d",
                polys[i].color.red,
                polys[i].color.green,
                polys[i].color.blue,
                polys[i].num_points,
                (num_verts+i));

            for (j = (polys[i].num_points-1); j >= 0; j--) {
                fprintf(fp, " %d", polys[i].points[j]);
            }

            fprintf(fp, "\n");
        }

        fprintf(fp, "\n");

        fclose(fp);
        return (0);
    } else {
        return (-1);
    }
}


int
OgreObject::saveFLOB(char *fname)
{
    float one_float;
    int i, j, temp;
    FILE *fp;


    fp = fopen(fname, "wb");

    if (fp != NULL) {
        temp = 0;
        for (i=0; i < num_polys; i++) {
            temp += polys[i].num_points;
        }

        // Write number_of_vertices
        fwrite(&num_verts, sizeof(int), 1, fp);

        // Write number_of_polygons
        fwrite(&num_polys, sizeof(int), 1, fp);

        // Write number_of_polypoints
        fwrite(&temp, sizeof(int), 1, fp);

        one_float = 1.0;

        for (i=0; i < num_verts; i++) {
            fwrite(&verts[i].x, sizeof(float), 1, fp);
            fwrite(&verts[i].y, sizeof(float), 1, fp);
            fwrite(&verts[i].z, sizeof(float), 1, fp);
            fwrite(&one_float,  sizeof(float), 1, fp);
        }

        // Write number_of_points_list
        for (i=0; i < num_polys; i++) {
            fwrite(&polys[i].num_points, sizeof(int), 1, fp);
        }

        // Write polypoint_list
        for (i=0; i < num_polys; i++) {
            for (j=0; j < polys[i].num_points; j++) {
                fwrite(&polys[i].points[j], sizeof(int), 1, fp);
            }
        }

        for (i=0; i < num_polys; i++) {
            fwrite(&polys[i].color.red, 1, 1, fp);
            fwrite(&polys[i].color.green, 1, 1, fp);
            fwrite(&polys[i].color.blue, 1, 1, fp);
        }

        fclose(fp);
        return (0);
    } else {
        return (-1);
    }
}


void
OgreObject::calc_normals(void)
{
    Vector v1, v2;
    float normal_len;
    int i, vert0, vert1, vertN;

    for (i=0; i < num_polys; i++) {
        // Compute cross-product to get poly normal
        vert0 = polys[i].points[0];
        vert1 = polys[i].points[1];
        vertN = polys[i].points[polys[i].num_points - 1];

        v1.x = verts[vert1].x - verts[vert0].x;
        v1.y = verts[vert1].y - verts[vert0].y;
        v1.z = verts[vert1].z - verts[vert0].z;

        v2.x = verts[vertN].x - verts[vert0].x;
        v2.y = verts[vertN].y - verts[vert0].y;
        v2.z = verts[vertN].z - verts[vert0].z;

        polys[i].norm_vect.x = ((v1.y * v2.z) - (v1.z * v2.y));
        polys[i].norm_vect.y = ((v1.z * v2.x) - (v1.x * v2.z));
        polys[i].norm_vect.z = ((v1.x * v2.y) - (v1.y * v2.x));

        normal_len = (polys[i].norm_vect.x * polys[i].norm_vect.x) +
                     (polys[i].norm_vect.y * polys[i].norm_vect.y) +
                     (polys[i].norm_vect.z * polys[i].norm_vect.z);

        normal_len = sqrt(normal_len);

        if (normal_len != 0.0) {
            polys[i].norm_vect.x = polys[i].norm_vect.x / normal_len;
            polys[i].norm_vect.y = polys[i].norm_vect.y / normal_len;
            polys[i].norm_vect.z = polys[i].norm_vect.z / normal_len;
        }
    }
}


void
OgreObject::calc_bsphere(void)
{
    float maxrad, temp;
    int i;

    maxrad = -1.0;

    for (i=0; i < num_verts; i++) {
        temp = (verts[i].x * verts[i].x) +
               (verts[i].y * verts[i].y) +
               (verts[i].z * verts[i].z);

        if (temp > maxrad) {
            maxrad = temp;
        }
    }

    bsphere = sqrt(maxrad);
}


int
OgreObject::backface(int face)
{
    if (polys[face].norm_vect.z >= 0) {
        // Faces away from viewer
        return 1;
    } else {
        // Faces toward viewer
        return 0;
    }
}


int
OgreObject::get_num_verts(void)
{
    return num_verts;
}


int
OgreObject::get_num_polys(void)
{
    return num_polys;
}


void
OgreObject::re_center(void)
{
    int i;
    float temp, avgx, avgy, avgz;
    float minx, maxx, miny, maxy, minz, maxz;

    minx = maxx = verts[0].x;
    miny = maxy = verts[0].y;
    minz = maxz = verts[0].z;

    for (i=1; i < num_verts; i++) {
        temp = verts[i].x;
        if (temp < minx) {
            minx = temp;
        } else if (temp > maxx) {
            maxx = temp;
        }

        temp = verts[i].y;
        if (temp < miny) {
            miny = temp;
        } else if (temp > maxy) {
            maxy = temp;
        }

        temp = verts[i].z;
        if (temp < minz) {
            minz = temp;
        } else if (temp > maxz) {
            maxz = temp;
        }
    }

    avgx = (minx + maxx) / 2.0;
    avgy = (miny + maxy) / 2.0;
    avgz = (minz + maxz) / 2.0;

    trans(-avgx, -avgy, -avgz);
}


void
OgreObject::re_floor(void)
{
    int i;
    float temp, minz, maxz;

    minz = maxz = verts[0].z;

    for (i=1; i < num_verts; i++) {
        temp = verts[i].z;
        if (temp < minz) {
            minz = temp;
        } else if (temp > maxz) {
            maxz = temp;
        }
    }

    if (minz > 0.0) {
        trans(0, 0, -minz);
    } else {
        trans(0, 0, minz);
    }
}


void
OgreObject::optimize_verts(float threshold)
{
    int i, j, verts_left;
    float thresh_squared;

    thresh_squared = SQUARE(threshold);

    verts_left = num_verts;

    for (i=0; i < verts_left; i++) {
        for (j=0; j < i; j++) {
            if (SQUARE(verts[i].x - verts[j].x) +
                SQUARE(verts[i].y - verts[j].y) +
                SQUARE(verts[i].z - verts[j].z) < thresh_squared) {
                // Change all instances of vertex I to vertex J
                // Decrement num_verts
                // Decrement all vertex numbers larger than I
            }
        }
    }
}

