/****************************************************************************
*  Module:  GA_rand.c                                                       *
*  Description:  A fast and effecive random number generation system using  *
*      as little of the standard library as possible to encourage maximum   *
*      portability across different computers and different compilers.      *
*      This library was originally developed to support work in Genetic     *
*      Algorithms, which may require tens of thousands of random numbers    *
*      per second.  GArand uses the r250 algorithm, and is based on an      *
*      article appearing in Doctor Dobb's Journal, May 1991.                *
****************************************************************************/

/* Static Variables */
static unsigned int GArand_buffer[250];
static int GArand_index;
static unsigned long bootstrap_seed;


/* Function Prototypes */
void GArand_seed(unsigned int seed);
unsigned int GArand(void);
unsigned int bootstrap_rand(void);


unsigned int
bootstrap_rand(void)
{
    /* Simple random number generator to "bootstrap" the more
     * effective r250 algorithm. */

    bootstrap_seed = (bootstrap_seed * 5709421L) + 1;

    return ((unsigned int) (bootstrap_seed & 0x0000FFFF));
}


void
GArand_seed(unsigned int seed)
{
    int i,k;
    unsigned int msb;
    unsigned int mask;

    bootstrap_seed = seed;
    GArand_index = 0;

    for (i=0; i < 250; i++) {
        GArand_buffer[i] = (((bootstrap_rand() & 0x00FF) << 8) |
                             (bootstrap_rand() & 0x00FF));
    }

    msb  = 0x8000;
    mask = 0xFFFF;

    for (i=0; i < 16; i++) {
        k = 11 * i + 3;              /* Select a word to operate on        */
        GArand_buffer[k] &= mask;    /* Turn off bits left of the diagonal */
        GArand_buffer[k] |= msb;     /* Turn on the diagonal bit           */
        mask >>= 1;
        msb  >>= 1;
    }
}


unsigned int
GArand(void)
{
    int j;
    unsigned int new_rand;

    if (GArand_index >= 147) {
        j = GArand_index - 147;    /* Wrap pointer around             */
    } else {
        j = GArand_index + 103;
    }

    new_rand = GArand_buffer[GArand_index] ^ GArand_buffer[j];
    GArand_buffer[GArand_index] = new_rand;

    if (GArand_index >= 249) {
        GArand_index = 0;          /* Increment pointer for next time */
    } else {
        GArand_index++;
    }

    return (new_rand);
}

