#include "viewwidget.h"
#include "viewwidget.moc"

#include <math.h>
#include <misc.h>



ViewWidget::ViewWidget(QWidget *p,char *n) :
	QWidget(p,n)  , view()
{
	printf("Create ViewWidget\n");

	qp = new QPainter(this);

	backcolor = QColor(200,200,200);
	drawcolor = QColor(0,0,0);
	selcolor = QColor(255,255,255);
	boxcolor = QColor(200,200,255);
	gridcolor = QColor(100,100,200);

	peye = Vector3(0,0,200);
	vview = Vector3(0,0,-1);
	vup = Vector3(0,1,0);
	vright = Vector3(1,0,0);

	setFOV(0.1);
/*
	setFrameStyle(WinPanel | Sunken);
	setLineWidth(2);
	setMidLineWidth(2);
*/
	xo = 0;yo = 0;
}

ViewWidget::~ViewWidget()
{
	delete qp;
}

int	ViewWidget::drawLine(Vector3 a,Vector3 b,Matrix44 m)
{
	Vector2	a2,b2;

	if(project(a,a2,m)) return -1;
	if(project(b,b2,m)) return -1;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	qp->drawLine((int)a2(0),(int)a2(1),(int)b2(0),(int)b2(1));

	return 0;
}

int	ViewWidget::drawLine(Vector3 a,Vector3 b,Matrix44 m,QColor color)
{
	Vector2	a2,b2;

	if(project(a,a2,m)) return -1;
	if(project(b,b2,m)) return -1;

	qp->setPen(color);

	qp->drawLine((int)a2(0),(int)a2(1),(int)b2(0),(int)b2(1));

	return 0;
}

int	ViewWidget::drawCross(Vector3 a,Matrix44 m)
{
	Vector2	a2;

	if(project(a,a2,m)) return -1;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	qp->drawLine((int)a2(0) - 10,(int)a2(1) - 10,(int)a2(0) + 10,(int)a2(1) + 10);
	qp->drawLine((int)a2(0) + 10,(int)a2(1) - 10,(int)a2(0) - 10,(int)a2(1) + 10);

	return 0;
}

int		ViewWidget::drawTriangle(Vector3 a,Vector3 b,Vector3 c,Matrix44 m)
{
	Vector2	a2,b2,c2;

	if(project(a,a2,m)) return -1;
	if(project(b,b2,m)) return -1;
	if(project(c,c2,m)) return -1;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	qp->drawLine((int)a2(0),(int)a2(1),(int)b2(0),(int)b2(1));
	qp->drawLine((int)b2(0),(int)b2(1),(int)c2(0),(int)c2(1));
	qp->drawLine((int)c2(0),(int)c2(1),(int)a2(0),(int)a2(1));

	return 0;
}

int	ViewWidget::drawTrapezium(Vector3 a,Vector3 b,Vector3 c,Matrix44 m)
{
	Vector3		d;
	Vector2	a2,b2,c2,d2;

	d = b;
	d -= a;
	d += c;

	if(project(a,a2,m)) return -1;
	if(project(b,b2,m)) return -1;
	if(project(c,c2,m)) return -1;
	if(project(d,d2,m)) return -1;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	qp->drawLine((int)a2(0),(int)a2(1),(int)b2(0),(int)b2(1));
	qp->drawLine((int)b2(0),(int)b2(1),(int)c2(0),(int)c2(1));
	qp->drawLine((int)c2(0),(int)c2(1),(int)d2(0),(int)d2(1));
	qp->drawLine((int)d2(0),(int)d2(1),(int)a2(0),(int)a2(1));

	return 0;
}

int	ViewWidget::drawCircle(Vector3 m,Vector3 a,Vector3 b,Matrix44 ma)
{
	Vector3		d,n;
	Vector2	n2,d2;
	int		t;
	double		alpha;

	alpha = (double)-1 / 16 * PI;
	n = m;
	n += a * cos(alpha) + b * sin(alpha);
	if(project(n,n2,ma)) return -1;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	for(t = 0;t < 32;t++)
	{
		alpha = (double)t / 16 * PI;
		d = m;
		d += a * cos(alpha) + b * sin(alpha);

		if(project(d,d2,ma)) return -1;

		qp->drawLine((int)n2(0),(int)n2(1),(int)d2(0),(int)d2(1));

		n2 = d2;
	}

	return 0;
}

int		ViewWidget::drawSemiCircle(Vector3 m,Vector3 a,Vector3 b,Matrix44 ma)
{
	Vector3		d,n;
	Vector2		n2,d2;
	int		t;
	double		alpha;

	alpha = (double)-1 / 16 * PI;
	n = m;
	n += a * cos(alpha) + b * sin(alpha);
	if(project(n,n2,ma)) return -1;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	for(t = 0;t < 16;t++)
	{
		alpha = (double)t / 16 * PI;
		d = m;
		d += a * cos(alpha) + b * sin(alpha);

		if(project(d,d2,ma)) return -1;

		qp->drawLine((int)n2(0),(int)n2(1),(int)d2(0),(int)d2(1));

		n2 = d2;
	}

	return 0;
}

int		ViewWidget::drawSphere(Vector3 m,Vector3 a,Vector3 b,Vector3 c,Matrix44 ma)
{
	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	drawCircle(m,a,b,ma);
	drawCircle(m,b,c,ma);
	drawCircle(m,c,a,ma);

	return 0;
}

int		ViewWidget::drawSemiSphere(Vector3 m,Vector3 a,Vector3 b,Vector3 c,Matrix44 ma)
{
	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	drawCircle(m,a,b,ma);
	drawSemiCircle(m,b,c,ma);
	drawSemiCircle(m,c,a,ma);

	return 0;
}

int		ViewWidget::drawBox(Vector3 m,Vector3 a,Vector3 b,Vector3 c,Matrix44 ma)
{
	Vector3		v[8];

	v[0] = v[1] = v[2] = v[3] = v[4] = v[5] = v[6] = v[7] = m;
	v[0] -= (a + b + c);
	v[1] -= (a - b + c);
	v[2] += (a + b - c);
	v[3] += (a - b - c);
	v[4] -= (a + b - c);
	v[5] -= (a - b - c);
	v[6] += (a + b + c);
	v[7] += (a - b + c);

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	drawLine(v[0],v[1],ma);
	drawLine(v[1],v[2],ma);
	drawLine(v[2],v[3],ma);
	drawLine(v[3],v[0],ma);
	drawLine(v[4],v[5],ma);
	drawLine(v[5],v[6],ma);
	drawLine(v[6],v[7],ma);
	drawLine(v[7],v[4],ma);
	drawLine(v[0],v[4],ma);
	drawLine(v[1],v[5],ma);
	drawLine(v[2],v[6],ma);
	drawLine(v[3],v[7],ma);

	return 0;
}

int		ViewWidget::drawCone(Vector3 m,Vector3 a,Vector3 b,Vector3 c,double d,Matrix44 ma)
{
	Vector3		v[8];

	v[0] = v[1] = v[2] = v[3] = v[4] = v[5] = v[6] = v[7] = m;
	v[0] += -a + c;
	v[1] += b + c;
	v[2] += a + c;
	v[3] += -b + c;
	v[4] += -a * d - c;
	v[5] += b * d - c;
	v[6] += a * d - c;
	v[7] += -b * d - c;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	drawLine(v[0],v[4],ma);
	drawLine(v[1],v[5],ma);
	drawLine(v[2],v[6],ma);
	drawLine(v[3],v[7],ma);

	drawCircle(c,a,b,ma);
	drawCircle(m,a * (1 + d) / 2,b * (1 + d) / 2,ma);
	drawCircle(-c,a * d,b * d,ma);

	return 0;
}

int		ViewWidget::drawCylinder(Vector3 m,Vector3 a,Vector3 b,Vector3 c,Matrix44 ma)
{
	Vector3		v[8];

	v[0] = v[1] = v[2] = v[3] = v[4] = v[5] = v[6] = v[7] = m;
	v[0] += -a - c;
	v[1] += b - c;
	v[2] += a - c;
	v[3] += -b - c;
	v[4] += -a + c;
	v[5] += b + c;
	v[6] += a + c;
	v[7] += -b + c;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	drawLine(v[0],v[4],ma);
	drawLine(v[1],v[5],ma);
	drawLine(v[2],v[6],ma);
	drawLine(v[3],v[7],ma);

	drawCircle(-c,a,b,ma);
	drawCircle(m,a,b,ma);
	drawCircle(c,a,b,ma);

	return 0;
}

int		ViewWidget::drawTorus(Vector3 m,Vector3 a,Vector3 b,Vector3 c,double d,Matrix44 ma)
{
	Vector3		v[8];

	v[0] = v[1] = v[2] = v[3] = v[4] = v[5] = v[6] = v[7] = m;
	v[0] += -a - c;
	v[1] += a - c;
	v[2] += a + c;
	v[3] += -a + c;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	drawCircle(m + b,a,c,ma);
	drawCircle(m - b,a,c,ma);
	drawCircle(m,a / a.length() * (a.length() - d),c / c.length() * (c.length() - d),ma);
	drawCircle(m,a / a.length() * (a.length() + d),c / c.length() * (c.length() + d),ma);

	drawCircle(m + a,a / a.length() * (a.length() - d),b,ma);
	drawCircle(m - a,a / a.length() * (a.length() - d),b,ma);
	drawCircle(m + c,c / c.length() * (c.length() - d),b,ma);
	drawCircle(m - c,c / c.length() * (c.length() - d),b,ma);

	return 0;
}

int		ViewWidget::drawIcon(Vector3 v,QPixmap p,int x,int y,Matrix44 m)
{
	Vector2	v2;

	if(project(v,v2,m)) return -1;

	qp->drawPixmap((int)v2(0) + x,(int)v2(1) + y,p);

	return 0;
}

int		ViewWidget::drawSymbol(Vector3 v,const char *name,Matrix44 m)
{
	extern	const char	*lightpoint_xpm;
	extern	const char	*spotlight_xpm;

	if(strcmp(name,"lightpoint") == 0)
		return drawIcon(v,lightpoint_xpm,12,12,m);
	if(strcmp(name,"spotlight") == 0)
		return drawIcon(v,spotlight_xpm,12,12,m);

	return -1;
}

void		ViewWidget::draw()
{
	Matrix44	m;
	Vector3		vm(0,0,0),vx(1,0,0),vy(0,1,0),vz(0,0,1),a,b;
	int	t;

	m.unify();

	qp->fillRect(1,1,width() - 2,height() - 2,backcolor);
	qp->setPen(QColor(0,0,0));
	qp->drawLine(0,0,width(),0);
	qp->drawLine(0,0,0,height());
	qp->drawLine(1,1,width() - 2,1);
	qp->setPen(QColor(255,255,255));
	qp->drawLine(width() - 1,0,width() - 1,height() - 1);
	qp->drawLine(0,height() - 1,width() - 1,height() - 1);

	qp->setPen(gridcolor);

	switch(projmode)
	{
		case SPACE:
			for(t = -10;t <= 10;t++)
			{
				a = vx * (double)t + vz * -10;
				b = vx * (double)t + vz * 10;
				drawLine(a,b,m,gridcolor);
				a = vz * (double)t + vx * -10;
				b = vz * (double)t + vx * 10;
				drawLine(a,b,m,gridcolor);
			}

			drawLine(vm,vx * 10,m,QColor(200,0,0));

			drawLine(vx * 10 + vx + vz,vx * 10 - vz,m,QColor(100,0,0));
			drawLine(vx * 10 + vx - vz,vx * 10 + vz,m,QColor(100,0,0));

			drawLine(vm,vy * 10,m,QColor(0,200,0));

			drawLine(vy * 10 + vy + vx,vy * 10 + vy * 0.5,m,QColor(0,100,0));
			drawLine(vy * 10 +      vx,vy * 10 + vy * 0.5,m,QColor(0,100,0));
			drawLine(vy * 10 + vy * 0.5,vy * 10 + vy * 0.5 - vx,m,QColor(0,100,0));

			drawLine(vm,vz * 10,m,QColor(100,0,200));

			drawLine(vz * 10 + vz + vx,vz * 10 + vx,m,QColor(50,0,100));
			drawLine(vz * 10 + vz - vx,vz * 10 - vx,m,QColor(50,0,100));
			drawLine(vz * 10 + vz - vx,vz * 10 + vx,m,QColor(50,0,100));
		break;
		case PARALLEL_XY:
			for(t = -10;t <= 10;t++)
			{
				a = vx * (double)t + vy * -10;
				b = vx * (double)t + vy * 10;
				drawLine(a,b,m,gridcolor);
				a = vy * (double)t + vx * -10;
				b = vy * (double)t + vx * 10;
				drawLine(a,b,m,gridcolor);
			}

			drawLine(vm,vx * 10,m,QColor(200,0,0));
			drawLine(vm,vy * 10,m,QColor(0,200,0));
		break;
		case PARALLEL_ZY:
			for(t = -10;t <= 10;t++)
			{
				a = vz * (double)t + vy * -10;
				b = vz * (double)t + vy * 10;
				drawLine(a,b,m,gridcolor);
				a = vy * (double)t + vz * -10;
				b = vy * (double)t + vz * 10;
				drawLine(a,b,m,gridcolor);
			}

			drawLine(vm,vz * 10,m,QColor(100,0,200));
			drawLine(vm,vy * 10,m,QColor(0,200,0));
		break;
		case PARALLEL_XZ:
			for(t = -10;t <= 10;t++)
			{
				a = vx * (double)t + vz * -10;
				b = vx * (double)t + vz * 10;
				drawLine(a,b,m,gridcolor);
				a = vz * (double)t + vx * -10;
				b = vz * (double)t + vx * 10;
				drawLine(a,b,m,gridcolor);
			}

			drawLine(vm,vz * 10,m,QColor(100,0,200));
			drawLine(vm,vx * 10,m,QColor(200,0,0));
		break;
	}

	qp->setPen(drawcolor);
	
	if(worldptr) worldptr->draw(this,m);
}

void		ViewWidget::setWorld(world *ww)
{
	worldptr = ww;
	draw();
}

void		ViewWidget::rotateView(double x,double y,double z)
{
	Vector3		vr(x,y,z);
	Matrix44	m;
	Vector4		v,u,r;

	m.rotateVector(vr);
	v = vview;
	u = vup;
	r = vright;

	v *= m;
	u *= m;
	r *= m;

	vview = v;
	vup = u;
	vright = r;

	calcVectors();
}

void		ViewWidget::rotate0View(double x,double y,double z)
{
	Vector3		vr(x,y,z);
	Matrix44	m;
	Vector4		v4;

	m.rotateVector(vr);
	v4 = peye;
	v4 *= m;
	peye = v4;

	rotateView(x,y,z);
}

void		ViewWidget::moveView(double x,double y,double z)
{
	Vector3		v(x,y,z);

	setEye(peye + v);
}

void		ViewWidget::changeFOV(double d)
{
	setFOV(fov + d);
}

void		ViewWidget::setViewMode(int vm)
{
	viewmode = vm;
}

void		ViewWidget::setProjectionMode(int pm)
{
	projmode = pm;
	draw();
}

void		ViewWidget::setSelected(base *bp)
{
	selectedptr = bp;
	draw();
}

void		ViewWidget::mouseMoveEvent(QMouseEvent *me)
{
	int		dx,dy;
	Vector3		v(1,1,1);

	dx = me->x() - xo;
	dy = me->y() - yo;

	switch(viewmode)
	{
		case 1:
			changeFOV(((double)(dx + dy)) / 1000.0);
		break;
		case 2:
			if(me->state() == LeftButton)
				moveView((double)dx / 10.0,(double)dy / 10.0,0);
			else
				moveView(0,(double)dy / 10.0,(double)dx / 10.0);
		break;
		case 3:
			if(me->state() == LeftButton)
				rotate0View((double)dx / 1800 * PI,0,0);
			else if(me->state() == RightButton)
				rotate0View(0,0,(double)dx / 1800 * PI);
			else
				rotate0View(0,(double)dx / 1800 * PI,0);
		break;
		case 4: // Object scale
			if(!selectedptr || selectedptr->getType() == NUM_WORLD) return;

			if(me->state() == LeftButton)
				v(0) = exp((double)dx / -100);
			else if(me->state() == RightButton)
				v(1) = exp((double)dx / -100);
			else
				v(2) = exp((double)dx / -100);
			((dim*)selectedptr)->scale(v);
			draw();
		break;
		case 5: // Object translate
			if(!selectedptr || selectedptr->getType() == NUM_WORLD) return;
			if(me->state() == LeftButton)
				((dim*)selectedptr)->translate(Vector3((double)dx / 10.0,(double)dy / 10.0,0));
			else
				((dim*)selectedptr)->translate(Vector3(0,(double)dy / 10.0,(double)dx / 10.0));
			draw();
		break;
		case 6: // Object rotate
			if(!selectedptr || selectedptr->getType() == NUM_WORLD) return;
			if(me->state() == LeftButton)
				((dim*)selectedptr)->rotate(Vector3((double)dx / 1800 * PI,0,0));
			else if(me->state() == RightButton)
				((dim*)selectedptr)->rotate(Vector3(0,0,(double)dx / 1800 * PI));
			else
				((dim*)selectedptr)->rotate(Vector3(0,(double)dx / 1800 * PI,0));
			draw();
		break;
	}
	draw();

	xo = me->x();
	yo = me->y();
}

void		ViewWidget::mousePressEvent(QMouseEvent *me)
{
	xo = me->x();
	yo = me->y();
}





void		ViewWidget::paintEvent(QPaintEvent*)
{
	setScreen(width(),height());
	createScreen(width(),height());

	draw();
}




#ifdef TEST_VIEW

int	main(int argc,char **argv)
{
	ViewWidget	*win;

	mykapp = new KApplication(argc,argv,"k3de");
	win = new ViewWidget();

	mykapp->setMainWidget(win);

	win->resize(500,345);
	win->setMaximumSize(1000,700);
	win->setMinimumSize(500,200);

	win->show();

	return mykapp->exec();
}

#endif

