/*****************************************************************************
* Abstraction of the RLE image format for input and output.		     *
******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                *
******************************************************************************
* Written by:  Bassarab Dmitri & Plavnik Michael       Ver 0.2, Apr. 1995    *
*****************************************************************************/

#ifdef HAVE_URT_RLE
#define NO_DECLARE_MALLOC /* For rle.h */
#include <rle.h>
#include <rle_raw.h>
#endif /* HAVE_URT_RLE */

#include "program.h"

#ifdef HAVE_URT_RLE
static rle_hdr *Header;
static rle_pixel **Rows;
#endif /* HAVE_URT_RLE */

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Opens output image file in RLE format.                                   M
*   Creates file header.                                                     M
*                                                                            *
* PARAMETERS:                                                                M
*   argv:      Pointer to the command line parameters.                       M
*   XSize:     X dimension of the image.                                     M
*   YSize:     Y dimension of the image.                                     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   RLEOpenFile, RLE, output image                                           M
*****************************************************************************/
void RLEOpenFile(char **argv, int XSize, int YSize)
{
#ifdef HAVE_URT_RLE
#ifdef URT_OLD_COMPAT
    Header = &rle_dflt_hdr;
    Header -> rle_file = rle_open_f("irender", NULL, "w");
#else
    Header = rle_hdr_init((rle_hdr *) NULL);
    rle_names(Header, *argv, NULL, 0);
    Header -> rle_file = rle_open_f(Header -> cmd, NULL, "w");
#endif /* URT_OLD_COMAP */
    rle_addhist(argv, (rle_hdr *) NULL, Header);
    RLE_SET_BIT(*Header, RLE_ALPHA);
    Header -> alpha = 1;
    Header -> xmax = --XSize;
    Header -> ymax = --YSize;
    rle_put_setup(Header);
    rle_row_alloc(Header, &Rows);
#else
    fprintf(stderr, "Utah raster tool kit is not supported\n");
    exit(1);
#endif /* HAVE_URT_RLE */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Outputs given line of color pixels and alpha correction values to the    M
*   ouput image file.                                                        M
*                                                                            *
* PARAMETERS:                                                                M
*   Alpha:  Array of alpha values.                                           M
*   Pixels: Array of color pixels.                                           M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   RLEPutLine, RLE, output image                                            M
*****************************************************************************/
void RLEPutLine(ByteType *Alpha, PixelStruct *Pixels)
{
#ifdef HAVE_URT_RLE
    int x;

    for (x = 0; x <= Header -> xmax; x++) {
        PixelStruct
	    *p = &Pixels[x];

        Rows[RLE_ALPHA][x] = Alpha[x];
        Rows[RLE_RED][x] = p -> r;
        Rows[RLE_GREEN][x] = p -> g;
        Rows[RLE_BLUE][x] = p -> b;
    }
    rle_putrow(Rows, Header -> xmax + 1, Header);
#endif /* HAVE_URT_RLE */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Closes output image file.                                                M
*                                                                            *
* PARAMETERS:                                                                M
*   None                                                                     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   RLECloseFile, RLE, output image                                          M
*****************************************************************************/
void RLECloseFile(void)
{
#ifdef HAVE_URT_RLE
    rle_puteof(Header);
#endif /* HAVE_URT_RLE */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Loads image file in RLE format.                                          M
*                                                                            *
* PARAMETERS:                                                                M
*   File:  Name of the image file.                                           M
*                                                                            *
* RETURN VALUE:                                                              M
*   ImageStruct *:  Pointer to dynamicaly created image.                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   RLELoadImage, RLE, image file, texture                                   M
*****************************************************************************/
ImageStruct *RLELoadImage(char *File)
{
#ifdef HAVE_URT_RLE
    rle_hdr Header;
    rle_pixel **Rows;
    ImageStruct *PImage;
    PixelStruct *p;
    int Error, x, y;

    Header.rle_file = rle_open_f_noexit("RleLoadImage", File, "r");
    if (!Header.rle_file)
        return NULL;
    if (Error = rle_get_setup(&Header)) {
        rle_get_error(Error, "RleLoadImage", File);
        return NULL;
    }
    rle_row_alloc(&Header, &Rows);
    PImage = MALLOC(ImageStruct, 1);
    PImage -> xSize = Header.xmax - Header.xmin;
    PImage -> ySize = Header.ymax - Header.ymin;
    PImage -> data = p = MALLOC(PixelStruct,
				(PImage -> ySize + 1) * (PImage -> xSize + 1));
    for (y = 0; y <= PImage -> ySize; y++) {
        rle_getrow(&Header, Rows);
        for (x = 0; x <= PImage -> xSize; x++, p++) {
            p -> r = Rows[RLE_RED][x];
            p -> g = Rows[RLE_GREEN][x];
            p -> b = Rows[RLE_BLUE][x];
        }
    }
    return PImage;
#else
    fprintf(stderr, "Utah raster tool kit is not supported\n");
    exit(1);
#endif /* HAVE_URT_RLE */
}
