/* Copyright (c) 1994 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Author: Timothy Rowley */

/* Module for 8 bit rendering */

/* {{{ Includes */

#include <stdlib.h>
#include <stdio.h>
#include "mgP.h"
#include "mgx11P.h"

/* }}} */

/* {{{ externs for dithering and dithering code */

extern int colorlevels;
extern int mgx11divN[256];
extern int mgx11modN[256];
extern int mgx11magic[16][16];
extern unsigned long mgx11colors[216];
extern int mgx11multab[256];

#define DMAP(v,x,y)  (mgx11modN[v]>mgx11magic[x][y] ? mgx11divN[v] + 1 : mgx11divN[v])

#ifdef __GNUC__
inline
#endif
static unsigned long
ditherRGB(int x, int y, int *color)
{
  int xdith, ydith;

  xdith = x % 16;
  ydith = y % 16;
  return mgx11colors[DMAP(color[0], xdith, ydith) +
                                mgx11multab[DMAP(color[1], xdith, ydith) +
                                  mgx11multab[DMAP(color[2], xdith, ydith)]]];
}

#ifdef __GNUC__
inline
#endif
static unsigned long
ditherRGBvals(int x, int y, int r, int g, int b)
{
  int xdith, ydith;

  xdith = x % 16;
  ydith = y % 16;
  return mgx11colors[DMAP(r, xdith, ydith) +
              mgx11multab[DMAP(g, xdith, ydith) +
	      mgx11multab[DMAP(b, xdith, ydith)]]];
}

/* }}} */

static endPoint *mug=NULL;
static mugSize = 0;

void
Xmgr_8clear(unsigned char *buf, float *zbuf, int zwidth, int width, int height,
	    int *color, int flag, int fullclear, int xmin, int ymin, int xmax, int ymax)
{
    int i, length, x, col = ditherRGB(0, 0, color);
    unsigned char *ptr;
    int pos;

    if (mug==NULL)
    {
	mug = (endPoint *)malloc(sizeof(endPoint)*height);
	mugSize = height;
    }
    if (height>mugSize)
    {
	mug = (endPoint *)realloc(mug, sizeof(endPoint)*height);
	mugSize = height;
    }

    if (fullclear)
    {
        memset(buf, col, width*height);
        if (flag)
            for (i=0; i<zwidth*height; i++)
                zbuf[i] = 1.0;
        return;
    }
    xmin = MAX(xmin,0);
    length = MIN(width-1,xmax)-xmin+1;
    ymin = MAX(ymin,0);
    ymax = MIN(height-1,ymax);
    for (i=ymin, ptr=buf+ymin*width+xmin; i<=ymax; i++, ptr+=width)
	memset(ptr, col, length);
    if (flag)
	for (i=ymin; i<=ymax; i++)
	{
	    pos = i*width+xmin;
	    for (x=0; x<length; x++)
		zbuf[pos+x] = 1.0;
	}
}

/* {{{ Internal wide line functions */

static void
wideline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
	 int x1, int y1,
	 int x2, int y2, int lwidth, int *color)
{
    register int d, x, y, ax, ay, sx, sy, dx, dy;
    int end, i;
    unsigned char col=ditherRGB(0,0,color);

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);
    sy = SGN(dy);

    x = x1;
    y = y1;

    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if ((x>=0) && (x<width)) 
	    {
		for (i=MAX(0,y-lwidth/2), end=MIN(height-1,y-lwidth/2+lwidth); i<end; i++)
		    buf[i*width+x] = col;
	    }
	    if (x==x2) break;
	    if (d>=0)
	    {
		y += sy;
		d -= ax;
	    }
	    x += sx;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if ((y>=0) && (y<height))
	    {
		for (i=MAX(0,x-lwidth/2), end=MIN(width-1,x-lwidth/2+lwidth); i<end; i++)
		    buf[y*width+i] = col;
	    }
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		d -= ay;
	    }
	    y += sy;
	    d += ax;
	}
    }
}

static void
wideDline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
	 int x1, int y1,
	 int x2, int y2, int lwidth, int *color)
{
    register int d, x, y, ax, ay, sx, sy, dx, dy;
    int end, i;

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);
    sy = SGN(dy);

    x = x1;
    y = y1;

    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if ((x>=0) && (x<width)) 
	    {
		for (i=MAX(0,y-lwidth/2), end=MIN(height-1,y-lwidth/2+lwidth); i<end; i++)
		    buf[i*width+x] =  ditherRGB(x, i, color);
	    }
	    if (x==x2) break;
	    if (d>=0)
	    {
		y += sy;
		d -= ax;
	    }
	    x += sx;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if ((y>=0) && (y<height))
	    {
		for (i=MAX(0,x-lwidth/2), end=MIN(width-1,x-lwidth/2+lwidth); i<end; i++)
		    buf[y*width+i] =  ditherRGB(i, y, color);
	    }
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		d -= ay;
	    }
	    y += sy;
	    d += ax;
	}
    }
}

wideZline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
	 int x1, int y1, float z1,
	 int x2, int y2, float z2, int lwidth, int *color)
{
    register int d, x, y, ax, ay, sx, sy, dx, dy;
    float delta=0, z;
    int jumps, total;
    int end, i;
    unsigned char col = ditherRGB(0, 0, color);

    z1 -= _mgc->zfnudge;
    z2 -= _mgc->zfnudge;

    dx = x2-x1;
    dy = y2-y1;
    total = ABS(dx)+ABS(dy);

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);
    sy = SGN(dy);

    x = x1;
    y = y1;
    z = z1;
    if (total)
        delta = (z2-z1)/total;
    jumps = 0;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if ((x>=0) && (x<width)) 
	    {
		for (i=MAX(0,y-lwidth/2), end=MIN(height-1,y-lwidth/2+lwidth); i<end; i++)
		{
		    if ((z<zbuf[i*width+x]) && (z>-1.0))
		    {
		    	buf[i*width+x] =  col;
		    	zbuf[i*width+x] = z;
		    }
		}
	    }
	    if (x==x2) break;
	    if (d>=0)
	    {
		y += sy;
		z += delta;
		d -= ax;
	    }
	    x += sx;
	    z += delta;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if ((y>=0) && (y<height))
	    {
		for (i=MAX(0,x-lwidth/2), end=MIN(width-1,x-lwidth/2+lwidth); i<end; i++)
		{
		    if ((z<zbuf[y*width+i]) && (z>-1.0))
		    {
		    	buf[y*width+i] =  col;
		    	zbuf[y*width+i] = z;
		    }
		}
	    }
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		z += delta;
		d -= ay;
	    }
	    y += sy;
	    z += delta;
	    d += ax;
	}
    }
}



static void
wideDZline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
	 int x1, int y1, float z1,
	 int x2, int y2, float z2, int lwidth, int *color)
{
    register int d, x, y, ax, ay, sx, sy, dx, dy;
    float delta=0, z;
    int jumps, total;
    int end, i;

    z1 -= _mgc->zfnudge;
    z2 -= _mgc->zfnudge;

    dx = x2-x1;
    dy = y2-y1;
    total = ABS(dx)+ABS(dy);

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);
    sy = SGN(dy);

    x = x1;
    y = y1;
    z = z1;
    if (total)
        delta = (z2-z1)/total;
    jumps = 0;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if ((x>=0) && (x<width)) 
	    {
		for (i=MAX(0,y-lwidth/2), end=MIN(height-1,y-lwidth/2+lwidth); i<end; i++)
		{
		    if ((z<zbuf[i*width+x]) && (z>-1.0))
		    {
		    	buf[i*width+x] =  ditherRGB(x, i, color);
		    	zbuf[i*width+x] = z;
		    }
		}
	    }
	    if (x==x2) break;
	    if (d>=0)
	    {
		y += sy;
		z += delta;
		d -= ax;
	    }
	    x += sx;
	    z += delta;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if ((y>=0) && (y<height))
	    {
		for (i=MAX(0,x-lwidth/2), end=MIN(width-1,x-lwidth/2+lwidth); i<end; i++)
		{
		    if ((z<zbuf[y*width+i]) && (z>-1.0))
		    {
		    	buf[y*width+i] =  ditherRGB(i, y, color);
		    	zbuf[y*width+i] = z;
		    }
		}
	    }
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		z += delta;
		d -= ay;
	    }
	    y += sy;
	    z += delta;
	    d += ax;
	}
    }
}

static void
wide8DGline(unsigned char *buf, float *zbuf, int zwidth, 
	      int width, int height, 
	     int x1, int y1, int r1, int g1, int b1,
	     int x2, int y2, int r2, int g2, int b2, int lwidth)
{
    register int d, x, y, ax, ay, sx, dx, dy;
    double r, g, b, rdelta, gdelta, bdelta;
    int total, i, end;

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);

    x = x1;
    y = y1;
    r = r1;
    g = g1;
    b = b1;
    total = ABS(dx)+ABS(dy);
    rdelta = (r2-r1)/(double)total;
    gdelta = (g2-g1)/(double)total;
    bdelta = (b2-b1)/(double)total;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if ((x >= 0) && (x < width))
		for (i=MAX(0,y-lwidth/2), end=MIN(height-1,y-lwidth/2+lwidth); i<end; i++)
		    buf[i*width+x] =  ditherRGBvals(x, i, r, g, b);
	    if (x==x2) break;
	    if (d>=0)
	    {
		y++;
		r += rdelta; g += gdelta; b += bdelta;
		d -= ax;
	    }
	    x += sx;
	    r += rdelta; g += gdelta; b += bdelta;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if ((y>=0) && (y<height))
		for (i=MAX(0,x-lwidth/2), end=MIN(width-1,x-lwidth/2+lwidth); i<end; i++)
		    buf[y*width+i] =  ditherRGBvals(i, y, r, g, b);
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		r += rdelta; g += gdelta; b += bdelta;
		d -= ay;
	    }
	    y++;
	    r += rdelta; g += gdelta; b += bdelta;
	    d += ax;
	}
    }
}

static void
wide8DGZline(unsigned char *buf, float *zbuf, int zwidth, 
	      int width, int height, 
	     int x1, int y1, double z1, int r1, int g1, int b1,
	     int x2, int y2, double z2, int r2, int g2, int b2, int lwidth)
{
    register int d, x, y, ax, ay, sx, dx, dy;
    double delta=0, z;
    double r, g, b, rdelta, gdelta, bdelta;
    int total, i, end;

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);

    x = x1;
    y = y1;
    z = z1;
    r = r1;
    g = g1;
    b = b1;
    total = ABS(dx)+ABS(dy);
    delta = (z2-z1)/total;
    rdelta = (r2-r1)/(double)total;
    gdelta = (g2-g1)/(double)total;
    bdelta = (b2-b1)/(double)total;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if ((x >= 0) && (x < width))
		for (i=MAX(0,y-lwidth/2), end=MIN(height-1,y-lwidth/2+lwidth); i<end; i++)
		{
		    if ((z<zbuf[i*width+x]) && (z>-1.0))
		    {
		    	buf[i*width+x] =  ditherRGBvals(x, i, r, g, b);
		    	zbuf[i*width+x] = z;
		    } 
		}
	    if (x==x2) break;
	    if (d>=0)
	    {
		y++;
		z += delta;
		r += rdelta; g += gdelta; b += bdelta;
		d -= ax;
	    }
	    x += sx;
	    z += delta;
	    r += rdelta; g += gdelta; b += bdelta;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if ((y>=0) && (y<height))
	    {
		for (i=MAX(0,x-lwidth/2), end=MIN(width-1,x-lwidth/2+lwidth); i<end; i++)
		{
		    if ((z<zbuf[y*width+i]) && (z>-1.0))
		    {
		    	buf[y*width+i] =  ditherRGBvals(i, y, r, g, b);
		    	zbuf[y*width+i] = z;
		    }
		}
	    }
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		z += delta;
		r += rdelta; g += gdelta; b += bdelta;
		d -= ay;
	    }
	    y++;
	    z += delta;
	    r += rdelta; g += gdelta; b += bdelta;
	    d += ax;
	}
    }
}

/* }}} */

/* {{{ Single line functions */

void
Xmgr_8line(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
	   int x1, int y1, float z1, int x2, int y2, float z2, int lwidth,
	   int *color)
{
    int d, x, y, ax, ay, sx, dx, dy, col;
    unsigned char *ptr;

    if (lwidth > 1)
    {
	wideline(buf, zbuf, zwidth, width, height, x1, y1, x2, y2,
		   lwidth, color);
	return;
    }

    if (y2<y1)
    {
	d = y2; y2 = y1; y1 = d;
	d = x2; x2 = x1; x1 = d;
    }
    col = ditherRGB(0,0,color);
    ptr=buf+y1*width+x1;

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);

    x = x1;
    y = y1;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if (y>=height) return;
	    if ((x>=0) && (x<width) && (y>=0))
		*ptr = col;
	    if (x==x2) return;
	    if (d>=0)
	    {
		y++;
		ptr += width;
		d -= ax;
	    }
	    x += sx;
	    ptr += sx;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if (y>=height) return;
	    if ((x>=0) && (x<width) && (y>=0))
		*ptr = col;
	    if (y==y2) return;
	    if (d>=0)
	    {
		x += sx;
		ptr += sx;
		d -= ay;
	    }
	    y++;
	    ptr += width;
	    d += ax;
	}
    }
}


void
Xmgr_8Dline(unsigned char *buf, float *zbuf, int zwidth, int width, int height,
	    int x1, int y1, float z1, int x2, int y2, float z2, 
	    int lwidth, int *color)
{
    int d, x, y, ax, ay, sx, dx, dy;
    unsigned char *ptr;

    if (lwidth > 1)
    {
	wideDline(buf, zbuf, zwidth, width, height, x1, y1, x2, y2,
		   lwidth, color);
	return;
    }

    if (y2<y1)
    {
	d = y2; y2 = y1; y1 = d;
	d = x2; x2 = x1; x1 = d;
    }
    ptr=buf+y1*width+x1;

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);

    x = x1;
    y = y1;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if (y>=height) return;
	    if ((x>=0) && (x<width) && (y>=0))
		*ptr = ditherRGB(x, y, color);
	    if (x==x2) return;
	    if (d>=0)
	    {
		y++;
		ptr += width;
		d -= ax;
	    }
	    x += sx;
	    ptr += sx;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if (y>=height) return;
	    if ((x>=0) && (x<width) && (y>=0))
		*ptr = ditherRGB(x, y, color);
	    if (y==y2) return;
	    if (d>=0)
	    {
		x += sx;
		ptr += sx;
		d -= ay;
	    }
	    y++;
	    ptr += width;
	    d += ax;
	}
    }
}


void
Xmgr_8Zline(unsigned char *buf, float *zbuf, int zwidth, int width, int height,
	    int x1, int y1, float z1,
	    int x2, int y2, float z2, int lwidth, int *color)
{
    register int d, x, y, ax, ay, sx, dx, dy, col;
    unsigned char *ptr;
    float *zptr;
    double delta=0, z;

    if (lwidth > 1)
    {
	wideZline(buf, zbuf, zwidth, width, height, x1, y1, z1, x2, y2,
		   z2, lwidth, color);
	return;
    }

    if (y2<y1)
    {
	d = y2; y2=y1; y1=d;
	d = x2; x2=x1; x1=d;
	z = z2; z2=z1; z1=z;
    }
    ptr = buf+y1*width+x1;
    zptr = zbuf+y1*width+x1;
    col = ditherRGB(0,0,color);

    z1 -= _mgc->zfnudge;
    z2 -= _mgc->zfnudge;

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);

    x = x1;
    y = y1;
    z = z1;
    delta = (z2-z1)/(ABS(dx)+ABS(dy));
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if (y>=height) return;
	    if ((x>=0) && (x<width) && (y>=0) && (z<*zptr) && (z>-1.0))
	    {
		*ptr = col;
		*zptr = z;
	    }
	    if (x==x2) break;
	    if (d>=0)
	    {
		y++;
		z += delta;
		ptr += width;
		zptr += width;
		d -= ax;
	    }
	    x += sx;
	    z += delta;
	    ptr += sx;
	    zptr += sx;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if (y>=height) return;
	    if ((x>=0) && (x<width) && (y>=0) && (z<*zptr) && (z>-1.0))
	    {
		*ptr = col;
		*zptr = z;
	    }
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		z += delta;
		ptr += sx;
		zptr += sx;
		d -= ay;
	    }
	    y++;
	    z += delta;
	    ptr += width;
	    zptr += width;
	    d += ax;
	}
    }
}

void
Xmgr_8DZline(unsigned char *buf, float *zbuf, int zwidth, 
	     int width, int height, int x1, int y1, float z1,
	     int x2, int y2, float z2, int lwidth, int *color)
{
    register int d, x, y, ax, ay, sx, dx, dy;
    unsigned char *ptr;
    float *zptr;
    double delta=0, z;

    if (lwidth > 1)
    {
	wideDZline(buf, zbuf, zwidth, width, height, x1, y1, z1, x2, y2,
		   z2, lwidth, color);
	return;
    }

    z1 -= _mgc->zfnudge;
    z2 -= _mgc->zfnudge;

    if (y2<y1)
    {
	d = y2; y2=y1; y1=d;
	d = x2; x2=x1; x1=d;
	z = z2; z2=z1; z1=z;
    }
    ptr = buf+y1*width+x1;
    zptr = zbuf+y1*width+x1;

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);

    x = x1;
    y = y1;
    z = z1;
    delta = (z2-z1)/(ABS(dx)+ABS(dy));
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if (y>=height) return;
	    if ((x>=0) && (x<width) && (y>=0) && (z<*zptr) && (z>-1.0))
	    {
		*ptr = ditherRGB(x, y, color);
		*zptr = z;
	    }
	    if (x==x2) break;
	    if (d>=0)
	    {
		y++;
		z += delta;
		ptr += width;
		zptr += width;
		d -= ax;
	    }
	    x += sx;
	    z += delta;
	    ptr += sx;
	    zptr += sx;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if (y>=height) return; 
	    if ((x>=0) && (x<width) && (y>=0) && (z<*zptr) && (z>-1.0))
	    {
		*ptr = ditherRGB(x, y, color);
		*zptr = z;
	    }
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		z += delta;
		ptr += sx;
		zptr += sx;
		d -= ay;
	    }
	    y++;
	    z += delta;
	    ptr += width;
	    zptr += width;
	    d += ax;
	}
    }
}

void
Xmgr_8DGline(unsigned char *buf, float *zbuf, int zwidth, 
	     int width, int height, CPoint3 *p1, CPoint3 *p2, int lwidth)
{
    register int d, x, y, ax, ay, sx, dx, dy;
    unsigned char *ptr;
    int r1, r2, g1, g2, b1, b2;
    double r, g, b, rdelta, gdelta, bdelta;
    int total, x1, y1, x2, y2;

    if (p2->y<p1->y)
    {
	x1 = p2->x; y1 = p2->y;
	x2 = p1->x; y2 = p1->y;
	r1 = 255*p2->vcol.r; g1 = 255*p2->vcol.g; b1 = 255*p2->vcol.b;
	r2 = 255*p1->vcol.r; g2 = 255*p1->vcol.g; b2 = 255*p1->vcol.b;
    }
    else
    {
	x1 = p1->x; y1 = p1->y;
	x2 = p2->x; y2 = p2->y;
	r1 = 255*p1->vcol.r; g1 = 255*p1->vcol.g; b1 = 255*p1->vcol.b;
	r2 = 255*p2->vcol.r; g2 = 255*p2->vcol.g; b2 = 255*p2->vcol.b;
    }
    if (lwidth > 1)
    {
	wide8DGline(buf, zbuf, zwidth, width, height, x1, y1, r1, g1, b1,
		   x2, y2, r2, g2, b2, lwidth);
	return;
    }

    ptr = buf+y1*width+x1;

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);

    x = x1;
    y = y1;
    r = r1;
    g = g1;
    b = b1;
    total = ABS(dx)+ABS(dy);
    rdelta = (r2-r1)/(double)total;
    gdelta = (g2-g1)/(double)total;
    bdelta = (b2-b1)/(double)total;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if (y>=height) return;
	    if ((x>=0) && (x<width) && (y>=0))
		*ptr = ditherRGBvals(x, y, r, g, b);
	    if (x==x2) break;
	    if (d>=0)
	    {
		y++;
		r += rdelta; g += gdelta; b += bdelta;
		ptr += width;
		d -= ax;
	    }
	    x += sx;
	    r += rdelta; g += gdelta; b += bdelta;
	    ptr += sx;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if (y>=height) return; 
	    if ((x>=0) && (x<width) && (y>=0))
		*ptr = ditherRGBvals(x, y, r, g, b);
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		r += rdelta; g += gdelta; b += bdelta;
		ptr += sx;
		d -= ay;
	    }
	    y++;
	    r += rdelta; g += gdelta; b += bdelta;
	    ptr += width;
	    d += ax;
	}
    }
}

void
Xmgr_8DGZline(unsigned char *buf, float *zbuf, int zwidth, 
	      int width, int height, CPoint3 *p1, CPoint3 *p2, int lwidth)
{
    register int d, x, y, ax, ay, sx, dx, dy;
    unsigned char *ptr;
    float *zptr;
    double delta=0, z;
    int r1, r2, g1, g2, b1, b2;
    double r, g, b, rdelta, gdelta, bdelta, z1, z2;
    int total, x1, y1, x2, y2;

    if (p2->y<p1->y)
    {
	x1 = p2->x; y1 = p2->y; z1 = p2->z - _mgc->zfnudge;
	x2 = p1->x; y2 = p1->y; z2 = p1->z - _mgc->zfnudge;
	r1 = 255*p2->vcol.r; g1 = 255*p2->vcol.g; b1 = 255*p2->vcol.b;
	r2 = 255*p1->vcol.r; g2 = 255*p1->vcol.g; b2 = 255*p1->vcol.b;
    }
    else
    {
	x1 = p1->x; y1 = p1->y; z1 = p1->z - _mgc->zfnudge;
	x2 = p2->x; y2 = p2->y; z2 = p2->z - _mgc->zfnudge;
	r1 = 255*p1->vcol.r; g1 = 255*p1->vcol.g; b1 = 255*p1->vcol.b;
	r2 = 255*p2->vcol.r; g2 = 255*p2->vcol.g; b2 = 255*p2->vcol.b;
    }
    if (lwidth > 1)
    {
	wide8DGZline(buf, zbuf, zwidth, width, height, x1, y1, z1, r1, g1, b1,
		   x2, y2, z2, r2, g2, b2, lwidth);
	return;
    }
    ptr = buf+y1*width+x1;
    zptr = zbuf+y1*width+x1;

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);

    x = x1;
    y = y1;
    z = z1;
    r = r1;
    g = g1;
    b = b1;
    total = ABS(dx)+ABS(dy);
    delta = (z2-z1)/total;
    rdelta = (r2-r1)/(double)total;
    gdelta = (g2-g1)/(double)total;
    bdelta = (b2-b1)/(double)total;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if (y>=height) return;
	    if ((x>=0) && (x<width) && (y>=0) && (z<*zptr) && (z>-1.0))
	    {
		*ptr = ditherRGBvals(x, y, r, g, b);
		*zptr = z;
	    }
	    if (x==x2) break;
	    if (d>=0)
	    {
		y++;
		z += delta;
		r += rdelta; g += gdelta; b += bdelta;
		ptr += width;
		zptr += width;
		d -= ax;
	    }
	    x += sx;
	    z += delta;
	    r += rdelta; g += gdelta; b += bdelta;
	    ptr += sx;
	    zptr += sx;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if (y>=height) return; 
	    if ((x>=0) && (x<width) && (y>=0) && (z<*zptr) && (z>-1.0))
	    {
		*ptr = ditherRGBvals(x, y, r, g, b);
		*zptr = z;
	    }
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		z += delta;
		r += rdelta; g += gdelta; b += bdelta;
		ptr += sx;
		zptr += sx;
		d -= ay;
	    }
	    y++;
	    z += delta;
	    r += rdelta; g += gdelta; b += bdelta;
	    ptr += width;
	    zptr += width;
	    d += ax;
	}
    }
}

/* }}} */

/* {{{ Polygon scan-conversion functions */

static void
Xmgr_doLines(unsigned char *buf, float *zbuf, int zwidth, int width, 
	     int height, int miny, int maxy, int *color, endPoint *mug)
{
    register unsigned char *ptr;           /* pixel pointers   */
    register int i, xdith, x2;            /* dithering junk   */
    int y, x1;                           /* current line     */
    int col;

    col = ditherRGB(0, 0, color);
    for(y = miny; y <= maxy; y++)
    {
	if ((x1=mug[y].P1x) < 0)
	    x1 = 0;
	if ((x2=mug[y].P2x) >= width)
	    x2 = width-1;
	ptr = buf+width*y+x1;
	for (i=x1; i<=x2; i++, ptr++)
	    *ptr = col;
    }
}

static void
Xmgr_DdoLines(unsigned char *buf, float *zbuf, int zwidth, int width, 
	      int height, int miny, int maxy, int *color, endPoint *mug)
{
    register unsigned char *ptr;           /* pixel pointers   */
    register int i, xdith, x2;            /* dithering junk   */
    int y, x1;                           /* current line     */
    int rmod, gmod, bmod, rdiv, gdiv, bdiv, rdiv1, gdiv1, bdiv1;
    int *magic;

    rmod = mgx11modN[color[0]]; rdiv = mgx11divN[color[0]]; rdiv1 = rdiv+1;
    gmod = mgx11modN[color[1]]; gdiv = mgx11divN[color[1]]; gdiv1 = gdiv+1;
    bmod = mgx11modN[color[2]]; bdiv = mgx11divN[color[2]]; bdiv1 = bdiv+1;
    for(y = miny; y <= maxy; y++)
    {
	if ((x1=mug[y].P1x) < 0)
	    x1 = 0;
	if ((x2=mug[y].P2x) >= width)
	    x2 = width-1;
	ptr = buf+width*y+x1;
	magic = mgx11magic[y & 0x0F];
	for (i=x1; i<=x2; i++, ptr++)
	{
	    xdith = magic[i & 0x0F];
	    *ptr = mgx11colors[((rmod>xdith)?rdiv1:rdiv) +
	               mgx11multab[((gmod>xdith)?gdiv1:gdiv) +
	                   mgx11multab[(bmod>xdith)?bdiv1:bdiv]]];
	}
    }
}

static void
Xmgr_ZdoLines(unsigned char *buf, float *zbuf, int zwidth, int width, 
	      int height, int miny, int maxy, int *color, endPoint *mug)
{
    register unsigned char *ptr;           /* pixel pointers   */
    register float *zptr;                 /* zbuff pointers   */
    register int i, x2;                  /* dithering junk   */
    int y, x1;                          /* current line     */
    double z, z2, zdelta;              /* z-buffer stuff   */
    int col;

    col = ditherRGB(0, 0, color);
    for(y = miny; y <= maxy; y++)
    {
	if (mug[y].P1x < 0)
	{
	    x1 = 0;
	    z = mug[y].P1z - (mug[y].P2z-mug[y].P1z) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	}
	else
	{
	    x1 = mug[y].P1x;
	    z = mug[y].P1z;
	}
	if (mug[y].P2x >= width)
	{
	    x2 = width-1;
	    z2 = mug[y].P1z + (mug[y].P2z-mug[y].P1z) *
		(width-1-mug[y].P1x)/(mug[y].P2x-mug[y].P1x);
	}
	else
	{
	    x2 = mug[y].P2x;
	    z2 = mug[y].P2z;
	}
	zdelta = (x1==x2) ? 0 : ((z2-z)/(x2-x1));
	ptr = buf+width*y+x1; zptr = zbuf+width*y+x1;
	if (((z<-1) && (z2<-1)) || ((z>1) && (z2>1)))
	    continue;
	if ((z<-1) || (z2<-1))
	{
	    for (i=x1; i<=x2; i++, z+=zdelta, zptr++, ptr++)
		if ((z < *zptr) && (z>-1.0))
		{
		    *ptr = col;
		    *zptr = z;
		}
	}
	else
	for (i=x1; i<=x2; i++, z+=zdelta, zptr++, ptr++)
	    if (z < *zptr)
	    {
		*ptr = col;
		*zptr = z;
	    }
    }
}

static void
Xmgr_DZdoLines(unsigned char *buf, float *zbuf, int zwidth, int width, 
	       int height, int miny, int maxy, int *color, endPoint *mug)
{
    register unsigned char *ptr;           /* pixel pointers   */
    register float *zptr;                 /* zbuff pointers   */
    register int i, xdith, x2;           /* dithering junk   */
    int y, x1;                          /* current line     */
    double z, z2, zdelta;              /* z-buffer stuff   */
    int rmod, gmod, bmod, rdiv, gdiv, bdiv, rdiv1, gdiv1, bdiv1;
    int *magic;

    rmod = mgx11modN[color[0]]; rdiv = mgx11divN[color[0]]; rdiv1 = rdiv+1;
    gmod = mgx11modN[color[1]]; gdiv = mgx11divN[color[1]]; gdiv1 = gdiv+1;
    bmod = mgx11modN[color[2]]; bdiv = mgx11divN[color[2]]; bdiv1 = bdiv+1;
    for(y = miny; y <= maxy; y++)
    {
	if (mug[y].P1x < 0)
	{
	    x1 = 0;
	    z = mug[y].P1z - (mug[y].P2z-mug[y].P1z) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	}
	else
	{
	    x1 = mug[y].P1x;
	    z = mug[y].P1z;
	}
	if (mug[y].P2x >= width)
	{
	    x2 = width-1;
	    z2 = mug[y].P1z + (mug[y].P2z-mug[y].P1z) *
		(width-1-mug[y].P1x)/(mug[y].P2x-mug[y].P1x);
	}
	else
	{
	    x2 = mug[y].P2x;
	    z2 = mug[y].P2z;
	}
	zdelta = (x1==x2) ? 0 : ((z2-z)/(x2-x1));
	ptr = buf+width*y+x1; zptr = zbuf+width*y+x1;
	magic = mgx11magic[y & 0x0F];
	if (((z<-1) && (z2<-1)) || ((z>1) && (z2>1)))
	    continue;
	if ((z<-1) || (z2<-1))
	{
	    for (i=x1; i<=x2; i++, z+=zdelta, zptr++, ptr++)
		if ((z < *zptr) && (z>-1.0))
		{
		    xdith = magic[i & 0x0F];
		    *ptr = mgx11colors[((rmod>xdith)?rdiv1:rdiv) +
		    mgx11multab[((gmod>xdith)?gdiv1:gdiv) +
		    mgx11multab[(bmod>xdith)?bdiv1:bdiv]]];
		    *zptr = z;
		}
	}
	else
	    for (i=x1; i<=x2; i++, z+=zdelta, zptr++, ptr++)
		if (z < *zptr)
		{
		    xdith = magic[i & 0x0F];
		    *ptr = mgx11colors[((rmod>xdith)?rdiv1:rdiv) +
		    mgx11multab[((gmod>xdith)?gdiv1:gdiv) +
		    mgx11multab[(bmod>xdith)?bdiv1:bdiv]]];
		    *zptr = z;
		}
    }
}

static void
Xmgr_DGdoLines(unsigned char *buf, float *zbuf, int zwidth, int width, 
	       int height, int miny, int maxy, endPoint *mug)
{
    register unsigned char *ptr;          /* pixel pointers   */
    register int i, xdith, x2;           /* dithering junk   */
    int y, x1;                          /* current line     */
    int sr, sg, sb, er, eg, eb, dr, dg, db, r, g, b, dx;
    int *magic;

    for(y = miny; y <= maxy; y++)
    {
	if ((mug[y].P2x-mug[y].P1x)==0) continue;
	if (mug[y].P1x < 0)
	{
	    x1 = 0;
	    r = mug[y].P1r - (mug[y].P2r-mug[y].P1r) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	    g = mug[y].P1g - (mug[y].P2g-mug[y].P1g) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	    b = mug[y].P1b - (mug[y].P2b-mug[y].P1b) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	}
	else
	{
	    x1 = mug[y].P1x;
	    r = mug[y].P1r;
	    g = mug[y].P1g;
	    b = mug[y].P1b;
	}
	if (mug[y].P2x >= width)
	{
	    x2 = width-1;
	    dr = (mug[y].P1r + (mug[y].P2r-mug[y].P1r) *
		  (x2-mug[y].P1x)/(mug[y].P2x-mug[y].P1x))-r;
	    dg = (mug[y].P1g + (mug[y].P2g-mug[y].P1g) *
		  (x2-mug[y].P1x)/(mug[y].P2x-mug[y].P1x))-g;
	    db = (mug[y].P1b + (mug[y].P2b-mug[y].P1b) *
		  (x2-mug[y].P1x)/(mug[y].P2x-mug[y].P1x))-b;
	}
	else
	{
	    x2 = mug[y].P2x;
	    dr = mug[y].P2r-r;
	    dg = mug[y].P2g-g;
	    db = mug[y].P2b-b;
	}
	dx = x2-x1;
	er = (dr << 1) - dx;
	eg = (dg << 1) - dx;
	eb = (db << 1) - dx;
	sr = SGN(dr); dr = abs(dr)<<1;
	sg = SGN(dg); dg = abs(dg)<<1;
	sb = SGN(db); db = abs(db)<<1;
	dx <<= 1;
	ptr = buf+width*y+x1;
	magic = mgx11magic[y & 0x0F];
	for (i=x1; i<=x2; i++, ptr++)
	{
	    xdith = magic[i & 0x0F];
	    *ptr = mgx11colors[
	     ((mgx11modN[r]>xdith)?(mgx11divN[r]+1):mgx11divN[r]) +
	     mgx11multab[((mgx11modN[g]>xdith)?(mgx11divN[g]+1):mgx11divN[g])+
	     mgx11multab[(mgx11modN[b]>xdith)?(mgx11divN[b]+1):mgx11divN[b]]]];
	    if (dx)
	    {
	    while (er>0)
	    {
		r += sr;
		er -= dx;
	    }
	    while (eg>0)
	    {
		g += sg;
		eg -= dx;
	    }
	    while (eb>0)
	    {
		b += sb;
		eb -= dx;
	    }
	    }
	    er += dr; eg += dg; eb += db;
	}
    }
}

static void
Xmgr_DGZdoLines(unsigned char *buf, float *zbuf, int zwidth, int width, 
		int height, int miny, int maxy, endPoint *mug)
{
    register unsigned char *ptr;           /* pixel pointers   */
    register float *zptr;                 /* zbuff pointers   */
    register int i, xdith, x2;           /* dithering junk   */
    int y, x1;                          /* current line     */
    double z, z2, zdelta;              /* z-buffer stuff   */
    int sr, sg, sb, er, eg, eb, dr, dg, db, r, g, b, dx;
    int *magic;

    for(y = miny; y <= maxy; y++)
    {
	if ((mug[y].P2x-mug[y].P1x)==0) continue;
	if (mug[y].P1x < 0)
	{
	    x1 = 0;
	    z = mug[y].P1z - (mug[y].P2z-mug[y].P1z) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	    r = mug[y].P1r - (mug[y].P2r-mug[y].P1r) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	    g = mug[y].P1g - (mug[y].P2g-mug[y].P1g) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	    b = mug[y].P1b - (mug[y].P2b-mug[y].P1b) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	}
	else
	{
	    x1 = mug[y].P1x;
	    z = mug[y].P1z;
	    r = mug[y].P1r;
	    g = mug[y].P1g;
	    b = mug[y].P1b;
	}
	if (mug[y].P2x >= width)
	{
	    x2 = width-1;
	    z2 = mug[y].P1z + (mug[y].P2z-mug[y].P1z) *
		(x2-mug[y].P1x)/(mug[y].P2x-mug[y].P1x);
	    dr = (mug[y].P1r + (mug[y].P2r-mug[y].P1r) *
		   (x2-mug[y].P1x)/(mug[y].P2x-mug[y].P1x))-r;
	    dg = (mug[y].P1g + (mug[y].P2g-mug[y].P1g) *
		   (x2-mug[y].P1x)/(mug[y].P2x-mug[y].P1x))-g;
	    db = (mug[y].P1b + (mug[y].P2b-mug[y].P1b) *
		   (x2-mug[y].P1x)/(mug[y].P2x-mug[y].P1x))-b;
	}
	else
	{
	    x2 = mug[y].P2x;
	    z2 = mug[y].P2z;
	    dr = mug[y].P2r-r;
	    dg = mug[y].P2g-g;
	    db = mug[y].P2b-b;
	}
	dx = x2-x1;
	zdelta = dx ? (z2-z)/dx : 0;
	er = (dr << 1) - dx;
	eg = (dg << 1) - dx;
	eb = (db << 1) - dx;
	sr = SGN(dr); dr = abs(dr)<<1;
	sg = SGN(dg); dg = abs(dg)<<1;
	sb = SGN(db); db = abs(db)<<1;
	dx <<= 1;
	ptr = buf+width*y+x1; zptr = zbuf+width*y+x1;
	magic = mgx11magic[y & 0x0F];
	if (((z<-1) && (z2<-1)) || ((z>1) && (z2>1)))
	    continue;
	if ((z<-1) || (z2<-1))
	    for (i=x1; i<=x2; i++, z+=zdelta, zptr++, ptr++)
	    {
		if ((z < *zptr) && (z>-1.0))
		{
		    xdith = magic[i & 0x0F];
		    *ptr = mgx11colors[
		    ((mgx11modN[r]>xdith)?(mgx11divN[r]+1):mgx11divN[r]) +
		    mgx11multab[((mgx11modN[g]>xdith)?(mgx11divN[g]+1):mgx11divN[g])+
		    mgx11multab[(mgx11modN[b]>xdith)?(mgx11divN[b]+1):mgx11divN[b]]]];
		    *zptr = z;
		}
		if (dx)
		{
		while (er>0)
		{
		    r += sr;
		    er -= dx;
		}
		while (eg>0)
		{
		    g += sg;
		    eg -= dx;
		}
		while (eb>0)
		{
		    b += sb;
		    eb -= dx;
		}
		}
		er += dr; eg += dg; eb += db;
	    }
	else
	    for (i=x1; i<=x2; i++, z+=zdelta, zptr++, ptr++)
	    {
		if (z < *zptr)
		{
		    xdith = magic[i & 0x0F];
		    *ptr = mgx11colors[
		    ((mgx11modN[r]>xdith)?(mgx11divN[r]+1):mgx11divN[r]) +
		    mgx11multab[((mgx11modN[g]>xdith)?(mgx11divN[g]+1):mgx11divN[g])+
		    mgx11multab[(mgx11modN[b]>xdith)?(mgx11divN[b]+1):mgx11divN[b]]]];
		    *zptr = z;
		}
		if (dx)
		{
		while (er>0)
		{
		    r += sr;
		    er -= dx;
		}
		while (eg>0)
		{
		    g += sg;
		    eg -= dx;
		}
		while (eb>0)
		{
		    b += sb;
		    eb -= dx;
		}
		}
		er += dr; eg += dg; eb += db;
	    }
    }
}


void
Xmgr_8DZpoly(unsigned char *buf, float *zbuf, int zwidth, 
	     int width, int height, CPoint3 *p, int n, int *color)
{
    Xmgr_Zpolyscan(buf, zbuf, zwidth, width, height, p, n, color, mug,
		   Xmgr_DZdoLines);
}

void
Xmgr_8Zpoly(unsigned char *buf, float *zbuf, int zwidth, 
	    int width, int height, CPoint3 *p, int n, int *color)
{
    Xmgr_Zpolyscan(buf, zbuf, zwidth, width, height, p, n, color, mug,
		   Xmgr_ZdoLines);
}

void
Xmgr_8Dpoly(unsigned char *buf, float *zbuf, int zwidth, 
	    int width, int height, CPoint3 *p, int n, int *color)
{
    Xmgr_polyscan(buf, zbuf, zwidth, width, height, p, n, color, mug,
		  Xmgr_DdoLines);
}

void
Xmgr_8poly(unsigned char *buf, float *zbuf, int zwidth, 
	   int width, int height, CPoint3 *p, int n, int *color)
{
    Xmgr_polyscan(buf, zbuf, zwidth, width, height, p, n, color, mug,
		  Xmgr_doLines);
}

void
Xmgr_8DGZpoly(unsigned char *buf, float *zbuf, int zwidth, 
	      int width, int height, CPoint3 *p, int n, int *dummy)
{
    Xmgr_GZpolyscan(buf, zbuf, zwidth, width, height, p, n, mug,
		    Xmgr_DGZdoLines);
}

void
Xmgr_8DGpoly(unsigned char *buf, float *zbuf, int zwidth, 
	      int width, int height, CPoint3 *p, int n, int *dummy)
{
    Xmgr_Gpolyscan(buf, zbuf, zwidth, width, height, p, n, mug,
		   Xmgr_DGdoLines);
}

/* }}} */

/* {{{ Multiple line scan-conversion functions */

void
Xmgr_8polyline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, CPoint3 *p, int n, int lwidth, int *color)
{
    CPoint3 *p0, *p1;
    int i;
    p0 = &p[0];
    if (n == 1)
    {
	if ((p0->x > 0) && (p0->x < width) && (p0->y > 0) && (p0->y <height))
	    buf[(int)(p0->y)*width+(int)p0->x] = ditherRGB(0, 0, color);
	return;
    }
    p1 = &p[1];
    Xmgr_8line(buf, zbuf, zwidth, width, height, p0->x, p0->y, 0, 
	        p1->x, p1->y, 0, lwidth, color);
    for (i=2; i<n; i++)
    {
	p0 = p1;
        p1 = &p[i];
        Xmgr_8line(buf, zbuf, zwidth, width, height, p0->x, p0->y, 0,
		    p1->x, p1->y, 0, lwidth, color);
    }
}

void
Xmgr_8Dpolyline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, CPoint3 *p, int n, int lwidth, int *color)
{
    CPoint3 *p0, *p1;
    int i;
    p0 = &p[0];
    if (n == 1)
    {
	if ((p0->x > 0) && (p0->x < width) && (p0->y > 0) && (p0->y <height))
	    buf[(int)(p0->y)*width+(int)p0->x] = ditherRGB(0, 0, color);
	return;
    }
    p1 = &p[1];
    Xmgr_8Dline(buf, zbuf, zwidth, width, height, p0->x, p0->y, 0, 
		p1->x, p1->y, 0, lwidth, color);
    for (i=2; i<n; i++)
    {
	p0 = p1;
        p1 = &p[i];
        Xmgr_8Dline(buf, zbuf, zwidth, width, height, p0->x, p0->y, 0,
		    p1->x, p1->y, 0, lwidth, color);
    }
}

void
Xmgr_8Zpolyline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, CPoint3 *p, int n, int lwidth, int *color)
{
    CPoint3 *p0, *p1;
    int i;
    p0 = &p[0];
    if (n == 1)
    {
	if ((p0->x > 0) && (p0->x < width) && (p0->y > 0) && (p0->y <height))
	    if ((p0->z < zbuf[(int)(p0->y)*zwidth+(int)p0->x]) && (p0->z>-1.0))
		buf[(int)(p0->y)*width+(int)p0->x] = ditherRGB(0, 0, color);
	return;
    }
    p1 = &p[1];
    Xmgr_8Zline(buf, zbuf, zwidth, width, height, p0->x, p0->y, p0->z, p1->x, p1->y, p1->z, lwidth, color);
    for (i=2; i<n; i++)
    {
	p0 = p1;
        p1 = &p[i];
        Xmgr_8Zline(buf, zbuf, zwidth, width, height, p0->x, p0->y, p0->z, p1->x, p1->y, p1->z, lwidth, color);
    }
}

void
Xmgr_8DZpolyline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, CPoint3 *p, int n, int lwidth, int *color)
{
    CPoint3 *p0, *p1;
    int i;

    p0 = &p[0];
    if (n == 1)
    {
	if ((p0->x > 0) && (p0->x < width) && (p0->y > 0) && (p0->y <height))
	    if ((p0->z < zbuf[(int)(p0->y*zwidth)+(int)p0->x]) && (p0->z>-1.0))
		buf[(int)(p0->y)*width+(int)p0->x] = ditherRGB(0, 0, color);
	return;
    }
    p1 = &p[1];
    Xmgr_8DZline(buf, zbuf, zwidth, width, height, p0->x, p0->y, p0->z, p1->x, p1->y, p1->z, lwidth, color);
    for (i=2; i<n; i++)
    {
	p0 = p1;
        p1 = &p[i];
        Xmgr_8DZline(buf, zbuf, zwidth, width, height, p0->x, p0->y, p0->z, p1->x, p1->y, p1->z, lwidth, color);
    }
}

void
Xmgr_8DGZpolyline(unsigned char *buf, float *zbuf, int zwidth, 
		  int width, int height, CPoint3 *p, int n, 
		  int lwidth, int *color)
{
    CPoint3 *p0, *p1;
    int i;

    p0 = &p[0];
    if (n == 1)
    {
	if ((p0->x > 0) && (p0->x < width) && (p0->y > 0) && (p0->y <height))
	    if ((p0->z < zbuf[(int)(p0->y*zwidth)+(int)p0->x]) && (p0->z>-1.0))
		buf[(int)(p0->y)*width+(int)p0->x] = ditherRGB(0, 0, color);
	return;
    }
    p1 = &p[1];
    Xmgr_gradWrapper(buf, zbuf, zwidth, width, height, p0, p1, lwidth,
			Xmgr_8DZline, Xmgr_8DGZline);
    for (i=2; i<n; i++)
    {
	p0 = p1;
        p1 = &p[i];
        Xmgr_gradWrapper(buf, zbuf, zwidth, width, height, p0, p1, lwidth,
			Xmgr_8DZline, Xmgr_8DGZline);
    }
}

void
Xmgr_8DGpolyline(unsigned char *buf, float *zbuf, int zwidth, 
		 int width, int height, CPoint3 *p, int n, 
		 int lwidth, int *color)
{
    CPoint3 *p0, *p1;
    int i;

    p0 = &p[0];
    if (n == 1)
    {
	if ((p0->x > 0) && (p0->x < width) && (p0->y > 0) && (p0->y <height))
	    buf[(int)(p0->y)*width+(int)p0->x] = ditherRGB(0, 0, color);
	return;
    }
    p1 = &p[1];
    Xmgr_gradWrapper(buf, zbuf, zwidth, width, height, p0, p1, lwidth,
			Xmgr_8Dline, Xmgr_8DGline);
    for (i=2; i<n; i++)
    {
	p0 = p1;
        p1 = &p[i];
        Xmgr_gradWrapper(buf, zbuf, zwidth, width, height, p0, p1, lwidth,
			Xmgr_8Dline, Xmgr_8DGline);
    }
}

/* }}} */

/*
Local variables:
folded-file: t
*/

